/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowSamples.h"
#include "WorkflowViewController.h"
#include "SceneSerializer.h"
#include <core_api/Log.h>
#include <core_api/Settings.h>

#include <workflow_support/WorkflowUtils.h>
#include <workflow_support/SchemaSerializer.h>

#include <QtCore/QDir>
#include <QtCore/QFile>
#include <QtCore/QUrl>
#include <QtGui/QLabel>
#include <QtGui/QMenu>
#include <QtGui/QToolButton>

#include <QtGui/QHeaderView>
#include <QtGui/QApplication>
#include <QtCore/QAbstractItemModel>
#include <QtGui/QTreeView>
#include <QtGui/QStyle>
#include <QtGui/QPainter>
#include <QtGui/QStyledItemDelegate>
#include <QtGui/QVBoxLayout>
#include <QtGui/QContextMenuEvent>
#include <QtGui/QTextDocument>
#include <QtGui/QAbstractTextDocumentLayout>

Q_DECLARE_METATYPE(QTextDocument*);

namespace GB2 {

static LogCategory log(ULOG_CAT_WD);

const QString SamplesWidget::MIME_TYPE("application/x-ugene-sample-id");
QList<SampleCategory> SampleRegistry::data;

static QPixmap genSnap(const QDomDocument& xml);

#define DATA_ROLE Qt::UserRole
#define INFO_ROLE Qt::UserRole + 1
#define ICON_ROLE Qt::UserRole + 2
#define DOC_ROLE Qt::UserRole + 3

class SampleDelegate : public QStyledItemDelegate {
public:
    SampleDelegate(QObject* parent = 0) : QStyledItemDelegate(parent){}
    QSize sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const
    {
        QVariant value = index.data(Qt::SizeHintRole);
        if (value.isValid())
            return qvariant_cast<QSize>(value);

        QStyleOptionViewItemV4 opt = option;
        initStyleOption(&opt, index);
        const QWidget *widget = qobject_cast<QWidget*>(parent());//QStyledItemDelegatePrivate::widget(option);
        QStyle *style = widget ? widget->style() : QApplication::style();
        opt.rect.setSize(widget->size());
        return style->sizeFromContents(QStyle::CT_ItemViewItem, &opt, QSize(), widget);
    }

//     void QStyledItemDelegate::paint(QPainter *painter,
//         const QStyleOptionViewItem &option, const QModelIndex &index) const
//     {
//         Q_ASSERT(index.isValid());
// 
//         QStyleOptionViewItemV4 opt = option;
//         initStyleOption(&opt, index);
// 
//         const QWidget *widget = qobject_cast<QWidget*>(parent());//QStyledItemDelegatePrivate::widget(option);
//         QStyle *style = widget ? widget->style() : QApplication::style();
//         style->drawControl(QStyle::CE_ItemViewItem, &opt, painter, widget);
//     }

    /*void initStyleOption(QStyleOptionViewItem *option, const QModelIndex &index) const
    {
        QStyledItemDelegate::initStyleOption(option, index);
        QWidget* owner;
        if ((owner = qobject_cast<QWidget*>(parent()))) {
            option->rect.setWidth(owner->width());
            option->rect.setHeight(0);
        }
//         QVariant value = index.data(Qt::FontRole);
//         if (value.isValid() && !value.isNull()) {
//             option->font = qvariant_cast<QFont>(value).resolve(option->font);
//             option->fontMetrics = QFontMetrics(option->font);
//         }
// 
//         value = index.data(Qt::TextAlignmentRole);
//         if (value.isValid() && !value.isNull())
//             option->displayAlignment = (Qt::Alignment)value.toInt();
// 
//         value = index.data(Qt::ForegroundRole);
//         if (qVariantCanConvert<QBrush>(value))
//             option->palette.setBrush(QPalette::Text, qvariant_cast<QBrush>(value));

        if (QStyleOptionViewItemV2 *v2 = qstyleoption_cast<QStyleOptionViewItemV2 *>(option)) {
            v2->features |= QStyleOptionViewItemV2::WrapText;
        }
    }*/
};

SamplesWidget::SamplesWidget(QWidget *parent) : QTreeWidget(parent) {
    setColumnCount(1);
    //header()->hide();
    setHeaderHidden(true);
    setItemDelegate(new SampleDelegate(this));
    setWordWrap(true);

    foreach(const SampleCategory& cat, SampleRegistry::getCategories()) {
        addCategory(cat);
    }

    expandAll();

    glass = new SamplePane();

    //connect(this, SIGNAL(itemActivated(QTreeWidgetItem*,int)), SLOT(handleItem(QTreeWidgetItem*)));
    connect(this, SIGNAL(currentItemChanged(QTreeWidgetItem*,QTreeWidgetItem*)), SLOT(handleTreeItem(QTreeWidgetItem*)));
    //connect(this, SIGNAL(itemSelectionChanged()), glass, SLOT(test()));
    connect(this, SIGNAL(itemDoubleClicked(QTreeWidgetItem*, int)), SLOT(activateItem(QTreeWidgetItem*)));
    connect(glass, SIGNAL(itemActivated(QTreeWidgetItem*)), SLOT(activateItem(QTreeWidgetItem*)));
    connect(glass, SIGNAL(cancel()), SLOT(cancelItem()));
}

void SamplesWidget::activateItem(QTreeWidgetItem * item) {
    if (item && item->data(0, DATA_ROLE).isValid()) {
        emit sampleSelected(item->data(0, DATA_ROLE).toString());
    }
}

void SamplesWidget::handleTreeItem(QTreeWidgetItem * item) {
//     bool show = false;
     if (item && !item->data(0, DATA_ROLE).isValid()) {
         item = NULL;
//         glass->setItem(item);
//         show = true;
     }
//     emit setupGlass(show ? glass : NULL);

    glass->setItem(item);
    emit setupGlass(glass);
}

void SamplesWidget::cancelItem() {
     selectionModel()->clear();
     if (isHidden()) {
         emit setupGlass(NULL);
         glass->setItem(NULL);
     } else {
         emit setupGlass(glass);
     }
}

void SamplesWidget::addCategory( const SampleCategory& cat )
{
    QTreeWidgetItem* ci = new QTreeWidgetItem(this, QStringList(cat.d.getDisplayName()));
    ci->setFlags(Qt::ItemIsEnabled);
    QFont cf;
    cf.setBold(true);
    ci->setData(0, Qt::FontRole, cf);
    ci->setData(0, Qt::BackgroundRole, QColor(255,255,160, 127));
    //QFont sf;
    //sf.setBold(true);
    //sf.setItalic(true);

    foreach(const Sample& item, cat.items) {
        QTreeWidgetItem* ib = new QTreeWidgetItem(ci, QStringList(item.d.getDisplayName()));
        ib->setData(0, DATA_ROLE, item.content.toString());
        //ib->setData(0, ICON_ROLE, item.ico.pixmap(200));
        //ib->setData(0, INFO_ROLE, qVariantFromValue<Descriptor>(item.d));
        QTextDocument* doc = new QTextDocument(this);
        ib->setData(0, DOC_ROLE, qVariantFromValue<QTextDocument*>(doc));
        //ib->setData(0, Qt::FontRole, sf);
        Descriptor d = item.d;
        QString text = 
            "<html>"
            "<table align='center' border='0' cellpadding='3' cellspacing='3'>"
            "<tr><td colspan='2'><h1 align='center'>%1</h1></td></tr>"
            "<tr><td valign='middle' width='20%'><img src=\"%2\"/></td><td valign='bottom'><br>%3</td></tr>"
            "<tr><td colspan='2' valign='top'>%4<br></td></tr>"
            "<tr><td colspan='2' bgcolor='gainsboro' align='center'><font color='maroon' size='+2' face='Courier'><b>%5</b></font></td></tr>"
            "</table>"
            "</html>";
        QString img("img://img");
        QIcon ico = item.ico;
        if (ico.isNull()) {
            ico.addPixmap(genSnap(item.content));
        }
        doc->addResource(QTextDocument::ImageResource, QUrl(img), ico.pixmap(200));
        QString body = Qt::escape(d.getDocumentation()).replace("\n", "<br>");
        int brk = body.indexOf("<br><br>");
        int shift = 8;
        if (brk <= 0) {
            brk = body.indexOf("<br>");
            shift = 4;
        }
        QString body2;
        if (brk > 0) {
            body2 = body.mid(brk + shift);
            body = body.left(brk);
        }
        text = text.arg(d.getDisplayName()).arg(img).arg(body).arg(body2)
            .arg(tr("Double click to load the sample"));
        doc->setHtml(text);
        QFont f;
        //f.setFamily("Times New Roman");
        f.setPointSizeF(12);
        doc->setDefaultFont(f);
    }
}

void SamplePane::setItem(QTreeWidgetItem* it) {
    item = it;
//     if (!item) {
//         m_document->clear();
//         return;
//     }
//     Descriptor d = it->data(0, INFO_ROLE).value<Descriptor>();
//     QString text = 
//         "<html>"
//         "<table align='center' border='0'>"
//         "<tr><td><h1 align='center'>%1</h1></td></tr>"
//         "<tr><td><img src=\"%2\"/>%3</td></tr>"
//         "<tr><td bgcolor='gainsboro' align='center'><font color='maroon' size='+2'>%4</font></td></tr>"
//         "</table>"
//         "</html>";
//     QString img("img://img");
//     m_document->addResource(QTextDocument::ImageResource, QUrl(img), it->data(0, ICON_ROLE));
//     
//     //QString img = QString("<table align='left' width='250' border='0'><tr><td><img src=\"img://img\"/></td></tr></table>");
//     QString body = Qt::escape(d.getDocumentation()).replace("\n", "<br>");
//     text = text.arg(d.getDisplayName()).arg(img).arg(body).arg(tr("Double click to load the sample"));
//     m_document->setHtml(text);
}

void SamplePane::test() {
    log.error("Acha!!!");
}

void SamplePane::mouseDoubleClickEvent( QMouseEvent *) {
    emit itemActivated(item);
}

void SamplePane::keyPressEvent(QKeyEvent * event ){
    if (event->key() == Qt::Key_Escape) {
        emit cancel();
    } else if (event->key() == Qt::Key_Enter) {
        emit itemActivated(item);
    }
}

SamplePane::SamplePane() : item(NULL) {
    m_document = new QTextDocument(this);
}

void SamplePane::paint(QPainter* painter)
{
    if (!item) {
        QPen pen(Qt::darkGray);
        pen.setWidthF(2);
        painter->setPen(pen);
        painter->setRenderHint(QPainter::SmoothPixmapTransform);
        QFont f = painter->font();
        painter->resetTransform();
        f.setFamily("Times New Roman");
        f.setPointSizeF(20);
        f.setItalic(true);
        painter->setFont(f);

        QRectF approx(50,50, 400, 400);
        QString txt = tr("Select a sample to start");
        QRectF res = painter->boundingRect(approx, Qt::AlignLeft | Qt::AlignTop, txt);
        res.adjust(-5,-3,15,3);

        QPainterPath p(QPointF(5, res.center().y()));
        p.lineTo(res.topLeft());
        p.lineTo(res.topRight());
        p.lineTo(res.bottomRight());
        p.lineTo(res.bottomLeft());
        p.closeSubpath();
        QColor yc = QColor(255,255,160);//QColor(Qt::yellow).lighter();yc.setAlpha(127);
        painter->fillPath(p, QBrush(yc));
        painter->drawPath(p);
        painter->setPen(Qt::black);
        painter->drawText(approx, Qt::AlignLeft | Qt::AlignTop, txt);

        return;
    }
    QTextDocument* doc = item->data(0, DOC_ROLE).value<QTextDocument*>();
    int pageWidth = qMax(width() - 100, 100);
    int pageHeight = qMax(height() - 100, 100);
    if (pageWidth != doc->pageSize().width()) {
        doc->setPageSize(QSize(pageWidth, pageHeight));
    }

    QSize ts = doc->size().toSize();

    QRect textRect(width() / 2 - pageWidth / 2,
        height() / 2 - pageHeight / 2,
        pageWidth,
        pageHeight);
    textRect.setSize(ts);
    int pad = 10;
    QRect clearRect = textRect.adjusted(-pad, -pad, pad, pad);
    painter->setPen(Qt::NoPen);
    painter->setBrush(QColor(0, 0, 0, 63));
    int shade = 10;
    painter->drawRect(clearRect.x() + clearRect.width() + 1,
        clearRect.y() + shade,
        shade,
        clearRect.height() + 1);
    painter->drawRect(clearRect.x() + shade,
        clearRect.y() + clearRect.height() + 1,
        clearRect.width() - shade + 1,
        shade);

    painter->setRenderHint(QPainter::Antialiasing, false);
    painter->setBrush(QColor(255, 255, 255/*, 220*/));
    painter->setPen(Qt::black);
    painter->drawRect(clearRect);

    painter->setClipRegion(textRect, Qt::IntersectClip);
    painter->translate(textRect.topLeft());

    QAbstractTextDocumentLayout::PaintContext ctx;

    QLinearGradient g(0, 0, 0, textRect.height());
    g.setColorAt(0, Qt::black);
    g.setColorAt(0.9, Qt::black);
    g.setColorAt(1, Qt::transparent);

    QPalette pal = palette();
    //pal.setBrush(QPalette::Text, g);

    ctx.palette = pal;
    ctx.clip = QRect(0, 0, textRect.width(), textRect.height());
    doc->documentLayout()->draw(painter, ctx);
}


LoadSamplesTask::LoadSamplesTask( const QStringList& lst) 
: Task(tr("Load workflow samples"), TaskFlag_None), dirs(lst) {}

void LoadSamplesTask::run()
{
    foreach(const QString& s, dirs) {
        scanDir(s);
    }
}

static QPixmap genSnap(const QDomDocument& xml) {
    WorkflowScene* scene = new WorkflowScene();
    QMap<ActorId, ActorId> stub;
    QString msg = SceneSerializer::xml2scene(xml.documentElement(), scene, stub, true, true);
    if (!msg.isEmpty()) {
        log.trace("Snapshot issues: " + msg);
    }

    QRectF bounds = scene->itemsBoundingRect();
    QPixmap pixmap(bounds.size().toSize());
    pixmap.fill();
    QPainter painter(&pixmap);
    painter.setRenderHint(QPainter::Antialiasing);
    scene->render(&painter, QRectF(), bounds);
    delete scene;
    return pixmap;
}

void LoadSamplesTask::scanDir( const QString& s)
{
    QDir dir(s);
    if (!dir.exists()) {
        log.error(tr("Sample dir does not exist: %1").arg(s));
        return;
    }
    SampleCategory category(s,dir.dirName());
    QStringList names(QString("*.") + DesignerUtils::SCHEMA_FILE_EXT);
    foreach(const QFileInfo& fi, dir.entryInfoList(names, QDir::Files|QDir::NoSymLinks)) {
        QFile f(fi.absoluteFilePath());
        QString err;
        QDomDocument doc;
        if (f.open(QIODevice::ReadOnly) 
            && doc.setContent(&f, &err) 
            && doc.doctype().name() == SchemaSerializer::WORKFLOW_DOC) 
        {
            Sample sample;
            sample.content = doc;

            Metadata meta;
            err = SchemaSerializer::readMeta(&meta, doc.documentElement());
            sample.d = Descriptor(fi.absoluteFilePath(), 
                meta.name.isEmpty()? fi.baseName() : meta.name, meta.comment);
            QString icoName = dir.absoluteFilePath(fi.baseName() + ".png");
            if (QFile::exists(icoName)) {
                sample.ico.addFile(icoName);
            } else {
                //sample.ico.addPixmap(genSnap(doc));
            }
            category.items << sample;
        } else {
            log.error(tr("Failed to read sample: %1 (%2)").arg(fi.absoluteFilePath()).arg(err));
        }
    }
    if (!category.items.isEmpty()) {
        result << category;
    }
    foreach(const QFileInfo& fi, dir.entryInfoList(QStringList(), QDir::AllDirs|QDir::NoSymLinks|QDir::NoDotAndDotDot)) {
        scanDir(fi.absoluteFilePath());
    }
}

Task::ReportResult LoadSamplesTask::report()
{
    SampleRegistry::data = result;
    return ReportResult_Finished;
}

Task* SampleRegistry::init( const QStringList& lst) {
    return new LoadSamplesTask(lst);
}

} //namespace
