/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "CoreLib.h"
#include "BaseDocWorker.h"
#include "DocWorkers.h"
#include "DocActors.h"
#include "GenericReadActor.h"
#include "FindWorker.h"
#include "SWWorker.h"
#include "SequenceSplitWorker.h"

#include <core_api/DocumentFormats.h>

#include <workflow/TypeSet.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowManager.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow/WorkflowEnv.h>

#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/LocalDomain.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/TypeMapEditor.h>
#include <workflow_support/DelegateEditors.h>

#include <core_api/core_api.h>
#include <util_gui/DialogUtils.h>
#include <util_gui/GUIUtils.h>

/* TRANSLATOR GB2::Workflow::CoreLib */

namespace GB2 {
using namespace LocalWorkflow;
namespace Workflow {


const QString CoreLib::READ_FASTA_ACTOR("read.fasta");
const QString CoreLib::WRITE_FASTA_ACTOR("write.fasta");
const QString CoreLib::READ_TEXT_ACTOR("read.text");
const QString CoreLib::WRITE_TEXT_ACTOR("write.text");

const QString CoreLib::READ_GENBANK_ACTOR("read.gbk");
const QString CoreLib::WRITE_GENBANK_ACTOR("write.gbk");

const QString CoreLib::READ_CLUSTAL_ACTOR("read.clustalw");
const QString CoreLib::WRITE_CLUSTAL_ACTOR("write.clustalw");

const QString CoreLib::READ_STOCKHOLM_ACTOR("read.stockholm");
const QString CoreLib::WRITE_STOCKHOLM_ACTOR("write.stockholm");

const QString CoreLib::GENERIC_MA_ACTOR("read.malignment");
const QString CoreLib::GENERIC_SEQ_ACTOR("read.sequence");

const QString CoreLib::URL_ATTR_ID("URL");
const QString CoreLib::APPEND_ATTR_ID("append");
const QString CoreLib::URL_SLOT_ID("URL");

const QString CoreLib::DATA_PORT_ID("data");

const QString CoreLib::FASTA_HEADER_SLOT_ID("fasta.header");
const QString CoreLib::GENBANK_ACN_SLOT_ID("genbank.ACN");

const Descriptor CoreLib::URL_ATTR(){return Descriptor(URL_ATTR_ID, tr("Location"), tr("Location of data file(s)"));}
const Descriptor CoreLib::READ_URL_SLOT() {return Descriptor(URL_SLOT_ID, tr("Source URL"), tr("Location of a corresponding input file."));}

const QString CoreLib::GENBANK_TYPESET_ID("genbank.content");
const QString CoreLib::FASTA_TYPESET_ID("fasta.content");
const QString CoreLib::TEXT_TYPESET_ID("text.content");
const QString CoreLib::MA_TYPESET_ID("ma.content");

void CoreLib::init() {

    Descriptor apnd(CoreLib::APPEND_ATTR_ID, tr("Accumulate objects"), 
        tr("Accumulate all incoming data in one file or create separate files for each input."
        "In the latter case, an incremental numerical suffix is added to the file name."));
    Descriptor writeUrlD(URL_SLOT_ID, tr("Location"), tr("Location for writing data"));
    Descriptor fastaHd(FASTA_HEADER_SLOT_ID, tr("FASTA header"), tr("A header line for the FASTA record."));
    //Descriptor gbAccd(GENBANK_ACN_SLOT_ID, tr("Genbank Accession Number"), tr("Accession Number within NCBI GenBank nucleotide sequence database"));

    DataTypeRegistry* dr = WorkflowEnv::getDataTypeRegistry();
    assert(dr);

//     DataTypePtr readMAType;
//     {
//         QMap<Descriptor, DataTypePtr> m;
//         m[CoreLib::READ_URL_SLOT()] = CoreDataTypes::STRING_TYPE();
//         m[BioActorLibrary::MA_SLOT()] = BioDataTypes::MULTIPLE_ALIGNMENT_TYPE();
//         readMAType = new DataTypeSet(Descriptor(CoreLib::MA_TYPESET_ID), m);
//         dr->registerEntry(readMAType);
//     }
    DataTypePtr writeMAType;
    {
        QMap<Descriptor, DataTypePtr> m;
        m[writeUrlD] = CoreDataTypes::STRING_TYPE();
        m[BioActorLibrary::MA_SLOT()] = BioDataTypes::MULTIPLE_ALIGNMENT_TYPE();
        writeMAType = new DataTypeSet(Descriptor(CoreLib::MA_TYPESET_ID), m);
        //dr->registerEntry(writeMAType);
    }

    WProtoRegistry* r = WorkflowEnv::getProtoRegistry();
    assert(r);
    r->registerProto(BioActorLibrary::CATEGORY_DATASRC(), new GenericMAActorProto());
    r->registerProto(BioActorLibrary::CATEGORY_DATASRC(), new GenericSeqActorProto());

    QMap<QString,QString> aliases;
    aliases["read.fasta"] = "read.sequence";
    aliases["read.gbk"] = "read.sequence";
    aliases["read.stockholm"] = "read.malignment";
    aliases["read.clustalw"] = "read.malignment";
    r->addAliases(aliases);


//     {
//         QMap<Descriptor, DataTypePtr> m;
//         m[CoreLib::READ_URL_SLOT()] = CoreDataTypes::STRING_TYPE();
//         m[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
//         m[fastaHd] = CoreDataTypes::STRING_TYPE();
//         DataTypePtr FASTA_TYPESET(new DataTypeSet(Descriptor(CoreLib::FASTA_TYPESET_ID), m));
//         dr->registerEntry(FASTA_TYPESET);
// 
//         QList<PortDescriptor*> p; QList<Attribute*> a;
//         Descriptor acd(CoreLib::READ_FASTA_ACTOR, tr("Read FASTA"), tr("Reads sequences from file(s) in textual FASTA format. The files can be local or Internet URLs."));
//         Descriptor pd(CoreLib::DATA_PORT_ID, tr("Sequence"), tr("A sequence along with FASTA header line."));
//         p << new PortDescriptor(pd, FASTA_TYPESET, false, true);
//         BusActorPrototype* proto = new DocActorProto(BaseDocumentFormats::PLAIN_FASTA, true, acd, p, a);
//         proto->setPrompter(new ReadDocPrompter(tr("Read all sequences from <u>%1</u> and output each in turn.")));
//         r->registerProto(BioActorLibrary::CATEGORY_DATASRC(), proto);
//     }

    {
        QMap<Descriptor, DataTypePtr> m;
        m[writeUrlD] = CoreDataTypes::STRING_TYPE();
        m[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
        m[fastaHd] = CoreDataTypes::STRING_TYPE();
        DataTypePtr FASTA_TYPESET(new DataTypeSet(Descriptor(CoreLib::FASTA_TYPESET_ID), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLib::WRITE_FASTA_ACTOR, tr("Write FASTA"), tr("Writes all supplied sequences to file(s) in FASTA format."));
        Descriptor pd(CoreLib::DATA_PORT_ID, tr("Sequence"), tr("A sequence along with FASTA header line."));
        p << new PortDescriptor(pd, FASTA_TYPESET, true);
        a << new Attribute(apnd, CoreDataTypes::BOOL_TYPE(), false, true);
        BusActorPrototype* proto = new DocActorProto(BaseDocumentFormats::PLAIN_FASTA, false, acd, p, a);
        proto->setPrompter(new WriteFastaPrompter());
        r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
    }

//     {
//         QMap<Descriptor, DataTypePtr> m;
//         m[CoreLib::READ_URL_SLOT()] = CoreDataTypes::STRING_TYPE();
//         m[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
//         m[BioActorLibrary::FEATURE_TABLE_SLOT()] = BioDataTypes::ANNOTATION_TABLE_TYPE();
//         //m[gbAccd] = CoreDataTypes::STRING_TYPE();
//         DataTypePtr GENBANK_TYPESET(new DataTypeSet(Descriptor(CoreLib::GENBANK_TYPESET_ID), m));
//         dr->registerEntry(GENBANK_TYPESET);
// 
//         QList<PortDescriptor*> p; QList<Attribute*> a;
//         Descriptor acd(CoreLib::READ_GENBANK_ACTOR, tr("Read Genbank"), tr("Reads sequences and annotations from file(s) in NCBI's Genbank format. The files can be local or Internet URLs."));
//         Descriptor pd(CoreLib::DATA_PORT_ID, tr("Output data"), tr("Sequence and set of annotations"));
//         p << new PortDescriptor(pd, GENBANK_TYPESET, false, true);
//         BusActorPrototype* proto = new DocActorProto(BaseDocumentFormats::PLAIN_GENBANK, true, acd, p, a);
//         proto->setPrompter(new ReadDocPrompter(tr("Read all Genbank entries from <u>%1</u>, output by turns each sequence and related set of annotations.")));
//         r->registerProto(BioActorLibrary::CATEGORY_DATASRC(), proto);
//     }

    {
        QMap<Descriptor, DataTypePtr> m;
        m[writeUrlD] = CoreDataTypes::STRING_TYPE();
        m[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
        m[BioActorLibrary::FEATURE_TABLE_SLOT()] = BioDataTypes::ANNOTATION_TABLE_LIST_TYPE();
        //m[gbAccd] = CoreDataTypes::STRING_TYPE();
        DataTypePtr dtl(new DataTypeSet(Descriptor("in.gb"), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLib::WRITE_GENBANK_ACTOR, tr("Write Genbank"), tr("Writes all supplied sequences and related annotations to file(s) in Genbank format."));
        Descriptor pd(CoreLib::DATA_PORT_ID, tr("Input data"), tr("Sequence and set of annotations"));
        p << new PortDescriptor(pd, dtl, true);
        a << new Attribute(apnd, CoreDataTypes::BOOL_TYPE(), false, true);
        BusActorPrototype* proto = new DocActorProto(BaseDocumentFormats::PLAIN_GENBANK, false, acd, p, a);
        proto->setPrompter(new WriteGenbankPrompter());
        r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
    }

    Descriptor txtd(DATA_PORT_ID, tr("Plain Text"), tr("Just a plain text w/o splitting to strings."));
    {
        QMap<Descriptor, DataTypePtr> m;
        m[CoreLib::READ_URL_SLOT()] = CoreDataTypes::STRING_TYPE();
        m[txtd] = CoreDataTypes::STRING_TYPE();
        DataTypePtr dtl(new DataTypeSet(Descriptor(CoreLib::TEXT_TYPESET_ID), m));
        dr->registerEntry(dtl);

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLib::READ_TEXT_ACTOR, tr("Read plain text"), tr("Plain text file reader"));
        p << new PortDescriptor(txtd, dtl, false, true);
        BusActorPrototype* proto = new DocActorProto(BaseDocumentFormats::PLAIN_TEXT, true, acd, p, a);
        proto->setPrompter(new ReadDocPrompter(tr("Read the whole text from <u>%1</u> and output it as a single string.")));
        proto->setIcon( GUIUtils::createRoundIcon(QColor(Qt::darkRed), 22) );
        r->registerProto(BioActorLibrary::CATEGORY_OTHER(), proto);
    }

    {
        QMap<Descriptor, DataTypePtr> m;
        m[writeUrlD] = CoreDataTypes::STRING_TYPE();
        m[txtd] = new ListDataType("string-list", CoreDataTypes::STRING_TYPE());
        DataTypePtr dtl(new DataTypeSet(Descriptor("in.text"), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLib::WRITE_TEXT_ACTOR, tr("Write plain text"), tr("Write strings to a file"));
        p << new PortDescriptor(txtd, dtl, true);
        a << new Attribute(apnd, CoreDataTypes::BOOL_TYPE(), false, true);
        BusActorPrototype* proto = new DocActorProto(BaseDocumentFormats::PLAIN_TEXT, false, acd, p, a);
        proto->setPrompter(new WriteDocPrompter(tr("Save text from <u>%1</u> to document <u>%2</u>."), DATA_PORT_ID));
        proto->setIcon( GUIUtils::createRoundIcon(QColor(Qt::darkRed), 22) );
        r->registerProto(BioActorLibrary::CATEGORY_OTHER(), proto);
    }

//     {
//         QList<PortDescriptor*> p; QList<Attribute*> a;
//         Descriptor acd(CoreLib::READ_CLUSTAL_ACTOR, tr("Read ClustalW"), tr("Reads alignments from file(s) in CLUSTALW format. The files can be local or Internet URLs."));
//         Descriptor pd(CoreLib::DATA_PORT_ID, tr("Multiple sequence alignment"), tr("Multiple sequence alignment"));
//         p << new PortDescriptor(pd, readMAType, false, true);
//         BusActorPrototype* proto = new DocActorProto(BaseDocumentFormats::CLUSTAL_ALN, true, acd, p, a);
//         proto->setPrompter(new ReadDocPrompter(tr("Read MSA (multiple sequence alignment) blocks from <u>%1</u>.")));
//         r->registerProto(BioActorLibrary::CATEGORY_DATASRC(), proto);
//     }

    {
        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLib::WRITE_CLUSTAL_ACTOR, tr("Write ClustalW"), tr("Writes all supplied alignments to file(s) in CLUSTALW format."));
        Descriptor pd(CoreLib::DATA_PORT_ID, tr("Multiple sequence alignment"), tr("Multiple sequence alignment"));
        p << new PortDescriptor(pd, writeMAType, true);
        BusActorPrototype* proto = new DocActorProto(BaseDocumentFormats::CLUSTAL_ALN, false, acd, p, a);
        proto->setPrompter(new WriteDocPrompter(tr("Save all MSAs from <u>%1</u> to document <u>%2</u>."), BioActorLibrary::MA_SLOT_ID));
        r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
    }
//     {
//         QList<PortDescriptor*> p; QList<Attribute*> a;
//         Descriptor acd(CoreLib::READ_STOCKHOLM_ACTOR, tr("Read Stockholm"), tr("Reads alignments from file(s) in Stockholm format. The files can be local or Internet URLs."));
//         Descriptor pd(CoreLib::DATA_PORT_ID, tr("Multiple sequence alignment"), tr("Multiple sequence alignment"));
//         p << new PortDescriptor(pd, readMAType, false, true);
//         BusActorPrototype* proto = new DocActorProto(BaseDocumentFormats::STOCKHOLM, true, acd, p, a);
//         proto->setPrompter(new ReadDocPrompter(tr("Read MSA (multiple sequence alignment) blocks from <u>%1</u>.")));
//         r->registerProto(BioActorLibrary::CATEGORY_DATASRC(), proto);
//     }

    {
        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor acd(CoreLib::WRITE_STOCKHOLM_ACTOR, tr("Write Stockholm"), tr("Writes all supplied alignments to file(s) in Stockholm format."));
        Descriptor pd(CoreLib::DATA_PORT_ID, tr("Multiple sequence alignment"), tr("Multiple sequence alignment"));
        p << new PortDescriptor(pd, writeMAType, true);
        a << new Attribute(apnd, CoreDataTypes::BOOL_TYPE(), false, true);
        BusActorPrototype* proto = new DocActorProto(BaseDocumentFormats::STOCKHOLM, false, acd, p, a);
        proto->setPrompter(new WriteDocPrompter(tr("Save all MSAs from <u>%1</u> to document <u>%2</u>."), BioActorLibrary::MA_SLOT_ID));
        r->registerProto(BioActorLibrary::CATEGORY_DATASINK(), proto);
    }

    DataWorkerFactory::init();
    FindWorkerFactory::init();
    SWWorkerFactory::init();
    SequenceSplitWorkerFactory::init();

}

}//Workflow namespace
}//GB2 namespace
