/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "GenericReadWorker.h"
#include "GenericReadActor.h"
#include "CoreLib.h"
#include <workflow_library/BioActorLibrary.h>
#include <workflow/WorkflowEnv.h>
#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/Log.h>
#include <core_api/DocumentModel.h>
#include <core_api/IOAdapter.h>
#include <util_algorithm/MSAUtils.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/GObjectUtils.h>
#include <gobjects/GObjectRelationRoles.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/MAlignmentObject.h>
#include <gobjects/AnnotationTableObject.h>
#include <document_format/DocumentFormatUtils.h>

namespace GB2 {
using namespace Workflow;
namespace LocalWorkflow {

static LogCategory log(ULOG_CAT_WD);

void GenericMSAReader::init() {
    mtype = WorkflowEnv::getDataTypeRegistry()->getById(GenericMAActorProto::TYPE);
    urls = DesignerUtils::expandToUrls(actor->getParameter("URL")->value.toString());
    assert(ports.size() == 1);
    ch = ports.values().first();
}

bool GenericMSAReader::isReady() {
    return !isDone();
}

Task* GenericMSAReader::tick() {
    if (cache.isEmpty() && !urls.isEmpty()) {
        Task* t = createReadTask(urls.takeFirst());
        connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
        return t;
    }
    while (!cache.isEmpty()) {
        ch->put(cache.takeFirst());
    }
    if (urls.isEmpty()) {
        done = true;
        ch->setEnded();
    }
    return NULL;
}

bool GenericMSAReader::isDone() {
    return done && cache.isEmpty();
}

void GenericMSAReader::sl_taskFinished() {
    LoadMSATask* t = qobject_cast<LoadMSATask*>(sender());
    if (!t->isFinished() || t->hasErrors()) {
        return;
    }
    foreach(MAlignment ma, t->results) {
        QVariantMap m;
        m.insert(CoreLib::URL_SLOT_ID, t->url);
        m.insert(BioActorLibrary::MA_SLOT_ID, qVariantFromValue<MAlignment>(ma)); 
        cache.append(Message(mtype, m));
    }
}

void LoadMSATask::run() {
    DocumentFormatConstraints mc;
    mc.supportedObjectTypes.append(GObjectTypes::MULTIPLE_ALIGNMENT);
    DocumentFormat* format = NULL;
    QList<DocumentFormat*> fs = DocumentFormatUtils::detectFormat(url);
    foreach(DocumentFormat* f, fs) {
        if (f->checkConstraints(mc)) {
            format = f;
            break;
        }
    }
    if (!format) {
        DocumentFormatConstraints sc;
        sc.supportedObjectTypes.append(GObjectTypes::DNA_SEQUENCE);
        foreach(DocumentFormat* f, fs) {
            if (f->checkConstraints(sc)) {
                format = f;
                break;
            }
        }
    }
    if (!format) {
        stateInfo.setError(  tr("Unsupported document format") );
        return;
    }
    log.info(tr("Reading MSA from %1 [%2]").arg(url).arg(format->getFormatName()));
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
    Document *doc = format->loadExistingDocument(iof, url, stateInfo, QVariantMap());
    assert(isCanceled() || doc!=NULL || hasErrors());
    assert(doc == NULL || doc->isLoaded());
    if (!isCanceled() && doc && doc->isLoaded()) {
        if (format->checkConstraints(mc)) {
            foreach(GObject* go, doc->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT)) {
                results.append(((MAlignmentObject*)go)->getMAlignment());
            }
        } else {
            QString err;
            MAlignment ma = MSAUtils::seq2ma(doc->findGObjectByType(GObjectTypes::DNA_SEQUENCE), err);
            if (err.isEmpty()) {
                results.append(ma);
            } else {
                setError(err);
            }
        }
    }
    if (doc && doc->isLoaded()) {
        doc->unload();
    }
}

void GenericSeqReader::init() {
    GenericMSAReader::init();
    mtype = WorkflowEnv::getDataTypeRegistry()->getById(GenericSeqActorProto::TYPE);
    GenericSeqActorProto::Mode mode = GenericSeqActorProto::Mode(actor->getParameter(GenericSeqActorProto::MODE_ATTR)->value.toInt());
    if (GenericSeqActorProto::MERGE == mode) {
        QString mergeToken = MERGE_MULTI_DOC_GAP_SIZE_SETTINGS;
        cfg[mergeToken] = actor->getParameter(GenericSeqActorProto::GAP_ATTR)->value.toInt();
    }
    selector.acc = actor->getParameter(GenericSeqActorProto::ACC_ATTR)->value.toString();
}

void GenericSeqReader::sl_taskFinished() {
    LoadSeqTask* t = qobject_cast<LoadSeqTask*>(sender());
    if (!t->isFinished() || t->hasErrors()) {
        return;
    }
    foreach(const QVariantMap& m, t->results) {
        cache.append(Message(mtype, m));
    }
}

void LoadSeqTask::run() {
    DocumentFormat* format = NULL;
    QList<DocumentFormat*> fs = DocumentFormatUtils::detectFormat(url);
    DocumentFormatConstraints constraints;
    constraints.supportedObjectTypes.append(GObjectTypes::DNA_SEQUENCE);
    DocumentFormatConstraints mc;
    mc.supportedObjectTypes.append(GObjectTypes::MULTIPLE_ALIGNMENT);
    foreach(DocumentFormat* f, fs) {
        if (f->checkConstraints(mc)) {
            format = f;
            break;
        }
    }
    if (!format) {
        foreach(DocumentFormat* f, fs) {
            if (f->checkConstraints(constraints)) {
                format = f;
                break;
            }
        }
    }
    if (!format) {
        stateInfo.setError(  tr("Unsupported document format") );
        return;
    }
    log.info(tr("Reading sequences from %1 [%2]").arg(url).arg(format->getFormatName()));
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
    Document *doc = format->loadExistingDocument(iof, url, stateInfo, cfg);
    assert(isCanceled() || doc!=NULL || hasErrors());
    assert(doc == NULL || doc->isLoaded());
    if (!isCanceled() && doc && doc->isLoaded()) {
        if (format->checkConstraints(constraints)) {
            foreach(GObject* go, doc->findGObjectByType(GObjectTypes::DNA_SEQUENCE)) {
                const DNASequence& dna = ((DNASequenceObject*)go)->getDNASequence();
                if (!selector->matches(dna)) {
                    continue;
                }
                QVariantMap m;
                m.insert(CoreLib::URL_SLOT_ID, url);
                m.insert(BioActorLibrary::SEQ_SLOT_ID, qVariantFromValue<DNASequence>(dna));
                QList<GObject*> allLoadedAnnotations = doc->findGObjectByType(GObjectTypes::ANNOTATION_TABLE);
                QList<GObject*> annotations = GObjectUtils::findObjectsRelatedToObjectByRole(go, 
                    GObjectTypes::ANNOTATION_TABLE, GObjectRelationRole::SEQUENCE, 
                    allLoadedAnnotations, UOF_LoadedOnly);
                if (!annotations.isEmpty()) {
                    QList<SharedAnnotationData> l;
                    AnnotationTableObject* att = qobject_cast<AnnotationTableObject*>(annotations.first());
                    foreach(Annotation* a, att->getAnnotations()) {
                        l << a->data();
                    }
                    m.insert(BioActorLibrary::FEATURE_TABLE_SLOT_ID, qVariantFromValue<QList<SharedAnnotationData> >(l));
                }

                results.append(m);
            }
        } else {
            //TODO merge seqs from alignment
//             QString mergeToken = MERGE_MULTI_DOC_GAP_SIZE_SETTINGS;
//             bool merge = cfg.contains(mergeToken);
//             int gaps = cfg.value(mergeToken).toInt();
            foreach(GObject* go, doc->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT)) {
                foreach(const DNASequence& s, MSAUtils::ma2seq(((MAlignmentObject*)go)->getMAlignment(), false)) {
                    if (!selector->matches(s)) {
                        continue;
                    }
                    QVariantMap m;
                    m.insert(CoreLib::URL_SLOT_ID, url);
                    m.insert(BioActorLibrary::SEQ_SLOT_ID, qVariantFromValue<DNASequence>(s));
                    results.append(m);
                }
            }
        }
    }
    if (doc && doc->isLoaded()) {
        doc->unload();
    }
}


bool DNASelector::matches( const DNASequence& dna)
{
    if (acc.isEmpty()) {
        return true;
    }
    if (dna.info.contains(DNAInfo::ACCESSION)) {
        return dna.info.value(DNAInfo::ACCESSION).toStringList().contains(acc);
    }
    return acc == dna.getName();
}

} // Workflow namespace
} // GB2 namespace
