/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <QtCore/QByteArray>

#include <core_api/DNAAlphabet.h>
#include <core_api/DNATranslation.h>
#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/Counter.h>
#include <util_gui/GUIUtils.h>
#include <gobjects/DNASequenceObject.h>

#include <gobject/uHMMObject.h>
#include <format/uHMMFormat.h>
#include <task_local_storage/uHMMSearchTaskLocalStorage.h>

#include "uHMM3SearhTask.h"

using namespace GB2;

namespace GB2 {

/*****************************************************
* UHMM3SWSearchTask
*****************************************************/

static P7_HMM* getHmmFromDocument( Document* doc, TaskStateInfo& ti ) {
    assert( NULL != doc );

    if( doc->getObjects().isEmpty() ) {
        ti.setError( UHMM3SWSearchTask::tr( "no_hmm_found_in_file" ) );
        return NULL;
    } else {
        UHMMObject* obj = qobject_cast< UHMMObject* >( doc->getObjects().first() );
        if( NULL == obj ) {
            ti.setError( UHMM3SWSearchTask::tr( "cannot_cast_to_hmm_object" ) );
            return NULL;
        }
        return (P7_HMM*)obj->getHMM();
    }
}

static void recountRegion( LRegion& region, bool isAmino, bool isCompl, LRegion globalR ) {
    int len = isAmino? region.len * 3 : region.len;
    int start = isAmino? region.startPos * 3 : region.startPos;

    if( isCompl ) {
        start = globalR.len - start - len;
    }
    region.startPos = globalR.startPos + start;
    region.len = len;
}

static void recountUHMM3SWSearchRegions( UHMM3SearchSeqDomainResult& res, bool isAmino, bool isCompl, LRegion globalR ) {
    recountRegion( res.seqRegion, isAmino, isCompl, globalR );
    recountRegion( res.envRegion, isAmino, isCompl, globalR );
}

UHMM3SWSearchTask::UHMM3SWSearchTask( const P7_HMM* h, const DNASequence& s, const UHMM3SearchTaskSettings& set )
: Task( "", TaskFlag_NoRun ), hmm( h ), sequence( s ), settings( set ), 
  complTranslation( NULL ), aminoTranslation( NULL ), swTask( NULL ), loadHmmTask( NULL ) {
    GCOUNTER( cvar, tvar, "UHMM3SWSearchTask" );
    
    if( NULL == hmm ) {
        setTaskName( tr( "Sequence_walker_HMM_search_task" ) );
        stateInfo.setError( Translations::badArgument( "hmm" ) );
        return;
    }
    assert( NULL != hmm->name );
    setTaskName( tr( "Sequence_walker_hmm_search_with_'%1'" ).arg( hmm->name ) );
    
    if( !sequence.seq.length() ) {
        stateInfo.setError( Translations::badArgument( "sequence" ) );
        return;
    }
}

UHMM3SWSearchTask::UHMM3SWSearchTask( const QString& hF, const DNASequence& seq, const UHMM3SearchTaskSettings& s)
: Task( "", TaskFlag_NoRun ), hmm( NULL ), sequence( seq ), settings( s ), 
  complTranslation( NULL ), aminoTranslation( NULL ), swTask( NULL ), loadHmmTask( NULL ), hmmFilename( hF ) {
    
    if( hmmFilename.isEmpty() ) {
        setTaskName( tr( "Sequence_walker_HMM_search_task" ) );
        stateInfo.setError( Translations::badArgument( "hmm_filename" ) );
        return;
    }
    setTaskName( tr( "Sequence_walker_hmm_search_with_'%1'" ).arg( hmmFilename ) );
    
    if( !sequence.seq.length() ) {
        stateInfo.setError( Translations::badArgument( "sequence" ) );
        return;
    }
}

SequenceWalkerTask* UHMM3SWSearchTask::getSWSubtask() {
    assert( !hasErrors() );
    assert( NULL != hmm );
    
    bool ok = checkAlphabets( hmm->abc->type, sequence.alphabet );
    if( !ok ) {
        assert( hasErrors() );
        return NULL;
    }
    ok = setTranslations( hmm->abc->type, sequence.alphabet );
    if( !ok ) {
        assert( hasErrors() );
        return NULL;
    }
    
    SequenceWalkerConfig config;
    config.seq          = sequence.seq.data();
    config.seqSize      = sequence.seq.size();
    config.complTrans   = complTranslation;
    config.aminoTrans   = aminoTranslation;
    config.overlapSize  = 0;
    config.chunkSize    = config.seqSize;
    config.nThreads     = MAX_PARALLEL_SUBTASKS_AUTO;
    
    return new SequenceWalkerTask( config, this, tr( "sequence_walker_hmmer3_search_task" ) );
}

void UHMM3SWSearchTask::prepare() {
    if( hasErrors() ) {
        return;
    }
    
    if( NULL != hmm ) {
        swTask = getSWSubtask();
        if( NULL == swTask ) {
            assert( hasErrors() );
            return;
        }
        addSubTask( swTask );
    } else {
        IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById( BaseIOAdapters::url2io( hmmFilename ) );
        assert( NULL != iof );
        loadHmmTask = new LoadDocumentTask( UHMMFormat::UHHMER_FORMAT_ID, hmmFilename, iof, QVariantMap() );
        addSubTask( loadHmmTask );
    }
}

QList< Task* > UHMM3SWSearchTask::onSubTaskFinished( Task* subTask ) {
    assert( NULL != subTask );
    QList< Task* > res;
    if( subTask->hasErrors() ) {
        stateInfo.setError(subTask->getError());
        return res;
    }
    
    if( loadHmmTask == subTask ) {
        hmm = getHmmFromDocument( loadHmmTask->getDocument(), stateInfo );
        swTask = getSWSubtask();
        if( NULL == swTask ) {
            assert( hasErrors() );
            return res;
        }
        res << swTask;
    } else {
        if( swTask != subTask ) {
            assert( 0 && "undefined_subtask_finished" );
        }
    }
    
    return res;
}

void UHMM3SWSearchTask::onRegion( SequenceWalkerSubtask* t, TaskStateInfo& ti ) {
    assert( NULL != t );
    if( stateInfo.hasErrors() || ti.hasErrors() ) {
        return;
    }
    
    const char* seq = t->getRegionSequence();
    int seqLen      = t->getRegionSequenceLen();
    bool isComl     = t->isDNAComplemented();
    bool isAmino    = t->isAminoTranslated();
    LRegion globalRegion = t->getGlobalRegion();
    
    UHMM3SearchTaskLocalStorage::createTaskContext( t->getTaskId() );
    UHMM3SearchResult generalRes = UHMM3Search::search( hmm, seq, seqLen, settings.inner, ti );
    if( ti.hasErrors() ) {
        UHMM3SearchTaskLocalStorage::freeTaskContext( t->getTaskId() );
        return;
    }
    
    writeResultsMtx.lock();
    
    foreach( const UHMM3SearchSeqDomainResult& domainRes, generalRes.domainResList ) {
        UHMM3SWSearchTaskDomainResult res;
        res.generalResult = domainRes;
        res.onCompl = isComl;
        res.onAmino = isAmino;
        recountUHMM3SWSearchRegions( res.generalResult, isAmino, isComl, globalRegion );
        results.append( res );
    }
    
    writeResultsMtx.unlock();
    UHMM3SearchTaskLocalStorage::freeTaskContext( t->getTaskId() );
}

Task::ReportResult UHMM3SWSearchTask::report() {
    /* post process overlaps filter here - we don't need it yet */
    return ReportResult_Finished;
}

bool UHMM3SWSearchTask::checkAlphabets( int hmmAl, DNAAlphabet* seqAl ) {
    assert( !hasErrors() );
    assert( NULL != seqAl );
    assert( 0 <= hmmAl );
    
    if( eslUNKNOWN == hmmAl || eslNONSTANDARD == hmmAl ) {
        stateInfo.setError( tr( "unknown_alphabet_type" ) );
        return false;
    }
    if( seqAl->isRaw() ) {
        stateInfo.setError( tr( "invalid_sequence_alphabet_type" ) );
        return false;
    }
    
    if( eslDNA == hmmAl || eslRNA == hmmAl ) {
        if( seqAl->isAmino() ) {
            stateInfo.setError( tr( "cannot_search_for_nucleic_hmm_in_amino_sequence" ) );
            return false;
        }
    }
    return true;
}

bool UHMM3SWSearchTask::setTranslations( int hmmAl, DNAAlphabet* seqAl ) {
    assert( !hasErrors() );
    assert( NULL != seqAl );
    assert( 0 <= hmmAl );
    
    if( seqAl->isNucleic() ) {
        DNATranslationRegistry* transReg = AppContext::getDNATranslationRegistry();
        assert( NULL != transReg );
        QList< DNATranslation* > complTs = transReg->lookupTranslation( seqAl, DNATranslationType_NUCL_2_COMPLNUCL );
        if (!complTs.empty()) {
            complTranslation = complTs.first();
        }
        if( hmmAl == eslAMINO ) {
            QList< DNATranslation* > aminoTs = transReg->lookupTranslation( seqAl, DNATranslationType_NUCL_2_AMINO );
            if( !aminoTs.empty() ) {
                aminoTranslation = aminoTs.first();
            }
        }
    } else {
        if( !seqAl->isAmino() ) {
            stateInfo.setError( "unrecognized_sequence_alphabet_found" );
            return false;
        }
    }
    
    return true;
}

UHMM3SWSearchTaskResult UHMM3SWSearchTask::getResults() const {
    return results;
}

QList< SharedAnnotationData > UHMM3SWSearchTask::getResultsAsAnnotations( const QString & name ) const {
    assert( !name.isEmpty() );
    QList< SharedAnnotationData > annotations;
    
    foreach( const UHMM3SWSearchTaskDomainResult & res, results ) {
        AnnotationData * annData = new AnnotationData();
        annData->name = name;
        annData->complement = res.onCompl;
        annData->aminoStrand = res.onAmino ? TriState_Yes :TriState_No;
        annData->location << res.generalResult.seqRegion;
        
        assert( NULL != hmm );
        QString hmmInfo = hmm->name;
        if( NULL != hmm->acc ) {
            hmmInfo += QString().sprintf( "\n Accession number in PFAM database: %s", hmm->acc );
        }
        if( NULL != hmm->desc ) {
            hmmInfo += QString().sprintf( "\n Description: %s", hmm->desc );
        }
        assert( !hmmInfo.isEmpty() );
        annData->qualifiers << Qualifier( "HMM model", hmmInfo );
        res.generalResult.writeQualifiersToAnnotation( annData );
        
        annotations << SharedAnnotationData( annData );
    }
    
    return annotations;
}

/*****************************************************
 * UHMM3SearchTaskSettings
 *****************************************************/

UHMM3SearchTaskSettings::UHMM3SearchTaskSettings() {
    setDefaultUHMM3SearchSettings( &inner );
}

/*****************************************************
* UHMM3SearchTask
*****************************************************/

UHMM3SearchTask::UHMM3SearchTask( const UHMM3SearchTaskSettings& s, P7_HMM* h, const char* se, int l )
: Task( "", TaskFlag_None ), settings( s ), hmm( h ), seq( se ), seqLen( l ) {
    if( NULL == hmm ) {
        setTaskName( tr( "HMM search task" ) );
        stateInfo.setError( tr( "no_hmm_given" ) );
        return;
    }
    assert( NULL != hmm->name );
    setTaskName( tr( "HMM search with '%1'" ).arg( hmm->name ) );
    
    if( NULL == seq || ( 0 >= seqLen ) ) {
        stateInfo.setError( tr( "empty_sequence_given" ) );
        return;
    }
}

UHMM3SearchTask::UHMM3SearchTask( const UHMM3SearchTaskSettings& s, const QString& hf, const char* se, int l )
: Task( "", TaskFlag_None ), settings( s ), hmm( NULL ), seq( se ), seqLen( l ) {
    const QString& hmmFilename = hf;
    if( hmmFilename.isEmpty() ) {
        setTaskName( tr( "HMM search task" ) );
        stateInfo.setError( tr( "no_hmm_file_given" ) );
        return;
    }
    setTaskName( tr( "HMM search with '%1' file" ).arg( hmmFilename ) );
    
    if( NULL == seq || ( 0 >= seqLen ) ) {
        stateInfo.setError( tr( "empty_sequence_given" ) );
        return;
    }
    
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById( BaseIOAdapters::url2io( hmmFilename ) );
    assert( NULL != iof );
    
    loadHmmTask = new LoadDocumentTask( UHMMFormat::UHHMER_FORMAT_ID, hmmFilename, iof, QVariantMap() );
    addSubTask( loadHmmTask );
}

QList< Task* > UHMM3SearchTask::onSubTaskFinished( Task* subTask ) {
    assert( NULL != subTask );
    QList< Task* > res;
    if( subTask->hasErrors() || subTask->isCanceled() ) {
        stateInfo.setError( subTask->getError() );
        return res;
    }
    
    if( loadHmmTask == subTask ) {
        hmm = getHmmFromDocument( loadHmmTask->getDocument(), stateInfo );
    } else {
        assert( 0 && "undefined_subtask_finished" );
    }
    return res;
}

UHMM3SearchResult UHMM3SearchTask::getResult() const {
    return result;
}

void UHMM3SearchTask::run() {
    if( stateInfo.hasErrors() ) {
        return;
    }
    assert( NULL != hmm );
    
    UHMM3SearchTaskLocalStorage::createTaskContext( getTaskId() );
    result = UHMM3Search::search( hmm, seq, seqLen, settings.inner, stateInfo );
    UHMM3SearchTaskLocalStorage::freeTaskContext( getTaskId() );
}

/*****************************************************
* UHMM3SWSearchToAnnotationsTask
*****************************************************/

void UHMM3SWSearchToAnnotationsTask::checkArgs() {
    if( hmmfile.isEmpty() ) {
        stateInfo.setError( Translations::badArgument( tr("hmm profile filename") ) );
        return;
    }
    if( NULL == annotationObj.data() ) {
        stateInfo.setError( Translations::badArgument( tr("annotation object") ) );
        return;
    }
    if( agroup.isEmpty() ) {
        stateInfo.setError( Translations::badArgument( tr( "annotations group name" ) ) );
        return;
    }
    if( aname.isEmpty() ) {
        stateInfo.setError( Translations::badArgument( tr( "annotations name" ) ) );
        return;
    }
}

UHMM3SWSearchToAnnotationsTask::UHMM3SWSearchToAnnotationsTask( const QString & hmmf, const DNASequence & s,
                                                                AnnotationTableObject * o, const QString & gr,
                                                                const QString & name, const UHMM3SearchTaskSettings & set )
: Task( "", TaskFlags_NR_FOSCOE | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled ),
hmmfile( hmmf ), sequence( s ), annotationObj( o ), agroup( gr ), aname( name ), searchSettings( set ),
loadSequenceTask( NULL ), searchTask( NULL ), createAnnotationsTask( NULL ) {
    
    setTaskName( tr( "HMMER3 search task" ) );
    checkArgs();
    if( sequence.isNull() ) {
        stateInfo.setError( Translations::badArgument( tr("dna sequence" ) ) );
    }
    if( stateInfo.hasErrors() ) {
        return;
    }
    setTaskName( tr( "HMMER3 search task with '%1' profile" ).arg( hmmfile ) );
    
    searchTask = new UHMM3SWSearchTask( hmmfile, sequence, searchSettings );
    addSubTask( searchTask );
}

UHMM3SWSearchToAnnotationsTask::UHMM3SWSearchToAnnotationsTask( const QString & hmmf, const QString & seqFile,
                                                                AnnotationTableObject * obj, const QString & gr,
                                                                const QString & name,
                                                                const UHMM3SearchTaskSettings & set )
: Task( "", TaskFlags_NR_FOSCOE | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled ), 
hmmfile( hmmf ), annotationObj( obj ), agroup( gr ), aname( name ), searchSettings( set ),
loadSequenceTask( NULL ), searchTask( NULL ), createAnnotationsTask( NULL ) {
    
    setTaskName( tr( "HMMER3 search task" ) );
    checkArgs();
    if( seqFile.isEmpty() ) {
        stateInfo.setError( Translations::badArgument( tr( "Sequence file" ) ) );
    }
    if( stateInfo.hasErrors() ) {
        return;
    }
    setTaskName( tr( "HMMER3 search task with '%1' profile" ).arg( hmmfile ) );
    
    loadSequenceTask = LoadDocumentTask::getDefaultLoadDocTask( seqFile );
    if( NULL == loadSequenceTask ) {
        stateInfo.setError( Translations::errorOpeningFileRead( seqFile ) );
        return;
    } else {
        addSubTask( loadSequenceTask );
    }
}

QString UHMM3SWSearchToAnnotationsTask::generateReport() const {
    QString res;
    res += "<table>";
    res+="<tr><td width=200><b>" + tr("HMM profile used") + "</b></td><td>" + QFileInfo( hmmfile ).absoluteFilePath() + "</td></tr>";
    
    if( hasErrors() || isCanceled() ) {
        res += "<tr><td width=200><b>" + tr("Task was not finished") + "</b></td><td></td></tr>";
        res += "</table>";
        return res;
    }
    
    res += "<tr><td><b>" + tr("Result annotation table") + "</b></td><td>" + annotationObj->getDocument()->getName() + "</td></tr>";
    res += "<tr><td><b>" + tr("Result annotation group") + "</b></td><td>" + agroup + "</td></tr>";
    res += "<tr><td><b>" + tr("Result annotation name") +  "</b></td><td>" + aname + "</td></tr>";
    
    int nResults = createAnnotationsTask == NULL ? 0 : createAnnotationsTask->getAnnotations().size();
    res += "<tr><td><b>" + tr("Results count") +  "</b></td><td>" + QString::number( nResults )+ "</td></tr>";
    res += "</table>";
    return res;
}

void UHMM3SWSearchToAnnotationsTask::setSequence() {
    assert( NULL != loadSequenceTask );
    
    Document * seqDoc = loadSequenceTask->getDocument();
    if( NULL == seqDoc ) {
        stateInfo.setError( tr( "Cannot load sequence document" ) );
        return;
    }
    QList< GObject* > objs = seqDoc->findGObjectByType( GObjectTypes::DNA_SEQUENCE );
    if( objs.isEmpty() ) {
        stateInfo.setError( tr( "No sequence objects loaded" ) );
        return;
    }
    DNASequenceObject * seqObj = qobject_cast< DNASequenceObject* >( objs.first() );
    if( NULL == seqObj ) {
        stateInfo.setError( tr( "Unknown sequence type loaded" ) );
        return;
    }
    
    sequence = seqObj->getDNASequence();
    if( sequence.isNull() ) {
        stateInfo.setError( tr( "Empty sequence loaded" ) );
        return;
    }
}

QList< Task* > UHMM3SWSearchToAnnotationsTask::onSubTaskFinished( Task * subTask ) {
    QMutexLocker locker( &mtx );
    QList< Task* > res;
    if( hasErrors() ) {
        return res;
    }
    assert( NULL != subTask );
    if( subTask->hasErrors() ) {
        stateInfo.setError( subTask->getError() );
        return res;
    }
    
    if( annotationObj.isNull() ) {
        stateInfo.setError( tr( "Annotation object removed" ) );
        return res;
    }
    
    if( loadSequenceTask == subTask ) {
        setSequence();
        if( hasErrors() ) {
            return res;
        }
        searchTask = new UHMM3SWSearchTask( hmmfile, sequence, searchSettings );
        res << searchTask;
    } else if( searchTask == subTask ) {
        QList< SharedAnnotationData > annotations = searchTask->getResultsAsAnnotations( aname );
        if( annotations.isEmpty() ) {
            return res;
        }
        
        createAnnotationsTask = new CreateAnnotationsTask( annotationObj, agroup, annotations );
        res << createAnnotationsTask;
    } else if( createAnnotationsTask != subTask ) {
        assert( false && "undefined task finished" );
    }
    
    return res;
}

} // GB2
