/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <cstdlib>
#include <core_api/AppContext.h>
#include <core_api/Counter.h>
#include <util_tasks/CreateAnnotationTask.h>
#include "primer3_main.h"
#include "boulder_input.h"
#include "Primer3Task.h"

namespace GB2 {

// Primer

Primer::Primer():
    start(0),
    length(0),
    meltingTemperature(0),
    gcContent(0),
    selfAny(0),
    selfEnd(0),
    endStability(0)
{
}

Primer::Primer(const primer_rec &primerRec):
    start(primerRec.start),
    length(primerRec.length),
    meltingTemperature(primerRec.temp),
    gcContent(primerRec.gc_content),
    selfAny(primerRec.self_any),
    selfEnd(primerRec.self_end),
    endStability(primerRec.end_stability)
{
}

int Primer::getStart()const
{
    return start;
}

int Primer::getLength()const
{
    return length;
}

double Primer::getMeltingTemperature()const
{
    return meltingTemperature;
}

double Primer::getGcContent()const
{
    return gcContent;
}

short Primer::getSelfAny()const
{
    return selfAny;
}

short Primer::getSelfEnd()const
{
    return selfEnd;
}

double Primer::getEndStabilyty()const
{
    return endStability;
}

void Primer::setStart(int start)
{
    this->start = start;
}

void Primer::setLength(int length)
{
    this->length = length;
}

void Primer::setMeltingTemperature(double meltingTemperature)
{
    this->meltingTemperature = meltingTemperature;
}

void Primer::setGcContent(double gcContent)
{
    this->gcContent = gcContent;
}

void Primer::setSelfAny(short selfAny)
{
    this->selfAny = selfAny;
}

void Primer::setSelfEnd(short selfEnd)
{
    this->selfEnd = selfEnd;
}

void Primer::setEndStability(double endStability)
{
    this->endStability = endStability;
}

// PrimerPair

PrimerPair::PrimerPair():
    leftPrimer(NULL),
    rightPrimer(NULL),
    internalOligo(NULL),
    complAny(0),
    complEnd(0),
    productSize(0)
{
}

PrimerPair::PrimerPair(const primer_pair &primerPair):
    leftPrimer((NULL == primerPair.left)? NULL:new Primer(*primerPair.left)),
    rightPrimer((NULL == primerPair.right)? NULL:new Primer(*primerPair.right)),
    internalOligo((NULL == primerPair.intl)? NULL:new Primer(*primerPair.intl)),
    complAny(primerPair.compl_any),
    complEnd(primerPair.compl_end),
    productSize(primerPair.product_size)
{
}

PrimerPair::PrimerPair(const PrimerPair &primerPair):
        leftPrimer((NULL == primerPair.leftPrimer.get())? NULL:new Primer(*primerPair.leftPrimer)),
        rightPrimer((NULL == primerPair.rightPrimer.get())? NULL:new Primer(*primerPair.rightPrimer)),
        internalOligo((NULL == primerPair.internalOligo.get())? NULL:new Primer(*primerPair.internalOligo)),
        complAny(primerPair.complAny),
        complEnd(primerPair.complEnd),
        productSize(primerPair.productSize)
{
}

const PrimerPair &PrimerPair::operator=(const PrimerPair &primerPair)
{
    leftPrimer.reset((NULL == primerPair.leftPrimer.get())? NULL:new Primer(*primerPair.leftPrimer));
    rightPrimer.reset((NULL == primerPair.rightPrimer.get())? NULL:new Primer(*primerPair.rightPrimer));
    internalOligo.reset((NULL == primerPair.internalOligo.get())? NULL:new Primer(*primerPair.internalOligo));
    complAny = primerPair.complAny;
    complEnd = primerPair.complEnd;
    productSize = primerPair.productSize;
    return *this;
}

Primer *PrimerPair::getLeftPrimer()const
{
    return leftPrimer.get();
}

Primer *PrimerPair::getRightPrimer()const
{
    return rightPrimer.get();
}

Primer *PrimerPair::getInternalOligo()const
{
    return internalOligo.get();
}

short PrimerPair::getComplAny()const
{
    return complAny;
}

short PrimerPair::getComplEnd()const
{
    return complEnd;
}

int PrimerPair::getProductSize()const
{
    return productSize;
}

void PrimerPair::setLeftPrimer(Primer *leftPrimer)
{
    this->leftPrimer.reset((NULL == leftPrimer)? NULL:new Primer(*leftPrimer));
}

void PrimerPair::setRightPrimer(Primer *rightPrimer)
{
    this->rightPrimer.reset((NULL == rightPrimer)? NULL:new Primer(*rightPrimer));
}

void PrimerPair::setInternalOligo(Primer *internalOligo)
{
    this->internalOligo.reset((NULL == internalOligo)? NULL:new Primer(*internalOligo));
}

void PrimerPair::setComplAny(short complAny)
{
    this->complAny = complAny;
}

void PrimerPair::setComplEnd(short complEnd)
{
    this->complEnd = complEnd;
}

void PrimerPair::setProductSize(int productSize)
{
    this->productSize = productSize;
}

// Primer3Task

Primer3Task::Primer3Task(const Primer3TaskSettings &settingsArg):
    Task(tr("Pick primers task"), TaskFlag_None),
    settings(settingsArg)
{
    GCOUNTER( cvar, tvar, "Primer3Task" );
    {
        QPair<int, int> region = settings.getIncludedRegion();
        region.first -= settings.getFirstBaseIndex();
        settings.setIncludedRegion(region);
    }
    if(!PR_START_CODON_POS_IS_NULL(settings.getSeqArgs()))
    {
        int startCodonPosition = PR_DEFAULT_START_CODON_POS;
        if(settings.getIntProperty("PRIMER_START_CODON_POSITION", &startCodonPosition))
        {
            settings.setIntProperty("PRIMER_START_CODON_POSITION", startCodonPosition - settings.getFirstBaseIndex());
        }
    }
    {
        QList<QPair<int, int> > regionList = settings.getTarget();
        for(int i=0;i < regionList.size();i++)
        {
            regionList[i].first -= settings.getFirstBaseIndex();
        }
        settings.setTarget(regionList);
    }
    {
        QList<QPair<int, int> > regionList = settings.getExcludedRegion();
        for(int i=0;i < regionList.size();i++)
        {
            regionList[i].first -= settings.getFirstBaseIndex();
        }
        settings.setExcludedRegion(regionList);
    }
    {
        QList<QPair<int, int> > regionList = settings.getInternalOligoExcludedRegion();
        for(int i=0;i < regionList.size();i++)
        {
            regionList[i].first -= settings.getFirstBaseIndex();
        }
        settings.setInternalOligoExcludedRegion(regionList);
    }
}

void Primer3Task::run()
{
    if(!settings.getRepeatLibrary().isEmpty())
    {
        read_seq_lib(&settings.getPrimerArgs()->repeat_lib, settings.getRepeatLibrary().constData(), "mispriming library");
        if(NULL != settings.getPrimerArgs()->repeat_lib.error.data)
        {
            pr_append_new_chunk(&settings.getPrimerArgs()->glob_err, settings.getPrimerArgs()->repeat_lib.error.data);
            pr_append_new_chunk(&settings.getSeqArgs()->error, settings.getPrimerArgs()->repeat_lib.error.data);
            return;
        }
    }
    if(!settings.getMishybLibrary().isEmpty())
    {
        read_seq_lib(&settings.getPrimerArgs()->io_mishyb_library, settings.getMishybLibrary().constData(), "internal oligo mishyb library");
        if(NULL != settings.getPrimerArgs()->io_mishyb_library.error.data)
        {
            pr_append_new_chunk(&settings.getPrimerArgs()->glob_err, settings.getPrimerArgs()->io_mishyb_library.error.data);
            pr_append_new_chunk(&settings.getSeqArgs()->error, settings.getPrimerArgs()->io_mishyb_library.error.data);
            return;
        }
    }
    primers_t primers = runPrimer3(settings.getPrimerArgs(), settings.getSeqArgs(), &stateInfo.cancelFlag, &stateInfo.progress);

    bestPairs.clear();
    for(int index = 0;index < primers.best_pairs.num_pairs;index++)
    {
        bestPairs.append(PrimerPair(primers.best_pairs.pairs[index]));
    }

    if(primers.best_pairs.num_pairs > 0)
    {
        std::free(primers.best_pairs.pairs);
        primers.best_pairs.pairs = NULL;
    }
    if(NULL != primers.left)
    {
        std::free(primers.left);
        primers.left = NULL;
    }
    if(NULL != primers.right)
    {
        std::free(primers.right);
        primers.right = NULL;
    }
    if(NULL != primers.intl)
    {
        std::free(primers.intl);
        primers.intl = NULL;
    }
}

Task::ReportResult Primer3Task::report()
{
    if(!settings.getError().isEmpty())
    {
        stateInfo.setError(settings.getError());
    }
    return Task::ReportResult_Finished;
}

QList<PrimerPair> Primer3Task::getBestPairs()const
{
    return bestPairs;
}

//////////////////////////////////////////////////////////////////////////
////Primer3ToAnnotationsTask

Primer3ToAnnotationsTask::Primer3ToAnnotationsTask( const Primer3TaskSettings &settings,
AnnotationTableObject * aobj_, const QString & groupName_, const QString & annName_ ) :
Task(tr("Search primers to annotations"), TaskFlags_NR_FOSCOE), settings(settings), aobj(aobj_), groupName(groupName_), annName(annName_)
{
}

void Primer3ToAnnotationsTask::prepare()
{
    searchTask = new Primer3Task(settings);
    addSubTask(searchTask);
}

Task::ReportResult Primer3ToAnnotationsTask::report()
{
    assert( searchTask );

    QList<PrimerPair> bestPairs = searchTask->getBestPairs();

    int index = 0;
    foreach(PrimerPair pair, bestPairs)
    {
        QList<SharedAnnotationData> annotations;
        if(NULL != pair.getLeftPrimer())
        {
            annotations.append(oligoToAnnotation("primer", *pair.getLeftPrimer(), false));
        }
        if(NULL != pair.getInternalOligo())
        {
            annotations.append(oligoToAnnotation("internalOligo", *pair.getInternalOligo(), false));
        }
        if(NULL != pair.getRightPrimer())
        {
            annotations.append(oligoToAnnotation("primer", *pair.getRightPrimer(), true));
        }
        AppContext::getTaskScheduler()->registerTopLevelTask(
                new CreateAnnotationsTask(aobj, groupName + "/pair " + QString::number(index + 1), annotations));
        index++;
    }
    return ReportResult_Finished;
}

SharedAnnotationData Primer3ToAnnotationsTask::oligoToAnnotation(QString title, const Primer &primer, bool complement)
{
    SharedAnnotationData annotationData(new AnnotationData());
    annotationData->name = title;
    int start = primer.getStart() + settings.getIncludedRegion().first - settings.getFirstBaseIndex();
    int length = primer.getLength();
    if(complement)
    {
        start -= length - 1;
    }
    annotationData->complement = complement;
    annotationData->location.append(LRegion(start, length));
    annotationData->qualifiers.append(Qualifier("tm", QString::number(primer.getMeltingTemperature())));
    annotationData->qualifiers.append(Qualifier("gc%", QString::number(primer.getGcContent())));
    annotationData->qualifiers.append(Qualifier("any", QString::number(0.01*primer.getSelfAny())));
    annotationData->qualifiers.append(Qualifier("3'", QString::number(0.01*primer.getSelfEnd())));
    return annotationData;
}

}
