/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SiteconWorkers.h"
#include "SiteconIOWorkers.h"
#include "SiteconPlugin.h"
#include "SiteconBuildDialogController.h"

#include <workflow/TypeSet.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>

#include <core_api/Log.h>
#include <datatype/MAlignment.h>

/* TRANSLATOR GB2::SiteconIO */
/* TRANSLATOR GB2::LocalWorkflow::SiteconBuildWorker */

namespace GB2 {
namespace LocalWorkflow {

static const QString IN_PORT("in");
static const QString OUT_PORT("out");
static const QString URL_SLOT("URL");

static const QString WINDOW_ATTR("1window");
static const QString LEN_ATTR("2len");
static const QString SEED_ATTR("3seed");
static const QString ALG_ATTR("4algo");

const QString SiteconBuildWorker::ACTOR_ID("sitecon.build");

static LogCategory log(ULOG_CAT_WD);

void SiteconBuildWorker::registerProto() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    QMap<Descriptor, DataTypePtr> m;
    Descriptor id(IN_PORT, SiteconBuildWorker::tr("Input alignment"), SiteconBuildWorker::tr("Input multiple sequence alignment for building statistical model."));
    Descriptor ud(URL_SLOT, SiteconBuildWorker::tr("Origin"), SiteconBuildWorker::tr("Location of input alignment, used as optional hint for model description."));
    m[ud] = CoreDataTypes::STRING_TYPE();
    m[id] = BioDataTypes::MULTIPLE_ALIGNMENT_TYPE();
    DataTypePtr t(new DataTypeSet(Descriptor("build.sitecon.content"), m));

    Descriptor od(OUT_PORT, SiteconBuildWorker::tr("Sitecon model"), SiteconBuildWorker::tr("Produced statistical model of specified TFBS data."));
    p << new PortDescriptor(id, t, true /*input*/);
    p << new PortDescriptor(od, SiteconWorkerFactory::SITECON_MODEL_TYPE(), false /*input*/, true /*multi*/);
    
    {
        Descriptor wd(WINDOW_ATTR, SiteconBuildWorker::tr("Window size, bp"), QApplication::translate("SiteconBuildDialog", "win_tip", 0, QApplication::UnicodeUTF8));
        Descriptor ld(LEN_ATTR, SiteconBuildWorker::tr("Calibration length"), QApplication::translate("SiteconBuildDialog", "seq_len_tip", 0, QApplication::UnicodeUTF8));
        Descriptor sd(SEED_ATTR, SiteconBuildWorker::tr("Random seed"), QApplication::translate("SiteconBuildDialog", "seed_tip", 0, QApplication::UnicodeUTF8));
        Descriptor ad(ALG_ATTR, SiteconBuildWorker::tr("Weight algorithm"), QApplication::translate("SiteconBuildDialog", "algo_tip", 0, QApplication::UnicodeUTF8));

        a << new Attribute(wd, CoreDataTypes::NUM_TYPE(), false, 40);
        a << new Attribute(ld, CoreDataTypes::NUM_TYPE(), false, 1000*1000);
        a << new Attribute(sd, CoreDataTypes::NUM_TYPE(), false, 0);
        a << new Attribute(ad, CoreDataTypes::BOOL_TYPE(), false, int(SiteconWeightAlg_None));
    }

    Descriptor desc(ACTOR_ID, tr("Build SITECON model"),
        tr("Builds statistical profile for SITECON. The SITECON is a program for probabilistic recognition of transcription factor binding sites."));
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;    
    
    {
        QVariantMap m2; m2["minimum"] = 0; m2["maximum"] = INT_MAX;
        delegates[SEED_ATTR] = new SpinBoxDelegate(m2);
    }
    {
        QVariantMap m2; m2["minimum"] = 1; m2["maximum"] = 1000;
        delegates[WINDOW_ATTR] = new SpinBoxDelegate(m2);
    }
    {
        QVariantMap modeMap; 
        modeMap["100K"] = 100*1000;
        modeMap["500K"] = 500*1000;
        modeMap["1M"] = 1000*1000;
        modeMap["5M"] = 5*1000*1000;
        delegates[LEN_ATTR] = new ComboBoxDelegate(modeMap);
    }
    {
        QVariantMap modeMap; 
        modeMap[tr("None")] = QVariant(SiteconWeightAlg_None);
        modeMap[tr("Algorithm2")] = QVariant(SiteconWeightAlg_Alg2);
        delegates[ALG_ATTR] = new ComboBoxDelegate(modeMap);
    }

    proto->setPrompter(new SiteconBuildPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":sitecon/images/sitecon.png");
    WorkflowEnv::getProtoRegistry()->registerProto(SiteconWorkerFactory::SITECON_CATEGORY(), proto);
}

QString SiteconBuildPrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(IN_PORT));
    Actor* msaProducer = input->getProducer(IN_PORT);

    QString msaName = msaProducer ? tr("For each MSA from <u>%1</u>,").arg(msaProducer->getLabel()) : "";
    QString doc = tr("%1 build SITECON model.")
        .arg(msaName);

    return doc;
}

void SiteconBuildWorker::init() {
    input = ports.value(IN_PORT);
    output = ports.value(OUT_PORT);

    cfg.props = SiteconPlugin::getDinucleotiteProperties();
    cfg.randomSeed = actor->getParameter(SEED_ATTR)->value.toInt();
    cfg.secondTypeErrorCalibrationLen = actor->getParameter(LEN_ATTR)->value.toInt();
    cfg.weightAlg = SiteconWeightAlg(actor->getParameter(ALG_ATTR)->value.toInt());
    cfg.windowSize = actor->getParameter(WINDOW_ATTR)->value.toInt();
    mtype = SiteconWorkerFactory::SITECON_MODEL_TYPE();
}

bool SiteconBuildWorker::isReady() {
    return (input && input->hasMessage());
}

Task* SiteconBuildWorker::tick() {
    QVariantMap data = input->get().getData().toMap();
    SiteconModel model = data.value(SiteconWorkerFactory::SITECON_MODEL_TYPE_ID).value<SiteconModel>();
    QString url = data.value(URL_SLOT).toString();

    const MAlignment& ma = data.value(IN_PORT).value<MAlignment>();
    Task* t = new SiteconBuildTask(cfg, ma, url);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void SiteconBuildWorker::sl_taskFinished() {
    SiteconBuildTask* t = qobject_cast<SiteconBuildTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    SiteconModel model = t->getResult();
    QVariant v = qVariantFromValue<SiteconModel>(model);
    output->put(Message(mtype, v));
    if (input->isEnded()) {
        output->setEnded();
    }
    log.info(tr("Built SITECON model from: %1").arg(model.aliURL));
}

bool SiteconBuildWorker::isDone() {
    return !input || input->isEnded();
}

} //namespace LocalWorkflow
} //namespace GB2
