/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*************************************************************************
 *
 *  File:       hw.c
 *  Content:    migration from frame buffer device driver's function.
 *
 ************************************************************************/
#include <math.h>
#include "via_driver.h"
#include "hw.h"
#include "drm.h"

/* Access I/O Function */
__inline__ void write_reg(CARD8 index, CARD16 io_port, CARD8 data);
void write_reg_mask(CARD8 index, int io_port, CARD8 data, CARD8 mask);
__inline__ CARD8 read_reg(int io_port, CARD8 index);
__inline__ void lock_crt(void);
__inline__ void unlock_crt(void);
int SearchModeSetting(int ModeInfoIndex);
int SearchLCDModeIndex(int ModeInfoIndex);
void load_fix_bit_crtc_reg(void);
void load_offset_reg(int h_addr, int bpp_byte, int set_iga);
void load_fetch_count_reg(int h_addr, int bpp_byte, int set_iga);
void load_FIFO_reg(VIABIOSInfoPtr pBIOSInfo, int hor_active, int ver_active, int set_iga);
CARD32 get_clk_value(VIABIOSInfoPtr pBIOSInfo, int clk);
void SetVCLK(VIABIOSInfoPtr pBIOSInfo, CARD32 CLK, int set_iga);
void set_sync_polarity(VIABIOSInfoPtr pBIOSInfo, int device, int hsync, int vsync);
void fill_crtc_timing(VIABIOSInfoPtr pBIOSInfo, struct crt_mode_table *crt_table, int mode_array, int bpp_byte, int device, int set_iga, Bool use_modeline);
void load_crtc_timing(VIABIOSInfoPtr pBIOSInfo, struct display_timing device_timing, int set_iga);
void load_reg(int timing_value, int load_reg_num, struct io_register *reg, int io_type);
void write_regx(struct io_reg RegTable[],int ItemNum);
void EnableSecondDisplayChannel();
void DisableSecondDisplayChannel();
void CalFromClock(VIABIOSInfoPtr pBIOSInfo, double dbDesiredClock, CARD32 * PLLValue);


__inline__ void write_reg(CARD8 index, CARD16 io_port, CARD8 data)
{
    MMIO_OUT8(MMIOMapBase, io_port, index);
    MMIO_OUT8(MMIOMapBase, io_port+1, data);
    DEBUG(ErrorF("Index=0x%2X, Value=0x%2X\n", index, data));
}

void write_reg_mask(CARD8 index, int io_port, CARD8 data, CARD8 mask)
{
    CARD8 tmp;

    MMIO_OUT8(MMIOMapBase, io_port, index);
    tmp = MMIO_IN8(MMIOMapBase, io_port+1);
    MMIO_OUT8(MMIOMapBase, io_port+1, (data & mask) | (tmp & (~mask)));
    DEBUG(ErrorF("Index=0x%2X, Value=0x%2X\n", index, ((data & mask) | (tmp & (~mask)))));
}

__inline__ CARD8 read_reg(int io_port, CARD8 index)
{
    MMIO_OUT8(MMIOMapBase, io_port, index);
    return(MMIO_IN8(MMIOMapBase, io_port+1));
}

__inline__ void lock_crt(void)
{
    write_reg_mask(CR11, VIACR, BIT7, BIT7);
}

__inline__ void unlock_crt(void)
{
    write_reg_mask(CR11, VIACR, 0, BIT7);
    write_reg_mask(CR47, VIACR, 0, BIT0);
}

/* Search Mode Index */
int SearchModeSetting( int ModeInfoIndex )
{
    int i=0;

    while ( ( i<NUM_TOTAL_MODETABLE ) && ( ModeInfoIndex != VIARoutineModes[i].ModeIndex) ){i++;}
    if( ( i >= NUM_TOTAL_MODETABLE ) ) i=0;
    return i;

}

/* Search LCD Mode Index */
int SearchLCDModeIndex(int ModeInfoIndex )
{
    int i=0;

    while ((i < NUM_TOTAL_LCD_MODETABLE) && (ModeInfoIndex != LCDModes[i].ModeIndex))
    {
        i++;
    }

    if (i >= NUM_TOTAL_LCD_MODETABLE)        
        i=0;
    
    return i;
}

void load_fix_bit_crtc_reg(void)
{
    DEBUG(ErrorF("VIA: load_fix_bit_crtc_reg\n"));

	unlock_crt();
	
    write_reg_mask(CR03, VIACR, 0x80, BIT7);               /* always set to 1 */
    write_reg(CR18, VIACR, 0xff);                          /* line compare should set all bits = 1 (extend modes) */
    write_reg_mask(CR07, VIACR, 0x10, BIT4);               /* line compare should set all bits = 1 (extend modes) */
    write_reg_mask(CR09, VIACR, 0x40, BIT6);               /* line compare should set all bits = 1 (extend modes) */
    write_reg_mask(CR35, VIACR, 0x10, BIT4);               /* line compare should set all bits = 1 (extend modes) */
    write_reg_mask(CR33, VIACR, 0x05, BIT0+BIT1+BIT2);     /* line compare should set all bits = 1 (extend modes) */
    /*write_reg_mask(CR32, VIACR, 0x01, BIT0);*/
    write_reg(CR17, VIACR, 0xe3);                          /* extend mode always set to e3h */
    write_reg(CR08, VIACR, 0x00);                          /* extend mode always set to 0h */
    write_reg(CR14, VIACR, 0x00);                          /* extend mode always set to 0h */

	lock_crt();
}

void load_offset_reg(int h_addr, int bpp_byte, int set_iga)
{
    int reg_value;
    int load_reg_num;
    struct io_register *reg;

    DEBUG(ErrorF("VIA: load_offset_reg\n"));
    switch(set_iga) {
        case IGA1_IGA2:
        case IGA1:
            reg_value = IGA1_OFFSET_FORMULA(h_addr, bpp_byte);
            load_reg_num = offset_reg.iga1_offset_reg.reg_num;
            reg = offset_reg.iga1_offset_reg.reg;
            load_reg(reg_value, load_reg_num, reg, VIACR);
            if (set_iga == IGA1)
                break;
        case IGA2:
            reg_value = IGA2_OFFSET_FORMULA(h_addr, bpp_byte);
            load_reg_num = offset_reg.iga2_offset_reg.reg_num;
            reg = offset_reg.iga2_offset_reg.reg;
            load_reg(reg_value, load_reg_num, reg, VIACR);
            break;
    }
}

void load_fetch_count_reg(int h_addr, int bpp_byte, int set_iga)
{
    int reg_value;
    int load_reg_num;
    struct io_register *reg=NULL;

    DEBUG(ErrorF("VIA: load_fetch_count_reg\n"));
    switch(set_iga) {
        case IGA1_IGA2:
        case IGA1:
            reg_value = IGA1_FETCH_COUNT_FORMULA(h_addr, bpp_byte);
            load_reg_num = fetch_count_reg.iga1_fetch_count_reg.reg_num;
            reg = fetch_count_reg.iga1_fetch_count_reg.reg;
            load_reg(reg_value, load_reg_num, reg, VIASR);
            if (set_iga == IGA1)
                break;
        case IGA2:
            reg_value = IGA2_FETCH_COUNT_FORMULA(h_addr, bpp_byte);
            load_reg_num = fetch_count_reg.iga2_fetch_count_reg.reg_num;
            reg = fetch_count_reg.iga2_fetch_count_reg.reg;
            load_reg(reg_value, load_reg_num, reg, VIACR);
            break;
    }
}

void load_FIFO_reg(VIABIOSInfoPtr pBIOSInfo, int hor_active, int ver_active, int set_iga)
{
    int reg_value;
    int load_reg_num;
    struct io_register *reg=NULL;
    int iga1_fifo_max_depth=0, iga1_fifo_threshold=0, iga1_fifo_high_threshold=0, iga1_display_queue_expire_num=0;
    int iga2_fifo_max_depth=0, iga2_fifo_threshold=0, iga2_fifo_high_threshold=0, iga2_display_queue_expire_num=0;


    DEBUG(ErrorF("VIA: load_FIFO_reg\n"));
    if (set_iga == IGA1)
    {
        switch (pBIOSInfo->Chipset)
        {
            case VIA_P4M800PRO:
            {
                iga1_fifo_max_depth = CN700_IGA1_FIFO_MAX_DEPTH;
                iga1_fifo_threshold = CN700_IGA1_FIFO_THRESHOLD;
                iga1_fifo_high_threshold = CN700_IGA1_FIFO_HIGH_THRESHOLD;

                /* If resolution > 1280x1024, expire length = 64, else expire length = 128 */
                if ((hor_active > 1280) && (ver_active > 1024))
                    iga1_display_queue_expire_num = 16;
                else
                    iga1_display_queue_expire_num = CN700_IGA1_DISPLAY_QUEUE_EXPIRE_NUM;

                break;
            }

            case VIA_CX700:
            {
                iga1_fifo_max_depth = CX700_IGA1_FIFO_MAX_DEPTH;
                iga1_fifo_threshold = CX700_IGA1_FIFO_THRESHOLD;
                iga1_fifo_high_threshold = CX700_IGA1_FIFO_HIGH_THRESHOLD;
                iga1_display_queue_expire_num = CX700_IGA1_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }
            
            case VIA_K8M890:
            {
                iga1_fifo_max_depth = K8M890_IGA1_FIFO_MAX_DEPTH;
                iga1_fifo_threshold = K8M890_IGA1_FIFO_THRESHOLD;
                iga1_fifo_high_threshold = K8M890_IGA1_FIFO_HIGH_THRESHOLD;
                iga1_display_queue_expire_num = K8M890_IGA1_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }         
            
            case VIA_P4M890:
            {
                iga1_fifo_max_depth = P4M890_IGA1_FIFO_MAX_DEPTH;
                iga1_fifo_threshold = P4M890_IGA1_FIFO_THRESHOLD;
                iga1_fifo_high_threshold = P4M890_IGA1_FIFO_HIGH_THRESHOLD;
                iga1_display_queue_expire_num = P4M890_IGA1_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }

            case VIA_P4M900:
            {
                iga1_fifo_max_depth = P4M900_IGA1_FIFO_MAX_DEPTH;
                iga1_fifo_threshold = P4M900_IGA1_FIFO_THRESHOLD;
                iga1_fifo_high_threshold = P4M900_IGA1_FIFO_HIGH_THRESHOLD;
                iga1_display_queue_expire_num = P4M900_IGA1_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }

            case VIA_VX800:
            {
                iga1_fifo_max_depth = VX800_IGA1_FIFO_MAX_DEPTH;
                iga1_fifo_threshold = VX800_IGA1_FIFO_THRESHOLD;
                iga1_fifo_high_threshold = VX800_IGA1_FIFO_HIGH_THRESHOLD;
                iga1_display_queue_expire_num = VX800_IGA1_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }
            case VIA_VX855:
            {
                iga1_fifo_max_depth = VX855_IGA1_FIFO_MAX_DEPTH;
                iga1_fifo_threshold = VX855_IGA1_FIFO_THRESHOLD;
                iga1_fifo_high_threshold = VX855_IGA1_FIFO_HIGH_THRESHOLD;
                iga1_display_queue_expire_num = VX855_IGA1_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }
        }
        
        /*Set Display FIFO Depath Select*/
        reg_value = IGA1_FIFO_DEPTH_SELECT_FORMULA(iga1_fifo_max_depth);
        load_reg_num = display_fifo_depth_reg.iga1_fifo_depth_select_reg.reg_num;
        reg = display_fifo_depth_reg.iga1_fifo_depth_select_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIASR);

        /*Set Display FIFO Threshold Select*/
        reg_value = IGA1_FIFO_THRESHOLD_FORMULA(iga1_fifo_threshold);
        load_reg_num = fifo_threshold_select_reg.iga1_fifo_threshold_select_reg.reg_num;
        reg = fifo_threshold_select_reg.iga1_fifo_threshold_select_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIASR);

        /*Set FIFO High Threshold Select*/
        reg_value = IGA1_FIFO_HIGH_THRESHOLD_FORMULA(iga1_fifo_high_threshold);
        load_reg_num = fifo_high_threshold_select_reg.iga1_fifo_high_threshold_select_reg.reg_num;
        reg = fifo_high_threshold_select_reg.iga1_fifo_high_threshold_select_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIASR);

        /*Set Display Gueue Expire Num*/
        reg_value = IGA1_DISPLAY_QUEUE_EXPIRE_NUM_FORMULA(iga1_display_queue_expire_num);
        load_reg_num = display_queue_expire_num_reg.iga1_display_queue_expire_num_reg.reg_num;
        reg = display_queue_expire_num_reg.iga1_display_queue_expire_num_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIASR);

    }
    else
    {
        switch (pBIOSInfo->Chipset)
        {
            case VIA_P4M800PRO:
            {
                iga2_fifo_max_depth = CN700_IGA2_FIFO_MAX_DEPTH;
                iga2_fifo_threshold = CN700_IGA2_FIFO_THRESHOLD;
                iga2_fifo_high_threshold = CN700_IGA2_FIFO_HIGH_THRESHOLD;

                /* If resolution > 1280x1024, expire length = 64, else expire length = 128 */
                if ((hor_active > 1280) && (ver_active > 1024))
                    iga2_display_queue_expire_num = 16;
                else
                    iga2_display_queue_expire_num = CN700_IGA2_DISPLAY_QUEUE_EXPIRE_NUM;

                break;
            }

            case VIA_CX700:
            {
                iga2_fifo_max_depth = CX700_IGA2_FIFO_MAX_DEPTH;
                iga2_fifo_threshold = CX700_IGA2_FIFO_THRESHOLD;
                iga2_fifo_high_threshold = CX700_IGA2_FIFO_HIGH_THRESHOLD;
                iga2_display_queue_expire_num = CX700_IGA2_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }
            
            case VIA_K8M890:
            {
                iga2_fifo_max_depth = K8M890_IGA2_FIFO_MAX_DEPTH;
                iga2_fifo_threshold = K8M890_IGA2_FIFO_THRESHOLD;
                iga2_fifo_high_threshold = K8M890_IGA2_FIFO_HIGH_THRESHOLD;
                iga2_display_queue_expire_num = K8M890_IGA2_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }
            
            case VIA_P4M890:
            {
                iga2_fifo_max_depth = P4M890_IGA2_FIFO_MAX_DEPTH;
                iga2_fifo_threshold = P4M890_IGA2_FIFO_THRESHOLD;
                iga2_fifo_high_threshold = P4M890_IGA2_FIFO_HIGH_THRESHOLD;
                iga2_display_queue_expire_num = P4M890_IGA2_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }

            case VIA_P4M900:
            {
                iga2_fifo_max_depth = P4M900_IGA2_FIFO_MAX_DEPTH;
                iga2_fifo_threshold = P4M900_IGA2_FIFO_THRESHOLD;
                iga2_fifo_high_threshold = P4M900_IGA2_FIFO_HIGH_THRESHOLD;
                iga2_display_queue_expire_num = P4M900_IGA2_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }

            case VIA_VX800:
            {
                iga2_fifo_max_depth = VX800_IGA2_FIFO_MAX_DEPTH;
                iga2_fifo_threshold = VX800_IGA2_FIFO_THRESHOLD;
                iga2_fifo_high_threshold = VX800_IGA2_FIFO_HIGH_THRESHOLD;
                iga2_display_queue_expire_num = VX800_IGA2_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }
            case VIA_VX855:
            {
                iga2_fifo_max_depth = VX855_IGA2_FIFO_MAX_DEPTH;
                iga2_fifo_threshold = VX855_IGA2_FIFO_THRESHOLD;
                iga2_fifo_high_threshold = VX855_IGA2_FIFO_HIGH_THRESHOLD;
                iga2_display_queue_expire_num = VX855_IGA2_DISPLAY_QUEUE_EXPIRE_NUM;
                break;
            }
        }
        
        /*Set Display FIFO Depath Select*/
        reg_value = IGA2_FIFO_DEPTH_SELECT_FORMULA(iga2_fifo_max_depth);
        load_reg_num = display_fifo_depth_reg.iga2_fifo_depth_select_reg.reg_num;
        reg = display_fifo_depth_reg.iga2_fifo_depth_select_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);

        /*Set Display FIFO Threshold Select*/
        reg_value = IGA2_FIFO_THRESHOLD_FORMULA(iga2_fifo_threshold);
        load_reg_num = fifo_threshold_select_reg.iga2_fifo_threshold_select_reg.reg_num;
        reg = fifo_threshold_select_reg.iga2_fifo_threshold_select_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);

        /*Set FIFO High Threshold Select*/
        reg_value = IGA2_FIFO_HIGH_THRESHOLD_FORMULA(iga2_fifo_high_threshold);
        load_reg_num = fifo_high_threshold_select_reg.iga2_fifo_high_threshold_select_reg.reg_num;
        reg = fifo_high_threshold_select_reg.iga2_fifo_high_threshold_select_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);

        /*Set Display Gueue Expire Num*/
        reg_value = IGA2_DISPLAY_QUEUE_EXPIRE_NUM_FORMULA(iga2_display_queue_expire_num);
        load_reg_num = display_queue_expire_num_reg.iga2_display_queue_expire_num_reg.reg_num;
        reg = display_queue_expire_num_reg.iga2_display_queue_expire_num_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);

    }
}

CARD32 get_clk_value(VIABIOSInfoPtr pBIOSInfo, int clk)
{
    int i;

    DEBUG(ErrorF("VIA: get_clk_value\n"));
    for(i=0; i<NUM_TOTAL_PLL_TABLE; i++)
    {
        if(clk==pll_value[i].clk)
        {
            switch(pBIOSInfo->Chipset)
            {
             	case VIA_P4M800PRO:
                    return (pll_value[i].k800_pll);
                case VIA_VX855:
                    return (pll_value[i].vx855_pll);
                default:      
                    return (pll_value[i].cx700_pll);
            }
        }
    }
    DEBUG(ErrorF("Can't find match PLL value\n\n"));
    return 0;
}

/* Set VCLK */
void SetVCLK(VIABIOSInfoPtr pBIOSInfo, CARD32 CLK, int set_iga)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    CARD32 RegTemp;

    DEBUG(ErrorF("Power on the PLL clock\n"));
    switch (set_iga)
    {
        case IGA1: /* VCK PLL Power Control */
            write_reg_mask(SR2D, VIASR, 0x30, BIT4+BIT5);
            break;
            
        case IGA2: /* LCK PLL Power Control */
            write_reg_mask(SR2D, VIASR, 0x0C, BIT2+BIT3);
            break;
            
        default:   /* VCK+LCK PLL Power Control */
            write_reg_mask(SR2D, VIASR, 0x3C, BIT2+BIT3+BIT4+BIT5);
            break;
    }

    DEBUG(ErrorF("VIA: SetVCLK\n"));
    /* H.W. Reset : ON */
    write_reg_mask(CR17, VIACR, 0x00, BIT7);

    if ((set_iga==IGA1) || (set_iga==IGA1_IGA2))
    {
        /* Change D,N FOR VCLK */
        write_reg(SR44, VIASR, CLK/0x10000);
        DEBUG(ErrorF("SR44=%lx\n", CLK/0x10000));
        write_reg(SR45, VIASR, (CLK&0xFFFF)/0x100);
        DEBUG(ErrorF("SR45=%lx\n", (CLK&0xFFFF)/0x100));
        write_reg(SR46, VIASR, CLK%0x100);
        DEBUG(ErrorF("SR46=%lx\n", CLK%0x100));
    }

    if ((set_iga==IGA2) || (set_iga==IGA1_IGA2))
    {
        /* Change D,N FOR LCK */
        write_reg(SR4A, VIASR, CLK/0x10000);
        write_reg(SR4B, VIASR, (CLK&0xFFFF)/0x100);
        write_reg(SR4C, VIASR, CLK%0x100);
    }

    /* H.W. Reset : OFF */
    write_reg_mask(CR17, VIACR, 0x80, BIT7);

    /* Reset PLL */
    if ((set_iga==IGA1) || (set_iga==IGA1_IGA2))
    {
        write_reg_mask(SR40, VIASR, 0x02, BIT1);
        write_reg_mask(SR40, VIASR, 0x00, BIT1);
    }

    if ((set_iga==IGA2) || (set_iga==IGA1_IGA2))
    {
        /* Reset LCDCK PLL. */
        write_reg_mask(SR40, VIASR, 0x04, BIT2);
        write_reg_mask(SR40, VIASR, 0x00, BIT2);
    }

    /* Fire! */
    RegTemp = VGAIN8(VIARMisc);
    VGAOUT8(VIAWMisc, RegTemp | (BIT2+BIT3));
}

void load_reg(int timing_value, int load_reg_num, struct io_register *reg, int io_type)
{
    int reg_mask;
    int bit_num=0;
    int data;
    int i,j;
    int shift_next_reg;
    int start_index, end_index, cr_index;
    CARD16  get_bit;

    for (i=0; i<load_reg_num; i++)
    {
	    reg_mask = 0;
	    data=0;
	    start_index = reg[i].start_bit;
	    end_index = reg[i].end_bit;
	    cr_index = reg[i].io_addr;

	    shift_next_reg = bit_num;
	    for(j=start_index;j<=end_index;j++)
	    {
		    /*if (bit_num==8) timing_value = timing_value >>8;*/
		    reg_mask = reg_mask | (BIT0 << j);
		    get_bit = (timing_value & (BIT0 << bit_num));
		    data = data | ((get_bit >> shift_next_reg)<< start_index);
		    bit_num ++;
	    }
	    if (io_type == VIACR)
	    {
		    /*DEBUG(ErrorF("i=%2d %2x %2x %2x\n", i, cr_index, data, reg_mask));*/

	        write_reg_mask(cr_index, VIACR, data, reg_mask);

	    }
	    else
	    {
		    /*DEBUG(ErrorF("SR = %2x\n", cr_index));*/
		    /*DEBUG(ErrorF("i=%2d %2x %2x %2x\n", i, cr_index, data, reg_mask));*/
	        write_reg_mask(cr_index, VIASR, data, reg_mask);
	    }
    }
}

/*
Set hsync and vsync polarity, but it's useful for CRT and DVI device
Normal CRT device: set misc register
External CRT and all DVI device: set the sync polarity register according to DI port
*/
void set_sync_polarity(VIABIOSInfoPtr pBIOSInfo, int device, int hsync, int vsync)
{
	CARD8 syncreg = (vsync<<1) | hsync; 
	CARD8 outport = VIA_DI_NONE;
	VIABIOSInfoPtr  pVia = pBIOSInfo;

	/* Normal CRT device */
	if ((device == VIA_DEVICE_CRT1) && (pBIOSInfo->CRTDIPort == VIA_DI_NONE))
	{
		VGAOUT8(VIAWMisc, (VGAIN8(VIARMisc) & (~(BIT6+BIT7))) | (syncreg<<6));
	}
	/* External CRT and all DVI device */
	else if ((device == VIA_DEVICE_CRT1) || (device == VIA_DEVICE_DFP))
	{
		if(device == VIA_DEVICE_CRT1)
			outport = pBIOSInfo->CRTDIPort;
		else
			outport = pBIOSInfo->DFPDIPort;

		switch (outport)
		{
			case VIA_DI_DVP0:
				write_reg_mask(CR96, VIACR, syncreg<<5, BIT6 | BIT5);
				break;
			case VIA_DI_DVP1:
				write_reg_mask(CR9B, VIACR, syncreg<<5, BIT6 | BIT5);
				break;
			case VIA_DI_DFPHIGH:
				write_reg_mask(CR97, VIACR, syncreg<<5, BIT6 | BIT5);
				break;
			case VIA_DI_DFPLOW:
				write_reg_mask(CR99, VIACR, syncreg<<5, BIT6 | BIT5);
				break;
			default:
				break;
		}			
	}
}


void load_crtc_timing(VIABIOSInfoPtr pBIOSInfo, struct display_timing device_timing, int set_iga)
{
    int i;
    int load_reg_num=0;
    int reg_value=0;
    struct io_register *reg=NULL;

    DEBUG(ErrorF("VIA: load_crtc_timing\n"));
    for (i=0;i<12;i++)
    {
        if (set_iga == IGA1)
        {
            /*CrtcWidth CrtcHeight record the value we down timing,
            is used for video parameter transfer*/
            pBIOSInfo->IGA1SettingInfo.CrtcWidth = device_timing.hor_addr;
            pBIOSInfo->IGA1SettingInfo.CrtcHeight = device_timing.ver_addr;
            
	        switch(i)
	        {
		        case H_TOTAL_INDEX:
	            	reg_value = IGA1_HOR_TOTAL_FORMULA(device_timing.hor_total);
			        load_reg_num = iga1_crtc_reg.hor_total.reg_num;
			        reg = iga1_crtc_reg.hor_total.reg;
			        break;
		        case H_ADDR_INDEX:
	            	reg_value = IGA1_HOR_ADDR_FORMULA(device_timing.hor_addr);
			        load_reg_num = iga1_crtc_reg.hor_addr.reg_num;
			        reg = iga1_crtc_reg.hor_addr.reg;
			        break;
		        case H_BLANK_START_INDEX:
	            	reg_value = IGA1_HOR_BLANK_START_FORMULA(device_timing.hor_blank_start);
			        load_reg_num = iga1_crtc_reg.hor_blank_start.reg_num;
			        reg = iga1_crtc_reg.hor_blank_start.reg;
			        break;
		        case H_BLANK_END_INDEX:
	            	reg_value = IGA1_HOR_BLANK_END_FORMULA(device_timing.hor_blank_start, device_timing.hor_blank_end);
			        load_reg_num = iga1_crtc_reg.hor_blank_end.reg_num;
			        reg = iga1_crtc_reg.hor_blank_end.reg;
			        break;
		        case H_SYNC_START_INDEX:
	            	reg_value = IGA1_HOR_SYNC_START_FORMULA(device_timing.hor_sync_start);
			        load_reg_num = iga1_crtc_reg.hor_sync_start.reg_num;
			        reg = iga1_crtc_reg.hor_sync_start.reg;
			        break;
		        case H_SYNC_END_INDEX:
                	reg_value = IGA1_HOR_SYNC_END_FORMULA(device_timing.hor_sync_start, device_timing.hor_sync_end);
			        load_reg_num = iga1_crtc_reg.hor_sync_end.reg_num;
			        reg = iga1_crtc_reg.hor_sync_end.reg;
			        break;
		        case V_TOTAL_INDEX:
	             	reg_value = IGA1_VER_TOTAL_FORMULA(device_timing.ver_total);
			        load_reg_num = iga1_crtc_reg.ver_total.reg_num;
			        reg = iga1_crtc_reg.ver_total.reg;
			        break;
		        case V_ADDR_INDEX:
		           	reg_value = IGA1_VER_ADDR_FORMULA(device_timing.ver_addr);
			        load_reg_num = iga1_crtc_reg.ver_addr.reg_num;
			        reg = iga1_crtc_reg.ver_addr.reg;
			        break;
		        case V_BLANK_START_INDEX:
		           	reg_value = IGA1_VER_BLANK_START_FORMULA(device_timing.ver_blank_start);
			        load_reg_num = iga1_crtc_reg.ver_blank_start.reg_num;
			        reg = iga1_crtc_reg.ver_blank_start.reg;
			        break;
		        case V_BLANK_END_INDEX:
		           	reg_value = IGA1_VER_BLANK_END_FORMULA(device_timing.ver_blank_start, device_timing.ver_blank_end);
			        load_reg_num = iga1_crtc_reg.ver_blank_end.reg_num;
			        reg = iga1_crtc_reg.ver_blank_end.reg;
			        break;
		        case V_SYNC_START_INDEX:
		            reg_value = IGA1_VER_SYNC_START_FORMULA(device_timing.ver_sync_start);
			        load_reg_num = iga1_crtc_reg.ver_sync_start.reg_num;
			        reg = iga1_crtc_reg.ver_sync_start.reg;
			        break;
		        case V_SYNC_END_INDEX:
			        reg_value = IGA1_VER_SYNC_END_FORMULA(device_timing.ver_sync_start, device_timing.ver_sync_end);
			        load_reg_num = iga1_crtc_reg.ver_sync_end.reg_num;
			        reg = iga1_crtc_reg.ver_sync_end.reg;
			        break;
	        }
	    }

        if (set_iga == IGA2)
        {
            /*CrtcWidth CrtcHeight record the value we down timing,
            is used for video parameter transfer*/
            pBIOSInfo->IGA2SettingInfo.CrtcWidth = device_timing.hor_addr;
            pBIOSInfo->IGA2SettingInfo.CrtcHeight = device_timing.ver_addr;
            
	        switch(i)
	        {
		        case H_TOTAL_INDEX:
		            reg_value = IGA2_HOR_TOTAL_FORMULA(device_timing.hor_total);
			        load_reg_num = iga2_crtc_reg.hor_total.reg_num;
			        reg = iga2_crtc_reg.hor_total.reg;
			        break;
		        case H_ADDR_INDEX:
		            reg_value = IGA2_HOR_ADDR_FORMULA(device_timing.hor_addr);
			        load_reg_num = iga2_crtc_reg.hor_addr.reg_num;
			        reg = iga2_crtc_reg.hor_addr.reg;
			        break;
		        case H_BLANK_START_INDEX:
		            reg_value = IGA2_HOR_BLANK_START_FORMULA(device_timing.hor_blank_start);
			        load_reg_num = iga2_crtc_reg.hor_blank_start.reg_num;
			        reg = iga2_crtc_reg.hor_blank_start.reg;
			        break;
		        case H_BLANK_END_INDEX:
		            reg_value = IGA2_HOR_BLANK_END_FORMULA(device_timing.hor_blank_start, device_timing.hor_blank_end);
			        load_reg_num = iga2_crtc_reg.hor_blank_end.reg_num;
			        reg = iga2_crtc_reg.hor_blank_end.reg;
			        break;
		        case H_SYNC_START_INDEX:
		            reg_value = IGA2_HOR_SYNC_START_FORMULA(device_timing.hor_sync_start);
                    load_reg_num = iga2_crtc_reg.hor_sync_start.reg_num;
			        reg = iga2_crtc_reg.hor_sync_start.reg;
			        break;
		        case H_SYNC_END_INDEX:
		            reg_value = IGA2_HOR_SYNC_END_FORMULA(device_timing.hor_sync_start, device_timing.hor_sync_end);
			        load_reg_num = iga2_crtc_reg.hor_sync_end.reg_num;
			        reg = iga2_crtc_reg.hor_sync_end.reg;
			        break;
		        case V_TOTAL_INDEX:
		            reg_value = IGA2_VER_TOTAL_FORMULA(device_timing.ver_total);
			        load_reg_num = iga2_crtc_reg.ver_total.reg_num;
			        reg = iga2_crtc_reg.ver_total.reg;
			        break;
		        case V_ADDR_INDEX:
		            reg_value = IGA2_VER_ADDR_FORMULA(device_timing.ver_addr);
			        load_reg_num = iga2_crtc_reg.ver_addr.reg_num;
			        reg = iga2_crtc_reg.ver_addr.reg;
			        break;
		        case V_BLANK_START_INDEX:
		            reg_value = IGA2_VER_BLANK_START_FORMULA(device_timing.ver_blank_start);
			        load_reg_num = iga2_crtc_reg.ver_blank_start.reg_num;
			        reg = iga2_crtc_reg.ver_blank_start.reg;
			        break;
		        case V_BLANK_END_INDEX:
		            reg_value = IGA2_VER_BLANK_END_FORMULA(device_timing.ver_blank_start, device_timing.ver_blank_end);
			        load_reg_num = iga2_crtc_reg.ver_blank_end.reg_num;
			        reg = iga2_crtc_reg.ver_blank_end.reg;
			        break;
		        case V_SYNC_START_INDEX:
		            reg_value = IGA2_VER_SYNC_START_FORMULA(device_timing.ver_sync_start);
			        load_reg_num = iga2_crtc_reg.ver_sync_start.reg_num;
			        reg = iga2_crtc_reg.ver_sync_start.reg;
			        break;
		        case V_SYNC_END_INDEX:
			        reg_value = IGA2_VER_SYNC_END_FORMULA(device_timing.ver_sync_start, device_timing.ver_sync_end);
			        load_reg_num = iga2_crtc_reg.ver_sync_end.reg_num;
			        reg = iga2_crtc_reg.ver_sync_end.reg;
			        break;
	        }
	     }
	     load_reg(reg_value, load_reg_num, reg, VIACR);
	}


    xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO,
           "IGA%d timing\n"
           "HActive=%d, HSyncStart=%d, HSyncEnd=%d, HTotal=%d \n"
           "VActive=%d, VSyncStart=%d, VSyncEnd=%d, VTotal=%d\n",           
           set_iga,
           device_timing.hor_addr, device_timing.hor_sync_start, device_timing.hor_sync_start+device_timing.hor_sync_end, device_timing.hor_total,
           device_timing.ver_addr, device_timing.ver_sync_start, device_timing.ver_sync_start+device_timing.ver_sync_end, device_timing.ver_total
           );

    
}

void set_color_depth(VIABIOSInfoPtr pBIOSInfo, int bpp_byte, int set_iga)
{
    if (set_iga == IGA1)
    {
        switch(bpp_byte) 
        {
            case MODE_8BPP:
                write_reg_mask(SR15, VIASR, 0xA2, 0xFE);
                break;
                
            case MODE_16BPP:
                write_reg_mask(SR15, VIASR, 0xB6, 0xFE);
                break;
                
            case MODE_32BPP:
                write_reg_mask(SR15, VIASR, 0xAE, 0xFE);
                break;
         }
    }
    else
    {
        switch(bpp_byte)
        {
            case MODE_8BPP:
                write_reg_mask(CR67, VIACR, 0x00, 0xC0);
                break;
                
            case MODE_16BPP:
                write_reg_mask(CR67, VIACR, 0x40, 0xC0);
                break;
                
            case MODE_32BPP:
                write_reg_mask(CR67, VIACR, 0xC0, 0xC0);
                break;
        }
    }
}

void fill_crtc_timing(VIABIOSInfoPtr pBIOSInfo, struct crt_mode_table *crt_table, int mode_array, int bpp_byte, int device, int set_iga, Bool use_modeline)
{
    int i;
    int index=0;
    struct display_timing crt_reg;
    int h_addr;
    int HalfLine,VSOffset,v_total;
    unsigned long HalfScreen;
    int h_active, v_active;
    int isNeedAdjust=TRUE;
    int hsync_polarity;
    int vsync_polarity;

    CARD32 pll_D_N;


    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "fill_crtc_timing\n"));

    if (use_modeline)
    {
        crt_reg = pBIOSInfo->UserSpecifiedMode.crtc;
        hsync_polarity = pBIOSInfo->UserSpecifiedMode.h_sync_polarity;
        vsync_polarity = pBIOSInfo->UserSpecifiedMode.v_sync_polarity;        
    }
    else /* Driver Built-in Mode Table */
    {   
        for(i=0;i<mode_array;i++)
        {
            index = i;

            if (crt_table[i].refresh_rate == pBIOSInfo->FoundRefresh)
            {
                break;
            }
        }

        crt_reg = crt_table[index].crtc;
        hsync_polarity = crt_table[index].h_sync_polarity;
        vsync_polarity = crt_table[index].v_sync_polarity;            
    }
    
    /* If the view size is smaller than the timing we set, */
    /* don't show the part out of real display range. */
    if (pBIOSInfo->ActiveDevice != VIA_DEVICE_LCD)
    {
        if (set_iga == IGA1)
        {
            h_active = pBIOSInfo->IGA1SettingInfo.HActive;
            v_active = pBIOSInfo->IGA1SettingInfo.VActive;
            if (pBIOSInfo->IGA1SettingInfo.IsDISP3DScaling)
            {
                isNeedAdjust = FALSE;
            }
        }
        else
        {
            h_active = pBIOSInfo->IGA2SettingInfo.HActive;
            v_active = pBIOSInfo->IGA2SettingInfo.VActive;
            if (pBIOSInfo->IGA2SettingInfo.IsDISP3DScaling)
            {
                isNeedAdjust = FALSE;
            }            
        }

        if (isNeedAdjust)
        {
            if (pBIOSInfo->ActualDesktopSizeX < h_active)
            {
                crt_reg.hor_addr = pBIOSInfo->ActualDesktopSizeX;
            }

            if (pBIOSInfo->ActualDesktopSizeY < v_active)
            {
                crt_reg.ver_addr = pBIOSInfo->ActualDesktopSizeY;
            }
        }
    }
    
    h_addr = crt_reg.hor_addr;
    v_total= crt_reg.ver_total;
    HalfLine=(h_addr*bpp_byte)/4;
    HalfScreen=(h_addr*v_total*bpp_byte)/4;
    VSOffset=h_addr*2;
    /*for mergedfb*/
    pBIOSInfo->HalfLine = HalfLine;
    pBIOSInfo->HalfScreen = HalfScreen;


    DEBUG(ErrorF("===Debug===\n"));
    DEBUG(ErrorF("Index=%d\n", index));
    DEBUG(ErrorF("Refresh_rate=%d\n", crt_table[index].refresh_rate));
    DEBUG(ErrorF("HT=%d\n",crt_reg.hor_total));
    DEBUG(ErrorF("HAdr=%d\n",crt_reg.hor_addr));
    DEBUG(ErrorF("HBS=%d\n",crt_reg.hor_blank_start));
    DEBUG(ErrorF("HBE=%d\n",crt_reg.hor_blank_end));
    DEBUG(ErrorF("HSS=%d\n",crt_reg.hor_sync_start));
    DEBUG(ErrorF("HSE=%d\n",crt_reg.hor_sync_end));

    DEBUG(ErrorF("VT=%d\n",crt_reg.ver_total));
    DEBUG(ErrorF("VAdr=%d\n",crt_reg.ver_addr));
    DEBUG(ErrorF("VBS=%d\n",crt_reg.ver_blank_start));
    DEBUG(ErrorF("VBE=%d\n",crt_reg.ver_blank_end));
    DEBUG(ErrorF("VSS=%d\n",crt_reg.ver_sync_start));
    DEBUG(ErrorF("VSE=%d\n",crt_reg.ver_sync_end));    

    set_sync_polarity(pBIOSInfo, device, hsync_polarity, vsync_polarity);

    unlock_crt();

    switch(set_iga)
    {
        case IGA1:
            load_crtc_timing(pBIOSInfo, crt_reg, IGA1);
            break;
            
        case IGA2:
            load_crtc_timing(pBIOSInfo, crt_reg, IGA2);
            break;
    }

    lock_crt();    
    write_reg_mask(CR17, VIACR, 0x80, BIT7); /* Hardware Reset */
    
    if(pBIOSInfo->MergedFB)
    {/* to set offset while mergedfb*/ 

        if((pBIOSInfo->Scrn2Position == viaRightOf)||(pBIOSInfo->Scrn2Position == viaLeftOf))
            load_offset_reg(VSOffset, bpp_byte, set_iga);
        if((pBIOSInfo->Scrn2Position == viaBelow)||(pBIOSInfo->Scrn2Position == viaAbove))
            load_offset_reg(h_addr, bpp_byte, set_iga);
    }
    else   
    {
        load_offset_reg(h_addr, bpp_byte, set_iga);
    }

    load_fetch_count_reg(h_addr, bpp_byte, set_iga);

    /*load FIFO*/
    load_FIFO_reg(pBIOSInfo, crt_reg.hor_addr, crt_reg.ver_addr, set_iga);

    /*load SR Register About Memory and Color part*/
    set_color_depth(pBIOSInfo, bpp_byte, set_iga);    


    if (use_modeline)
    {
        CalFromClock(pBIOSInfo, (double)(pBIOSInfo->UserSpecifiedMode.clk/1000000), &pll_D_N);    
        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO," PixelClock=%ld Hz\n", pBIOSInfo->UserSpecifiedMode.clk);
        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO,"-----TIMING END-----\n");                
    }
    else
    {
        /* Set PLL     */
        pll_D_N = get_clk_value(pBIOSInfo, crt_table[index].clk);
        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO," PixelClock=%ld Hz\n", crt_table[index].clk);        
        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO,"-----TIMING END-----\n");                        
    }
    
    DEBUG(ErrorF("PLL=0x%x\n", pll_D_N));
    SetVCLK(pBIOSInfo, pll_D_N, set_iga);
}

/* Write Registers */
void write_regx(struct io_reg RegTable[],int ItemNum)
{
    int             i;

    for( i=0; i< ItemNum; i++ ) {
        write_reg_mask(RegTable[i].index, RegTable[i].port, RegTable[i].value, RegTable[i].mask);
    }
}


void EnableSecondDisplayChannel()
{
    /* It's better to use the following sequence */
    /* to enable second display channel. */
    write_reg_mask(CR6A, VIACR, 0x00, BIT6);
    write_reg_mask(CR6A, VIACR, BIT7, BIT7);
    write_reg_mask(CR6A, VIACR, BIT6, BIT6);
}

void DisableSecondDisplayChannel()
{
    /* It's better to use the following sequence */
    /* to disable second display channel. */
    write_reg_mask(CR6A, VIACR, 0x00, BIT6);
    write_reg_mask(CR6A, VIACR, 0x00, BIT7);
    write_reg_mask(CR6A, VIACR, BIT6, BIT6);
}



void VIADIPortPadOn(int DIPort)
{
    switch (DIPort)
    {
        case VIA_DI_DVP0:
            write_reg_mask(SR1E, VIASR, 0xC0, BIT6+BIT7);
            break;

        case VIA_DI_DVP1:
            write_reg_mask(SR1E, VIASR, 0x30, BIT4+BIT5);
            break;

        case VIA_DI_DFPHIGH:
            write_reg_mask(SR2A, VIASR, 0x0C, BIT2+BIT3);
            break;

        case VIA_DI_DFPLOW:
            write_reg_mask(SR2A, VIASR, 0x03, BIT0+BIT1);
            break;

        case VIA_DI_DFPHIGHLOW:
            write_reg_mask(SR2A, VIASR, 0x0F, BIT0+BIT1+BIT2+BIT3);
            break;
    }
}


void VIADIPortPadOff(int DIPort)
{
    switch (DIPort)
    {
        case VIA_DI_DVP0:
            write_reg_mask(SR1E, VIASR, 0x00, BIT6+BIT7);
            break;

        case VIA_DI_DVP1:
            write_reg_mask(SR1E, VIASR, 0x00, BIT4+BIT5);
            break;

        case VIA_DI_DFPHIGH:
            write_reg_mask(SR2A, VIASR, 0x00, BIT2+BIT3);
            break;

        case VIA_DI_DFPLOW:
            write_reg_mask(SR2A, VIASR, 0x00, BIT0+BIT1);
            break;

        case VIA_DI_DFPHIGHLOW:
            write_reg_mask(SR2A, VIASR, 0x00, BIT0+BIT1+BIT2+BIT3);
            break;
    }
}



static void NMR2Reg(int N, int M, int R, int* dwSR47, int* dwSR48, int* dwSR49)
{
    int DTZ = BIT1 | BIT0;    /* b'11*/

    M   &= 0x3FF;   /* 10 bits */
    N   &= 0x3F;    /* 6 bits */
    R   &= 0x7;     /* 3 bits */

    *dwSR47    = M & 0xFF;
    M       >>= 8;
    *dwSR48    |= M;

    *dwSR49    = N;

    *dwSR48    |= (R << 2);

    *dwSR48    |= ((DTZ & 1) << 7);
    DTZ     >>= 1;
    *dwSR49    |= ((DTZ & 1) << 7);

    DEBUG(ErrorF("dwSR47 = 0x%x\n", *dwSR47));
    DEBUG(ErrorF("dwSR48 = 0x%x\n", *dwSR48));
    DEBUG(ErrorF("dwSR49 = 0x%x\n", *dwSR49));

}


static void CREATE_CLOCK_SETTING(CLOCK_SETTING* pNewClockSetting, double dbClock, int iM, int iN, int iR, double dbf_refDivBW, double dbPhaseMargin)
{        
    pNewClockSetting->clock = dbClock;
    pNewClockSetting->M   = iM;
    pNewClockSetting->N   = iN;
    pNewClockSetting->R   = iR;
    pNewClockSetting->f_refDivBW  = dbf_refDivBW;
    pNewClockSetting->PhaseMargin = dbPhaseMargin;

    DEBUG(ErrorF("====================================\n"));
    DEBUG(ErrorF("clock = %f\n", pNewClockSetting->clock));
    DEBUG(ErrorF("M = %d\n", pNewClockSetting->M));
    DEBUG(ErrorF("N = %d\n", pNewClockSetting->N));
    DEBUG(ErrorF("R = %d\n", pNewClockSetting->R));
    DEBUG(ErrorF("f_refDivBW = %f\n", pNewClockSetting->f_refDivBW));
    DEBUG(ErrorF("PhaseMargin = %f\n", pNewClockSetting->PhaseMargin));
}




static void InputClock(CLOCK_SETTING* pClockSetting, double dbFout, Bool bVT3314, unsigned int iSortType)
{       
    double m_dbFin = FIN;
    int    m_iN = 1000;
    int    m_iD = 5;
    int    m_iP = 5;
    double m_dbERR = 5000;

    double m_dbFxo = m_dbFin * 1E6;     /* unit = MHz*/
    double m_dbIp = 10 * 1E-6;          /* unit = uA */
    double m_dbKpfd = m_dbIp / (2*PI);  /* unit = uA/rad */
    double m_dbKvco = 400 * 1E6;        /* unit = MHz*/
    double m_dbC1 = 225 * 1E-12;        /* unit = pf*/
    double m_dbC2 = 15 * 1E-12;         /* unit = pf*/
    double m_dbR1 = 40 * 1E3;           /* unit = k-ohm*/
    double m_dbZERO = 1 / (TWO_PI*m_dbR1*m_dbC1);
    double m_dbPOLE =(m_dbC1+m_dbC2) / (TWO_PI*m_dbR1*m_dbC1*m_dbC2);        

    int N=2,D=2,R=0;    
    double dbTest   = dbFout/m_dbFin;
    double dbFvco   = 0;
    int iMaxN   = m_iN;
    double iBestValue = 0;
    
    if(bVT3314)
    {
        /* 3314 M[9:8] must 0, so maximum value of M is equal to 0xFF, and N = M+2*/
        iMaxN = (m_iN<0xFF+2)?m_iN:(0xFF+2);
    }

    while (N<=iMaxN) 
    {
        while(D<=m_iD)
        {
            while(R<=m_iP)
            {
                dbFvco  = dbFout * pow(2.0f,R);
                double dbF_ref  = m_dbFxo / D;
                double dbBW     = ((m_dbKpfd * m_dbKvco)/N)*m_dbR1*((m_dbC1/m_dbC2)/(1+(m_dbC1/m_dbC2)));
                double dbPHASE_M1       =atan2(dbBW,m_dbPOLE);
                double dbPHASE_M2       =atan2(dbBW,m_dbZERO);
                double dbPHASE_MARGIN   =(dbPHASE_M2-dbPHASE_M1)*180/PI;
                double dbClock  = (double)N/(D*pow(2.0f,R));

                if ( (dbClock > (dbTest-dbTest*m_dbERR/1E6)) &&
                     (dbClock < (dbTest+dbTest*m_dbERR/1E6)) &&
                     ((dbFvco>=300) && (dbFvco<=600)))
                {  
                    switch (iSortType)
                    {
                        case SORT_BY_PHASE_MARGIN:
                            if (iBestValue == 0)
                            {
                                iBestValue = dbPHASE_MARGIN;
                            }

                            /* Choose the biggest value. */
                            if (dbPHASE_MARGIN > iBestValue)
                            {
                                CREATE_CLOCK_SETTING(pClockSetting, dbClock*m_dbFin, N-2, D-2, R, dbF_ref/dbBW, dbPHASE_MARGIN);
                                iBestValue = dbPHASE_MARGIN;
                            }                    
                        break;
                        case SORT_BY_F_REF_BW:
                            if (iBestValue == 0)
                            {
                                iBestValue = (dbF_ref/dbBW);
                            }

                            /* Choose the biggest value. */
                            if ((dbF_ref/dbBW) > iBestValue)
                            {
                                CREATE_CLOCK_SETTING(pClockSetting, dbClock*m_dbFin, N-2, D-2, R, dbF_ref/dbBW, dbPHASE_MARGIN);
                                iBestValue = (dbF_ref/dbBW);
                            }                   
                        break;
                        case SORT_BY_PIXEL_CLOCK:
                            if (iBestValue == 0)
                            {
                                iBestValue = dbClock*m_dbFin;
                            }

                            /* Choose the most close value. */
                            if (fabs(dbClock*m_dbFin - dbFout) < fabs(iBestValue - dbFout))
                            {
                                CREATE_CLOCK_SETTING(pClockSetting, dbClock*m_dbFin, N-2, D-2, R, dbF_ref/dbBW, dbPHASE_MARGIN);
                                iBestValue = dbClock*m_dbFin;
                            }                    
                        break;
                    }
                }
                
                R++;
            }

            R   = 0;
            D++;
        }

        R   = 0;
        D   = 2;
        N++;
    }
}



void CalFromClock(VIABIOSInfoPtr pBIOSInfo, double dbDesiredClock, CARD32* PLLValue)
{    
    CLOCK_SETTING clockSetting;
    double dbClock  = dbDesiredClock;
    int dwSR47=0, dwSR48=0, dwSR49=0;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "Caculate PLL from Clock. Input Clock = %f\n",dbDesiredClock));            

    memset(&clockSetting, 0, sizeof(CLOCK_SETTING));
    InputClock(&clockSetting, dbClock, FALSE, SORT_BY_PHASE_MARGIN);
    
    if(pBIOSInfo->Chipset == VIA_VX855)
    {
    	/* 
    		For Vt3409, M and N dosn't need to add 2, and since clockSetting.N and clockSetting.M have been sub 2 in
		function InputClock(), here should add 2 to them. 
		*/
        NMR2Reg(clockSetting.N+2,clockSetting.M+2, clockSetting.R, &dwSR47, &dwSR48, &dwSR49);
    }
    else
    {
        NMR2Reg(clockSetting.N,clockSetting.M, clockSetting.R, &dwSR47, &dwSR48, &dwSR49);
    }        

    *PLLValue = ((dwSR47<<16) | (dwSR48<<8) | dwSR49);
    
    return;
                        
}

