/*
 * Copyright 2006 Thomas Hellstrom. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#ifndef VIA_3D_H
#define VIA_3D_H

#include "xf86.h"
#include "picturestr.h"

#include "via_dmabuffer.h"

/* A flag to make H5 ps instruction easy reading/writing */
#define VIA_H5_SHADER_INSTRUCTION_TRANSLATE_ENABLE  1
#define TEX_PITCH_LIMIT_INBYTE    8192
#define TEX_DIMENSION_LIMIT_INPIXEL   2048
#define DIMENSION_2D_LIMIT_INPIXEL  4096
#define HWDIMENSION_BOX_WIDTH_INPIXEL   1024
#define HWDIMENSION_BOX_HEIGHT_INPIXEL   1024
/* H2 chips 3D engine only support clipping window range: 0 ~ 2047 */
#define HW_H2_CLIPRANG(x)    min(x, ((1<<11)-1))
/* H5 chips 3D engine only support clipping window range: 0 ~ 4095 */
#define HW_H5_CLIPRANG(x)    min(x, ((1<<12)-1))

#define VIA_NUM_TEXUNITS 2

typedef enum
{
    via_single,
    via_clamp,
    via_repeat,
    via_mirror,
    via_warp
} ViaTextureModes;

typedef enum
{
    via_src,
    via_src_onepix_mask,
    via_src_onepix_comp_mask,
    via_mask,
    via_comp_mask,

    src_Aa,
    src_Ca,			
    mask_Ca,
} ViaTexBlendingModes;

typedef enum
{
    via_FilterNearest,
    via_FilterBilinear,
    via_FilterFast,
    via_FilterGood,
    via_FilterBest
} ViaTexFilterModes;
#if VIA_H5_SHADER_INSTRUCTION_TRANSLATE_ENABLE
/* ensure the pack instruction to avoid the compiler apply byte alignment optimization */
#pragma pack(push,1)

typedef enum
{
    via_mad = 0,
    /* reserved 1 */
    via_dp3 = 2,
    via_dp4,
    via_exp,
    via_log,
    via_rcp,
    via_rsq,
    /* reserved 8 */
    via_cmp = 9,
    via_cmd,
    via_max,
    via_min,
    via_mov,
    via_frc,
    via_sincos,
    via_madfix,
    via_blendfix,
    /* reserved 18 */
    /* reserved 19 */
    via_texkill3 = 20,
    via_texkill4,
    /* reserved 22 ~ 31 */
} via3DALUOp;

typedef enum
{
    via_texld = 0,
    via_tesldb,
    via_teskill,
    via_texldp,
    /* reserved 4 */
    /* reserved 5 ~ 7 */
} via3DTAUOp;

typedef enum
{
    selT_r = 0,
    selT_g,
    selT_b,
    selT_a,
} via3DTAUSrcSelector;

typedef enum
{
    selA_r = 0,
    selA_g,
    selA_b,
    selA_a,
    selA_1f,
    selA_0f,
    /* reserved 6 */
    /* reserved 7 */
} via3DALUSrcSelector;

typedef enum
{
    dstT_nodefined = 0,
    dstT_temp,
} via3DTAUDstTempReg;

typedef enum
{
    modi_no = 0,
    modi_bias,
    modi_invert,
    /* reserved 3 */
    modi_scaleX2 = 4,
    /* reserved 5 ~ 7 */
    modi_negate = 8,
    /* reserved 9 ~ 15 */
} via3DSrcModifier;

typedef enum
{
    src_temp = 0,
    src_color,
    src_const,
    src_texture,
    src_aluOut,
    src_preConstant,
    src_fogCoordinate,
    src_notrequired,
} via3DSrcRegtype;

typedef enum
{
    dst_temp = 0,
    dst_texture,
    dst_output,
    dst_internaltemp,
    dst_nodefined = dst_internaltemp,
} via3DDstRegtype;

#define NO_DEST_INDEX  15

typedef enum
{
    scale_disable = 0,
    scale_x2,
    scale_x4,
    scale_x8,
    /* reserved 4 */
    scale_d2 = 5,
    scale_d4,
    scale_d8,
} via3DInsScale;

typedef enum
{
    mask_none= 0,
    mask_r = 1,
    mask_g = 2,
    mask_b = 4,
    mask_a = 8,
    mask_all = 15,
} via3DWriteMask;

typedef union _Via3DTAUInstruction
{
    struct {
        CARD32  s1index: 4;      /*[3:0]*/
        CARD32  s0sel_a:  2;     /*[5:4]*/
        CARD32  s0sel_b:  2;     /*[7:6]*/
        CARD32  s0sel_g:  2;     /*[9:8]*/
        CARD32  s0sel_r:  2;     /*[11:10]*/
        CARD32  s0index:  4;    /*[15:12]*/
        CARD32  dindex:  4;      /*[19:16]*/
        CARD32  dreg:  1;         /*[20]*/
        CARD32  op:  3;            /*[23:21]*/
        CARD32  tstage:  4;      /*[27:24]*/
        CARD32  sfire:  1;         /*[28]*/
        CARD32  pvalid:  1;      /*[29]*/
        CARD32  debug:  1;      /*[30]*/
        CARD32  dfire:  1;        /*[31]*/
    };
    CARD32  uint[1];
} Via3DTAUInstructionRec, *Via3DTAUInstructionPtr;

typedef union _Via3DALUInstruction
{
    struct {
        CARD32  s2mod: 4;                   /*[3:0]*/
        CARD32  s2sel_a:  3;                /*[6:4]*/
        CARD32  s2sel_b:  3;                /*[9:7]*/
        CARD32  s2sel_g:  3;                /*[12:10]*/
        CARD32  s2sel_r:  3;                 /*[15:13]*/
        CARD32  s2index:  6;                /*[21:16]*/
        CARD32  s2reg:  3;                    /*[24:22]*/
        CARD32  s2fire:  1;                    /*[25]*/
        CARD32  s2reserved: 2;            /*[27:26]*/
        CARD32  s1mod:  4;                   /*[31:28]*/
        CARD32  s1sel_a:  3;                 /*[34:32]*/
        CARD32  s1sel_b:  3;                 /*[37:35]*/
        CARD32  s1sel_g:  3;                 /*[40:38]*/
        CARD32  s1sel_r:  3;                 /*[43:41]*/
        CARD32  s1index:  6;                /*[49:44]*/
        CARD32  s1reg:  3;                    /*[52:50]*/
        CARD32  s1fire:  1;                    /*[53]*/
        CARD32  s1reserved:  2;           /*[55:54]*/  
        CARD32  s0mod:  4;                   /*[59:56]*/
        CARD32  s0sel_a:  3;                 /*[62:60]*/
        CARD32  s0sel_b:  3;                 /*[65:63]*/
        CARD32  s0sel_g:  3;                 /*[68:66]*/
        CARD32  s0sel_r:  3;                 /*[71:69]*/ 
        CARD32  s0index:  6;                /*[77:72]*/
        CARD32  s0reg:  3;                   /*[80:78]*/
        CARD32  s0fire:  1;                   /*[81]*/
        CARD32  s0reserved:  2;          /*[83:82]*/
        CARD32  dwmask:  4;               /*[87:84]*/
        CARD32  dreserved: 8;             /*[95:88]*/
        CARD32  dindex1:  4;                /*[99:96]*/
        CARD32  dreg1:  2;                    /*[101:100]*/
        CARD32  dindex0:  4;                /*[105:102]*/
        CARD32  dreg0:  2;                    /*[107:106]*/
        CARD32  isat:  1;                       /*[108]*/
        CARD32  iscale:  3;                    /*[111:109]*/
        CARD32  op:  5;                         /*[116:112]*/
        CARD32  opreserved:  5;          /*[121:117]*/
        CARD32  coissue:  1;                 /*[122]*/
        CARD32  debug:  1;                   /*[123]*/
        CARD32  d1fire:  1;                   /*[124]*/
        CARD32  d0fire:  1;                   /*[125]*/
        CARD32  reserved:  2;              /*[127:126]*/
    };
    CARD32  uint[4];
} Via3DALUInstructionRec, *Via3DALUInstructionPtr;

#pragma pack(pop)
#endif

typedef struct _ViaTextureUnit
{
    CARD32 textureLevel0Offset;
    CARD32 textureLevel0UOffset; /* used for video Texture YV12 format only */
    CARD32 textureLevel0VOffset; /* used for video Texture YV12 format only */
    CARD32 textureLevel0Pitch;
    CARD32 textureLevel0Width;
    CARD32 textureLevel0Height;
    CARD32 textureLevel0WExp;
    CARD32 textureLevel0HExp;
    CARD32 textureBlendMode;
    CARD32 textureDrawable_width;
    CARD32 textureDrawable_height;
    CARD32 bytePerPixel;
    CARD32 textureFormat;
    CARD32 textureModesT;
    CARD32 textureModesS;
    CARD32 texCsat;
    CARD32 texRCa;
    CARD32 texAsat;
    CARD32 texRAa;
    int pictureFormat;
    int pictureFilter;
    CARD32 PictureOffset;
    Bool agpTexture;
    Bool textureDirty;
    Bool texBColDirty;
    Bool npot;
    Bool textureRepeat;
    /** Transform pointers for teture, or NULL if identity */
    PictTransform *transform;
    CARD32 texturefilter;
} ViaTextureUnit;

typedef struct _Via3DState
{
    Bool destDirty;
    Bool blendDirty;
    Bool enableDirty;
    Bool drawingDirty;
    CARD32 rop;
    CARD32 planeMask;
    CARD32 solidColor;
    CARD32 solidAlpha;
    CARD32 destOffset;
    CARD32 destPitch;
    CARD32 destFormat;
    int destDepth;
    int numTextures;
    Bool blend;
    CARD32 blendCol0;
    CARD32 blendCol1;
    CARD32 blendAl0;
    CARD32 blendAl1;
    Bool writeAlpha;
    Bool writeColor;
    Bool useDestAlpha;
    Bool forceUpload;
    Bool componentAlpha;
    Bool srcRepeat;
    Bool maskRepeat;
    ViaTextureUnit tex[VIA_NUM_TEXUNITS];
#if VIA_H5_SHADER_INSTRUCTION_TRANSLATE_ENABLE
    Via3DTAUInstructionRec  viaTAUIns;
    Via3DALUInstructionRec  viaALUIns;
#endif
    void (*setDestination) (struct _Via3DState * v3d, CARD32 offset,
	CARD32 pitch, int format);
    void (*setDrawing) (struct _Via3DState * v3d, int rop,
	CARD32 planeMask, CARD32 solidColor, CARD32 solidAlpha);
    void (*setFlags) (struct _Via3DState * v3d, int numTextures,
	Bool writeAlpha, Bool writeColor, Bool blend);
    Bool(*setTexture) (struct _Via3DState * v3d, int tex, CARD32 offset, 
        CARD32 pitch, Bool nPot, CARD32 width, CARD32 height, int format, 
        ViaTextureModes sMode, ViaTextureModes tMode, ViaTexBlendingModes blendingMode, 
        Bool agpTexture,PictTransformPtr matrix, ViaTexFilterModes filter);
    Bool(*setTexUVOffset) (struct _Via3DState * v3d, int tex, CARD32 uoffset, CARD32 voffset);
    void (*setTexBlendCol) (struct _Via3DState * v3d, int tex, Bool component,
	CARD32 color);
    void (*setCompositeOperator) (struct _Via3DState * v3d, CARD8 op,Bool comp);
        Bool(*opSupported) (CARD8 op);
    void (*emitQuad) (struct _Via3DState * v3d, ViaCommandBuffer * cb,
	int dstX, int dstY, int src0X, int src0Y, int src1X, int src1Y, int w,
	int h);
    void (*emitState) (struct _Via3DState * v3d, ViaCommandBuffer * cb,
	Bool forceUpload);
    void (*emitClipRect) (struct _Via3DState * v3d, ViaCommandBuffer * cb,
	int x, int y, int w, int h);
    void (*emitPixelShader) (struct _Via3DState * v3d, ViaCommandBuffer * cb, 
        int srcFormat, int maskFormat, int dstFormat);
    Bool(*dstSupported) (int format);
    Bool(*texSupported) (int format);
    int (*MarkSync)(ScreenPtr pScreen);
    void (*WaitMarker)(ScreenPtr pScreen, int marker);
} Via3DState;

typedef union _via2DBltSrc {
    struct {
        unsigned bpp;
        unsigned long offset;
        unsigned pitch;
        unsigned x;
        unsigned y;
        unsigned long colorKey;
        unsigned memLoc;
    };
    struct {
        unsigned bpp;
        unsigned long color;
    };
} via2DBltSrcRec, *via2DBltSrcPtr;

typedef struct {
    unsigned bpp;
    unsigned long offset;
    unsigned pitch;
    unsigned x;
    unsigned y;
    unsigned w;
    unsigned h;
    unsigned long colorKey;
} via2DBltDstRec, *via2DBltDstPtr;

typedef union _via2DBltConfig {
    struct {
        unsigned srckey:1;
        unsigned dstey:1;
        unsigned solidfill:1;
        unsigned reservered:29;
    };
    unsigned unit; 
} via2DBltConfigRec, *via2DBltConfigPtr;

typedef struct {
    unsigned format;        /* src format in viaFormats[] */
    unsigned long offset;   /* src offset */
    unsigned long uoffset;  /* src u(cb) offset */
    unsigned long voffset;  /* src v(cr) offset */
    unsigned pitch;         /* src pitch */
    unsigned x;             /* src start x */
    unsigned y;             /* src start y */
    unsigned w;             /* src width */
    unsigned h;             /* src height */
    unsigned surfwidth;     /* src surface width */
    unsigned surfheight;    /* src surface height */
    unsigned filter;        /* src filter mode */
    unsigned memLoc;        /* src surface memory location */
} viaTexture3DBltSrcRec, *viaTexture3DBltSrcPtr;

typedef struct {
    unsigned format;       /* dst format in viaFormats[] */
    unsigned long offset;  /* dst offset */
    unsigned pitch;        /* dst pitch */
    unsigned x;            /* dst start x */
    unsigned y;            /* dst start y */
    unsigned w;            /* dst width */
    unsigned h;            /* dst height */
} viaTexture3DBltDstRec, *viaTexture3DBltDstPtr;

typedef struct {
    unsigned rotate;         /* rotation/reflection flag in R&R define */
    unsigned width;          /* rotate/reflect area width */
    unsigned height;         /* rotate/reflect area height */
} viaTexture3DBltRotationRec, *viaTexture3DBltRotationPtr;

void
viaAccelTexture3DBlt(ScrnInfoPtr pScrn, viaTexture3DBltSrcPtr src, viaTexture3DBltDstPtr dst,
                    viaTexture3DBltRotationPtr rotation, RegionPtr clip_region);

Bool viaOrder(CARD32 val, CARD32 * shift);
void viaInit3DState(Via3DState * v3d);
void viaInit3DState_H5(Via3DState * v3d);
void viaInit3DState_H6(Via3DState * v3d);
void Modulate_H5_via_src(Via3DState *v3d, ViaCommandBuffer * cb);
void Modulate_H5_via_src_onepix_mask_no_alpha(Via3DState *v3d, ViaCommandBuffer * cb);
void Modulate_H5_via_src_no_alpha(Via3DState *v3d, ViaCommandBuffer * cb);
void Modulate_H5_via_src_onepix_mask(Via3DState *v3d, ViaCommandBuffer * cb);
void Modulate_H5_via_mask(Via3DState *v3d, ViaCommandBuffer * cb);
void Modulate_H5_via_mask2(Via3DState *v3d, ViaCommandBuffer * cb);
void Modulate_H5_via_mask2_no_alpha(Via3DState *v3d, ViaCommandBuffer * cb);
void Modulate_H5_No_Tex(Via3DState *v3d, ViaCommandBuffer * cb);
#endif
