 /*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <stdarg.h> /* for variable argument */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>

#include "via_driver.h"
#include "via_dri.h"
#include "via_rotate.h" /* for rotate feature.*/
#include "via_Hw3DScaling.h"
#include "via_eng_regs.h"

void hw3d_CursorPositionUpdate(VIABIOSInfoPtr pBIOSInfo)
{
    int x = 0, y = 0;
    ScrnInfoPtr pScrn = xf86Screens[pBIOSInfo->scrnIndex];
    VIAPtr      pVia = VIAPTR(pScrn);
    
    if (pBIOSInfo->IGA1SettingInfo.IsDISP3DScaling)
    {
        x = pVia->pBIOSInfo->IGA1SettingInfo.DISP3DSCALCTRLInfo.CursorX;
        y = pVia->pBIOSInfo->IGA1SettingInfo.DISP3DSCALCTRLInfo.CursorY;    
    }
    if(pBIOSInfo->IGA2SettingInfo.IsDISP3DScaling)
    {
        x = pVia->pBIOSInfo->IGA2SettingInfo.DISP3DSCALCTRLInfo.CursorX;
        y = pVia->pBIOSInfo->IGA2SettingInfo.DISP3DSCALCTRLInfo.CursorY;    
    }
    VIASetCursorPosition(pScrn, x, y);
}

void hw3d_MonoCursorImageAdjust(VIAPtr pVia) {}

void hw3d_ARGBCursorImageAdjust(VIAPtr pVia, VIADISP3DSCALCTRLPtr p3DSCALCTRLInfo, 
    CARD32 srcAddr, CARD32 dstAddr, RECTL rSrc, CARD32 lSrcPitch, CARD32 dwSrcWidth,CARD32 dwSrcHeight)
{
    VIABIOSInfoPtr pBIOSInfo = pVia->pBIOSInfo;
    ScrnInfoPtr pScrn = xf86Screens[(pVia->IsSecondary)? 1 : 0];
    viaTexture3DBltSrcRec texBltSrc;
    viaTexture3DBltDstRec texBltDst;
    viaTexture3DBltRotationRec texBltRotation;
    RECTL rDest;
    CARD32  lDestPitch;
    DWORD     lSrcAddr=0,lDestAddr = 0;

    rDest.left = rDest.top = 0;
    rDest.right = rSrc.right* (p3DSCALCTRLInfo->RealHActive) /(pBIOSInfo->CrtcHDisplay);
    rDest.bottom = rSrc.bottom* (p3DSCALCTRLInfo->RealVActive)/(pBIOSInfo->CrtcVDisplay);
    lDestPitch = MAX_CURS *4;
    
    lSrcAddr = srcAddr;
    lDestAddr = dstAddr;

    memset(&texBltSrc, 0x00, sizeof(texBltSrc));
    memset(&texBltDst, 0x00, sizeof(texBltDst));
    memset(&texBltRotation, 0x00, sizeof(texBltRotation));

    texBltSrc.format = PICT_a8r8g8b8;
    texBltDst.format =  PICT_a8r8g8b8;

    texBltDst.offset = lDestAddr;
    texBltDst.pitch = lDestPitch;
    texBltDst.x = 0;
    texBltDst.y = 0;
    texBltDst.w = rDest.right;
    texBltDst.h = rDest.bottom;

    texBltSrc.offset = lSrcAddr;
    texBltSrc.pitch = lSrcPitch;
    texBltSrc.x = 0;
    texBltSrc.y = 0;
    texBltSrc.w = rSrc.right;
    texBltSrc.h = rSrc.bottom;
    texBltSrc.surfwidth = dwSrcWidth;
    texBltSrc.surfheight = dwSrcHeight;
    texBltSrc.filter = via_FilterBilinear;
    texBltSrc.memLoc = LOC_SL;

    texBltRotation.rotate = VIA_TO_RR(0);
    texBltRotation.width = 0;
    texBltRotation.height = 0;

    viaAccelTexture3DBlt(pScrn, &texBltSrc, &texBltDst, &texBltRotation, NULL);
}


/*3D DISPLAY SCaling supportting*/
void Reserve3DScalingMemory(ScrnInfoPtr pScrn)
{
    int needBufferSet;
    VIAPtr pVia = VIAPTR(pScrn);
    DevUnion* pPriv;
    VIAEntPtr pVIAEnt;
    VIAPtr    pVia1 = NULL;
    unsigned int size = 0;
    unsigned int addr = 0;

    pPriv = xf86GetEntityPrivate(pScrn->entityList[0],gVIAEntityIndex);
    pVIAEnt = pPriv->ptr;
    if(pVIAEnt->pPrimaryScrn != NULL)
    {
        pVia1 = VIAPTR(pVIAEnt->pPrimaryScrn);
    }
    
    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Reserve3DScalingMemory begin\n"));

    /*now add the double buffle mechiem to get good qualtity display when enable 3D display function*/
    if(!pVia->IsSecondary)
    {
        needBufferSet =  pVia->NumOfDevNeed3dScl;
        size = DISP_3D_SCALING_BUFFER_SIZE*DISP_3D_SCALING_BUFNUM_ONESET*(needBufferSet);
        addr = viaFBAlloctor(pScrn, size, INVERT_SEQ, "3D Scaling Buf");
        if(addr) {
            pVia->DISP3DScalingBufferStart = addr;
        }
        else {
            return ;
        }
        if ( 1 == needBufferSet)
        {
            pVia->iga1DISP3DScalingBufferStart = pVia->iga2DISP3DScalingBufferStart = pVia->DISP3DScalingBufferStart;
        }

        if (needBufferSet > 1)
        {
            pVia->iga1DISP3DScalingBufferStart = pVia->DISP3DScalingBufferStart;
            pVia->iga2DISP3DScalingBufferStart = pVia->DISP3DScalingBufferStart + DISP_3D_SCALING_BUFFER_SIZE*DISP_3D_SCALING_BUFNUM_ONESET;
        }
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Reserve3DScalingMemory , need buffer set =%d\n", needBufferSet));
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Reserve3DScalingMemory pVia->iga1DISP3DScalingBufferStart=%x, pVia->iga2DISP3DScalingBufferStart=%x\n",pVia->iga1DISP3DScalingBufferStart,pVia->iga2DISP3DScalingBufferStart));
        size = MAX_CURS * MAX_CURS * 4;
        addr = viaFBAlloctor(pScrn, size, INVERT_SEQ, "3D Scaling Cursor Buf");
        if(addr) {
            pVia->Cursor3DScalingBufferStart = addr;
        } 
        else {
            return ;
        }
    }
    /*  Needn't to reserve 3D buffer in second scrn because the first scrn has reserved all necessary buffer.
        Here Just need to save the 3D buffer address in second scrn.
    */
    else
    {
        pVia->iga1DISP3DScalingBufferStart = pVia1->iga1DISP3DScalingBufferStart;
        pVia->iga2DISP3DScalingBufferStart = pVia1->iga2DISP3DScalingBufferStart;
        pVia->DISP3DScalingBufferStart = pVia1->DISP3DScalingBufferStart;
        pVia->Cursor3DScalingBufferStart = pVia1->Cursor3DScalingBufferStart;
        
    }
    
    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Reserve3DScalingMemory exit\n"));
}


void VIABlankRectangle( VIAPtr pVia, CARD32 RectAddr, CARD32 RectDimension, CARD32 RectPitch)
{
    CARD32  GeMode = 0;
    VIABIOSInfoPtr pBIOSInfo = pVia->pBIOSInfo;

    switch(pBIOSInfo->bitsPerPixel)
    {
        case 32:
            GeMode = 0x300;
            break;
        case 16:
            GeMode = 0x100;
            break;
        default:
            break;            
    }

    WaitIdle();
    if(pVia->Chipset == VIA_VX800||pVia->Chipset== VIA_VX855)
    {
        VIASETREG(VIA_REG_GEMODE_M1, GeMode);                                          
        VIASETREG(VIA_REG_MONOPATFGC_M1, 0x00000000); 
        VIASETREG(VIA_REG_DSTBASE_M1 , (VIA_REG_GECMD + RectAddr) >> 3 );
        VIASETREG(VIA_REG_PITCH_M1, (RectPitch & 0xFFF8)<<13 );   
        VIASETREG(VIA_REG_DSTPOS_M1, 0x0);                                           
        VIASETREG(VIA_REG_DIMENSION_M1, RectDimension);                                
        VIASETREG(VIA_REG_GECMD_M1, VIA_GEC_FIXCOLOR_PAT|VIA_GEC_BLT | (0xF0 << 24));  
    }
    else
    {
        VIASETREG(VIA_REG_GEMODE, GeMode);                                          /* 0x04*/
        VIASETREG(VIA_REG_DSTCOLORKEY, 0x00000000);                                 /* 0x18*/
        VIASETREG(VIA_REG_DSTBASE , (VIA_REG_GECMD + RectAddr) >> 3 );
        VIASETREG(VIA_REG_PITCH, ((RectPitch & 0xFFF8)<<13 )|VIA_PITCH_ENABLE);    /* 0x38*/
        VIASETREG(VIA_REG_DSTPOS, 0x0);                                             /* 0x0C*/
        VIASETREG(VIA_REG_DIMENSION, RectDimension);                                /* 0x10*/
        VIASETREG(VIA_REG_GECMD, VIA_GEC_FIXCOLOR_PAT|VIA_GEC_BLT | (0xF0 << 24));  /* 0x00*/
    }
}  /* VIABlankRectangle()*/

Bool BlackSclScrnBoundary(VIAPtr pVia, VIADISP3DSCALCTRLPtr p3DSCALCTRLInfo, RECTL rDest, int index)
{
    VIABIOSInfoPtr pBIOSInfo = pVia->pBIOSInfo;
    CARD32 RectAddr = p3DSCALCTRLInfo->DISP3DScalingBufAddr[index];
    CARD32 ulTopLeftOffset;
    long dwWidth, dwHeigght;
    CARD32 Demension;
    int BytePP;
    long dwShrinkMemWidth, dwShrinkMemHeight;
    CARD32 MemPitch;

    BytePP = pBIOSInfo->bitsPerPixel >> 3;
    dwShrinkMemWidth = p3DSCALCTRLInfo->RealHActive;
    dwShrinkMemHeight = p3DSCALCTRLInfo->RealVActive;

    MemPitch = dwShrinkMemWidth * BytePP;
    MemPitch = (MemPitch+31)&~31;

    /*black left side rectangle  */
    /*we clear a little bit more to avoid align problem*/
    ulTopLeftOffset = 0; 
    dwWidth = rDest.left; 
    dwHeigght = dwShrinkMemHeight;
    Demension = (dwHeigght<<16) | dwWidth;
    if( (dwWidth > 0) && (dwHeigght > 0) )
        VIABlankRectangle( pVia, RectAddr, Demension, MemPitch);
        
    /*black right side rectangle*/
    ulTopLeftOffset = (rDest.right ) * BytePP ;
    dwWidth = dwShrinkMemWidth - rDest.right; 
    dwHeigght = dwShrinkMemHeight;
    Demension = (dwHeigght<<16) | dwWidth;
    if( (dwWidth > 0) && (dwHeigght > 0) )
            VIABlankRectangle( pVia, RectAddr+ulTopLeftOffset, Demension, MemPitch);


    /*black up rectangle*/
    ulTopLeftOffset = 0;
    dwWidth = dwShrinkMemWidth;
    dwHeigght = rDest.top;
    Demension = (dwHeigght<<16) | dwWidth;    
    if( (dwWidth > 0) && (dwHeigght > 0) )
        VIABlankRectangle( pVia, RectAddr+ulTopLeftOffset, Demension, MemPitch);

    /*draw bottom rectangle*/
    ulTopLeftOffset =  (rDest.bottom) * MemPitch;
    dwWidth = dwShrinkMemWidth;
    dwHeigght = dwShrinkMemHeight - rDest.bottom;
    Demension = (dwHeigght<<16) | dwWidth;
    if( (dwWidth > 0) && (dwHeigght > 0) )
        VIABlankRectangle( pVia, RectAddr+ulTopLeftOffset, Demension, MemPitch);

    return TRUE;


}


void patchForVidColorKey(VIAPtr pVia,CARD32 pSrc,CARD32 pDest,CARD32 ColoyKey,CARD32 pitch,
                            RECTL* prclDst,RECTL* prclClip)
{
#if  VIA_APPLY_UNIFORM_2DBLT
    ScrnInfoPtr pScrn = xf86Screens[(pVia->IsSecondary)? 1 : 0];
    via2DBltSrcRec src;
    via2DBltDstRec dst;
    via2DBltConfigRec config;

    memset(&src, 0x00, sizeof(via2DBltSrcRec));
    memset(&dst, 0x00, sizeof(via2DBltDstRec));
    memset(&config, 0x00, sizeof(via2DBltConfigRec));

    src.bpp = 32;
    src.offset = pSrc;
    src.pitch = pitch;
    src.x = prclClip->left;
    src.y = prclClip->top;
    
    dst.bpp = 32;
    dst.offset = pDest;
    dst.pitch = pitch;
    dst.colorKey = ColoyKey;
    dst.x = prclClip->left;
    dst.y = prclClip->top;
    dst.w = prclClip->right - prclClip->left;
    dst.h = prclClip->bottom - prclClip->top;

    config.srckey = TRUE;

    viaAccel2DBlt(pScrn, &src, &dst, &config);
    
#else
    CARD32   uGEMode;
    
    CARD16   height =0,width = 0;

    height = (CARD16)(prclClip->bottom - prclClip->top);
    width = (CARD16)(prclClip->right - prclClip->left);
        
    WaitIdle();
    if(pVia->Chipset == VIA_VX800||pVia->Chipset ==VIA_VX855)
    {
        uGEMode = VIAGETREG(VIA_REG_GEMODE_M1);
        VIASETREG(VIA_REG_GEMODE_M1, uGEMode|VIA_GEC_CLIP_ENABLE);    
        VIASETREG(VIA_REG_SRCBASE_M1,  pSrc >>3);    
        VIASETREG(VIA_REG_DSTBASE_M1 , pDest >> 3  );
        VIASETREG(VIA_REG_PITCH_M1, ((pitch)>>3) | ((pitch)<<13));    /* 0x38*/
        VIASETREG(VIA_REG_SRCPOS_M1, (prclClip->top << 16 | prclClip->left));   
        VIASETREG(VIA_REG_DSTPOS_M1, (prclClip->top << 16 | prclClip->left));   
        VIASETREG(VIA_REG_DIMENSION_M1, PACK2W(height - 1, width - 1));     
        VIASETREG(VIA_REG_KEYCONTROL_M1, VIA_GEC_FIXCOLOR_PAT|VIA_GEC_DECY);    
        VIASETREG(VIA_REG_SRCCOLORKEY_M1, ColoyKey);     
        VIASETREG(VIA_REG_CLIPTL_M1, PACKYX(prclClip->top,prclClip->left)); 
        VIASETREG(VIA_REG_CLIPBR_M1, PACKYX(prclClip->bottom - 1 ,prclClip->right - 1)); 
        VIASETREG(VIA_REG_GECMD_M1, VIA_GEC_FIXCOLOR_PAT | VIA_GEC_BLT | (0xcc << 24));  
        VIASETREG(VIA_REG_KEYCONTROL_M1, 0);
    }
    else
    {
        uGEMode = VIAGETREG(VIA_REG_GEMODE);
        VIASETREG(VIA_REG_GEMODE, uGEMode|VIA_GEC_CLIP_ENABLE);    
        VIASETREG(VIA_REG_SRCBASE,  pSrc >>3);    
        VIASETREG(VIA_REG_DSTBASE , pDest >> 3  );
        VIASETREG(VIA_REG_PITCH, ((pitch)>>3) | ((pitch)<<13)|VIA_PITCH_ENABLE);    /* 0x38*/

        VIASETREG(VIA_REG_SRCPOS, (prclClip->top << 16 | prclClip->left));   

        VIASETREG(VIA_REG_DSTPOS, (prclClip->top << 16 | prclClip->left));   

        VIASETREG(VIA_REG_DIMENSION, PACK2W(height - 1, width - 1));     

        VIASETREG(VIA_REG_KEYCONTROL, VIA_GEC_FIXCOLOR_PAT|VIA_GEC_DECY);    
        VIASETREG(VIA_REG_SRCCOLORKEY, ColoyKey);     

        //set clip rect here
        VIASETREG(VIA_REG_CLIPTL, PACKYX(prclClip->top,prclClip->left)); 

        VIASETREG(VIA_REG_CLIPBR, PACKYX(prclClip->bottom - 1 ,prclClip->right - 1)); 
                                             
        VIASETREG(VIA_REG_GECMD, VIA_GEC_FIXCOLOR_PAT | VIA_GEC_BLT | (0xcc << 24)); 

        VIASETREG(VIA_REG_KEYCONTROL, 0);
    }
    WaitIdle();    
 #endif   
}

Bool DISP3DScalWindowAdjust(RECTLPtr prSrc, RECTLPtr prDest, VIA3DSclSCNParasPtr pDISP3DScalScreenInfo)
{
    CARD32 dwShrinkMemWidth;
    CARD32 dwShrinkMemHeight;
    CARD32 dwSrcMemWidth, dwSrcMemHeight;
    CARD32 dwDesMemWidth, dwDesMemHeight;

    dwShrinkMemWidth = pDISP3DScalScreenInfo->XSIZEMax;
    dwShrinkMemHeight = pDISP3DScalScreenInfo->YSIZEMax;

    dwSrcMemWidth = prSrc->right - prSrc->left;
    dwSrcMemHeight = prSrc->bottom - prSrc->top;

    /*Size debug information*/
    DEBUG(xf86DrvMsg(0, X_INFO, "3DScalInfo SIZE: XSIZEMax=%ld, YSIZEMax=%ld, XSIZEMaxRange=%ld, YSIZEMaxRange=%ld\n", pDISP3DScalScreenInfo->XSIZEMax, pDISP3DScalScreenInfo->YSIZEMax, pDISP3DScalScreenInfo->XSIZEMaxRange,pDISP3DScalScreenInfo->YSIZEMaxRange));
    DEBUG(xf86DrvMsg(0, X_INFO, "3DScalInfo SIZE: XScaleStep=%ld, YScaleStep=%ld, XScaleMaximum=%ld, YScaleMaximum=%ld\n", pDISP3DScalScreenInfo->XScaleStep, pDISP3DScalScreenInfo->YScaleStep, pDISP3DScalScreenInfo->XScaleMaximum,pDISP3DScalScreenInfo->YScaleMaximum));
    DEBUG(xf86DrvMsg(0, X_INFO, "3DScalInfo SIZE: XScaleLevel=%ld, YScaleLevel=%ld, XSIZEOffset=%ld, YSIZEOffset=%ld\n", pDISP3DScalScreenInfo->XScaleLevel, pDISP3DScalScreenInfo->YScaleLevel, pDISP3DScalScreenInfo->XSIZEOffset,pDISP3DScalScreenInfo->YSIZEOffset));
    DEBUG(xf86DrvMsg(0, X_INFO, "3DScalInfo SIZE: XSIZENow=%ld, YSIZENow=%ld\n", pDISP3DScalScreenInfo->XSIZENow, pDISP3DScalScreenInfo->YSIZENow));
        
    /*Position debug information*/
    DEBUG(xf86DrvMsg(0, X_INFO, "3DScalInfo POSITION: XPOSITIONMaxRange=%ld, YPOSITIONMaxRange=%ld\n", pDISP3DScalScreenInfo->XPOSITIONMaxRange,pDISP3DScalScreenInfo->YPOSITIONMaxRange));
    DEBUG(xf86DrvMsg(0, X_INFO, "3DScalInfo POSITION: XPOSITIONStep=%ld, YPOSITIONStep=%ld, XPOSITIONMaximum=%ld, YPOSITIONMaximum=%ld\n", pDISP3DScalScreenInfo->XPOSITIONStep, pDISP3DScalScreenInfo->YPOSITIONStep, pDISP3DScalScreenInfo->XPOSITIONMaximum,pDISP3DScalScreenInfo->YPOSITIONMaximum));
    DEBUG(xf86DrvMsg(0, X_INFO, "3DScalInfo POSITION: XPOSITIONLevel=%ld, YPOSITIONLevel=%ld, XPOSITIONOffset=%ld, YPOSITIONOffset=%ld\n", pDISP3DScalScreenInfo->XPOSITIONLevel, pDISP3DScalScreenInfo->YPOSITIONLevel, pDISP3DScalScreenInfo->XPOSITIONOffset,pDISP3DScalScreenInfo->YPOSITIONOffset));
    DEBUG(xf86DrvMsg(0, X_INFO, "3DScalInfo POSITION: XPOSITIONNow=%ld, YPOSITIONNow=%ld\n", pDISP3DScalScreenInfo->XPOSITIONNow, pDISP3DScalScreenInfo->YPOSITIONNow));
    
    /*software shrink adjust windows size*/
    prDest->left += pDISP3DScalScreenInfo->XSIZEOffset;
    prDest->top += pDISP3DScalScreenInfo->YSIZEOffset;
    prDest->right -= pDISP3DScalScreenInfo->XSIZEOffset; 
    prDest->bottom -= pDISP3DScalScreenInfo->YSIZEOffset;
  
    prDest->left += pDISP3DScalScreenInfo->XPOSITIONOffset;
    prDest->right += pDISP3DScalScreenInfo->XPOSITIONOffset;
    prDest->top -= pDISP3DScalScreenInfo->YPOSITIONOffset;
    prDest->bottom -= pDISP3DScalScreenInfo->YPOSITIONOffset;

    dwDesMemWidth = prDest->right - prDest->left;
    dwDesMemHeight = prDest->bottom - prDest->top;
  
    if(prDest->left < 0)
    {   
        prSrc->left += -prDest->left * dwSrcMemWidth /dwDesMemWidth;
        prDest->left = 0;
    }
  
    if(prDest->top < 0)
    {
        prSrc->top += -prDest->top * dwSrcMemHeight / dwDesMemHeight;
        prDest->top = 0;
    }
  
    if(prDest->right > dwShrinkMemWidth)
    {
        prSrc->right -= (prDest->right -dwShrinkMemWidth) * dwSrcMemWidth /dwDesMemWidth;
        prDest->right = dwShrinkMemWidth;
    }
  
    if(prDest->bottom > dwShrinkMemHeight)
    {
        prSrc->bottom -= (prDest->bottom - dwShrinkMemHeight) * dwSrcMemHeight / dwDesMemHeight;
        prDest->bottom = dwShrinkMemHeight;
    }
    return TRUE;
}


Bool VIASetLCD3DScalScreenParas(VIABIOSInfoPtr  pBIOSInfo)
{
       VIA3DSclSCNParasPtr pLCD3DScalInfo = &(pBIOSInfo->LVDSSettingInfo.LCD3DScalInfo);
       /*Get the  physical size*/
       pLCD3DScalInfo->XSIZEMax = pBIOSInfo->LVDSSettingInfo.HActive;
       pLCD3DScalInfo->YSIZEMax = pBIOSInfo->LVDSSettingInfo.VActive;
       
       /*make adjust SIZE range 50% of physical mode, 25% up and down*/
       pLCD3DScalInfo->XSIZEMaxRange= (pBIOSInfo->LVDSSettingInfo.HActive >> 1);
       pLCD3DScalInfo->YSIZEMaxRange= (pBIOSInfo->LVDSSettingInfo.VActive >> 1);    


       pLCD3DScalInfo->XScaleStep = 3;
       pLCD3DScalInfo->YScaleStep = 3;

       pLCD3DScalInfo->XScaleMaximum = (pLCD3DScalInfo->XSIZEMaxRange / pLCD3DScalInfo->XScaleStep);
       pLCD3DScalInfo->YScaleMaximum = (pLCD3DScalInfo->YSIZEMaxRange / pLCD3DScalInfo->YScaleStep);
       
       pLCD3DScalInfo->XScaleLevel = pLCD3DScalInfo->XScaleDefault = pLCD3DScalInfo->XScaleMaximum >>1;
       pLCD3DScalInfo->YScaleLevel = pLCD3DScalInfo->YScaleDefault = pLCD3DScalInfo->YScaleMaximum >>1;
       
       pLCD3DScalInfo->XSIZEOffset = (pLCD3DScalInfo->XScaleDefault - pLCD3DScalInfo->XScaleLevel)*pLCD3DScalInfo->XScaleStep;
       pLCD3DScalInfo->YSIZEOffset = (pLCD3DScalInfo->YScaleDefault - pLCD3DScalInfo->YScaleLevel)*pLCD3DScalInfo->YScaleStep;

       pLCD3DScalInfo->XSIZENow = pLCD3DScalInfo->XSIZEMax - (pLCD3DScalInfo->XScaleLevel*pLCD3DScalInfo->XScaleStep);
       pLCD3DScalInfo->YSIZENow = pLCD3DScalInfo->YSIZEMax - (pLCD3DScalInfo->YScaleLevel*pLCD3DScalInfo->YScaleStep);

       /*make adjust POSITION range 50% of physical mode, 25% up and down       */
       pLCD3DScalInfo->XPOSITIONMaxRange = (pBIOSInfo->LVDSSettingInfo.HActive >> 1);
       pLCD3DScalInfo->YPOSITIONMaxRange = (pBIOSInfo->LVDSSettingInfo.VActive >> 1);
       
       pLCD3DScalInfo->XPOSITIONStep = 3;
       pLCD3DScalInfo->YPOSITIONStep = 3;

       pLCD3DScalInfo->XPOSITIONMaximum = (pLCD3DScalInfo->XPOSITIONMaxRange / pLCD3DScalInfo->XPOSITIONStep);
       pLCD3DScalInfo->YPOSITIONMaximum = (pLCD3DScalInfo->YPOSITIONMaxRange / pLCD3DScalInfo->YPOSITIONStep);
       
       pLCD3DScalInfo->XPOSITIONLevel = pLCD3DScalInfo->XPOSITIONDefault = pLCD3DScalInfo->XPOSITIONMaximum >>1;
       pLCD3DScalInfo->YPOSITIONLevel = pLCD3DScalInfo->YPOSITIONDefault = pLCD3DScalInfo->YPOSITIONMaximum >>1;

       pLCD3DScalInfo->XPOSITIONOffset= (pLCD3DScalInfo->XPOSITIONLevel - pLCD3DScalInfo->XPOSITIONDefault)*pLCD3DScalInfo->XPOSITIONStep;
       pLCD3DScalInfo->YPOSITIONOffset= (pLCD3DScalInfo->YPOSITIONLevel - pLCD3DScalInfo->YPOSITIONDefault)*pLCD3DScalInfo->YPOSITIONStep;

       pLCD3DScalInfo->XPOSITIONNow =( pLCD3DScalInfo->XPOSITIONLevel - pLCD3DScalInfo->XPOSITIONDefault)*pLCD3DScalInfo->XPOSITIONStep;
       pLCD3DScalInfo->YPOSITIONNow =( pLCD3DScalInfo->YPOSITIONLevel - pLCD3DScalInfo->YPOSITIONDefault)*pLCD3DScalInfo->YPOSITIONStep;

       pLCD3DScalInfo->bNeedBlack3dScalingBorders = FALSE;

       if((pBIOSInfo->LVDSSettingInfo.IsPreSetting))
       {
            pLCD3DScalInfo->XScaleLevel = pBIOSInfo->LVDSSettingInfo.XScalingLevel;
            pLCD3DScalInfo->YScaleLevel = pBIOSInfo->LVDSSettingInfo.YScalingLevel; 
            pLCD3DScalInfo->XSIZEOffset = (pLCD3DScalInfo->XScaleDefault - pLCD3DScalInfo->XScaleLevel)*pLCD3DScalInfo->XScaleStep;
            pLCD3DScalInfo->YSIZEOffset = (pLCD3DScalInfo->YScaleDefault - pLCD3DScalInfo->YScaleLevel)*pLCD3DScalInfo->YScaleStep;
            
            pLCD3DScalInfo->XPOSITIONLevel = pBIOSInfo->LVDSSettingInfo.XScalPosLevel;
            pLCD3DScalInfo->YPOSITIONLevel = pBIOSInfo->LVDSSettingInfo.YScalPosLevel;
            pLCD3DScalInfo->XPOSITIONOffset= (pLCD3DScalInfo->XPOSITIONLevel - pLCD3DScalInfo->XPOSITIONDefault)*pLCD3DScalInfo->XPOSITIONStep;
            pLCD3DScalInfo->YPOSITIONOffset= (pLCD3DScalInfo->YPOSITIONLevel - pLCD3DScalInfo->YPOSITIONDefault)*pLCD3DScalInfo->YPOSITIONStep;
       }
       return TRUE;
       
}

Bool    VIASet3DScalCtrlParas(ScrnInfoPtr pScrn, VIADISP3DSCALCTRLPtr p3DSCALCTRLInfo)
{
    int i;
    CARD32  Pitch;
    CARD32  Dimension;
    CARD32  ScalingAddress;
    VIAPtr pVia = VIAPTR(pScrn);    
    VIABIOSInfoPtr pBIOSInfo = pVia->pBIOSInfo;

    /*Initialize the 3d scaling control for display */
    p3DSCALCTRLInfo->VirtualHActive = pBIOSInfo->displayWidth;
    p3DSCALCTRLInfo->BitsPerPixel = pBIOSInfo->bitsPerPixel;
    
    Pitch = p3DSCALCTRLInfo->RealHActive*(p3DSCALCTRLInfo->BitsPerPixel>>3);

    /* 3D engine of VT3353/3409 require that the src and dst pitch must be aligned to 32 */
    Pitch = ALIGN_TO(Pitch, 32);

    p3DSCALCTRLInfo->DISP3DScalingBufPitch= Pitch;
    if (pBIOSInfo->IsSecondary)
    {
        p3DSCALCTRLInfo->OrigFBAddress = pScrn->fbOffset;
    }
    else
    {
        p3DSCALCTRLInfo->OrigFBAddress=0;
    }

    p3DSCALCTRLInfo->PanningX=0;
    p3DSCALCTRLInfo->PanningY=0;
    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "pVia->videoRambytes=%lud\n",pVia->videoRambytes));        

    /*Re-allocate the scaling buffers for 3d scaling*/
    ScalingAddress = pVia->DISP3DScalingBufferStart;    
    if(ScalingAddress)
    {       
        Dimension = (p3DSCALCTRLInfo->RealVActive<<16) | p3DSCALCTRLInfo->RealHActive;
        if (p3DSCALCTRLInfo->DISP3DScalIGAPath ==IGA1)
        {
            ScalingAddress = pVia->iga1DISP3DScalingBufferStart;
        }
        else
        {
            ScalingAddress = pVia->iga2DISP3DScalingBufferStart;
        }
        for(i=0; i < DISP_3D_SCALING_BUFNUM_ONESET;i++)
        {
            p3DSCALCTRLInfo->DISP3DScalingBufAddr[i]= ScalingAddress+ DISP_3D_SCALING_BUFFER_SIZE*i;
            /*256byte alignment for 3D scaling function*/
            p3DSCALCTRLInfo->DISP3DScalingBufAddr[i] = ((p3DSCALCTRLInfo->DISP3DScalingBufAddr[i]+ 255) & ~255);
//                VIABlankRectangle( pVia, p3DSCALCTRLInfo->DISP3DScalingBufAddr[i], Dimension, Pitch);
            DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "p3DSCALCTRLInfo->DISP3DScalingBufAddr=%x\n",p3DSCALCTRLInfo->DISP3DScalingBufAddr[i]));        
        }
    }
    return TRUE;
}

void vid3DScalColorKeyWindowAdjust(RECTL* lpSrcRectVideoWin, RECTL* lpDstRectVideoWin, HW3D_Scaling_INFO   *lpgfx3DScal_info)
{

    int OrigHActive,OrigVActive,RealHActive,RealVActive;
    int tmpLeft, tmpRight,tmpTop, tmpBottom;

    int screenX, screenY;
    OrigHActive = lpgfx3DScal_info->OrigHActive;
    OrigVActive = lpgfx3DScal_info->OrigVActive;
    RealHActive = lpgfx3DScal_info->RealHActive;
    RealVActive = lpgfx3DScal_info->RealVActive;
    
    tmpLeft = (lpSrcRectVideoWin->left)*RealHActive/OrigHActive;
    tmpRight = (lpSrcRectVideoWin->right)*RealHActive/OrigHActive;
    tmpTop = (lpSrcRectVideoWin->top)*RealVActive/OrigVActive;
    tmpBottom = (lpSrcRectVideoWin->bottom)*RealVActive/OrigVActive;
    
    screenX = RealHActive - lpgfx3DScal_info->XSIZEOffset * 2;
    screenY = RealVActive - lpgfx3DScal_info->YSIZEOffset * 2;

    tmpLeft = (screenX /2) - (RealHActive /2 - tmpLeft) * screenX /RealHActive;
    tmpRight = (screenX/2 ) - (RealHActive/2 - tmpRight) * screenX /RealHActive;
    tmpTop = (screenY/2) - (RealVActive /2 - tmpTop) * screenY /RealVActive;
    tmpBottom = (screenY/2) - (RealVActive/2  - tmpBottom) * screenY /RealVActive;
      
    /*software shrink adjust windows size*/
    tmpLeft += (lpgfx3DScal_info->XSIZEOffset + lpgfx3DScal_info->XPOSITIONOffset);
    tmpRight += (lpgfx3DScal_info->XSIZEOffset + lpgfx3DScal_info->XPOSITIONOffset);
    tmpTop +=  (lpgfx3DScal_info->YSIZEOffset - lpgfx3DScal_info->YPOSITIONOffset);
    tmpBottom +=  (lpgfx3DScal_info->YSIZEOffset - lpgfx3DScal_info->YPOSITIONOffset);

    lpDstRectVideoWin->left =(unsigned long) tmpLeft;
    lpDstRectVideoWin->right =(unsigned long) tmpRight;
    lpDstRectVideoWin->top =  (unsigned long)tmpTop;
    lpDstRectVideoWin->bottom = (unsigned long) tmpBottom;

}


void VIADISP3DScalingParasSetting(ScrnInfoPtr pScrn)
{
    CARD32 OffsetWidthByQWord, CountWidthByQWord;
    VIAPtr pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr pBIOSInfo = pVia->pBIOSInfo;
    IGASETTINGINFO     *pTargetIGASettingInfo = &(pBIOSInfo->IGA1SettingInfo);
    VIADISP3DSCALCTRLPtr p3DSCALCTRLInfo = &(pTargetIGASettingInfo->DISP3DSCALCTRLInfo);
    int panelWidth=0, panelHeight=0;

    /*Reset the setting*/
    pBIOSInfo->IGA1SettingInfo.IsDISP3DScaling = FALSE;
    pBIOSInfo->IGA2SettingInfo.IsDISP3DScaling = FALSE;     
    
    if (pBIOSInfo->Is3DScalingEnable & VIA_DEVICE_LCD)
    {
        LVDSSETTINGINFOPTR    pLVDSSettingInfo = &(pBIOSInfo->LVDSSettingInfo);
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "into VIADISP3DScalingParasSetting for LCD  :PanelID=%d, PanelSizeX=%d, PanelSizeY=%d\n",pBIOSInfo->LVDSSettingInfo.PanelSizeID, pBIOSInfo->LVDSSettingInfo.PanelSizeX, pBIOSInfo->LVDSSettingInfo.PanelSizeY));       
                 
        switch (pBIOSInfo->LVDSSettingInfo.IGAPath )
        {
            case IGA2:
                pTargetIGASettingInfo = &(pBIOSInfo->IGA2SettingInfo);
                p3DSCALCTRLInfo = & (pTargetIGASettingInfo->DISP3DSCALCTRLInfo);
                break;
            case IGA1:
            default:
                pTargetIGASettingInfo = &(pBIOSInfo->IGA1SettingInfo);
                p3DSCALCTRLInfo = &(pTargetIGASettingInfo->DISP3DSCALCTRLInfo);            
                break;
        }
        p3DSCALCTRLInfo->DISP3DScalIGAPath = pBIOSInfo->LVDSSettingInfo.IGAPath;
        pTargetIGASettingInfo->IsDISP3DScaling = TRUE;
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VIADISP3DScalingParasSetting begin\n"));
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "pLVDSSettingInfo:HActive =%d, VActive=%d, IGAPath=%d,ModeIndex =%d\n", pLVDSSettingInfo->HActive, pLVDSSettingInfo->VActive, pLVDSSettingInfo->IGAPath,pLVDSSettingInfo->ModeIndex));
         
        /*1. first to get the physical size, and store the current mode size*/
        /*Replace the LCD setting mode to native mode*/
        
        p3DSCALCTRLInfo->OrigHActive = pLVDSSettingInfo->HActive;
        p3DSCALCTRLInfo->OrigVActive = pLVDSSettingInfo->VActive;

        panelWidth = VIA_PANEL_WIDTH(pLVDSSettingInfo->PanelSizeID);
        panelHeight = VIA_PANEL_HEIGHT(pLVDSSettingInfo->PanelSizeID);

        p3DSCALCTRLInfo->RealHActive = panelWidth;
        p3DSCALCTRLInfo->RealVActive = panelHeight;

        pLVDSSettingInfo->HActive = panelWidth;
        pLVDSSettingInfo->VActive = panelHeight;

        pLVDSSettingInfo->ModeIndex = VIAGetModeIndex(panelWidth, panelHeight);

        if ((p3DSCALCTRLInfo->OrigHActive == p3DSCALCTRLInfo->RealHActive) &&
            (p3DSCALCTRLInfo->OrigVActive == p3DSCALCTRLInfo->RealVActive)) {
            pBIOSInfo->Is3DScalingEnable &= ~VIA_DEVICE_LCD;
            return;
        }    

        /*2.initial the scaling screen paras */
        VIASetLCD3DScalScreenParas(pBIOSInfo);
#if 1       
         /*Size debug information*/
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "3DScalInfo SIZE: XSIZEMax=%ld, YSIZEMax=%ld, XSIZEMaxRange=%ld, YSIZEMaxRange=%ld\n", pLVDSSettingInfo->LCD3DScalInfo.XSIZEMax, pLVDSSettingInfo->LCD3DScalInfo.YSIZEMax, pLVDSSettingInfo->LCD3DScalInfo.XSIZEMaxRange,pLVDSSettingInfo->LCD3DScalInfo.YSIZEMaxRange));
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "3DScalInfo SIZE: XScaleStep=%ld, YScaleStep=%ld, XScaleMaximum=%ld, YScaleMaximum=%ld\n", pLVDSSettingInfo->LCD3DScalInfo.XScaleStep, pLVDSSettingInfo->LCD3DScalInfo.YScaleStep, pLVDSSettingInfo->LCD3DScalInfo.XScaleMaximum,pLVDSSettingInfo->LCD3DScalInfo.YScaleMaximum));
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "3DScalInfo SIZE: XScaleLevel=%ld, YScaleLevel=%ld, XSIZEOffset=%ld, YSIZEOffset=%ld\n", pLVDSSettingInfo->LCD3DScalInfo.XScaleLevel, pLVDSSettingInfo->LCD3DScalInfo.YScaleLevel, pLVDSSettingInfo->LCD3DScalInfo.XSIZEOffset,pLVDSSettingInfo->LCD3DScalInfo.YSIZEOffset));
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "3DScalInfo SIZE: XSIZENow=%ld, YSIZENow=%ld\n", pLVDSSettingInfo->LCD3DScalInfo.XSIZENow, pLVDSSettingInfo->LCD3DScalInfo.YSIZENow));
        
        /*Position debug information*/
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "3DScalInfo POSITION: XPOSITIONMaxRange=%ld, YPOSITIONMaxRange=%ld\n", pLVDSSettingInfo->LCD3DScalInfo.XPOSITIONMaxRange,pLVDSSettingInfo->LCD3DScalInfo.YPOSITIONMaxRange));
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "3DScalInfo POSITION: XPOSITIONStep=%ld, YPOSITIONStep=%ld, XPOSITIONMaximum=%ld, YPOSITIONMaximum=%ld\n", pLVDSSettingInfo->LCD3DScalInfo.XPOSITIONStep, pLVDSSettingInfo->LCD3DScalInfo.YPOSITIONStep, pLVDSSettingInfo->LCD3DScalInfo.XPOSITIONMaximum,pLVDSSettingInfo->LCD3DScalInfo.YPOSITIONMaximum));
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "3DScalInfo POSITION: XPOSITIONLevel=%ld, YPOSITIONLevel=%ld, XPOSITIONOffset=%ld, YPOSITIONOffset=%ld\n", pLVDSSettingInfo->LCD3DScalInfo.XPOSITIONLevel, pLVDSSettingInfo->LCD3DScalInfo.YPOSITIONLevel, pLVDSSettingInfo->LCD3DScalInfo.XPOSITIONOffset,pLVDSSettingInfo->LCD3DScalInfo.YPOSITIONOffset));
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "3DScalInfo POSITION: XPOSITIONNow=%ld, YPOSITIONNow=%ld\n", pLVDSSettingInfo->LCD3DScalInfo.XPOSITIONNow, pLVDSSettingInfo->LCD3DScalInfo.YPOSITIONNow));
                  
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VIASet3DScalScreenParas exit\n")); 
#endif     
       
        /*3. initial the scaling control paras*/
        VIASet3DScalCtrlParas(pScrn, p3DSCALCTRLInfo);
       
        /*4. Update the timing factor to IGA */
        CountWidthByQWord = OffsetWidthByQWord = p3DSCALCTRLInfo->DISP3DScalingBufPitch >> 3;
        
        pTargetIGASettingInfo->IsPanning = FALSE;
        pTargetIGASettingInfo->HActive = p3DSCALCTRLInfo->RealHActive;
        pTargetIGASettingInfo->VActive = p3DSCALCTRLInfo->RealVActive;
        pTargetIGASettingInfo->OffsetWidthByQWord = OffsetWidthByQWord;
        pTargetIGASettingInfo->CountWidthByQWord = CountWidthByQWord;
        pTargetIGASettingInfo->SetStartingAddrFirstTime = TRUE;
        pTargetIGASettingInfo->IsActive = TRUE;

        /*5. Enable the 3D scaling */
        pBIOSInfo->Is3DScalingEnable |= DISP_3D_SCALING_ENABLE;  
    }
}


Bool doVIADISP3DScalingImageProcessing(VIAPtr pVia, VIADISP3DSCALCTRLPtr p3DSCALCTRLInfo, VIA3DSclSCNParasPtr pDISP3DScalScreenInfo)
{
    ScrnInfoPtr pScrn = xf86Screens[(pVia->IsSecondary)? 1 : 0];
    viaTexture3DBltSrcRec texBltSrc;
    viaTexture3DBltDstRec texBltDst;
    viaTexture3DBltRotationRec texBltRotation;
    RECTL rSrc, rDest,rDestofColorKey;
    CARD32 Panningoffset, dwSrcWidth, dwSrcHeight;
    CARD32 lSrcPitch;
    CARD32 SourcScalAddr, DestScalAddr, VidDestAddr;
    CARD32 ScalingMode = TRUE;

    Bool bVidDuoViewTwoOverlay = FALSE;
    Bool bNeedAdjust = FALSE;
    int i, bufIndexInPrecessing,numberofvideostream=0;
    int vidStreamOnIGA=0;
    RECTL rVideoSrc, rVideoDest;
    /*set the array size is 3, in fact, it maybe should be 2, need study more*/
    unsigned long VideoFlag[3];
    Bool isColorKeyOn[3];
    unsigned long ColorKey[3], tmpColorKey;
    LPOVERLAYRECORD lpTargetOverlayRecord[3];
    ViaXvRectRec dwOriDestRect[2];

    VIABIOSInfoPtr pBIOSInfo = pVia->pBIOSInfo;
    PVIDDATA pVidData = pVia->pVidData;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;
    unsigned long curIGAIndex = 0;

    LPVIDHWDIFFERENCE lpVideoHWDifference = &pVidData->VideoHWDifference;
    HW3D_Scaling_INFO *    lpGFX3DScal_info;

    Bool isH2 = TRUE;

    if(H5_UMA_CHIPID||H6_UMA_CHIPID)
        isH2 = FALSE;

   isColorKeyOn[0]=FALSE;
   isColorKeyOn[1]=FALSE;
   isColorKeyOn[2]=FALSE;

    rSrc.left = rSrc.top = 0;
    rSrc.right = pBIOSInfo->CrtcHDisplay;
    rSrc.bottom = pBIOSInfo->CrtcVDisplay;
    dwSrcWidth = pBIOSInfo->CrtcHDisplay;
    dwSrcHeight = pBIOSInfo->CrtcVDisplay;
    
    /* 3D engine of VT3353/3409 require that the src and dst pitch must be aligned to 32 */
    lSrcPitch = ALIGN_TO(p3DSCALCTRLInfo->VirtualHActive*(p3DSCALCTRLInfo->BitsPerPixel>> 3), 32);
    
    /*get the source address*/
    Panningoffset =( (p3DSCALCTRLInfo->VirtualHActive)*(p3DSCALCTRLInfo->PanningY)+ (p3DSCALCTRLInfo->PanningX))*(p3DSCALCTRLInfo->BitsPerPixel>> 3);
    SourcScalAddr = (p3DSCALCTRLInfo->OrigFBAddress) + Panningoffset;

    /*get the buffer index and address will to be processing and updated to on screen*/
    bufIndexInPrecessing=(p3DSCALCTRLInfo->DISP3DScalingBufIndex+1)%(DISP_3D_SCALING_BUFNUM_ONESET-1);
    DestScalAddr = p3DSCALCTRLInfo->DISP3DScalingBufAddr[bufIndexInPrecessing];

    rDest.left = rDest.top = 0;
    rDest.right = p3DSCALCTRLInfo->RealHActive;
    rDest.bottom = p3DSCALCTRLInfo->RealVActive;

    DISP3DScalWindowAdjust(&rSrc , &rDest, pDISP3DScalScreenInfo);

   if((viaGfxInfo->screenAttr.duoview)&&(!lpVideoHWDifference->dwOnlyOneVideoOverlay))
   {
        bVidDuoViewTwoOverlay = TRUE;
        DEBUG(xf86DrvMsg(0, X_INFO, "video special case is %d \n",bVidDuoViewTwoOverlay));
   }

   if (via_module_loaded)
  {
        numberofvideostream = vvaGet3DScalingVideoInfo(
                &VideoFlag[0],
                &isColorKeyOn[0],
                &ColorKey[0],
                &bVidDuoViewTwoOverlay,
                &dwOriDestRect[0]
                );
   }

    for(i=0; i<2; i++)
    {
        if(pVia->pPriv[i] != NULL){
            if((pVia->pPriv[i]->videoFlag & VIDEO_ACTIVE) && !pVia->pPriv[i]->redirected)
            {
                VideoFlag[i] =  pVia->pPriv[i]->videoFlag;
                curIGAIndex = pVia->pPriv[i]->curIGA-1;
                isColorKeyOn[i] = pVia->pPriv[i]->ovlInfo[curIGAIndex].dwColorKeyOn;
                ColorKey[i] = pVia->pPriv[i]->ovlInfo[curIGAIndex].dwKeyLow;
                lpTargetOverlayRecord[i] = &pVia->pPriv[i]->ovlInfo[curIGAIndex];               
                numberofvideostream++;
                DEBUG(xf86DrvMsg(0, X_INFO, "video stream is SW %d \n",i));
                DEBUG(xf86DrvMsg(0, X_INFO, "NO. %d \n",numberofvideostream));
            }
        }
    }

    /*first do the scaling using 3D engine linear mode*/
    /*if Video enable and set "GoodPerformance" option, we only do no linear 3D scaling once*/
    if(numberofvideostream && pVia->GoodPerformance)
    {
        ScalingMode = FALSE;
    }

    memset(&texBltSrc, 0x00, sizeof(texBltSrc));
    memset(&texBltDst, 0x00, sizeof(texBltDst));
    memset(&texBltRotation, 0x00, sizeof(texBltRotation));

    switch(pBIOSInfo->bitsPerPixel)
    {
        case 15:
            texBltSrc.format = PICT_x1r5g5b5;
            texBltDst.format = PICT_x1r5g5b5;
            break;
        case 16:
            texBltSrc.format = PICT_r5g6b5;
            texBltDst.format = PICT_r5g6b5;
            break;
            
        case 24:
            texBltSrc.format = PICT_x8r8g8b8;
            texBltDst.format = PICT_x8r8g8b8;
            break;
            
        case 32:
        default:
            texBltSrc.format = PICT_a8r8g8b8;
            texBltDst.format = PICT_a8r8g8b8;
            break;
    }

    /* NOTE: the first time, Blt destination address: DestScalAddr */
    texBltDst.offset = DestScalAddr;
    texBltDst.pitch = p3DSCALCTRLInfo->DISP3DScalingBufPitch;
    texBltDst.x = rDest.left;
    texBltDst.y = rDest.top;
    texBltDst.w = rDest.right-rDest.left;
    texBltDst.h = rDest.bottom-rDest.top;

    texBltSrc.offset = SourcScalAddr;
    texBltSrc.pitch = lSrcPitch;
    texBltSrc.x = rSrc.left;
    texBltSrc.y = rSrc.top;
    texBltSrc.w = rSrc.right-rSrc.left;
    texBltSrc.h = rSrc.bottom-rSrc.top;
    texBltSrc.surfwidth = isH2 ? dwSrcWidth : pVia->pBIOSInfo->SaveVirtualX;
    texBltSrc.surfheight = isH2 ? dwSrcHeight: pVia->pBIOSInfo->SaveVirtualY;
    texBltSrc.filter = ScalingMode ? via_FilterBilinear : via_FilterNearest;
    texBltSrc.memLoc = LOC_SL;

    texBltRotation.rotate = VIA_TO_RR(0);
    texBltRotation.width = 0;
    texBltRotation.height = 0;

    /* Must acquire the DRM LOCK before accessing HW 3D engine */
#ifdef XF86DRI 
    if(pVia->directRenderingEnabled)
    {
        DRILock(pScrn->pScreen, 0);
    }
#endif   
    viaAccelTexture3DBlt(pScrn, &texBltSrc, &texBltDst, &texBltRotation, NULL);
#ifdef XF86DRI
    if(pVia->directRenderingEnabled)
    {
        DRIUnlock(pScrn->pScreen);
    }
#endif  
    /***********************************************************************/
    if (!pVia->GoodPerformance && numberofvideostream)  /*The video steam is playing*/
    {
        for (i=0; i < 2; i++)
        {
            bNeedAdjust = FALSE;
            
            if(isColorKeyOn[i])  /*this video steam enable the color key*/
            {
                tmpColorKey = ColorKey[i];
                
                DEBUG(xf86DrvMsg(0, X_INFO, "Check the NO. %d video stream \n",i));

                if(bVidDuoViewTwoOverlay)  /*special case, we must to adjust the scaling*/
                {
                    bNeedAdjust = TRUE;
                }
                else  /*to check if the video is on the IGA we are scaling*/
                { 
                    if(VideoFlag[i]&VIDEO_ON_IGA2)                     
                     {
                        vidStreamOnIGA = IGA2;
                     }
                     else
                     {
                        vidStreamOnIGA = IGA1;
                     }
                     if(vidStreamOnIGA == p3DSCALCTRLInfo->DISP3DScalIGAPath)
                     {
                        bNeedAdjust = TRUE;
                     }
                }
            }
            
            if(bNeedAdjust)  /*need to do the adjust for video     */
            {
                VidDestAddr = p3DSCALCTRLInfo->DISP3DScalingBufAddr[2];
                
                memset(&texBltSrc, 0x00, sizeof(texBltSrc));
                memset(&texBltDst, 0x00, sizeof(texBltDst));
                memset(&texBltRotation, 0x00, sizeof(texBltRotation));

                switch(pBIOSInfo->bitsPerPixel)
                {
                    case 15:
                        texBltSrc.format = PICT_x1r5g5b5;
                        texBltDst.format = PICT_x1r5g5b5;
                        break;
                    case 16:
                        texBltSrc.format = PICT_r5g6b5;
                        texBltDst.format = PICT_r5g6b5;
                        break;
                        
                    case 24:
                        texBltSrc.format = PICT_x8r8g8b8;
                        texBltDst.format = PICT_x8r8g8b8;
                        break;
                        
                    case 32:
                    default:
                        texBltSrc.format = PICT_a8r8g8b8;
                        texBltDst.format = PICT_a8r8g8b8;
                        break;
                }

                /* NOTE: the second time, Blt destination address: VidDestAddr */
                texBltDst.offset = VidDestAddr;
                texBltDst.pitch = p3DSCALCTRLInfo->DISP3DScalingBufPitch;
                texBltDst.x = 0;
                texBltDst.y = 0;
                texBltDst.w = rDest.right;
                texBltDst.h = rDest.bottom;

                texBltSrc.offset = SourcScalAddr;
                texBltSrc.pitch = lSrcPitch;
                texBltSrc.x = 0;
                texBltSrc.y = 0;
                texBltSrc.w = rSrc.right;
                texBltSrc.h = rSrc.bottom;
                texBltSrc.surfwidth = isH2 ? dwSrcWidth : pVia->pBIOSInfo->SaveVirtualX;
                texBltSrc.surfheight = isH2 ? dwSrcHeight: pVia->pBIOSInfo->SaveVirtualY;
                texBltSrc.filter = via_FilterNearest;
                texBltSrc.memLoc = LOC_SL;

                texBltRotation.rotate = VIA_TO_RR(0);
                texBltRotation.width = 0;
                texBltRotation.height = 0;
                
                /* Must acquire the DRM LOCK before accessing HW 3D engine */
#ifdef XF86DRI 
                if(pVia->directRenderingEnabled)
                {
                    DRILock(pScrn->pScreen, 0);
                }
#endif  
                viaAccelTexture3DBlt(pScrn, &texBltSrc, &texBltDst, &texBltRotation, NULL);
#ifdef XF86DRI
                if(pVia->directRenderingEnabled)
                {
                    DRIUnlock(pScrn->pScreen);
                }
#endif                   
                /*get the desternation video window position parameters*/
                switch(p3DSCALCTRLInfo->DISP3DScalIGAPath)
                {
                    case IGA2:
                        lpGFX3DScal_info = &(pVidData->viaGfxInfo->igaInfo[IGA2-1].igagfx3DScaling_info);
                        break;
                    case IGA1:
                      default:
                        lpGFX3DScal_info = &(pVidData->viaGfxInfo->igaInfo[IGA1-1].igagfx3DScaling_info);
                        break;
                }
                if(pVia->pPriv[i] != NULL){
                    if ((pVia->pPriv[i]->videoFlag & VIDEO_ACTIVE) && !pVia->pPriv[i]->redirected){
                            rVideoSrc.left = (long)lpTargetOverlayRecord[i]->dwOriDestRect.left;
                            rVideoSrc.top = (long)lpTargetOverlayRecord[i]->dwOriDestRect.top;
                            rVideoSrc.right = (long)lpTargetOverlayRecord[i]->dwOriDestRect.right;
                            rVideoSrc.bottom = (long)lpTargetOverlayRecord[i]->dwOriDestRect.bottom;
                    }
                }else if (via_module_loaded){
                        rVideoSrc.left = (long)dwOriDestRect[i].left;
                        rVideoSrc.top = (long)dwOriDestRect[i].top;
                        rVideoSrc.right = (long)dwOriDestRect[i].right;
                        rVideoSrc.bottom = (long)dwOriDestRect[i].bottom;
                }

                rVideoSrc.left = max(0, rVideoSrc.left);
                rVideoSrc.top = max(0, rVideoSrc.top);
                rVideoSrc.right = min(p3DSCALCTRLInfo->OrigHActive, rVideoSrc.right);
                rVideoSrc.bottom= min(p3DSCALCTRLInfo->OrigVActive, rVideoSrc.bottom);
                vid3DScalColorKeyWindowAdjust(&rVideoSrc , &rVideoDest, lpGFX3DScal_info);

                rVideoDest.left    = range_limit(rVideoDest.left, 0, p3DSCALCTRLInfo->RealHActive);
                rVideoDest.top     = range_limit(rVideoDest.top, 0, p3DSCALCTRLInfo->RealVActive);
                rVideoDest.right   = range_limit(rVideoDest.right, 0, p3DSCALCTRLInfo->RealHActive);
                rVideoDest.bottom  = range_limit(rVideoDest.bottom, 0, p3DSCALCTRLInfo->RealVActive);

                rDestofColorKey.left = rDestofColorKey.top =0;
                rDestofColorKey.right = p3DSCALCTRLInfo->RealHActive;
                rDestofColorKey.bottom = p3DSCALCTRLInfo->RealVActive;

                /* NOTE: Blt Color key from VidDest Addr to DestScalAddr with 2D engine */
                patchForVidColorKey(pVia,VidDestAddr,DestScalAddr, tmpColorKey,\
                p3DSCALCTRLInfo->DISP3DScalingBufPitch,&rDestofColorKey,&rVideoDest);

            }
        }

    }

    /*clean the shrink buffer if adjusted the size or position*/
    if(pDISP3DScalScreenInfo->bNeedBlack3dScalingBorders ==  TRUE)
    {
        for(i=0; i<DISP_3D_SCALING_BUFNUM_ONESET-1;i++)
        {
            BlackSclScrnBoundary(pVia, p3DSCALCTRLInfo, rDest,i);
        }
        pDISP3DScalScreenInfo->bNeedBlack3dScalingBorders = FALSE;
    }

    /*update the current display buffer index*/
    p3DSCALCTRLInfo->DISP3DScalingBufIndex = bufIndexInPrecessing;
    /*now with the double buffers, we need to show the stabel display,so first switch to the good one*/
    /*temply solution, it seems that the VT3364 MMIO setting staraddress have some issue, will study it laterly*/
    /* Unlock Extended Regs */
    VGAOUT8(0x3c4, 0x10);
    VGAOUT8(0x3c5, VGAIN8(0x3c5) | 0x01);
    /* Unlock CRTC register protect */
    VGAOUT8(0x3d4, 0x47);
    VGAOUT8(0x3d5, VGAIN8(0x3d5) & ~0x01);

    switch(p3DSCALCTRLInfo->DISP3DScalIGAPath)
    {
        case IGA1:
            VIASETREG(0x214, DestScalAddr>>1);
            break;
        case IGA2:
            VIASETREG(0x218, DestScalAddr);
            break;
    }

    return TRUE;
}

Bool VIADISP3DScalingImageProcessing(VIAPtr pVia)
{
    VIABIOSInfoPtr pBIOSInfo = pVia->pBIOSInfo;
    VIADISP3DSCALCTRLPtr p3DSCALCTRLInfo = NULL;
    VIA3DSclSCNParasPtr pDISP3DScalScreenInfo = NULL;
          
    if (pBIOSInfo->Is3DScalingEnable & VIA_DEVICE_LCD)
    {
        switch (pBIOSInfo->LVDSSettingInfo.IGAPath )
        {
            case IGA2:
                p3DSCALCTRLInfo =& (pBIOSInfo->IGA2SettingInfo.DISP3DSCALCTRLInfo);
                break;
            case IGA1:
            default:
                p3DSCALCTRLInfo =& (pBIOSInfo->IGA1SettingInfo.DISP3DSCALCTRLInfo);
                break;
        }
        pDISP3DScalScreenInfo = &(pBIOSInfo->LVDSSettingInfo.LCD3DScalInfo);
        doVIADISP3DScalingImageProcessing(pVia, p3DSCALCTRLInfo, pDISP3DScalScreenInfo);        
    } 

    return TRUE;
} 

