 /*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
 

/*************************************************************************
 *
 *  File:       via_accel.c
 *  Content:    2D acceleration function for VIA/S3G UniChrome
 *
 ************************************************************************/

#include "via_driver.h"
#include "xaalocal.h"	/* Include dir /usr/include/xorg/ */
#include "xaarop.h"
#include "misc.h"       /* For MACRO max() */
#include "via_regs.h"
#include "via_eng_regs.h"
#include "via_rotate.h" /* For rotate feature */
#include "agpctl.h"
#include "via_xaa.h"
#include "via_exa.h"
#include "via_dmabuffer.h"

void VIAEnableVQ(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    CARD32  dwVQStartAddr, dwVQEndAddr;
    CARD32  dwVQLen, dwVQStartL, dwVQEndL, dwVQStartEndH;

    if(b3DRegsInitialized==0)
    {
        /* Enable VQ */
        dwVQStartAddr = pVia->VQStart;
        dwVQEndAddr = pVia->VQEnd;
        switch ( pVia->Chipset )
        {
            case VIA_K8M890:
            case VIA_P4M900: 
    		case VIA_VX800:
            case VIA_VX855:
                dwVQStartL = 0x70000000 | (dwVQStartAddr & 0xFFFFFF);
                dwVQEndL = 0x71000000 | (dwVQEndAddr & 0xFFFFFF);
                dwVQStartEndH = 0x72000000 | ((dwVQStartAddr & 0xFF000000) >> 24) |
                        ((dwVQEndAddr & 0xFF000000) >> 16);
                dwVQLen = 0x73000000 | (VIA_VQ_SIZE >> 3);
                break;

            default:
                dwVQStartL = 0x50000000 | (dwVQStartAddr & 0xFFFFFF);
                dwVQEndL = 0x51000000 | (dwVQEndAddr & 0xFFFFFF);
                dwVQStartEndH = 0x52000000 | ((dwVQStartAddr & 0xFF000000) >> 24) |
                        ((dwVQEndAddr & 0xFF000000) >> 16);
                dwVQLen = 0x53000000 | (VIA_VQ_SIZE >> 3);
                break; 
        }

        switch ( pVia->Chipset )
        { 
            case VIA_K8M890:
            case VIA_P4M900:
    		case VIA_VX800:
            case VIA_VX855:
                VIASETREG(0x41c, 0x00100000);
                VIASETREG(0x420, dwVQStartEndH);
                VIASETREG(0x420, dwVQStartL);
                VIASETREG(0x420, dwVQEndL);
                VIASETREG(0x420, dwVQLen);
                VIASETREG(0x420, 0x74301001);
                VIASETREG(0x420, 0x00000000);
                break;
            
            default: 
                VIASETREG(0x43c, 0x00fe0000);
                VIASETREG(0x440, 0x080003fe);
                VIASETREG(0x440, 0x0a00027c);
                VIASETREG(0x440, 0x0b000260);
                VIASETREG(0x440, 0x0c000274);
                VIASETREG(0x440, 0x0d000264);
                VIASETREG(0x440, 0x0e000000);
                VIASETREG(0x440, 0x0f000020);
                VIASETREG(0x440, 0x1000027e);
                VIASETREG(0x440, 0x110002fe);
                VIASETREG(0x440, 0x200f0060);

                VIASETREG(0x440, 0x00000006);
                VIASETREG(0x440, 0x40008c0f);
                VIASETREG(0x440, 0x44000000);
                VIASETREG(0x440, 0x45080c04);
                VIASETREG(0x440, 0x46800408);

                VIASETREG(0x440, dwVQStartEndH);
                VIASETREG(0x440, dwVQStartL);
                VIASETREG(0x440, dwVQEndL);
                VIASETREG(0x440, dwVQLen);
                break;

        }
        
    }
}

void VIADisableVQ(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    /* Diable VQ */
    switch ( pVia->Chipset)
    {
        case VIA_K8M890: 	      
        case VIA_P4M900:
        case VIA_VX800:
        case VIA_VX855:
            break;
        
        default: 
            VIASETREG(0x43c, 0x00fe0000);
            VIASETREG(0x440, 0x00000004);
            VIASETREG(0x440, 0x40008c0f);
            VIASETREG(0x440, 0x44000000);
            VIASETREG(0x440, 0x45080c04);
            VIASETREG(0x440, 0x46800408);
            break;
    }   
}


void
VIAInitialize2DEngine(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    CARD32  dwGEMode;

    /* init 2D engine regs to reset 2D engine */
    VIASETREG(0x04, 0x0);
    VIASETREG(0x08, 0x0);
    VIASETREG(0x0c, 0x0);
    VIASETREG(0x10, 0x0);
    VIASETREG(0x14, 0x0);
    VIASETREG(0x18, 0x0);
    VIASETREG(0x1c, 0x0);
    VIASETREG(0x20, 0x0);
    VIASETREG(0x24, 0x0);
    VIASETREG(0x28, 0x0);
    VIASETREG(0x2c, 0x0);
    VIASETREG(0x30, 0x0);
    VIASETREG(0x34, 0x0);
    VIASETREG(0x38, 0x0);
    VIASETREG(0x3c, 0x0);
    VIASETREG(0x40, 0x0);

    if (pVia->Chipset == VIA_VX800 || pVia->Chipset == VIA_VX855)
    {
        VIASETREG(0x44, 0x0);
        VIASETREG(0x48, 0x0);
        VIASETREG(0x4C, 0x0);
        VIASETREG(0x50, 0x0);
        VIASETREG(0x54, 0x0);
        VIASETREG(0x58, 0x0);
        VIASETREG(0x5C, 0x0);
    }

    if(b3DRegsInitialized==0)   
    {
        /*cliff add this switch because doing this vexp will hang when chipset is K8M800*/
        /* Init AGP and VQ regs */
        switch( pVia->Chipset )  
        {
            case VIA_K8M890: 
            case VIA_P4M900:
			case VIA_VX800:
            case VIA_VX855:
                VIASETREG(0x41c, 0x00100000);
                VIASETREG(0x420, 0x680A0000);
                VIASETREG(0x420, 0x02000000);
                break; 

            default: 
                VIASETREG(0x43c, 0x00100000);
                VIASETREG(0x440, 0x00000000);
                VIASETREG(0x440, 0x00333004);
                VIASETREG(0x440, 0x60000000);
                VIASETREG(0x440, 0x61000000);
                VIASETREG(0x440, 0x62000000);
                VIASETREG(0x440, 0x63000000);
                VIASETREG(0x440, 0x64000000);
                VIASETREG(0x440, 0x7D000000);
                VIASETREG(0x43c, 0xfe020000);
                VIASETREG(0x440, 0x00000000);
                break;
        }  
        
    }

    if (pVia->VQEnable && pVia->VQStart)
    {
        VIAEnableVQ(pScrn);
    }
    else
    {
        switch ( pVia->Chipset)
        {
            case VIA_K8M890: 	      
            case VIA_P4M900:
            case VIA_VX800:
            case VIA_VX855:
                VIASETREG(0x41c, 0x00100000);
                VIASETREG(0x420, 0x74301000);
                break;
            
            default: 
                VIASETREG(0x43c, 0x00fe0000);
                VIASETREG(0x440, 0x00000004);
                VIASETREG(0x440, 0x40008c0f);
                VIASETREG(0x440, 0x44000000);
                VIASETREG(0x440, 0x45080c04);
                VIASETREG(0x440, 0x46800408);
                break;
        }  
    }

    dwGEMode = 0;

    switch (pScrn->bitsPerPixel)
    {
        case 16:
            dwGEMode |= VIA_GEM_16bpp;
            break;
        case 32:
            dwGEMode |= VIA_GEM_32bpp;
            break;
        default:
            dwGEMode |= VIA_GEM_8bpp;
            break;
    }

    
    if (pVia->Chipset == VIA_VX800 || pVia->Chipset == VIA_VX855)
    {
        /* Set BPP and Pitch */
        VIASETREG(VIA_REG_GEMODE_M1, dwGEMode);
        /* Set Src and Dst base address and pitch, pitch is qword */
        VIASETREG(VIA_REG_SRCBASE_M1, 0x0 + (pScrn->fbOffset >> 3));
        VIASETREG(VIA_REG_DSTBASE_M1, 0x0 + (pScrn->fbOffset >> 3));
        VIASETREG(VIA_REG_PITCH_M1, ((pScrn->displayWidth * pScrn->bitsPerPixel >> 3) >> 3) |
                  (((pScrn->displayWidth * pScrn->bitsPerPixel >> 3) >> 3) << 16));
    }
    else
    {
        VIASETREG(VIA_REG_GEMODE, dwGEMode);

        /* Set Src and Dst base address and pitch, pitch is qword */
        /*=* Add fbOffset for IGA2 2D hardware acceleration
         *   function. If IGA1, fbOffset will be zero. *=*/    
        VIASETREG(VIA_REG_SRCBASE, 0x0 + (pScrn->fbOffset >> 3));
        VIASETREG(VIA_REG_DSTBASE, 0x0 + (pScrn->fbOffset >> 3));
        VIASETREG(VIA_REG_PITCH, VIA_PITCH_ENABLE |
                  ((pScrn->displayWidth * pScrn->bitsPerPixel >> 3) >> 3) |
                  (((pScrn->displayWidth * pScrn->bitsPerPixel >> 3) >> 3) << 16));

        /* Set rotation temporary memory base and pitch. */
        if (pVia->IsHWRotateEnabled)
        {
            VIASETREG(VIA_REG_ROT_TMP_ADDR, pVia->RotateFBStart);
            VIASETREG(VIA_REG_ROT_TMP_PITCH, (pScrn->displayWidth * (pScrn->bitsPerPixel/8))>>3);
        }
    }   
}

/* Acceleration init function, sets up pointers to our accelerated functions */
Bool
VIAInitAccel(ScreenPtr pScreen)
{
    ScrnInfoPtr     pScrn = xf86Screens[pScreen->myNum];
    VIAPtr          pVia = VIAPTR(pScrn);
    ScrnInfoPtr pScrn0 = xf86Screens[0];
    VIAPtr pVia0 = VIAPTR(pScrn0);
    BoxRec AvailFBArea;
	Bool ret;
	unsigned int addr = 0;

    /*  HW Limitation are described here:
     *
     *  1. H2/H5/H6 2D source and destination:
     *     Pitch: (1 << 14) - 1 = 16383
     *     Dimension: (1 << 12) = 4096
     *     X, Y position: (1 << 12) - 1 = 4095.
     *
     *  2. H2 3D engine Render target:
     *     Pitch: (1 << 14) - 1 = 16383
     *     Clip Rectangle: 0 - 2047
     *
     *  3. H5/H6 3D engine Render target:
     *     Pitch: ((1 << 10) - 1)*32 = 32736
     *     Clip Rectangle: Color Window, 12bits. As Spec saied: 0 - 2048
     *                     Scissor is the same as color window.
     * */

    if(!pVia->IsSecondary)
    {
        pVia->VQStart = 0;
        if (pVia->VQEnable) {
            /* Reserved space for VQ */
            addr = viaFBAlloctor(pScrn, VIA_VQ_SIZE, INVERT_SEQ, "Accel VQ Buf");
            if(addr) {
                pVia->VQStart = addr;
                pVia->VQEnd = pVia->VQStart + VIA_VQ_SIZE - 1;
            }
            else {
                 return FALSE;
            }
        }

        /* Reserved space for rotate temporary memory */
        if (pVia->Chipset < VIA_VX800)
        {
            if (pVia->IsHWRotateEnabled)
            {        
                addr = viaFBAlloctor(pScrn, (pScrn->virtualX * pScrn->virtualY * (pScrn->bitsPerPixel/8)), 
                                                   INVERT_SEQ, "HW Rotate temp Buf");
                if(addr) {
                    pVia->RotateFBStart = addr;
                    VIASETREG(VIA_REG_ROT_TMP_ADDR, pVia->RotateFBStart);
                } 
                else {
                    return FALSE;
                }
            }
        }
    }
    /*  The second Scrn just need to copy the reserved buffer address from the first scrn */
    else
    {
        pVia->VQStart = pVia0->VQStart;
        pVia->VQEnd = pVia0->VQEnd;
        if (pVia->Chipset != VIA_VX800 && pVia->Chipset != VIA_VX855)
        {
            pVia->RotateFBStart = pVia0->RotateFBStart;
        }
    }
    /*
    After RotateFB alloc,it must set to hw! Or the Blt use in rotate,it will also draw in
    on the point before rotate(err place), besides point after rotate(right place)
    */
    /* Set up screen parameters. */
    pVia->Bpp = pScrn->bitsPerPixel >> 3;
    pVia->Bpl = pScrn->displayWidth * pVia->Bpp;

#ifdef VIA_HAVE_EXA
#ifdef XF86DRI
    pVia->texAddr = NULL;
    pVia->dBounce = NULL;
#endif /* XF86DRI */
    if (pVia->useEXA) {

    /* Sync marker space. */
    addr = viaFBAlloctor( pScrn, 32, INVERT_SEQ, "EXA Sync Marker");
    if(addr) { 
        pVia->markerOffset = addr;
    }
    else {
        return FALSE;    
    }
    pVia->markerOffset = ( pVia->markerOffset + 31) & ~31;
    pVia->markerBuf = (volatile CARD32 *) ((char *)pVia->FBBase + pVia->markerOffset);
    *pVia->markerBuf = 0;
    pVia->curMarker = 0;
    pVia->lastMarkerRead = 0;
		
        pVia->exaDriverPtr = viaInitExa(pScreen);
        if (!pVia->exaDriverPtr) {

            /*
             * Docs recommend turning off also Xv here, but we handle this
             * case with the old linear offscreen FB manager through
             * VIAInitLinear.
             */

            pVia->NoAccel = TRUE;
            return FALSE;
        }

        xf86DrvMsg(pScrn->scrnIndex, X_INFO,
                   "[EXA] Trying to enable EXA acceleration.\n");

        return TRUE;
    }
#endif /* VIA_HAVE_EXA */

    /* No EXA path :
       Init the 3D pipeline API for the uniform 3D texture blting interface */
    viaExa3DInit(pScreen);
    viaExa2DInit(pScreen);
    /* Sync marker space. */
    addr = viaFBAlloctor( pScrn, 32, INVERT_SEQ, "CMD Sync Marker");
    if(addr) { 
        pVia->markerOffset = addr;
    }
    else {
        return FALSE;    
    }
    pVia->markerOffset = ( pVia->markerOffset + 31) & ~31;
    pVia->markerBuf = (volatile CARD32 *) ((char *)pVia->FBBase + pVia->markerOffset);
    *pVia->markerBuf = 0;
    pVia->curMarker = 0;
    pVia->lastMarkerRead = 0;

    /* We reserve space for pixel cache */
    pVia->ScissB = (pVia->FBFreeStart + VIA_PIXMAP_CACHE_SIZE) / pVia->Bpl;        

    /* Currently, this value should not be bigger than 4095, becuase there is only 12 bits to
     * store the X/Y-position value (SRC and DST)for Bitblt hardware accelerator.
     * if bigger than 4095, PIXMAP acceleration (e.g.VIASubsequentScreenToScreenCopy())
     * may be abnormal.
     */
    if (pVia->ScissB > 4095)
        pVia->ScissB = 4095;

    /*
     * Finally, we set up the memory space available to the pixmap
     * cache.
     */
    AvailFBArea.x1 = 0;
    AvailFBArea.y1 = 0;
    AvailFBArea.x2 = pScrn->displayWidth;        

    if (pVia->IsHWRotateEnabled)   
    {        
        /* In noRanR rotate in 90 or 270 degree, pScrn->virutalX, virtualY are swapped 
        */
        if( (pVia->RotateDegree == VIA_ROTATE_DEGREE_90) || 
            (pVia->RotateDegree == VIA_ROTATE_DEGREE_270)) {
            AvailFBArea.x2 = pScrn->virtualX;
        }

        AvailFBArea.y2 = pScrn->virtualY + 1;
        pVia->FBFreeStart = (AvailFBArea.y2 + 1) * pVia->Bpl;

    }
    else /* Non-rotate */
    {
        AvailFBArea.y2 = pVia->ScissB;
        pVia->FBFreeStart = (AvailFBArea.y2 + 1) * pVia->Bpl;
    }
    
    /*
    *   Initialization of the XFree86 framebuffer manager is done via
    *   Bool xf86InitFBManager(ScreenPtr pScreen, BoxPtr FullBox) 
    *   FullBox represents the area of the framebuffer that the manager
    *   is allowed to manage. This is typically a box with a width 
    *   of pScrn->displayWidth and a height of as many lines as can be fit
    *   within the total video memory
    */
    ret = xf86InitFBManager(pScreen, &AvailFBArea);
	if(ret != TRUE) {
    	xf86DrvMsg(pScrn->scrnIndex, X_ERROR,"VIAInitAccel xf86InitFBManager init failed\n");
	}
    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO,
               "Frame Buffer From (%d,%d) To (%d,%d)\n",
               AvailFBArea.x1, AvailFBArea.y1, AvailFBArea.x2, AvailFBArea.y2));
    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO,
               "Using %d lines for offscreen memory.\n",
               AvailFBArea.y2));
        
    return VIAInitXAA(pScreen);
}

void
viaFinishInitAccel(ScreenPtr pScreen)
{
	ScrnInfoPtr     pScrn = xf86Screens[pScreen->myNum];
    VIAPtr          pVia = VIAPTR(pScrn);

#ifdef XF86DRI
    int size, ret;
    if (pVia->directRenderingEnabled && pVia->useEXA &&  \
						!(H5_UMA_CHIPID || H6_UMA_CHIPID)) {
        if (!pVia->IsPCI) {
            /* Allocate upload and scratch space. */
            size = VIA_AGP_UPL_SIZE * 2 + 32;
            pVia->texAGPBuffer.context = 1;
            pVia->texAGPBuffer.size = size;
            pVia->texAGPBuffer.type = VIA_MEM_AGP;
            ret = drmCommandWriteRead(pVia->drmFD, DRM_VIA_ALLOCMEM,
                                      &pVia->texAGPBuffer,
                                      sizeof(drm_via_mem_t));

            if (ret || size != pVia->texAGPBuffer.size) {
                 pVia->texAGPBuffer.size = 0;
         	   } else {
               DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO,
                           "Allocated %u kiB of AGP memory for "
                           "system-to-framebuffer transfer.\n",
                           size / 1024));
                pVia->texOffset = (pVia->texAGPBuffer.offset + 31) & ~31;
                pVia->texAddr = (char *)pVia->agpMappedAddr + pVia->texOffset;
            }
        }
    }
#endif /* XF86DRI */

	if(!pVia->texAddr && pVia->useEXA){
		pVia->texFBBuffer = exaOffscreenAlloc(pScreen, VIA_AGP_UPL_SIZE * 2,
                                  256, TRUE, NULL, NULL);
		if(pVia->texFBBuffer)
			{
			pVia->texOffset = pVia->texFBBuffer->offset;
			pVia->texAddr = (char*)pVia->FBBase + pVia->texOffset;
		  	DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO,
                       "Allocated %u kiB of framebuffer memory for "
                       "EXA texture area.\n", VIA_AGP_UPL_SIZE*2));
			}
	}
	
	if (Success != viaSetupCBuffer(pScrn, &pVia->cb, 0)) 
	{
        pVia->NoAccel = TRUE;
        viaExitAccel(pScreen);
        return;
    }
}

void
viaExitAccel(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);

    viaAccelSync(pScrn);
    viaTearDownCBuffer(&pVia->cb);

#ifdef VIA_HAVE_EXA
    if (pVia->useEXA) {
		
#ifdef XF86DRI
        if (pVia->directRenderingEnabled) {
            if (pVia->texAddr) {
                drmCommandWrite(pVia->drmFD, DRM_VIA_FREEMEM,
                                &pVia->texAGPBuffer, sizeof(drm_via_mem_t));
                pVia->texAddr = NULL;
            }
        }
#endif /* XF86DRI */
		
		if (pVia->texAddr) {
            exaOffscreenFree(pScreen, pVia->texFBBuffer);
            pVia->texAddr = NULL;
        }
		
        if (pVia->exaDriverPtr) {
            exaDriverFini(pScreen);
        }
        xfree(pVia->exaDriverPtr);
        pVia->exaDriverPtr = NULL;
        return;
    }
#endif /* VIA_HAVE_EXA */
    if (pVia->AccelInfoRec) {
        XAADestroyInfoRec(pVia->AccelInfoRec);
        pVia->AccelInfoRec = NULL;
    }
}


