/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*************************************************************************
 *
 *  File:       via_bios.c
 *  Content:    Get the mode table from VGA BIOS and set mode by this table
 *
 ************************************************************************/

#include "via_driver.h"
#include "hw.h"
#include "xf86Priv.h"
#include "via_refresh.h"
#include "via_rotate.h"
#include "via_serial.h"
#include "via_lcd.h"

/* Functions Prototype Declaration */
Bool IsValidMode(CARD16 HDisplay, CARD16 VDisplay);

/* Extern Functions Prototypes Declaration */
extern __inline__ void write_reg(CARD8 index, CARD16 io_port, CARD8 data);
extern void write_reg_mask(CARD8 index, int io_port, CARD8 data, CARD8 mask);
extern __inline__ CARD8 read_reg(int io_port, CARD8 index);
extern int SearchModeSetting(int ModeInfoIndex);
extern void write_regx(struct io_reg RegTable[],int ItemNum);
extern void unlock_crt(void);
extern void lock_crt(void);
extern void fill_crtc_timing(VIABIOSInfoPtr pBIOSInfo, struct crt_mode_table *crt_table, int mode_array, int bpp_byte, int device, int set_iga, Bool use_modeline);
extern void load_crtc_timing(VIABIOSInfoPtr pBIOSInfo, struct display_timing device_timing, int set_iga);
extern CARD32 get_clk_value(VIABIOSInfoPtr pBIOSInfo, int clk);
extern void SetVCLK(VIABIOSInfoPtr pBIOSInfo, CARD32 CLK, int set_iga);
extern void EnableSecondDisplayChannel();
extern void DisableSecondDisplayChannel();
extern void load_offset_reg(int h_addr, int bpp_byte, int set_iga);
extern void set_color_depth(VIABIOSInfoPtr pBIOSInfo, int bpp_byte, int set_iga);

/* Support Mode, the order must follow the rule "small to big". */
struct _VIASUPPORTMODE VIASupportMode[] = {
/*  Index,         HActive, VActive*/
    {"480x640",     M480x640,     480,     640},
    {"640x480",     M640x480,     640,     480},
    {"720x480",     M720x480,     720,     480},
    {"720x576",     M720x576,     720,     576},
    {"800x480",     M800x480,     800,     480},
    {"800x600",     M800x600,     800,     600},
    {"848x480",     M848x480,     848,     480},
    {"852x480",     M852x480,     856,     480},
    {"960x600",     M960x600,     960,     600}, 
    {"1000x600",    M1000x600,    1000,    600}, 
    {"1024x512",    M1024x512,    1024,    512},  
    {"1024x600",    M1024x600,    1024,    600},     
    {"1024x576",    M1024x576,    1024,    576},        
    {"1024x768",    M1024x768,    1024,    768},
    {"1088x612",    M1088x612,    1088,    612}, 
    {"1152x720",    M1152x720,    1152,    720},
    {"1152x864",    M1152x864,    1152,    864},
    {"1200x720",    M1200x720,    1200,    720},
    {"1280x600",    M1280x600,    1280,    600},
    {"1280x720",    M1280x720,    1280,    720},
    {"1280x768",    M1280x768,    1280,    768},
    {"1280x800",    M1280x800,    1280,    800},
    {"1280x960",    M1280x960,    1280,    960},
    {"1280x1024",   M1280x1024,   1280,    1024},
    {"1360x768",    M1360x768,    1360,    768},
    {"1366x768",    M1366x768,    1366,    768},
    {"1400x1050",   M1400x1050,   1400,    1050},
    {"1440x900",    M1440x900,    1440,    900},
    {"1440x1050",   M1440x1050,   1440,    1050},
    {"1600x900",    M1600x900,    1600,    900},
    {"1600x1024",   M1600x1024,   1600,    1024},
    {"1600x1200",   M1600x1200,   1600,    1200},
    {"1680x1050",   M1680x1050,   1680,    1050},
    {"1792x1344",   M1792x1344,   1792,    1344},
    {"1856x1392",   M1856x1392,   1856,    1392},
    {"1920x1080",   M1920x1080,   1920,    1080},
    {"1920x1200",   M1920x1200,   1920,    1200},
    {"1920x1440",   M1920x1440,   1920,    1440},   
    {"2048x1536",   M2048x1536,   2048,    1536}, 
    {"720x400",     M720x400,     720,     400},
    {"832x624",     M832x624,     832,     624},
    {"",            -1,    0,       0}
};


/* Search Valid mode */
Bool IsValidMode(CARD16 HDisplay, CARD16 VDisplay)
{
    int i = 0;
    
    while (VIASupportMode[i].ModeIndex > -1)
    {
        if ((HDisplay==VIASupportMode[i].HActive)&&(VDisplay==VIASupportMode[i].VActive))
        {
			return 1;
        }
        i++;
    }
    return 0;
}

/*=*
 *
 * unsigned char VIAGetActiveDisplay(VIABIOSInfoPtr, unsigned char)
 *
 *     - Get Active Display Device
 *
 * Return Type:    unsigned char
 *
 * The Definition of Input Value:
 *
 *                 VIABIOSInfoPtr
 *
 * The Definition of Return Value:
 *
 *                 Bit[7] 2nd Path
 *                 Bit[6] 1/0 MHS Enable/Disable
 *                 Bit[5] 0 = Bypass Callback, 1 = Enable Callback
 *                 Bit[4] 0 = Hot-Key Sequence Control (OEM Specific)
 *                 Bit[3] LCD
 *                 Bit[1] CRT
 *                 Bit[0] DVI
 *=*/

unsigned char VIAGetActiveDisplay(VIABIOSInfoPtr pBIOSInfo)
{
    VIABIOSInfoPtr  pVia;
    unsigned char tmp;

    pVia = pBIOSInfo;

    tmp = (read_reg(VIACR, CR3E) & 0xF0) >> 4;

    tmp |= (read_reg(VIACR, CR3B) & 0x18) << 3;

    return tmp;
}

/*=*
 *
 * unsigned char VIAGetDeviceDetect(VIABIOSInfoPtr)
 *
 *     - Get Display Device Attched
 *
 * Return Type:    unsigned char
 *
 * The Definition of Input Value:
 *
 *                 VIABIOSInfoPtr
 *
 * The Definition of Return Value:
 *
 *                 Bit[5] Reserved
 *                 Bit[3] DFP
 *                 Bit[1] LCD
 *                 Bit[0] CRT
 *=*/

CARD32 VIAGetDeviceDetect(VIABIOSInfoPtr pBIOSInfo)
{
    CARD32   ret;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIAGetDeviceDetect\n"));

    ret = VIA_DEVICE_CRT1; /* Default assume color CRT attached */

    ret |= VIAGetDeviceDetectLCD(pBIOSInfo);
    ret |= VIAGetDeviceDetectDVI(pBIOSInfo);

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, 
                    "Connect Device is 0x%x\n", ret));
    
    return ret;
}

Bool VIAFindRefreshRate(VIABIOSInfoPtr pBIOSInfo, int ModeIndex)
{
    struct VideoModeTable   *VideoMode;
    int     i;
    int     needRefresh = pBIOSInfo->Refresh;

    if (pBIOSInfo->IsEnableInterlaceMode)
    {
        switch (ModeIndex)
        {
            case M1024x768:
            case M1152x864:
            case M1280x1024:
            case M1600x1200:
            case M1920x1080:
                ModeIndex |= 0x200;
                break;
        }
    }
    
    VideoMode = &VIARoutineModes[SearchModeSetting(ModeIndex)];

    for (i = 0; i < VideoMode->mode_array; i++)
    {
        pBIOSInfo->FoundRefresh = VideoMode->crtc[i].refresh_rate;
        
        if (VideoMode->crtc[i].refresh_rate == needRefresh)
        {            
            return TRUE;
        }
    }

    xf86DrvMsg(pBIOSInfo->scrnIndex, X_WARNING, "Can't find refresh rate %d Hz.\n", needRefresh);

    return FALSE;
}

void VIAGetPanelSizeByIndex(int PanelSizeIndex, int *pSizeX, int *pSizeY)
{
	*pSizeX = VIA_PANEL_WIDTH(PanelSizeIndex);
	*pSizeY = VIA_PANEL_HEIGHT(PanelSizeIndex);
}


int VIAGetModeIndex(int HorResolution, int VerResolution)
{
    int ModeIndex = M1024x768;
    int i = 0;    
    
    while (VIASupportMode[i].ModeIndex > -1)
    {        
        if ((HorResolution == VIASupportMode[i].HActive) &&
            (VerResolution == VIASupportMode[i].VActive))
        {
            ModeIndex = VIASupportMode[i].ModeIndex;
            break;
        }

        i++;
    }    
    
    return ModeIndex;
}


void VIAGetModeSizeByName(char *pName, int *pHSize, int *pVSize)
{
    int i = 0;

    if (!pName)
    {
        return;
    }

    while (VIASupportMode[i].ModeIndex > -1)
    {
        if (strcmp(pName, VIASupportMode[i].Name) == 0)
        {
            *pHSize = VIASupportMode[i].HActive;
            *pVSize = VIASupportMode[i].VActive;
            break;
        }

        i++;
    }
}

/*
*   Get mode size by mode index.
*/
void VIAGetModeSizeByIndex(int ModeIndex, CARD32 *pHSize, CARD32 *pVSize)
{
    int i = 0;
    
    while (VIASupportMode[i].ModeIndex > -1)
    {
        if (VIASupportMode[i].ModeIndex == ModeIndex)
        {
            *pHSize = VIASupportMode[i].HActive;
            *pVSize = VIASupportMode[i].VActive;
            break;
        }

        i++;
    }
}

CARD32 VIAGetPixelClockByMode(CARD32 ModeIndex)
{        
    CARD32 i = 0;
    CARD32 DesireCLK = 0;

    for (i=0; i<NUM_TOTAL_MODETABLE; i++)
    {
        if (ModeIndex == VIARoutineModes[i].ModeIndex)
        {
             DesireCLK = VIARoutineModes[i].crtc->clk;
        }
    }    

    return DesireCLK;
}

/* Determine whether to use reduce blanking mode. */
void VIAReduceBlanking(VIABIOSInfoPtr pBIOSInfo, CARD32 ModeIndex, CARD32 DeviceType)
{        
    CARD32 DesiredModePixelClock;    

    /* if NoDDCValue is enable, we don't know the max pixel clock of monitor. */
    if (pBIOSInfo->NoDDCValue)
        return;

    /* Only 60Hz can reduce blanking */
    if (pBIOSInfo->FoundRefresh != 60)
        return;    
    
    xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIAReduceBlanking!!\n");
    
    switch (ModeIndex)
    {   
        /* Following modes have reduce blanking mode. */
        case M1360x768:
        case M1400x1050:
        case M1440x900:
        case M1600x900:
        case M1680x1050:
        case M1920x1080:
        case M1920x1200:                                           
            break;
            
        default:              
            DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "Reduce Blanking Mode doesn't exist!\n"));
            return; 
            break;
    }
    
    /* Start to determine whether to use reduce blanking mode. */
    DesiredModePixelClock = VIAGetPixelClockByMode(ModeIndex)/1000000; /* MHz */
    
    switch (DeviceType)
    {
        case VIA_DEVICE_CRT1:
            DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO,
                    "Max Pixel Clock of CRT = %d\n", pBIOSInfo->CRTSettingInfo.MaxPixelClock));
            if ((DesiredModePixelClock > pBIOSInfo->CRTSettingInfo.MaxPixelClock) &&
                 pBIOSInfo->CRTSettingInfo.IsFlatMonitor)       
            {             
                 pBIOSInfo->CRTSettingInfo.ModeIndex |= 0x100;       
                 DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO,
                    "CRT Use CVT Reduce Blaning Mode, Mode Index = %x\n", pBIOSInfo->CRTSettingInfo.ModeIndex));
            }
            break;

        case VIA_DEVICE_DFP:
            DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO,
                    "Max Pixel Clock of DVI = %d\n!", pBIOSInfo->TMDSSettingInfo.MaxPixelClock));
            if (DesiredModePixelClock > pBIOSInfo->TMDSSettingInfo.MaxPixelClock)
            {
                pBIOSInfo->TMDSSettingInfo.ModeIndex |= 0x100;   
                DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO,
                    "DVI Use CVT Reduce Blaning Mode, Mode Index = %x\n", pBIOSInfo->TMDSSettingInfo.ModeIndex));
            }
            break;
    }
    
}

void VIAInitSettingInfo_IGA(VIABIOSInfoPtr pBIOSInfo, int DisplayPath, int IsPanning, int IsDownScaling, int HActive, int VActive, CARD16 activeDevice)

{
    int OffsetWidthByQWord, CountWidthByQWord;
    OffsetWidthByQWord = (pBIOSInfo->displayWidth * (pBIOSInfo->bitsPerPixel >> 3)) >> 3;
    CountWidthByQWord = ((HActive * (pBIOSInfo->bitsPerPixel >> 3) + 15 ) & ~15) >> 3; /* Do 16 bytes alignment */
    
    if (DisplayPath == IGA1)
    {
        pBIOSInfo->IGA1SettingInfo.IsPanning = IsPanning;
        pBIOSInfo->IGA1SettingInfo.IsDownScaling = IsDownScaling;		
        pBIOSInfo->IGA1SettingInfo.HActive = HActive;
        pBIOSInfo->IGA1SettingInfo.VActive = VActive;
        pBIOSInfo->IGA1SettingInfo.OffsetWidthByQWord = OffsetWidthByQWord;
        pBIOSInfo->IGA1SettingInfo.CountWidthByQWord = CountWidthByQWord;
        pBIOSInfo->IGA1SettingInfo.SetStartingAddrFirstTime = TRUE;
        pBIOSInfo->IGA1SettingInfo.IsActive = TRUE;
        pBIOSInfo->IGA1SettingInfo.deviceActived = activeDevice;
    }
    else if (DisplayPath == IGA2)
    {
        pBIOSInfo->IGA2SettingInfo.IsPanning = IsPanning;
		pBIOSInfo->IGA2SettingInfo.IsDownScaling = IsDownScaling;
        pBIOSInfo->IGA2SettingInfo.HActive = HActive;
        pBIOSInfo->IGA2SettingInfo.VActive = VActive;
        pBIOSInfo->IGA2SettingInfo.OffsetWidthByQWord = OffsetWidthByQWord;
        pBIOSInfo->IGA2SettingInfo.CountWidthByQWord = CountWidthByQWord;
        pBIOSInfo->IGA2SettingInfo.SetStartingAddrFirstTime = TRUE;
        pBIOSInfo->IGA2SettingInfo.IsActive = TRUE;
        pBIOSInfo->IGA2SettingInfo.deviceActived = activeDevice;        
    }
}




void VIAAdjustActiveSize_LCD(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo, int *pHActive, int *pVActive,
                          unsigned long whichLCD)
{
    int HResToSet = pBIOSInfo->CrtcHDisplay;
    int VResToSet = pBIOSInfo->CrtcVDisplay;

    /*When LCD scaling function is enable, we neednt to take care of Panning. */
    if (pBIOSInfo->Is3DScalingEnable & whichLCD)
    {
         pLVDSSettingInfo->IsPanning = FALSE; 
    }
    else
    {
        if ((HResToSet > pLVDSSettingInfo->PanelSizeX) || 
            (VResToSet > pLVDSSettingInfo->PanelSizeY))
        {
            if ((*pHActive > pLVDSSettingInfo->PanelSizeX) || 
                (*pVActive > pLVDSSettingInfo->PanelSizeY))
            {
                /* Need to adjust active size: (Panning)*/
                if (*pHActive > pLVDSSettingInfo->PanelSizeX)
                   *pHActive = pLVDSSettingInfo->PanelSizeX;
                *pVActive = pLVDSSettingInfo->PanelSizeY;
                pLVDSSettingInfo->IsPanning = TRUE;
                pLVDSSettingInfo->IsDownScaling = FALSE;
            }                            
        }
        else if ((pBIOSInfo->VirtualX > HResToSet) || (pBIOSInfo->VirtualY > VResToSet))
        {
            /* If use Virtual Desktop, and the size is larger than the resolution, */
            /* we need to do panning, too. */
            pLVDSSettingInfo->IsPanning = TRUE;
            pLVDSSettingInfo->IsDownScaling = FALSE;
        }
        else 
        {
            pLVDSSettingInfo->IsPanning = FALSE;
            pLVDSSettingInfo->IsDownScaling = FALSE;  
        }
    }
}



void VIAAdjustActiveSize_CRT(VIABIOSInfoPtr pBIOSInfo, CRTSETTINGINFOPTR pCRTSettingInfo, int* pHActive, int* pVActive)
{
    int HResToSet = pBIOSInfo->CrtcHDisplay;
    int VResToSet = pBIOSInfo->CrtcVDisplay;
	

    
    if ((HResToSet > pCRTSettingInfo->MonitorSizeH) ||
        (VResToSet > pCRTSettingInfo->MonitorSizeV))
    {
        if ((*pHActive > pCRTSettingInfo->MonitorSizeH) ||
            (*pVActive > pCRTSettingInfo->MonitorSizeV))
        {
            *pHActive = pCRTSettingInfo->MonitorSizeH;
            *pVActive = pCRTSettingInfo->MonitorSizeV;
        }

        pCRTSettingInfo->IsPanning = TRUE;
    }
    else if ((pBIOSInfo->VirtualX > HResToSet) || (pBIOSInfo->VirtualY > VResToSet))
    {
        /* If use Virtual Desktop, and the size is larger than the resolution, */
        /* we need to do panning, too. */
        pCRTSettingInfo->IsPanning = TRUE;
    }
    else
    {
        pCRTSettingInfo->IsPanning = FALSE;
    }
    
}


void VIAAdjustActiveSize_DVI(VIABIOSInfoPtr pBIOSInfo, TMDSSETTINGINFOPTR pTMDSSettingInfo, int *pHActive, int *pVActive)
{
    int HResToSet = pBIOSInfo->CrtcHDisplay;
    int VResToSet = pBIOSInfo->CrtcVDisplay;

    if ((pBIOSInfo->TMDSSettingInfo.PanelSizeH == 0) &&
        (pBIOSInfo->TMDSSettingInfo.PanelSizeV == 0) &&
        (pBIOSInfo->TMDSSettingInfo.PanelSizeH != 0xFFF))
    {
        VIAGetModeSizeByIndex(pBIOSInfo->TMDSSettingInfo.DFPSize,
                             &pBIOSInfo->TMDSSettingInfo.PanelSizeH,
                             &pBIOSInfo->TMDSSettingInfo.PanelSizeV);
    }
    
    if ((HResToSet > pBIOSInfo->TMDSSettingInfo.PanelSizeH) ||
        (VResToSet > pBIOSInfo->TMDSSettingInfo.PanelSizeV))
    {        
        if ((*pHActive > pBIOSInfo->TMDSSettingInfo.PanelSizeH) ||
            (*pVActive > pBIOSInfo->TMDSSettingInfo.PanelSizeV))
        {            
            /* Panning */
            *pHActive = pBIOSInfo->TMDSSettingInfo.PanelSizeH;
            *pVActive = pBIOSInfo->TMDSSettingInfo.PanelSizeV;
            pTMDSSettingInfo->IsPanning = TRUE;
            pTMDSSettingInfo->IsDownScaling = FALSE;
        }            
    }
    else if ((pBIOSInfo->VirtualX > HResToSet) || (pBIOSInfo->VirtualY > VResToSet))
    {
        /* If use Virtual Desktop, and the size is larger than the resolution, */
        /* we need to do panning, too. */
        pBIOSInfo->TMDSSettingInfo.IsPanning = TRUE;
    }
    else
    {
        pBIOSInfo->TMDSSettingInfo.IsPanning = FALSE;
    }

}


void VIAAdjustActiveSize(VIABIOSInfoPtr pBIOSInfo)
{
    int     HActive = pBIOSInfo->CrtcHDisplay;
    int     VActive = pBIOSInfo->CrtcVDisplay;    

    /* Try to find a suitable view size, we should use the minimum one if 
       more than one device */
    /* need to panning */

    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_CRT1)    
        VIAAdjustActiveSize_CRT(pBIOSInfo, &(pBIOSInfo->CRTSettingInfo),
                                &HActive, &VActive);
    
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_DFP)
        VIAAdjustActiveSize_DVI(pBIOSInfo, &(pBIOSInfo->TMDSSettingInfo),
                                &HActive, &VActive);    

    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD)
        VIAAdjustActiveSize_LCD(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo),
                                &HActive, &VActive, VIA_DEVICE_LCD);

    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD2)
        VIAAdjustActiveSize_LCD(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo2),
                                &HActive, &VActive, VIA_DEVICE_LCD2);  

    /* pBIOSInfo->frameX1 pBIOSInfo->frameY1 always remain the actual view size
       whenever rotation or not */
    pBIOSInfo->frameX1 = HActive - 1;
    pBIOSInfo->frameY1 = VActive - 1;
    
    pBIOSInfo->HDisplay = HActive;
    pBIOSInfo->VDisplay = VActive;
}

void VIAInitSettingInfo_CRT(VIABIOSInfoPtr pBIOSInfo, CRTSETTINGINFOPTR pCRTSettingInfo, CARD16 activeDevice)
{
    int     HActive, VActive;
    Bool    IsPanning = pCRTSettingInfo->IsPanning;
	Bool    IsDownScaling = FALSE;

    if (IsPanning)
    {
        HActive = pBIOSInfo->HDisplay;
        VActive = pBIOSInfo->VDisplay;

        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "CRT Max Resolution %dx%d, set virtual desktop!!\n", 
            pCRTSettingInfo->MonitorSizeH, pCRTSettingInfo->MonitorSizeV);
    }
    else
    {
        HActive = pBIOSInfo->CrtcHDisplay;
        VActive = pBIOSInfo->CrtcVDisplay;
    }

    /* support two crt, so we need update the relative iga path rather than the first crt's iga path */
    VIAInitSettingInfo_IGA(pBIOSInfo, pCRTSettingInfo->IGAPath, IsPanning, IsDownScaling, HActive, VActive, activeDevice);

    pCRTSettingInfo->ModeIndex = VIAGetModeIndex(HActive, VActive);
    pCRTSettingInfo->HActive = HActive;
    pCRTSettingInfo->VActive = VActive;    

    /* switch to interlace mode index. */
    if (pBIOSInfo->IsEnableInterlaceMode)
    {
        switch (pCRTSettingInfo->ModeIndex)
        {
            case M1024x768:
            case M1152x864:
            case M1280x1024:
            case M1600x1200:
            case M1920x1080:
                pCRTSettingInfo->ModeIndex |= 0x200;
                break;
        }
    }

}


void VIAInitSettingInfo_DVI(VIABIOSInfoPtr pBIOSInfo)
{
    xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIAInitSettingInfo_DVI!!\n");
    
    int     HActive, VActive;
    Bool    IsPanning = pBIOSInfo->TMDSSettingInfo.IsPanning;
    Bool    IsDownScaling = pBIOSInfo->TMDSSettingInfo.IsDownScaling;
    
    if (IsPanning)
    {
        HActive = pBIOSInfo->HDisplay;
        VActive = pBIOSInfo->VDisplay;

        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "DFP Max Resolution %lux%lu, set virtual desktop!!\n", 
            pBIOSInfo->TMDSSettingInfo.PanelSizeH, pBIOSInfo->TMDSSettingInfo.PanelSizeV);
    }
    else if (IsDownScaling)
    {
        HActive = pBIOSInfo->HorDownScaleSize;
        VActive = pBIOSInfo->VerDownScaleSize;
    }
    else
    {
        HActive = pBIOSInfo->CrtcHDisplay;
        VActive = pBIOSInfo->CrtcVDisplay;
    }

    
 	VIAInitSettingInfo_IGA (pBIOSInfo, pBIOSInfo->TMDSSettingInfo.IGAPath, IsPanning, IsDownScaling, HActive, VActive, VIA_DEVICE_DFP);

    pBIOSInfo->TMDSSettingInfo.ModeIndex = VIAGetModeIndex(HActive, VActive);
    pBIOSInfo->TMDSSettingInfo.HActive = HActive;
    pBIOSInfo->TMDSSettingInfo.VActive = VActive;        
    pBIOSInfo->IsDownScaleEnable = IsDownScaling;        

    /* switch to interlace mode index. */
    if (pBIOSInfo->IsEnableInterlaceMode)
    {
        switch (pBIOSInfo->TMDSSettingInfo.ModeIndex)
        {
            case M1024x768:
            case M1152x864:
            case M1280x1024:
            case M1600x1200:
            case M1920x1080:
                pBIOSInfo->TMDSSettingInfo.ModeIndex |= 0x200;
                break;
        }
    }
    else
    {
        /* Determine whether to use reduce blanking mode. */
        VIAReduceBlanking(pBIOSInfo, pBIOSInfo->TMDSSettingInfo.ModeIndex, VIA_DEVICE_DFP);
    }
}


void VIAInitSettingInfo_LCD(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo,  CARD16 activeDevice)
{
    int     HActive, VActive;
    Bool    IsPanning = pLVDSSettingInfo->IsPanning;
    Bool    IsDownScaling = pLVDSSettingInfo->IsDownScaling;

    if (IsPanning)
    {
        HActive = pBIOSInfo->HDisplay;
        VActive = pBIOSInfo->VDisplay;

        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "LCD Max Resolution %dx%d, set virtual desktop!!\n", 
            pLVDSSettingInfo->PanelSizeX, pLVDSSettingInfo->PanelSizeY);
    }
    else if (IsDownScaling)
    {
        HActive = pBIOSInfo->HorDownScaleSize;
        VActive = pBIOSInfo->VerDownScaleSize;
    }

    else
    {
        HActive = pBIOSInfo->CrtcHDisplay;
        VActive = pBIOSInfo->CrtcVDisplay;
    }

    VIAInitSettingInfo_IGA(pBIOSInfo, pLVDSSettingInfo->IGAPath, IsPanning, IsDownScaling, HActive, VActive, activeDevice); 
    

    pLVDSSettingInfo->ModeIndex = VIAGetModeIndex(HActive, VActive);
    pLVDSSettingInfo->HActive = HActive;
    pLVDSSettingInfo->VActive = VActive;        
    pBIOSInfo->IsDownScaleEnable = IsDownScaling;  
}


Bool VIAFindModeUseBIOSTable(ScrnInfoPtr pScrn)
{
    VIAPtr          pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;
    int             needRefresh;
    int             ModeIndex;
    Bool            IsFoundRefresh = FALSE;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIAFindModeUseBIOSTable\n"));

    /* Initial down scaling variable. */
    pBIOSInfo->IsDownScaleEnable = FALSE;

	if (!pVia->useRandR)
	{
	    if ((pBIOSInfo->ActiveDevice & VIA_DEVICE_DFP) &&
	        (pBIOSInfo->TMDSSettingInfo.DFPSize == M_INVALID))
	    {
	        VIAGetPanelInfo(pBIOSInfo);
	    }
		/*ActiveDevice will be removed in new randr. TODO!*/
	    if (!pBIOSInfo->ActiveDevice)
	    {
	        pBIOSInfo->ActiveDevice = VIAGetDeviceDetect(pBIOSInfo);
	    }
	} 
    
    if (pBIOSInfo->VirtualX > 0)
    {
        /* If set Virtual Desktop Size in xorg.conf: */
        pBIOSInfo->ActualDesktopSizeX = pBIOSInfo->VirtualX;
        pBIOSInfo->ActualDesktopSizeY = pBIOSInfo->VirtualY;
    }
    else
    {
        pBIOSInfo->ActualDesktopSizeX = pBIOSInfo->CrtcHDisplay;
        pBIOSInfo->ActualDesktopSizeY = pBIOSInfo->CrtcVDisplay;
    }

    /* Initial device's setting information: */
    VIAAdjustActiveSize(pBIOSInfo);
                        
    /* Find Resolution index: */
    ModeIndex = VIAGetModeIndex(pBIOSInfo->CrtcHDisplay, pBIOSInfo->CrtcVDisplay);

    /* Find suitable refresh rate: */
    /* Set the refresh rate to 60MHz for DuoView. */
    /* Note: We should support other refresh rate later. */
    if (/*pBIOSInfo->DuoView ||*/ 
        !(pBIOSInfo->ActiveDevice & (VIA_DEVICE_LCD | VIA_DEVICE_LCD2)))
    {
        if (pBIOSInfo->OptRefresh)
        {
            pBIOSInfo->Refresh = pBIOSInfo->OptRefresh;           
            IsFoundRefresh = VIAFindRefreshRate(pBIOSInfo, ModeIndex);            
        }
        else if (pBIOSInfo->ActiveDevice & VIA_DEVICE_DFP)
        {
            /* DVI will reference the CRT's capability to 
               determine the refresh rate. But the most DVI cannot support the
               higher refresh rate. I consider that we should to use option to 
               set refresh rate if we want to set the higher refresh rate for DVI.
               Otherwise the default refresh rate of DVI is 60Hz.
            */
            pBIOSInfo->FoundRefresh = 60;
            IsFoundRefresh = TRUE;
        }
        
        if (!IsFoundRefresh)
        {
            xf86DrvMsg(pBIOSInfo->scrnIndex, X_WARNING, "Refresh rate setting in xorg.conf is not supported!!\n");
            xf86DrvMsg(pBIOSInfo->scrnIndex, X_WARNING, "Driver will try to find another refresh rate instead.\n");
                
            /* use the monitor information */
            /* needRefresh = (pBIOSInfo->Clock * 1000) / (pBIOSInfo->HTotal * pBIOSInfo->VTotal); */
            /* Do rounding */
            needRefresh = ((pBIOSInfo->Clock * 10000) / (pBIOSInfo->HTotal * pBIOSInfo->VTotal) + 5) / 10;
            pBIOSInfo->Refresh = needRefresh;            
            IsFoundRefresh = VIAFindRefreshRate(pBIOSInfo, ModeIndex);            
        }
        
        if (!IsFoundRefresh)
        {
            pBIOSInfo->FoundRefresh = 60;
            xf86DrvMsg(pBIOSInfo->scrnIndex, X_WARNING, "Can't find suitable refresh rate, use 60Hz as default.\n");
        }
    }
    else
    {
        pBIOSInfo->FoundRefresh = 60;
    }
    
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_CRT1)
    {        
        VIAInitSettingInfo_CRT(pBIOSInfo, &(pBIOSInfo->CRTSettingInfo), VIA_DEVICE_CRT1 );
    }
    
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_DFP)
    {
        VIAInitSettingInfo_DVI(pBIOSInfo);
    }
    
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD)
    {
        VIAInitSettingInfo_LCD(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo), VIA_DEVICE_LCD);
    }

    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD2)
    {
        VIAInitSettingInfo_LCD(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo2), VIA_DEVICE_LCD2);
    }    
    /* pBIOSInfo->widthByQWord = (pBIOSInfo->displayWidth * (pBIOSInfo->bitsPerPixel >> 3)) >> 3; */
    pBIOSInfo->offsetWidthByQWord = (pBIOSInfo->displayWidth * (pBIOSInfo->bitsPerPixel >> 3)) >> 3;
    pBIOSInfo->countWidthByQWord = (pBIOSInfo->CrtcHDisplay * (pBIOSInfo->bitsPerPixel >> 3)) >> 3;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "pBIOSInfo->FoundRefresh: %d\n", pBIOSInfo->FoundRefresh));
    return TRUE;
}

void VIAPrimaryDisplayOffsetPatch(VIABIOSInfoPtr pBIOSInfo)
{
    int     countWidthByQWord, offsetWidthByQWord;
    CARD8   cr13, cr35, sr1c, sr1d;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIAPrimaryDisplayOffsetPatch\n"));    
    
    countWidthByQWord = pBIOSInfo->IGA1SettingInfo.CountWidthByQWord;
    offsetWidthByQWord = pBIOSInfo->IGA1SettingInfo.OffsetWidthByQWord;
    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "countWidthByQWord = %d\n", countWidthByQWord));
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "offsetWidthByQWord = %d\n", offsetWidthByQWord));    

    
    /* IGA1's offset */
    cr13 = offsetWidthByQWord & 0xFF;
    cr35 = read_reg(VIACR, CR35) & 0x1F;    
    cr35 |= (offsetWidthByQWord & 0x700) >> 3;        /* bit 7:5: offset 10:8 */
    write_reg(CR13, VIACR, cr13);
    write_reg(CR35, VIACR, cr35);
    
    /* IGA1's quad-word count. */
    sr1c = ((countWidthByQWord >> 1)+1) & 0xFF;
    sr1d = read_reg(VIASR, SR1D);
    sr1d = (sr1d & 0xFC) | (countWidthByQWord >> 9);
    write_reg(SR1C, VIASR, sr1c);
    write_reg(SR1D, VIASR, sr1d); 

    /* For down-scaling used. */
    if ((pBIOSInfo->IsDownScaleEnable) 
     && ((pBIOSInfo->ActiveDevice & VIA_DEVICE_DFP) && (pBIOSInfo->TMDSSettingInfo.IGAPath == IGA1)))
    {
        load_offset_reg(pBIOSInfo->HDisplay, pBIOSInfo->bitsPerPixel >> 3, IGA1);
    }
}

void VIASecondDisplayOffsetPatch(VIABIOSInfoPtr pBIOSInfo)
{    
    /* If use virtual desktop, we should update */
    /* second display horizontal offset to the width of desktop. */
    int     countWidthByQWord = 0, offsetWidthByQWord = 0;    
    CARD8   cr65, cr67;

    /*the CountWidthByQWord and the OffsetWidthByQWord variables always keep the 
        effective value when 3D scaling or not. So we load the offset and fetch count
        registers using these two variables.*/
    countWidthByQWord = pBIOSInfo->IGA2SettingInfo.CountWidthByQWord;
    offsetWidthByQWord = pBIOSInfo->IGA2SettingInfo.OffsetWidthByQWord;
    if (pBIOSInfo->IsDownScaleEnable)
    {
        if (pBIOSInfo->displayWidth > pBIOSInfo->HDisplay)
        {
            load_offset_reg(pBIOSInfo->displayWidth, pBIOSInfo->bitsPerPixel >> 3, IGA2);
        }
        else
        {
            load_offset_reg(pBIOSInfo->HDisplay, pBIOSInfo->bitsPerPixel >> 3, IGA2);
        }
    }
    else 
    {
        /* offset patch */
        load_offset_reg((offsetWidthByQWord<<3) / (pBIOSInfo->bitsPerPixel >> 3), pBIOSInfo->bitsPerPixel >> 3, IGA2);        

        /* fetch count patch */
        cr65 = (countWidthByQWord >> 1) & 0xFF;
        cr67 = read_reg(VIACR, CR67);
        cr67 = (cr67 & 0xF3) | ((countWidthByQWord >> 9) << 2);
        write_reg(CR65, VIACR, cr65);
        write_reg(CR67, VIACR, cr67);
    }   
}

Bool VIASetModeForMHS(VIABIOSInfoPtr pBIOSInfo)
{
    VIABIOSInfoPtr  pVia;

    pVia = pBIOSInfo;
    int IGAToUse = 0;
    
    if (pBIOSInfo->FirstInit)
    {
        /* Clear Memory. Sometimes it will appear some garbage under 1366x768 SAMM(LCD+CRT).  */      
        memset(pBIOSInfo->FBBase, 0x00, pBIOSInfo->videoRambytes - VIA_FB_CURSOR_SIZE); 
    }
    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIASetModeForMHS\n"));
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "Active Device is 0x%x\n", pBIOSInfo->ActiveDevice));
    
    /* Turn off Screen */
    write_reg_mask(CR17, VIACR, 0x0, BIT7);

    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_CRT1)
    {
        VIACRTSetMode(pBIOSInfo,pBIOSInfo->CRTSettingInfo.ModeIndex, pBIOSInfo->CRTSettingInfo.IGAPath);
        VIACRTEnable();
        IGAToUse = pBIOSInfo->CRTSettingInfo.IGAPath;
    }
    
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD)
    {
        VIALCDSetMode(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo));
        VIAEnableLCD(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo));
        IGAToUse = pBIOSInfo->LVDSSettingInfo.IGAPath;
    }

    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD2)
    {
        VIALCDSetMode(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo2));
        VIAEnableLCD(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo2));
        IGAToUse = pBIOSInfo->LVDSSettingInfo2.IGAPath;
    }

    /*move DFP setmode after LCD,else DVI will be turned off by VIADisableLCD on EPIA*/
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_DFP)
    {
        VIASetDFPMode(pBIOSInfo);
        VIAEnableDFP(pBIOSInfo);
        IGAToUse = pBIOSInfo->TMDSSettingInfo.IGAPath;
    }      


    /* Set Quadword offset and counter */
    /*
    These code will patch IGA2 regs,according secondary display size.
    When LCD2+LCD1,LCD is primary on IGA2,
    In this case,IGA2 regs should not patch.
    */

    if(!(pBIOSInfo->UseIGA2 && pBIOSInfo->TwoLCD))
    {
        if (IGAToUse == IGA1)
        {
            VIAPrimaryDisplayOffsetPatch(pBIOSInfo);
        }
        else
        {
            /* Replaced by function call. */
            VIASecondDisplayOffsetPatch(pBIOSInfo);
        }
    }

    set_color_depth(pBIOSInfo, pBIOSInfo->bitsPerPixel>>3, IGA2);    

    /* Enable CRT Controller (3D5.17 Hardware Reset) */
    write_reg_mask(CR17, VIACR, 0x80, BIT7);

    /* Open Screen */
    (void) VGAIN8(0x3da);
    VGAOUT8(0x3c0, 0x20);
    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "-- VIASetModeForMHS Done!\n"));
    return TRUE;
}

Bool VIASaveGamma(ScrnInfoPtr pScrn, LOCO *colors) {
    VIAPtr  pVia = VIAPTR(pScrn);
    CARD8   sr1a;
    int     i;

    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VIASaveGamma\n"));
    /* 8 bpp mode can't adjust gamma */
    if (pScrn->bitsPerPixel == 8)
        return FALSE;

    /* Enable Gamma */
    write_reg_mask(CR33, VIACR, 0x80, BIT7);

    /* map to Primary Display's LUT */
    sr1a = read_reg(VIASR, SR1A);
    write_reg_mask(SR1A, VIASR, 0x0, BIT0);

    VGAOUT8(0x3c7, 0);
    for (i = 0; i < 256; i++)
    {
        colors[i].red = VGAIN8(0x3c9);
        colors[i].green = VGAIN8(0x3c9);
        colors[i].blue = VGAIN8(0x3c9);
    }

    write_reg(SR1A, VIASR, sr1a);

    return TRUE;
}

Bool VIARestoreGamma(ScrnInfoPtr pScrn, LOCO *colors) {
    VIAPtr pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr pBIOSInfo = pVia->pBIOSInfo;
    int i, sr1a, cr6a, sr1b, sr2d;

    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VIARestorePalette\n"));
    /* 8 bpp mode can't adjust gamma */
    if (pScrn->bitsPerPixel == 8)
        return FALSE;

    /* Note:
       VT3409 has a hardware issue, when we enable IGA1 gamma, the display color
       will become bad if playing video.
       Solution: disable IGA1 gamma on VT3409 (16/32bpp)
    */
    if (pBIOSInfo->Chipset != VIA_VX855)
    {
        /* Enable IGA1 Gamma */
        write_reg_mask(CR33, VIACR, 0x80, BIT7);
    }
    
    /* 3C6/3C7/3C8/3C9 map to IGA1 gamma */
    sr1a = read_reg(VIASR, SR1A);
    write_reg_mask(SR1A, VIASR, 0x0, BIT0);                    

    sr1b = read_reg(VIASR, SR1B);
    write_reg_mask(SR1B, VIASR, 0xC0, BIT6+BIT7);  

    sr2d = read_reg(VIASR, SR2D);
    write_reg_mask(SR2D, VIASR, 0x0C, BIT2+BIT3);
            
    VGAOUT8(0x3c8, 0);
    for (i = 0; i < 256; i++)
    {
        VGAOUT8(0x3c9, colors[i].red);
        VGAOUT8(0x3c9, colors[i].green);
        VGAOUT8(0x3c9, colors[i].blue);
    }    

    /* 3C6/3C7/3C8/3C9 map to IGA2 gamma */
    write_reg_mask(SR1A, VIASR, 0x01, BIT0);

    /* Enable IGA2 Gamma */   
    write_reg_mask(CR6A, VIACR, 0x02, BIT1);

    /* We should set the format of IGA2's LUT to 8-BITs, because we use 8-BITs LUT except VT3123Ax. */
    /* This register added in VT3314 or later chipsets. */
    write_reg_mask(CR6A, VIACR, BIT5, BIT5);

    /* 
     Situation:
     On VT3324, when we switch to two devices, the second LUT maybe fault.
     Solution:
     Before we fill the second LUT, we have to enable second display channel.
    */
    /* Save */
    VGAOUT8(0x3D4, CR6A);
    cr6a = VGAIN8(0x3D5);
        
    VGAOUT8(0x3D4, CR6A);           
    if(!(VGAIN8(0x3D5) & 0x80))
    {                
        EnableSecondDisplayChannel();
    }

    /* Fill Second LUT */
    VGAOUT8(0x3c8, 0);
    for (i = 0; i < 256; i++)
    {
        VGAOUT8(0x3c9, colors[i].red);
        VGAOUT8(0x3c9, colors[i].green);
        VGAOUT8(0x3c9, colors[i].blue);
    }

    /* After filled second LUT, we should restore to previous value. */
    VGAOUT8(0x3D4, CR6A);
    VGAOUT8(0x3D5, cr6a);   

    write_reg(SR1A, VIASR, sr1a);
    write_reg(SR1B, VIASR, sr1b);
    write_reg(SR2D, VIASR, sr2d);
    return TRUE;
}


/* query NB to get frame buffer size */
int VIAGetFBSize(ScrnInfoPtr pScrn)
{
    unsigned long   configid,deviceid,FBSize = 0;
    int   VideoMemSize;
    int   DeviceFound = FALSE;
    VIAPtr pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr pBIOSInfo = pVia->pBIOSInfo;
    
    for ( configid = 0x80000000; configid < 0x80010800; configid += 0x100 )
    {
        outl((unsigned long)0xCF8,configid);
        deviceid = (inl((unsigned long)0xCFC)>>16) & 0xffff;

        /* Add KM800 case*/
        switch (deviceid)
        {
            case CN700_FUNCTION3:
            case CX700_FUNCTION3:
            case KM890_FUNCTION3:
            case P4M890_FUNCTION3:
            case P4M900_FUNCTION3:
            case VX800_FUNCTION3:
            case VX855_FUNCTION3:
                outl((unsigned long)0xCF8,configid+0xA0);
                FBSize = inl((unsigned long)0xCFC);
                DeviceFound = TRUE; /* Found device id */
                break;
                
            /* Can't set default, otherwise FBSize will be modified incorrectly */    
            default:    
                break;
         }

         if (DeviceFound)
         {
            break; /* Found device id, so exit for loop */
         }
    }

    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Device ID = %lx\n", deviceid));
    
    FBSize = FBSize & 0x00007000;
    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "FB Size = %lx\n", FBSize));
    
    if (pBIOSInfo->Chipset < VIA_CX700)
    {
        switch(FBSize)
        {
            case 0x00004000:
                VideoMemSize = (16 << 10);  /*16M*/              
                break;

            case 0x00005000:
                VideoMemSize = (32 << 10);  /*32M*/                
                break;

            case 0x00006000:
                VideoMemSize = (64 << 10);  /*64M*/                
                break;

            default:
                VideoMemSize = (32 << 10);  /*32M*/
                break;     
        }
    }
    else
    {
        switch(FBSize)
        {
            case 0x00001000:
                VideoMemSize = (8 << 10);   /*8M*/                
                break;

            case 0x00002000:
                VideoMemSize = (16 << 10);  /*16M*/               
                break;

            case 0x00003000:
                VideoMemSize = (32 << 10);  /*32M*/                
                break;

            case 0x00004000:
                VideoMemSize = (64 << 10);  /*64M*/               
                break;

            case 0x00005000:
                VideoMemSize = (128 << 10); /*128M*/
                break;

            case 0x00006000:
                VideoMemSize = (256 << 10); /*256M*/
                break;

            default:
                VideoMemSize = (32 << 10);  /*32M*/                
                break;
        }
    }

    pVia->pChipInfo->biosVideoMemSize = VideoMemSize;

    return VideoMemSize;
}

unsigned char BIOS_GetActiveDevice(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    int RealOff;
    pointer page = NULL;
    unsigned char   ret = 0x0;

    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "BIOS_GetActiveDevice\n"));

    /* For MAMM. */
    if (pVia->IsMAMMEnable)
        return 0xFF;
    
    if (pVia->pInt10==NULL)
        return 0xFF;
    
    page = xf86Int10AllocPages(pVia->pInt10, 1, &RealOff);
    
    if (!page)
        return 0xFF;
    
    pVia->pInt10->ax = 0x4F14;
    pVia->pInt10->bx = 0x0103;
    pVia->pInt10->cx = 0;
    pVia->pInt10->num = 0x10;
    xf86ExecX86int10(pVia->pInt10);

    if ((pVia->pInt10->ax & 0xFF) != 0x4F)
    {
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
              "BIOS Get Active Device fail!\n"));

        if (page)
            xf86Int10FreePages(pVia->pInt10, page, 1);
        
        return 0xFF;
    }

    if (page)
        xf86Int10FreePages(pVia->pInt10, page, 1);

    if (pVia->pInt10->cx & 0x01)
        ret = VIA_DEVICE_CRT1;
    if (pVia->pInt10->cx & 0x02)
        ret |= VIA_DEVICE_LCD;
    if (pVia->pInt10->cx & 0x20)
        ret |= VIA_DEVICE_DFP;

    pVia->pChipInfo->biosActiveDev = ret;
    
    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Active Dev is: 0x%x\n", ret));
    return ret;
}



Bool BIOS_GetBIOSDate(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    int RealOff;
    pointer page = NULL;

    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "BIOS_GetBIOSDate\n"));

    /* For MAMM. */
    if (pVia->IsMAMMEnable)
        return FALSE;
    
    /*weird*/
    if (pVia->pInt10==NULL)
        return FALSE; 

    page = xf86Int10AllocPages(pVia->pInt10, 1, &RealOff);
    if (!page)
        return FALSE;
    pVia->pInt10->ax = 0x4F14;
    pVia->pInt10->bx = 0x0100;
    pVia->pInt10->cx = 0;
    pVia->pInt10->dx = 0;
    pVia->pInt10->si = 0;
    pVia->pInt10->num = 0x10;
    xf86ExecX86int10(pVia->pInt10);

    if ((pVia->pInt10->ax & 0xff) != 0x4f)
    {
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Get BIOS Date fail!\n"));
        if (page)
            xf86Int10FreePages(pVia->pInt10, page, 1);
        
        return FALSE;
    }

    pVia->pChipInfo->biosDateYear = ((pVia->pInt10->bx >> 8) - 48) + ((pVia->pInt10->bx & 0xFF) - 48)*10;
    pVia->pChipInfo->biosDateMonth = ((pVia->pInt10->cx >> 8) - 48) + ((pVia->pInt10->cx & 0xFF) - 48)*10;
    pVia->pChipInfo->biosDateDay = ((pVia->pInt10->dx >> 8) - 48) + ((pVia->pInt10->dx & 0xFF) - 48)*10;

    if (page)
        xf86Int10FreePages(pVia->pInt10, page, 1);

    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VGA BIOS Release Date Is: %d/%d/%d\n",
               pVia->pChipInfo->biosDateYear + 2000,
               pVia->pChipInfo->biosDateMonth,
               pVia->pChipInfo->biosDateDay);
    
    return TRUE;
}

int BIOS_GetBIOSVersion(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    int RealOff, ret;
    pointer page = NULL;

    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "BIOS_GetBIOSVersion\n"));

    /* For MAMM. */
    if (pVia->IsMAMMEnable)
        return 0xFFFF;
    
    /*strange but need to check*/
    if (pVia->pInt10==NULL)
        return FALSE;
    
    page = xf86Int10AllocPages(pVia->pInt10, 1, &RealOff);
    
    if (!page)
        return 0xFFFF;
    
    pVia->pInt10->ax = 0x4F14;
    pVia->pInt10->bx = 0x0000;
    pVia->pInt10->cx = 0;
    pVia->pInt10->num = 0x10;
    xf86ExecX86int10(pVia->pInt10);

    if ((pVia->pInt10->ax & 0xff) != 0x4f)
    {
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Get BIOS Version fail!\n"));

        if (page)
            xf86Int10FreePages(pVia->pInt10, page, 1);

        return 0xFFFF;
    }

    if (page)
        xf86Int10FreePages(pVia->pInt10, page, 1);

    ret = pVia->pInt10->bx & 0xFFFF;

    if (pVia->Chipset >= VIA_P4M900){
        pVia->pChipInfo->biosVerMajorNum = ret & 0xFF;
        pVia->pChipInfo->biosVerMinorNum = 0;        
    }else{            
        pVia->pChipInfo->biosVerMajorNum = ret >> 8;
        pVia->pChipInfo->biosVerMinorNum = ret & 0xFF;           
    }
    
    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VGA BIOS Version is = %x.%x\n",
               pVia->pChipInfo->biosVerMajorNum, 
               pVia->pChipInfo->biosVerMinorNum);
    
    return ret;
}


Bool BIOS_QueryChipInfo(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    int RealOff;
    pointer page = NULL;
    int supportDev = 0;
    int biosType = 0;
    
    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "BIOS_QueryChipInfo\n"));

    /* For MAMM. */
    if (pVia->IsMAMMEnable)
        return FALSE;
    
    /*strange but need to check*/
    if (pVia->pInt10==NULL)
        return FALSE;
    
    page = xf86Int10AllocPages(pVia->pInt10, 1, &RealOff);
    
    if (!page)
        return FALSE;
    
    pVia->pInt10->ax = 0x4F14;
    pVia->pInt10->bx = 0x0000;
    pVia->pInt10->cx = 0;
    pVia->pInt10->dx = 0;
    pVia->pInt10->di = 0;
    pVia->pInt10->si = 0;
    pVia->pInt10->num = 0x10;

    /* Real execution */
    xf86ExecX86int10(pVia->pInt10);

    if (pVia->pInt10->ax != 0x4F)
    {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "BIOS Query Chip Info fail!\n");
        
        if (page)
            xf86Int10FreePages(pVia->pInt10, page, 1);
        
        return FALSE;
    }

    if (pVia->Chipset >= VIA_P4M900){
        biosType = (pVia->pInt10->bx & 0x0C000000)>>26;
        switch(biosType)
        {
        case 0:
            DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Mobile BIOS!\n"));
            pVia->pChipInfo->biosType = BIOS_TYPE_MOBILE;
            break;
        case 1:
            DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Embed BIOS!\n"));
            pVia->pChipInfo->biosType = BIOS_TYPE_EMBED;        
            break;
        case 2:
            DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Desktop BIOS!\n"));
            pVia->pChipInfo->biosType = BIOS_TYPE_DESKTOP;        
            break;
        }        
    }else{
        biosType = (pVia->pInt10->bx & 0x08000000)>>27;
        switch(biosType)
        {
        case 0:
            DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Mobile BIOS!\n"));
            pVia->pChipInfo->biosType = BIOS_TYPE_MOBILE;
            break;
        case 1:            
            DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Desktop BIOS!\n"));
            pVia->pChipInfo->biosType = BIOS_TYPE_DESKTOP;        
            break;
        }        
        
    }

    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO,
          "pVia->pInt10->cx = 0x%x\n", pVia->pInt10->cx)); 
    
    supportDev = (pVia->pInt10->cx & 0xFFFF0000)>>16;
    if (supportDev & 0x1){
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Support CRT1!\n"));
        pVia->pChipInfo->biosSupportDev |= VIA_DEVICE_CRT1;
    }
    
    if (supportDev & 0x2){
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Support LCD!\n"));
        pVia->pChipInfo->biosSupportDev |= VIA_DEVICE_LCD;        
    }
    
    if (supportDev & 0x8){
        if (pVia->Chipset >= VIA_P4M900){
            DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Support LCD2!\n"));
            pVia->pChipInfo->biosSupportDev |= VIA_DEVICE_LCD2;        
        }            
    }
    
    if (supportDev & 0x10){
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Support DVI2!\n"));
        pVia->pChipInfo->biosSupportDev |= VIA_DEVICE_DFP2;        
    }
    
    if (supportDev & 0x20){
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Support DVI!\n"));        
        pVia->pChipInfo->biosSupportDev |= VIA_DEVICE_DFP;        
    }
    
    if (supportDev & 0x80){
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Device Reserved!\n"));
    }
    
    if (supportDev & 0x100){
        if (pVia->Chipset >= VIA_P4M900){
            DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Support LCD2!\n"));            
            pVia->pChipInfo->biosSupportDev |= VIA_DEVICE_LCD2;        
        }
    }

    return TRUE;
}



void VIADelayIn_usec(VIABIOSInfoPtr pBIOSInfo, int usec)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    volatile long   i;
    unsigned char   tmp;

    if (usec > 0)
    {
        for (i = usec * 4; i > 0; i--)
        {
            tmp = VGAIN8(0x3c3);
        }
    }
}


/* CRT DAC Sense */
Bool VIASenseCRT(ScrnInfoPtr pScrn)
{    
    VIAPtr pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;    
    
    CARD8   sr2d, sr1b, sr01, cr36, sr40;
    Bool    IsAttch = FALSE;

    /* Save Value */
    sr2d = read_reg(VIASR, SR2D);
    sr1b = read_reg(VIASR, SR1B);
    sr01 = read_reg(VIASR, SR01);
    cr36 = read_reg(VIACR, CR36);
    sr40 = read_reg(VIASR, SR40);
    
    write_reg_mask(SR2D, VIASR, BIT4+BIT5, BIT4+BIT5);      /* Turn on PLL of VCK */
    write_reg_mask(SR1B, VIASR, BIT4+BIT5, BIT4+BIT5);      /* Turn on VCK */
    write_reg_mask(SR01, VIASR, 0x0, BIT5);                 /* Screen On */
    write_reg_mask(CR36, VIACR, 0x0, BIT4+BIT5+BIT6+BIT7);  /* Power On DPMS */                        

    /* This method could be fail. So I add delay. */
    while(!(INREG(0x200) & BIT1));                          /* Delay for V-sync */
    while(INREG(0x200) & BIT1);                             /* 1: Display, 0:VerticalBlank*/

    VIADelay_Nmsec(pBIOSInfo, 16);
    
    write_reg_mask(SR40, VIASR, BIT7, BIT7);                /* Enable CRT Sense */

    /* Read Status Bit */
    if (VGAIN8(0x3C2) & BIT4)
    {
        IsAttch = TRUE;        
    }
    else
    {
        IsAttch = FALSE;        
    }

    /* Restore */
    write_reg(SR40, VIASR, sr40);
    write_reg(CR36, VIASR, cr36);
    write_reg(SR01, VIASR, sr01);
    write_reg(SR1B, VIASR, sr1b);
    write_reg(SR2D, VIASR, sr2d);
    
    return IsAttch;
}


Bool VIASetMode(VIABIOSInfoPtr pBIOSInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    unsigned char tmp = 0;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIASetMode!!\n"));
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "Active Device is 0x%x\n",
                     pBIOSInfo->ActiveDevice));

    /* Turn off Screen */
    write_reg_mask(CR17, VIACR, 0x00, BIT7);

    /* Clean Second Path Status */
    if(!pBIOSInfo->SAMM || pBIOSInfo->FirstInit)
    {
        /*=* Reset IGA2 channel before disable IGA2 channel
             or it may cause some line garbage. *=*/
        /* Modify for sequence issue. */
        DisableSecondDisplayChannel();
        
        write_reg_mask(CR6A, VIACR, 0x00, 0x3D);
        write_reg(CR6B, VIACR, 0x00);
        write_reg(CR6C, VIACR, 0x00);
        write_reg(CR79,VIACR,0x0);

        /*=* Clear CR47[3,6-7]
             (IGA1 Timing Plus 2, added in VT3259 A3 or later) *=*/
        write_reg_mask(CR47, VIACR, 0x00, BIT7+BIT6+BIT3);
    }

    /* Disable LCD backlight to avoid garbage line */
    if ((pBIOSInfo->Chipset >= VIA_VX800) &&
        (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD) &&
        (pBIOSInfo->LVDSSettingInfo.DIPort & VIA_DI_TTL)) {    
        write_reg_mask(CR91, VIACR, 0x00, 0x02);
    }

    /* CRT set mode */
    if((pBIOSInfo->s3utility==FALSE)||(!pBIOSInfo->SAMM))
    {
        int i;
   
        VGAIN8(VIAStatus);
        VGAOUT8(VIAAR, 0x00);

        /* Temporary solution for the problem: on VT3364, the device on IGA2
           will display incorrect when the system resumes from suspends. 
           Because we don't know why the value of starting address register 
           CR62 will be changed from 0x00 to 0xff once we set Misc register, 
           so we are forced to change the flow to set Misc register before 
           common setting for video mode temporarily. Write Misc Register: 
        */
        VGAOUT8(VIAWMisc, VPIT.Misc);
        
        /* Write Common Setting for Video Mode */
        switch(pBIOSInfo->Chipset)
        {   
            case VIA_P4M800PRO:
            case VIA_K8M890:
            case VIA_P4M890:
            case VIA_P4M900:
                write_regx(CN700_ModeXregs, NUM_TOTAL_CN700_ModeXregs);
                /* 
                  Under MAMM, If VIA is secondary, vga BIOS will not initial 
                  registers. So we have to load initial table by ourself.
                 */
                if (pBIOSInfo->IsMAMMEnable) {
                    write_regx(CN700_ModeXregs_FOR_MAMM, 
                               NUM_TOTAL_CN700_ModeXregs_FOR_MAMM);
                }
                break;

            case VIA_CX700:
            case VIA_VX800:
                write_regx(CX700_ModeXregs, NUM_TOTAL_CX700_ModeXregs);
                break;
            case VIA_VX855:
                write_regx(VX855_ModeXregs, NUM_TOTAL_VX855_ModeXregs);
                break;
        }

        load_fix_bit_crtc_reg();

        VIACRTDisable();

        VIADisableLCD(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo));

        if (pBIOSInfo->LVDSSettingInfo2.ChipID) {       
            VIADisableLCD(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo2));
        }
                
        VIADisableDFP(pBIOSInfo);
        
        /* Fill VPIT Parameters */
        /* Write Sequencer */
        for ( i=1; i<=StdSR ; i++ ) {
            write_reg((CARD8)i, VIASR, VPIT.SR[i-1]);
        }
    
        /* Write Graphic Controller */
        for ( i=0; i<StdGR ; i++ ) {
            write_reg((CARD8)i, VIAGR, VPIT.GR[i]);
        }

        /* Write Attribute Controller */
        for ( i=0; i<StdAR ; i++ ) {
            VGAIN8(VIAStatus);
            VGAOUT8(VIAAR, i);
            VGAOUT8(VIAAR, VPIT.AR[i]);
        }

        VGAIN8(VIAStatus);
        VGAOUT8(VIAAR, 0x20);
    }

    /* On SAMM case(LCD+any device), if we press lcd expand checkbox,
       It will call this unnecessary CRT setmode and modify the IGA1's timing 
       using default value. So we have to skip this setmode operation under 
       the following condition. 
    */       
    if(!(pBIOSInfo->SAMM && (pBIOSInfo->PrimaryDevice == VIA_DEVICE_LCD)
       &&(pBIOSInfo->s3utility == TRUE))){           
        VIACRTSetMode(pBIOSInfo, M640x480, IGA1);
    }

    if(pBIOSInfo->ActiveDevice & VIA_DEVICE_CRT1){    
        VIACRTSetMode(pBIOSInfo, pBIOSInfo->CRTSettingInfo.ModeIndex, 
                      pBIOSInfo->CRTSettingInfo.IGAPath);
    }

    /* Set mode according to active device */
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_DFP)    
        VIASetDFPMode(pBIOSInfo);                       
    
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD)    
        VIALCDSetMode(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo));    

    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD2)    
        VIALCDSetMode(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo2));      

    /* Patch IGA1's quad-word count and offset for virtual desktop. */     
    if(!((pBIOSInfo->SAMM)
       &&((pBIOSInfo->PrimaryDevice == VIA_DEVICE_LCD) || (pBIOSInfo->PrimaryDevice == VIA_DEVICE_LCD2))
       &&(pBIOSInfo->s3utility == TRUE)))       
    {
        VIAPrimaryDisplayOffsetPatch(pBIOSInfo);  
    }

    /* We need to do this patch when IGA2 is in use. 
       And because LCD always uses IGA2,so we should do this patch for LCD, too.
    */
    if ((pBIOSInfo->DuoView) ||
        ((pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD) && (pBIOSInfo->LVDSSettingInfo.IGAPath == IGA2)) ||
        ((pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD2) && (pBIOSInfo->LVDSSettingInfo2.IGAPath == IGA2)))
    {
        /* Patch for using virtual desktop. */
        VIASecondDisplayOffsetPatch(pBIOSInfo);
    }

    /* Disable MMIO & PCI burst (0 wait state)
     * Ensure the PCI burst set to 0.
     *
     * Fix for video Red Bar issue.
     */    
    VGAOUT8(0x3c4, SR1A);
    tmp = VGAIN8(0x3c5);
    VGAOUT8(0x3c5, tmp &(~0x04));

    /* Enable modify CRTC starting address */
    write_reg_mask(CR11, VIACR, 0x00, BIT7);

    if (pBIOSInfo->FirstInit){        
        /* Clear Memory */
        memset(pBIOSInfo->FBBase, 0x00, 
               pBIOSInfo->videoRambytes - VIA_FB_CURSOR_SIZE);
        /*pBIOSInfo->FirstInit = FALSE;*/
    }

    VIAEnabledPrimaryExtendedFIFO(pBIOSInfo);

    /* Enable CRT Controller (3D5.17 Hardware Reset) */
    write_reg_mask(CR17, VIACR, BIT7, BIT7);

    /* CRT */
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_CRT1){
        VIACRTEnable();

        /* Disable unused DI port */
        if (pBIOSInfo->ActiveDevice == VIA_DEVICE_CRT1){
            write_reg_mask(SR1E, VIASR, 0x00, 0xF0); 
            write_reg_mask(SR2A, VIASR, 0x00, 0x0F);              
        }
    }
    else{
         if(pBIOSInfo->s3utility==FALSE)
            VIACRTDisable();
    }

    /*=* For Advantech Project, support 2 LCD SAMM.
         If mode is 640x480, disable IGA1 Horizontal and Vertical border. *=*/
    if((pBIOSInfo->TwoLCD)&&(pBIOSInfo->LVDSSettingInfo.ModeIndex == M640x480))
    {
        write_reg(CR03, VIACR, 0x83);
        write_reg(CR15, VIACR, 0xDF);
        write_reg(CR16, VIACR, 0x0C);
        /*=* Enable DVP1 *=*/
        write_reg_mask(SR1E, VIASR, 0xC0, 0xC0);
    }  

    /* Enable DVI after set mode */
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_DFP)    
        VIAEnableDFP(pBIOSInfo);      
    
    /* When enable LCD only, the bit of Spread Spectrum (SR1E[3]) and 
       Spectrum IO selected bit (SR2C[0]) should be enabled, or the screen 
       will scramble. 
    */
    if (pBIOSInfo->IsSpreadSpectrumEnabled &&
        (pBIOSInfo->ActiveDevice == VIA_DEVICE_LCD)){
        if ((pBIOSInfo->Chipset == VIA_CX700)||
            (pBIOSInfo->Chipset == VIA_VX800)||
            (pBIOSInfo->Chipset == VIA_VX855)){
            write_reg_mask(SR3D, VIASR, 0x01, BIT0);
        }
        else {       
            write_reg_mask(SR2C, VIASR, 0x01, BIT0);
        }

        write_reg_mask(SR1E, VIASR, 0x08, BIT3);
    }
    else{
        if ((pBIOSInfo->Chipset == VIA_CX700) ||
            (pBIOSInfo->Chipset == VIA_VX800) ||
            (pBIOSInfo->Chipset == VIA_VX855)){
            write_reg_mask(SR3D, VIASR, 0x00, BIT0);
        }
        else {       
            write_reg_mask(SR2C, VIASR, 0x00, BIT0);
        }

        write_reg_mask(SR1E, VIASR, 0x00, BIT3);
    }


    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD)            
        VIAEnableLCD(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo));
    
    if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD2)    
        VIAEnableLCD(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo2));        


    /* Open Screen */
    (void) VGAIN8(0x3da);
    VGAOUT8(0x3c0, 0x20);
    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "-- VIASetMode Done!\n"));
    return TRUE;
}

void VIACRTSetMode(VIABIOSInfoPtr pBIOSInfo, int vmode_index, int set_iga)
{
    struct VideoModeTable   *VideoMode = NULL;
    struct crt_mode_table   *crt_timing = NULL;

    /* for center timing used. */
    struct VideoModeTable   *ProjectorMode = NULL;
    struct crt_mode_table   *projector_crt_timing = NULL;
    struct display_timing   crt_reg;
    struct display_timing   projector_crt_reg; 
    struct display_timing   center_crt_reg; 
    CARD32                  pll_D_N;
    BOOL                    crt_use_modeline = FALSE;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIACRTSetMode!!\n"));

	/* Use the mode line in two cases:
	1. SAMM and Mode line is user-setting
	2. DuoView/single, EDIDType=true, and Mode line is user-setting*/
    if (((pBIOSInfo->SAMM) ||
		 (!pBIOSInfo->SAMM && (pBIOSInfo->EDIDType == VIA_DEVICE_CRT1))) && 
		 (pBIOSInfo->ModeLineStatus & (MODE_LINE_TYPE_USER | MODE_LINE_TYPE_EDID)))
    {
        crt_use_modeline = TRUE;
    }

    VideoMode = &VIARoutineModes[SearchModeSetting(vmode_index)];
    crt_timing = VideoMode->crtc;

    xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "-----CRT TIMING-----\n");

    /* Write CRTC */
    fill_crtc_timing(pBIOSInfo, crt_timing, VideoMode->mode_array, (pBIOSInfo->bitsPerPixel >> 3), VIA_DEVICE_CRT1, set_iga, crt_use_modeline);

    if (crt_use_modeline)
    {
        /* Set CRT center timing for projector used. */
        if ((vmode_index == M800x480) || (vmode_index == M1024x600))
        {
            DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "CRT Center Timing!!\n"));
            
            /* projector can't support 800x480, so we have to send 800x600 timing. */
            if (vmode_index == M800x480)
            {        
                ProjectorMode = &VIARoutineModes[SearchModeSetting(M800x600)];
            }
            else if (vmode_index == M1024x600)
            {
                ProjectorMode = &VIARoutineModes[SearchModeSetting(M1024x768)];
            }
            
            projector_crt_timing = ProjectorMode->crtc;

            crt_reg = crt_timing->crtc;
            projector_crt_reg = projector_crt_timing->crtc;        

            /* compute the center mode timing */
            center_crt_reg.hor_total = projector_crt_reg.hor_total;
            center_crt_reg.hor_addr = crt_reg.hor_addr;
            center_crt_reg.hor_blank_start = (projector_crt_reg.hor_addr - crt_reg.hor_addr)/2 + center_crt_reg.hor_addr;
            center_crt_reg.hor_blank_end = projector_crt_reg.hor_blank_end;
            center_crt_reg.hor_sync_start = (projector_crt_reg.hor_sync_start - projector_crt_reg.hor_blank_start) + center_crt_reg.hor_blank_start;
            center_crt_reg.hor_sync_end = projector_crt_reg.hor_sync_end;

            center_crt_reg.ver_total = projector_crt_reg.ver_total;
            center_crt_reg.ver_addr = crt_reg.ver_addr;
            center_crt_reg.ver_blank_start = (projector_crt_reg.ver_addr - crt_reg.ver_addr)/2 + center_crt_reg.ver_addr;
            center_crt_reg.ver_blank_end = projector_crt_reg.ver_blank_end;
            center_crt_reg.ver_sync_start = (projector_crt_reg.ver_sync_start - projector_crt_reg.ver_blank_start) + center_crt_reg.ver_blank_start;
            center_crt_reg.ver_sync_end = projector_crt_reg.ver_sync_end;                

            /* Patch the crtc timing */
            unlock_crt();
			load_crtc_timing(pBIOSInfo, center_crt_reg, IGA1);        
            lock_crt();  
               
            /* Set PLL */
            pll_D_N = get_clk_value(pBIOSInfo, projector_crt_timing[0].clk);        
            SetVCLK(pBIOSInfo, pll_D_N, set_iga);
            xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO," PixelClock=%ld Hz\n", projector_crt_timing[0].clk);        
            xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO,"-----TIMING END-----\n");                    

        }
    }

    
    /* If K8M800, enable Prefetch Mode. */
    if(pBIOSInfo->Chipset == VIA_K8M890)
    {
        write_reg_mask(CR33, VIACR, BIT3, BIT3);
    }
 
    if(!(pBIOSInfo->s3utility && (pBIOSInfo->SAMM || pBIOSInfo->DuoView)))
    {
        VIASetOutputPath(pBIOSInfo, VIA_DEVICE_CRT1,set_iga, 0);
    }

    /* Enable Interlace mode. */
    if (pBIOSInfo->IsEnableInterlaceMode)
    {
        if (set_iga == IGA1)
        {
            write_reg_mask(CR33, VIACR, BIT6, BIT6);
        }
        else if (set_iga == IGA2)
        {
            write_reg_mask(CR67, VIACR, BIT5, BIT5);
        }
    }
}


void VIACRTDisable(void)
{
    write_reg_mask(CR36, VIACR, BIT5+BIT4, BIT5+BIT4);
}

void VIACRTEnable(void)
{
    write_reg_mask(CR36, VIACR, 0x0, BIT5+BIT4);
}

void VIASetOutputPath(VIABIOSInfoPtr pBIOSInfo, int device, int set_iga, int output_interface)
{
    switch(device)
    {
        case VIA_DEVICE_CRT1:
            VIASetCRTOutputPath(pBIOSInfo, set_iga, output_interface);
            break;
        case VIA_DEVICE_DFP:
            VIASetDVIOutputPath(pBIOSInfo, set_iga, output_interface);
            break;
        case VIA_DEVICE_LCD:
            VIASetLCDOutputPath(pBIOSInfo, set_iga, output_interface);
            break;      
    }
}

void VIASetCRTOutputPath(VIABIOSInfoPtr pBIOSInfo, int set_iga, int output_interface)
{
    switch(set_iga)
    {
        case IGA1:
            /* normal crt */
            if(!output_interface)
            {
                write_reg_mask(SR16, VIASR, 0x00, BIT6);
            }            
            break;
            
        case IGA2:
        case IGA1_IGA2:
            /* Modify for sequence issue. */
            EnableSecondDisplayChannel();
            /* normal crt */
            if(!output_interface)
            {
                write_reg_mask(SR16, VIASR, 0x40, BIT6);
            }           
            if (set_iga == IGA1_IGA2)
                write_reg_mask(CR6B, VIACR, 0x08, BIT3);
            break;
    }

    if(!output_interface)
    {
        return ;
    }

    /*Set Iga source and turn on DI port clk*/
    switch(output_interface)
    {
        case VIA_DI_DVP0:
            if (set_iga == IGA1)
            {
                /* DVP0 Data Source Selection.*/
                write_reg_mask(CR96, VIACR, 0x00, BIT4);
            }
            else
            {
                /* DVP0 Data Source Selection.*/
                write_reg_mask(CR96, VIACR, 0x10, BIT4);
            }	
            /* enable dvp0 under CX700 */
            if (pBIOSInfo->Chipset == VIA_CX700)
            {
                write_reg_mask(CR91, VIACR, 0x00, BIT5);
            }
            
            /*Turn on DVP0 clk*/
            write_reg_mask(SR1E, VIASR, 0xc0, BIT6+BIT7);
            break;

        case VIA_DI_DVP1:
            if (set_iga == IGA1)
            {
                write_reg_mask(CR9B, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg_mask(CR9B, VIACR, 0x10, BIT4);
            }
            /* enable dvp1 under this chipset */
            if (pBIOSInfo->Chipset == VIA_CX700 ||
                pBIOSInfo->Chipset == VIA_VX800 ||
                pBIOSInfo->Chipset == VIA_VX855)
            {
                write_reg_mask(CRD3, VIACR, 0x00, BIT5);
            }
            /*Turn on DVP1 clk*/
            write_reg_mask(SR1E, VIASR, 0x30, BIT4+BIT5);
            break;
            
        case VIA_DI_DFPHIGH:
            if (set_iga == IGA1)
            {
                /*3324,3353 CR96[4] control DVP0*/
                if ((pBIOSInfo->Chipset == VIA_CX700 ) ||
                    (pBIOSInfo->Chipset == VIA_VX800 ) || 
                    (pBIOSInfo->Chipset == VIA_VX855))
                {
                    write_reg_mask(CR96, VIACR, 0x00, BIT4);
                }
                write_reg_mask(CR97, VIACR, 0x00, BIT4);
            }
            else
            {
                /*3324,3353 CR96[4] control DVP0*/
                if ((pBIOSInfo->Chipset == VIA_CX700 ) ||
                    (pBIOSInfo->Chipset == VIA_VX800 ) ||
                    (pBIOSInfo->Chipset == VIA_VX855))
                {
                    write_reg_mask(CR96, VIACR, 0x10, BIT4);
                }
                write_reg_mask(CR97, VIACR, 0x10, BIT4);
            }

            /*Turn on DFPH clk*/
            write_reg_mask(SR2A, VIASR, 0x0C, BIT2+BIT3);
            break;
            
        case VIA_DI_DFPLOW:
            if (set_iga == IGA1)
            {
                /*3324,3353 CR9B[4] control DVP1*/
                if ((pBIOSInfo->Chipset == VIA_CX700 ) ||
                    (pBIOSInfo->Chipset == VIA_VX800 ) ||
                    (pBIOSInfo->Chipset == VIA_VX855))
                {
                    write_reg_mask(CR9B, VIACR, 0x00, BIT4);
                }
                write_reg_mask(CR99, VIACR, 0x00, BIT4);
            }
            else
            {
                /*3324,3353 CR9B[4] control DVP1*/
                if ((pBIOSInfo->Chipset == VIA_CX700 ) ||
                    (pBIOSInfo->Chipset == VIA_VX800 ) ||
                    (pBIOSInfo->Chipset == VIA_VX855))
                {
                    write_reg_mask(CR9B, VIACR, 0x10, BIT4);
                }
                write_reg_mask(CR99, VIACR, 0x10, BIT4);
            }

            /*Turn on DFPL clk*/
            write_reg_mask(SR2A, VIASR, 0x03, BIT1+BIT0);
            break;
    }

    /*If Iga2 is used, enable Iga2 output for DI port. In order to enable Iga2, 3324,3353,3293 use CR91 and CRD3 to control
    DVP0 and DVP1 seperately, but other chip only use CR91 to control all Di port*/
    if(IGA2 == set_iga && pBIOSInfo->Chipset != VIA_CX700)
    {
        write_reg_mask(CR91, VIACR, 0x00, BIT5);
    }    

}

void VIASetDisplayChannel(VIABIOSInfoPtr pBIOSInfo)
{
    BOOL lcd_is_internal = (pBIOSInfo->ActualActiveDevice & VIA_DEVICE_LCD) && 
                           (pBIOSInfo->LVDSSettingInfo.ChipID == VIA_INTEGRATED_LVDS);
    BOOL lcd2_is_internal = (pBIOSInfo->ActualActiveDevice & VIA_DEVICE_LCD2) &&
                            (pBIOSInfo->LVDSSettingInfo2.ChipID == VIA_INTEGRATED_LVDS);
    BOOL tmds_is_internal = (pBIOSInfo->ActualActiveDevice & VIA_DEVICE_DFP) &&
                            (pBIOSInfo->TMDSSettingInfo.ChipID == VIA_INTEGRATED_TMDS);

    if ((lcd_is_internal && pBIOSInfo->LVDSSettingInfo.IsDualEdge) ||
        (lcd2_is_internal && pBIOSInfo->LVDSSettingInfo2.IsDualEdge))
    {
        /* For dual channel LCD: */
        /* Set to Dual LVDS channel. */
        write_reg_mask(CRD2, VIACR, 0x20, BIT4+BIT5);
    }
    else if (tmds_is_internal)
    {
        if (lcd_is_internal || lcd2_is_internal)
        {
            /* For Internal LCD + Internal DFP: */
            /* Set to LVDS1 + TMDS channel. */
            write_reg_mask(CRD2, VIACR, 0x10, BIT4+BIT5);
        }
        else
        {
            /* Set to single TMDS channel. */
            write_reg_mask(CRD2, VIACR, 0x30, BIT4+BIT5);   
        }
    }
    else
    {
        /* Set to LVDS0 + LVDS1 channel. */
        write_reg_mask(CRD2, VIACR, 0x00, BIT4+BIT5);
    }   
}

/*
    If monitor's max support resolution is a strange mode that
    we don't support, we have to correct it. Find a new mode that we support to replace it.

*/
Bool VIACorrectMonitorSize(int* MonitorH, int* MonitorV)
{   
    int i = 0;

    while (VIASupportMode[i].ModeIndex > -1)
    {
        if (*MonitorH < VIASupportMode[i].HActive)
        {
            i--;
            break;
        }
        else if ((*MonitorH == VIASupportMode[i].HActive)
              && (*MonitorV < VIASupportMode[i].VActive))
        {
            i--;
            break;
        }
        else if ((*MonitorH == VIASupportMode[i].HActive)
              && (*MonitorV == VIASupportMode[i].VActive))
        {
            break;
        }
        else
        {
            i++;
        }
    }
    
    if ((VIASupportMode[i].ModeIndex < 0) || (i < 0))
    {
        return FALSE;
    }
    else
    {
        *MonitorH = VIASupportMode[i].HActive;
        *MonitorV = VIASupportMode[i].VActive;        
        return TRUE;
    }
}


/*
*   Attach a new mode to the modeline.
*   So we don't need to add modeline in the config file.
*/
void ViaModesAttachHelper(ScrnInfoPtr pScrn, MonPtr monitorp, DisplayModePtr Modes)
{
    DisplayModePtr mode;
    DisplayModePtr last = monitorp->Last;
    int i;

    for (i = 0; Modes[i].name; i++)
    {        
        mode = xnfalloc(sizeof(DisplayModeRec));
        memcpy(mode, &Modes[i], sizeof(DisplayModeRec));
        mode->name = xnfstrdup(Modes[i].name);
        
        if (last)
        {
            mode->prev = last;
            last->next = mode;
        }
        else
        { 
            /* this is the first mode */
            monitorp->Modes = mode;
            mode->prev = NULL;
        }
        last = mode;
    }
    
    monitorp->Last = last;
}


Bool VIAEnumModes(VIABIOSInfoPtr pBIOSInfo, int IndexOfTable, VIASUPPORTMODE *pVIASupportMode, Bool *pIsSupported)
{
    int ItemNum;
    
    ItemNum = sizeof(VIASupportMode) / sizeof(VIASUPPORTMODE);

    *pIsSupported = FALSE;

    if (IndexOfTable < ItemNum)
    {
        memcpy(pVIASupportMode, &VIASupportMode[IndexOfTable], sizeof(VIASUPPORTMODE));
        
        /* Validate for CRT: */
        if ((pBIOSInfo->ActiveDevice & VIA_DEVICE_CRT1))
        {
            if ((pBIOSInfo->CRTSettingInfo.MonitorSizeH >= pVIASupportMode->HActive) &&
                (pBIOSInfo->CRTSettingInfo.MonitorSizeV >= pVIASupportMode->VActive))
            {
                *pIsSupported = TRUE;
            }
        }

        /* Validate for DVI: */
        if (pBIOSInfo->ActiveDevice & VIA_DEVICE_DFP)
        {
            if ((pBIOSInfo->TMDSSettingInfo.PanelSizeH >= pVIASupportMode->HActive) &&
                (pBIOSInfo->TMDSSettingInfo.PanelSizeV >= pVIASupportMode->VActive))
            {
                *pIsSupported = TRUE;
            }
        }

        /* Validate for LCD: */
        if (pBIOSInfo->ActiveDevice & VIA_DEVICE_LCD)
        {
            if ((pBIOSInfo->LVDSSettingInfo.PanelSizeX >= pVIASupportMode->HActive) &&
                (pBIOSInfo->LVDSSettingInfo.PanelSizeY >= pVIASupportMode->VActive))
            {
                *pIsSupported = TRUE;
            }
        }

        /* Don't support if mode size is larger than virtual size. */
        if (pVIASupportMode->HActive >= pBIOSInfo->SaveVirtualX)
        {
            if (pVIASupportMode->VActive > pBIOSInfo->SaveVirtualY)
            {
                *pIsSupported = FALSE;
            }
        }
        
        return TRUE;
    }

    return FALSE;
}


void VIAEnumRefreshRate(int ModeIndex, int RefreshRateNum, int *pRefreshRate)
{
    struct VideoModeTable   *VideoMode;
    
    VideoMode = &VIARoutineModes[SearchModeSetting(ModeIndex)];

    if (RefreshRateNum < VideoMode->mode_array)
    {
        *pRefreshRate = VideoMode->crtc[RefreshRateNum].refresh_rate;
    }
    else
    {
        *pRefreshRate = 0;
    }
}

/*
*   Purpose: Set DPA value to register.
*/
void VIASetDPA_Gfx(int DIPort, GFX_DPA_VALUE_PTR pGfxDPASetting)
{
    DEBUG(ErrorF("VIASetDPA_Gfx, Use Port %x.\n", DIPort));
    
    switch (DIPort)
    {
        case VIA_DI_DVP0:
        {
            /* DVP0 Clock Polarity and Adjust: */
            write_reg_mask(CR96, VIACR, pGfxDPASetting->DVP0, 0x0F);

            /* DVP0 Clock and Data Pads Driving: */
            write_reg_mask(SR1E, VIASR, pGfxDPASetting->DVP0ClockDri_S, BIT2);
            write_reg_mask(SR2A, VIASR, pGfxDPASetting->DVP0ClockDri_S1, BIT4);
            write_reg_mask(SR1B, VIASR, pGfxDPASetting->DVP0DataDri_S, BIT1);
            write_reg_mask(SR2A, VIASR, pGfxDPASetting->DVP0DataDri_S1, BIT5);

            DEBUG(ErrorF("DVP0 = %x.\n", pGfxDPASetting->DVP0));
            DEBUG(ErrorF("DVP0ClockDri_S = %x.\n", pGfxDPASetting->DVP0ClockDri_S));
            DEBUG(ErrorF("DVP0ClockDri_S1 = %x.\n", pGfxDPASetting->DVP0ClockDri_S1));
            DEBUG(ErrorF("DVP0DataDri_S = %x.\n", pGfxDPASetting->DVP0DataDri_S));
            DEBUG(ErrorF("DVP0DataDri_S1 = %x.\n", pGfxDPASetting->DVP0DataDri_S1));
            break;
        }

        case VIA_DI_DVP1:
        {
            /* DVP1 Clock Polarity and Adjust: */
            write_reg_mask(CR9B, VIACR, pGfxDPASetting->DVP1, 0x0F);

            /* DVP1 Clock and Data Pads Driving: */
            write_reg_mask(SR65, VIASR, pGfxDPASetting->DVP1Driving, 0x0F);
            
            DEBUG(ErrorF("DVP1 = %x.\n", pGfxDPASetting->DVP1));
            DEBUG(ErrorF("DVP1Driving = %x.\n", pGfxDPASetting->DVP1Driving));
            break;
        }

        case VIA_DI_DFPHIGH:
        {
            write_reg_mask(CR97, VIACR, pGfxDPASetting->DFPHigh, 0x0F);
            DEBUG(ErrorF("DFPHigh = %x.\n", pGfxDPASetting->DFPHigh));
            break;
        }

        case VIA_DI_DFPLOW:
        {
            write_reg_mask(CR99, VIACR, pGfxDPASetting->DFPLow, 0x0F);
            DEBUG(ErrorF("DFPLow = %x.\n", pGfxDPASetting->DFPLow));
            break;
        }

        case VIA_DI_DFPHIGHLOW:
        {
            write_reg_mask(CR97, VIACR, pGfxDPASetting->DFPHigh, 0x0F);
            write_reg_mask(CR99, VIACR, pGfxDPASetting->DFPLow, 0x0F);
            DEBUG(ErrorF("DFPHigh = %x.\n", pGfxDPASetting->DFPHigh));
            DEBUG(ErrorF("DFPLow = %x.\n", pGfxDPASetting->DFPLow));
            break;
        }
    }
}


/* Source Data Offset Control */
void VIASetDownScaleOffsetReg(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;    
    CARD32 reg_value;    
    CARD32 source_pitch = 0;
    
    /* Load Horizontal offset */
    reg_value    = pBIOSInfo->HDisplay;    
    write_reg_mask(CRD9, VIACR, reg_value, 0xFF);
    write_reg_mask(CRDB, VIACR, reg_value>>8, BIT0+BIT1+BIT2);    
        
    /* Load Vertical offset */
    reg_value    = pBIOSInfo->VDisplay;    
    write_reg_mask(CRDA, VIACR, reg_value, 0xFF);
    write_reg_mask(CRDB, VIACR, (reg_value>>8)<<3, BIT3+BIT4+BIT5); 


    /* Load down scaling source frame buffer stride(pitch). */
    if (pBIOSInfo->IsDownScaleEnable)
    {
        if (pBIOSInfo->displayWidth > pBIOSInfo->HDisplay)
        {
            source_pitch = pBIOSInfo->displayWidth;
        }
        else
        {
            source_pitch = pBIOSInfo->HDisplay;
        }
    }
    reg_value    = (((source_pitch * pBIOSInfo->bitsPerPixel/8) + 15) & ~15)>>4;    
  
    write_reg_mask(CRE4, VIACR, reg_value, 0xFF);
    write_reg_mask(CRE3, VIACR, (reg_value>>8)<<4, BIT4+BIT5); 
    
    
}

 
/* Set down scaling factor to control the screen size. */
void VIASetDownScaleFactor(ScrnInfoPtr pScrn, int DownScaleHSize, int DownScaleVSize)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;
    
    CARD32 reg_value;

    /* Load Horizontal down scaling factor. */
    if(pBIOSInfo->HDisplay > DownScaleHSize)
    {
        reg_value    = (pBIOSInfo->HDisplay * 4096) / DownScaleHSize;        
        write_reg_mask(CRDC, VIACR, reg_value, 0xFF);
        write_reg_mask(CRDD, VIACR, reg_value>>8, 0x3F);     

        /* Enable Horizontal down scale. */         
        write_reg_mask(CRDD, VIACR, 0x40, 0x40);            
    } 
    else
    {
        /* Clear */
        write_reg(CRDC, VIACR, 0x00);
        write_reg_mask(CRDD, VIACR, 0x00, 0x3F);   
        /* Disable Horizontal down scale. */         
        write_reg_mask(CRDD, VIACR, 0x00, 0x40);   
    }

    /* Load Vertical down scaling factor. */
    if(pBIOSInfo->VDisplay > DownScaleVSize)
    {
        reg_value    = (pBIOSInfo->VDisplay * 4096) / DownScaleVSize;        
        /* For Qunta IL1's case, fine tune to make the screen more smoth.*/
        if (pBIOSInfo->LVDSSettingInfo.PanelSizeID == VIA_800X480)
        {
            reg_value -= 5;
        }
        write_reg_mask(CRDE, VIACR, reg_value, 0xFF);
        write_reg_mask(CRDF, VIACR, reg_value>>8, 0x3F);

        /* Enable Vertical down scale. */        
        write_reg_mask(CRDF, VIACR, 0x40, 0x40);             
    }    
    else
    {
        /* Clear */
        write_reg_mask(CRDE, VIACR, 0x00, 0xFF);
        write_reg_mask(CRDF, VIACR, 0x00, 0x3F);
        /* Disable Vertical down scale. */        
        write_reg_mask(CRDF, VIACR, 0x00, 0x40);             
    }    
}

 
void VIASetDownScalingEngineReg(ScrnInfoPtr pScrn, int IGAPath)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;    
    CARD32 dwBufferStartAddr;        
    unsigned int addr = 0;
    static Bool IsDownScaleMemAllocated = FALSE;

    /* Disabling down-scaling to reset the destination buffers.*/
    if (IGAPath == IGA1) {
        write_reg_mask(CREC, VIACR, 0x00, BIT0);    /* IGA1 Scaling Down Disable */
    } else {
        write_reg_mask(CRE8, VIACR, 0x00, BIT4);    /* IGA2 Scaling Down Disable */
    }
    
    /* Reserved 3 buffer */
    if (!IsDownScaleMemAllocated)    
    {
        addr = viaFBAlloctor(pScrn, (VIA_DOWN_SCALE_BUFFER_SIZE * 3), INVERT_SEQ, "IGA Down Scaling Buf");
        if(addr) {
            pVia->DownScaleBufferStartAddr = addr; 
        }
        else {
           return;
        }
        IsDownScaleMemAllocated = TRUE;
    }

    /* Clear the buffer */
    memset((pVia->FBBase + pVia->DownScaleBufferStartAddr), 0, VIA_DOWN_SCALE_BUFFER_SIZE*3);    

    /* Set Buffer0 address. */
    dwBufferStartAddr = pVia->DownScaleBufferStartAddr >> 3;
    write_reg(CRE0, VIACR, (CARD8)(dwBufferStartAddr & 0x000000FF));
    write_reg(CRE1, VIACR, (CARD8)((dwBufferStartAddr >> 8) & 0x000000FF));
    write_reg(CRE2, VIACR, (CARD8)((dwBufferStartAddr >> 16) & 0x000000FF));

    write_reg_mask(CRE3, VIACR, (CARD8)((dwBufferStartAddr >> 24) & 0x00000003), 0x03);

    /* Set Buffer1 address. */       
    dwBufferStartAddr += (VIA_DOWN_SCALE_BUFFER_SIZE >> 3);
    write_reg(CRE5, VIACR, (CARD8)(dwBufferStartAddr & 0x000000FF));
    write_reg(CRE6, VIACR, (CARD8)((dwBufferStartAddr >> 8) & 0x000000FF));
    write_reg(CRE7, VIACR, (CARD8)((dwBufferStartAddr >> 16) & 0x000000FF));
    write_reg_mask(CRE8, VIACR, (CARD8)((dwBufferStartAddr >> 24) & 0x00000003), 0x03);
 
    /* Set Buffer2 address. */
    dwBufferStartAddr += (VIA_DOWN_SCALE_BUFFER_SIZE >> 3);
    write_reg(CRE9, VIACR, (CARD8)(dwBufferStartAddr & 0x000000FF));
    write_reg(CREA, VIACR, (CARD8)((dwBufferStartAddr >> 8) & 0x000000FF));
    write_reg(CREB, VIACR, (CARD8)((dwBufferStartAddr >> 16) & 0x000000FF));
    write_reg_mask(CRE8, VIACR, (CARD8)((dwBufferStartAddr >> 24) & 0x00000003), 0x0c);                
    
    /* Set down scaling horizontal and vertical offset. */
    VIASetDownScaleOffsetReg(pScrn);
        
    /* Set down scaling horizontal and vertical scaling factor. */
    VIASetDownScaleFactor(pScrn, pBIOSInfo->HorDownScaleSize, pBIOSInfo->VerDownScaleSize);     
    
    /* Set down scaling bpp. */    
    switch(pBIOSInfo->bitsPerPixel/8)
    {
        case MODE_8BPP:
            write_reg_mask(CRE3, VIACR, 0x40, 0xC0);
            break;
            
        case MODE_16BPP:
            write_reg_mask(CRE3, VIACR, 0x80, 0xC0);
            break;

        case MODE_32BPP:
            write_reg_mask(CRE3, VIACR, 0x00, 0xC0);
            break;
    }

    if (IGAPath == IGA1)
    {
        /* Select IGA1 Path to run down scale. */
        write_reg_mask(CRE8, VIACR, 0x00, BIT7); 
        
        /* IGA1 Down Scale Enable */
        write_reg_mask(CREC, VIACR, 0x01, BIT0);        
    }
    else
    {
        /* Select IGA2 Path to run down scale. */
        write_reg_mask(CRE8, VIACR, 0x80, BIT7); 

        /* IGA2 Down Scale Enable */
        write_reg_mask(CRE8, VIACR, 0x10, BIT4);
    }    
    
    /* Enable auto flip. */
    write_reg_mask(CRDD, VIACR, 0x80, BIT7);     

    
}

void VIADisableDownScaling(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);

    /* Disable auto flip. */
    VGAOUT8(0x3D4, CRDD);
    VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0x7F);

    /* IGA1 Down Scale Disable */    
    VGAOUT8(0x3D4, CREC);
    VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xFE);

    /* IGA2 Down Scale Disable */    
    VGAOUT8(0x3D4, CRE8);
    VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xEF);
}

CARD32  VIAGetRevisionOfVX800()
{    
    /* Get VT3353 Revision num. */
    outb(0x3C4, 0x3B);    
    if (inb(0x3C5) < 0x10)
    {        
        return REVISION_VX800_A;
    }    
    else if (inb(0x3C5) < 0x12)
    {        
        return REVISION_VX800_B;
    }
    else
    {        
        return REVISION_VX800_B2;
    }
}


