/*
 * Copyright 1998-2009 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2009 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */


/*
	File Name: via_common.c
	Description: write vga misc register 
	Purpose: add for V85a new structure,xrandr1.2 code refine
	Date:2008-12-16
*/
#include "via_common.h"
#define MASK_START(x)   (((x) == 0x80000000) ? (0x80000000) : (((x) ^ ((x) - 1) + 1) >> 1))
#define MASK_END(x)    (((x) & 0x80000000) ? (0x80000000) : (((x) + MASK_START(x)) >> 1))

/*
 * Function Name:
 * Description: Transfer the IO name(CRXX/SRXX to the address in MMIO base)
 * Parameters:
 * IN: IO name
 * OUT: the address of the register in MMIO
 * Status Return: none
 * Note: now it just support 32 bit OS, how to support 64bit OS will be done laterly
 * */
__inline__ CARD32 viaGetVgaIoAddress(CARD16 ioName)
{
    CARD32 ioAddress;
	
    if (IS_ZSR & ioName)
        ioAddress = REG_ZSR;      
    else
        ioAddress = REG_ZCR;              
    ioAddress = (CARD32)(MMIOMapBase + ioAddress);
    return ioAddress;
}

/*
 * Function Name: ReadMiscIo
 * Description: Read the data of MISC register
 * Parameters:
 * IN:
 * OUT: Data of MISC register
 * Status Return:
 * Note:    
 *
 * */
__inline__ CARD8  viaReadMiscIo(void)
{
    CARD8 ioData;
	
    ioData = MMIO_RD8(MMIOMapBase + REG_RMISC);
    return ioData;
}

/*
 * Function Name: WriteMiscIo
 * Description: Write data to MISC register
 * Parameters: 
 * IN: data
 * OUT:
 * Status Return:
 * Note:    
 * */
__inline__ CARD32  viaWriteMiscIo(CARD8 data)
{
    MMIO_WR8((MMIOMapBase + REG_WMISC), data);
    return TRUE;
}


/*
 * Function Name:
 * Description:
 * Parameters:
 * IN:
 * OUT:
 * Status Return:
 * Note:    
 * */
CARD8  viaReadVgaIo(CARD16 ioName)
{
    CARD8  ioData;
    CARD8  ioPortIndex;
    CARD32 ioAddress;

    /*To use the index's bit8 to represent CR or SR. 00=CR ; 01=SR.*/
    ioPortIndex = ioName & REG_INDEX_MASK;
    ioAddress = viaGetVgaIoAddress(ioName);
    MMIO_WR8(ioAddress, ioPortIndex);
    ioData = MMIO_RD8((ioAddress + 1));
    return ioData;
}

/*
 * Function Name:
 * Description:
 * Parameters:
 * IN:
 * OUT:
 * Status Return:
 * Note:    
 * */
CARD32   viaWriteVgaIo(CARD16 ioName, CARD8 data)
{
    CARD8 ioPortIndex;
    CARD32 ioAddress;

    /*To use the index's bit8 to represent CR or SR. 00=CR ; 01=SR.*/
    ioPortIndex = ioName & REG_INDEX_MASK;
    ioAddress = viaGetVgaIoAddress(ioName);
    MMIO_WR8(ioAddress, ioPortIndex);
    MMIO_WR8((ioAddress + 1), data);
    return TRUE;
}

/*
 * Function Name:
 * Description:
 * Parameters:
 * IN:
 * OUT:
 * Status Return:
 * Note:    
 * */
CARD32   viaWriteVgaIoBits(CARD16 ioName, CARD8 data, CARD8 bitsMask)
{
    CARD8  tmpIoData;
    CARD8  ioPortIndex;
    CARD32 ioAddress;

    /*To use the index's bit8 to represent CR or SR. 00=CR ; 01=SR.*/
    ioPortIndex = ioName & REG_INDEX_MASK;
    ioAddress = viaGetVgaIoAddress(ioName);
    MMIO_WR8(ioAddress, ioPortIndex);
    tmpIoData = MMIO_RD8(ioAddress + 1);    
    MMIO_WR8((ioAddress + 1), (data & bitsMask) | (tmpIoData & (~bitsMask)));
    return TRUE;
}

/*
 * Function Name:
 * Description:
 * Parameters:
 * IN:
 * OUT:
 * Status Return:
 * Note:    
 * */
CARD32   viaWriteVgaIoMultiBits(ViaIoReg regTable[], CARD32 itemNum)
{
    CARD32 i;
	
    for (i = 0; i < itemNum; i++)
        viaWriteVgaIoBits(regTable[i].index, regTable[i].value, regTable[i].mask);
    return TRUE;
}

/*
 * Function Name: LoadRegs
 * Description: Write centain value to some bits of multiple register
 * Parameters:
 * IN:
 * regValue: centain value
 * regAmount: the amount of multiple register
 * pRegBit: the struct indicated which bits of register should be written
 * OUT:
 * Status Return:
 * Note:    
 * */
void viaLoadRegs(CARD32 regValue, CARD32 regAmount, ViaIoBitPtr pRegBit)
{
    CARD32 regMask;
    CARD32 bitNum = 0;
    CARD32 data;
    CARD32 i,j;
    CARD32 shiftNextReg;
    CARD32 startIndex, endIndex;
    CARD32 crIndex;
    CARD32 getBit;

    for (i = 0; i < regAmount; i++) {
        regMask = 0;
        data = 0;
        startIndex = pRegBit[i].startBit;
        endIndex = pRegBit[i].endBit;
        crIndex = pRegBit[i].ioAddr;

        shiftNextReg = bitNum;

        for (j = startIndex; j <= endIndex; j++) {
            regMask = regMask | (BIT0 << j);
            getBit = (regValue & (BIT0 << bitNum));
            data = data | ((getBit >> shiftNextReg) << startIndex);
            bitNum++;
        }
       
        viaWriteVgaIoBits(crIndex, data, regMask);
    }
}

/*
 * Function Name: UnlockCrtc
 * Description: Unlock CRTC register
 * Parameters:
 * IN:
 * OUT:
 * Status Return:
 * Note:    
 * */
__inline__ void viaUnlockCrtc(void)
{
    viaWriteVgaIoBits(REG_CR11, 0, BIT7);
    viaWriteVgaIoBits(REG_CR47, 0, BIT0);
}

/*
 * Function Name: LockCrtc
 * Description: Lock CRTC register
 * Parameters:
 * IN:
 * OUT:
 * Status Return:
 * Note:    
 * */
__inline__ void viaLockCrtc(void)
{
    viaWriteVgaIoBits(REG_CR11, BIT7, BIT7);
}

/*
 * Function Name:MemSet
 *       
 * Description:This function set the memory region started at "pStart" and
 * length "a_LengthInBytes" to 0.
 *                     
 * Parameters:
 * IN:pStart------memory start address
 * lengthInBytes------memory length(in bytes).
 * */
void viaMemSet(void* pStart,  CARD32 lengthInBytes)
{
    CARD8 *pchar = (CARD8 *)pStart;
    CARD32 i = 0;

    if(pStart == NULL)
        return ;

    for(i = 0; i < lengthInBytes; i++)
        *pchar++ = (CARD8)0;
    return;
}

