/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*************************************************************************
 *
 *  File:       via_cursor.c
 *  Content:    Hardware cursor support for VIA/S3G UniChrome
 *
 ************************************************************************/

#include "via_driver.h"
#include "cursorstr.h"
#include "via_eng_regs.h"
#include "via_rotate.h"

#define	USESwScaling

static void VIALoadCursorImage(ScrnInfoPtr pScrn, unsigned char *src);
void VIASetCursorPosition(ScrnInfoPtr pScrn, int x, int y);
static void VIASetCursorColors(ScrnInfoPtr pScrn, int bg, int fg);
static Bool VIAUseHWCursor(ScreenPtr pScreen, CursorPtr pCurs);
static Bool VIAHiUseHWCursorARGB(ScreenPtr pScreen,CursorPtr pCursor);
static void VIAHiLoadCursorARGB (ScrnInfoPtr pScrn, CursorPtr pCurs);
extern void VIACheckCursorTypeToUse(ScrnInfoPtr pScrn);
void VIALoadIconImage(ScrnInfoPtr pScrn, CARD32* src);

void VIAARGBCursorInit(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;
    CARD32 dwHiControl;
    
    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VIAARGBCursorInit\n"));

    switch(pBIOSInfo->Chipset )
    {
        case VIA_CX700:
        case VIA_P4M890:
        case VIA_P4M900:
        case VIA_VX800:
		case VIA_VX855:
            /* set 0 as transparent color key */
            VIASETREG(PRIM_HI_TRANSCOLOR, 0);
            VIASETREG(PRIM_HI_FIFO,0x0D000D0F);
            VIASETREG(PRIM_HI_INVTCOLOR, 0X00FFFFFF);
            VIASETREG(V327_HI_INVTCOLOR, 0X00FFFFFF);
            dwHiControl = VIAGETREG(PRIM_HI_CTRL);
            
            /* Turn cursor off. */
            VIASETREG(PRIM_HI_CTRL, dwHiControl & 0xFFFFFFFA);                    

			if (pVia->useRandR) {
	            VIASETREG(PRIM_HI_FBOFFSET, pVia->HiStartBuf[0]);
	            VIASETREG(HI_FBOFFSET, pVia->HiStartBuf[1]);
			} else {
				/* two hi use the same memory*/
	            VIASETREG(PRIM_HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIBufStart);
	            VIASETREG(HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIBufStart);
			}
            
            /* set 0 as transparent color key */
            VIASETREG(HI_TRANSPARENT_COLOR, 0);
            VIASETREG(HI_INVTCOLOR, 0X00FFFFFF);
            VIASETREG(ALPHA_V3_PREFIFO_CONTROL, 0xE0000);
            VIASETREG(ALPHA_V3_FIFO_CONTROL, 0xE0F0000);
            dwHiControl = VIAGETREG(HI_CONTROL);
            
            /* Turn cursor off. */
            VIASETREG(HI_CONTROL, dwHiControl & 0xFFFFFFFA);
            break;
            
        default:     
			if (pVia->useRandR) 
				VIASETREG(HI_FBOFFSET, pVia->HiStartBuf[0]);
			else
            	VIASETREG(HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIBufStart);
            VIASETREG(HI_TRANSPARENT_COLOR, 0);
            VIASETREG(HI_INVTCOLOR, 0X00FFFFFF);
            VIASETREG(ALPHA_V3_PREFIFO_CONTROL, 0xE0000);
            VIASETREG(ALPHA_V3_FIFO_CONTROL, 0xE0F0000);
            dwHiControl = VIAGETREG(HI_CONTROL);

            /* Turn cursor off. */
            VIASETREG(HI_CONTROL, dwHiControl & 0xFFFFFFFA);                         
    }
}

/*Note: Randr1.2 doesn't use this function*/
Bool
VIAHWCursorInit(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    xf86CursorInfoPtr infoPtr;

    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VIAHWCursorInit\n"));
    infoPtr = xf86CreateCursorInfoRec();
    if (!infoPtr)
        return FALSE;

    pVia->CursorInfoRec = infoPtr;

    infoPtr->MaxWidth = MAX_CURS;
    infoPtr->MaxHeight = MAX_CURS;
    infoPtr->Flags = HARDWARE_CURSOR_SOURCE_MASK_INTERLEAVE_64 |
                     HARDWARE_CURSOR_AND_SOURCE_WITH_MASK |
                     /*HARDWARE_CURSOR_SWAP_SOURCE_AND_MASK |*/
                     HARDWARE_CURSOR_TRUECOLOR_AT_8BPP |
                     HARDWARE_CURSOR_INVERT_MASK |
                     HARDWARE_CURSOR_BIT_ORDER_MSBFIRST|
                     0;
    
    infoPtr->SetCursorColors = VIASetCursorColors;
    infoPtr->SetCursorPosition = VIASetCursorPosition;
    infoPtr->LoadCursorImage = VIALoadCursorImage;
    infoPtr->HideCursor = VIAHideCursor;
    infoPtr->ShowCursor = VIAShowCursor;
    infoPtr->UseHWCursor = VIAUseHWCursor;

  
#ifdef ARGB_CURSOR
    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "We will use HWCursor ARGB\n"));
    
    /*add to enable ARGB cursor */
    infoPtr->UseHWCursorARGB = VIAHiUseHWCursorARGB;
    infoPtr->LoadCursorARGB = VIAHiLoadCursorARGB;
#endif
 
    pVia->pScreensPublicInfo->CursorInfo.CursorBG = 0xffffff;
    pVia->pScreensPublicInfo->CursorInfo.CursorFG = 0x0;

#ifdef ARGB_CURSOR
    VIAARGBCursorInit(pScrn);
#endif

    return xf86InitCursor(pScreen, infoPtr);
}

/*Note: Randr1.2 doesn't use this function*/
static Bool VIAUseHWCursor(ScreenPtr pScreen, CursorPtr pCurs)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);

    /* Robust enhancement, for OLPC/F11 Big Cursor/Pointer Theme,
     * When choose 64x64 cursor, the XServer will pass large width (68 exactly) 
     * to X driver. 
     * */
    if ( (pCurs->bits->width > MAX_CURS) 
         ||(pCurs->bits->height > MAX_CURS) ) {
        return FALSE;
    }
    
    pVia->pScreensPublicInfo->CursorInfo.width = pCurs->bits->width;
    pVia->pScreensPublicInfo->CursorInfo.height = pCurs->bits->height;
    
    /*Need to confine hotx, hoty to MAX_CURS?*/
    pVia->pScreensPublicInfo->CursorInfo.hotx = pCurs->bits->xhot;
    pVia->pScreensPublicInfo->CursorInfo.hoty = pCurs->bits->yhot;
	
    return pVia->pScreensPublicInfo->UseHwCursor ? TRUE: FALSE;
}


void VIAShowCursor(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;
    viaGfxInfoPtr viaGfxInfo = pVia->pVidData->viaGfxInfo;   
    
    switch(pBIOSInfo->Chipset )
    {
        case VIA_CX700:
        case VIA_P4M890:
        case VIA_P4M900:
        case VIA_VX800:
		case VIA_VX855:	
             /* Turn on Hardware icon Cursor */
             if(!pBIOSInfo->DuoView)
            {
                switch(viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse)   
                {
                     /* Turn on Hardware icon Cursor */
                    case IGA1:
                      default:
                        VIASETREG(PRIM_HI_CTRL, 0x36000005);
                        break;
                    case IGA2:
                        VIASETREG(HI_CONTROL, 0xB6000005);
                        break;
                }
            }
            else
            {
                VIASETREG(PRIM_HI_CTRL, 0x36000005);
                VIASETREG(HI_CONTROL, 0xb6000005);
            }
             break;
        
        default:
            switch(viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse)
            {
                 /* Turn on Hardware icon Cursor */
            case IGA1:
              default:    
                /*enableV4 Window Pre-fetch [bit 30]for solving VT3336  bug: when do some actions such as play video cursor will jump*/
                VIASETREG(HI_CONTROL, 0x76000005);
                break;                  
            case IGA2:                    
                VIASETREG(HI_CONTROL, 0xf6000005);
                break;
            }
    }
}


void VIAHideCursor(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;
    CARD32 dwHiControl;
    viaGfxInfoPtr viaGfxInfo = pVia->pVidData->viaGfxInfo;
    
    switch(pVia->pBIOSInfo->Chipset )
    {
        case VIA_CX700:
        case VIA_P4M890:
        case VIA_P4M900:
        case VIA_VX800:
		case VIA_VX855:	
            if(!pBIOSInfo->DuoView)
            {
                switch(viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse)   
                {
                case IGA1:
                  default:
                    dwHiControl = VIAGETREG(PRIM_HI_CTRL);
                    /* Turn hardware icon cursor off. */
                    VIASETREG(PRIM_HI_CTRL, dwHiControl & 0xFFFFFFFA);
                    break;
                case IGA2:
                    dwHiControl = VIAGETREG(HI_CONTROL);
                    /* Turn hardware icon cursor off. */
                    VIASETREG(HI_CONTROL, dwHiControl & 0xFFFFFFFA);
                    break;
                }
            }
            else
            {
                dwHiControl = VIAGETREG(PRIM_HI_CTRL);
                /* Turn hardware icon cursor off. */
                VIASETREG(PRIM_HI_CTRL, dwHiControl & 0xFFFFFFFA);
                dwHiControl = VIAGETREG(HI_CONTROL);
                /* Turn hardware icon cursor off. */
                VIASETREG(HI_CONTROL, dwHiControl & 0xFFFFFFFA);
            }
            break;
        default:
            dwHiControl = VIAGETREG(HI_CONTROL);
            /* Turn hardware icon cursor off. */
            VIASETREG(HI_CONTROL, dwHiControl & 0xFFFFFFFA);
    }
}

void VIAGetHWCursorRealPosition(ScrnInfoPtr pScrn, int iga_path, int x, int y, int frameX0, int frameY0,
        int *xpos, int *ypos, unsigned char *xoff, unsigned char *yoff)
{
    VIAPtr          pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;
    IGASETTINGINFO *IGASettingInfo = NULL;
    VIA3DSclSCNParasPtr pTarget3DSclSCRNInfo = NULL;
    int IsShrinkEnable = pBIOSInfo->Is3DScalingEnable & DISP_3D_SCALING_ENABLE;
    int ScreenSizeH, ScreenSizeV;
    int Hotx = 0, Hoty = 0;
    int ActiveDevs = 0;
    int dx, dy;

    switch (iga_path)
    {
        case IGA1:
          default:
            IGASettingInfo = &pBIOSInfo->IGA1SettingInfo;
            break;
        case IGA2:
            IGASettingInfo = &pBIOSInfo->IGA2SettingInfo;
            break;
    }

    Hotx = pVia->pScreensPublicInfo->CursorInfo.hotx;
    Hoty = pVia->pScreensPublicInfo->CursorInfo.hoty;

    DoConvertCoordinates(&Hotx, &Hoty, MAX_CURS - 1, MAX_CURS - 1,
                              pVia->RotateDegree);

    if (IGASettingInfo->IsPanning)
    {
        ScreenSizeH = pBIOSInfo->HDisplay;
        ScreenSizeV = pBIOSInfo->VDisplay;

        /* Panning + Rotate support, to adjust position of X,Y */
        switch(pVia->RotateDegree) {
        /*On Rotate 90/270, virtualX/virtualY is swapped*/
        case VIA_ROTATE_DEGREE_90:
            dx = pScrn->virtualY - pBIOSInfo->HDisplay;
            x -= dx;
            break;

        case VIA_ROTATE_DEGREE_270:
            dy = pScrn->virtualX - pBIOSInfo->VDisplay;
            y -= dy;
            break;

        case VIA_ROTATE_DEGREE_180:
            dx = pScrn->virtualX - pBIOSInfo->HDisplay;
            dy = pScrn->virtualY - pBIOSInfo->VDisplay;
            x -= dx;
            y -= dy;
            break;

        default:
            break;
        }
    }
    else if (IGASettingInfo->IsDownScaling)
    {
        ScreenSizeH = pBIOSInfo->HorDownScaleSize;
        ScreenSizeV = pBIOSInfo->VerDownScaleSize;
        x += frameX0;
        y += frameY0;
        x = (x * ScreenSizeH) / pBIOSInfo->HDisplay;
        y = (y * ScreenSizeV) / pBIOSInfo->VDisplay;
        Hotx = (Hotx * ScreenSizeH) / pBIOSInfo->HDisplay;
        Hoty = (Hoty * ScreenSizeV) / pBIOSInfo->VDisplay;
    }
    else if (IsShrinkEnable && IGASettingInfo->IsDISP3DScaling)
    {
        ScreenSizeH = IGASettingInfo->DISP3DSCALCTRLInfo.RealHActive;
        ScreenSizeV = IGASettingInfo->DISP3DSCALCTRLInfo.RealVActive;
        ActiveDevs = IGASettingInfo->deviceActived;
        IGASettingInfo->DISP3DSCALCTRLInfo.CursorX = x;
        IGASettingInfo->DISP3DSCALCTRLInfo.CursorY = y;
        x += frameX0;
        y += frameY0;
        x = (x * ScreenSizeH) / pBIOSInfo->CrtcHDisplay;
        y = (y * ScreenSizeV) / pBIOSInfo->CrtcVDisplay;

        switch(ActiveDevs)
        {
            case VIA_DEVICE_DFP:
                pTarget3DSclSCRNInfo = &(pBIOSInfo->TMDSSettingInfo.DVI3DScalInfo);
                break;                
            case VIA_DEVICE_LCD:
                pTarget3DSclSCRNInfo = &(pBIOSInfo->LVDSSettingInfo.LCD3DScalInfo);
                break;
        }

        x -= (x * 2 -ScreenSizeH) * (pTarget3DSclSCRNInfo->XSIZEOffset) /ScreenSizeH;
        y -= (y * 2 -ScreenSizeV) * (pTarget3DSclSCRNInfo->YSIZEOffset) /ScreenSizeV;
        
        x += pTarget3DSclSCRNInfo->XPOSITIONOffset;
        y -= pTarget3DSclSCRNInfo->YPOSITIONOffset;

        Hotx = (Hotx * ScreenSizeH) / pBIOSInfo->CrtcHDisplay;
        Hoty = (Hoty * ScreenSizeV) / pBIOSInfo->CrtcVDisplay;
    }
    else
    {
        ScreenSizeH = pBIOSInfo->VirtualX;
        ScreenSizeV = pBIOSInfo->VirtualY;
        x += frameX0;
        y += frameY0;
    }

    /*compute x*/
    if(x > ScreenSizeH-1)
    {
        *xpos = ScreenSizeH-1;
        *xoff = 0;
    }
    else if (x < 0) 
    {
        *xpos = 0;
        *xoff = (-x) & 0xFF;
        if (Hotx > 0)
        {
            if (*xoff >= Hotx)
            {
                *xoff = Hotx;
            }
        }
        else
        {
            *xoff = 0; 
        }
    } 
    else 
    {
        *xpos = x;
        *xoff = 0;
    }

    /*compute y*/
    if(y > ScreenSizeV-1)
    {
        *ypos = ScreenSizeV-1;
        *yoff = 0;
    }
    else if (y < 0) 
    {
        *ypos = 0;
        *yoff = (-y) & 0xFF;
        if (Hoty > 0)
        {
            if (*yoff >= Hoty)
            {
                *yoff = Hoty;
            }
        }
        else
        {
            *yoff = 0; 
        }
    } 
    else 
    {
        *ypos = y;
        *yoff = 0;
    }
}

void VIAMonoToArgbTrans(ScrnInfoPtr pScrn, unsigned char* src, CARD32* argb)
{
    VIAPtr pVia = VIAPTR(pScrn);
    VIACursorInfoPtr pCursorInfo = &pVia->pScreensPublicInfo->CursorInfo;
    int i, j, k;
    unsigned int bg, fg;
    unsigned char *AndPlane;
    unsigned char *XorPlane;
    unsigned int pitch = pVia->CursorInfoRec->MaxWidth >> 3;
    CARD32 pixel;
    int x_rot, y_rot;
    int w, h;
   
    w = (pitch << 3) - 1;
    h = pCursorInfo->height - 1;

    ConvertSize(pScrn, &w, &h);

    /* bg and fg should set alpha value to 0xff, then it can show on screens */
    bg = 0xff000000 | pVia->pScreensPublicInfo->CursorInfo.CursorBG;
    fg = 0xff000000 | pVia->pScreensPublicInfo->CursorInfo.CursorFG;
    XorPlane = src;
    AndPlane = XorPlane + pitch;
    for(i = 0; i < pVia->pScreensPublicInfo->CursorInfo.height; i++)
    {
        for(j = 0; j < pitch; j++)
        {
            for(k = 7; k >= 0; k--)
            {
                if (((AndPlane[j] >> k) & 0x1) != 0)
                    pixel = 0;  /* transparent */
                else
                    pixel = ((XorPlane[j] >> k) & 0x1) ? fg : bg;

                x_rot = (j << 3) + 7 - k;
                y_rot = i;
                DoConvertCoordinates(&x_rot, &y_rot, w, h,
                                     pVia->RotateDegree);

                argb[(y_rot * (pitch<<3)) + x_rot] = pixel;
            }
        }
        XorPlane += pitch * 2;
        AndPlane = XorPlane + pitch;
    }
}


static void VIALoadCursorImage(ScrnInfoPtr pScrn, unsigned char* src)
{
    VIAPtr pVia = VIAPTR(pScrn);
    ScrnInfoPtr pScrn0 = xf86Screens[0];
    VIAPtr pVia0 = VIAPTR(pScrn0);
    WaitIdle();
   
    memcpy(pVia0->pScreensPublicInfo->Image_buf, src, MAX_CURS * MAX_CURS/8*2 ); 
    memset((CARD8 *)pVia0->FBBase + pVia->pScreensPublicInfo->CursorInfo.HIBufStart, 0, 
               (pVia->CursorInfoRec->MaxWidth * pVia->CursorInfoRec->MaxHeight * 4));
    VIAMonoToArgbTrans(pScrn, src, (CARD32 *)(pVia0->FBBase + pVia->pScreensPublicInfo->CursorInfo.HIBufStart));

    if(!(pVia0->pBIOSInfo->Is3DScalingEnable & DISP_3D_SCALING_ENABLE)) {
        VIALoadIconImage(pScrn, (CARD32 *)(pVia0->FBBase + pVia->pScreensPublicInfo->CursorInfo.HIBufStart));
    }
}

void VIASetCursorPosition(ScrnInfoPtr pScrn, int x, int y)
{
    VIAPtr          pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;
    int             xpos, ypos;    
    unsigned char   xoff, yoff;
    viaGfxInfoPtr viaGfxInfo = pVia->pVidData->viaGfxInfo;
    
    ConvertCoordinatesOne(pScrn, &x, &y, MAX_CURS, MAX_CURS);

    switch(pBIOSInfo->Chipset )
    {
        case VIA_CX700:
        case VIA_P4M890:
        case VIA_P4M900:
        case VIA_VX800:
		case VIA_VX855:	
            if(!pBIOSInfo->DuoView)
            {
                switch(viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse)
                {
                case IGA1:
                  default:
                    /*Get IGA1 cursor position*/
                    VIAGetHWCursorRealPosition(pScrn, IGA1, x, y, pScrn->frameX0, pScrn->frameY0, &xpos, &ypos, &xoff, &yoff);
                    /*set hi position*/
                    VIASETREG(PRIM_HI_POSSTART, ((xpos << 16) | (ypos & 0x07ff)));
                    VIASETREG(PRIM_HI_CENTEROFFSET, ((xoff<< 16) | (yoff & 0x07ff)));
                    break;
                case IGA2:
                    /*Get IGA2 cursor position*/
                    VIAGetHWCursorRealPosition(pScrn, IGA2, x, y, pScrn->frameX0, pScrn->frameY0, &xpos, &ypos, &xoff, &yoff);
                    /*set hi position*/
                    VIASETREG(HI_POSSTART, ((xpos << 16) | (ypos & 0x07ff)));
                    VIASETREG(HI_CENTEROFFSET, ((xoff<< 16) | (yoff & 0x07ff)));
                    break;
                }
            }
            else
            {
                /*Get IGA1 cursor position*/
                VIAGetHWCursorRealPosition(pScrn, IGA1, x, y, pScrn->frameX0, pScrn->frameY0, &xpos, &ypos, &xoff, &yoff);

                /*set hi position*/
                VIASETREG(PRIM_HI_POSSTART, ((xpos << 16) | (ypos & 0x07ff)));
                VIASETREG(PRIM_HI_CENTEROFFSET, ((xoff<< 16) | (yoff & 0x07ff)));

                /*Get IGA2 cursor position*/
                VIAGetHWCursorRealPosition(pScrn, IGA2, x, y, pScrn->frameX0, pScrn->frameY0, &xpos, &ypos, &xoff, &yoff);
                
                /*set hi position*/
                VIASETREG(HI_POSSTART, ((xpos << 16) | (ypos & 0x07ff)));
                VIASETREG(HI_CENTEROFFSET, ((xoff<< 16) | (yoff & 0x07ff)));
            }
            break;
        default:
            /*Get IGA cursor position*/
            VIAGetHWCursorRealPosition(pScrn, viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse,
                                        x, y, pScrn->frameX0, pScrn->frameY0, &xpos, &ypos, &xoff, &yoff);
            /*set hi position*/
            VIASETREG(HI_POSSTART, ((xpos << 16) | (ypos & 0x07ff)));
            VIASETREG(HI_CENTEROFFSET, ((xoff<< 16) | (yoff & 0x07ff)));
            break;
    }
}


static void VIASetCursorColors(ScrnInfoPtr pScrn, int bg, int fg)
{
    ScrnInfoPtr pScrn0 = xf86Screens[0];
    VIAPtr pVia0 = VIAPTR(pScrn0);

    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VIASetCursorColors\n"));
    /* 
     * Because Xserver always load cursor image before set cursor color, if
     * use hardware icon simulate hardware cursor, hardware cursor should
     * retransfer and reload new hardware icon image.
     */
    if((fg != pVia0->pScreensPublicInfo->CursorInfo.CursorFG)
        || (bg != pVia0->pScreensPublicInfo->CursorInfo.CursorBG))
    {
        pVia0->pScreensPublicInfo->CursorInfo.CursorFG = fg;
        pVia0->pScreensPublicInfo->CursorInfo.CursorBG = bg;
        VIAMonoToArgbTrans(pScrn, (unsigned char *)(pVia0->pScreensPublicInfo->Image_buf),
                          (CARD32 *)(pVia0->FBBase + pVia0->pScreensPublicInfo->CursorInfo.HIBufStart));        
        if(!(pVia0->pBIOSInfo->Is3DScalingEnable & DISP_3D_SCALING_ENABLE)) {
            VIALoadIconImage(pScrn, (CARD32 *)(pVia0->FBBase + pVia0->pScreensPublicInfo->CursorInfo.HIBufStart));
        }
    }
}

#ifdef ARGB_CURSOR

/*enable ALPHA cursor. Note: Randr1.2 doesn't use this function*/
static Bool VIAHiUseHWCursorARGB(ScreenPtr pScreen, CursorPtr pCurs)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);

    /* Robust enhancement, for OLPC/F11 Big Cursor/Pointer Theme,
     * When choose 64x64 cursor, the XServer will pass large width (68 exactly) 
     * to X driver. It will cause garbage for some Icons. Fix it here.  
     * */
    if ( (pCurs->bits->width > MAX_CURS) 
         ||(pCurs->bits->height > MAX_CURS) ) {
        return FALSE;
    }
    
    /* DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VIAUseHWCursorARGB\n")); */
    pVia->pScreensPublicInfo->CursorInfo.width = pCurs->bits->width;
    pVia->pScreensPublicInfo->CursorInfo.height = pCurs->bits->height;
    
    /*Need to confine hotx, hoty to MAX_CURS?*/
    pVia->pScreensPublicInfo->CursorInfo.hotx = pCurs->bits->xhot;
    pVia->pScreensPublicInfo->CursorInfo.hoty = pCurs->bits->yhot;
	
    VIACheckCursorTypeToUse(pScrn);
    /* Switch SW/HW cursor dynamically. */
    if (pVia->pScreensPublicInfo->UseHwCursor)
    {
        return TRUE;
    }
    else
    {
        return FALSE;
    }
}


static void VIAHiLoadCursorARGB (ScrnInfoPtr pScrn, CursorPtr pCurs)
{
    VIAPtr pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr pBIOSInfo;
    ScrnInfoPtr pScrn0 = xf86Screens[0];
    VIAPtr pVia0 = VIAPTR(pScrn0);
	int x, y;
    CARD32 *image = pCurs->bits->argb;
    CARD32 *IconStartAddr; 
    xf86CursorInfoPtr pCursInfo;
    VIACursorInfoPtr pViaCursInfo;
    int x_rot, y_rot, w_rot, h_rot;

    /* DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "VIAHiLoadCursorARGB\n")); */
    pBIOSInfo = pVia->pBIOSInfo;
    pCursInfo = pVia->CursorInfoRec;
    pViaCursInfo = &pVia->pScreensPublicInfo->CursorInfo;
    IconStartAddr = (CARD32*) (pVia0->FBBase + pViaCursInfo->HIBufStart);

    /* fill with transparent */
    memset(IconStartAddr, 0,
           pCursInfo->MaxWidth * pCursInfo->MaxHeight * sizeof(CARD32));

    if (pBIOSInfo->Is3DScalingEnable & DISP_3D_SCALING_ENABLE) {
        for (y = 0; y < pViaCursInfo->height; y++) {
            for (x = 0; x < pViaCursInfo->width; x++) {
                IconStartAddr[(y * pCursInfo->MaxWidth) + x] = *(image++);
            }
        }
    }
    else {
        w_rot = pCursInfo->MaxWidth - 1;
        h_rot = pCursInfo->MaxHeight - 1;

        ConvertSize(pScrn, &w_rot, &h_rot);

        for (y = 0; y < pViaCursInfo->height; y++)
        {
            for (x = 0; x < pViaCursInfo->width; x++)
            {
                /* FIXME:  Lift the rotation calculation out of these loops. */
                x_rot = x;
                y_rot = y;
                DoConvertCoordinates(&x_rot, &y_rot,
                                     w_rot, h_rot,
                                     pVia->RotateDegree);
                IconStartAddr[(y_rot * pCursInfo->MaxWidth) + x_rot] = *(image++);
            }
        }
    }
    VIALoadIconImage(pScrn, IconStartAddr);
}

#endif

#ifdef USESwScaling
/*
   Use linear interpolation algorithm to scale up/down cursor image.
*/
int VIAARGBCursorSWScaling(VIAPtr pVia, unsigned char *pSrc, unsigned char *pDest,
	                     int SrcWidth, int SrcHeight, int DestWidth, int DestHeight)
{
    int xSrc, ySrc, xDest, yDest;
    int xParam, yParam;
    int i;
    int Bpp = 4;
    unsigned char *pPrevLine, *pNextLine;
    unsigned char *pTmp;
    unsigned char *pLeftUp, *pRightUp, *pLeftDown, *pRightDown;
    ScrnInfoPtr pScrn = xf86Screens[0];

    if( (DestWidth <= 0) && (DestHeight <= 0) )
    {
        return -1;
    }

    ConvertSize(pScrn, &SrcWidth, &SrcHeight);
    ConvertSize(pScrn, &DestWidth, &DestHeight);    

    for(yDest = 0; yDest <= DestHeight; yDest++)
    {
        pTmp = pDest + (pVia->CursorInfoRec->MaxWidth * yDest * Bpp);
        ySrc = yDest * SrcHeight / DestHeight;
        yParam = DestHeight - (yDest * SrcHeight % DestHeight);
        pPrevLine = pSrc + (pVia->CursorInfoRec->MaxWidth * ySrc * Bpp);
        ySrc++;

        if(yParam == DestHeight)
            pNextLine = pPrevLine;
        else
            pNextLine = pSrc + (pVia->CursorInfoRec->MaxWidth * ySrc * Bpp);
		
        for(xDest = 0; xDest <= DestWidth; xDest++)
        {
            xSrc = xDest * SrcWidth / DestWidth * Bpp;
            xParam = DestWidth - xDest * SrcWidth % DestWidth;
            pLeftUp = pPrevLine + xSrc;
            pRightUp = pLeftUp + Bpp;
            pLeftDown = pNextLine + xSrc;
            pRightDown = pLeftDown + Bpp;
            if ( xParam == DestWidth )
            {
                pRightUp = pLeftUp;
                pRightDown = pLeftDown;
            }
            for(i = 0; i < Bpp; i++)
            {
                *pTmp++ = (unsigned char)(unsigned int)(
                    (xParam * yParam * ((*pLeftUp++) - *pRightUp - *pLeftDown + *pRightDown) 
                    + DestWidth * yParam * (*pRightUp++)
                    + DestHeight * xParam * (*pLeftDown++) 
                    + (DestWidth * DestHeight - DestHeight * xParam - DestWidth * yParam) * (*pRightDown++)
                    + DestWidth * DestHeight / 2)
                    / ( DestWidth * DestHeight ));
            }
        }	
    }
    return 0;
}
#endif

int VIADoIconScaling(VIAPtr pVia, int iga_path, CARD32* src, int IconOffset)
{
    VIAPtr pVia0 = VIAPTR(xf86Screens[0]);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;
    VIADISP3DSCALCTRLPtr p3DSCALCTRLInfo = NULL;
    VIA3DSclSCNParasPtr pTarget3DSclSCRNInfo = NULL;
    IGASETTINGINFO *pIGASettingInfo = NULL;
    CARD32 *IconStartAddr = NULL;
    CARD32 *ptmp_buf = NULL;
    int x_rot, y_rot, w_rot, h_rot;
    ScrnInfoPtr pScrn = xf86Screens[0];
    xf86CursorInfoPtr pCursInfo;

    int x, y;
    CARD32 newHIwidth,newHIheight;
    CARD32 newHActive, newVActive;
    RECTL rScreenDest;
    int ret = -1;
#ifndef USESwScaling
    CARD32 lSrcPitch, dwSrcWidth, dwSrcHeight;
    RECTL rIconSrc;
#endif

    pCursInfo = pVia->CursorInfoRec;
    switch (iga_path)
    {
        case IGA1:
          default:
            pIGASettingInfo = &pBIOSInfo->IGA1SettingInfo;
            break;
        case IGA2:
            pIGASettingInfo = &pBIOSInfo->IGA2SettingInfo;
            break;
    }
    p3DSCALCTRLInfo = &(pIGASettingInfo->DISP3DSCALCTRLInfo);   
    switch(pIGASettingInfo->deviceActived)
    {
        case VIA_DEVICE_DFP:
            pTarget3DSclSCRNInfo = &(pBIOSInfo->TMDSSettingInfo.DVI3DScalInfo);
            break;                             
        case VIA_DEVICE_LCD:
            pTarget3DSclSCRNInfo = &(pBIOSInfo->LVDSSettingInfo.LCD3DScalInfo);
            break;
    }
    rScreenDest.left = rScreenDest.top = 0;
    rScreenDest.right = p3DSCALCTRLInfo->RealHActive;
    rScreenDest.bottom = p3DSCALCTRLInfo->RealVActive;

    /*software shrink adjust windows size*/
    rScreenDest.left += pTarget3DSclSCRNInfo->XSIZEOffset;
    rScreenDest.top += pTarget3DSclSCRNInfo->YSIZEOffset;
    rScreenDest.right -= pTarget3DSclSCRNInfo->XSIZEOffset; 
    rScreenDest.bottom -= pTarget3DSclSCRNInfo->YSIZEOffset;

    rScreenDest.left += pTarget3DSclSCRNInfo->XPOSITIONOffset;
    rScreenDest.right += pTarget3DSclSCRNInfo->XPOSITIONOffset;
    rScreenDest.top -= pTarget3DSclSCRNInfo->YPOSITIONOffset;
    rScreenDest.bottom -= pTarget3DSclSCRNInfo->YPOSITIONOffset;

    newHActive = rScreenDest.right - rScreenDest.left;
    newVActive = rScreenDest.bottom - rScreenDest.top;
    newHIwidth = pVia->pScreensPublicInfo->CursorInfo.width * newHActive / (pBIOSInfo->CrtcHDisplay);
    newHIheight = pVia->pScreensPublicInfo->CursorInfo.height * newVActive / (pBIOSInfo->CrtcVDisplay);
#ifdef USESwScaling
    ptmp_buf = (CARD32 *)pVia->pScreensPublicInfo->Image_buf;
    if(ptmp_buf) {
        memset(ptmp_buf, 0, pCursInfo->MaxWidth * pCursInfo->MaxHeight * sizeof(CARD32)); 
    }
    IconStartAddr = (CARD32 *)(pVia0->FBBase + IconOffset);
    /* fill with transparent */
    memset(IconStartAddr, 0,
           pCursInfo->MaxWidth * pCursInfo->MaxHeight * sizeof(CARD32));

    if(pVia->RotateDegree){     /*Rotate Case (No Zero Degree)*/
        ret = VIAARGBCursorSWScaling(pVia, (unsigned char *)src, (unsigned char *)(ptmp_buf),
            (pVia->pScreensPublicInfo->CursorInfo.width - 1), (pVia->pScreensPublicInfo->CursorInfo.height - 1),
            (newHIwidth - 1), (newHIheight - 1));
        if(ret){
            return -1; 
        }
    }
    else {      /*Normal Case*/
        ret = VIAARGBCursorSWScaling(pVia, (unsigned char *)src, (unsigned char *)(IconStartAddr),
            (pVia->pScreensPublicInfo->CursorInfo.width - 1), (pVia->pScreensPublicInfo->CursorInfo.height - 1),
            (newHIwidth - 1), (newHIheight - 1));
        if(ret){
            return -1; 
        }
    }
#else            
    rIconSrc.left = rIconSrc.top = 0;
    rIconSrc.right = pVia->pScreensPublicInfo->CursorInfo.width;
    rIconSrc.bottom = pVia->pScreensPublicInfo->CursorInfo.height;
    dwSrcWidth = dwSrcHeight = MAX_CURS;
    /*Note the alpha cursor should be 32bpp */
    lSrcPitch = MAX_CURS * 4;
    /* do cursor image adjust*/
    hw3d_ARGBCursorImageAdjust(pVia, p3DSCALCTRLInfo, (CARD32)pVia->pScreensPublicInfo->CursorInfo.HIBufStart, 
                                   IconOffset, rIconSrc, lSrcPitch, dwSrcWidth, dwSrcHeight);
#endif

    if(pVia->RotateDegree){
        w_rot = pCursInfo->MaxWidth - 1;
        h_rot = pCursInfo->MaxHeight - 1;

        ConvertSize(pScrn, &w_rot, &h_rot);

        for (y = 0; y < pCursInfo->MaxHeight; y++)
        {
            for (x = 0; x < pCursInfo->MaxWidth; x++)
            {
                /* FIXME:  Lift the rotation calculation out of these loops. */
                x_rot = x;
                y_rot = y;
                DoConvertCoordinates(&x_rot, &y_rot,
                                     w_rot, h_rot,
                                     pVia->RotateDegree);
                IconStartAddr[(y_rot * pCursInfo->MaxWidth) + x_rot] = *(ptmp_buf++);
            }
        }
    }
    return 0;
}

void VIALoadIconImage(ScrnInfoPtr pScrn, CARD32* src)
{
    VIAPtr pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo = pVia->pBIOSInfo;
    int     IconOnIGA1IsHandled = FALSE, IconOnIGA2IsHandled = FALSE;
    viaGfxInfoPtr viaGfxInfo = pVia->pVidData->viaGfxInfo;
    int ret = -1;

    if (pBIOSInfo->DuoView)
    {
        if (pBIOSInfo->Is3DScalingEnable & DISP_3D_SCALING_ENABLE)
        {   
            if (pVia->pBIOSInfo->IGA1SettingInfo.IsDISP3DScaling)
            {
                ret = VIADoIconScaling(pVia, IGA1, src, pVia->pScreensPublicInfo->CursorInfo.HIPrimScalingBufStart);
                if(ret) {
                    return;
                }
                /* Point to Hw Icon Primary Scaling buffer */
                VIASETREG(PRIM_HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIPrimScalingBufStart);
                IconOnIGA1IsHandled = TRUE;
            }

            if (pVia->pBIOSInfo->IGA2SettingInfo.IsDISP3DScaling)
            {
                ret = VIADoIconScaling(pVia, IGA2, src, pVia->pScreensPublicInfo->CursorInfo.HISecScalingBufStart);
                if(ret) {
                    return;
                }
                VIASETREG(HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HISecScalingBufStart);
                IconOnIGA2IsHandled = TRUE;
            }
        }

        if (!IconOnIGA1IsHandled)
        {
            VIASETREG(PRIM_HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIBufStart);
        }
        if (!IconOnIGA2IsHandled)
        {
            VIASETREG(HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIBufStart);
        }
    }
    else
    {
        switch(pBIOSInfo->Chipset)
        {
            case VIA_CX700:
            case VIA_P4M890:
            case VIA_P4M900:
            case VIA_VX800:
            case VIA_VX855: 
                if(viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse != IGA2)
                {
                    if (pBIOSInfo->Is3DScalingEnable & DISP_3D_SCALING_ENABLE)
                    {
                        ret = VIADoIconScaling(pVia, IGA1, src, pVia->pScreensPublicInfo->CursorInfo.HIPrimScalingBufStart);
                        if(ret) {
                            return;
                        }
                        VIASETREG(PRIM_HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIPrimScalingBufStart);
                    }
                    else
                        VIASETREG(PRIM_HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIBufStart);
                }
                else
                {
                    if (pBIOSInfo->Is3DScalingEnable & DISP_3D_SCALING_ENABLE)
                    {
                        ret = VIADoIconScaling(pVia, IGA2, src, pVia->pScreensPublicInfo->CursorInfo.HISecScalingBufStart);
                        if(ret) {
                            return;
                        }
                        VIASETREG(HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HISecScalingBufStart);                        
                    }
                    else
                        VIASETREG(HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIBufStart);
                }
                break;
            default:
                /*One HW Icon case: Duo view case uses SW cursor!*/
                if (pBIOSInfo->Is3DScalingEnable & DISP_3D_SCALING_ENABLE)
                {
                    ret = VIADoIconScaling(pVia, viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse, 
                                    src, pVia->pScreensPublicInfo->CursorInfo.HIPrimScalingBufStart);
                    if(ret) {
                        return;
                    }

                    VIASETREG(HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIPrimScalingBufStart);
                }
                else
                    VIASETREG(HI_FBOFFSET, pVia->pScreensPublicInfo->CursorInfo.HIBufStart);
                break;
        }
    }
}

