/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
 

#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <linux/version.h>
#include "via_driver.h"
#ifdef VIA_RANDR12_SUPPORT
#include "globals.h"
#include "debug.h"      /* for DBG_DD */
#include "xf86RAC.h"
#include "xf86Priv.h"
#include "xf86Crtc.h"
#include "xf86RandR12.h"
#include "via_bios.h"
#include "via_rotate.h"
#include "cursorstr.h"
#include "via_eng_regs.h"
#include "macros.h"

#include "via_modedata.h"
#include "via_output.h"

extern Bool via_module_loaded;
extern NEWVIAGRAPHICINFO NEWVIAGraphicInfo;
extern void TranslateGFXInfo(viaGfxInfoPtr viaGfxInfo, LPNEWVIAGRAPHICINFO lpNEWVIAGraphicInfo);



/* Fill VPIT parameters, write Misc/SR/GR/AR register*/
static void 
viaLoadVpitRegs(void)
{
	unsigned long  i = 0;
	unsigned char data = 0;
    
    /* Switching AR00[5](PAS) to 0 will allow the CPU to change the palette registers.Meanwhile, the monitor will be blank.  */
    data = STANDVGA_R8(REG_STATUS);
    STANDVGA_W8(REG_ZAR, 0x00);

    /* Fill VPIT Parameters */
    /* Write Misc Register */
    STANDVGA_W8(REG_WMISC, VpitRegs.Misc);

    /* Write Sequencer */
    for (i = 1; i <= STD_SR_NUM ; i++) {
        STANDVGA_W8(REG_ZSR, i);
        STANDVGA_W8(REG_ZSR+1,VpitRegs.SR[i-1]);
    }

    /* Write Graphic Controller */
    for (i = 0; i < STD_GR_NUM ; i++) {
        STANDVGA_W8(REG_ZGR, i);
        STANDVGA_W8(REG_ZGR+1, VpitRegs.GR[i]);
    }

    /* Write Attribute Controller */
    for (i = 0; i < STD_AR_NUM ; i++) {
        data = STANDVGA_R8(REG_STATUS);
        STANDVGA_W8(REG_ZAR, i);
        STANDVGA_W8(REG_ZAR, VpitRegs.AR[i]);
    }

    /* Switching AR00[5](PAS) to 1 will NOT allow the CPU to change the palette registers.
       Meanwhile, the monitor will display normally. */
    data = STANDVGA_R8(REG_STATUS);
    STANDVGA_W8(REG_ZAR, 0x20);
}

/*Load CRTC timing registers*/
static void 
viaLoadCrtcTiming(DisplayTiming crtcTiming, int igaPath)
{
    unsigned char	i = 0;
    unsigned char 	regAmount = 0;
    unsigned long 	regValue = 0;
    ViaIoBitPtr		reg = NULL;

    /* Check input variable igaPath*/
    if ((igaPath != IGA1) && (igaPath != IGA2))
        return;
	
    for (i = 0; i < 12; i++) {
        if (igaPath == IGA1) {
            switch (i) {
                case VIA_H_TOTAL_INDEX:
                    regValue = VIA_IGA1_HOR_TOTAL_FORMULA(crtcTiming.HorTotal);
                    regAmount = Iga1CrtcRegs.HTotal.RegNum;
                    reg = Iga1CrtcRegs.HTotal.Reg;
                    break;
                case VIA_H_ADDR_INDEX:
                    regValue = VIA_IGA1_HOR_ADDR_FORMULA(crtcTiming.HorAddr);
                    regAmount = Iga1CrtcRegs.HAddr.RegNum;
                    reg = Iga1CrtcRegs.HAddr.Reg;
                    break;
                case VIA_H_BLANK_START_INDEX:
                    regValue = VIA_IGA1_HOR_BLANK_START_FORMULA(crtcTiming.HorBlankStart);
                    regAmount = Iga1CrtcRegs.HBlankSt.RegNum;
                    reg = Iga1CrtcRegs.HBlankSt.Reg;
                    break;
                case VIA_H_BLANK_END_INDEX:
                    regValue = VIA_IGA1_HOR_BLANK_END_FORMULA(crtcTiming.HorBlankEnd);
                    regAmount = Iga1CrtcRegs.HBlankEd.RegNum;
                    reg = Iga1CrtcRegs.HBlankEd.Reg;
                    break;
                case VIA_H_SYNC_START_INDEX:
                    regValue = VIA_IGA1_HOR_SYNC_START_FORMULA(crtcTiming.HorSyncStart);
                    regAmount = Iga1CrtcRegs.HSyncSt.RegNum;
                    reg = Iga1CrtcRegs.HSyncSt.Reg;
                    break;
                case VIA_H_SYNC_END_INDEX:
                    regValue = VIA_IGA1_HOR_SYNC_END_FORMULA(crtcTiming.HorSyncEnd);
                    regAmount = Iga1CrtcRegs.HSyncEd.RegNum;
                    reg = Iga1CrtcRegs.HSyncEd.Reg;
                    break;
                case VIA_V_TOTAL_INDEX:
                    regValue = VIA_IGA1_VER_TOTAL_FORMULA(crtcTiming.VerTotal);
                    regAmount = Iga1CrtcRegs.VTotal.RegNum;
                    reg = Iga1CrtcRegs.VTotal.Reg;
                    break;
                case VIA_V_ADDR_INDEX:
                    regValue = VIA_IGA1_VER_ADDR_FORMULA(crtcTiming.VerAddr);
                    regAmount = Iga1CrtcRegs.VAddr.RegNum;
                    reg = Iga1CrtcRegs.VAddr.Reg;
                    break;
                case VIA_V_BLANK_START_INDEX:
                    regValue = VIA_IGA1_VER_BLANK_START_FORMULA(crtcTiming.VerBlankStart);
                    regAmount = Iga1CrtcRegs.VBlankSt.RegNum;
                    reg = Iga1CrtcRegs.VBlankSt.Reg;
                    break;
                case VIA_V_BLANK_END_INDEX:
                    regValue = VIA_IGA1_VER_BLANK_END_FORMULA(crtcTiming.VerBlankEnd);
                    regAmount = Iga1CrtcRegs.VBlankEd.RegNum;
                    reg = Iga1CrtcRegs.VBlankEd.Reg;
                    break;
                case VIA_V_SYNC_START_INDEX:
                    regValue = VIA_IGA1_VER_SYNC_START_FORMULA(crtcTiming.VerSyncStart);
                    regAmount = Iga1CrtcRegs.VSyncSt.RegNum;
                    reg = Iga1CrtcRegs.VSyncSt.Reg;
                    break;
                case VIA_V_SYNC_END_INDEX:
                    regValue = VIA_IGA1_VER_SYNC_END_FORMULA(crtcTiming.VerSyncEnd);
                    regAmount = Iga1CrtcRegs.VSyncEd.RegNum;
                    reg = Iga1CrtcRegs.VSyncEd.Reg;
                    break;
            }
        } else {
            switch (i) {
                case VIA_H_TOTAL_INDEX:
                    regValue = VIA_IGA2_HOR_TOTAL_FORMULA(crtcTiming.HorTotal);
                    regAmount = Iga2CrtcRegs.HTotal.RegNum;
                    reg = Iga2CrtcRegs.HTotal.Reg;
                    break;
                case VIA_H_ADDR_INDEX:
                    regValue = VIA_IGA2_HOR_ADDR_FORMULA(crtcTiming.HorAddr);
                    regAmount = Iga2CrtcRegs.HAddr.RegNum;
                    reg = Iga2CrtcRegs.HAddr.Reg;
                    break;
                case VIA_H_BLANK_START_INDEX:
                    regValue = VIA_IGA2_HOR_BLANK_START_FORMULA(crtcTiming.HorBlankStart);
                    regAmount = Iga2CrtcRegs.HBlankSt.RegNum;
                    reg = Iga2CrtcRegs.HBlankSt.Reg;
                    break;
                case VIA_H_BLANK_END_INDEX:
                    regValue = VIA_IGA2_HOR_BLANK_END_FORMULA(crtcTiming.HorBlankEnd);
                    regAmount = Iga2CrtcRegs.HBlankEd.RegNum;
                    reg = Iga2CrtcRegs.HBlankEd.Reg;
                    break;
                case VIA_H_SYNC_START_INDEX:
                    regValue = VIA_IGA2_HOR_SYNC_START_FORMULA(crtcTiming.HorSyncStart);
                    regAmount = Iga2CrtcRegs.HSyncSt.RegNum;                        
                    reg = Iga2CrtcRegs.HSyncSt.Reg;
                    break;
                case VIA_H_SYNC_END_INDEX:
                    regValue = VIA_IGA2_HOR_SYNC_END_FORMULA(crtcTiming.HorSyncEnd);
                    regAmount = Iga2CrtcRegs.HSyncEd.RegNum;
                    reg = Iga2CrtcRegs.HSyncEd.Reg;
                    break;
                case VIA_V_TOTAL_INDEX:
                    regValue = VIA_IGA2_VER_TOTAL_FORMULA(crtcTiming.VerTotal);
                    regAmount = Iga2CrtcRegs.VTotal.RegNum;
                    reg = Iga2CrtcRegs.VTotal.Reg;
                    break;
                case VIA_V_ADDR_INDEX:
                    regValue = VIA_IGA2_VER_ADDR_FORMULA(crtcTiming.VerAddr);
                    regAmount = Iga2CrtcRegs.VAddr.RegNum;
                    reg = Iga2CrtcRegs.VAddr.Reg;
                    break;
                case VIA_V_BLANK_START_INDEX:
                    regValue = VIA_IGA2_VER_BLANK_START_FORMULA(crtcTiming.VerBlankStart);
                    regAmount = Iga2CrtcRegs.VBlankSt.RegNum;
                    reg = Iga2CrtcRegs.VBlankSt.Reg;
                    break;
                case VIA_V_BLANK_END_INDEX:
                    regValue = VIA_IGA2_VER_BLANK_END_FORMULA(crtcTiming.VerBlankEnd);
                    regAmount = Iga2CrtcRegs.VBlankEd.RegNum;
                    reg = Iga2CrtcRegs.VBlankEd.Reg;
                    break;
                case VIA_V_SYNC_START_INDEX:
                    regValue = VIA_IGA2_VER_SYNC_START_FORMULA(crtcTiming.VerSyncStart);
                    regAmount = Iga2CrtcRegs.VSyncSt.RegNum;
                    reg = Iga2CrtcRegs.VSyncSt.Reg;
                    break;
                case VIA_V_SYNC_END_INDEX:
                    regValue = VIA_IGA2_VER_SYNC_END_FORMULA(crtcTiming.VerSyncEnd);
                    regAmount = Iga2CrtcRegs.VSyncEd.RegNum;
                    reg = Iga2CrtcRegs.VSyncEd.Reg;
                   break;
            }
        }
        viaLoadRegs(regValue, regAmount, reg);
    }

    xf86DrvMsg(1, X_INFO,
       "IGA%d timing\n"
       "HActive=%d, HSyncStart=%d, HSyncEnd=%d, HTotal=%d \n"
       "VActive=%d, VSyncStart=%d, VSyncEnd=%d, VTotal=%d\n",           
       igaPath,
       crtcTiming.HorAddr, crtcTiming.HorSyncStart, crtcTiming.HorSyncEnd, crtcTiming.HorTotal,
       crtcTiming.VerAddr, crtcTiming.VerSyncStart, crtcTiming.VerSyncEnd, crtcTiming.VerTotal
       );

}

/* Load offset registers*/
static void 
viaLoadOffsetRegs(VIAPtr pVia, unsigned long offset, int igaPath)
{
    unsigned long 	regValue = offset;
    unsigned char 	regAmount = 0;
    ViaIoBitPtr		reg = NULL;

    /* Check input variable igaPath*/
    if ((igaPath != IGA1) && (igaPath != IGA2))
        return;

    if (igaPath == IGA1) {
        regAmount = ViaOffsetRegs.Iga1.RegNum;
        reg = ViaOffsetRegs.Iga1.Reg;
    } else {
        switch (pVia->Chipset) {
        case VIA_VX800:
        case VIA_VX855:
            regAmount = ViaIga2Offset13BitRegs.RegNum;
            reg = ViaIga2Offset13BitRegs.Reg;
            break;
        default:
            regAmount = ViaOffsetRegs.Iga2.RegNum;
            reg = ViaOffsetRegs.Iga2.Reg;
            break;
        }
    }
    viaLoadRegs(regValue, regAmount, reg);
}

/* Load fetch count registers*/
static void 
viaLoadFetchCountRegs(unsigned long fetchCount, int igaPath)
{
    unsigned long	regValue = fetchCount;
    unsigned long	regAmount = 0;
    ViaIoBitPtr		reg = NULL;

    /* Check input variable igaPath*/
    if ((igaPath != IGA1) && (igaPath != IGA2))
        return;

    if (igaPath == IGA1) {
        regAmount = ViaFetchCountRegs.Iga1.RegNum;
        reg = ViaFetchCountRegs.Iga1.Reg;
    } else {
        regAmount = ViaFetchCountRegs.Iga2.RegNum;
        reg = ViaFetchCountRegs.Iga2.Reg;
    }
    viaLoadRegs(regValue, regAmount, reg);
}

/*Load fifo registers (depending on chip name),The function should be called when setting mode*/
void 
viaLoadFifoRegs(VIAPtr pVia, unsigned long modeH, unsigned long modeV, int igaPath)
{
    unsigned long	regValue;
    unsigned long	regAmount;
    ViaIoBitPtr 	regBit = NULL;
    FifoValue 		iga1Fifo;
    FifoValue 		iga2Fifo;
    unsigned long  	i = 0;

    viaMemSet(&iga1Fifo, sizeof(FifoValue));
    viaMemSet(&iga2Fifo, sizeof(FifoValue));

    for (i = 0; i < NUM_TOTAL_DISP_FIFO_TABLE; i++) {
        if (DispFifoTable[i].ChipName == pVia->Chipset) {
            iga1Fifo = DispFifoTable[i].Iga1Fifo;
            iga2Fifo = DispFifoTable[i].Iga2Fifo;
            break;
        }
    }

    /*Learn from linux driver*/
    /*For chip VT3204/3259/3314, if resolution > 1280x1024, expire length = 64*/
    switch (pVia->Chipset) {
        case VIA_P4M800PRO:
            if ((modeH > 1280) && (modeV > 1024)) {
                iga1Fifo.DispQueueExpiredNum = 16;
                iga2Fifo.DispQueueExpiredNum = 16;
            }
			break;
        default:
            break;
    }

    if (IGA1 == igaPath) {
        /*Set IGA1 Display FIFO Depath*/
        regValue = VIA_IGA1_FIFO_DEPTH_SELECT_FORMULA(iga1Fifo.MaxDepth);
        regAmount = ViaDisplayFifoDepthReg.Iga1FifoDepthSelectReg.RegNum;
        regBit = ViaDisplayFifoDepthReg.Iga1FifoDepthSelectReg.Reg;
        viaLoadRegs(regValue, regAmount, regBit);

        /*Set IGA1 Display FIFO Threshold Select*/
        regValue = VIA_IGA1_FIFO_THRESHOLD_FORMULA(iga1Fifo.Threshold);
        regAmount = ViaFifoThresholdSelectReg.Iga1FifoThresholdSelectReg.RegNum;
        regBit = ViaFifoThresholdSelectReg.Iga1FifoThresholdSelectReg.Reg;
        viaLoadRegs(regValue, regAmount, regBit);

        /*Set IGA1 GFX PREG Threshold Select*/
        regValue = VIA_IGA1_FIFO_HIGH_THRESHOLD_FORMULA(iga1Fifo.HighThreshold);
        regAmount = ViaFifoHighThresholdSelectReg.Iga1FifoHighThresoldSelectReg.RegNum;
        regBit = ViaFifoHighThresholdSelectReg.Iga1FifoHighThresoldSelectReg.Reg;
        viaLoadRegs(regValue, regAmount, regBit);

        /*Set IGA1 Display Gueue Expire Num*/
        regValue = VIA_IGA1_DISPLAY_QUEUE_EXPIRE_NUM_FORMULA(iga1Fifo.DispQueueExpiredNum);
        regAmount = ViaDisplayQueueExpireNumReg.Iga1DisplayQueueExpireNumReg.RegNum;
        regBit = ViaDisplayQueueExpireNumReg.Iga1DisplayQueueExpireNumReg.Reg;
        viaLoadRegs(regValue, regAmount, regBit);
    } else {
        /*Set IGA2 Display FIFO Depath Select*/
        regValue = VIA_IGA2_FIFO_DEPTH_SELECT_FORMULA(iga2Fifo.MaxDepth);
        regAmount = ViaDisplayFifoDepthReg.Iga2FifoDepthSelectReg.RegNum;
        regBit = ViaDisplayFifoDepthReg.Iga2FifoDepthSelectReg.Reg;
        viaLoadRegs(regValue, regAmount, regBit);

        /*Set IGA2 Display FIFO Threshold Select*/
        regValue = VIA_IGA2_FIFO_THRESHOLD_FORMULA(iga2Fifo.Threshold);
        regAmount = ViaFifoThresholdSelectReg.Iga2FifoThresholdSelectReg.RegNum;
        regBit = ViaFifoThresholdSelectReg.Iga2FifoThresholdSelectReg.Reg;
        viaLoadRegs(regValue, regAmount, regBit);

        /*Set IGA2 GFX PREG Threshold Select*/
        regValue = VIA_IGA2_FIFO_HIGH_THRESHOLD_FORMULA(iga2Fifo.HighThreshold);
        regAmount = ViaFifoHighThresholdSelectReg.Iga2FifoHighThresoldSelectReg.RegNum;
        regBit = ViaFifoHighThresholdSelectReg.Iga2FifoHighThresoldSelectReg.Reg;
        viaLoadRegs(regValue, regAmount, regBit);

        /*Set IGA2 Display Queue Expire Num*/
        regValue = VIA_IGA2_DISPLAY_QUEUE_EXPIRE_NUM_FORMULA(iga2Fifo.DispQueueExpiredNum);
        regAmount = ViaDisplayQueueExpireNumReg.Iga2DisplayQueueExpireNumReg.RegNum;
        regBit = ViaDisplayQueueExpireNumReg.Iga2DisplayQueueExpireNumReg.Reg;
        viaLoadRegs(regValue, regAmount, regBit);
    }    
}

/* Load color depth registers*/
static void 
viaLoadColorDepthRegs(unsigned char bpp, int igaPath)
{
    /* Check input variable a_IgaPath*/
    if ((igaPath != IGA1) && (igaPath != IGA2))
        return;

    if (igaPath == IGA1) {
        switch (bpp) {
            case BPP8:
                viaWriteVgaIoBits(REG_SR15, 0x22, 0xFE);
                break;
            case BPP16:
                viaWriteVgaIoBits(REG_SR15, 0xB6, 0xFE);
                break;
            case BPP32:
                viaWriteVgaIoBits(REG_SR15, 0xAE, 0xFE);
                break;
            default:
                break;
        }
    } else {
        switch (bpp) {
            case BPP8:
                viaWriteVgaIoBits(REG_CR67, 0x00, 0xC0);
                break;
            case BPP16:
                viaWriteVgaIoBits(REG_CR67, 0x40, 0xC0);
                break;
            case BPP32:
                viaWriteVgaIoBits(REG_CR67, 0xC0, 0xC0);
                break;
            default:
                break;
        }
    }
}

#define FILL_CLOCK_SETTING(pClockSetting, clock, m, n, r, refDivBW, phaseMargin) \
    (pClockSetting)->Clock = (clock); 			\
    (pClockSetting)->M = (m); 						\
    (pClockSetting)->N = (n); 						\
    (pClockSetting)->R = (r); 						\
    (pClockSetting)->RefDivBW = (refDivBW); 	\
    (pClockSetting)->PhaseMargin = (phaseMargin);
	
/*Compute IGA clock*/
static void 
viaInputClock(ClockSettingPtr pClockSetting, double fout, Bool isVT3314, unsigned int sortType)
{       
    double fin = VIA_FIN;
    int    constN = 1000;
    int    constD = 5;
    int    constP = 5;
    double constERR = 5000;

    double constFxo = fin * 1E6;     /* unit = MHz*/
    double constIp = 10 * 1E-6;          /* unit = uA */
    double constKpfd = constIp / (2*VIA_PI);  /* unit = uA/rad */
    double constKvco = 400 * 1E6;        /* unit = MHz*/
    double constC1 = 225 * 1E-12;        /* unit = pf*/
    double constC2 = 15 * 1E-12;         /* unit = pf*/
    double constR1 = 40 * 1E3;           /* unit = k-ohm*/
    double constZERO = 1 / (VIA_TWO_PI * constR1 * constC1);
    double constPOLE =(constC1 + constC2) / (VIA_TWO_PI * constR1 * constC1 * constC2);        

    int    n = 2, d = 2, r = 0;    
    double test = fout / fin;
    double fvco = 0;
    int	   maxN = constN;
    double bestValue = 0;
    
    if (isVT3314) {
        /* 3314 M[9:8] must 0, so maximum value of M is equal to 0xFF, and N = M+2*/
        maxN = (constN < 0xFF + 2) ? constN : (0xFF + 2);
    }

    while (n <= maxN) {
        while (d <= constD) {
            while (r <= constP) {
                fvco = fout * pow(2.0f, r);
                double fref = constFxo / d;
                double bw = ((constKpfd * constKvco) / n) * constR1 * ((constC1 / constC2) / (1 + (constC1 / constC2)));
                double phaseM1 = atan2(bw, constPOLE);
                double phaseM2 = atan2(bw, constZERO);
                double phaseMargin = (phaseM2 - phaseM1) * 180 / VIA_PI;
                double clock = (double)n / (d * pow(2.0f, r));

                if ((clock > (test - test * constERR / 1E6)) &&
                     (clock < (test + test * constERR / 1E6)) &&
                     ((fvco >= 300) && (fvco <= 600))) {  
                    switch (sortType) {
                        case VIA_SORT_BY_PHASE_MARGIN:
                            if (bestValue == 0)
                                bestValue = phaseMargin;

                            /* Choose the biggest value. */
                            if (phaseMargin > bestValue) {
                                FILL_CLOCK_SETTING(pClockSetting, clock*fin, n-2, d-2, r, fref/bw, phaseMargin);
                                bestValue = phaseMargin;
                            }                    
                        	break;
                        case VIA_SORT_BY_F_REF_BW:
                            if (bestValue == 0)
                                bestValue = (fref/bw);

                            /* Choose the biggest value. */
                            if ((fref/bw) > bestValue) {
                                FILL_CLOCK_SETTING(pClockSetting, clock*fin, n-2, d-2, r, fref/bw, phaseMargin);
                                bestValue = (fref / bw);
                            }                   
                        	break;
                        case VIA_SORT_BY_PIXEL_CLOCK:
                            if (bestValue == 0)
                                bestValue = clock*fin;

                            /* Choose the most close value. */
                            if (fabs(clock * fin - fout) < fabs(bestValue - fout)) {
                                FILL_CLOCK_SETTING(pClockSetting, clock*fin, n-2, d-2, r, fref/bw, phaseMargin);
                                bestValue = clock * fin;
                            }                    
                        	break;
                    }
                }
                r++;
            }
            r = 0;
            d++;
        }
        r = 0;
        d = 2;
        n++;
    }
}

static void 
viaNMR2Reg(int n, int m, int r, int* sr47, int* sr48, int* sr49)
{
    int dtz = BIT1 | BIT0;    /* b'11*/

    m &= 0x3FF;   /* 10 bits */
    n &= 0x3F;    /* 6 bits */
    r &= 0x7;     /* 3 bits */

    *sr47 = m & 0xFF;
    m >>= 8;
    *sr48 |= m;
    *sr49 = n;
    *sr48 |= (r << 2);
    *sr48 |= ((dtz & 1) << 7);
    dtz >>= 1;
    *sr49 |= ((dtz & 1) << 7);

    DEBUG(ErrorF("dwSR47 = 0x%x\n", *sr47));
    DEBUG(ErrorF("dwSR48 = 0x%x\n", *sr48));
    DEBUG(ErrorF("dwSR49 = 0x%x\n", *sr49));

}

static void
viaCalFromClock(int chipSet, double desiredClock, unsigned long* pllValue)
{    
    ClockSetting clockSetting;
    double clock  = desiredClock;
    int sr47 = 0, sr48 = 0, sr49 = 0;

    memset(&clockSetting, 0, sizeof(ClockSetting));

	/*Shall we care 3314?*/
    viaInputClock(&clockSetting, clock, FALSE, VIA_SORT_BY_PHASE_MARGIN);
    
    if (chipSet == VIA_VX855) {
    	/* 
    	For Vt3409, M and N dosn't need to add 2, and since clockSetting.N and clockSetting.M have been sub 2 in
		function InputClock(), here should add 2 to them. 
		*/
        viaNMR2Reg(clockSetting.N+2,clockSetting.M+2, clockSetting.R, &sr47, &sr48, &sr49);
    } else {
        viaNMR2Reg(clockSetting.N,clockSetting.M, clockSetting.R, &sr47, &sr48, &sr49);
    }        

    *pllValue = ((sr47 << 16) | (sr48 << 8) | sr49);
    return;
}

/*Set Iga clock*/
void 
viaSetVCLK(unsigned long pllValue, unsigned char igaPath)
{
    /* Check input variable a_IgaPath*/
    if ((igaPath != IGA1) && (igaPath != IGA2))
        return;

    if (igaPath == IGA1) {
        viaWriteVgaIo(REG_SR44, (unsigned char)((pllValue & 0x00FF0000)>>16));
        viaWriteVgaIo(REG_SR45, (unsigned char)((pllValue & 0x0000FF00)>>8));
        viaWriteVgaIo(REG_SR46, (unsigned char)((pllValue & 0x000000FF)));
    } else {
        viaWriteVgaIo(REG_SR4A, (unsigned char)((pllValue & 0x00FF0000)>>16));
        viaWriteVgaIo(REG_SR4B, (unsigned char)((pllValue & 0x0000FF00)>>8));
        viaWriteVgaIo(REG_SR4C, (unsigned char)((pllValue & 0x000000FF)));
    }

    /* Reset PLL */
    if (igaPath == IGA1) {
        viaWriteVgaIoBits(REG_SR40, 0x02, 0x02);    /*SR40[1] = 1*/
        viaWriteVgaIoBits(REG_SR40, 0x00, 0x02);    /*SR40[1] = 0*/
    } else {
        viaWriteVgaIoBits(REG_SR40, 0x04, 0x04);    /*SR40[2] = 1*/
        viaWriteVgaIoBits(REG_SR40, 0x00, 0x04);    /*SR40[2] = 0*/
    }

    /* Fire */
    viaWriteMiscIo(viaReadMiscIo() | (BIT2+BIT3));
}

/*Patch for iga1 clock*/
static void
viaPlusVCKToIGA1Timing(unsigned short hTotal)
{
	switch (hTotal % 8) {
		case 0:
		default:
			viaWriteVgaIoBits(REG_CR47, 0x00, BIT7+BIT6+BIT3);
			break;
		case 2:
			viaWriteVgaIoBits(REG_CR47, BIT7, BIT7+BIT6+BIT3);
			break;
		case 4:
			viaWriteVgaIoBits(REG_CR47, BIT6, BIT7+BIT6+BIT3);
			break;
		case 6:
			viaWriteVgaIoBits(REG_CR47, BIT3, BIT7+BIT6+BIT3);
			break;
	}
}

/* Get the pointer of crtc structure matched with the iga number*/
xf86CrtcPtr
viaGetCrtcFromIga(ScrnInfoPtr pScrn, int iga)
{
    xf86CrtcConfigPtr   xf86Config = XF86_CRTC_CONFIG_PTR(pScrn);
	int n;
	
	for (n = 0; n < xf86Config->num_crtc; n++) {
		VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(xf86Config->crtc[n]);
		if (viaCrtc->iga == iga)
			return xf86Config->crtc[n];
	}
	return NULL;
}

/*Set timing for iga, it's a very important function!*/
Bool
viaSetIgaTiming(ScrnInfoPtr pScrn, DisplayModePtr mode, DisplayTiming* pTiming, unsigned int clk, unsigned int igaPath)
{
    VIAPtr			pVia = VIAPTR(pScrn); 
	unsigned long	offset = 0;
	unsigned long	fetchCount =0;
	unsigned long	pllValue = 0;
	xf86CrtcPtr     crtc = NULL; 

	/*1. Sanity check*/
	if(pTiming == NULL || (igaPath != IGA1 && igaPath != IGA2))
		return FALSE;

	/*2.vpit register is just for IGA1, but for more stable, we set it also when single IGA2,it will done in .commit*/
	if (igaPath == IGA1)
		viaLoadVpitRegs();
	
	/*3. Unlock*/
	viaUnlockCrtc();
	
	/*4. IGA1 reset*/
	if (igaPath == IGA1) {
		/* initial CR09=0 */
		viaWriteVgaIo(REG_CR09, 0x00);  
		viaWriteVgaIoBits(REG_CR11, 0x00, BIT4 | BIT5 | BIT6);
		viaWriteVgaIoBits(REG_CR17, 0x00, BIT7);
	}

	/*5. Load crtc timing*/
	viaLoadCrtcTiming(*pTiming, igaPath);
	if (igaPath == IGA1)
		viaPlusVCKToIGA1Timing(pTiming->HorTotal);
	
	/*6. Lock*/
	viaLockCrtc();
	
	/*7. HW reset*/
	viaWriteVgaIoBits(REG_CR17, 0x80, BIT7); 
	
	/*8. Load offset register*/
	crtc = viaGetCrtcFromIga(pScrn, igaPath);
	if (!crtc)
		return FALSE;

	if (crtc->rotatedData) {	
        offset = CMDISP_ALIGN_TO(mode->HDisplay * pScrn->bitsPerPixel / 8, 32) / 8;	
    } else {
	    offset = CMDISP_ALIGN_TO(pScrn->displayWidth * pScrn->bitsPerPixel / 8, 16) / 8;
    }

	viaLoadOffsetRegs(pVia,offset, igaPath);
	
	/*9. Load fetch count register*/
	fetchCount = CMDISP_ALIGN_TO(mode->HDisplay * pScrn->bitsPerPixel / 8, 16) / 16;
	fetchCount += 1;
	viaLoadFetchCountRegs(fetchCount, igaPath);
	
	/*10. Load FIFO*/
	viaLoadFifoRegs(pVia,mode->HDisplay, mode->VDisplay, igaPath);
	
	/*11. Load color depth*/
	viaLoadColorDepthRegs(pScrn->bitsPerPixel, igaPath);
	
	/*12. Set clock*/
	if (clk) {
		viaCalFromClock(pVia->Chipset, (double)clk / 1000000, &pllValue);
		viaSetVCLK(pllValue, igaPath);
	}
	
	return TRUE;
}

/* Get the timing information which will be flushed into HW from the given mode*/
static Bool 
GetTimingInfoFromMode(DisplayModePtr adjusted_mode, ViaTimingTablePtr pTiming)
{
 	if (adjusted_mode == NULL)
 		return FALSE;
	
	pTiming->Timing.HorTotal = adjusted_mode->CrtcHTotal;
	pTiming->Timing.HorAddr = adjusted_mode->CrtcHDisplay;
	pTiming->Timing.HorBlankStart = adjusted_mode->CrtcHBlankStart;
	pTiming->Timing.HorBlankEnd = adjusted_mode->CrtcHBlankEnd;
	pTiming->Timing.HorSyncStart = adjusted_mode->CrtcHSyncStart;
	pTiming->Timing.HorSyncEnd =  adjusted_mode->CrtcHSyncEnd;
	pTiming->Timing.VerTotal = adjusted_mode->CrtcVTotal;
	pTiming->Timing.VerAddr = adjusted_mode->CrtcVDisplay;
	pTiming->Timing.VerBlankStart = adjusted_mode->CrtcVBlankStart;
	pTiming->Timing.VerBlankEnd = adjusted_mode->CrtcVBlankEnd;
	pTiming->Timing.VerSyncStart = adjusted_mode->CrtcVSyncStart;
	pTiming->Timing.VerSyncEnd = adjusted_mode->CrtcVSyncEnd;

	pTiming->Clk = adjusted_mode->Clock * 1000;
	
	if (adjusted_mode->Flags & V_PHSYNC)
		pTiming->HSyncPolarity = POSITIVE;

	if (adjusted_mode->Flags & V_NHSYNC)
		pTiming->HSyncPolarity = NEGATIVE;

	if (adjusted_mode->Flags & V_PVSYNC)
		pTiming->VSyncPolarity = POSITIVE;

	if (adjusted_mode->Flags & V_NVSYNC)
		pTiming->VSyncPolarity = NEGATIVE;
	
	pTiming->RefreshRate = adjusted_mode->VRefresh;

	DEBUG(ErrorF("pTiming->Timing.HorTotal = %d\n",pTiming->Timing.HorTotal ));
	DEBUG(ErrorF("pTiming->Timing.HorAddr  = %d\n",pTiming->Timing.HorAddr));
	DEBUG(ErrorF("pTiming->Timing.HorBlankStart = %d\n",pTiming->Timing.HorBlankStart));
	DEBUG(ErrorF("pTiming->Timing.HorBlankEnd = %d\n",pTiming->Timing.HorBlankEnd));
	DEBUG(ErrorF("pTiming->Timing.HorSyncStart = %d\n",pTiming->Timing.HorSyncStart));
	DEBUG(ErrorF("pTiming->Timing.HorSyncEnd = %d\n",pTiming->Timing.HorSyncEnd));
	DEBUG(ErrorF("pTiming->Timing.VerTotal = %d\n",pTiming->Timing.VerTotal));
	DEBUG(ErrorF("pTiming->Timing.VerAddr = %d\n",pTiming->Timing.VerAddr));
	DEBUG(ErrorF("pTiming->Timing.VerBlankStart = %d\n",pTiming->Timing.VerBlankStart));
	DEBUG(ErrorF("pTiming->Timing.VerBlankEnd = %d\n",pTiming->Timing.VerBlankEnd));
	DEBUG(ErrorF("pTiming->Timing.VerSyncStart = %d\n",pTiming->Timing.VerSyncStart));
	DEBUG(ErrorF("pTiming->Timing.VerSyncEnd = %d\n",pTiming->Timing.VerSyncEnd));
	DEBUG(ErrorF("pTiming.Clk = %ld\n",pTiming->Clk));
	DEBUG(ErrorF("pTiming.HSyncPolarity = %ld\n",pTiming->HSyncPolarity));
	DEBUG(ErrorF("pTiming.VSyncPolarity = %ld\n",pTiming->VSyncPolarity));
	DEBUG(ErrorF("pTiming.RefreshRate= %ld\n",pTiming->RefreshRate));
	
	return TRUE;
}

/* Set iga start address*/
void
viaSetIgaStartAddr(xf86CrtcPtr crtc, int x, int y)
{
    ScrnInfoPtr pScrn = crtc->scrn;
	VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);

	unsigned int startAddr = 0;

	if (crtc->rotatedData) {
        startAddr = viaCrtc->rotateMem.base;
	} else { /*No rotate and reflect*/
		startAddr = (pScrn->displayWidth * y + x) * (pScrn->bitsPerPixel / 8);
	}

	if (viaCrtc->iga == IGA1) {
        startAddr = startAddr >> 1;
        viaWriteVgaIo(REG_CR0D, (unsigned char)(startAddr & 0x000000FF));
        viaWriteVgaIo(REG_CR0C, (unsigned char)((startAddr & 0x0000FF00) >> 8));
        viaWriteVgaIoBits(REG_CR48, (unsigned char)((startAddr & 0xFF000000) >> 24) & 0x1f, 0x1f);
		/*Note: For 353,the register 3X5.48, 3x5.0C, 3x5.0D, 3x5.EC must ready before 3x5.34*/
		viaWriteVgaIo(REG_CR34, (unsigned char)((startAddr & 0x00FF0000) >> 16));
	} else {
        startAddr = startAddr >> 3;
        viaWriteVgaIoBits(REG_CR62, (unsigned char)((startAddr & 0x0000007F) << 1), 0xFE);
        viaWriteVgaIo(REG_CR63, (unsigned char)((startAddr & 0x00007F80) >> 7));
        viaWriteVgaIo(REG_CR64, (unsigned char)((startAddr & 0x007F8000) >> 15));
        viaWriteVgaIo(REG_CRA3, (unsigned char)((startAddr & 0x03800000) >> 23));
	}
	
	return;
}

static void
viaEnableSecDispChannel(void)
{
    /* to enable second display channel. */
    viaWriteVgaIoBits(REG_CR6A, 0x00, BIT6);
    viaWriteVgaIoBits(REG_CR6A, BIT7, BIT7);
    viaWriteVgaIoBits(REG_CR6A, BIT6, BIT6);
}

static void 
viaDisableSecDispChannel(void)
{
    /* to disable second display channel. */
    viaWriteVgaIoBits(REG_CR6A, 0x00, BIT6);
    viaWriteVgaIoBits(REG_CR6A, 0x00, BIT7);
    viaWriteVgaIoBits(REG_CR6A, BIT6, BIT6);
}

static void 
viaSetLUT(int whichIGA, LOCO* colors)
{
    unsigned long i;
    unsigned char  sr1a, sr1b, cr67;

    sr1a = viaReadVgaIo(REG_SR1A);
    sr1b = viaReadVgaIo(REG_SR1B);
    cr67 = viaReadVgaIo(REG_CR67);     

    if (whichIGA == IGA2) {
        /* Prepare for IGA2's LUT initialization: */
		/* 1. Change Shadow to Secondary Display's LUT. */
        viaWriteVgaIo(REG_SR1A, sr1a | 0x01);
		/* 2. Enable Secondary Display Engine. */
        viaWriteVgaIo(REG_SR1B, sr1b | 0x80);
		/* 3. Second Display Color Depth should be 8bpp. */
        viaWriteVgaIo(REG_CR67, cr67 & 0x3F);

        if (!(viaReadVgaIo(REG_CR6A) & BIT7)) { 
           viaEnableSecDispChannel();
        }

        /* Fill in IGA2's LUT. */
        for (i = 0; i < 256; i++) {
            STANDVGA_W8(REG_PEL, i);
            STANDVGA_W8(REG_PEL+1, colors[i].red);
            STANDVGA_W8(REG_PEL+1, colors[i].green);
            STANDVGA_W8(REG_PEL+1, colors[i].blue);
        }
        /*  Disable gamma in case it was enabled previously */
        viaWriteVgaIoBits(REG_CR6A, 0x0, BIT1);
    }

    if (whichIGA == IGA1) {
        /* Prepare for initialize IGA1's LUT: */
        viaWriteVgaIo(REG_SR1A, sr1a & 0xFE);   /* Change to Primary Display's LUT. */
        viaWriteVgaIo(REG_SR1B, sr1b);
        viaWriteVgaIo(REG_CR67, cr67);

        /* Fill in IGA1's LUT. */
        for (i = 0; i < 256; i++) {
            STANDVGA_W8(REG_PEL, i);
            STANDVGA_W8(REG_PEL + 1, colors[i].red);
            STANDVGA_W8(REG_PEL + 1, colors[i].green);
            STANDVGA_W8(REG_PEL + 1, colors[i].blue);
        }
        /* enable LUT */
        viaWriteVgaIoBits(REG_SR1B, 0x0, BIT0);
        /*  Disable gamma in case it was enabled previously */
        viaWriteVgaIoBits(REG_CR33, 0x0, BIT7);
    }
    viaWriteVgaIo(REG_SR1A, sr1a & 0xFE);   /* access Primary Display's LUT. */

	return ;
}

void 
viaSetGamma(int whichIGA, LOCO* colors)
{
    int  i, sr1a;

    sr1a = viaReadVgaIo(REG_SR1A);

    if (whichIGA == IGA1) {
        /* Enable Gamma */
        viaWriteVgaIoBits(REG_CR33, 0x80, BIT7);
        viaWriteVgaIoBits(REG_SR1A, 0x0, BIT0);
            
        /* Fill in IGA1's gamma. */
        for (i = 0; i < 256; i++) {
            STANDVGA_W8(REG_PEL, i);
            STANDVGA_W8(REG_PEL + 1, colors[i].red);
            STANDVGA_W8(REG_PEL + 1, colors[i].green);
            STANDVGA_W8(REG_PEL + 1, colors[i].blue);
        }
    }

    if (whichIGA == IGA2) {
        viaWriteVgaIoBits(REG_SR1A, 0x01, BIT0);
        /* Enable Gamma */
        viaWriteVgaIoBits(REG_CR6A, 0x02, BIT1);

        /* Before we fill the second LUT, we have to enable second display channel.
        If it's enabled before, we don't need to do that, or else the secondary
        display will be dark for about 1 sec and then be turned on again.*/
        if (!(viaReadVgaIo(REG_CR6A) & BIT7))
            viaEnableSecDispChannel();

        /* Fill in IGA2's gamma. */
        for (i = 0; i < 256; i++) {
            STANDVGA_W8(REG_PEL, i);
            STANDVGA_W8(REG_PEL + 1, colors[i].red);
            STANDVGA_W8(REG_PEL + 1, colors[i].green);
            STANDVGA_W8(REG_PEL + 1, colors[i].blue);
        }
    }
    viaWriteVgaIo(REG_SR1A, sr1a);
	
    return;
}

static Bool 
viaSetGammaOrLUT(xf86CrtcPtr crtc)
{
	ScrnInfoPtr  pScrn = crtc->scrn;
	VIAPtr  pVia = VIAPTR(pScrn); 
	VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);
	
	/* Get the affected IGA path first, we only need to set the affected iga path */       
	if (pScrn->bitsPerPixel == 8)
		viaSetLUT(viaCrtc->iga, viaCrtc->colors);
    /* Note:
       VT3409 has a hardware issue, when we enable IGA1 gamma, the display color
       will become bad if playing video.
       Solution: disable IGA1 gamma on VT3409 (16/32bpp)
    */
	else if ((viaCrtc->iga != IGA1) || (pVia->Chipset != VIA_VX855))
		viaSetGamma(viaCrtc->iga, viaCrtc->colors);

	return TRUE;
}

/* For the case only iga2 is used, simulate a 800x600 timing for iga1 to
make sure there must be timing on iga1!*/
static Bool 
viaIga1RequirePatchClk(ScrnInfoPtr pScrn)
{
	xf86CrtcPtr crtc = NULL;
    VIACrtcPrivatePtr viaCrtc = NULL; 
	ViaTimingTable	viaTiming;

	crtc = viaGetCrtcFromIga(pScrn, IGA1);
	if (!crtc)
		return FALSE;
	viaCrtc = VIACrtcPrivate(crtc);
	if (viaCrtc->enabled)
		return FALSE;

	/*Patch a default mode(800x600) on IGA1*/
	static DisplayModeRec	mode = {
		NULL, NULL, "800x600", MODE_OK, M_T_DEFAULT,
		40000,
		800, 840, 968, 1056, 0,
		600, 601, 605, 628, 0,
		V_PHSYNC | V_PVSYNC,
		0, 0,
		0, 0, 0, 0, 0, 0, 0,
		0, 0, 0, 0, 0, 0,
		FALSE, FALSE, 0, NULL, 0, 0.0, 0.0
	};
	
	xf86SetModeCrtc (&mode, 0);
	
	if (!GetTimingInfoFromMode(&mode, &viaTiming)) {
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "viaIga1RequirePatchClk has no mode to set.\n");
		return FALSE;
	};
	
	viaSetIgaTiming(pScrn, &mode, &(viaTiming.Timing), viaTiming.Clk, viaCrtc->iga);
	return TRUE;
}

/* Disable Iga2 scaling up function*/
static void
disableIga2ScalingUp(void)
{
	/*Horizontal scaling disable*/
	viaWriteVgaIoBits(REG_CRA2, 0x00, BIT7);
	/*Vertical scaling disable*/
	viaWriteVgaIoBits(REG_CRA2, 0x00, BIT3);
	/*Disable IGA2 scaling*/
	viaWriteVgaIoBits(REG_CR79, 0x00, 0x07);
}

/*Compute the cursor position*/
static void
get_crtc_cursor_real_position(xf86CrtcPtr crtc, int x, int y, int *xpos, int *ypos, 
        unsigned char *xoff, unsigned char *yoff)
{
    /*compute x*/
    if (x < 0) {
        *xpos = 0;
        *xoff = (-x) & 0xFF;
    } else {
        *xpos = x;
        *xoff = 0;
    }

    /*compute y*/
    if (y < 0) {
        *ypos = 0;
        *yoff = (-y) & 0xFF;
    } else {
        *ypos = y;
        *yoff = 0;
    }
}

/* Video playback needs to know the relationship between outputs(single, extend or overlap)*/
static int 
probe_extend_status(xf86CrtcPtr crtc, DisplayModePtr mode)
{
    ScrnInfoPtr pScrn = crtc->scrn;
    int i = 0, j = 0;
    int crtcWidth[2];
    int crtcHeight[2];
    int posX[2];
    int posY[2];
    xf86CrtcPtr crtcPtr[2];
    VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);
    xf86CrtcConfigPtr xf86_config = XF86_CRTC_CONFIG_PTR(pScrn);
    Rotation  rotateMask = RR_Rotate_0 | RR_Rotate_90 | RR_Rotate_270 | RR_Rotate_180;
    
    for (i = 0; i < xf86_config->num_output; i++) {
        if (xf86_config->output[i]->crtc) {
            posX[j] = xf86_config->output[i]->crtc->x;
            posY[j] = xf86_config->output[i]->crtc->y;
            crtcPtr[j] = xf86_config->output[i]->crtc;
            /*this is important,because when we dynamicly change the mode, the desiredMode.HDisplay
            is not the mode now, but the mode we successfully last time,so if we do not use the mode
            parameter the position will not right*/
            if (xf86_config->output[i]->crtc == crtc) {
                crtcWidth[j] = mode->HDisplay;
                crtcHeight[j] = mode->VDisplay;
            } else {
                crtcWidth[j] = xf86_config->output[i]->crtc->desiredMode.HDisplay;
                crtcHeight[j] = xf86_config->output[i]->crtc->desiredMode.VDisplay;
            }
            j++;
        }
    }

    /* Only one output*/
    if (j == 1)
        return RELATION_NONE;

    viaCrtc = VIACrtcPrivate(crtcPtr[1]);

    switch(crtcPtr[0]->rotation & rotateMask) {
        case RR_Rotate_90:
        case RR_Rotate_270:
            EXCHANGE(crtcWidth[0], crtcHeight[0]);
            break;
    }

    switch(crtcPtr[1]->rotation & rotateMask) {
        case RR_Rotate_90:
        case RR_Rotate_270:
            EXCHANGE(crtcWidth[1], crtcHeight[1]);
            break;
    }
    
    if ((posX[0] == posX[1]) && (posY[1] == posY[0]))
        return RELATION_SAME_OF;

    if (posX[0]>posX[1]) {
        if (viaCrtc->iga == IGA1) {
            return RELATION_RIGHT_OF;
        } else {
            return RELATION_LEFT_OF;
        }    
    } else if (posX[0]<posX[1]) {
        if (viaCrtc->iga == IGA1) {
            return RELATION_LEFT_OF;
        } else {
            return RELATION_RIGHT_OF;
        }
    } else if (posY[0]>posY[1]) {
        if (viaCrtc->iga == IGA1) {
            return RELATION_BELOW_OF;
        } else {
            return RELATION_ABOVE_OF;
        }    
    } else if (posY[0]<posY[1]) {
        if (viaCrtc->iga == IGA1) {
            return RELATION_ABOVE_OF;
        } else {
            return RELATION_BELOW_OF;
        }
    }
    
    return RELATION_OVERLAP_OF;
}

/**
 * Sets the power management mode of the pipe and plane.
 * We just deal with cursor and iga, maybe video also should be done.
 */
static void
via_crtc_dpms(xf86CrtcPtr crtc, int mode)
{
	DEBUG(ErrorF("via_crtc_dpms, mode = %d\n", mode));

	ScrnInfoPtr pScrn = crtc->scrn;
	VIAPtr pVia = VIAPTR(pScrn); 
	VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);    
    PVIDDATA pVidData = pVia->pVidData;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;
	
	switch (mode) {
		case DPMSModeOn:		/* 0 */
			/*1. Enable IGA input data*/
			if (viaCrtc->iga == IGA1) {
				viaWriteVgaIoBits(REG_SR01, 0x00, BIT5);
			} else {
				viaEnableSecDispChannel();
				if ((VIA_CX700 == pVia->Chipset) ||
					(VIA_VX800 == pVia->Chipset) ||
					(VIA_VX855 == pVia->Chipset)) {
					viaWriteVgaIoBits(REG_CR6B, 0x00, BIT2);
				}
			}
			
			/*2. Show cursor*/
			if (!pVia->ForceSWCursor)
				crtc->funcs->show_cursor(crtc);

            /*3. Update video information*/	
            viaGfxInfo->igaInfo[viaCrtc->iga-1].igaStatus.unit = viaCrtc->igastatus;
    	
	        if (via_module_loaded) {
                TranslateGFXInfo(viaGfxInfo, &NEWVIAGraphicInfo);
		        vvaSyncInfo(&NEWVIAGraphicInfo, VIA_SET_2D_INFO);
	        }
	
	        VIARandR12UpdataOverlay(pScrn->scrnIndex, crtc->x, crtc->y, 0);
			break;
		case DPMSModeStandby:	/* 1 */
		case DPMSModeSuspend:	/* 2 */
		case DPMSModeOff:		/* 3 */
			/*1. Hide cursor*/
			if (!pVia->ForceSWCursor)
				crtc->funcs->hide_cursor(crtc);
			
			/*2. Disable IGA input data.
				 Disable scaling down/up function
			*/
			if (viaCrtc->iga == IGA1) {
				/* a. Disable IGA1 input data*/
				viaWriteVgaIoBits(REG_SR01, 0x20, BIT5);
				/* b. Disable down scaling*/
				if ((VIA_VX800 == pVia->Chipset) ||
				    (VIA_VX855 == pVia->Chipset)) {
					/* Disable auto flip. */
					viaWriteVgaIoBits(REG_CRDD, 0x00, BIT7);
					/* IGA1 Down Scale Disable */	 
					viaWriteVgaIoBits(REG_CREC, 0x00, BIT0);
				}
			} else {
				/* a. Disable IGA2 input data for 3324/3353*/
				if ((VIA_CX700 == pVia->Chipset) ||
					(VIA_VX800 == pVia->Chipset) ||
					(VIA_VX855 == pVia->Chipset)) {
					viaWriteVgaIoBits(REG_CR6B, 0x04, BIT2);
				}
				/*b. Disable IGA2 display channel*/	 
				viaDisableSecDispChannel();
				/*c. Disable down scaling*/
				if ((VIA_VX800 == pVia->Chipset) ||
				    (VIA_VX855 == pVia->Chipset)) {
					/* Disable auto flip. */
					viaWriteVgaIoBits(REG_CRDD, 0x00, BIT7);
					/* IGA2 Down Scale Disable */	 
					viaWriteVgaIoBits(REG_CRE8, 0x00, BIT4);
				}
				/*d. Disable scaling up function*/
				disableIga2ScalingUp();
			}
            /*3. clear CR6C every time for tv clock
              Fixme:here we don't clear 314,364,
              because 314 iga1 iga2 use the same reg bit to control clock source, 
              so it is not necessary to clear it, but it may be have some bugs*/
            if ((VIA_CX700 == pVia->Chipset) ||
				(VIA_VX800 == pVia->Chipset) ||
			    (VIA_VX855 == pVia->Chipset)) {
			    if(viaCrtc->iga == IGA1) {
                    viaWriteVgaIoBits(REG_CR6C, 0x00, 0xF0);
                } else {
                    viaWriteVgaIoBits(REG_CR6C, 0x00, 0x0F);
                }
            }
            
            /*4. update video info && ddmpeg info*/
            if (viaGfxInfo->igaInfo[viaCrtc->iga-1].actived) {
                viaGfxInfo->igaInfo[viaCrtc->iga-1].actived = 0;
                viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse -= viaCrtc->iga;
                if (viaGfxInfo->screenAttr.clone || viaGfxInfo->screenAttr.extend) {
                    viaGfxInfo->screenAttr.uint = VIA_SINGLEVIEW;
                } else {
                    viaGfxInfo->screenAttr.uint = 0;
                }

                /*updata ddmpeg info*/
                if (via_module_loaded) {
                    TranslateGFXInfo(viaGfxInfo, &NEWVIAGraphicInfo);
		            vvaSyncInfo(&NEWVIAGraphicInfo, VIA_SET_2D_INFO);
	            }
                
                VIARandR12UpdataOverlay(pScrn->scrnIndex, crtc->x, crtc->y, 0);
            }
			break;
	}
}

static Bool
via_crtc_lock (xf86CrtcPtr crtc)
{   
	ScrnInfoPtr pScrn = crtc->scrn;
	VIAPtr      pVia = VIAPTR(pScrn); 

	/* Wait Hardware Engine idle to switch graphicd mode */
	WaitIdle();

	/*Returns whether unlock is needed,if return FALSE,do not need unlock.*/ 
	return FALSE;
}

static void
via_crtc_unlock (xf86CrtcPtr crtc)
{
	/*Unlock CRTC after mode setting, mostly for DRI*/	
	/*Note: But now our xrandr do not support DRI yet,so we do nothing here*/
}

static void
via_crtc_prepare (xf86CrtcPtr crtc)
{
    DEBUG(ErrorF("via_crtc_prepare\n"));
    ScrnInfoPtr pScrn = crtc->scrn;
    VIAPtr pVia = VIAPTR(pScrn);
    VIACrtcPrivatePtr   viaCrtc = VIACrtcPrivate(crtc);     

	/*1. DPMS off*/
	crtc->funcs->dpms(crtc, DPMSModeOff);
	
	/*2. Disable VQ, should do it?*/
	if (pVia->VQEnable) { 
		/* if we use VQ, disable it before we exit */
		switch (pVia->Chipset) {
			case VIA_P4M890:			 
			case VIA_K8M890:
				VIASETREG(0x41c, 0x00100000);
				VIASETREG(0x420, 0x74301000);
				break;
			default:  
				VIASETREG(0x43c, 0x00fe0000);
				VIASETREG(0x440, 0x00000004);
				VIASETREG(0x440, 0x40008c0f);
				VIASETREG(0x440, 0x44000000);
				VIASETREG(0x440, 0x45080c04);
				VIASETREG(0x440, 0x46800408);
				break;
		}
	}	

	/*3. Update crtc private data structure*/
	viaCrtc->enabled = FALSE;
}


/* Video playback needs to know how many devices are enabled*/
static ActiveDeviceRec
getActiveDevice(char *pname)
{
    ActiveDeviceRec activeDevice;

    activeDevice.unit = 0;
	
    if (!xf86NameCmp(pname, "CRT"))
        activeDevice.crt = TRUE;

	if (!xf86NameCmp(pname, "LCD"))
		activeDevice.lcd = TRUE;

	if (!xf86NameCmp(pname, "LCD-2"))
		activeDevice.lcd2 = TRUE;

	if (!xf86NameCmp(pname, "DVI"))
		activeDevice.dvi = TRUE;

	if (!xf86NameCmp(pname, "DVI-2"))
		activeDevice.dvi2 = TRUE;

	return activeDevice;
}


/**
 * Sets up registers for the given mode/adjusted_mode pair.
 *
 * The clocks, CRTCs and outputs attached to this CRTC must be off.
 *
 * This shouldn't enable any clocks, CRTCs, or outputs, but they should
 * be easily turned on/off after this.
 */
static void
via_crtc_mode_set(xf86CrtcPtr crtc, DisplayModePtr mode,
                                        DisplayModePtr adjusted_mode,
                                        int x, int y)
{    
    ScrnInfoPtr pScrn = crtc->scrn;
    VIAPtr pVia = VIAPTR(pScrn);
    ViaTimingTable  viaTiming;
    VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);
    int i;
    xf86CrtcConfigPtr xf86_config = XF86_CRTC_CONFIG_PTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;
    Rotation  rotateMask = RR_Rotate_0 | RR_Rotate_90 | RR_Rotate_270 | RR_Rotate_180;
    Rotation  reflectMask = RR_Reflect_X | RR_Reflect_Y;
    ActiveDeviceRec activeDevice;

    /*1. Get Timing from adjusted_mode*/
    if (!GetTimingInfoFromMode(adjusted_mode, &viaTiming)) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "via_crtc_mode_set has no mode to set.\n");
        return;
    };

    /*2. Set IGA timing*/
    viaSetIgaTiming(pScrn, mode, &(viaTiming.Timing), viaTiming.Clk, viaCrtc->iga);

    /*3. Set IGA start address*/
    viaSetIgaStartAddr(crtc, x, y);

    /*4. Crtc private data structure update*/

    /*5. Update video information*/	
    memset(&(viaGfxInfo->screenAttr), 0x0, sizeof(viaGfxInfo->screenAttr));	
    memset(&(viaGfxInfo->igaAttr), 0x0, sizeof(viaGfxInfo->igaAttr));
    switch (probe_extend_status(crtc,mode)) {
        case RELATION_RIGHT_OF:
            viaGfxInfo->screenAttr.uint = VIA_EXTEND;
            viaGfxInfo->igaAttr.iga2_right = TRUE;
            break;
        case RELATION_LEFT_OF:
            viaGfxInfo->screenAttr.uint = VIA_EXTEND;
            viaGfxInfo->igaAttr.iga2_left = TRUE;
            break;
        case RELATION_ABOVE_OF:
            viaGfxInfo->screenAttr.uint = VIA_EXTEND;
            viaGfxInfo->igaAttr.iga2_above = TRUE;
            break;
        case RELATION_BELOW_OF:
            viaGfxInfo->screenAttr.uint = VIA_EXTEND;
            viaGfxInfo->igaAttr.iga2_below = TRUE;
            break;
        case RELATION_NONE:
            viaGfxInfo->screenAttr.uint = VIA_SINGLEVIEW;
            break;
        case RELATION_OVERLAP_OF:
            viaGfxInfo->screenAttr.uint = VIA_EXTEND;
            viaGfxInfo->igaAttr.overlapped = TRUE;
            break;
        case RELATION_SAME_OF:
            viaGfxInfo->screenAttr.uint = VIA_CLONE;
            break;
    }

    memset(&(viaGfxInfo->igaInfo[viaCrtc->iga-1]), 0x0, sizeof(viaGfxInfo->igaInfo[viaCrtc->iga - 1]));
    /*update rotation info xrandr's rotation is counter-clockwise*/
    switch (crtc->rotation & rotateMask) {		
        case RR_Rotate_90:
            viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_270 = TRUE;
            break;
        case RR_Rotate_180:
            viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_180 = TRUE;
            break;
        case RR_Rotate_270:
            viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_90 = TRUE;
            break;
    }

    /*update reflection info*/
    switch (crtc->rotation & reflectMask) {
        case RR_Reflect_X:
            viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.reflect_x = TRUE;
            break;
        case RR_Reflect_Y:
            viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.reflect_y = TRUE;
            break;
        case RR_Reflect_X|RR_Reflect_Y:
            viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.reflect_x = TRUE;
            viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.reflect_y = TRUE;
            break;
        default:
            break;
    }
    
    /*simplify igaRRStatus info for xy reflection .It equals to 180 degree rotation in 
    addtion to original rotation degree at nature*/
    if ((viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.reflect_x == TRUE) &&
        (viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.reflect_y == TRUE)) {
        switch (crtc->rotation&rotateMask) {	
            case RR_Rotate_0:
                viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_180 = TRUE;
                break;
            case RR_Rotate_90:
                viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_90 = TRUE;
                viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_270 = FALSE;
                break;
            case RR_Rotate_180:
                viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_180 = FALSE;
                break;
            case RR_Rotate_270:
                viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_90 = FALSE;
                viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_270 = TRUE;
                break;
        }
        viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.reflect_x = FALSE;
        viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.reflect_y = FALSE;
    }
    
    /*fill the related information of IGA*/
    for (i = 0; i < xf86_config->num_output; i++) {
        xf86OutputPtr output = xf86_config->output[i];
        if (output->crtc == crtc) {
            activeDevice = getActiveDevice(output->name);
        }
    }
    viaGfxInfo->igaInfo[viaCrtc->iga-1].start_x = crtc->x;
    viaGfxInfo->igaInfo[viaCrtc->iga-1].start_y = crtc->y;
    viaGfxInfo->igaInfo[viaCrtc->iga-1].crtc_width = adjusted_mode->CrtcHDisplay;
    viaGfxInfo->igaInfo[viaCrtc->iga-1].crtc_height = adjusted_mode->CrtcVDisplay;
    viaGfxInfo->igaInfo[viaCrtc->iga-1].desired_width = mode->HDisplay;
    viaGfxInfo->igaInfo[viaCrtc->iga-1].desired_height = mode->VDisplay;
    viaGfxInfo->igaInfo[viaCrtc->iga-1].refreshrate = adjusted_mode->VRefresh;
    viaGfxInfo->igaInfo[viaCrtc->iga-1].activeDevice = activeDevice;
    viaGfxInfo->igaInfo[viaCrtc->iga-1].actived = TRUE;

    /*if we do left or right rotate, we need to exchange the HDisplay VDisplay update to video*/ 
    if ((viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_90)
        ||(viaGfxInfo->igaInfo[viaCrtc->iga-1].igaRRStatus.rotate_270)) {
       viaGfxInfo->igaInfo[viaCrtc->iga-1].visible_width = mode->VDisplay;
       viaGfxInfo->igaInfo[viaCrtc->iga-1].visible_height = mode->HDisplay;
    } else {
       viaGfxInfo->igaInfo[viaCrtc->iga-1].visible_width = mode->HDisplay;
       viaGfxInfo->igaInfo[viaCrtc->iga-1].visible_height = mode->VDisplay;
    }
    
    /*fill the related information of Screen*/
    memset(&(viaGfxInfo->screenInfo[pScrn->scrnIndex]), 0x0, sizeof(viaGfxInfo->screenInfo[pScrn->scrnIndex]));
    viaGfxInfo->screenInfo[pScrn->scrnIndex].bpp= pScrn->bitsPerPixel;
    for (i = 0; i < xf86_config->num_output; i++) {
        if (xf86_config->output[i]->crtc){
            VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(xf86_config->output[i]->crtc);
            viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse |= viaCrtc->iga;
        }
    }
}

static void
via_crtc_commit (xf86CrtcPtr crtc)
{
    ScrnInfoPtr pScrn = crtc->scrn;
    VIAPtr pVia = VIAPTR(pScrn);
    VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);
    PVIDDATA pVidData = pVia->pVidData;

	/*1. If only IGA2 is used, set a default timing on IGA1*/
	if (!viaCrtc->enabled && (viaCrtc->iga != IGA1)) {
		if (!viaIga1RequirePatchClk(pScrn)) {
			DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "via_crtc_commit, fail to patch Iga1 clock!\n"));    
		}
	}
    
	/*3. Reload cursor*/
    xf86_reload_cursors(pScrn->pScreen);

	/*4. DPMS on*/
	crtc->funcs->dpms(crtc, DPMSModeOn);
	
	/*5. Crtc private data structure update*/
	viaCrtc->enabled = TRUE;
}

static Bool
via_crtc_mode_fixup(xf86CrtcPtr crtc, DisplayModePtr mode,
                        DisplayModePtr adjusted_mode)
{
    return TRUE;
}

/** Sets the color ramps on behalf of RandR */
static void
via_crtc_gamma_set(xf86CrtcPtr crtc, CARD16 *red, CARD16 *green, CARD16 *blue, int size)
{
	VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);
	int i;

	assert(size == 256);

	for (i = 0; i < 256; i++) {
		viaCrtc->colors[i].red = red[i] >> 8;
		viaCrtc->colors[i].green = green[i] >> 8;
		viaCrtc->colors[i].blue = blue[i] >> 8;	
	}

	viaSetGammaOrLUT(crtc);
}


/**
 * Allocates memory for a locked-in-framebuffer shadow of the given
 * width and height for this CRTC's rotated shadow framebuffer.
 */
 
static void *
via_crtc_shadow_allocate (xf86CrtcPtr crtc, int width, int height)
{
	ScrnInfoPtr pScrn = crtc->scrn;
	VIAPtr pVia = VIAPTR(pScrn);
	VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc); 

    int size = height * CMDISP_ALIGN_TO(width * (pScrn->bitsPerPixel/8), 32);
    int ret = viaVideoMemAlloc(pScrn, &viaCrtc->rotateMem, size);
    if((ret == BadAlloc) ||(!viaCrtc->rotateMem.pool) || (!viaCrtc->rotateMem.base)) {
        return NULL;
    }
    memset(pVia->FBBase + viaCrtc->rotateMem.base, 0, size);
    return (pVia->FBBase + viaCrtc->rotateMem.base);
}
    
/**
 * Creates a pixmap for this CRTC's rotated shadow framebuffer.
 */
static PixmapPtr
via_crtc_shadow_create(xf86CrtcPtr crtc, void *data, int width, int height)
{
	ScrnInfoPtr		pScrn = crtc->scrn;
	VIAPtr			pVia = VIAPTR(pScrn);
	PixmapPtr   	rotatePixmap;
	unsigned long 	rotatePitch;

	if (!data)
		data = via_crtc_shadow_allocate(crtc, width, height);

    rotatePitch = CMDISP_ALIGN_TO(width * pVia->Bpp, 32);

	rotatePixmap = GetScratchPixmapHeader(pScrn->pScreen,
                       		width, height,
                       		pScrn->depth,
                       		pScrn->bitsPerPixel,
                       		rotatePitch,
                       		data);
	
	if (rotatePixmap == NULL)
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Couldn't allocate shadow pixmap for rotated CRTC\n");
	
	return rotatePixmap;
}

static void
via_crtc_shadow_destroy(xf86CrtcPtr crtc, PixmapPtr rotate_pixmap, void *data)
{
    ScrnInfoPtr pScrn = crtc->scrn;
    VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc); 

    viaVideoMemFree(pScrn,&viaCrtc->rotateMem);

	if (rotate_pixmap)
		FreeScratchPixmapHeader(rotate_pixmap); 	
}

static void
via_crtc_set_cursor_colors (xf86CrtcPtr crtc, int bg, int fg)
{
	/*Only used when crtc->funcs->load_cursor_image != NULL
	but now we set crtc->funcs->load_cursor_image == NULL, so we need do nothing here*/
}

static void
via_crtc_show_cursor (xf86CrtcPtr crtc)
{
	ScrnInfoPtr     pScrn = crtc->scrn;
	VIAPtr          pVia = VIAPTR(pScrn);
	VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);

	/*1.Set HI start address to reg*/
	/*2.Turn on Hardware icon Cursor */
	switch (pVia->Chipset) {
	default:
		if (viaCrtc->iga == IGA1) {
			VIASETREG(PRIM_HI_FBOFFSET, pVia->HiStartBuf[0]);
			VIASETREG(PRIM_HI_CTRL, 0x36000005);
		} else {
			VIASETREG(HI_FBOFFSET, pVia->HiStartBuf[1]);
			VIASETREG(HI_CONTROL, 0xB6000005);
		}            
		break;
	case VIA_P4M800PRO:
		if (viaCrtc->iga == IGA1) {
			VIASETREG(HI_FBOFFSET, pVia->HiStartBuf[0]);
			/*enableV4 Window Pre-fetch [bit 30]for solving VT3336  bug:
			when do some actions such as play video cursor will jump*/
			VIASETREG(HI_CONTROL, 0x76000005);
		} else {
			VIASETREG(HI_FBOFFSET, pVia->HiStartBuf[1]);
			VIASETREG(HI_CONTROL, 0xf6000005);
		}
		break;
    }
}

static void
via_crtc_hide_cursor (xf86CrtcPtr crtc)
{
	ScrnInfoPtr     pScrn = crtc->scrn;    
	VIAPtr          pVia = VIAPTR(pScrn);
	VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);

	/* Turn hardware icon cursor off. */
	switch (pVia->Chipset) {
	default:
		if (viaCrtc->iga == IGA1)
			VIASETREG(PRIM_HI_CTRL, VIAGETREG(PRIM_HI_CTRL)&0xFFFFFFFA);
		else
			VIASETREG(HI_CONTROL, VIAGETREG(HI_CONTROL)&0xFFFFFFFA);
		break;
	case VIA_P4M800PRO:
		VIASETREG(HI_CONTROL, VIAGETREG(HI_CONTROL)&0xFFFFFFFA);
		break;
	}
}

static void
via_crtc_set_cursor_position (xf86CrtcPtr crtc, int x, int y)
{
	ScrnInfoPtr pScrn = crtc->scrn;  
	VIAPtr pVia = VIAPTR(pScrn);
	int xpos, ypos;
	unsigned char xoff, yoff;
	VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);

	/*1.Get IGA cursor position*/
	get_crtc_cursor_real_position(crtc, x, y, &xpos, &ypos, &xoff, &yoff);

	/*2.Set the position to registers*/
	switch(pVia->Chipset){
	case VIA_CX700:
	case VIA_P4M890:
	case VIA_P4M900:
	case VIA_VX800:
	case VIA_VX855:
		if (viaCrtc->iga == IGA1) {
			VIASETREG(PRIM_HI_POSSTART, ((xpos << 16) | (ypos & 0x07ff)));
			VIASETREG(PRIM_HI_CENTEROFFSET, ((xoff << 16) | (yoff & 0x07ff)));
		} else {
			VIASETREG(HI_POSSTART, ((xpos << 16) | (ypos & 0x07ff)));
			VIASETREG(HI_CENTEROFFSET, ((xoff << 16) | (yoff & 0x07ff)));
		}
		break;
	default:
		VIASETREG(HI_POSSTART, ((xpos << 16) | (ypos & 0x07ff)));
		VIASETREG(HI_CENTEROFFSET, ((xoff << 16) | (yoff & 0x07ff)));
		break;
	 }
}

static void
via_crtc_load_cursor_image (xf86CrtcPtr crtc, CARD8 *src)
{
	/*Now xrandr1.2 do not use this function*/
}

static void
via_crtc_load_cursor_argb (xf86CrtcPtr crtc, CARD32 *image)
{
	ScrnInfoPtr pScrn = crtc->scrn;
	VIAPtr pVia = VIAPTR(pScrn);  
	VIACrtcPrivatePtr viaCrtc = VIACrtcPrivate(crtc);
	int hiBufIndex = (viaCrtc->iga == IGA1) ? 0 : 1;

	/*3314 only has one HW Icon, other chips have two*/
	memset(pVia->FBBase + pVia->HiStartBuf[hiBufIndex], 0, 
			MAX_CURS * MAX_CURS * 4);
	memcpy(pVia->FBBase + pVia->HiStartBuf[hiBufIndex], image, 
			MAX_CURS * MAX_CURS * 4);
}


#if RANDR_13_INTERFACE
static void
via_crtc_set_origin(xf86CrtcPtr crtc, int x, int y)
{
    if (crtc->enabled)
	    viaSetIgaStartAddr(crtc, x, y);
}
#endif


static const xf86CrtcFuncsRec via_crtc_funcs = {
    .dpms = via_crtc_dpms,
    .save = NULL, /* XXX */
    .restore = NULL, /* XXX */
    .lock = via_crtc_lock,
    .unlock = via_crtc_unlock,
    .mode_fixup = via_crtc_mode_fixup,
    .prepare = via_crtc_prepare,
    .mode_set = via_crtc_mode_set,
    .commit = via_crtc_commit,
    .gamma_set = via_crtc_gamma_set,
    .shadow_create = via_crtc_shadow_create,
    .shadow_allocate = via_crtc_shadow_allocate,  
    .shadow_destroy = via_crtc_shadow_destroy,
    .set_cursor_colors = via_crtc_set_cursor_colors,
    .set_cursor_position = via_crtc_set_cursor_position,
    .show_cursor = via_crtc_show_cursor,
    .hide_cursor = via_crtc_hide_cursor,    
    /*.load_cursor_image = via_crtc_load_cursor_image,     */ /* Use HWIcon instead of HWCursor*/
    .load_cursor_argb = via_crtc_load_cursor_argb,    
    .destroy = NULL, /* XXX */
#if RANDR_13_INTERFACE
    .set_origin = via_crtc_set_origin,
#endif    
};


void via_crtc_init(ScrnInfoPtr pScrn, int igaNum)
{
	xf86CrtcPtr crtc;
	VIACrtcPrivatePtr viaCrtc;
	int i, j;

	for (i = 0; i < igaNum; i++) {		 
		crtc = xf86CrtcCreate (pScrn, &via_crtc_funcs);

		if (crtc == NULL) {
			xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "xf86CrtcCreate failed.\n");
			return;
		}
		viaCrtc = xnfcalloc (sizeof (VIACrtcPrivateRec), 1);
		viaCrtc->iga = (0 == i) ? IGA1 : IGA2;

		 /* Initialize the LUTs for when we turn on the CRTC. */
		for (j = 0; j < 256; j++) {
			viaCrtc->colors[j].red = j;
			viaCrtc->colors[j].green = j;
			viaCrtc->colors[j].blue = j;
		}
		
		crtc->driver_private = viaCrtc;
	}	 
}

#endif

