/*
 * Copyright 1998-2009 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2009 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "via_driver.h"
#ifdef VIA_RANDR12_SUPPORT
#include "via_modedata.h"
#include "via_output.h"
#include "via_displcd.h"
#include "via_serial.h"

void 
viaDelayIn_usec(VIAPtr pVia, int usec)
{
	volatile long i;
	unsigned char tmp;

	if (usec > 0) {
		for (i = usec * 4; i > 0; i--) {
			tmp = STANDVGA_R8(0x3c3);
		}
	}
}

Bool 
viaDelay_Nmsec(VIAPtr pVia, CARD32 dwCounter)
{
	CARD32 i = 0;
    
	if (pVia->Chipset >= VIA_VX800 || pVia->Chipset == VIA_CX700) {
		viaWriteVgaIoBits(REG_SR5D, 0x70, BIT4+BIT5+BIT6);
        
		while (1){
			if (viaReadVgaIo(REG_SR5D)&BIT7) {
				i++;
				viaWriteVgaIoBits(REG_SR5D, 0x70, BIT4+BIT5+BIT6);
			}

			if (i==dwCounter) {
				break;
			}
		}
	} else {
        viaDelayIn_usec(pVia, dwCounter*1000);
	}
	
    return TRUE;
}


/* Set LCD software power sequence off */
void 
viaSWPowerSequenceOFF(VIAPtr pVia, unsigned long diPort, unsigned long panelSizeId)
{    
	CARD32 TD0 = 0, TD1 = 0, TD2 = 0, TD3 = 0;        
        
	TD0 = LCD_POWER_SEQUENCE_TD0;
    TD1 = LCD_POWER_SEQUENCE_TD1;
    TD2 = LCD_POWER_SEQUENCE_TD2;
    TD3 = LCD_POWER_SEQUENCE_TD3;
			
	if (diPort == DISP_DI_DFPH) {
		DEBUG(xf86DrvMsg(0, X_PROBED, "LVDS1 SW Power Sequence OFF\n"));
        
		/* Back-Light OFF */
		viaWriteVgaIoBits(REG_CRD3, 0x0, BIT1);     

		/* Delay TD3 msec. */
		viaDelay_Nmsec(pVia, TD3);

		/* VEE OFF */
		viaWriteVgaIoBits(REG_CRD3, 0x0, BIT2);

		/* Delay TD2 msec. */
		viaDelay_Nmsec(pVia, TD2);        
        
		/* DATA OFF */
		viaWriteVgaIoBits(REG_CRD3, 0x0, BIT3);

		/* Delay TD1 msec. */
		viaDelay_Nmsec(pVia, TD1);
        
		/* VDD OFF */    
		viaWriteVgaIoBits(REG_CRD3, 0x0, BIT4); 

    } else {
  		DEBUG(xf86DrvMsg(0, X_PROBED, "LVDS0 SW Power Sequence OFF\n"));

		/* Back-Light OFF */
		viaWriteVgaIoBits(REG_CR91, 0x0, BIT1);

		/* Delay TD3 msec. */
		viaDelay_Nmsec(pVia, TD3);

		/* VEE OFF (unused on vt3353)*/
		viaWriteVgaIoBits(REG_CR91, 0x0, BIT2);     

		/* Delay TD2 msec. */
		viaDelay_Nmsec(pVia, TD2);
        
		/* DATA OFF */    
		viaWriteVgaIoBits(REG_CR91, 0x0, BIT3); 

		/* Delay TD1 msec. */
		viaDelay_Nmsec(pVia, TD1);
        
		/* VDD OFF */
		viaWriteVgaIoBits(REG_CR91, 0x0, BIT4); 
	} 
}

/* Set LCD software power sequence on */
void 
viaSWPowerSequenceON(VIAPtr pVia, unsigned long diPort, unsigned long panelSizeId)
{   
	CARD32 TD0 = 0, TD1 = 0, TD2 = 0, TD3 = 0;    

	TD0 = LCD_POWER_SEQUENCE_TD0;
    TD1 = LCD_POWER_SEQUENCE_TD1;
    TD2 = LCD_POWER_SEQUENCE_TD2;
    TD3 = LCD_POWER_SEQUENCE_TD3;
	   
	if (diPort == DISP_DI_DFPH) {
		DEBUG(xf86DrvMsg(0, X_PROBED, "LVDS1 SW Power Sequence ON\n"));

		/* Secondary power hardware power sequence enable 0:off 1: on */
		viaWriteVgaIoBits(REG_CRD4, 0x0, BIT1);
        
		/* Software control power sequence ON*/
		viaWriteVgaIoBits(REG_CRD3, 0x1, BIT0);

		/* Delay TD0 msec. */
		viaDelay_Nmsec(pVia, TD0);

		/* VDD ON*/
		viaWriteVgaIoBits(REG_CRD3, 0x10, BIT4);

 		/* Delay TD1 msec. */
		viaDelay_Nmsec(pVia, TD1);

		/* DATA ON */
		viaWriteVgaIoBits(REG_CRD3, 0x08, BIT3);

		/* Delay TD2 msec. */
		viaDelay_Nmsec(pVia, TD2);

		/* VEE ON (unused on vt3353)*/
		viaWriteVgaIoBits(REG_CRD3,0x04, BIT2);    

		/* Delay TD3 msec. */
		viaDelay_Nmsec(pVia, TD3);

		/* Back-Light ON */
		viaWriteVgaIoBits(REG_CRD3, 0x02, BIT1);

	} else {	
		DEBUG(xf86DrvMsg(0, X_PROBED, "LVDS0 SW Power Sequence ON\n"));

		/* Software control power sequence ON*/    
		viaWriteVgaIoBits(REG_CR91,0x0,BIT7);
		viaWriteVgaIoBits(REG_CR91,0x1,BIT0);


		/* Delay TD0 msec.  */
		viaDelay_Nmsec(pVia, TD0);

		/* VDD ON*/
		viaWriteVgaIoBits(REG_CR91,0x10,BIT4);

		/* Delay TD1 msec.*/
		viaDelay_Nmsec(pVia, TD1);    

		/* DATA ON */    
		viaWriteVgaIoBits(REG_CR91,0x08,BIT3);

		/* Delay TD2 msec. */
		viaDelay_Nmsec(pVia, TD2);

		/* VEE ON (unused on vt3353)*/
		viaWriteVgaIoBits(REG_CR91, 0x04, BIT2);  

		/* Delay TD3 msec. */
		viaDelay_Nmsec(pVia, TD3);

		/* Back-Light ON */
		viaWriteVgaIoBits(REG_CR91, 0x2, BIT1);
	} 
}

void 
disableInternalLvds(VIAPtr pVia, unsigned long diPort, unsigned long panelSizeId, BOOL dualChannel)
{
	Bool firstPowerSequenceOff = FALSE;
	Bool secondPowerSequenceOff = FALSE;

	if (DISP_DI_DFPH == diPort) {
		secondPowerSequenceOff = TRUE;
	} else {
		firstPowerSequenceOff = TRUE;
	}

	/*1. Turn off LCD panel*/
	if (firstPowerSequenceOff) {
		if (pVia->Chipset == VIA_VX800 ||pVia->Chipset == VIA_CX700) {
			viaSWPowerSequenceOFF(pVia, diPort, panelSizeId);
		} else {
			/* Use first power sequence control: */
			/* Turn off power sequence. */
			viaWriteVgaIoBits(REG_CR6A, 0, BIT3);

			/* Turn off back light and panel path. */
			viaWriteVgaIoBits(REG_CR91, 0xC0, BIT6+BIT7);
		}
	}

	if (secondPowerSequenceOff) {
		if (pVia->Chipset == VIA_VX800 ||pVia->Chipset == VIA_CX700) {
			viaSWPowerSequenceOFF(pVia, diPort, panelSizeId);
		} else {
			/* Use second power sequence control: */
			/* Turn off power sequence. */
			viaWriteVgaIoBits(REG_CRD4, 0, BIT1);
			/* Turn off back light and panel path. */
			viaWriteVgaIoBits(REG_CRD3, 0xC0, BIT6+BIT7);
		}
	}
	
	/*2. Power off LVDS channel. */
    if (dualChannel) {
        /* high resolution LCD (internal), power off both LVDS0 and LVDS1 */
        viaWriteVgaIoBits(REG_CRD2, 0xC0, BIT7+BIT6);
    } else {
    	switch (diPort) {
    	case DISP_DI_DFPL:
    		viaWriteVgaIoBits(REG_CRD2, 0x80, BIT7);
    		break;
    	case DISP_DI_DFPH:
    		viaWriteVgaIoBits(REG_CRD2, 0x40, BIT6);
    		break;
    	default:
    		DEBUG(xf86DrvMsg(0, X_PROBED, "disableInternalLvds:Invalid DIport\n"));
    		break;
    	}
    }	
}

void 
enableInternalLvds(VIAPtr pVia, unsigned long diPort, unsigned long panelSizeId, BOOL dualChannel)
{
	Bool firstPowerSequenceOn = FALSE;
	Bool secondPowerSequenceOn = FALSE;

	if (DISP_DI_DFPH == diPort) {
		secondPowerSequenceOn = TRUE;
	} else {
		firstPowerSequenceOn = TRUE;
	}

	/*1. Turn on LCD panel*/
	if (firstPowerSequenceOn) {
		if (pVia->Chipset == VIA_VX800 || pVia->Chipset == VIA_CX700) {
			viaSWPowerSequenceON(pVia, diPort, panelSizeId);
		} else {
			/* Use first power sequence control: */
			/* Use hardware control power sequence. */
			viaWriteVgaIoBits(REG_CR91, 0, BIT0);
			/* Turn on back light and panel path. */
			viaWriteVgaIoBits(REG_CR91, 0, BIT6+BIT7);
			/* Turn on hardware power sequence. */
			viaWriteVgaIoBits(REG_CR6A, 0x08, BIT3);
		}
	}
	
	if (secondPowerSequenceOn) {
		if (pVia->Chipset == VIA_VX800 ||pVia->Chipset == VIA_CX700) {
			viaSWPowerSequenceON(pVia, diPort, panelSizeId);
		} else {
			/* Use hardware control power sequence. */
			viaWriteVgaIoBits(REG_CRD3, 0, BIT0);
			/* Turn on back light and panel path. */
			viaWriteVgaIoBits(REG_CRD3, 0, BIT6+BIT7);
			/* Turn on hardware power sequence. */
			viaWriteVgaIoBits(REG_CRD4, 0x02, BIT1);
		}
	}

	/*2. Power on LVDS channel. */
	if (dualChannel) {
	    /* high resolution LCD (internal), power on both LVDS0 and LVDS1 */
	    viaWriteVgaIoBits(REG_CRD2, 0x0, BIT6+BIT7);
	} else {
    	switch (diPort){
    	case DISP_DI_DFPL:
    		viaWriteVgaIoBits(REG_CRD2, 0x0, BIT7);
    		break;
    	case DISP_DI_DFPH:
    		viaWriteVgaIoBits(REG_CRD2, 0x0, BIT6);
    		break;
    	default:
    		DEBUG(xf86DrvMsg(0, X_PROBED, "enableInternalLvds:Invalid DIport\n"));
    		break;
    	}
	}
}

/*Just turn on Lcd panel power sequence*/
void 
enableLcdViaVt1636(unsigned long serialPort)
{
	viaSerialWriteByteMask(serialPort, SUBCHIP_VT1636_SLAVE_ADDR, 0x10, 0x20, 0x20);
}

/*Just turn off Lcd panel power sequence*/
void 
disableLcdViaVt1636(unsigned long serialPort)
{
	viaSerialWriteByteMask(serialPort, SUBCHIP_VT1636_SLAVE_ADDR, 0x10, 0x00, 0x20);
}

/* TTL-LCD should use software power-sequence control. */
void 
ttlLcdPowerSequenceOn(VIAPtr pVia)
{    
	DEBUG(xf86DrvMsg(0, X_PROBED, "TTLLCDPowerSequenceOn\n"));

	/* Software control power sequence ON*/
	viaWriteVgaIoBits(REG_CR91, 0x01, BIT0);

	/* VDD ON*/
	viaWriteVgaIoBits(REG_CR91, 0x10, BIT4);

	/* Delay 25 msec. (25msec is HW proposed) */
	viaDelay_Nmsec(pVia, 25);    

	/* DATA ON (TTL-LCD's data is output from DVP1) */
	viaWriteVgaIoBits(REG_SR1E, 0x30, BIT4+BIT5);

	/* Delay 225 msec. (225msec is HW proposed) */
	viaDelay_Nmsec(pVia, 225);

	/* Back-Light ON */
	viaWriteVgaIoBits(REG_CR91, 0x02, BIT1);
}

/* TTL-LCD should use software power-sequence control. */
void
ttlLCDPowerSequenceOff(VIAPtr pVia)
{
	DEBUG(xf86DrvMsg(0, X_PROBED, "TTLLCDPowerSequenceOff\n"));

	/* Back-Light OFF */ 
	viaWriteVgaIoBits(REG_CR91, 0x0, BIT1);

	/* Delay 225 msec. (225msec is HW proposed) */
	viaDelay_Nmsec(pVia, 225);
    
	/* DATA OFF (TTL-LCD's data is output from DVP1) */
	viaWriteVgaIoBits(REG_SR1E, 0x0, BIT4+BIT5);

	/* Delay 25 msec. (25msec is HW proposed) */
	viaDelay_Nmsec(pVia, 25);
    
	/* VDD OFF */
	viaWriteVgaIoBits(REG_CR91, 0x0, BIT4);
}

/* according to the width and height,get the built-in timing */
ViaTimingTablePtr
getDriverTiming(CARD32 devModeTableIndex, unsigned long modeIndex)
{
	ViaTimingTablePtr devModeTable = NULL;
	int i, arrayNum = 0;
	
	switch (devModeTableIndex) {
	case LCD_MODE_TABLE:
		devModeTable = VIALCDModes;
		arrayNum = NUM_VIALCD_TIMING_TBL;
		break;
	default:
		break;
	}
	
	if (devModeTable) {
		for (i = 0; i < arrayNum; i++) {
			if (devModeTable[i].ModeIndex == modeIndex)
				return &devModeTable[i];
		}
	}
	return NULL;
}

Bool 
fillModeByDriverTiming(ViaTimingTablePtr pTimingTable, DisplayModePtr mode)
{
	DisplayTimingPtr pCrtcTiming = NULL;

	if (!pTimingTable)	
		return FALSE;
    
	pCrtcTiming = &pTimingTable->Timing;
	/*Valuate mode from build-in timing*/	
	mode->HDisplay = pCrtcTiming->HorAddr;
	mode->HSyncStart = pCrtcTiming->HorSyncStart;
	mode->HSyncEnd = pCrtcTiming->HorSyncStart+pCrtcTiming->HorSyncEnd;
	mode->HTotal = pCrtcTiming->HorTotal;
	mode->VDisplay = pCrtcTiming->VerAddr;
	mode->VSyncStart = pCrtcTiming->VerSyncStart;
	mode->VSyncEnd = pCrtcTiming->VerSyncStart+pCrtcTiming->VerSyncEnd;
	mode->VTotal = pCrtcTiming->VerTotal;
	mode->Clock = pTimingTable->Clk/1000;
	return TRUE;
}

/* Get a build-in mode record according to panel id. */
DisplayModePtr 
viaGetPanelModeRecord(ScrnInfoPtr pScrn, CARD32 panelID)
{
    DisplayModePtr  panelMode;
	ViaTimingTablePtr panelCrtcTable = NULL;

	panelCrtcTable = getDriverTiming(LCD_MODE_TABLE, panelID);
	if (!panelCrtcTable)
		return NULL;

    panelMode = xnfalloc(sizeof(DisplayModeRec));
    memset(panelMode, 0, sizeof(*panelMode));
	fillModeByDriverTiming(panelCrtcTable, panelMode);
    panelMode->type |= M_T_PREFERRED;
    xf86SetModeDefaultName(panelMode);

    /*When we using panel 1366x768,want to light mode 1366x768
      but in our mode list we just have 1368x768,so change
      the name to 1366x768*/
    if (panelMode->HDisplay==1368&&panelMode->VDisplay==768) {
        strcpy(panelMode->name, "1366x768");        
    }
    return panelMode;
}

/* caculate the cetering timing using mode and adjusted_mode*/
void
centeringTiming(DisplayModePtr mode, DisplayModePtr adjusted_mode)
{
	int panelHSyncTime = adjusted_mode->HSyncEnd - adjusted_mode->HSyncStart;
	int panelVSyncTime = adjusted_mode->VSyncEnd - adjusted_mode->VSyncStart;
	int panelHBlankStart = adjusted_mode->HDisplay;
	int panelVBlankStart = adjusted_mode->VDisplay;	
	int hBorder = (adjusted_mode->HDisplay - mode->HDisplay)/2;
	int vBorder = (adjusted_mode->VDisplay - mode->VDisplay)/2;
	int newHBlankStart = hBorder + mode->HDisplay;
	int newVBlankStart = vBorder + mode->VDisplay;	

	adjusted_mode->HDisplay = mode->HDisplay;
	adjusted_mode->HSyncStart = (adjusted_mode->HSyncStart - panelHBlankStart) + newHBlankStart;
	adjusted_mode->HSyncEnd = adjusted_mode->HSyncStart + panelHSyncTime;
	adjusted_mode->VDisplay = mode->VDisplay;
	adjusted_mode->VSyncStart = (adjusted_mode->VSyncStart - panelVBlankStart) + newVBlankStart;
	adjusted_mode->VSyncEnd = adjusted_mode->VSyncStart + panelVSyncTime;
	/*Adjust Crtc H and V*/
	adjusted_mode->CrtcHDisplay = adjusted_mode->HDisplay;
	adjusted_mode->CrtcHBlankStart = newHBlankStart;
	adjusted_mode->CrtcHBlankEnd = adjusted_mode->CrtcHTotal - hBorder;
	adjusted_mode->CrtcHSyncStart = adjusted_mode->HSyncStart;
	adjusted_mode->CrtcHSyncEnd = adjusted_mode->HSyncEnd;
	adjusted_mode->CrtcVDisplay = adjusted_mode->VDisplay;
	adjusted_mode->CrtcVBlankStart = newVBlankStart;
	adjusted_mode->CrtcVBlankEnd = adjusted_mode->CrtcVTotal - vBorder;
	adjusted_mode->CrtcVSyncStart = adjusted_mode->VSyncStart;
	adjusted_mode->CrtcVSyncEnd = adjusted_mode->VSyncEnd;
}

void
loadScaleFactorRegs(CARD32 mode, CARD32 panel, Bool isHorizontal)
{
    CARD32 regValue = 0;
    CARD32 regAmount = 0;
    ViaIoBitPtr reg = NULL;

    if (isHorizontal) {
        regValue = LCD_HOR_SCALE_FACTOR_FORMULA(mode, panel);
        regAmount = VIALcdScalingRegs.HorRegs.RegNum;
        reg = VIALcdScalingRegs.HorRegs.Reg;
        viaLoadRegs(regValue, regAmount, reg);
		/*Horizontal scaling enable*/
        viaWriteVgaIoBits(REG_CRA2, BIT7, BIT7);
    } else {
        regValue = LCD_VER_SCALE_FACTOR_FORMULA(mode, panel);
        regAmount = VIALcdScalingRegs.VerRegs.RegNum;
        reg = VIALcdScalingRegs.VerRegs.Reg;
        viaLoadRegs(regValue, regAmount, reg);
		/*Vertical scaling enable*/
        viaWriteVgaIoBits(REG_CRA2, 0x08, 0x08);
    }
	/* LCD scaling enabled and set H & V to Interpolation */
    viaWriteVgaIoBits(REG_CR79, 0x07, 0x07);
}


void 
initVt1636(ViaLcdPrivateInfoPtr lcdPriv)
{
    CARD32 RegNum = 0;
	CARD32 SerialPort = lcdPriv->commonInfo.serialPort;
	CARD32 IsDualChannel = lcdPriv->dualChannel;
	CARD32 MSB = lcdPriv->msb;
	CARD32 IsDithering = !lcdPriv->noDithering;
	
    int i = 0;

    /*Registers setting*/
    if (SerialPort) {
        RegNum = CMDISP_ARRAYSIZE(INIT_TBL_VT1636);
        for(i = 0; i < RegNum; i++) {
            viaSerialWriteByteMask(SerialPort, SUBCHIP_VT1636_SLAVE_ADDR, INIT_TBL_VT1636[i].index, INIT_TBL_VT1636[i].data, INIT_TBL_VT1636[i].mask);
        }

        /*Dual channel: register offset 0x08*/
        if (IsDualChannel) {
            viaSerialWriteByteMask(SerialPort, SUBCHIP_VT1636_SLAVE_ADDR, 0x08, 0xE0, 0xE0);
        } else {
            viaSerialWriteByteMask(SerialPort, SUBCHIP_VT1636_SLAVE_ADDR, 0x08, 0x00, 0xE0);    
        }

        /*MSB or LSB: register offset 0x08*/
        if (MSB) {
            viaSerialWriteByteMask(SerialPort, SUBCHIP_VT1636_SLAVE_ADDR, 0x08, 0x10, 0x10);
        } else {
            viaSerialWriteByteMask(SerialPort, SUBCHIP_VT1636_SLAVE_ADDR, 0x08, 0x00, 0x10);    
        }

        /*Dither: register offset 0x0A*/
        if (IsDithering) {
            viaSerialWriteByteMask(SerialPort, SUBCHIP_VT1636_SLAVE_ADDR, 0x0A, 0x50, 0x70);
        } else {
            viaSerialWriteByteMask(SerialPort, SUBCHIP_VT1636_SLAVE_ADDR, 0x0A, 0x00, 0x70);    
        }
    }
}

/*some of the LVDS parameters, we only need to set once so we abstract them out,do it in here*/
void 
initLVDS(ViaLcdPrivateInfoPtr lcdPriv)
{
	unsigned char RegCR88 = 0;
	
	if (lcdPriv->commonInfo.subChipName == SUBCHIP_VT1636) {
		initVt1636(lcdPriv);
	} else if (lcdPriv->commonInfo.subChipName == SUBCHIP_INTEGRATED_LVDS) {
		/*MSB*/
		if (lcdPriv->commonInfo.diPort == DISP_DI_DFPH) {
            if (lcdPriv->msb) {
				viaWriteVgaIoBits(REG_CRD2, 0x00, BIT0);
			} else {
				viaWriteVgaIoBits(REG_CRD2, 0x01, BIT0);
			}
		} else { 
			if (lcdPriv->msb) {
				viaWriteVgaIoBits(REG_CRD2, 0x00, BIT1);
			} else {
				viaWriteVgaIoBits(REG_CRD2, 0x02, BIT1);
			}

		}
		/*Dual channel*/
		if (lcdPriv->dualChannel) {
			RegCR88 = BIT4;
		}
		/*Dithering*/
		if (!lcdPriv->noDithering) {
			RegCR88 |= BIT0;
		}
		viaWriteVgaIoBits(REG_CR88, RegCR88, BIT4+BIT0);
	}
}

Bool 
isPanelSizeValid(CARD32 width, CARD32 height)
{
	int i;
	
	for (i = 0; i<NUM_VIA_SUPPORT_PANEL; i++) {
		if ((ViaSupportLCDPanel[i].Width == width)&&
			    (ViaSupportLCDPanel[i].Height == height))
			return TRUE;
		else
			continue;
	}
	return FALSE;
}

Bool 
viaGetPanelModeLine(xf86OutputPtr output)
{
    ViaLcdPrivateInfoPtr viaLcdInfo = output->driver_private;
    Bool ret = FALSE;
    DisplayModePtr panelModePtr;
    DisplayModePtr mode;

    viaLcdInfo->confMode = NULL;
    if (output->conf_monitor)  {        
        panelModePtr = xf86GetMonitorModes(output->scrn, output->conf_monitor);

        for (mode = panelModePtr; ; mode = mode->next) {
            if (mode == NULL)
                break;
            
            if (!memcmp(mode->name, "PanelMode", 9)) {
                viaLcdInfo->commonInfo.physicalWidth = mode->HDisplay;
                viaLcdInfo->commonInfo.physicalHeight = mode->VDisplay;
                viaLcdInfo->panelIndex = VIA_MAKE_ID(mode->HDisplay, mode->VDisplay);
                viaLcdInfo->confMode = xnfalloc(sizeof(DisplayModeRec));
                memcpy(viaLcdInfo->confMode, mode, sizeof(DisplayModeRec));
                ret = TRUE;
            }
                
            if ((mode->next == panelModePtr)  || (mode->next == NULL))
                break;  /* The end of mode list. */        
        }   
    }
    return ret;
}
#endif

