/*
 * Copyright 1998-2009 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2009 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */


#include "via_dmabuffer.h"
#include "via_driver.h"
#include "h1hwreg.h"

/* Prepare the common resource for (EXA) usermode command buffer management */

#ifdef XF86DRI
/*
 * Flush the command buffer using DRM. If in PCI mode, we can bypass DRM,
 * but not for command buffers that contain 3D engine state, since then
 * the DRM command verifier will lose track of the 3D engine state.
 */
void viaFlushDRIEnabled(ViaCommandBuffer * cb)
{
    ScrnInfoPtr pScrn = cb->pScrn;
    VIAPtr pVia = VIAPTR(pScrn);
    char *tmp = (char *)cb->buf;
    int tmpSize;
    drm_via_cmdbuffer_t b;

    if (!pVia->IsPCI && cb->mode == 2 && cb->rindex != EXA_HC_ParaType_CmdVdata
        && (cb->pos & 1)) {
        OUT_RING(EXA_HC_DUMMY);
    }

    tmpSize = cb->pos * sizeof(CARD32);
    if (!pVia->IsPCI || (pVia->directRenderingEnabled && cb->has3dState)) {
        cb->mode = 0;
        cb->has3dState = FALSE;
        while (tmpSize > 0) {
            b.size = (tmpSize > VIA_DMASIZE) ? VIA_DMASIZE : tmpSize;
            tmpSize -= b.size;
            b.buf = tmp;
            tmp += b.size;
            if (drmCommandWrite(pVia->drmFD, ((!pVia->IsPCI)
                                              ? DRM_VIA_CMDBUFFER :
                                              DRM_VIA_PCICMD), &b, sizeof(b))) {
                ErrorF("DRM command buffer submission failed.\n");
                return;
            }
        }
        cb->pos = 0;
        cb->mode = 0;
        cb->has3dState = 0;
    } else {
        viaFlushPCI(cb);
    }
    cb->needFire=FALSE;
}


/*
 * Flush the command buffer using DRM. If in PCI mode, we can bypass DRM,
 * but not for command buffers that contain 3D engine state, since then
 * the DRM command verifier will lose track of the 3D engine state.
 */
void viaFlushDRIEnabled_H5(ViaCommandBuffer * cb)
{
    ScrnInfoPtr pScrn = cb->pScrn;
    VIAPtr pVia = VIAPTR(pScrn);
    int *tmp = (int *)cb->buf;
    int tmpSize;
	int ret, i;
    drm_s3g_flush_t b;

    while(cb->pos & 0x3) OUT_RING(0xCC000000);
	
    tmpSize = cb->pos;

    b.dma_cmd_type = flush_dma_buffer;
    if (!pVia->IsPCI || (pVia->directRenderingEnabled && cb->has3dState)) {
        cb->mode = 0;
        cb->has3dState = FALSE;
        
#if VIA_APPLY_AGP_BRANCH
        if(pVia->umdBranchQuery.branch_buf_enabled) {
            pVia->umdBranchBuf.cmd_size = cb->pos;
            viaFlushBranchAgpBuffer(pVia);
            if(viaRequestBranchAgpBuffer(pVia)) {
                /* The usermode cmd buffer size in unit of CARD32 */
                cb->bufSize = cb->bufSize;    /* keep the size of intial phase*/
                cb->buf = (CARD32 *) pVia->umdBranchBuf.user_virtual;
            } else {
                /* Fall back to normal AGP cmd buffer */
                cb->bufSize = min(cb->bufSize, VIA_CMDBUFSIZE>>2); /* may be changed here */
                cb->buf = (CARD32 *) xcalloc(cb->bufSize, sizeof(CARD32));
            }
        } 
        else
#endif
        {
            while (tmpSize > 0) {
                b.cmd_size = (tmpSize > VIA_DMASIZE) ? VIA_DMASIZE : tmpSize;
                tmpSize -= b.cmd_size;
                b.usermode_dma_buf = tmp;
                tmp += b.cmd_size;
                do {
                ret = drmCommandWrite(pVia->drmFD, DRM_S3G_FLUSH, \
                								&b, sizeof(b));
                } while (ret == -EAGAIN);
            }
        }
        cb->pos = 0;
        cb->mode = 0;
        cb->has3dState = 0;
    } else {
        viaFlushPCI_H5(cb);
    }
    cb->needFire=FALSE;
}

#endif

void
viaBeginHeaderAgp(ViaCommandBuffer * cb, unsigned headType, unsigned regIndex)
{
    CARD32 *hb;

    cb->mode = headType;
    cb->rindex = regIndex;
    while(cb->pos & 1)
        OUT_RING(EXA_HC_DUMMY);
    cb->header_start = cb->pos;
    cb->pos +=4;
}
void
viaFinishHeaderAgp(ViaCommandBuffer * cb)
{
    int numDWords, i;

    CARD32 *hb;

    if (!cb->mode)
	return;
    numDWords = cb->pos - cb->header_start - 4;
    hb = cb->buf + cb->header_start;
    switch (cb->mode) {
        case 4:
            hb[0] = HC_AGPHeader5 | cb->rindex;
            hb[1] = numDWords;
            hb[2] = 0x00F50000;	       /* SW debug flag. (?) */
            break;
        default:
            hb[0] = HC_AGPHeader6;
            hb[1] = numDWords >> 1;
            hb[2] = 0x00F60000;	       /* SW debug flag. (?) */
            break;
    }
    hb[3] = 0;
    if (numDWords & 3) {
	for (i = 0; i < (4 - (numDWords & 3)); ++i)
	    OUT_RING(EXA_HC_DUMMY);
    }
    cb->mode = 0;
}

void
viaBeginHeaderAgp_H5(ViaCommandBuffer * cb, unsigned headType, unsigned regIndex)
{
    CARD32 *hb;

    cb->mode = headType;
    cb->rindex = regIndex;
    while(cb->pos & 3)
        OUT_RING(H5_HC_DUMMY);
    cb->header_start = cb->pos;
    cb->pos +=4;
}

void
viaFinishHeaderAgp_H5(ViaCommandBuffer * cb)
{
    int numDWords, i;

    CARD32 *hb;

    if (!cb->mode)
	return;
    numDWords = cb->pos - cb->header_start - 4;
    hb = cb->buf + cb->header_start;
    switch (cb->mode) {
        case 4:
            hb[0] = INV_AGPHeader4 | cb->rindex;
            hb[1] = numDWords;
            hb[2] = 0x00F50000;	       /* SW debug flag. (?) */
            break;
        case 5:
            hb[0] = INV_AGPHeader5;
            hb[1] = numDWords >> 1;
            hb[2] = 0x00F60000;	       /* SW debug flag. (?) */
            break;
        default:
            return;
    }
    hb[3] = 0;
    if (numDWords & 3) {
	for (i = 0; i < (4 - (numDWords & 3)); ++i)
	    OUT_RING(H5_HC_DUMMY);
    }
    cb->mode = 0;
}

void
viaFlushPCI(ViaCommandBuffer * buf)
{
    VIAPtr pVia = VIAPTR(buf->pScrn);
    register CARD32 *bp = buf->buf;
    CARD32 transSetting;
    CARD32 *endp = bp + buf->pos;
    unsigned loop = 0;
    register CARD32 offset = 0;
    register CARD32 cmdCount = 0;

    while (bp < endp) {

        if(*bp == HC_DUMMY) {
            bp ++;
            continue;
        }
        
        switch(*bp & HC_AGPHeader_MASK)
        {
            case HC_AGPHeader0:
                /* AGP Header 0 is designed for Vertex Parameters Transmission, 
                    it is aways work together with AGP Header 2, so we handle AGP Header 0
                    in AGP Header2 case internally 
                */
                /* FIXME. Never reached here, just skip to the following 32 bits position */
                bp++;
                break;
                
            case HC_AGPHeader1:
                while (bp < endp) {
                    if (((*bp & HC_AGPHeader_MASK) == HC_AGPHeader0)||
                         ((*bp & HC_AGPHeader_MASK) == HC_AGPHeader2)||
                         ((*bp & HC_AGPHeader_MASK) == HC_AGPHeader3)||
                         ((*bp & HC_AGPHeader_VIDEOMASK) == HC_AGPHeader5)||
                         ((*bp & HC_AGPHeader_VIDEOMASK) == HC_AGPHeader6)||
                         ((*bp & HC_AGPHeader_VIDEOMASK) == HC_AGPHeader7))
                        break;
                    
                    /* update the I/O Address for a new AGP Header 1 */
                    if((*bp & HC_AGPHeader_MASK) == HC_AGPHeader1) {
                        offset = (*bp++ & 0x1FF) << 2;
                    }
                    if (offset == 0) {
                        /*
                        * Not doing this wait will probably stall the processor
                        * for an unacceptable amount of time in VIASETREG while
                        * other high priority interrupts may be pending.
                        */
                        pVia->myWaitIdle(pVia);
                    }
                    VIASETREG(offset, *bp++);
                }
                break;
                
            case HC_AGPHeader2:
                if (++bp >= endp)     /* skip to the following 32 bits position */
                    return;

                VIASETREG(VIA_REG_TRANSET, transSetting = *bp++);
                
                while (bp < endp) {
                    if ((transSetting != EXA_HC_ParaType_CmdVdata) &&
                        (((*bp & HC_AGPHeader_MASK)== HC_AGPHeader1)||
                          ((*bp & HC_AGPHeader_MASK)== HC_AGPHeader2)||
                          ((*bp & HC_AGPHeader_MASK)== HC_AGPHeader3)||
                         ((*bp & HC_AGPHeader_VIDEOMASK) == HC_AGPHeader5)||
                         ((*bp & HC_AGPHeader_VIDEOMASK) == HC_AGPHeader6)||
                         ((*bp & HC_AGPHeader_VIDEOMASK) == HC_AGPHeader7)))
                        break;
                    
                    VIASETREG(VIA_REG_TRANSPACE, *bp++);
                }
                break;
                
            case HC_AGPHeader3:
                offset = (*bp & 0x1FF)<<2;
                cmdCount = (*bp & 0x1FFF000) >>12;
                while(cmdCount--) {
                    VIASETREG(offset, *bp++);
                    offset +=4;
                }
                break;
                
            case HC_AGPHeader4:
                /* FIXME. Never reached here, AGP Header 4 is not used */
                bp++;
                break;

            case HC_AGPHeader_MASK:
                /* It is Video AGP Header */
                switch(*bp & HC_AGPHeader_VIDEOMASK)
                {
                    case HC_AGPHeader5:
                        offset = *bp & 0xFFFF;
                        cmdCount = *(bp+1) & 0xFFFF;
                        bp += 4;
                        while(cmdCount--)
                            VIASETREG(offset, *bp++);
                        break;

                    case HC_AGPHeader6:
                        cmdCount = *(bp+1) & 0xFFFF;
                        bp += 4;
                        while(cmdCount--) {
                            VIASETREG(*bp & 0xFFFF, *(bp+1));
                            bp +=2;
                        }
                        break;

                    case HC_AGPHeader7:
                        offset = *bp++ & 0xFFFF;
                        VIASETREG(offset, *bp++);
                        offset = *bp++ & 0xFFFF;
                        VIASETREG(offset, *bp++);
                        
                        break;
                    default:
                        /* FIXME. No regular path treament, for example of dummy commands, 
                           Just skip to the following 32 bits position */
                        bp++;
                        break;
                }
                break;
                
            default:
                /* FIXME. No regular path treament, for example of dummy commands, 
                   Just skip to the following 32 bits position */
                bp++;
                break;
        }
        
    }
    buf->pos = 0;
    buf->mode = 0;
    buf->has3dState = FALSE;
}

void
viaFlushPCI_H5(ViaCommandBuffer * buf)
{
    VIAPtr pVia = VIAPTR(buf->pScrn);
    register CARD32 *bp = buf->buf;
    CARD32 transSetting;
    CARD32 *endp = bp + buf->pos;
    unsigned loop = 0;
    register CARD32 offset = 0;
    register CARD32 cmdCount = 0;

    while (bp < endp) {

        if(*bp == H5_HC_DUMMY) {
            bp ++;
            continue;
        }
        
        switch(*bp & INV_AGPHeader_MASK)
        {
            case INV_AGPHeader0:
                cmdCount = *(bp+1);
                bp += 4;
                while(cmdCount--) {
                    offset = *bp++ & 0xFFFF;
                    if(offset == 0) {
                        pVia->myWaitIdle(pVia);
                    }
                    VIASETREG(offset, *bp++);
                }
                break;
                
            case INV_AGPHeader1:
                offset = *bp & 0xFFFF;
                cmdCount = *(bp+1);
                bp += 4;
                while(cmdCount--) {
                    VIASETREG(offset, *bp++);
                    offset += 4;
                }
                break;
                
            case INV_AGPHeader2:
                bp += 2;
                if (bp >= endp)     /* skip to the following 32 bits position */
                    return;

                VIASETREG(INV_REG_3D_TRANS, transSetting = *bp);
                bp += 2;
                
                while (bp < endp) {
                    if ((transSetting != INV_ParaType_CmdVdata) &&
                        (((*bp &INV_AGPHeader_MASK)== INV_AGPHeader0)||
                         ((*bp & INV_AGPHeader_MASK)== INV_AGPHeader1)||
                         ((*bp & INV_AGPHeader_MASK)== INV_AGPHeader2)||
                         ((*bp & INV_AGPHeader_MASK)== INV_AGPHeader3)||
                         ((*bp & INV_AGPHeader_MASK)== INV_AGPHeader4)||
                         ((*bp & INV_AGPHeader_MASK) == INV_AGPHeader5) ||
                         ((*bp & INV_AGPHeader_MASK) == INV_AGPHeader7)))
                        break;
                    
                    VIASETREG(INV_REG_3D_BEGIN, *bp++);
                }
                break;
                
            case INV_AGPHeader3:
                offset = *bp & 0xFFFF;
                cmdCount = *(bp+1);
                bp += 4;
                while(cmdCount--) {
                    VIASETREG(offset, *bp++);
                    offset +=4;
                }
                break;
 
            case INV_AGPHeader4:
                offset = *bp & 0xFFFF;
                cmdCount = *(bp+1);
                bp += 4;
                while(cmdCount--)
                    VIASETREG(offset, *bp++);
                break;

            case INV_AGPHeader5:
                cmdCount = *(bp+1);
                bp += 4;
                while(cmdCount--) {
                    VIASETREG(*bp & 0xFFFF, *(bp+1));
                    bp +=2;
                }
                break;
                
            case INV_AGPHeader6:
                /* FIXME. Never reached here, AGP Header 6 is not used */
                bp++;
                break;
                
            case INV_AGPHeader7:
                cmdCount = *(bp+1);
                bp += 4;
                while(cmdCount--) {
                    offset = *bp++ & 0xFFFF;
                    VIASETREG(offset, *bp++);
                }
                break;
  
             default:
                /* FIXME. No regular path treament, for example of dummy commands, 
                   Just skip to the following 32 bits position */
                bp++;
                break;
        }
        
    }
    buf->pos = 0;
    buf->mode = 0;
    buf->has3dState = FALSE;
}

int
viaSetupCBuffer(ScrnInfoPtr pScrn, ViaCommandBuffer * buf, unsigned size)
{
#ifdef XF86DRI
    VIAPtr pVia = VIAPTR(pScrn);
#endif

    buf->pScrn = pScrn;

#if VIA_APPLY_AGP_BRANCH
    if( pVia->directRenderingEnabled &&
        pVia->umdBranchQuery.branch_buf_enabled &&
        viaRequestBranchAgpBuffer(pVia)) {
        /* The usermode cmd buffer size in unit of CARD32 */
        buf->bufSize = ((size == 0) ? \
            pVia->umdBranchBuf.size : min(pVia->umdBranchBuf.size, size))>> 2;
        buf->buf = (CARD32 *) pVia->umdBranchBuf.user_virtual;
    }  /* Fall back to normal AGP cmd buffer */
    else
#endif

    {
        buf->bufSize = ((size == 0) ? \
            VIA_CMDBUFSIZE : min(VIA_CMDBUFSIZE, size)) >> 2;
        buf->buf = (CARD32 *) xcalloc(buf->bufSize, sizeof(CARD32));
    }

    if (!buf->buf)
        return BadAlloc;
    buf->waitFlags = 0;
    buf->pos = 0;
    buf->mode = 0;
    buf->header_start = 0;
    buf->rindex = 0;
    buf->has3dState = FALSE;
    buf->needFire=FALSE;

    switch (pVia->ChipId) {
        case PCI_CHIP_VT3336:
        case PCI_CHIP_VT3364:
            buf->flushFunc = viaFlushPCI_H5;
            buf->beginFunc = viaBeginHeaderAgp_H5;
            buf->finishFunc = viaFinishHeaderAgp_H5;
#ifdef XF86DRI
            if (pVia->directRenderingEnabled) {
                buf->flushFunc = viaFlushDRIEnabled_H5;
            }
#endif
            break;

        case PCI_CHIP_VT3353:
        case PCI_CHIP_VT3409:
            buf->flushFunc = viaFlushPCI_H5;
            buf->beginFunc = viaBeginHeaderAgp_H5;
            buf->finishFunc = viaFinishHeaderAgp_H5;
#ifdef XF86DRI
            if (pVia->directRenderingEnabled) {
                buf->flushFunc = viaFlushDRIEnabled_H5;
            }
#endif
            break;

        default:
            buf->flushFunc = viaFlushPCI;
            buf->beginFunc = viaBeginHeaderAgp;
            buf->finishFunc = viaFinishHeaderAgp;
#ifdef XF86DRI
            if (pVia->directRenderingEnabled) {
                buf->flushFunc = viaFlushDRIEnabled;
            }
#endif
            break;
    }
						
    return Success;
}

void
viaTearDownCBuffer(ViaCommandBuffer * buf)
{
    ScrnInfoPtr pScrn = buf->pScrn;
#ifdef XF86DRI
    VIAPtr pVia = VIAPTR(pScrn);
#endif

#if VIA_APPLY_AGP_BRANCH
    if(pVia->directRenderingEnabled && 
       pVia->umdBranchQuery.branch_buf_enabled) {
        pVia->umdBranchBuf.cmd_size = buf->pos;
        viaFlushBranchAgpBuffer(pVia);
        buf->buf = NULL;
        buf->bufSize = 0;
    }
#endif

    if (buf && buf->buf)
        xfree(buf->buf);
    buf->buf = NULL;
}


/*
 * Wait for acceleration engines idle. An expensive way to sync.
 */
 
 
