 /*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include <sys/ioctl.h>
#include <unistd.h>
#include <errno.h>                  /* For Flag "EPERM" */
#include "via_driver.h"
#include "xf86_OSproc.h"
#include "xf86Priv.h"
#include "xf86PciInfo.h"
#include "xf86Pci.h"
#include "GL/glxtokens.h"
#include "via_dri.h"
#include "xf86drm.h"
#include "xf86drmVIA.h"
#include "via_drm.h"
#include "xf86.h"
#include "via_rotate.h"
#include "agpctl.h"
#include "via_common.h"

#ifdef DRM_SAMM_VIA
drm_context_t samm_ctx_hdl = 0;
#endif

extern void GlxSetVisualConfigs(
    int nconfigs,
    __GLXvisualConfig *configs,
    void **configprivs
);

#define H2_AGP_TEXTURE_ENABLE 1
#define H5_AGP_TEXTURE_ENABLE 1
#define H6_PCIE_TEXTURE_ENABLE 1

#define VIDEO	0 
#define AGP		1
#define AGP_PAGE_SIZE 4096
/*agp tex*/
#define AGP_PAGES 4096
#define AGP_TEXBUF_PAGES_H5 0x3000
#define AGP_TEX_SIZE_H5 (AGP_PAGE_SIZE * AGP_TEXBUF_PAGES_H5)/*AGP_TEX_SIZE_H5 is used by H5*/
#define AGP_TEXBUF_PAGES_H2 0x6000
#define AGP_TEX_SIZE_H2 (AGP_PAGE_SIZE * AGP_TEXBUF_PAGES_H2)/*AGP_TEX_SIZE_H2 is used by H2*/
#define AGP_CMDBUF_PAGES 4096
#define AGP_CMDBUF_SIZE (AGP_PAGE_SIZE * AGP_CMDBUF_PAGES)/*AGP_CMDBUF_SIZE is used by H2,H5&H6*/

#define RING_BUFFER_INIT_FLAG 1
#define RING_BUFFER_CLEANUP_FLAG 2

#if H5_AGP_TEXTURE_ENABLE
#define AGP_SIZE_H5 (AGP_CMDBUF_SIZE + AGP_TEX_SIZE_H5)/*AGP_SIZE_H5 is used by H5*/
#else
#define AGP_SIZE_H5 (AGP_CMDBUF_SIZE)
#endif

#if H2_AGP_TEXTURE_ENABLE
#define AGP_SIZE_H2 (AGP_CMDBUF_SIZE + AGP_TEX_SIZE_H2)/*AGP_SIZE_H2 is used by H2*/
#else
#define AGP_SIZE_H2 (AGP_CMDBUF_SIZE)
#endif

#if H6_PCIE_TEXTURE_ENABLE
#define PCIE_TEX_SIZE (AGP_PAGE_SIZE * 0x3000)/*default 48M*/
#endif

#define VIAH2KernelDriverName "via"
#define VIAH2ClientDriverName  "via_unichrome"
#define VIAH5KernelDriverName "via_chrome9"
#define VIAH5ClientDriverName  "via_chrome9"

int test_alloc_FB(ScreenPtr pScreen, VIAPtr pVia, int Size);
int test_alloc_AGP(ScreenPtr pScreen, VIAPtr pVia, int Size);
int VIAMapAGP(ScreenPtr pScreen, VIAPtr pVia);
int VIAS3GFinishScreenInit(ScreenPtr pScreen);
static Bool VIAInitVisualConfigs(ScreenPtr pScreen);
Bool VIASet3DSyncInfoParas(ScreenPtr pScreen);
static Bool VIADRIAgpInit(ScreenPtr pScreen, VIAPtr pVia);
static Bool VIADRIPciInit(ScreenPtr pScreen, VIAPtr pVia);
Bool VIADRIFBInit(ScreenPtr pScreen, VIAPtr pVia);
static Bool VIADRIKernelInit(ScreenPtr pScreen, VIAPtr pVia);
static Bool VIADRIMapInit(ScreenPtr pScreen, VIAPtr pVia);

static Bool VIACreateContext(ScreenPtr pScreen, VisualPtr visual, 
                   drmContext hwContext, void *pVisualConfigPriv,
                   DRIContextType contextStore);
static void VIADestroyContext(ScreenPtr pScreen, drmContext hwContext,
                   DRIContextType contextStore);
static void VIADRISwapContext(ScreenPtr pScreen, DRISyncType syncType, 
                   DRIContextType readContextType, 
                   void *readContextStore,
                   DRIContextType writeContextType, 
                   void *writeContextStore);
static void VIADRIInitBuffers(WindowPtr pWin, RegionPtr prgn, CARD32 index);
static void VIADRIMoveBuffers(WindowPtr pParent, DDXPointRec ptOldOrg, 
                   RegionPtr prgnSrc, CARD32 index);
/*for H5/H6*/
static int VIAInitS3gConf(ScreenPtr);
static int VIAReserveFBBuffer(ScreenPtr);
static int S3GInitVisualConfigs(ScreenPtr);


Bool 
VIASet3DSyncInfoParas(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    VIADISP3DSCALCTRLPtr p3DSCALCTRLInfo = NULL;
 
    if (pVia->pBIOSInfo->IGA1SettingInfo.IsDISP3DScaling)
    {
        p3DSCALCTRLInfo =&(pVia->pBIOSInfo->IGA1SettingInfo.DISP3DSCALCTRLInfo); 
        p3DSCALCTRLInfo->drmFD = pVia->drmFD;
        p3DSCALCTRLInfo->xcontext = DRIGetContext(pScreen);
        p3DSCALCTRLInfo->plock = (drmLock *)&(((XF86DRISAREARec *)DRIGetSAREAPrivate(pScreen))->lock);
        p3DSCALCTRLInfo->getShareArea = TRUE;
    }
    if(pVia->pBIOSInfo->IGA2SettingInfo.IsDISP3DScaling)
    {
        p3DSCALCTRLInfo =&(pVia->pBIOSInfo->IGA2SettingInfo.DISP3DSCALCTRLInfo); 
        p3DSCALCTRLInfo->drmFD = pVia->drmFD;
        p3DSCALCTRLInfo->xcontext = DRIGetContext(pScreen);
        p3DSCALCTRLInfo->plock = (drmLock *)&(((XF86DRISAREARec *)DRIGetSAREAPrivate(pScreen))->lock);
        p3DSCALCTRLInfo->getShareArea = TRUE;
    }     
    return TRUE;
}

static Bool 
VIADRIAgpInit(ScreenPtr pScreen, VIAPtr pVia)
{
    unsigned long  agp_phys;
    VIADRIPtr pVIADRI;
    DRIInfoPtr pDRIInfo;
    unsigned long AGPSize;
    pDRIInfo = pVia->pDRIInfo;
    pVIADRI = pDRIInfo->devPrivate;
    pVia->agpSize = 0;
    
/*For AMD64*/
#ifdef __x86_64__
    return FALSE;
#endif

    if (drmAgpAcquire(pVia->drmFD) < 0) {
        xf86DrvMsg(pScreen->myNum, X_ERROR, "[drm] drmAgpAcquire failed\n");
        return FALSE;
    }

    if (drmAgpEnable(pVia->drmFD, drmAgpGetMode(pVia->drmFD)&~0x0) < 0) {
         xf86DrvMsg(pScreen->myNum, X_ERROR, "[drm] drmAgpEnable failed\n");
        return FALSE;
    }
    
    xf86DrvMsg(pScreen->myNum, X_INFO, "[drm] drmAgpEnabled succeeded\n");

    if(H5_UMA_CHIPID)
        AGPSize = min(drmAgpSize(pVia->drmFD),AGP_SIZE_H5);
    else
        AGPSize = min(drmAgpSize(pVia->drmFD),AGP_SIZE_H2);
	
    if (drmAgpAlloc(pVia->drmFD, AGPSize, 0, &agp_phys, &pVia->agpHandle) < 0) {
        xf86DrvMsg(pScreen->myNum, X_ERROR,
                 "[drm] drmAgpAlloc failed\n");
        drmAgpRelease(pVia->drmFD);
        return FALSE;
    }
   
    if (drmAgpBind(pVia->drmFD, pVia->agpHandle, 0) < 0) {
        xf86DrvMsg(pScreen->myNum, X_ERROR,
                 "[drm] drmAgpBind failed\n");
        drmAgpFree(pVia->drmFD, pVia->agpHandle);
        drmAgpRelease(pVia->drmFD);

        return FALSE;
    }

    pVia->agpSize = AGPSize;
    pVia->agpAddr = drmAgpBase(pVia->drmFD);
    DEBUG(xf86DrvMsg(pScreen->myNum, X_INFO,
                 "[drm] agpAddr = 0x%08lx\n",pVia->agpAddr));
		 
    /*h5s1/H6 needn't share agp handle, and addmap drm buf is in VIAMapAGP()*/
    if(!(H5_UMA_CHIPID||H6_UMA_CHIPID))
    {
        pVIADRI->agp.size = pVia->agpSize;
        if (drmAddMap(pVia->drmFD, (drmHandle)0,
                     pVIADRI->agp.size, DRM_AGP, 0, 
                     &pVIADRI->agp.handle) < 0) {
    	xf86DrvMsg(pScreen->myNum, X_ERROR,
    	    "[drm] Failed to map public agp area\n");
            pVIADRI->agp.size = 0;
            return FALSE;
        }  
        /* Map AGP from kernel to Xserver - Not really needed */
        drmMap(pVia->drmFD, pVIADRI->agp.handle,pVIADRI->agp.size,
            (drmAddressPtr)&pVia->agpMappedAddr);
        
        /*agp flush drm*/
        pVia->agp_handle=pVIADRI->agp.handle;
        /* For H2 AGP chips: agpTexBaseOffset is the agp_handle returned by DrmAddMap() */
        pVia->agpTexBaseOffset = pVIADRI->agp.handle;

        if(pVIADRI->agp.size > AGP_CMDBUF_SIZE)
            drmVIAAgpInit(pVia->drmFD, AGP_CMDBUF_SIZE, pVIADRI->agp.size - AGP_CMDBUF_SIZE);
            /*Only let drm to handle AGP texture buffer*/
    }
    
    DEBUG(xf86DrvMsg(pScreen->myNum, X_INFO, 
                "[drm] agpBase = 0x%08lx\n", (unsigned long)pVia->agpBase));
    DEBUG(xf86DrvMsg(pScreen->myNum, X_INFO, 
                "[drm] agpAddr = 0x%08lx\n", pVia->agpAddr));
    DEBUG(xf86DrvMsg(pScreen->myNum, X_INFO, 
                "[drm] agpSize = 0x%08x\n", pVia->agpSize));
    DEBUG(xf86DrvMsg(pScreen->myNum, X_INFO, 
                "[drm] agp physical addr = 0x%08lx\n", agp_phys));
    return TRUE;
  
}

Bool 
VIADRIFBInit(ScreenPtr pScreen, VIAPtr pVia)
{   
    ScrnInfoPtr     pScrn;
    int FBSize = pVia->FBFreeEnd - pVia->FBFreeStart;
    int FBOffset = pVia->FBFreeStart; 
    VIADRIPtr pVIADRI = pVia->pDRIInfo->devPrivate;
    pVIADRI->fbOffset = FBOffset;
    pVIADRI->fbSize = pVia->videoRambytes;
    pScrn = xf86Screens[pScreen->myNum];
   
#ifdef DRM_SAMM_VIA
  if (!(pVia->IsSecondary)) {
#endif
    if(H5_UMA_CHIPID||H6_UMA_CHIPID) /*for H5s1/H6_UMA*/
    {
       if(!(VIAReserveFBBuffer(pScreen)))
        	return FALSE;
        
        return TRUE;
    }
    else
    {
        if(FBSize < 0) {
            xf86DrvMsg(pScreen->myNum, X_ERROR, "Allocate mem for H2 DRM failed, No enough MEM\n");
		    return FALSE;
        }

        if (FBSize < pVia->Bpl) {
            xf86DrvMsg(pScreen->myNum, X_ERROR, 
                       "[drm] H2 no DRM framebuffer heap available.\n"
                       "[drm] Please increase the frame buffer\n"
                       "[drm] memory area in the BIOS. Disabling DRI.\n");
		    return FALSE;
	    }

        if (drmVIAFBInit(pVia->drmFD, FBOffset, FBSize) < 0) {
            xf86DrvMsg(pScreen->myNum, X_ERROR,"[drm] failed to init frame buffer area\n");
            return FALSE;		
        }
        else {
    	    DEBUG(xf86DrvMsg(pScreen->myNum, X_INFO,"[drm] FBFreeStart= 0x%08x FBFreeEnd= \
		    0x%08x FBSize= 0x%08x\n", pVia->FBFreeStart, pVia->FBFreeEnd, FBSize));
            return TRUE;	
        }
    }
#ifdef DRM_SAMM_VIA
  } 
#endif
    return TRUE;	
}

static Bool 
VIADRIPciInit(ScreenPtr pScreen, VIAPtr pVia)
{
    return TRUE;	
}

static Bool 
viaDoInitVisual(ScreenPtr pScreen, int * numConfigs, __GLXvisualConfig *pConfigs, 
                VIAConfigPrivPtr pVIAConfigs, VIAConfigPrivPtr *pVIAConfigPtrs)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    int i, db, stencil, accum;

    /*for visual config */
    *numConfigs = 16;
	    
    if (!(pConfigs = (__GLXvisualConfig*)xcalloc(sizeof(__GLXvisualConfig),
					   *numConfigs)))
        return FALSE;
	
    if (!(pVIAConfigs = (VIAConfigPrivPtr)xcalloc(sizeof(VIAConfigPrivRec),
					    *numConfigs))) {
        xfree(pConfigs);
        return FALSE;
    }

    if (!(pVIAConfigPtrs = (VIAConfigPrivPtr*)xcalloc(sizeof(VIAConfigPrivPtr),
	  				  *numConfigs))) {
        xfree(pConfigs);
        xfree(pVIAConfigs);
        return FALSE;
    }
	
    for ( i=0; i< (*numConfigs); i++ ) {
        pVIAConfigPtrs[i] = &pVIAConfigs[i];
    }

    i = 0;
    for (accum = 0; accum <= 1; accum++) {
            /*for visual config */
            for (stencil = 0; stencil <= 3; stencil++) {
                    for (db = 0; db <= 1; db++) {
                            pConfigs[i].vid = -1;
                            pConfigs[i].class = -1;
                            pConfigs[i].rgba = TRUE;

                            if( 16 == pScrn->bitsPerPixel ) {
                                    pConfigs[i].redSize = -1;
                                    pConfigs[i].greenSize = -1;
                                    pConfigs[i].blueSize = -1;
                                    pConfigs[i].redMask = -1;
                                    pConfigs[i].greenMask = -1;
                                    pConfigs[i].blueMask = -1;
                                    pConfigs[i].alphaMask = 0;
                            }
                            else if( 32 == pScrn->bitsPerPixel ) {
                                    pConfigs[i].redSize = 8;
                                    pConfigs[i].greenSize = 8;
                                    pConfigs[i].blueSize = 8;
                                    pConfigs[i].alphaSize = 8;
                                    pConfigs[i].redMask = 0x00FF0000;
                                    pConfigs[i].greenMask = 0x0000FF00;
                                    pConfigs[i].blueMask = 0x000000FF;
                                    pConfigs[i].alphaMask = 0xFF000000;
                            }

                            if (accum) {
                                    pConfigs[i].accumRedSize = 16;
                                    pConfigs[i].accumGreenSize = 16;
                                    pConfigs[i].accumBlueSize = 16;
                                    pConfigs[i].accumAlphaSize = 16;
                            }
                            else {
                                    pConfigs[i].accumRedSize = 0;
                                    pConfigs[i].accumGreenSize = 0;
                                    pConfigs[i].accumBlueSize = 0;
                                    pConfigs[i].accumAlphaSize = 0;
                            }

                            if (!db)
                                    pConfigs[i].doubleBuffer = TRUE;
                            else
                                    pConfigs[i].doubleBuffer = FALSE;

                            pConfigs[i].stereo = FALSE;
                            if( 16 == pScrn->bitsPerPixel) {
                                    pConfigs[i].bufferSize = -1;
                            }
                            else if (32 == pScrn->bitsPerPixel) {
                                    pConfigs[i].bufferSize = 32; 
                            } 

                            switch (stencil) {
                                    case 0:
                                            pConfigs[i].depthSize = 0;
                                            pConfigs[i].stencilSize = 0;
                                            break;
                                    case 1:
                                            if( 16 == pScrn->bitsPerPixel) {
                                                    pConfigs[i].depthSize = 16;
                                            }
                                            else if (32 == pScrn->bitsPerPixel) {
                                                    pConfigs[i].depthSize = 24;
                                            }
                                            pConfigs[i].stencilSize = 0;
                                            break;
                                    case 2:
                                            pConfigs[i].depthSize = 32;
                                            pConfigs[i].stencilSize = 0;
                                            break;
                                    case 3:
                                            pConfigs[i].depthSize = 24;
                                            pConfigs[i].stencilSize = 8;
                                            break;
                            }

                            pConfigs[i].auxBuffers = 0;
                            pConfigs[i].level = 0;
                            pConfigs[i].visualRating = GLX_NONE_EXT;
                            pConfigs[i].transparentPixel = 0;
                            pConfigs[i].transparentRed = 0;
                            pConfigs[i].transparentGreen = 0;
                            pConfigs[i].transparentBlue = 0;
                            pConfigs[i].transparentAlpha = 0;
                            pConfigs[i].transparentIndex = 0;
                            i++;
                    }
            }
    }

    if (i != (*numConfigs) ) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                   "[dri] Incorrect initialization of visuals.  Disabling DRI.\n");
        return FALSE;
    }

    return TRUE;
}

static Bool
VIAInitVisualConfigs(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    int numConfigs = 0;
    __GLXvisualConfig *pConfigs = NULL;
    VIAConfigPrivPtr pVIAConfigs = NULL;
    VIAConfigPrivPtr *pVIAConfigPtrs = NULL;

    switch (pScrn->bitsPerPixel) {
	case 8:
	case 24:
	    break;

	case 16:
	/* add to expose RGBA visual configs for 32 bpp*/
	case 32:
        /*Diff by param. "pScrn->bitsPerPixel" */
	    viaDoInitVisual(pScreen, &numConfigs, pConfigs, pVIAConfigs, pVIAConfigPtrs);    
	break;
    }
    
    pVia->numVisualConfigs = numConfigs;
    pVia->pVisualConfigs = pConfigs;
    pVia->pVisualConfigsPriv = pVIAConfigs;

    GlxSetVisualConfigs(numConfigs, pConfigs, (void**)pVIAConfigPtrs);

    return TRUE;
}

/*initialize AGP ring buffer in 2D driver*/
void 
VIADRIRingBufferCleanup(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    int i = RING_BUFFER_CLEANUP_FLAG;
    drm_via_dma_init_t ringBufInit;
    xf86DrvMsg(pScrn->scrnIndex, X_INFO, 
		   "[drm] Cleaning up DMA ring-buffer.\n");

    if (!pVia->IsPCI)
    {
#ifdef DRM_SAMM_VIA
      if ( (pVia->drmSammEnabled) && (1 == pScrn->scrnIndex)) {
	    if(H5_UMA_CHIPID || H6_UMA_CHIPID)
	    {
  	      if(ioctl(pVia->drmFD, DRM_IOCTL_VIA_CHROME9_DMA_INIT, &i))
		      xf86DrvMsg(pScrn->scrnIndex, X_WARNING, 
		       "[drm] Failed to clean up DMA ring-buffer: %d\n", errno);
       
 	    }
	    else
	    {
  	  	    if(!pVia->useAgp)
			    return ;
		    ringBufInit.func = VIA_CLEANUP_DMA;
		    if (drmCommandWrite(pVia->drmFD, DRM_VIA_DMA_INIT, &ringBufInit,
				    sizeof(ringBufInit))) {
	    	    xf86DrvMsg(pScrn->scrnIndex, X_WARNING, 
			               "[drm] Failed to clean up DMA ring-buffer: %d\n", errno);
            }
        }
        return;
    }
            
    if ((pVia->drmSammEnabled) && (0 == pScrn->scrnIndex)) {
         return;
    }
#endif
	    if(H5_UMA_CHIPID || H6_UMA_CHIPID)
	    {
  	      if(ioctl(pVia->drmFD, DRM_IOCTL_VIA_CHROME9_DMA_INIT, &i))
		      xf86DrvMsg(pScrn->scrnIndex, X_WARNING, 
		       "[drm] Failed to clean up DMA ring-buffer: %d\n", errno);
       
 	    }
	    else
	    {
  	  	    if(!pVia->useAgp)
			    return ;
		    ringBufInit.func = VIA_CLEANUP_DMA;
		    if (drmCommandWrite(pVia->drmFD, DRM_VIA_DMA_INIT, &ringBufInit,
				    sizeof(ringBufInit))) {
	    	    xf86DrvMsg(pScrn->scrnIndex, X_WARNING, 
			               "[drm] Failed to clean up DMA ring-buffer: %d\n", errno);
            }
        }
    }

    return ;
}

Bool 
VIADRIRingBufferInit(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    int i = RING_BUFFER_INIT_FLAG;

    if (!pVia->IsPCI)
    {
        if(H5_UMA_CHIPID||H6_UMA_CHIPID)
        {
            if(0 == ioctl(pVia->drmFD, DRM_IOCTL_VIA_CHROME9_DMA_INIT,&i))
	    	return TRUE;            
        }
        else
        {
            drm_via_dma_init_t ringBufInit,init;
            /*
            * Check whether AGP DMA has been initialized.
            */
            memset(&init, 0, sizeof(init));
            memset(&ringBufInit, 0, sizeof(ringBufInit));
            init.func = VIA_DMA_INITIALIZED;

            if(0 == drmCommandWrite(pVia->drmFD, DRM_VIA_DMA_INIT,&init, sizeof(init)))
            return TRUE;

#ifdef DRM_SAMM_VIA
            if ( (pVia->drmSammEnabled) && (1 == pScrn->scrnIndex)) {
                ringBufInit.reg_pause_addr = 0x418;
                ringBufInit.offset = 0x0;
                ringBufInit.size = AGP_CMDBUF_SIZE;

                ringBufInit.func = VIA_INIT_DMA;
                if (drmCommandWrite(pVia->drmFD, DRM_VIA_DMA_INIT, &ringBufInit,
                sizeof(ringBufInit))) {
                xf86DrvMsg(pScrn->scrnIndex, X_ERROR, 
                "[drm] Failed to initialize DMA ring-buffer: %d\n", errno);
                return FALSE;
                }

                return TRUE;
            }
             
            if ( (pVia->drmSammEnabled) && (0 == pScrn->scrnIndex)) {
                return FALSE;
            }
#endif

            ringBufInit.reg_pause_addr = 0x418;
            ringBufInit.offset = 0x0;
            ringBufInit.size = AGP_CMDBUF_SIZE;

            ringBufInit.func = VIA_INIT_DMA;
            if (drmCommandWrite(pVia->drmFD, DRM_VIA_DMA_INIT, &ringBufInit,
            sizeof(ringBufInit))) {
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, 
            "[drm] Failed to initialize DMA ring-buffer: %d\n", errno);
            return FALSE;
            }
            DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, 
            "[drm] Initialized AGP ring-buffer, size 0x%lx at AGP offset 0x%lx.\n",
            ringBufInit.size, ringBufInit.offset));
            return TRUE;		   
        }    
    }

    return FALSE;
}

Bool 
VIACheckDrmAlreadyInit(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
#ifdef DRM_SAMM_VIA
    VIAPtr pVia0 = VIAPTR(xf86Screens[0]);
#endif
    int drmfd, initjudge=0;
    char *drmDriverName, *clientDriverName;
    char busstring[64];

#ifdef DRM_SAMM_VIA
    if (pVia->pBIOSInfo->SAMM) {

        if(pVia0->drmFD)
        {
            pVia0->drmSammEnabled = TRUE;
            pVia->drmFD = pVia0->drmFD;
            return TRUE;
        }
    }
#endif

    if (!xf86LoaderCheckSymbol("drmAvailable"))
        return FALSE;

    sprintf(busstring, "PCI:%d:%d:%d",
#if XSERVER_LIBPCIACCESS
		((pVia->PciInfo->domain << 8) | pVia->PciInfo->bus),
		pVia->PciInfo->dev, pVia->PciInfo->func
#else
		((pciConfigPtr)pVia->PciInfo->thisCard)->busnum,
		((pciConfigPtr)pVia->PciInfo->thisCard)->devnum,
		((pciConfigPtr)pVia->PciInfo->thisCard)->funcnum
#endif
	);    
    
    if(H5_UMA_CHIPID||H6_UMA_CHIPID) 
    {
        drmDriverName = VIAH5KernelDriverName;
        clientDriverName = VIAH5ClientDriverName;
    }
    else
    {
        drmDriverName = VIAH2KernelDriverName;
        clientDriverName = VIAH2ClientDriverName;
    }
    if((drmfd=drmOpen(drmDriverName, busstring)) == -1)
    {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "DRM open error \n");
        return FALSE;
    }

    if (!drmCommandRead(drmfd, DRM_VIA_INIT_JUDGE, &initjudge, sizeof(int)))
    {
        if(initjudge == 1)
        {
            drmClose(drmfd);
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "DRM initjudge==1 \n");
            return TRUE;          
        }
    }
    
    drmClose(drmfd);
    return FALSE;
}

Bool 
VIADRIScreenInit(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    VIAPtr pVia0 = VIAPTR(xf86Screens[0]);
    DRIInfoPtr pDRIInfo;
    VIADRIPtr pVIADRI;
    /*set the default value,becuase the default valuse setted by compiler is zero, but we need 1*/
    pVia->IsPCI = TRUE;
    char *busstring, *drmDriverName, *clientDriverName;
    
    if (H5_UMA_CHIPID||H6_UMA_CHIPID) {
        switch(pVia->ChipId)
        {
            case PCI_CHIP_VT3336:
            case PCI_CHIP_VT3364:
                xf86DrvMsg(pScreen->myNum, X_INFO,
                "[drm] Detect H5s1 chipset: %04x  chipid: %04x\n",
                pVia->Chipset, pVia->ChipId);
                break;
            case PCI_CHIP_VT3353:
            case PCI_CHIP_VT3409:
                xf86DrvMsg(pScreen->myNum, X_INFO,
                "[drm] Detect H6 chipset: %04x  chipid: %04x\n",
                pVia->Chipset, pVia->ChipId);
                break;
            default:
                break;
        }

        drmDriverName = VIAH5KernelDriverName;
        clientDriverName = VIAH5ClientDriverName;
    }
    else {  /* H2 */
        drmDriverName = VIAH2KernelDriverName;
        clientDriverName = VIAH2ClientDriverName;
                
        xf86DrvMsg(pScreen->myNum, X_INFO,
                  "[drm] Detect H2 chipset: %04x  chipid: %04x\n",
                  pVia->Chipset, pVia->ChipId);
    }

    busstring = xalloc(64);
    sprintf(busstring, "PCI:%d:%d:%d",
#if XSERVER_LIBPCIACCESS
        ((pVia->PciInfo->domain << 8) | pVia->PciInfo->bus),
        pVia->PciInfo->dev, pVia->PciInfo->func
#else
        ((pciConfigPtr)pVia->PciInfo->thisCard)->busnum,
        ((pciConfigPtr)pVia->PciInfo->thisCard)->devnum,
        ((pciConfigPtr)pVia->PciInfo->thisCard)->funcnum
#endif
    );
		
    /* Check that the GLX, DRI, and DRM modules have been loaded by testing
    * for canonical symbols in each module. */
    if (!xf86LoaderCheckSymbol("GlxSetVisualConfigs")) return FALSE;
    if (!xf86LoaderCheckSymbol("DRIScreenInit"))       return FALSE;
    if (!xf86LoaderCheckSymbol("drmAvailable"))        return FALSE;
    if (!xf86LoaderCheckSymbol("DRIQueryVersion")) {
	xf86DrvMsg(pScreen->myNum, X_ERROR,
                 "[dri] VIADRIScreenInit failed (libdri.a too old)\n");
        return FALSE;
    }

    /* if drm modules already initialized, it means that one x server has been run in dri,
        all the other x will be in-dri. */
#ifdef DRM_SAMM_VIA
    if(!pVia->IsSecondary){
#endif
        if (VIACheckDrmAlreadyInit(pScrn)) {
            return FALSE;
        }
#ifdef DRM_SAMM_VIA
    }
#endif

    /* Check the DRI version 
       change for get the libDRI version 5.0.0, so remove it first.*/
    if (!(H5_UMA_CHIPID||H6_UMA_CHIPID)) {
        int major, minor, patch;
        DRIQueryVersion(&major, &minor, &patch);
        if (major < 4 || minor < 0) {
            xf86DrvMsg(pScreen->myNum, X_ERROR,
                    "[dri] VIADRIScreenInit failed because of a version mismatch.\n"
                    "[dri] libDRI version is %d.%d.%d but version 4.0.x is needed.\n"
                    "[dri] Disabling DRI.\n",
                    major, minor, patch);
            return FALSE;
        }
    }
    else {
        /*for support H5/H6, now allocate additional record*/
        ((VIARec *)(pScrn->driverPrivate))->pScrnConf = xnfcalloc(sizeof(ScrnConfRec), 1);
        ((VIARec *)(pScrn->driverPrivate))->pDriConf = xnfcalloc(sizeof(DriConfRec), 1);
        ((VIARec *)(pScrn->driverPrivate))->pServerInfo = xnfcalloc(sizeof(ServerInfoRec), 1);
        ((VIARec *)(pScrn->driverPrivate))->pAddrConf = xnfcalloc(sizeof(AddrConfRec), 1);
        ((VIARec *)(pScrn->driverPrivate))->pPciConf = xnfcalloc(sizeof(PciConfRec), 1);
        ((VIARec *)(pScrn->driverPrivate))->pChipConf = xnfcalloc(sizeof(ChipConfRec), 1);
    }

    pDRIInfo = DRICreateInfoRec();
    
    if (!pDRIInfo) return FALSE;
    
    pVia->pDRIInfo = pDRIInfo;
    pDRIInfo->drmDriverName = drmDriverName;
    pDRIInfo->clientDriverName = clientDriverName;
    pDRIInfo->busIdString = busstring;
    pDRIInfo->ddxDriverMajorVersion = VIA_VERSION_MAJOR;
    pDRIInfo->ddxDriverMinorVersion = VIA_VERSION_MINOR;
    pDRIInfo->ddxDriverPatchVersion = PATCHLEVEL;
    pDRIInfo->frameBufferPhysicalAddress = (void *)pVia->FrameBufferBase;
    pDRIInfo->frameBufferSize = pVia->videoRambytes;    
    pDRIInfo->frameBufferStride = (pScrn->displayWidth *
					               pScrn->bitsPerPixel / 8);
    pDRIInfo->ddxDrawableTableEntry = VIA_MAX_DRAWABLES;

	pDRIInfo->maxDrawableTableEntry = min(SAREA_MAX_DRAWABLES, VIA_MAX_DRAWABLES);

    /* For now the mapping works by using a fixed size defined
    * in the SAREA header
    */
    if(!(H5_UMA_CHIPID||H6_UMA_CHIPID)) {
        if (sizeof(XF86DRISAREARec)+sizeof(VIASAREAPriv) > SAREA_MAX) {
    	    xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
	    		    "Data does not fit in SAREA\n");
	        return FALSE;
        }
    } else {
        if (sizeof(XF86DRISAREARec)+sizeof(VIA_CHROME9SAREAPriv) > SAREA_MAX) {
    	    xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
	    		    "Data does not fit in SAREA\n");
	        return FALSE;
        }
    }

    pDRIInfo->SAREASize = SAREA_MAX;

    if (!(pVIADRI = (VIADRIPtr)xcalloc(sizeof(VIADRIRec),1))) {
	    DRIDestroyInfoRec(pVia->pDRIInfo);
	    pVia->pDRIInfo = 0;
	    return FALSE;
    }
    /*for h5s1/H6_UMA, copy the s3g rec ptr from VIARec*/
    if (H5_UMA_CHIPID||H6_UMA_CHIPID) {
        pVIADRI->pServerInfo = pVia->pServerInfo;
    }
  
    pDRIInfo->devPrivate= pVIADRI;
    pDRIInfo->devPrivateSize = sizeof(VIADRIRec);
    pDRIInfo->contextSize = sizeof(VIADRIContextRec);
   
    pDRIInfo->CreateContext = VIACreateContext;
    pDRIInfo->DestroyContext = VIADestroyContext;
    pDRIInfo->SwapContext = VIADRISwapContext;
    pDRIInfo->InitBuffers = VIADRIInitBuffers;
    pDRIInfo->MoveBuffers = VIADRIMoveBuffers;
    pDRIInfo->bufferRequests = DRI_ALL_WINDOWS;

#ifdef DRM_SAMM_VIA
    if (!(pVia->pBIOSInfo->SAMM)) {
#endif
        if (!DRIScreenInit(pScreen, pDRIInfo, &pVia->drmFD)) {
              xf86DrvMsg(pScreen->myNum, X_ERROR,
                     "[dri] DRIScreenInit failed.  Disabling DRI.\n");
              xfree(pDRIInfo->devPrivate);
              pDRIInfo->devPrivate=0;
              DRIDestroyInfoRec(pVia->pDRIInfo);
              pVia->pDRIInfo=0;
              pVia->drmFD = -1;
              return FALSE;
        }

#ifdef DRM_SAMM_VIA
    }
    else {
        drmSetVersion saveSv, sv;
        Bool drmWasAvailable;
        drmVersionPtr drmlibv;
        int drmFD;
        int drmlibmajor, drmlibminor;
        const char *openBusID;
        int count;
        int err;

        char *busID;
        char *drmDriverName;
        
        pVia->drmSammEnabled = FALSE;
        pVia->IsPCI = TRUE;

        if(pVia0->drmFD) {
            pVia->drmFD = pVia0->drmFD;
            pVia->drmSammEnabled = TRUE;    /*Sync to the same as the Primary Screen*/
            return TRUE;
        }

        busID = xalloc(64);
        sprintf(busID, "PCI:%d:%d:%d",
#if XSERVER_LIBPCIACCESS
                        ((pVia->PciInfo->domain << 8) | pVia->PciInfo->bus),
                        pVia->PciInfo->dev, pVia->PciInfo->func
#else
                        ((pciConfigPtr)pVia->PciInfo->thisCard)->busnum,
                        ((pciConfigPtr)pVia->PciInfo->thisCard)->devnum,
                        ((pciConfigPtr)pVia->PciInfo->thisCard)->funcnum
#endif
               );

        if (H5_UMA_CHIPID||H6_UMA_CHIPID) {
            drmDriverName = VIAH5KernelDriverName;
        }
        else {
            drmDriverName = VIAH2KernelDriverName;
        }

        drmWasAvailable = drmAvailable();

        /* Check the DRM lib version.
         * drmGetLibVersion was not supported in version 1.0, so check for
         * symbol first to avoid possible crash or hang.
         */
        drmlibmajor = 1;
        drmlibminor = 0;
        if (xf86LoaderCheckSymbol("drmGetLibVersion")) {
                drmlibv = drmGetLibVersion(-1);
                if (drmlibv != NULL) {
                        drmlibmajor = drmlibv->version_major;
                        drmlibminor = drmlibv->version_minor;
                        drmFreeVersion(drmlibv);
                }
        }

        /* Check if the libdrm can handle falling back to loading based on name
         * if a busid string is passed.
         */
        openBusID = (drmlibmajor == 1 && drmlibminor >= 2) ? busID : NULL;

        drmFD = -1;
        sv.drm_di_major = 1;
        sv.drm_di_minor = 1;
        sv.drm_dd_major = -1;

        saveSv = sv;
        count = 10;
        while (count--) {
                if(!pVia0->drmFD)
                {
                        drmFD = drmOpen(drmDriverName, openBusID);
                }

                if (drmFD < 0) {
                        xf86DrvMsg( pScrn->scrnIndex, X_ERROR, "[drm] drmOpen failed.\n");
                        return FALSE;
                }

                err = drmSetInterfaceVersion(drmFD, &sv);

                if (err != -EPERM)
                        break;

                sv = saveSv;
                drmClose(drmFD);
                drmFD = -1;
                usleep(100000);
        }

        /*gen the samm + drm context*/
        err = drmCreateContext(drmFD, &samm_ctx_hdl);
        if(err) {
                xf86DrvMsg( pScrn->scrnIndex, X_ERROR, "[drm] drmCreateContext failed err = %d\n", err);
                return FALSE;
        }

        err = drmGetLock(drmFD, samm_ctx_hdl, 0);
        if(err) {
        xf86DrvMsg( pScrn->scrnIndex, X_ERROR, "[drm] drmGetLock failed\n");
        return FALSE;
        }

        if (drmFD <= 0) {
                xf86DrvMsg( pScrn->scrnIndex, X_ERROR, "[drm] DRM was busy with another master.\n");
        }

        if (!drmWasAvailable) {
                xf86DrvMsg( pScrn->scrnIndex, X_INFO, "[drm] loaded kernel module for \"%s\" driver.\n", drmDriverName);
        }

        if (err != 0) {
                sv.drm_di_major = 1;
                sv.drm_di_minor = 0;
        }

        xf86DrvMsg( pScrn->scrnIndex, X_INFO, "[drm] DRM interface version %d.%d\n",
                        sv.drm_di_major, sv.drm_di_minor);

        if (sv.drm_di_major == 1 && sv.drm_di_minor >= 1)
                err = 0;
        else
                err = drmSetBusid(drmFD, busID);

        if (err) {
                xf86DrvMsg( pScrn->scrnIndex, X_ERROR, "[drm] Could not set DRM device bus ID.\n");
        }

        if(!pVia0->drmFD)
        {
                pVia->drmFD = drmFD;
        }

        /* pass all the test, so now we can enable variable safely */        
        pVia->drmSammEnabled = TRUE;
    }
#endif

    if (H6_UMA_CHIPID) {
        /* We can only know whether PCIE is OK or not after init drm modules
         * PCIe will try to enable later
         * */
        pVia->IsPCI = TRUE;
    }
    else {
        pVia->IsPCI = !VIADRIAgpInit(pScreen, pVia);
        
        if (pVia->IsPCI) {
            VIADRIPciInit(pScreen, pVia);
            xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[dri] use pci.\n" );
        } else {
            xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[dri] use agp.\n" );
        }
    }
    
    if(!(H5_UMA_CHIPID||H6_UMA_CHIPID))
    {
        if (!(VIAInitVisualConfigs(pScreen))) 
        {
            VIADRICloseScreen(pScreen);
            return FALSE;
        }
    }      
    else /*for  h5s1/H6 UMA*/
    {
        if(!(S3GInitVisualConfigs(pScreen)))
        {
            VIADRICloseScreen(pScreen);
            return FALSE;
        }
    }
    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[dri] visual configs initialized.\n" );
  
    /* DRIScreenInit doesn't add all the common mappings.  Add additional mappings here. */
    if (!VIADRIMapInit(pScreen, pVia)) {
        VIADRICloseScreen(pScreen);
        return FALSE;
    }
    pVIADRI->regs.size = VIA_MMIO_REGSIZE;
    pVIADRI->regs.map = 0;
    pVIADRI->regs.handle = pVia->registerHandle;
    xf86DrvMsg(pScreen->myNum, X_INFO, "[drm] mmio Registers = 0x%08u\n",
	pVIADRI->regs.handle);
    
    pVIADRI->drixinerama = pVia->drixinerama;
    /*Tuxracer & VQ*/
    pVIADRI->VQEnable = pVia->VQEnable;

    /* Get the rotate degree */
    pVIADRI->RotateDegree = pVia->RotateDegree;

    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[dri] mmio maped.\n" );

    return TRUE;
}

void
VIADRICloseScreen(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);

    if(H5_UMA_CHIPID||H6_UMA_CHIPID)
    {
        drm_s3g_init_t init;
        /*call drm clean up ioctl before close screen*/
        DriConfPtr pDriConf = pVia->pDriConf;
        memset(&init, 0, sizeof(drm_s3g_init_t));
        init.func = S3G_CLEANUP;
        drmCommandWrite(pDriConf->drmFd, DRM_S3G_INIT, &init, sizeof(drm_s3g_init_t));
    }
    else
    {
        drm_via_init_t init;
        memset(&init, 0, sizeof(drm_via_init_t));
        init.func = VIA_CLEANUP_MAP;
        ioctl(pVia->drmFD, DRM_IOCTL_VIA_MAP_INIT, &init);        
    }

    if (!pVia->pBIOSInfo->SAMM) {
        DRICloseScreen(pScreen);
    }
#ifdef DRM_SAMM_VIA
    else {
        if (!pVia->IsSecondary) {
            if (!samm_ctx_hdl) {
                drmUnlock(pVia->drmFD, samm_ctx_hdl);
                drmDestroyContext(pVia->drmFD, samm_ctx_hdl);    
            }

            if (pVia->drmFD) {
                drmClose(pVia->drmFD);
                pVia->drmFD = 0;
            }
        }
    }
#endif

    if (pVia->pDRIInfo) {
        if (pVia->pDRIInfo->devPrivate) {
            xfree(pVia->pDRIInfo->devPrivate);
            pVia->pDRIInfo->devPrivate=0;
	    }
        DRIDestroyInfoRec(pVia->pDRIInfo);
        pVia->pDRIInfo=0;
    }
    
    if (pVia->pVisualConfigs) xfree(pVia->pVisualConfigs);
    if (pVia->pVisualConfigsPriv) xfree(pVia->pVisualConfigsPriv);
    if (pVia->agpSize) {
        xf86DrvMsg(pScreen->myNum, X_INFO, "[drm] Freeing agp memory\n");
        drmAgpFree(pVia->drmFD, pVia->agpHandle);
        xf86DrvMsg(pScreen->myNum, X_INFO, "[drm] Releasing agp module\n");
    	drmAgpRelease(pVia->drmFD);
    }
}

/* TODO: xserver receives driver's swapping event and do something
 *       according the data initialized in this function
 */
static Bool
VIACreateContext(ScreenPtr pScreen, VisualPtr visual, 
          drmContext hwContext, void *pVisualConfigPriv,
          DRIContextType contextStore)
{
    return TRUE;
}

static void
VIADestroyContext(ScreenPtr pScreen, drmContext hwContext, 
           DRIContextType contextStore)
{
}

Bool
VIADRIFinishScreenInit(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    VIADRIPtr pVIADRI;
    pVIADRI=(VIADRIPtr)pVia->pDRIInfo->devPrivate;

#ifdef DRM_SAMM_VIA
    if (!pVia->pBIOSInfo->SAMM)
    {
#endif
            pVia->pDRIInfo->driverSwapMethod = DRI_HIDE_X_CONTEXT;

            DRIFinishScreenInit(pScreen);

            if(H5_UMA_CHIPID||H6_UMA_CHIPID)
            {
                    VIAS3GFinishScreenInit(pScreen);
            }
            if (!VIADRIKernelInit(pScreen, pVia)) {
                    VIADRICloseScreen(pScreen);
                    return FALSE;
            }
            xf86DrvMsg(pScreen->myNum, X_INFO, "[dri] kernel data initilized.\n");

            /* set SAREA value */
            if(!(H5_UMA_CHIPID||H6_UMA_CHIPID)) {
                    VIASAREAPriv *saPriv;
                    saPriv=(VIASAREAPriv*)DRIGetSAREAPrivate(pScreen);
                    assert(saPriv);
                    memset(saPriv, 0, sizeof(*saPriv));
                    saPriv->ctx_owner = -1;

            } else {
                    VIA_CHROME9SAREAPriv *saPriv;
                    saPriv=(VIA_CHROME9SAREAPriv *)DRIGetSAREAPrivate(pScreen);
                    assert(saPriv);
                    memset(saPriv, 0, sizeof(*saPriv));
                    saPriv->ctx_owner = -1;
            }

    pVIADRI=(VIADRIPtr)pVia->pDRIInfo->devPrivate;
    pVIADRI->deviceID=pVia->Chipset;  
    pVIADRI->width=pScrn->displayWidth;
    /*pass the pitch to DRI directly*/
    pVIADRI->pitch = CMDISP_ALIGN_TO(pScrn->displayWidth * pScrn->bitsPerPixel / 8, 16);
    pVIADRI->height=pScrn->virtualY;
    /*For XV set Clipping Windows, Rotate for 90/270 degree, virtualX/virtualY is swapped*/
    if ((pVIADRI->RotateDegree == VIA_ROTATE_DEGREE_90) ||
        (pVIADRI->RotateDegree == VIA_ROTATE_DEGREE_270)) {
        pVIADRI->height = pScrn->virtualX;
    }

            pVIADRI->mem=pScrn->videoRam*1024;
            pVIADRI->bytesPerPixel= (pScrn->bitsPerPixel+7) / 8; 
            pVIADRI->sarea_priv_offset = sizeof(XF86DRISAREARec);
            /* TODO */
            pVIADRI->scrnX=pVIADRI->width;
            pVIADRI->scrnY=pVIADRI->height;

            /*for vt3336*/
            pVia->useAgp = 0;
            if(H5_UMA_CHIPID||H6_UMA_CHIPID)
            {
                    pVia->pDRIInfo->devPrivate= pVia->pServerInfo;
                    pVia->pDRIInfo->devPrivateSize = sizeof(ServerInfoRec);
                    pVia->pDRIInfo->contextSize = 256;
            }
            else
            {
                    /*initia AGP ring buffer in 2D driver*/
                    pVia->useAgp=VIADRIRingBufferInit(pScrn);
            }
#ifdef DRM_SAMM_VIA
    }
    else {
#ifdef DRM_SAMM_VIA
        if (!(pVia->IsSecondary)) {
#endif
            if(H5_UMA_CHIPID||H6_UMA_CHIPID)
            {
                    VIAS3GFinishScreenInit(pScreen);
            }
            if (!VIADRIKernelInit(pScreen, pVia)) {
                    VIADRICloseScreen(pScreen);
                    return FALSE;
            }
#ifdef DRM_SAMM_VIA
        }
#endif
            pVia->useAgp=VIADRIRingBufferInit(pScrn);
            return FALSE;
    }
#endif
    return TRUE;
}

static void
VIADRISwapContext(ScreenPtr pScreen, DRISyncType syncType, 
           DRIContextType oldContextType, void *oldContext,
           DRIContextType newContextType, void *newContext)
{
  /*ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
  VIAPtr pVia = VIAPTR(pScrn);
  */   
  return;
}

static void
VIADRIInitBuffers(WindowPtr pWin, RegionPtr prgn, CARD32 index)
{
  /*ScreenPtr pScreen = pWin->drawable.pScreen;
  ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
  VIAPtr pVia = VIAPTR(pScrn);
  */
  return;
}

static void
VIADRIMoveBuffers(WindowPtr pParent, DDXPointRec ptOldOrg, 
           RegionPtr prgnSrc, CARD32 index)
{
  /*ScreenPtr pScreen = pParent->drawable.pScreen;
  ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
  VIAPtr pVia = VIAPTR(pScrn);
  */
  return;
}

/* Initialize the kernel data structures. */
static Bool 
VIADRIKernelInit(ScreenPtr pScreen, VIAPtr pVia)
{
    drmVIAInit drmInfo;
    int ret;

    drmInfo.sarea_priv_offset   = sizeof(XF86DRISAREARec);
    drmInfo.fb_offset           = pVia->FrameBufferBase;
    drmInfo.mmio_offset         = pVia->registerHandle;

    if (pVia->IsPCI) {
        drmInfo.agpAddr = (CARD32)NULL;
    }
    else {
/*For AMD64*/
#ifndef __x86_64__
	drmInfo.agpAddr = (CARD32)pVia->agpAddr;
#else
	drmInfo.agpAddr = (CARD64)pVia->agpAddr;
#endif
    }

    /*agp flush drm*/
    drmInfo.agp_offset=(CARD32)pVia->agp_handle;
    if(H5_UMA_CHIPID||H6_UMA_CHIPID)/*drm init for H5/H6 UMA*/
    {
        DriConfPtr pDriConf = pVia->pDriConf;
        ChipConfPtr pChipConf = pVia->pChipConf;
        AddrConfPtr pAddrConf = pVia->pAddrConf;
        ServerInfoPtr pServerInfo = pVia->pServerInfo;
        
        drm_s3g_init_t init;
        {
            init.func = S3G_INIT;
            init.chip_index = S3G_INV;
            switch(pVia->ChipId)
            {
            case PCI_CHIP_VT3336:
            case PCI_CHIP_VT3364:
                init.chip_sub_index = CHIP_H5S1;
                break;
            case PCI_CHIP_VT3353:
            case PCI_CHIP_VT3409:
                init.chip_sub_index = CHIP_H6S2;
                break;
            
            default:
                break;			   			   	
            }
            init.garttable_size = 0x0;
            /*because h5s1 is AGP, but not PCIE, we don't need to create garttable*/
            init.garttable_offset = 0x0;  
            /*for h5s1, dma buffer is in agp*/
            init.dma_handle = pDriConf->dmaHandle;   
        }

        if (!H6_UMA_CHIPID) {
             /*tell drm how to init dma-agp*/
             /*if agp init failed, then use pci*/
             if(pVia->IsPCI)
                  init.agp_type = AGP_DISABLED;
             /*For H5(vt3336 and vt3364), the default agp type is ring buffer mode*/
             else
                 init.agp_type = AGP_RING_BUFFER;
             /*end*/
        }
        else {
             /*default choice*/
             init.agp_type = AGP_RING_BUFFER;
        }

        init.sarea_priv_offset = sizeof(XF86DRISAREARec);
        init.fb_cpp = pVia->Bpp;
        init.front_offset = 0;
        init.depth_offset = pDriConf->depthOffset;
        
        init.mmio_handle = pDriConf->mmioHandle;
        init.hostBlt_handle = (unsigned int)pVia->hostBltHandle;
       
        init.fb_handle = pDriConf->fbHandle;
        init.front_handle = pDriConf->frontHandle;
        init.back_handle = pDriConf->frontHandle;
        init.depth_handle = pDriConf->frontHandle;

        /* Set total available video memory for 3D apps,
         * because the meaning of fbFree in VIA and s3g is contrary to via
         */
        init.available_fb_size = pVia->FBFreeEnd - pVia->FBFreeStart;
        if( init.available_fb_size < 0) {
            init.available_fb_size = 0;
            xf86DrvMsg(pScreen->myNum, X_ERROR, "Allocate mem for H5/H6 DRM failed, No enough MEM\n");
        }

        init.fb_base_address = pAddrConf->fbPhyAddr;    
        init.back_offset = pVia->FBFreeStart;  /*give the region follow on-screen to 3d*/
        
        init.fb_tex_offset = pDriConf->fbTexOffset;
        init.fb_tex_size = pDriConf->fbTexSize;
        init.agp_tex_size = pDriConf->agpTexSize;
        init.agp_tex_handle = pDriConf->agpTexHandle;

        init.chip_agp = pChipConf->chipType;
        init.chip_sub_index = pServerInfo->chipSubIndex;
        init.chip_index = pServerInfo->chipIndex;

        init.usec_timeout = 50000; /*???this value should consult to clb driver*/
        /*now shadow is in video memory, so if agp init failed, shadow is still available*/
        init.shadow_size = pDriConf->shadowSize;
        init.shadow_handle = pDriConf->shadowHandle;


        if(H6_UMA_CHIPID)
        { 
            init.garttable_offset = init.back_offset;/*reserved for gart table*/
            init.garttable_offset = (init.garttable_offset + 4095)&(~4095);/*4k align*/
            init.DMA_size = AGP_CMDBUF_SIZE;
            #if H6_PCIE_TEXTURE_ENABLE
            pServerInfo->agpTexSize = pDriConf->agpTexSize = 
            init.agp_tex_size = PCIE_TEX_SIZE;
            #endif
            init.garttable_size = (init.DMA_size + init.agp_tex_size) >> 10;
            init.garttable_size = (init.garttable_size + 4095)&(~4095);
            init.available_fb_size -= (init.garttable_size + init.garttable_offset - init.back_offset);
            init.back_offset = init.garttable_offset + init.garttable_size;
        }
        else
        {
            if (pVia->IsPCI) 
            {
                init.DMA_size = 0x0;
            }
            else
            {
                init.DMA_phys_address = pDriConf->agpBase; ;  /*dma buffer is follow the shadow in AGP*/
                init.DMA_size = AGP_CMDBUF_SIZE;   /*the real size of ring buffer agp buffer size*/
                init.agp_tex_size = pDriConf->agpTexSize;
            }
        }
        
        ret = drmCommandWriteRead(pDriConf->drmFd, DRM_S3G_INIT, &init, sizeof(drm_s3g_init_t));
        if (ret < 0)  
        {
            xf86DrvMsg(pScreen->myNum, X_ERROR,"[drm] ERROR! S3GDRIKernelInit:failed to "
                       "initialize kernel module! (%d)\n", ret );
            return FALSE;
        }

        if(H6_UMA_CHIPID)
        {
            if((init.agp_type == AGP_RING_BUFFER)||(init.agp_type == AGP_DOUBLE_BUFFER)){
                pServerInfo->isAGP = pServerInfo->DriConf.dmaBufAGP = 
                pDriConf->dmaBufAGP = TRUE;
                pVia->IsPCI = FALSE;
                xf86DrvMsg(pScreen->myNum, X_INFO,"[drm] use pcie\n");
                /* addmap pcie tex buffer and get the handle */
                pVia->agpTexSize = init.agp_tex_size;
                ret = drmAddMap(pDriConf->drmFd, init.DMA_size, pVia->agpTexSize,
                    DRM_SCATTER_GATHER, 0, &pVia->agpTexHandle);

                if(ret < 0)
                {
                    /* Invalidate them */
                    pVia->agpTexHandle = 0;
                    pVia->agpTexMappedAddr = 0;
                    pVia->agpTexSize = 0;
                    xf86DrvMsg(pScreen->myNum, X_INFO,
                        "Add pcie memory error Branch buffer can't work well but it won't affect any other functions\n");
                }
                /* For H6 PCIe chips: agpTexBaseOffset is the size of DMA Ring buffer */
                pVia->agpTexBaseOffset = init.DMA_size;
            }/*PCIE path enable*/
            else
            {
                   pServerInfo->isAGP = pServerInfo->DriConf.dmaBufAGP = 
                   pDriConf->dmaBufAGP = FALSE;
                   pVia->IsPCI = TRUE;
                   pVia->agpTexHandle = 0;
                   pVia->agpTexMappedAddr = 0;
                   pVia->agpTexSize = 0;
                   xf86DrvMsg(pScreen->myNum, X_INFO,"[drm] use pci\n");
            }

            /* After getting the pcie tex buffer handle, we deliver it to dri data structure for 3D drv use */
            pServerInfo->agpTexSize = pVia->agpTexSize;
            pServerInfo->agpTexHandle = pVia->agpTexHandle;
        }

        if((H5_UMA_CHIPID || H6_UMA_CHIPID) && (pVia->agpTexHandle && pVia->agpTexSize))
        {
            /* Map Tex AGP/PCIE from kernel to Xserver - 
              It is really needed for EXA Texture AGP and AGP Branch buffer. 
            */
            ret = drmMap(pDriConf->drmFd, pVia->agpTexHandle,
            pVia->agpTexSize, &pVia->agpTexMappedAddr);
            if (ret < 0)
            {
                xf86DrvMsg(pScreen->myNum, X_INFO,
                "Map agp/pcie texture memory error, H6 chipset branch buffer mechanism won't work! \n");
                pVia->agpTexHandle = 0;
                pVia->agpTexMappedAddr = 0;
                pVia->agpTexSize = 0;
            }
            else
            {
                xf86DrvMsg(pScreen->myNum, X_INFO,
                "Map agp/pcie texture memory successfully and branch buffer will work for H6 chipset! \n");
            }
        }

        if(!pVia->IsPCI)
        {
            viaQuerryBranchAgpBufferInfo(pVia);
        }

        return TRUE;
    }
    else
    {
        if (drmVIAInitMAP(pVia->drmFD, &drmInfo) < 0) 
            return FALSE;
    }

    return TRUE;
}

/* Add a map for the MMIO registers */
static Bool 
VIADRIMapInit(ScreenPtr pScreen, VIAPtr pVia)
{
    int flags = 0;
    drm_handle_t hdl;

    if (drmAddMap(pVia->drmFD, pVia->MmioBase, VIA_MMIO_REGSIZE,
		  DRM_REGISTERS, flags, &pVia->registerHandle) < 0) {
	return FALSE;
    }

    /*for non-root access issue*/
    pVia->pVidData->viaGfxInfo->mmiohandle = pVia->registerHandle;
    pVia->pVidData->viaGfxInfo->mmioPhybase = pVia->MmioBase;
    pVia->pVidData->viaGfxInfo->mmioSize = VIA_MMIO_REGSIZE;

    if (!(pVia->IsSecondary)){
        /* V4L/DDMpeg need it to call drmMap, just map for All video memory ONE time */    
        if (drmAddMap(pVia->drmFD, pVia->FrameBufferBase, pVia->pScreensPublicInfo->VRamSize,
                      DRM_FRAME_BUFFER, flags, &hdl) < 0) {
            return FALSE;
        }
    }
    pVia->pVidData->viaGfxInfo->fbhandle = hdl;
    pVia->pVidData->viaGfxInfo->fbPhybase = pVia->FrameBufferBase;
    pVia->pVidData->viaGfxInfo->fbSize = pVia->videoRambytes;
    
#ifdef DRM_SAMM_VIA
    if (!(pVia->IsSecondary)){
#endif
        if(H5_UMA_CHIPID||H6_UMA_CHIPID) /*for H5s1_UMA*/
        {
             /*FIXME, Now init pVia->pDriConf->fbHandle here*/
             pVia->pDriConf->fbHandle = hdl;

            /* Because if UMA use pci path, the agphead1 data will transfer to 
             * 2d host bitblt map region for 2d host bitblt map region, now 
             * only the firest 32 bit is used, so only map 0x9000 is enough*/
            if (drmAddMap(pVia->drmFD, pVia->MmioBase + VIA_MMIO_BLTBASE, VIA_MMIO_REGSIZE,
                DRM_REGISTERS, flags, &pVia->hostBltHandle) < 0) {
                return FALSE;
            }
        
            VIAInitS3gConf(pScreen);
            VIAMapAGP(pScreen, pVia);
        }
#ifdef DRM_SAMM_VIA
    }
#endif

    DEBUG(xf86DrvMsg(pScreen->myNum, X_INFO,
	"[drm] register handle = 0x%08lx, HostBlt handle = 0x%08lx\n", 
    (unsigned long)pVia->registerHandle, (unsigned long)pVia->hostBltHandle));

    return TRUE;
}

/*following functions are only used by H5s1_UMA(vt3336/vt3364)*/
int 
VIAMapAGP(ScreenPtr pScreen, VIAPtr pVia)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    
    DriConfPtr pDriConf = pVia->pDriConf;
    /* Linear frame buffer handle: pVia->pDriConf->fbHandle 
     * had been set on VIADRIMapInit*/

    ChipConfPtr pChipConf = pVia->pChipConf;
    AddrConfPtr pAddrConf = pVia->pAddrConf;
    ScrnConfPtr pScrnConf = pVia->pScrnConf;
    /*get pDriConf agp para*/
    pDriConf->agpHandle = pVia->agpHandle;
    pDriConf->agpBase = (unsigned long)pVia->agpBase;
    pDriConf->agpSize = pVia->agpSize;
	
    /*at first, get mmio handle from pVia*/
    pDriConf->mmioHandle = pVia->registerHandle;

    /* alloc shadow region */
    pDriConf->shadowSize = 16 * 1024; /* 16K */
    pDriConf->shadowOffset = 0x0;

    /*now remove the shadow region to pci space*/
    #define _DRM_CONSISTENT 5
    if (drmAddMap(pVia->drmFD,
                  0,
                  pDriConf->shadowSize,/* size of shadow region */
                  _DRM_CONSISTENT, 0,
                  &pDriConf->shadowHandle) < 0) {
    xf86DrvMsg(pScreen->myNum, X_ERROR,
                   "[drm] Could not add shadow region mapping\n");
        return FALSE;
    }

    if (!pVia->IsPCI) /*AGP has been enabled*/
    {
        pDriConf->dmaBufAGP = TRUE;   /*tell user space DRI, agp ok*/
        
        /* add dma buffer */
        int dmaBufSize = AGP_CMDBUF_SIZE;
        if (drmAddMap(pVia->drmFD,
              0x0,/* offset of AGP = 0x0, because shadow is in sys mem */
              dmaBufSize,/* size of AGP */
              DRM_AGP, 0,
              &pDriConf->dmaHandle) < 0) {
            xf86DrvMsg(pScreen->myNum, X_ERROR,
                   "[drm] Could not add DMA buffers mapping\n");
            return FALSE;
        }
        pDriConf->dmaBufOffs = 0x0;
        
        int AGPTexSize = pVia->agpSize - dmaBufSize;
        if(AGPTexSize > 0)
        {
            if (drmAddMap(pVia->drmFD,
            dmaBufSize,/*Follow the AGP DMA buffer*/
            AGPTexSize,/* size of AGP */
            DRM_AGP, 0,
            &pDriConf->agpTexHandle) < 0) {
            xf86DrvMsg(pScreen->myNum, X_ERROR,
            "[drm] Could not add AGP tex buffers mapping\n");
            return FALSE;
            }
            
            pDriConf->agpTexOffset = dmaBufSize;
            pDriConf->agpTexSize = AGPTexSize;
            pVia->agpTexHandle = pDriConf->agpTexHandle;
            pVia->agpTexSize = AGPTexSize;
            /* For H5 AGP chips: agpTexBaseOffset is the agpTexHandle returned by DrmAddMap() */
            pVia->agpTexBaseOffset = pVia->agpTexHandle;
        }
    }
    else  /*AGP is disable, use PCI*/
    {
        pDriConf->dmaBufAGP = FALSE;   /*tell user module, agp init fail and use PCI*/
    }

    pDriConf->agpMode   = drmAgpGetMode(pVia->drmFD);
    pDriConf->agpVendor = drmAgpVendorId(pVia->drmFD);
    pDriConf->agpDevice = drmAgpDeviceId(pVia->drmFD);

    pDriConf->frontPhyAddr = pVia->FrameBufferBase + pDriConf->frontOffset;
    pDriConf->backPhyAddr = pVia->FrameBufferBase + pDriConf->backOffset;
    pDriConf->depthPhyAddr = pVia->FrameBufferBase + pDriConf->depthOffset;

    /*fill in Rec*/
    strcpy(pChipConf->chipName, pScrn->chipset);
    /*pChipConf->chipName = pScrn->chipset;*/
    pChipConf->maxIgas = 2;
    pChipConf->maxApts = 16; /*like deltachrome*/
    if(H5_UMA_CHIPID||H6_UMA_CHIPID)
    {
        pChipConf->pciID = pVia->ChipId;
        pChipConf->primary = 1;
        if(H6_UMA_CHIPID)
            pChipConf->chipType = CHIP_PCIE;
        else
            pChipConf->chipType = CHIP_AGP;

        pChipConf->samm = 0;
        pChipConf->mergefb = 0;
        pChipConf->bcipm = 1; /*BCI power management on, like clb*/
        pChipConf->docking = 0;
        pChipConf->swizzleOff = 1;
    }
    /*fill in AddrChonf*/
    pAddrConf->fbVirtAddr = (unsigned long)pVia->FBBase;
    pAddrConf->fbPhyAddr = pVia->FrameBufferBase;
    pAddrConf->fbSize = pScrnConf->fbSize;
    /*    pAddrConf->fbFree = pVia->FBFreeStart;*/
    /*!!!this line is error, for s3g, fbFree means the highest side of videomem, which is contrary to via*/  
    pAddrConf->fbFree = pVia->FBFreeEnd; /*give the available end of fb*/
    pAddrConf->mmioPhyAddr = pVia->MmioBase;
    pAddrConf->mmioVirtAddr = (unsigned long)pVia->MapBase;
    pAddrConf->mmioSize = VIA_MMIO_REGSIZE;
    pAddrConf->bciPhyAddr = pAddrConf->mmioPhyAddr + 0x10000;
    pAddrConf->bciVirtAddr = pAddrConf->mmioVirtAddr + 0x10000;
    pAddrConf->bciSize = 0x10000;
    pAddrConf->aptPhyAddr[0] = pScrnConf->fbPhyAddr;
    pAddrConf->aptPhyAddr[1] = pScrnConf->fbPhyAddr + pScrnConf->fbOffset;
    pAddrConf->aptVirtAddr[0] = pAddrConf->fbVirtAddr;
    pAddrConf->aptVirtAddr[1] = pAddrConf->fbVirtAddr + pScrnConf->fbOffset;
    pAddrConf->aptPitch[0] = pScrnConf->scrnDelta;
    pAddrConf->aptUsing[0] = 1;/*what meaning? just set like Deltachrome*/

    return TRUE;
}

int 
VIAReserveFBBuffer(ScreenPtr pScreen)
{
	/*get via rec*/
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr    pVia = VIAPTR(pScrn);
    /*get s3g rec*/
    ScrnConfPtr pScrnConf = VIAPTR(pScrn)->pScrnConf;
    DriConfPtr pDriConf = VIAPTR(pScrn)->pDriConf;
    /*although now 3d will dynamic alloc and free vidmem with drm, these setting seems to be unuseful*/
    /*but such steps will effect set modes in 3d dri*/
    /*when call s3gFillInModes() to setmodes, if (pDevInfo->backOffset != pDevInfo->depthOffset), DRI will think that there is no backbuf*/
    ulong bufSize = pScrnConf->scrnScanline * pScrnConf->scrnDelta;
    int totalBufs = pScrnConf->memScanline / pScrnConf->scrnScanline;
    int texScanline,XBuffers,needBufs;
    Bool flag;

    if (pScrnConf->Bpp == 2) /*in s3g, &&pScrnConf->z24, no use?*/
    {
        needBufs = 4;
        flag = TRUE;
    } 
    else 
    {
        needBufs = 3;
        flag = FALSE;
    }
    pScrnConf->xaaBufs = (totalBufs - needBufs) / 3;  /*reserve 1/3 of fb for off-screen*/

    XBuffers = pScrnConf->xaaBufs + 1;         
    pScrnConf->XScanline = XBuffers * pScrnConf->scrnScanline;
    /* reserver memory for back/depth */
    pDriConf->frontOffset = pScrnConf->fbOffset;  
    /*frontOffset means on-screen start?*/
    pDriConf->backOffset = pVia->FBFreeStart; /*+ XBuffers * bufSize;*/
    /*because we do not use s3g XAA driver, so XBuffers is unuseful?*/
    /*backOffset begins after XAA(offscreen)*/
    pDriConf->depthOffset = pDriConf->backOffset + bufSize;
    /*depthOffset just after back buffer, backbuffer size = bufSize?so small?*/
    pDriConf->backY = pDriConf->backOffset / pScrnConf->scrnDelta;
    pDriConf->depthY = pDriConf->depthOffset / pScrnConf->scrnDelta;
    /*reserve tex buffer*/
    texScanline = pScrnConf->memScanline - ((needBufs-1) + XBuffers) * pScrnConf->scrnScanline;
    if (flag) 
        pDriConf->fbTexOffset = pDriConf->depthOffset + 2 * bufSize;
    else
        pDriConf->fbTexOffset = pDriConf->depthOffset + bufSize;

    pDriConf->fbTexY = pDriConf->fbTexOffset / pScrnConf->scrnDelta;
    pDriConf->fbTexSize = texScanline * pScrnConf->scrnDelta;
    return TRUE;
}

#define TILEDFBINUSE 0
/*collect VIA screen info to fill in that of s3g*/
int 
VIAInitS3gConf(ScreenPtr pScreen)
{
    /*via rec*/
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr    pVia = VIAPTR(pScrn);
    /*s3g rec*/
    ScrnConfPtr pScrnConf = VIAPTR(pScrn)->pScrnConf;
    DriConfPtr pDriConf = VIAPTR(pScrn)->pDriConf;

    /*get the physical address*/
    pScrnConf->fbPhyAddr=pVia->FrameBufferBase;
	
    /*copy samm flag*/
    pScrnConf->samm=pVia->pBIOSInfo->SAMM;   
    /*now set the default single-screen*/
    pScrnConf->master=1;
    /*set screen*/
    pScrnConf->virtualX = pScrn->virtualX;
    pScrnConf->virtualY = pScrn->virtualY;
    pScrnConf->bpp = pScrn->bitsPerPixel;
    pScrnConf->Bpp = pScrn->bitsPerPixel >> 3;
    pScrnConf->scrnDelta = pScrn->virtualX * pScrnConf->Bpp;
#if TILEDFBINUSE
    if (pScrnConf->bpp == 8)
    {
#endif
        /* 32: Alignment for nontiled mode */
        pScrnConf->scrnDelta = (pScrnConf->scrnDelta + 31) & (~31);
        pScrnConf->scrnScanline = pScrn->virtualY;
        /* no use */
        pScrnConf->scrnWit = pScrnConf->scrnDelta  >> 7;
#if TILEDFBINUSE
    } 
    else 
    {
        /* align to a tile width */
        pScrnConf->scrnDelta = (pScrnConf->scrnDelta + 127) & (~127);
        /* align to a tile height */
        /* 4k per tile */
        pScrnConf->scrnScanline = (pScrn->virtualY + 31) & (~31);
        pScrnConf->scrnWit = pScrnConf->scrnDelta  >> 7;
    }
#endif
    pScrnConf->scrnWip = pScrnConf->scrnDelta / pScrnConf->Bpp;

    /* set framebuffer parameter:size/offset/linear address */
    if (!pScrnConf->samm) 
    {
        /* non-samm case */
        pScrnConf->fbOffset = 0; 
     /*seems that pScrnConf->fbOffset in s3g is different from  pVIADRI->fbOffset in VIA*/
    /*pVIADRI->fbOffset = pVia->FBFreeStart*/
    /*pScrnConf->linearVideoOffset = (pAddrConf->fbFree - 800*600*2) & (~(0x1000-1));*/
        pScrnConf->fbSize = (unsigned long)pVia->videoRambytes;
        pScrnConf->scrnLinear = (unsigned long)pVia->FBBase;
    } 
    /* allocate offscreen scanline by scanline */
    pScrnConf->memDelta = pScrnConf->scrnDelta;
    /* usable scanline(subtrace the scrnScanline) */
    pScrnConf->memScanline = pScrnConf->fbSize / pScrnConf->memDelta;
    pScrnConf->XScanline = pScrnConf->memScanline;
    /*fb start and fb pitch*/
    pScrnConf->scrnStart = (unsigned long)pVia->FBBase;
    pScrnConf->scrnPitch = pScrnConf->scrnDelta;
    pScrnConf->scrnPhyAddr = pVia->FrameBufferBase;
    pScrnConf->fbPhyAddr = pVia->FrameBufferBase + pScrnConf->fbOffset;
    /*pScrnConf->fbSize = pVia->FBFreeEnd - pVia->FBFreeStart;*/
#if TILEDFBINUSE
    pScrnConf->tiled = 1; /*fb is tiled, like clb*/
#else
    pScrnConf->tiled = 0;
#endif
    pScrnConf->driEnabled = 1;
    pScrnConf->accel = 1;
    pScrnConf->video = 1;
#if TILEDFBINUSE
    pScrnConf->scrnTile = pScrnConf->scrnLinear; /*cracked, exactly should be pAddrConf->aptVirtAddr[0]*/
#else
    pScrnConf->scrnTile = 0;
#endif
    /*
    pDriConf->dmaBufNum = S3G_NUM_BUFFERS;
    pDriConf->dmaBufSize = S3G_BUFFER_SIZE;
    */
    /*others*/
    pDriConf->drmFd = pVia->drmFD;
    pDriConf->pDRIInfo= pVia->pDRIInfo;

    return TRUE;
}

int 
VIAS3GFinishScreenInit(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    /*DrvInfoPtr  pDrv = DRVPTR(pScrn);*/
    VIAPtr pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr pBIOSInfo = pVia->pBIOSInfo;
    ScrnConfPtr pScrnConf=VIAPTR(pScrn)->pScrnConf;
    DriConfPtr  pDriConf=VIAPTR(pScrn)->pDriConf;
    ServerInfoPtr pServerInfo = VIAPTR(pScrn)->pServerInfo;

    /*get additional datas for vt3336*/
    memcpy(&pServerInfo->AddrConf, pVia->pAddrConf, sizeof(AddrConfRec));
    memcpy(&pServerInfo->ChipConf, pVia->pChipConf, sizeof(ChipConfRec));
    memcpy(&pServerInfo->ScrnConf, pVia->pScrnConf, sizeof(ScrnConfRec));
    memcpy(&pServerInfo->DriConf, pVia->pDriConf, sizeof(DriConfRec));
    memcpy(&pServerInfo->PciConf, pVia->pPciConf, sizeof(PciConfRec));

    if(H5_UMA_CHIPID||H6_UMA_CHIPID)
    {
        if(H6_UMA_CHIPID)
            pServerInfo->chipType = CHIP_PCIE;
        else
            pServerInfo->chipType = CHIP_AGP; /*CHIP_AGP*/

        pServerInfo->chipIndex = S3G_INV; /*S3G_CLB*/

        switch(pVia->ChipId) {
        case PCI_CHIP_VT3336:
        case PCI_CHIP_VT3364:
                 pServerInfo->chipSubIndex =CHIPID_H5S1;
        break;

        case PCI_CHIP_VT3353:
        case PCI_CHIP_VT3409:
                pServerInfo->chipSubIndex =CHIPID_H6S2;
        break;

        default:
        break;			   			   	
        }
        /*CHIP_CLB*/
        pServerInfo->isAGP=!pVia->IsPCI;
    }

#ifdef DRM_SAMM_VIA
    /*DRM + Xinerama, DRI Arch is NOT enabled*/
    if( !(pVia->pBIOSInfo->SAMM)) { 
#endif
            pServerInfo->sareaPrivOffset = sizeof(XF86DRISAREARec);
            pServerInfo->dmaBufOffs = pDriConf->dmaBufOffs;

            pServerInfo->bufGBD = pScrnConf->gbd;
            pServerInfo->bufBpp = pScrnConf->Bpp;
            pServerInfo->bufSize = pScrnConf->scrnScanline * pScrnConf->scrnPitch;
            pServerInfo->bufPitch = pScrnConf->scrnPitch;
            pServerInfo->bufWit = pScrnConf->scrnWit;

            if ((pScrnConf->Bpp == 4) || 
                            (pScrnConf->Bpp == 2 && !pDriConf->z24)) {
                    pServerInfo->zbufGBD = pScrnConf->gbd;
                    pServerInfo->zbufBpp = pScrnConf->Bpp;
                    pServerInfo->zbufSize = pScrnConf->scrnScanline * pScrnConf->scrnPitch;
                    pServerInfo->zbufPitch = pScrnConf->scrnPitch;
                    pServerInfo->zbufWit = pScrnConf->scrnWit;
            } else { /* 16bpp mode but 32bpp depth buffer */
                    pServerInfo->zbufGBD = (2 * pScrnConf->scrnDelta) | (1<<16) | (1<<18);
                    pServerInfo->zbufBpp = 2 * pScrnConf->Bpp;
                    pServerInfo->zbufSize = pScrnConf->scrnScanline * (2 * pScrnConf->scrnPitch);
                    pServerInfo->zbufPitch = 2 * pScrnConf->scrnPitch;
                    pServerInfo->zbufWit = 2 * pScrnConf->scrnWit;
            }

            pServerInfo->frontOffset = pDriConf->frontOffset;
            pServerInfo->backOffset = pDriConf->backOffset;
            pServerInfo->depthOffset = pDriConf->depthOffset;
            pServerInfo->backY = pDriConf->backY;
            pServerInfo->depthY = pDriConf->depthY;

            pServerInfo->dmaHandle = pDriConf->dmaHandle;
            pServerInfo->frontHandle = pDriConf->frontHandle;
            pServerInfo->backHandle = pDriConf->backHandle;
            pServerInfo->depthHandle = pDriConf->depthHandle;
            pServerInfo->fbTexOffset = pDriConf->fbTexOffset;
            pServerInfo->fbTexSize = pDriConf->fbTexSize;

            pServerInfo->shadowSize = pDriConf->shadowSize;
            pServerInfo->shadowHandle = pDriConf->shadowHandle;

            pServerInfo->agpTexOffset = pDriConf->agpTexOffset;
            pServerInfo->agpTexSize = pDriConf->agpTexSize;
            pServerInfo->agpTexHandle = pDriConf->agpTexHandle;

            pServerInfo->mmioHandle = pVia->registerHandle;
            pServerInfo->hostBltHandle = pVia->hostBltHandle;
            pServerInfo->RotateDegree = pVia->RotateDegree;

            /* To get the screen's width and height */
            pServerInfo->width = pBIOSInfo->HDisplay;
            pServerInfo->height	= pBIOSInfo->VDisplay;

            pDriConf->allowPageFlip = 0;

            /*sarea init*/
            VIA_CHROME9SAREAPriv  *pSAREA = DRIGetSAREAPrivate(pScreen);
            pSAREA->page_flip=0;
#ifdef DRM_SAMM_VIA
    }
#endif

    return TRUE;
}

static __GLXvisualConfig * 
S3GInitVisConf16Bpp(ScreenPtr pScreen,int *numConfigs)
{
    __GLXvisualConfig *pConfigs = NULL;
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    DriConfPtr  pDriConf=VIAPTR(pScrn)->pDriConf;
    int i, db, stencil, accum;

    *numConfigs = 8;

    pConfigs = (__GLXvisualConfig*)xcalloc(sizeof(__GLXvisualConfig),*numConfigs);
    if (!pConfigs) {
        return NULL;
    }
    
    i = 0;
    for (db = 0 ; db <= 1 ; db++) 
    { /* for accum buffer */
       for (accum = 0 ; accum <= 1 ; accum++) 
       { /* for stencil buffer */
          for (stencil = 0 ; stencil <= 1 ; stencil++) 
          { /* for depth buffer */
              pConfigs[i].vid         = -1;
              pConfigs[i].class       = -1;
              pConfigs[i].rgba        = TRUE;
              pConfigs[i].redSize     = 5;
              pConfigs[i].greenSize   = 6;
              pConfigs[i].blueSize    = 5;
              pConfigs[i].alphaSize   = 0;
              pConfigs[i].redMask     = 0x0000F800;
              pConfigs[i].greenMask   = 0x000007E0;
              pConfigs[i].blueMask    = 0x0000001F;
              pConfigs[i].alphaMask   = 0;
              if (accum) 
              {
                  pConfigs[i].accumRedSize    = 16;
                  pConfigs[i].accumGreenSize  = 16;
                  pConfigs[i].accumBlueSize   = 16;
                  pConfigs[i].accumAlphaSize  = 0;
              } 
              else 
              {
                  pConfigs[i].accumRedSize    = 0;
                  pConfigs[i].accumGreenSize  = 0;
                  pConfigs[i].accumBlueSize   = 0;
                  pConfigs[i].accumAlphaSize  = 0;
              }
              if (!db) 
              {
                  pConfigs[i].doubleBuffer    = TRUE;
              } 
              else 
              {
                  pConfigs[i].doubleBuffer    = FALSE;
              }
              pConfigs[i].stereo        = FALSE;
              pConfigs[i].bufferSize    = 16;             
              if (pDriConf->z24) 
              { 
                  pConfigs[i].depthSize = 24;
              } else 
              {
                  pConfigs[i].depthSize = 16;
              }
              if (stencil) 
              {
                  pConfigs[i].stencilSize  = 8;
              } 
              else 
              {
                  pConfigs[i].stencilSize  = 0;
              }
              pConfigs[i].auxBuffers       = 0;
              pConfigs[i].level            = 0;
              if (accum) 
              {
                  pConfigs[i].visualRating    = GLX_SLOW_CONFIG;
              } else 
              {
                  pConfigs[i].visualRating    = GLX_NONE;
              }
              pConfigs[i].transparentPixel    = GLX_NONE;
              pConfigs[i].transparentRed      = 0;
              pConfigs[i].transparentGreen    = 0;
              pConfigs[i].transparentBlue     = 0;
              pConfigs[i].transparentAlpha    = 0;
              pConfigs[i].transparentIndex    = 0;
              i++;
           }
        }
    }

    return pConfigs;
}

static __GLXvisualConfig * 
S3GInitVisConf32Bpp(ScreenPtr pScreen,int *numConfigs)
{
    __GLXvisualConfig *pConfigs = NULL;
    int i, db, accum,stencil;

    *numConfigs = 8;
    pConfigs = (__GLXvisualConfig*)xcalloc(sizeof(__GLXvisualConfig),*numConfigs);
    if (!pConfigs) 
    {
        return NULL;
    }
    
    i = 0;
    for (db = 0 ; db <= 1 ; db++) 
    {
        for (accum = 0 ; accum <= 1 ; accum++) 
        { /* and stencil */
            for (stencil = 0 ; stencil <= 1 ; stencil++) 
            {
                pConfigs[i].vid         = -1;
                pConfigs[i].class       = -1;
                pConfigs[i].rgba        = TRUE;
                pConfigs[i].redSize     = 8;
                pConfigs[i].greenSize   = 8;
                pConfigs[i].blueSize    = 8;
                pConfigs[i].alphaSize   = 8;
                pConfigs[i].redMask     = 0x00FF0000;
                pConfigs[i].greenMask   = 0x0000FF00;
                pConfigs[i].blueMask    = 0x000000FF;
                pConfigs[i].alphaMask   = 0xFF000000;
                if (accum) 
                {
                    pConfigs[i].accumRedSize    = 16;
                    pConfigs[i].accumGreenSize  = 16;
                    pConfigs[i].accumBlueSize   = 16;
                    pConfigs[i].accumAlphaSize  = 16;
                }
                else 
                {
                    pConfigs[i].accumRedSize    = 0;
                    pConfigs[i].accumGreenSize  = 0;
                    pConfigs[i].accumBlueSize   = 0;
                    pConfigs[i].accumAlphaSize  = 0;
                }
                if (!db) 
                {
                    pConfigs[i].doubleBuffer    = TRUE;
                } else 
                {
                    pConfigs[i].doubleBuffer    = FALSE;
                }
                pConfigs[i].stereo            = FALSE;
                pConfigs[i].bufferSize        = 32;
                pConfigs[i].depthSize     = 24;
                if (stencil) {
                    pConfigs[i].stencilSize   = 8;
                }
                else 
                {
                    pConfigs[i].stencilSize   = 0;
                }
                pConfigs[i].auxBuffers   = 0;
                pConfigs[i].level        = 0;
                if (accum) 
                {
                    pConfigs[i].visualRating    = GLX_SLOW_CONFIG;
                } 
                else 
                {
                    pConfigs[i].visualRating    = GLX_NONE;
                }
                pConfigs[i].transparentPixel    = GLX_NONE;
                pConfigs[i].transparentRed      = 0;
                pConfigs[i].transparentGreen    = 0;
                pConfigs[i].transparentBlue     = 0;
                pConfigs[i].transparentAlpha    = 0;
                pConfigs[i].transparentIndex    = 0;
                i++;
            }
        }
    }
    return pConfigs;    
}

static int 
S3GInitVisualConfigs(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    DriConfPtr pDriConf = VIAPTR(pScrn)->pDriConf;
    __GLXvisualConfig *pConfigs = NULL;
    int numConfigs = 0, i;
    VIAConfigVisualPrivPtr pVIAConfigs = NULL;
    VIAConfigVisualPrivPtr* pVIAConfigPtrs = NULL;
    
    if (pScrn->bitsPerPixel == 16) 
    {
        pConfigs = S3GInitVisConf16Bpp(pScreen,&numConfigs);
        if (!(pVIAConfigs = (VIAConfigVisualPrivPtr)xcalloc(sizeof(VIAConfigVisualPrivRec),
  						    numConfigs))) 
        {
            xfree(pConfigs);
            return FALSE;
        }
  
  	if (!(pVIAConfigPtrs = (VIAConfigVisualPrivPtr*)xcalloc(sizeof(VIAConfigVisualPrivPtr),
  			  numConfigs))) {
            xfree(pConfigs);
            xfree(pVIAConfigs);
            return FALSE;
        }
  	   
        for (i=0; i<numConfigs; i++) 
      	{
      	    pVIAConfigs[i].class = -1;
            pVIAConfigs[i].depth = 16;
      	    pVIAConfigPtrs[i] = &pVIAConfigs[i];
        }
    } 
    else if (pScrn->bitsPerPixel == 32) 
    {
        pConfigs = S3GInitVisConf32Bpp(pScreen,&numConfigs); 
        if (!(pVIAConfigs = (VIAConfigVisualPrivPtr)xcalloc(sizeof(VIAConfigVisualPrivRec),
  						    numConfigs))) 
        {
            xfree(pConfigs);
            return FALSE;
        }
  
        if (!(pVIAConfigPtrs = (VIAConfigVisualPrivPtr*)xcalloc(sizeof(VIAConfigVisualPrivPtr),
                                 numConfigs))) {
            xfree(pConfigs);
            xfree(pVIAConfigs);
            return FALSE;
        }

        for (i=0; i<numConfigs; i++) 
        {
            pVIAConfigs[i].class = -1;
            pVIAConfigs[i].depth = 32;
            pVIAConfigPtrs[i] = &pVIAConfigs[i];
        }
    } 

    pDriConf->pVisualConfigs = pConfigs;
    GlxSetVisualConfigs(numConfigs, pConfigs, (void**)pVIAConfigPtrs);

    return TRUE;
}

