/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
#include "via_driver.h"
#include "via_dvi.h"
#include "hw.h"


void VIADVIPatchSkew(VIABIOSInfoPtr pBIOSInfo, int DIPort)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIADVIPatchSkew: DIPort=%d\n", DIPort));
    
    switch (DIPort)
    {
        case VIA_DI_DVP0:
            VIADVIPatchSkew_DVP0(pBIOSInfo);
            break;

        case VIA_DI_DVP1:
            break;
            
        case VIA_DI_DFPLOW:
            VIADVIPatchSkew_DFPLOW(pBIOSInfo);            
            break;

        case VIA_DI_DFPHIGH:
            break;

        default:
            break;
    }

    /* Replace the default patch if get user DPA settings from xorg: */
    if (pBIOSInfo->UserDPASettingInfo.HasUserSetting_Gfx)
    {
        VIASetDPA_Gfx(DIPort, &(pBIOSInfo->UserDPASettingInfo.GfxDPA));
    }
}

void VIADVIPatchSkew_DVP0(VIABIOSInfoPtr pBIOSInfo)
{
    switch (pBIOSInfo->Chipset)
    {
        case VIA_P4M890:
        {
            if((pBIOSInfo->CrtcHDisplay==1600)&&(pBIOSInfo->CrtcVDisplay==1200))
                write_reg_mask(CR96, VIACR, 0x03, BIT0+BIT1+BIT2);
            else
                write_reg_mask(CR96, VIACR, 0x07, BIT0+BIT1+BIT2);
            break;
        }

        case VIA_P4M900:
        {
            /* Validate by CN896(VT5943A-2), VN896(VT6363A1-2)*/
            /* and P4M900(VT8450C/VT8450D). */
            write_reg_mask(CR96, VIACR, 0x02, BIT0+BIT1+BIT2);
            break;
        }

        default:
        {
           break;
        }
    }
}


void VIADVIPatchSkew_DFPLOW(VIABIOSInfoPtr pBIOSInfo)
{
    switch (pBIOSInfo->Chipset)
    {
        case VIA_K8M890:
        {
            write_reg_mask(CR99, VIACR, 0x03, BIT0+BIT1);
            break;
        }
        
        case VIA_P4M900:        
        {
            /* Validate by CN896(VT5943A-2), VN896(VT6363A1-2)*/
            /* and P4M900(VT8450C/VT8450D). */
            write_reg_mask(CR99, VIACR, 0x09, BIT0+BIT1+BIT2+BIT3);
            break;
        }

        case VIA_P4M890:        
        {
            write_reg_mask(CR99, VIACR, 0x0F, BIT0+BIT1+BIT2+BIT3);
            break;
        }

        default:
        {
           break;
        }
    }
}

void VIASetDVIOutputPath(VIABIOSInfoPtr pBIOSInfo, int set_iga, int output_interface)
{
    switch(output_interface)
    {
        case VIA_DI_DVP0:
            write_reg_mask(CR6B, VIACR, 0x01, BIT0);

            if (set_iga == IGA1)
                write_reg_mask(CR96, VIACR, 0x00, BIT4);
            else
                write_reg_mask(CR96, VIACR, 0x10, BIT4);
                 
            write_reg_mask(SR1E, VIASR, 0xC0, BIT7+BIT6);
            break;

        case VIA_DI_DVP1:
            if (set_iga== IGA1)
                write_reg_mask(CR9B, VIACR, 0x00, BIT4);
            else
                write_reg_mask(CR9B, VIACR, 0x10, BIT4);

            write_reg_mask(SR1E, VIASR, 0x30, BIT4+BIT5);
            break;

        case VIA_DI_DFPHIGH:
            if (set_iga== IGA1)
            {
                write_reg_mask(CR96, VIACR, 0x00, BIT4);
                write_reg_mask(CR97, VIACR, 0x03, BIT0+BIT1+BIT4);
            }
            else
            {
                write_reg_mask(CR96, VIACR, 0x10, BIT4);
                write_reg_mask(CR97, VIACR, 0x13, BIT0+BIT1+BIT4);
            }
       
            write_reg_mask(SR2A, VIASR, 0x0C, BIT2+BIT3);
            break;
            
        case VIA_DI_DFPLOW:
            if (set_iga == IGA1)
            {
                write_reg_mask(CR99, VIACR, 0x00, BIT4);
                write_reg_mask(CR9B, VIACR, 0x00, BIT4);
            }
            else
            {                
                write_reg_mask(CR99, VIACR, 0x10, BIT4);
                write_reg_mask(CR9B, VIACR, 0x10, BIT4);
            }

            write_reg_mask(SR2A, VIASR, 0x03, BIT0+BIT1);
            break;

        case VIA_DI_TMDS:
            if (set_iga == IGA1)
            {
                write_reg_mask(CR99, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg_mask(CR99, VIACR, 0x10, BIT4);
            }
            
            break;
    }
}


Bool VIATMDSIdentify_VT1632(VIABIOSInfoPtr pBIOSInfo, TMDSSETTINGINFOPTR pTMDSSettingInfo, CARD32 PortNum)
{
    CARD8 Buffer[2];

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIATMDSIdentify_VT1632A: Try Port No. 0x%x\n", PortNum));
    
    /* Sense VT1636 LVDS Transmiter */
    pTMDSSettingInfo->I2CPort = PortNum;
    pTMDSSettingInfo->I2CAddress = 0x10;

    /* Check vendor ID first: */
    if (!GPIOI2CRead_TMDS(pBIOSInfo, pTMDSSettingInfo, 0x00, &(Buffer[0])))
    {
        return FALSE;
    }
    
    if (!GPIOI2CRead_TMDS(pBIOSInfo, pTMDSSettingInfo, 0x01, &(Buffer[1])))
    {
        return FALSE;
    }

    if (!((Buffer[0] == 0x06) && (Buffer[1] == 0x11)))
    {        
        return FALSE;        
    }

    /* Vendor ID is correct, so continue to check Chip ID: */
    if (!GPIOI2CRead_TMDS(pBIOSInfo, pTMDSSettingInfo, 0x02, &(Buffer[0])))
    {
        return FALSE;
    }
    
    if (!GPIOI2CRead_TMDS(pBIOSInfo, pTMDSSettingInfo, 0x03, &(Buffer[1])))
    {
        return FALSE;
    }

    if ((Buffer[0] == 0x92) && (Buffer[1] == 0x31))
    {
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "Found VIA VT1632 TMDS on port %x!\n", PortNum));
        pTMDSSettingInfo->ChipID = VIA_VT1632;
        return TRUE;
    }

    return FALSE;
}


int VIATMDSIdentify_IntegratedTMDS(VIABIOSInfoPtr pBIOSInfo)
{
    int ret= NOT_FOUND_DEV;
    
    if (pBIOSInfo->Chipset == VIA_CX700) {
        pBIOSInfo->TMDSSettingInfo.ChipID = VIA_INTEGRATED_TMDS;
        pBIOSInfo->TMDSSettingInfo.I2CPort = PORT_INDEX_I2C_31;
        ret = FOUND_INTERNAL_DEV;
    } else if (pBIOSInfo->Chipset == VIA_VX800) {
        pBIOSInfo->TMDSSettingInfo.ChipID = VIA_INTEGRATED_TMDS;
		/*3353 A0, A1 and B0 use 3D port, B2 and B4 use 31 port*/
		int port = (VIAGetRevisionOfVX800() == REVISION_VX800_B2) ? PORT_INDEX_I2C_31 : PORT_INDEX_GPIO_3D;
        pBIOSInfo->TMDSSettingInfo.I2CPort = port;
        pBIOSInfo->GPIOI2CInfo.bGPIOPort = port;    
        ret = FOUND_INTERNAL_DEV;
    }
    
    return ret;

}


void VIATMDSRegInit_VT1632(VIABIOSInfoPtr pBIOSInfo)
{  
    TMDSSETTINGINFOPTR pTMDSSettingInfo = &(pBIOSInfo->TMDSSettingInfo);

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIATMDSRegInit_VT1632\n"));
       
    pTMDSSettingInfo->I2CAddress = 0x10;
    
    /* Now our hardware only support dvi buswide 12 Bits,
       we should use dual edge. */
    GPIOI2CWrite_TMDS(pBIOSInfo, pTMDSSettingInfo, 0x08, 0x3B);
}


void VIATMDSRegInit(VIABIOSInfoPtr pBIOSInfo)
{
    unsigned char   sr2a = 0, sr1e = 0, sr3e = 0;

    sr1e = read_reg(VIASR, SR1E);
    sr2a = read_reg(VIASR, SR2A);
    
    VIADIPortPadOn(pBIOSInfo->DFPDIPort);
    
    /* SR3E[1]Multi-function selection:
       0 = Emulate I2C and DDC bus by GPIO2/3/4. */
    sr3e = read_reg(VIASR, SR3E);
    write_reg_mask(SR3E, VIASR, 0x0, BIT5);
    
    switch (pBIOSInfo->TMDSSettingInfo.ChipID)
    {
        case VIA_VT1632:
            VIATMDSRegInit_VT1632(pBIOSInfo);
            break;

        default:
            break;
    }

    /* Restore status */
    write_reg(SR1E, VIASR, sr1e);
    write_reg(SR2A, VIASR, sr2a);
    write_reg(SR3E, VIASR, sr3e);
}


void VIAInitDVIDIPort(VIABIOSInfoPtr pBIOSInfo, int IdentifyState)
{
    if (pBIOSInfo->DFPDIPort != VIA_DI_NONE)
    {
        /* Do nothing, because DFP DIPort can be assigned by option. */
        return;
    }
        
    switch (pBIOSInfo->Chipset)
    {
       case VIA_CX700:
       case VIA_VX800:
            /* Add TMDS(VT1632A) on AMR with DVP1 support in VT3324*/
            if (IdentifyState == FOUND_INTERNAL_DEV)
            {
                /*Integrated TMDS*/
                pBIOSInfo->DFPDIPort = VIA_DI_TMDS;
            }
            else 
            {
                /*External TMDS*/
                pBIOSInfo->DFPDIPort = VIA_DI_DVP1;            
            }
            break;
            
        case VIA_K8M890:
        case VIA_P4M900:
        case VIA_P4M890:
            /* TMDS on PCIE, we set DFPLOW as default. */
            pBIOSInfo->DFPDIPort = VIA_DI_DFPLOW;            
            break;

        case VIA_VX855:
            /* 409 doesn't support internal TMDS */
            pBIOSInfo->DFPDIPort = VIA_DI_DVP1;
            break;

        default:
            if (IdentifyState == FOUND_EXTERNAL_DEV_BY_I2C){
                /* TMDS on AMR, we set DVP0 as default. */
                pBIOSInfo->DFPDIPort = VIA_DI_DVP0;
            }
            else
            {
                /* TMDS on AGP, we set DVP1 as default. */
                pBIOSInfo->DFPDIPort = VIA_DI_DVP1;
            }
            
            break;
    }
}

Bool VIATMDSIdentify(VIABIOSInfoPtr pBIOSInfo)
{
    int             IdentifyState = NOT_FOUND_DEV;
    unsigned char   sr2a = 0, sr1e = 0, sr3e = 0;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIATMDSIdentify\n"));

    switch (pBIOSInfo->Chipset)
    {
        case VIA_K8M890:
            /*=* DFP Low Pad on *=*/
            sr2a = read_reg(VIASR, SR2A);
            write_reg_mask(SR2A, VIASR, 0x03, BIT0+BIT1);
            break;

        case VIA_P4M900:
        case VIA_P4M890:
            /* DFP Low Pad on */
            sr2a = read_reg(VIASR, SR2A);
            write_reg_mask(SR2A, VIASR, 0x03, BIT0+BIT1);

            /* VT3364 also supports DVI on AMR, so we need to sense DVP0, too. */
            /* DVP0 Pad on */
            sr1e = read_reg(VIASR, SR1E);
            write_reg_mask(SR1E, VIASR, 0xC0, BIT6+BIT7);
            break;

        default:
            /*=* DVP0/DVP1 Pad on *=*/
            sr1e = read_reg(VIASR, SR1E);
            write_reg_mask(SR1E, VIASR, 0xF0, BIT4+BIT5+BIT6+BIT7);
            /*=* SR3E[1]Multi-function selection:
                 0 = Emulate I2C and DDC bus by GPIO2/3/4. *=*/
            sr3e = read_reg(VIASR, SR3E);
            write_reg_mask(SR3E, VIASR, 0x0, BIT5);
            break;
    }
       
    if (VIATMDSIdentify_VT1632(pBIOSInfo, &(pBIOSInfo->TMDSSettingInfo), PORT_INDEX_GPIO_2C))
    {
        pBIOSInfo->GPIOI2CInfo.bGPIOPort = PORT_INDEX_GPIO_2C;
        IdentifyState = FOUND_EXTERNAL_DEV_BY_GPIO;
    }
    else if (VIATMDSIdentify_VT1632(pBIOSInfo, &(pBIOSInfo->TMDSSettingInfo), PORT_INDEX_I2C_31))
    {
        pBIOSInfo->GPIOI2CInfo.bGPIOPort = PORT_INDEX_I2C_31;
        IdentifyState = FOUND_EXTERNAL_DEV_BY_I2C;
    }
    else if (VIATMDSIdentify_VT1632(pBIOSInfo, &(pBIOSInfo->TMDSSettingInfo), PORT_INDEX_GPIO_25))
    {
        pBIOSInfo->GPIOI2CInfo.bGPIOPort = PORT_INDEX_GPIO_25;
        IdentifyState = FOUND_EXTERNAL_DEV_BY_GPIO;
    }
    else
    {
        IdentifyState = NOT_FOUND_DEV;
    }
    
    if (IdentifyState == NOT_FOUND_DEV)
    {
        IdentifyState = VIATMDSIdentify_IntegratedTMDS(pBIOSInfo);
    }

    /* Restore status */
    switch (pBIOSInfo->Chipset)
    {
        case VIA_K8M890:
            write_reg(SR2A, VIASR, sr2a);
            break;

        case VIA_P4M900:
        case VIA_P4M890:
            write_reg(SR2A, VIASR, sr2a);
            write_reg(SR1E, VIASR, sr1e);
            break;

        default: 
            write_reg(SR1E, VIASR, sr1e);
            write_reg(SR3E, VIASR, sr3e);
            break;
    }

    if (IdentifyState == NOT_FOUND_DEV)
    {
        return FALSE;
    }
    else
    {
        VIAInitDVIDIPort(pBIOSInfo, IdentifyState);
        return TRUE;
    }
}




/*=*
 *
 * Bool VIASenseDVI(pBIOSInfo)
 *
 *     - Sense DVI Connector
 *
 * Return Type:    Bool
 *
 * The Definition of Input Value:
 *
 *                 VIABIOSInfoPtr
 *
 * The Definition of Return Value:
 *
 *                 DVI Attached  - TRUE
 *                 DVI Not Attached - FALSE
 *=*/
Bool VIASenseDVI(VIABIOSInfoPtr pBIOSInfo)
{
    Bool            ret = FALSE;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIASenseDVI\n"));
    
    if (pBIOSInfo->NoDDCValue)
        return TRUE;

    /* If we sense DVI connection status from TMDS, the step of */
    /* turning software power sequence on may effects LCD when we use utility to turn LCD off. */
    /* And since we can just get DVI connection status by querying DVI EDID, I remove */ 
    /* the codes about sense DVI connection status from TMDS. */
    /* Note: Not include CLE266. */

    if (!ret)
    {
        if(VIAQueryDVIEDIDType(pBIOSInfo))
        {
            ret = TRUE;
        }
    }

    return ret;
}



int VIAQueryDVIEDIDType(VIABIOSInfoPtr pBIOSInfo)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIAQueryDVIEDIDType\n"));
    
    CARD8 Buffer[2];
        
    pBIOSInfo->TMDSSettingInfo.I2CAddress = 0xA0;
    
    if (!GPIOI2CRead_TMDS(pBIOSInfo, &(pBIOSInfo->TMDSSettingInfo), 0x00, &(Buffer[0])))    
        return FALSE;
    
    
    if (!GPIOI2CRead_TMDS(pBIOSInfo, &(pBIOSInfo->TMDSSettingInfo), 0x01, &(Buffer[1])))    
        return FALSE;    

    if ((Buffer[0] == 0) && (Buffer[1] == 0xFF))
            return 1; /* Found EDID1 Table */
    else
    {
        pBIOSInfo->TMDSSettingInfo.I2CAddress = 0xA2;
        if (!GPIOI2CRead_TMDS(pBIOSInfo, &(pBIOSInfo->TMDSSettingInfo), 0x00, &(Buffer[0])))                
            return FALSE;
        
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "Buffer[0]=%d\n", Buffer[0]));
        
        if (Buffer[0] == 0x20)
            return 2; /* Found EDID2 Table */
        else
            return 0; /* Found No EDID Table */
    }

}


void VIAGetPanelSizeFromEDID(VIABIOSInfoPtr pBIOSInfo)
{        
    xf86MonPtr          edid_mon;      
    int                 i, max_h, max_v;

    unsigned char       *R_Buffer;    
    
    R_Buffer = xcalloc(1,sizeof(unsigned char) * (128));
    
    pBIOSInfo->TMDSSettingInfo.I2CAddress = 0xA0;
    
    /* Read EDID v1.1 Block */
    for (i=0; i<128; i++)
    {
        GPIOI2CRead_TMDS(pBIOSInfo, &(pBIOSInfo->TMDSSettingInfo), i, R_Buffer+i);
    }    

    /* Parse the EDID block */
    edid_mon = xf86InterpretEDID(pBIOSInfo->scrnIndex,R_Buffer);
    
    if (edid_mon)
    {             
        /*Find max resolution from Established timing*/
        VIAFindMaxResFromEstablishedTiming(edid_mon, &max_h, &max_v);

        /*Find max resolution from Standard timing*/
        for (i = 0; i < 8; i++)
        {
            if (edid_mon->timings2[i].hsize > max_h)
            {
                max_h = edid_mon->timings2[i].hsize;
                max_v = edid_mon->timings2[i].vsize;
            }            
        }

        /*Find max resolution from Detail timing*/
        for (i = 0; i < 4; i++)
        {
            if ((edid_mon->det_mon[i].type == DT) && 
                (edid_mon->det_mon[i].section.d_timings.h_active > max_h))
            {
                max_h = edid_mon->det_mon[i].section.d_timings.h_active;
                max_v = edid_mon->det_mon[i].section.d_timings.v_active;

                if (edid_mon->det_mon[i].section.d_timings.interlaced)                
                    max_v = max_v*2;                
            }       
        }
        
        /* Correct monitor size to avoid invalid mode. */
        VIACorrectMonitorSize(&max_h, &max_v);            

        /* To prevent to get a wrong EDID data. We should assign a default value.*/
        if ((max_h == 0) || (max_v == 0))
        {
            max_h = 1024;
            max_v = 768;
        }
        
        pBIOSInfo->TMDSSettingInfo.DFPSize = VIAGetModeIndex(max_h, max_v);        
    }                                
            
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "DVI Panel Size ID = 0x%x\n", pBIOSInfo->TMDSSettingInfo.DFPSize));
    

}

/*=*
 *
 * unsigned char VIAGetPanelInfo(VIABIOSInfoPtr pBIOSInfo)
 *
 *     - Get Panel Size
 *
 * Return Type:    unsigned char
 *
 * The Definition of Input Value:
 *
 *                 VIABIOSInfoPtr
 *
 * The Definition of Return Value:
 *
 *                 0 - 640x480
 *                 1 - 800x600
 *                 2 - 1024x768
 *                 3 - 1280x768
 *                 4 - 1280x1024
 *                 5 - 1400x1050
 *                 6 - 1600x1200
 *                 0xFF - Not Supported Panel Size
 *=*/
unsigned char VIAGetPanelInfo(VIABIOSInfoPtr pBIOSInfo)
{
    unsigned char   sr1e = 0, sr3e = 0, cr91 = 0, cr9b = 0;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIAGetPanelInfo\n"));

    /*If the attach device is DVI which just is used as a DVI,
    then get the panel size from the 128 bytes EDID*/
    if (pBIOSInfo->TMDSSettingInfo.DFPSize == M_INVALID)
    {
        /*=* DVP0/DVP1 Pad on *=*/
        sr1e = read_reg(VIASR, SR1E);
        write_reg_mask(SR1E, VIASR, 0xF0, BIT4+BIT5+BIT6+BIT7);

        /*=* SR3E[1]Multi-function selection:
                    0 = Emulate I2C and DDC bus by GPIO2/3/4. *=*/
        sr3e = read_reg(VIASR, SR3E);
        write_reg_mask(SR3E, VIASR, 0x0, BIT5);

        /*=* CR91[4] VDD On [3] Data On [2] VEE On [1] Back Light Off
                 [0] Software Control Power Sequence *=*/
        cr91 = read_reg(VIACR, CR91);
        write_reg(CR91, VIACR, 0x1D);

        /*=* CR9B[4] DVP1 Data Source Selection: 1 = From secondary display.
             CR9B[2:0] DVP1 Clock Adjust *=*/
        cr9b = read_reg(VIACR, CR9B);
        write_reg(CR9B, VIACR, 0x1F);
        
        switch (VIAQueryDVIEDIDType(pBIOSInfo))
        {
            case 1:
            case 2:
                VIAGetPanelSizeFromEDID(pBIOSInfo);
                break;
                           
            default:
                break;
        }

        /* Restore status */
        write_reg(SR1E, VIASR, sr1e);
        write_reg(SR3E, VIASR, sr3e);
        write_reg(CR91, VIACR, cr91);
        write_reg(CR9B, VIACR, cr9b);
    }
    
    VIAGetModeSizeByIndex(pBIOSInfo->TMDSSettingInfo.DFPSize, &pBIOSInfo->TMDSSettingInfo.PanelSizeH, &pBIOSInfo->TMDSSettingInfo.PanelSizeV);
    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "DFPSize = %d\n", pBIOSInfo->TMDSSettingInfo.DFPSize));
    return (unsigned char)(pBIOSInfo->TMDSSettingInfo.DFPSize);
}

Bool VIASetDFPMode(VIABIOSInfoPtr pBIOSInfo)
{
    struct io_register      *reg = NULL;
    int                     load_reg_num=0;
    int                     reg_value=0;
    int                     offset;
	
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIASetDFPMode\n"));

    VideoSetMode(pBIOSInfo, pBIOSInfo->TMDSSettingInfo.IGAPath);

    if ((pBIOSInfo->IsSecondary)||(pBIOSInfo->DuoView))
    {
        /* Second Display Channel Enable */
        /* Modify for sequence issue. */
        EnableSecondDisplayChannel();
        

        /*=* Fixed Set /DVT+CRT/SAMM mode,
             the secondary monitor without display *=*/
        write_reg_mask(CR91, VIACR, 0x0, BIT5);
    }

    /* Disable LCD Scaling */
    if(pBIOSInfo->TMDSSettingInfo.IGAPath == IGA2)
    {
        write_reg(CR79, VIACR, 0x0);
    }

    /* Enable Interlace mode. */
    if (pBIOSInfo->IsEnableInterlaceMode)
    {
        if (pBIOSInfo->TMDSSettingInfo.IGAPath == IGA1)
        {
            write_reg_mask(CR33, VIACR, BIT6, BIT6);
        }
        else if (pBIOSInfo->TMDSSettingInfo.IGAPath == IGA2)
        {
            write_reg_mask(CR67, VIACR, BIT5, BIT5);
        }
    }
        
    if(pBIOSInfo->EPIA_DVI && pBIOSInfo->DuoView == FALSE && pBIOSInfo->SAMM== FALSE)
    {
        VideoSetMode(pBIOSInfo, pBIOSInfo->TMDSSettingInfo.IGAPath);
        switch(pBIOSInfo->TMDSSettingInfo.ModeIndex)
        {
            case M640x480:
                offset = 80;
                break;
            case M800x480:
            case M800x600:
                offset = 110;
                break;
            case M1024x768:
                offset = 150;
                break;
            case M1280x768:
            case M1280x800:
            case M1280x1024:
            case M1440x1050:
                offset = 190;
                break;
            case M1600x1200:
                offset = 250;
                break;
            case M1366x768:
            case M1360x768:
                offset = 212;                             
                break;
            default:
                offset = 140;
                break;
        }
        /* Offset for simultaneous */
        reg_value = offset;
        load_reg_num = offset_reg.iga2_offset_reg.reg_num;
        reg = offset_reg.iga2_offset_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);
       
        write_reg_mask(CR6B, VIACR, 0x08, BIT3);
        write_reg_mask(CR6A, VIACR, 0x00, BIT6);
        write_reg_mask(CR6A, VIACR, 0x40, BIT6);
        write_reg_mask(SR16, VIASR, 0x40, BIT6);
    }

    switch (pBIOSInfo->Chipset)
    {
        case VIA_P4M800PRO:
            if (pBIOSInfo->TMDSSettingInfo.IGAPath == IGA2)
            {
                write_reg(CR9B, VIACR, 0x11);
            }
            else
            {
                if(pBIOSInfo->IsDemoBoard)
                {
                    write_reg(CR9B, VIACR, 0x01);
                }
                else
            	{
                    write_reg_mask(CR97, VIACR, 0x01, BIT0);
            	}
            }
            break;
        default:
            break;
    }

    VIASetOutputPath(pBIOSInfo, VIA_DEVICE_DFP, pBIOSInfo->TMDSSettingInfo.IGAPath, pBIOSInfo->DFPDIPort);

    /* Adjust DPA Setting for clock skew problem: */
    VIADVIPatchSkew(pBIOSInfo, pBIOSInfo->DFPDIPort);

    /* Set display channel. */
    if ((pBIOSInfo->Chipset == VIA_CX700)||
        (pBIOSInfo->Chipset == VIA_VX800))
    {
        VIASetDisplayChannel(pBIOSInfo);
    }
    
    return TRUE;
}

/* If Enable DFP, turn on pad */
void VIAEnableDFP(VIABIOSInfoPtr pBIOSInfo)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIAEnableDFP\n"));
    
    VIABIOSInfoPtr  pVia = pBIOSInfo;

    if (pBIOSInfo->DFPDIPort & VIA_DI_DVP0)
    {
        VGAOUT8(0x3c4, 0x1E);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) | 0xC0);
    }
    if (pBIOSInfo->DFPDIPort & VIA_DI_DVP1)
    {
        VGAOUT8(0x3c4, 0x1E);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) | 0x30);
    }
    if (pBIOSInfo->DFPDIPort & VIA_DI_DFPHIGH)
    {
        VGAOUT8(0x3c4, 0x2A);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) | 0x0C);
    }
    if (pBIOSInfo->DFPDIPort & VIA_DI_DFPLOW)
    {
        VGAOUT8(0x3c4, 0x2A);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) | 0x03);
    }
    if(pBIOSInfo->EPIA_DVI)      /*Support DVI on EPIA*/
    {
        write_reg(CR91, VIACR, 0x1f);
        write_reg(CR88, VIACR, 0x00);
    }
    if (pBIOSInfo->DFPDIPort & VIA_DI_TMDS)
    {
        VGAOUT8(0x3d4, 0x91);
        VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0x7F);   /* Turn on Display period in the panel path. */
        
        VGAOUT8(0x3d4, 0xD2);
        VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0xF7);   /* Turn on TMDS power. */
    }
}

/* If Disable DFP, turn off pad */
void VIADisableDFP(VIABIOSInfoPtr pBIOSInfo)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIADisableDFP\n"));
    
    VIABIOSInfoPtr  pVia = pBIOSInfo;

    if (pBIOSInfo->DFPDIPort & VIA_DI_DVP0)
    {
        VGAOUT8(0x3c4, 0x1E);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) & 0x3F);
    }

    if (pBIOSInfo->DFPDIPort & VIA_DI_DVP1)
    {
        VGAOUT8(0x3c4, 0x1E);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) & 0xCF);
    }

    if (pBIOSInfo->DFPDIPort & VIA_DI_DFPHIGH)
    {
        VGAOUT8(0x3c4, 0x2A);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) & 0xF3);
    }

    if (pBIOSInfo->DFPDIPort & VIA_DI_DFPLOW)
    {
        VGAOUT8(0x3c4, 0x2A);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) & 0xFC);
    }

    if (pBIOSInfo->DFPDIPort & VIA_DI_TMDS)
    {
        VGAOUT8(0x3d4, 0xD2);
        VGAOUT8(0x3d5, VGAIN8(0x3d5) | 0x08);   /* Turn off TMDS power. */
    }
}

void VideoSetMode(VIABIOSInfoPtr pBIOSInfo, int set_iga)
{
    struct  VideoModeTable *VideoMode = NULL;
    struct display_timing   mode_crt_reg;
    struct crt_mode_table   *mode_crt_table = NULL;
    BOOL    dvi_use_modeline = FALSE;
    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VideoSetMode: Set IGA%d Mode!\n", set_iga));

	/* Use the mode line in two cases:
	1. SAMM and Mode line is user-setting
	2. DuoView/single, EDIDType=true, and Mode line is user-setting*/
    if (((pBIOSInfo->SAMM) ||
		 (!pBIOSInfo->SAMM && (pBIOSInfo->EDIDType == VIA_DEVICE_DFP))) && 
		 (pBIOSInfo->ModeLineStatus & (MODE_LINE_TYPE_USER | MODE_LINE_TYPE_EDID)))
	{
        dvi_use_modeline = TRUE;
    }

    VideoMode = &VIARoutineModes[SearchModeSetting(pBIOSInfo->TMDSSettingInfo.ModeIndex)];
    mode_crt_table = VideoMode->crtc;
    mode_crt_reg = mode_crt_table->crtc;

    xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "-----DVI TIMING-----\n");
	/* Write CRTC */
    fill_crtc_timing(pBIOSInfo, mode_crt_table, VideoMode->mode_array, (pBIOSInfo->bitsPerPixel >> 3), VIA_DEVICE_DFP, set_iga, dvi_use_modeline);

    if(pBIOSInfo->EPIA_DVI && (pBIOSInfo->DuoView == FALSE))
    {
        fill_crtc_timing(pBIOSInfo, mode_crt_table, VideoMode->mode_array, (pBIOSInfo->bitsPerPixel >> 3), VIA_DEVICE_DFP, IGA2, dvi_use_modeline);

        if(pBIOSInfo->TMDSSettingInfo.ModeIndex == M640x480 && pBIOSInfo->FoundRefresh == 60)
        {
            /* The border is 8 pixels. */
            mode_crt_reg.hor_blank_start = mode_crt_reg.hor_blank_start - 8;

            /* Blanking time should add left and right borders. */
            mode_crt_reg.hor_blank_end = mode_crt_reg.hor_blank_end + 16;
        }
        
        load_crtc_shadow_timing(pBIOSInfo, mode_crt_reg, mode_crt_reg);       /* Fill shadow registers */
        load_fetch_count_reg(mode_crt_reg.hor_addr, 4, IGA2);           /* Fetch count for simultaneous*/        
    }
    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VideoSetMode --Done--\n"));
}

unsigned char VIAGetDeviceDetectDVI(VIABIOSInfoPtr pBIOSInfo)
{
    unsigned char   ret = 0x0;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIAGetDeviceDetectDVI\n"));

    if(pBIOSInfo->TMDSSettingInfo.ChipID)
    {
        if(VIASenseDVI(pBIOSInfo))
        {
            ret = VIA_DEVICE_DFP;
            DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "DVI has Attachment.\n"));
            if(pBIOSInfo->TMDSSettingInfo.DFPSize == M_INVALID)
                VIAGetPanelInfo(pBIOSInfo);
        }
        else
        {
            DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "DVI hasn't Attachment.\n"));
        }
    }

    return ret;
}

