/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "via_driver.h"
#include "via_video.h"
#include "via_eng_regs.h"
#include "via_eng_reg_fops.h"
#include "debug.h"
#include "xf86.h"
#include "via_dmabuffer.h"


/*
 * In:      Video MMIO base address
 *          dwVideoFlag,Video control flag
 *          dwIndex,Video register offset
 * Out:     dwRegValue,Video register value
 * Ret:
 * Desc:    HQV register read function
 */
unsigned long vfCMHQVR(unsigned long dwIndex,ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    unsigned long *pVideoFlag = &pPriv->videoFlag;
    unsigned long dwRegValue    = 0;
    
    if  (pPriv->hqvEngUpdate == HQV_1){
        dwRegValue = VIDInD(dwIndex|REG_HQV1_INDEX);
    }else{
        dwRegValue = VIDInD( dwIndex);
    }
    
    return dwRegValue;
}


/*
 * In:      Video MMIO base address
 *          dwVideoFlag,Video control flag
 *          dwIndex,Video register offset
 *          dwData,Value to be set to Video register
 *          dwAGPRequest,AGP command buffer allocated flag
 * Out:
 * Ret:
 * Desc:    HQV register write function
 */
void vfCMHQVW(unsigned long dwIndex,unsigned long dwData,ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;      
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;
    unsigned long *pVideoFlag = &pPriv->videoFlag;

#if VIA_APPLY_XVOVERLAY_AGP
    RING_VARS;
#endif

    if (pPriv->hqvEngUpdate == HQV_1){
        dwIndex|=REG_HQV1_INDEX;
    }
    
    if (*pVideoFlag & (VIDEO_VQ_INUSE|VIDEO_AGP_INUSE)){
        dwIndex |= VID_THROUGH_CR_OFFSET;
    }
        
    DBG_DD(("W0 %08lx %08lx dwVideoFlag=%lx\n",dwIndex,dwData,*pVideoFlag));

#if VIA_APPLY_XVOVERLAY_AGP
    if(viaGfxInfo->chipId != VIA_DEVICE_VT3314) {
        OUT_RING_QW(dwIndex, dwData);
    } else
#endif
    VIDOutD( dwIndex, dwData);
}


/*
 * In:      Video MMIO base address
 *          dwVideoFlag,Video control flag
 *          dwIndex,Video register offset
 *          dwData,Value to be set to Video register
 *          dwAGPRequest,AGP command buffer allocated flag
 * Out:
 * Ret:
 * Desc:    Video register write function
 */
void vfCM(unsigned long dwIndex,unsigned long dwData,ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;
    unsigned long *pVideoFlag = &pPriv->videoFlag;

#if VIA_APPLY_XVOVERLAY_AGP
    RING_VARS;
#endif

    /*
     * if hw supports VQ, we use VQ  for both sw and hw decoding case
     * otherwise in PIP mpeg2 and mpeg1 case, time delay introduced by mpeg2 VQ 
     * has sync problems with none time delay in sw decoding 
     */
    if (*pVideoFlag & (VIDEO_VQ_INUSE|VIDEO_AGP_INUSE)){
        dwIndex |= VID_THROUGH_CR_OFFSET;
    }
    
#if VIA_APPLY_XVOVERLAY_AGP
    if(viaGfxInfo->chipId != VIA_DEVICE_VT3314) {
        OUT_RING_QW(dwIndex, dwData);
    } else
#endif
    /* should be disabled in first phase of implementing video agp command */
    VIDOutD( dwIndex, dwData);        
}


/*
 * In:      Video MMIO base address
 *          dwVideoFlag,Video control flag
 *          dwFunSel,Video wait type
 * Out:
 * Ret:
 * Desc:    Video wait function
 */
void vfHM(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv, unsigned long dwFunSel)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;
    unsigned long *pVideoFlag = &pPriv->videoFlag;

    if (*pVideoFlag & (VIDEO_VQ_INUSE | VIDEO_AGP_INUSE)){
        dwFunSel |= HM_WAIT_VIRTUAL_QUEUE;
    }

#if  VIA_APPLY_XVOVERLAY_AGP
    dwFunSel |= HM_WAIT_VIRTUAL_QUEUE;
#endif

    switch (dwFunSel) 
    {
    case HM_WAIT_FIRE:
        if (pPriv->vidEngUpdate == VIDEO_1){
        	vidWaitVideoCommandFire(VIDEO_1,pVia->MapBase);
        }else{	
            vidWaitVideoCommandFire(VIDEO_3,pVia->MapBase);
        }    
        break;
        
    case HM_WAIT_SUBFLIP:
        if (pPriv->hqvEngUpdate == HQV_0){
        	vidWaitSUBFlip(HQV_0, pVia->MapBase);
        }else if (pPriv->hqvEngUpdate == HQV_1){	        
            vidWaitSUBFlip(HQV_1, pVia->MapBase);
        }
        break;
        
    case HM_WAIT_HQV_DONE:
        if (pPriv->hqvEngUpdate == HQV_0){
            vidWaitHQVDone(HQV_0, pVia->MapBase);
        }else if (pPriv->hqvEngUpdate == HQV_1){   
            vidWaitHQVDone(HQV_1, pVia->MapBase);
        }    
        break;
        
    case HW_WAIT_HQV_FIFOEMPTY:
        if (pPriv->hqvEngUpdate == HQV_0){
        	vidWaitHQVFIFOEmpty(HQV_0, pVia->MapBase);
        }else if (pPriv->hqvEngUpdate == HQV_1){	
            vidWaitHQVFIFOEmpty(HQV_1, pVia->MapBase);
        }    
        break;
        
    case HW_WAIT_HQV_FINISH:    
        if (pPriv->hqvEngUpdate == HQV_0){
            vidWaitHQVFinish(HQV_0, pVia->MapBase);
        }else if (pPriv->hqvEngUpdate == HQV_1){   
            vidWaitHQVFinish(HQV_1, pVia->MapBase);
        }
        break;
        
        /*hardware wait.only in CME path*/
    case HM_WAIT_FIRE_VQ:
        if (pPriv->vidEngUpdate == VIDEO_1){
            vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_V1_FIRE, pScrn, pPriv);
        }else{	
            vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_V3_FIRE, pScrn, pPriv);
        }    
        break;
        
    case HM_WAIT_SUBFLIP_VQ:
        if (pPriv->vidEngUpdate == VIDEO_1){
            vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_V1_SUBFLIP, pScrn, pPriv);
        }else{	
            vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_V3_SUBFLIP, pScrn, pPriv);
        }    
        break;
        
    case HM_WAIT_HQV_DONE_VQ:/* SW FLIP */
        if (pPriv->vidEngUpdate == VIDEO_1){
            vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_V1_HQV_SW_FLIP, pScrn, pPriv);
        }else{	
            vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_V3_HQV_SW_FLIP, pScrn, pPriv);
        }    
        break;
        
    case HW_WAIT_HQV_FIFOEMPTY_VQ:
        if (pPriv->vidEngUpdate == VIDEO_1){
            vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_V1_HQV_FLIP_FIFO, pScrn, pPriv);
        }else{	
            vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_V3_HQV_FLIP_FIFO, pScrn, pPriv);
        }    
        break;
        
    /* HW wait for HQV finished. */
    case HW_WAIT_HQV_FINISH_VQ:
        if (pPriv->vidEngUpdate == VIDEO_1){
            vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_V1_HQV_DONE, pScrn, pPriv);
        }else{
            vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_V3_HQV_DONE, pScrn, pPriv);
        }
        break;
        
    case HM_WAIT_3D_IDLE:
        /* vidWait3DIdle(pVia->MapBase); */
        /* use the uniform wait chip idle function provided instead */
        pVia->myWaitIdle(pVia);
        break;
        
    case HM_WAIT_3D_IDLE_VQ:
#if VIA_APPLY_XVOVERLAY_AGP
        switch(viaGfxInfo->chipId) {
            /* support 4-cmd CR Lock with register 0x60/0x3260 */
            case VIA_DEVICE_VT3324:
                {
                    RING_VARS;
                    static unsigned long Local_Working_Buffer_ID = 0;
                    unsigned long dwSyncSetting =0;
                    if(IN_VIDEO_HEADER4) {
                        END_HEADER4_VIDEOAGP;
                    } else if(IN_VIDEO_HEADER5) {
                        END_HEADER5_VIDEOAGP;
                    }
                    dwSyncSetting = HWSYMC_3D_2D_Start;
                    dwSyncSetting |= Local_Working_Buffer_ID << 24;
                    dwSyncSetting |= (pPriv->vidEngUpdate == VIDEO_1)? \
                            HWSYMC_3D_2D_comad_stream_kinds_HQV0 : \
                            HWSYMC_3D_2D_comad_stream_kinds_HQV1;
                    BEGIN_HEAD1_2D(REG_2D3D_SYNCCTL, dwSyncSetting);
                    dwSyncSetting = HWSYMC_3D_2D_End_and_wait_3D_Idle;
                    dwSyncSetting |= Local_Working_Buffer_ID << 24;
                    dwSyncSetting |= (pPriv->vidEngUpdate == VIDEO_1)? \
                            HWSYMC_3D_2D_comad_stream_kinds_HQV0 : \
                            HWSYMC_3D_2D_comad_stream_kinds_HQV1;
                    BEGIN_HEAD1_2D(REG_2D3D_SYNCCTL, dwSyncSetting);

                    BEGIN_HEADER5_VIDEOAGP(2);
                    dwSyncSetting = HWSYMC_DVD_VIDEO_Start;
                    dwSyncSetting |= Local_Working_Buffer_ID << 24;
                    dwSyncSetting |= (pPriv->vidEngUpdate == VIDEO_1)? \
                            HWSYMC_DVD_VIDEO_comad_stream_kinds_HQV0 :\
                            HWSYMC_DVD_VIDEO_comad_stream_kinds_HQV1;
                    vfCM(REG_VID_SYNCCTL, dwSyncSetting, pScrn, pPriv);
                    dwSyncSetting = HWSYMC_DVD_VIDEO_End;
                    dwSyncSetting |= Local_Working_Buffer_ID << 24;
                    dwSyncSetting |= (pPriv->vidEngUpdate == VIDEO_1)? \
                            HWSYMC_DVD_VIDEO_comad_stream_kinds_HQV0 :\
                            HWSYMC_DVD_VIDEO_comad_stream_kinds_HQV1;
                    vfCM(REG_VID_SYNCCTL, dwSyncSetting, pScrn, pPriv);
                    END_HEADER5_VIDEOAGP;
                    
                    Local_Working_Buffer_ID ++;   
                    Local_Working_Buffer_ID &= CR_WorkingBufferMask;
                } 
                break;

            /* support 1-cmd CR Lock with register 0x326c */
            case VIA_DEVICE_VT3364:
            case VIA_DEVICE_VT3353:
            case VIA_DEVICE_VT3409:
                {
                    Bool newHeader5 = TRUE;
                    RING_VARS;
                    
                    if(IN_VIDEO_HEADER4) {
                        END_HEADER4_VIDEOAGP;
                    } else if(IN_VIDEO_HEADER5) {
                        newHeader5 = FALSE; 
                    }
                    
                    if(newHeader5){
                        BEGIN_HEADER5_VIDEOAGP(1);
                    }
                    vfCM(REG_HW_WAIT_VID_CME, VAL_HM_WAIT_2D3D, pScrn, pPriv);
                    if(newHeader5){
                        END_HEADER5_VIDEOAGP;
                    }
                }
                break;
                
            default:
                break;
        }
#else
        /* vidWait3DIdle(pVia->MapBase); */
        /* use the uniform wait chip idle function provided instead */
        pVia->myWaitIdle(pVia);
#endif
        break;

    case HM_WAIT_VQ_EMPTY:
    case HM_WAIT_VQ_EMPTY_VQ:
        vidWaitVirtualQueueEmpty(pVia->MapBase);
        break;
        
    default:
        DBG_DD(("vfHM :Undefined wait command!\n"));
        break;
    }
}


/*HQV register write*/
void cmdWriteHQV(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv, unsigned long hqvIndex)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    LPVIDEOHWINFO lphwinfo  = &pPriv->vidHWinfo;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    unsigned long dwIndex=0;

    if(hqvIndex == HQV_1){
        dwIndex |= REG_HQV1_INDEX;
    }

    if(viaGfxInfo->chipId == VIA_DEVICE_VT3353)
    {
        /* 0x380 */
        vfCM(HQV_SRC_DATA_OFFSET_CONTROL1|dwIndex, lphwinfo->dwHQVSrcDataOffsetStart, pScrn, pPriv);
        /* 0x388 */
        vfCM(HQV_SRC_DATA_OFFSET_CONTROL3|dwIndex, lphwinfo->dwHQVSrcDataOffsetEnd, pScrn, pPriv);

        /* VT3353 has register for subpicture start/end setting */
        /* 0x384 */
        vfCM(HQV_SRC_DATA_OFFSET_CONTROL2|dwIndex, lphwinfo->dwHQVSrcDataOffsetStart, pScrn, pPriv);
        /* 0x38c */
        vfCM(HQV_SRC_DATA_OFFSET_CONTROL4|dwIndex, lphwinfo->dwHQVSrcDataOffsetEnd, pScrn, pPriv);
        /* 0x3cc*/
        vfCM(HQV_SRC_OFFSET|dwIndex, lphwinfo->dwHQVOffset,pScrn, pPriv);

    }


    if(viaGfxInfo->chipId == VIA_DEVICE_VT3409)
    {
        /* 0x370 */
         vfCM(HQV_SRC_DATA_OFFSET_CTRL1_409|dwIndex, lphwinfo->dwHQVSrcDataOffsetStart, pScrn, pPriv);
        /* 0x378 */
         vfCM(HQV_SRC_DATA_OFFSET_CTRL3_409|dwIndex, lphwinfo->dwHQVSrcDataOffsetEnd, pScrn, pPriv);
        /* 0x374 */
        vfCM(HQV_SRC_DATA_OFFSET_CTRL2_409|dwIndex, lphwinfo->dwHQVSrcDataOffsetStart, pScrn, pPriv);
        /* 0x37c */
        vfCM(HQV_SRC_DATA_OFFSET_CTRL4_409|dwIndex, lphwinfo->dwHQVSrcDataOffsetEnd, pScrn, pPriv);
        
        /* 0x380 */
         vfCM(HQV_DST_DATA_OFFSET_CTRL1_409|dwIndex, lphwinfo->dwHQVDstDataOffsetStart, pScrn, pPriv);
        /* 0x388 */
         vfCM(HQV_DST_DATA_OFFSET_CTRL3_409|dwIndex, lphwinfo->dwHQVDstDataOffsetEnd, pScrn, pPriv);
        /* 0x384 */
        vfCM(HQV_DST_DATA_OFFSET_CTRL2_409|dwIndex, lphwinfo->dwHQVDstDataOffsetStart, pScrn, pPriv);
        /* 0x38c */
        vfCM(HQV_DST_DATA_OFFSET_CTRL4_409|dwIndex, lphwinfo->dwHQVDstDataOffsetEnd, pScrn, pPriv);
        /* 0x3cc */
        vfCM(HQV_BACKGROUND_DATA_OFFSET_409|dwIndex,lphwinfo->dwHQVBgOffset,pScrn, pPriv);
        /*no scaling for subpictures for the time being*/
        /*color adjustment*/
        vfCM(HQV_COLOR_ADJUSTMENT_PRE_CTRL1|dwIndex,lphwinfo->dwHQVCAC[0],pScrn, pPriv);
        vfCM(HQV_COLOR_ADJUSTMENT_PRE_CTRL2|dwIndex,lphwinfo->dwHQVCAC[1],pScrn, pPriv);    
        vfCM(HQV_COLOR_ADJUSTMENT_PRE_CTRL3|dwIndex,lphwinfo->dwHQVCAC[2],pScrn, pPriv);        
        vfCM(HQV_COLOR_ADJUSTMENT_PRE_CTRL4|dwIndex,lphwinfo->dwHQVCAC[3],pScrn, pPriv);
        vfCM(HQV_COLOR_ADJUSTMENT_CTRL1|dwIndex,lphwinfo->dwHQVCAC[4],pScrn, pPriv);
        vfCM(HQV_COLOR_ADJUSTMENT_CTRL2|dwIndex,lphwinfo->dwHQVCAC[5],pScrn, pPriv);
        vfCM(HQV_COLOR_ADJUSTMENT_CTRL3|dwIndex,lphwinfo->dwHQVCAC[6],pScrn, pPriv);
        vfCM(HQV_COLOR_ADJUSTMENT_CTRL4|dwIndex,lphwinfo->dwHQVCAC[7],pScrn, pPriv);
        vfCM(HQV_STATIC_RECORD_FB_STRIDE|dwIndex,lphwinfo->dwSRFStride,pScrn,pPriv);
        vfCM(HQV_EXTENDED_CONTROL|dwIndex,lphwinfo->dwHQVExtCtrl,pScrn, pPriv);

        /*0x3e0*/
        vfCM(HQV_SUBP_HSCALE_CTRL|dwIndex,0,pScrn, pPriv);
        /*0x3e8*/
        vfCM(HQV_SUBP_VSCALE_CTRL|dwIndex,0,pScrn, pPriv);
    }

    if( (viaGfxInfo->chipId == VIA_DEVICE_VT3353) ||
        (viaGfxInfo->chipId == VIA_DEVICE_VT3409) )
    {
        /* 0x3B8 */
        vfCM(HQV_DEFAULT_VIDEO_COLOR|dwIndex, HQV_FIX_COLOR, pScrn, pPriv); 
    }

    /* 0x3d4,2,X,3   Only for Video and MPEG S/W flip use */
    if(pPriv->xv_refine_using_pcie) {
        vfCM(HQV_SRC_STARTADDR_Y|dwIndex, (unsigned long)lphwinfo->dwHQVSrcAddr[0]|LOC_SF, pScrn, pPriv);
    } else {
        vfCM(HQV_SRC_STARTADDR_Y|dwIndex, (unsigned long)lphwinfo->dwHQVSrcAddr[0],pScrn, pPriv);
    }
    
    /* 0x3d8,X,X,4   Only for Video and MPEG S/W flip use */
    if(pPriv->xv_refine_using_pcie) {
        vfCM(HQV_SRC_STARTADDR_U|dwIndex, (unsigned long)lphwinfo->dwHQVSrcAddr[1]|LOC_SF,pScrn, pPriv);
    } else {
        vfCM(HQV_SRC_STARTADDR_U|dwIndex, (unsigned long)lphwinfo->dwHQVSrcAddr[1],pScrn, pPriv);
    }
    
    /* 0x3dc,1,1,1 */
    if (viaGfxInfo->chipId != VIA_DEVICE_VT3314)
    {
	    vfCM(HQV_TILE_CSC_CONTROL|dwIndex, lphwinfo->dwHQVTCGDCtl,pScrn, pPriv);
    }else {
        vfCM(HQV_SRC_STARTADDR_V|dwIndex, (unsigned long)lphwinfo->dwHQVSrcAddr[2],pScrn, pPriv);
    }
    
    /*0x3e0*/
    if (viaGfxInfo->chipId != VIA_DEVICE_VT3409)
    {
        vfCM(HQV_SRC_FETCH_LINE|dwIndex, lphwinfo->dwHQVSrcCountWH,pScrn, pPriv); 
    }

    /* 0x3e4,X,2,2   Only for Capture and MPEG use,we don't use this in Video */
    if (viaGfxInfo->chipId != VIA_DEVICE_VT3314)
    {
         vfCM(HQV_DEINTERLACE_CONTROL|dwIndex, lphwinfo->reg_3e4.dwHQVDeinterlaceCtl,pScrn, pPriv);
    }else{
         vfCM(HQV_FILTER_CONTROL|dwIndex, lphwinfo->reg_3e4.dwHQVFilterCtl,pScrn, pPriv);
    }
    
    /* 0x3e8,17,6 */ 
    if( (viaGfxInfo->chipId == VIA_DEVICE_VT3353) ||
        (viaGfxInfo->chipId == VIA_DEVICE_VT3409) )
    {
        vfCM(HQV_H_SCALE_CONTROL|dwIndex, lphwinfo->dwHQVScaleCtlH,pScrn, pPriv);
        vfCM(HQV_V_SCALE_CONTROL|dwIndex, lphwinfo->dwHQVScaleCtlV,pScrn, pPriv);
    }else{
        vfCM(HQV_MINI_CONTROL|dwIndex, lphwinfo->dwHQVMiniZoomCtl,pScrn, pPriv);    
    }
    	
    /* 0x3ec */
    vfCM(HQV_DST_STARTADDR0|dwIndex, lphwinfo->dwHQVDstAddr[0],pScrn, pPriv);                
    /* 0x3f0 */
    vfCM(HQV_DST_STARTADDR1|dwIndex, lphwinfo->dwHQVDstAddr[1],pScrn, pPriv);
    /* 0x3fc */
    vfCM(HQV_DST_STARTADDR2|dwIndex, lphwinfo->dwHQVDstAddr[2],pScrn, pPriv);                

    /* 0x3f4,7,X,10 */
    vfCM(HQV_DST_STRIDE|dwIndex, lphwinfo->dwHQVDstPitch,pScrn, pPriv);            
    /* 0x3f8,8,X,11 */
    vfCM(HQV_SRC_STRIDE|dwIndex, lphwinfo->dwHQVSrcPitch,pScrn, pPriv);
}


/*
 * In:      dwVideoFlag,Video control flag
 *          lphwinfo,Video register values to be set.
 * Out:
 * Ret:
 * Desc:    Video registers write function
 */
void cmdWriteV1(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{    
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;
    LPVIDEOHWINFO lphwinfo  = &pPriv->vidHWinfo;
    
    DBG_DD(("cmdwriteV1\n"));
    /* 0x254,1,5 */
    vfCM(V1_STARTADDR0, (unsigned long)lphwinfo->dwVSrcAddr[0],pScrn, pPriv);         
    /* 0x238,X,6 */
    vfCM(V1_STARTADDR1, (unsigned long)lphwinfo->dwVSrcAddr[1],pScrn, pPriv);         
    /* 0x248,X,7 */
    vfCM(V1_STARTADDR2, (unsigned long)lphwinfo->dwVSrcAddr[2],pScrn, pPriv);   
    /* 0x23C,2 */
    vfCM(V1_STRIDE, lphwinfo->dwVPitch,pScrn, pPriv);              
    /* 0x240,3,1 */
    vfCM(V1_WIN_START_X_Y, lphwinfo->dwWinStartXY,pScrn, pPriv);         
    /* 0x244,4,2 */
    vfCM(V1_WIN_END_X_Y, lphwinfo->dwWinEndXY,pScrn, pPriv);           
    /* 0x234,5,4 */
    vfCM(V12_QWORD_PER_LINE, lphwinfo->dwFetch,pScrn, pPriv);   

    /* 0x24C,9,9 */
    vfCM(V1_ZOOM_CONTROL, lphwinfo->dwzoomCtl,pScrn, pPriv);
    /* 0x26C,6,3 */
    vfCM(V1_DISPLAY_COUNT, lphwinfo->dwVDisplayCount,pScrn, pPriv);
    
    /* 0x2C0,8,8 */
    vfCM(V1_MINI_CONTROL, lphwinfo->dwminiCtl,pScrn, pPriv);        
    /* 0x258,7,14 */
    vfCM(V_FIFO_CONTROL, lphwinfo->dwV1FIFO,pScrn, pPriv);
    /* 0x220,10,10 */
    vfCM(V_COLOR_KEY, lphwinfo->dwPriDispColorKey, pScrn, pPriv);    
    /* 0x264,11,11   Only for v1 use LCD */
    vfCM(SND_COLOR_KEY, lphwinfo->dwSndDispColorKey, pScrn, pPriv);       
    /* 0x228,13,13   Not use currently */
    vfCM(V_CHROMAKEY_LOW, lphwinfo->dwChromaKeyLow,pScrn, pPriv);            
    /* 0x22c,12,12   Not use currently */
    vfCM(V_CHROMAKEY_HIGH, lphwinfo->dwChromaKeyHigh,pScrn, pPriv);            
}


/*
 * In:      dwVideoFlag,Video control flag
 *          lphwinfo,Video register values to be set.
 * Out:
 * Ret:
 * Desc:    Video registers write function, used for Video,Capture,MPEG
 */
void cmdWriteV3(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{   
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;
    LPVIDEOHWINFO lphwinfo  = &pPriv->vidHWinfo;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    

    DBG_DD(("cmdwriteV3")); 
    /* 0x2A4,1,5 */
    vfCM(V3_STARTADDR_0, (unsigned long)lphwinfo->dwVSrcAddr[0],pScrn, pPriv);
    lphwinfo->dwV3SrcAddr[0]=lphwinfo->dwVSrcAddr[0];
    /* 0x2A8,X,6 */
    vfCM(V3_STARTADDR_1, (unsigned long)lphwinfo->dwVSrcAddr[1],pScrn, pPriv);
    lphwinfo->dwV3SrcAddr[1]=lphwinfo->dwVSrcAddr[1];
    /* 0x29C,X,7 */
    vfCM(V3_STARTADDR_2, (unsigned long)lphwinfo->dwVSrcAddr[2],pScrn, pPriv);
    lphwinfo->dwV3SrcAddr[2]=lphwinfo->dwVSrcAddr[2];
    /* 0x2AC,2    Capture set this register in UpdateVPort_CME() */
    vfCM(V3_STRIDE, lphwinfo->dwVPitch,pScrn, pPriv); 
    /* 0x2B0,3,1 */
    vfCM(V3_WIN_START_X_Y, lphwinfo->dwWinStartXY,pScrn, pPriv);
    /* 0x2B4,4,2 */
    vfCM(V3_WIN_END_X_Y, lphwinfo->dwWinEndXY,pScrn, pPriv);
    /* 0x2B8,5,4 */
    vfCM(V3_ALPHA_FETCH_COUNT, lphwinfo->dwFetch,pScrn, pPriv);

    /* 0x2BC,10,9 */
    vfCM(V3_ZOOM_CONTROL, lphwinfo->dwzoomCtl,pScrn, pPriv);
    /* 0x27C,6,3 */
    vfCM(V3_DISPLAY_COUNT, lphwinfo->dwVDisplayCount,pScrn, pPriv);
    
    /* 0x2C0,9,8 */
    vfCM(V3_MINI_CONTROL, lphwinfo->dwminiCtl,pScrn, pPriv);
    /* 0x278,7,13 */
    vfCM(ALPHA_V3_FIFO_CONTROL, lphwinfo->dwV3FIFO,pScrn, pPriv);
    /* 0x268,8,14 */
    vfCM(ALPHA_V3_PREFIFO_CONTROL, lphwinfo->dwV3PreFIFO,pScrn, pPriv);

    switch (viaGfxInfo->chipId)
    {
        /*v3 exist*/
        case VIA_DEVICE_VT3314:
            /* 0x220 */
            vfCM(V_COLOR_KEY, lphwinfo->dwPriDispColorKey, pScrn, pPriv);
            /*0x264*/
            vfCM(SND_COLOR_KEY, lphwinfo->dwSndDispColorKey, pScrn, pPriv);  
            break;
                                        
        /*v1 v3 both exist*/
        case VIA_DEVICE_VT3324:
        case VIA_DEVICE_VT3409:
        case VIA_DEVICE_VT3353:                        
            /* 0x280 */
            vfCM(V3_COLOR_KEY, lphwinfo->dwPriDispSndColorKey,pScrn, pPriv);
            break;
                                    
        default:                    
            break;    
    }

    /*0x228,13,12   Not use currently*/
    vfCM(V_CHROMAKEY_LOW, lphwinfo->dwChromaKeyLow,pScrn, pPriv);  
    /* 0x22c,12,11   //Not use currently */
    vfCM(V_CHROMAKEY_HIGH, lphwinfo->dwChromaKeyHigh,pScrn, pPriv);
}


unsigned long vidWriteHQVandVideoCmd(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    unsigned long *pVideoFlag = &pPriv->videoFlag;

   /*In duoview,when V3 is being updated,
    * we don't need to write HQV0 again.HQV0 is now bounded with V1 as
    * HQV HW triggers V1
    */
    if(pPriv->hqvEngUpdate == HQV_0){
        if ( (pPriv->vidEngUpdate== VIDEO_1)
            ||(viaGfxInfo->chipId == VIA_DEVICE_VT3314)){
                cmdWriteHQV(pScrn, pPriv, HQV_0);
            }
    }
    else if(pPriv->hqvEngUpdate == HQV_1){
        cmdWriteHQV(pScrn, pPriv, HQV_1);
    }

    if (pPriv->vidEngUpdate == VIDEO_1){
        cmdWriteV1(pScrn, pPriv);
    }else if (pPriv->vidEngUpdate == VIDEO_3){
        cmdWriteV3(pScrn, pPriv);
    }
    return TRUE;
}

