/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
 

/*************************************************************************
 *
 *  File:       via_exa_h5.c
 *  Content:    2D acceleration function for VIA/S3G UniChrome
 *
 ************************************************************************/
#include "via_exa_h5.h"


void viaInit2DState_H5(ViaTwodContext *td)
{
    td->setModeHelper = viaAccelSetMode_H5;
    td->planeMaskHelper = viaAccelPlaneMaskHelper_H5;
    td->transparentHelper = viaAccelTransparentHelper_H5;
    td->copyHelper = viaAccelCopyHelper_H5;
    td->solidHelper = viaAccelSolidHelper_H5;
}

/*
 * Wait for the value to get blitted, or in the PCI case for engine idle.
 */
void
viaAccelWaitMarker_H5(ScreenPtr pScreen, int marker)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    CARD32 uMarker = marker;

    if (!pVia->IsPCI) {
        while ((pVia->lastMarkerRead - uMarker) > (1 << 24))
            pVia->lastMarkerRead = *(volatile CARD32 *)pVia->markerBuf;
    } else {
        viaAccelSync_H5(pScrn);
    }   
}


/*
 * Wait for acceleration engines idle. An expensive way to sync.
 */
void
viaAccelSync_H5(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    int loop = 0;

    mem_barrier();

	
    switch (pVia->ChipId) {
        case PCI_CHIP_VT3336:
        case PCI_CHIP_VT3364:
            while ((VIAGETREG(VIA_REG_STATUS) &
                    (VIA_CMD_RGTR_BUSY_H5| VIA_2D_ENG_BUSY_H5))
                   && (loop++ < MAXLOOP)) ;
            break;
        default:
            break;
    }
}

/*
 * Mark Sync using the 2D blitter for AGP. NoOp for PCI.
 * In the future one could even launch a NULL PCI DMA command
 * to have an interrupt generated, provided it is possible to
 * write to the PCI DMA engines from the AGP command stream.
 */
 
int
viaAccelMarkSync_H5(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);

    RING_VARS;

    ++pVia->curMarker;

    /* Wrap around without affecting the sign bit. */
    pVia->curMarker &= 0x7FFFFFFF;
    if (!pVia->IsPCI) {
        Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
        BEGIN_HEADER0_2D_H5(1);
        OUT_RING_QW(VIA_REG_KEYCONTROL, 0x00);
        viaAccelSolidHelper_H5(cb, 0, 0, 1, 1, pVia->markerOffset,
                            VIA_GEM_32bpp, 4, pVia->curMarker,
                            (0xF0 << 24) | VIA_GEC_BLT | VIA_GEC_FIXCOLOR_PAT);
        Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
        ADVANCE_RING;
    }
    return pVia->curMarker;
}



/*
 * Emit a solid blit operation to the command buffer. 
 */
static void
viaAccelSolidHelper_H5(ViaCommandBuffer * cb, int x, int y, int w, int h,
                    unsigned fbBase, CARD32 mode, unsigned pitch,
                    CARD32 fg, CARD32 cmd)
{
    BEGIN_HEADER0_2D_H5(7);
    OUT_RING_QW(VIA_REG_GEMODE, mode);
    OUT_RING_QW(VIA_REG_DSTBASE, fbBase >> 3);
    OUT_RING_QW(VIA_REG_PITCH, VIA_PITCH_ENABLE | (pitch >> 3) << 16);
    OUT_RING_QW(VIA_REG_DSTPOS, (y << 16) | (x & 0xFFFF));
    OUT_RING_QW(VIA_REG_DIMENSION, ((h - 1) << 16) | (w - 1));
    OUT_RING_QW(VIA_REG_FGCOLOR, fg);
    OUT_RING_QW(VIA_REG_GECMD, cmd);
}

Bool
viaExaPrepareCopy_H5(PixmapPtr pSrcPixmap, PixmapPtr pDstPixmap, int xdir,
                  int ydir, int alu, Pixel planeMask)
{
    ScrnInfoPtr pScrn = xf86Screens[pDstPixmap->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    ViaTwodContext *tdc = &pVia->td;

    RING_VARS;

    if (pSrcPixmap->drawable.bitsPerPixel != pDstPixmap->drawable.bitsPerPixel)
        return FALSE;

    if ((tdc->srcPitch = exaGetPixmapPitch(pSrcPixmap)) & 3)
        return FALSE;

    if (exaGetPixmapPitch(pDstPixmap) & 7)
        return FALSE;

    tdc->srcOffset = exaGetPixmapOffset(pSrcPixmap) + pScrn->fbOffset;
    tdc->cmd = VIA_GEC_BLT | VIAACCELCOPYROP(alu);
    if (xdir < 0)
        tdc->cmd |= VIA_GEC_DECX;
    if (ydir < 0)
        tdc->cmd |= VIA_GEC_DECY;

    if (!viaAccelSetMode_H5(pDstPixmap->drawable.bitsPerPixel, tdc))
        return FALSE;

    if (!viaAccelPlaneMaskHelper_H5(tdc, planeMask))
        return FALSE;

    return TRUE;
}



/*
 * Emit transparency state and color to the command buffer.
 */
static void
viaAccelTransparentHelper_H5(ViaTwodContext * tdc, ViaCommandBuffer * cb,
                          CARD32 keyControl, CARD32 transColor,
                          Bool usePlaneMask)
{
    tdc->keyControl &= ((usePlaneMask) ? 0xF0000000 : 0x00000000);
    tdc->keyControl |= (keyControl & 0x0FFFFFFF);
    BEGIN_HEADER0_2D_H5(1);
    OUT_RING_QW(VIA_REG_KEYCONTROL, tdc->keyControl);
    if (keyControl) {
    	BEGIN_HEADER0_2D_H5(1);
        OUT_RING_QW(VIA_REG_SRCCOLORKEY, transColor);
    }
}


void
viaExaCopy_H5(PixmapPtr pDstPixmap, int srcX, int srcY, int dstX, int dstY,
           int width, int height)
{
    ScrnInfoPtr pScrn = xf86Screens[pDstPixmap->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    ViaTwodContext *tdc = &pVia->td;
    CARD32 srcOffset = tdc->srcOffset;
    CARD32 dstOffset = exaGetPixmapOffset(pDstPixmap) + pScrn->fbOffset;
        
    RING_VARS;

    if (!width || !height)
        return;
    Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
    viaAccelTransparentHelper_H5(tdc, cb, 0x0, 0x0, TRUE);
    viaAccelCopyHelper_H5(cb, srcX, srcY, dstX, dstY, width, height,
                       srcOffset, dstOffset, tdc->mode, tdc->srcPitch,
                       exaGetPixmapPitch(pDstPixmap), tdc->cmd);
    Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
    ADVANCE_RING;
}



/*
 * Emit a copy blit operation to the command buffer.
 */
static void
viaAccelCopyHelper_H5(ViaCommandBuffer * cb, int xs, int ys, int xd, int yd,
                   int w, int h, unsigned srcFbBase, unsigned dstFbBase,
                   CARD32 mode, unsigned srcPitch, unsigned dstPitch,
                   CARD32 cmd)
{
    if (cmd & VIA_GEC_DECY) {
        ys += h - 1;
        yd += h - 1;
    }

    if (cmd & VIA_GEC_DECX) {
        xs += w - 1;
        xd += w - 1;
    }

    BEGIN_HEADER0_2D_H5(8);
    OUT_RING_QW(VIA_REG_GEMODE, mode);
    OUT_RING_QW(VIA_REG_SRCBASE, srcFbBase >> 3);
    OUT_RING_QW(VIA_REG_DSTBASE, dstFbBase >> 3);
    OUT_RING_QW(VIA_REG_PITCH, VIA_PITCH_ENABLE |
                ((dstPitch >> 3) << 16) | (srcPitch >> 3));
    OUT_RING_QW(VIA_REG_SRCPOS, (ys << 16) | (xs & 0xFFFF));
    OUT_RING_QW(VIA_REG_DSTPOS, (yd << 16) | (xd & 0xFFFF));
    OUT_RING_QW(VIA_REG_DIMENSION, ((h - 1) << 16) | (w - 1));
    OUT_RING_QW(VIA_REG_GECMD, cmd);
}

void
viaExaDoneSolidCopy_H5(PixmapPtr pPixmap)
{
}


Bool
viaExaPrepareSolid_H5(PixmapPtr pPixmap, int alu, Pixel planeMask, Pixel fg)
{
    ScrnInfoPtr pScrn = xf86Screens[pPixmap->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    ViaTwodContext *tdc = &pVia->td;

    RING_VARS;

    if (exaGetPixmapPitch(pPixmap) & 7)
        return FALSE;

    if (!viaAccelSetMode_H5(pPixmap->drawable.bitsPerPixel, tdc))
        return FALSE;

    if (!viaAccelPlaneMaskHelper_H5(tdc, planeMask))
        return FALSE;

    tdc->cmd = VIA_GEC_BLT | VIA_GEC_FIXCOLOR_PAT | VIAACCELPATTERNROP(alu);

    tdc->fgColor = fg;

    return TRUE;
}


void
viaExaSolid_H5(PixmapPtr pPixmap, int x1, int y1, int x2, int y2)
{
    ScrnInfoPtr pScrn = xf86Screens[pPixmap->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    ViaTwodContext *tdc = &pVia->td;
    CARD32 dstPitch, dstOffset;

    RING_VARS;

    int w = x2 - x1, h = y2 - y1;

    dstPitch = exaGetPixmapPitch(pPixmap);
    dstOffset = exaGetPixmapOffset(pPixmap) + pScrn->fbOffset;
    Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
    viaAccelTransparentHelper_H5(tdc, cb, 0x0, 0x0, TRUE);
    viaAccelSolidHelper_H5(cb, x1, y1, w, h, dstOffset,
                        tdc->mode, dstPitch, tdc->fgColor, tdc->cmd);
    Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
    ADVANCE_RING;
}



/*
 * Update our 2D state (TwoDContext) with a new mode.
 */
Bool
viaAccelSetMode_H5(int bpp, ViaTwodContext * tdc)
{
    switch (bpp) {
        case 16:
            tdc->mode = VIA_GEM_16bpp;
            tdc->bytesPPShift = 1;
            return TRUE;
        case 32:
            tdc->mode = VIA_GEM_32bpp;
            tdc->bytesPPShift = 2;
            return TRUE;
        case 8:
            tdc->mode = VIA_GEM_8bpp;
            tdc->bytesPPShift = 0;
            return TRUE;
        default:
            tdc->bytesPPShift = 0;
            return FALSE;
    }
}



/*
 * Check if we can use a planeMask and update the 2D context accordingly.
 */
static Bool
viaAccelPlaneMaskHelper_H5(ViaTwodContext * tdc, CARD32 planeMask)
{
    CARD32 modeMask = (1 << ((1 << tdc->bytesPPShift) << 3)) - 1;
    CARD32 curMask = 0x00000000;
    CARD32 curByteMask;
    int i;

    if ((planeMask & modeMask) != modeMask) {

        /* Masking doesn't work in 8bpp. */
        if (modeMask == 0xFF) {
            tdc->keyControl &= 0x0FFFFFFF;
            return FALSE;
        }

        /* Translate the bit planemask to a byte planemask. */
        for (i = 0; i < (1 << tdc->bytesPPShift); ++i) {
            curByteMask = (0xFF << (i << 3));

            if ((planeMask & curByteMask) == 0) {
                curMask |= (1 << i);
            } else if ((planeMask & curByteMask) != curByteMask) {
                tdc->keyControl &= 0x0FFFFFFF;
                return FALSE;
            }
        }
        ErrorF("DEBUG: planeMask 0x%08x, curMask 0%02x\n",
               (unsigned)planeMask, (unsigned)curMask);

        tdc->keyControl = (tdc->keyControl & 0x0FFFFFFF) | (curMask << 28);
    }

    return TRUE;
}


Bool
viaExaCheckComposite_H5(int op, PicturePtr pSrcPicture,
                     PicturePtr pMaskPicture, PicturePtr pDstPicture)
{
    ScrnInfoPtr pScrn = xf86Screens[pDstPicture->pDrawable->pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    Via3DState *v3d = &pVia->v3d;

    /* Reject small composites early. They are done much faster in software. */
    if (!pSrcPicture->repeat &&
        pSrcPicture->pDrawable->width *
        pSrcPicture->pDrawable->height < VIA_MIN_COMPOSITE)
        return FALSE;

    if (pMaskPicture &&
        !pMaskPicture->repeat &&
        pMaskPicture->pDrawable->width *
        pMaskPicture->pDrawable->height < VIA_MIN_COMPOSITE)
        return FALSE;

    if (pMaskPicture && pMaskPicture->componentAlpha) {
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "use componentAlpha \n"));
        if((op==PictOpAdd)||(op==PictOpOver)||(op==PictOpAtop)||(op==PictOpAtopReverse)||(op==PictOpXor))
        return FALSE;
    }

	if(pMaskPicture && pMaskPicture->repeatType > RepeatReflect){
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Unsupported picture repeat %d\n", pMaskPicture->repeatType));
        return FALSE;
	}

	if (pMaskPicture && (pMaskPicture->filter > PictFilterBest))
    {
        ErrorF("Unsupported filter 0x%x\n", pMaskPicture->filter);
		return FALSE;
    }
	
    if (!v3d->opSupported(op)) {
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Operator not supported \n"));
        return FALSE;
    }

    if (((pSrcPicture->pDrawable->width > TEX_DIMENSION_LIMIT_INPIXEL ||
        pSrcPicture->pDrawable->height > TEX_DIMENSION_LIMIT_INPIXEL) || 
        (pMaskPicture && (pMaskPicture->pDrawable->width > TEX_DIMENSION_LIMIT_INPIXEL ||
        pMaskPicture->pDrawable->height > TEX_DIMENSION_LIMIT_INPIXEL))) &&
        (pDstPicture->pDrawable->width > TEX_DIMENSION_LIMIT_INPIXEL || 
        pDstPicture->pDrawable->height > TEX_DIMENSION_LIMIT_INPIXEL)) {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "The Composite is not support the region bigger than 2048 \n");
        return FALSE;
        }
    
    /*
     * FIXME: A8 destination formats are currently not supported and do not
     * seem supported by the hardware, although there are some leftover
     * register settings apparent in the via_3d_reg.h file. We need to fix this
     * (if important), by using component ARGB8888 operations with bitmask.
     */

    if (!v3d->dstSupported(pDstPicture->format)) {
        DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Destination format not supported \n"));
        return FALSE;
    }

    if (v3d->texSupported(pSrcPicture->format)) {
        if (pMaskPicture && (PICT_FORMAT_A(pMaskPicture->format) == 0 ||
                             !v3d->texSupported(pMaskPicture->format))) {
            DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Mask format not supported \n"));
            return FALSE;
        }
        return TRUE;
    }
    DEBUG(xf86DrvMsg(pScrn->scrnIndex, X_INFO, "Src format not supported \n"));
    return FALSE;
}

Bool
viaExaPrepareComposite_H5(int op, PicturePtr pSrcPicture,
                       PicturePtr pMaskPicture, PicturePtr pDstPicture,
                       PixmapPtr pSrc, PixmapPtr pMask, PixmapPtr pDst)
{
    ScrnInfoPtr pScrn = xf86Screens[pDst->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    Via3DState *v3d = &pVia->v3d;
    ViaTextureUnit *vTex = v3d->tex;
    int curTex = 0;
    ViaTexBlendingModes srcMode,maskMode;
    unsigned long offset;
    v3d->componentAlpha = v3d->srcRepeat = v3d->maskRepeat = FALSE;

    offset = exaGetPixmapOffset(pDst) + pScrn->fbOffset;
    v3d->setDestination(v3d, offset/*exaGetPixmapOffset(pDst)*/,
                        exaGetPixmapPitch(pDst), pDstPicture->format);
    if(pMaskPicture && pMaskPicture->componentAlpha)
        v3d->componentAlpha = TRUE;
    else
        v3d->componentAlpha = FALSE;
    v3d->setCompositeOperator(v3d, op, v3d->componentAlpha);
    v3d->setDrawing(v3d, 0x0c, 0xFFFFFFFF, 0x000000FF, 0xFF);

    /*
     * For one-pixel repeat mask pictures we avoid using multitexturing by
     * modifying the src's texture blending equation and feed the pixel
     * value as a constant alpha for the src's texture. Multitexturing on the
     * Unichromes seems somewhat slow, so this speeds up translucent windows.
     */

    srcMode = via_src;
    pVia->maskP = NULL;
    if (pMaskPicture &&
        (pMaskPicture->pDrawable->height == 1) &&
        (pMaskPicture->pDrawable->width == 1) &&
        pMaskPicture->repeat && viaExpandablePixel(pMaskPicture->format)) {
        pMask->devPrivate.ptr = exaGetPixmapOffset(pMask) + pVia->FBBase;
        pVia->maskP = pMask->devPrivate.ptr;
        pVia->componentAlpha = pMaskPicture->componentAlpha;
        v3d->maskRepeat = TRUE;
        srcMode = ((pMaskPicture->componentAlpha)
                   ? via_src_onepix_comp_mask : via_src_onepix_mask);
    }

    if(pMaskPicture) {
        pVia->maskFormat = pMaskPicture->format;
    }

    /*
     * One-Pixel repeat src pictures go as solid color instead of textures.
     * Speeds up window shadows.
     */

    pVia->srcP = NULL;
    if (pSrcPicture && pSrcPicture->repeat
        && (pSrcPicture->pDrawable->height == 1)
        && (pSrcPicture->pDrawable->width == 1)
        && viaExpandablePixel(pSrcPicture->format)) {
        pSrc->devPrivate.ptr = exaGetPixmapOffset(pSrc) + pVia->FBBase;
        pVia->srcP = pSrc->devPrivate.ptr;
        v3d->srcRepeat = TRUE;
    }

    if(pSrcPicture) {
        pVia->srcFormat = pSrcPicture->format;
    }

    /* Exa should be smart enough to eliminate this IN operation. */
    if (pVia->srcP && pVia->maskP) {
        ErrorF("Bad one-pixel IN composite operation. "
               "EXA needs to be smarter.\n");
        return FALSE;
    }
    maskMode=via_mask;
    if(pMaskPicture && pMaskPicture->componentAlpha){
	     switch(op){
            case PictOpOutReverse:
			case PictOpInReverse:
			    srcMode =src_Aa;
			    maskMode =mask_Ca;
			    break;
			case PictOpOverReverse:
			case PictOpIn:
			case PictOpOut:
			case PictOpAdd:
			case PictOpSrc:
			case PictOpDisjointSrc:
			case PictOpConjointSrc:
			    srcMode=src_Ca;
				maskMode =mask_Ca;
				break;
			case PictOpClear:
			case PictOpDisjointClear:
			case PictOpConjointClear:
			case PictOpDst:
			case PictOpDisjointDst:
			case PictOpConjointDst:
            
			    pVia->srcP=NULL;
			    pVia->maskP = NULL;
			    curTex=0;
			    goto AlphaBlendOnly;
			    break;
			default:
			    break;
		 }
		 pVia->srcP=NULL;
		 pVia->maskP = NULL;
    }
    if (!pVia->srcP) {
        offset = exaGetPixmapOffset(pSrc) + pScrn->fbOffset;
        vTex = v3d->tex + curTex;
        vTex->PictureOffset= offset;
        vTex->npot = pVia->nPOT[curTex];
        vTex->textureLevel0Pitch = exaGetPixmapPitch(pSrc);
        vTex->pictureFormat= pSrcPicture->format;
        vTex->textureBlendMode = srcMode;
        vTex->transform = pSrcPicture->transform;
        vTex->bytePerPixel = pSrcPicture->pDrawable->bitsPerPixel >> 3;
        vTex->textureDrawable_width = pSrc->drawable.width;
        vTex->textureDrawable_height = pSrc->drawable.height;
        vTex->pictureFilter= pSrcPicture->filter;
        vTex->textureRepeat = v3d->srcRepeat;
        
        curTex++;
    }

    if (pMaskPicture && !pVia->maskP) {
        offset = exaGetPixmapOffset(pMask) + pScrn->fbOffset;
        vTex = v3d->tex + curTex;                             
        vTex->PictureOffset = offset;
        vTex->npot = pVia->nPOT[curTex];
        vTex->textureLevel0Pitch = exaGetPixmapPitch(pMask);
        vTex->pictureFormat = pMaskPicture->format;
        vTex->textureBlendMode = maskMode;
        vTex->transform = pMaskPicture->transform;
        vTex->bytePerPixel = pMaskPicture->pDrawable->bitsPerPixel >> 3;
        vTex->textureDrawable_width = pMask->drawable.width;
        vTex->textureDrawable_height= pMask->drawable.height;
        vTex->pictureFilter = pMaskPicture->filter;
        vTex->textureRepeat = v3d->maskRepeat;
        curTex++;
    }
AlphaBlendOnly:
    v3d->setFlags(v3d, curTex, FALSE, TRUE, TRUE);

    return TRUE;
}


void
viaExaComposite_H5(PixmapPtr pDst, int srcX, int srcY, int maskX, int maskY,
                int dstX, int dstY, int width, int height)
{
    ScrnInfoPtr pScrn = xf86Screens[pDst->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    Via3DState *v3d = &pVia->v3d;
    CARD32 dstOffset;
    ViaTextureUnit *vTex = v3d->tex;
    CARD32 col, i, j,textureDrawable_width,textureDrawable_height;
    int dst_X_wasted =0,dst_Y_wasted=0;
    v3d->destOffset = exaGetPixmapOffset(pDst) + pScrn->fbOffset;
    v3d->forceUpload = viaCheckUpload(pScrn, v3d);
    RING_VARS;
    
    for(i = 0; i < v3d->numTextures; i++) {
        vTex = v3d->tex + i;
        textureDrawable_width =
            vTex->textureDrawable_width > TEX_DIMENSION_LIMIT_INPIXEL ?
            TEX_DIMENSION_LIMIT_INPIXEL : vTex->textureDrawable_width;
        textureDrawable_height =
            vTex->textureDrawable_height > TEX_DIMENSION_LIMIT_INPIXEL ?
            TEX_DIMENSION_LIMIT_INPIXEL : vTex->textureDrawable_height;
        
        v3d->setTexture(v3d, i, vTex->PictureOffset,
                 vTex->textureLevel0Pitch, vTex->npot,
                 textureDrawable_width,textureDrawable_height,
                 vTex->pictureFormat,
                 via_repeat, via_repeat,
                  vTex->textureBlendMode, FALSE, 
                  vTex->transform, vTex->pictureFilter);
        }


    if (pVia->maskP) {
        viaPixelARGB8888(pVia->maskFormat, pVia->maskP, &col);
        v3d->setTexBlendCol(v3d, 0, pVia->componentAlpha, col);
    }
    if (pVia->srcP) {
        viaPixelARGB8888(pVia->srcFormat, pVia->srcP, &col);
        v3d->setDrawing(v3d, 0x0c, 0xFFFFFFFF, col & 0x00FFFFFF, col >> 24);
        srcX = maskX;
        srcY = maskY;
    }
    
    if (((dstX + width) >= TEX_DIMENSION_LIMIT_INPIXEL) ||
        ((dstY + height) >= TEX_DIMENSION_LIMIT_INPIXEL)) {
        dst_X_wasted = dstX % 256;
        dst_Y_wasted = dstY % 256;
        dstX = dstX - dst_X_wasted;
        dstY = dstY - dst_Y_wasted;
        v3d->destOffset += dstY * v3d->destPitch + dstX * (pDst->drawable.bitsPerPixel >> 3);
        dstX = dst_X_wasted;
        dstY = dst_Y_wasted;
        }

    Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_3D_IDLE);
    v3d->emitState(v3d, &pVia->cb, 1);  /*force upload all Setting*/
    v3d->emitClipRect(v3d, &pVia->cb, dstX, dstY, width,height);
    v3d->emitPixelShader(v3d, &pVia->cb, pVia->srcFormat, 0, 0) ;
    v3d->emitQuad(v3d, &pVia->cb, dstX, dstY, srcX, srcY, maskX, maskY,
                  width, height);
    Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_3D_IDLE);
    ADVANCE_RING;
}



Bool
viaExaTexUploadToScreen_H5(PixmapPtr pDst, int x, int y, int w, int h, char *src,
                        int src_pitch)
{
    ScrnInfoPtr pScrn = xf86Screens[pDst->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    unsigned dstPitch = exaGetPixmapPitch(pDst);
    unsigned wBytes = (w * pDst->drawable.bitsPerPixel + 7) >> 3;
    CARD32 dstOffset;
    char *dst;
    return FALSE;
    #if 0
    dstOffset = x * pDst->drawable.bitsPerPixel;
    if ((dstOffset & 7) || (wBytes & 3))
        return FALSE;
    dst = (char *)pVia->FBBase + (exaGetPixmapOffset(pDst) + y * dstPitch + (dstOffset >> 3));
    while(h--) {
        via_ExaMoveDwords((CARD32*)dst, (CARD32*)src, wBytes/4);

        src += src_pitch;
        dst += dstPitch;
        }
    return TRUE;
    #endif
}

