/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
 

/*************************************************************************
 *
 *  File:       via_exa_h6.c
 *  Content:    2D acceleration function for VIA/S3G UniChrome
 *
 ************************************************************************/
#include "via_exa_h6.h"

void viaInit2DState_H6(ViaTwodContext *td)
{
    td->setModeHelper = viaAccelSetMode_H6;
    td->planeMaskHelper = viaAccelPlaneMaskHelper_H6;
    td->transparentHelper = viaAccelTransparentHelper_H6;
    td->copyHelper = viaAccelCopyHelper_H6;
    td->solidHelper = viaAccelSolidHelper_H6;
}

/*
 * Wait for the value to get blitted, or in the PCI case for engine idle.
 */
void
viaAccelWaitMarker_H6(ScreenPtr pScreen, int marker)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    CARD32 uMarker = marker;

    if (!pVia->IsPCI) {
        while ((pVia->lastMarkerRead - uMarker) > (1 << 24))
            pVia->lastMarkerRead = *(volatile CARD32 *)pVia->markerBuf;
    } else {
        viaAccelSync_H6(pScrn);
    }
}
/*
 * Wait for acceleration engines idle. An expensive way to sync.
 */
void
viaAccelSync_H6(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    int loop = 0;

    mem_barrier();

    switch (pVia->ChipId) {
        case PCI_CHIP_VT3353:
        case PCI_CHIP_VT3409:
            while ((VIAGETREG(VIA_REG_STATUS) &
                    (VIA_CMD_RGTR_BUSY_M1 | VIA_2D_ENG_BUSY_M1 | VIA_3D_ENG_BUSY_M1))
                   && (loop++ < MAXLOOP)) ;
            break;

		default:
			break;
    }
}

/*
 * Mark Sync using the 2D blitter for AGP. NoOp for PCI.
 * In the future one could even launch a NULL PCI DMA command
 * to have an interrupt generated, provided it is possible to
 * write to the PCI DMA engines from the AGP command stream.
 */
 
int
viaAccelMarkSync_H6(ScreenPtr pScreen)
{
    ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);

    RING_VARS;

    ++pVia->curMarker;

    /* Wrap around without affecting the sign bit. */
    pVia->curMarker &= 0x7FFFFFFF;
    if (!pVia->IsPCI) {
        Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
        BEGIN_HEADER0_2D_H5(1);
        OUT_RING_QW(VIA_REG_KEYCONTROL_M1, 0x00);
        viaAccelSolidHelper_H6(cb, 0, 0, 1, 1, pVia->markerOffset,
                            VIA_GEM_32bpp, 4, pVia->curMarker,
                            (0xF0 << 24) | VIA_GEC_BLT | VIA_GEC_FIXCOLOR_PAT);
        Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
        ADVANCE_RING;
    }   
    return pVia->curMarker;    
}

/*
 * Emit a solid blit operation to the command buffer. 
 */
static void
viaAccelSolidHelper_H6(ViaCommandBuffer * cb, int x, int y, int w, int h,
                    unsigned fbBase, CARD32 mode, unsigned pitch,
                    CARD32 fg, CARD32 cmd)
{
    BEGIN_HEADER0_2D_H5(7);
    OUT_RING_QW(VIA_REG_GEMODE_M1, mode);
    OUT_RING_QW(VIA_REG_DSTBASE_M1, fbBase >> 3);
    OUT_RING_QW(VIA_REG_PITCH_M1, (pitch >> 3) << 16);
    OUT_RING_QW(VIA_REG_DSTPOS_M1, (y << 16) | (x & 0xFFFF));
    OUT_RING_QW(VIA_REG_DIMENSION_M1, ((h - 1) << 16) | (w - 1));
    OUT_RING_QW(VIA_REG_MONOPATFGC_M1, fg);
    OUT_RING_QW(VIA_REG_GECMD_M1, cmd);
}

Bool viaExaPrepareCopy_H6 (PixmapPtr pSrcPixmap, PixmapPtr pDstPixmap, int xdir,
                  int ydir, int alu, Pixel planeMask)
{
    ScrnInfoPtr pScrn = xf86Screens[pDstPixmap->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    ViaTwodContext *tdc = &pVia->td;

    RING_VARS;

    if (pSrcPixmap->drawable.bitsPerPixel != pDstPixmap->drawable.bitsPerPixel)
        return FALSE;

    if ((tdc->srcPitch = exaGetPixmapPitch(pSrcPixmap)) & 3)
        return FALSE;

    if (exaGetPixmapPitch(pDstPixmap) & 7)
        return FALSE;

    tdc->srcOffset = exaGetPixmapOffset(pSrcPixmap) + pScrn->fbOffset;

    tdc->cmd = VIA_GEC_BLT | VIAACCELCOPYROP(alu);
    if (xdir < 0)
        tdc->cmd |= VIA_GEC_DECX;
    if (ydir < 0)
        tdc->cmd |= VIA_GEC_DECY;

    if (!viaAccelSetMode_H6(pDstPixmap->drawable.bitsPerPixel, tdc))
        return FALSE;

    if (!viaAccelPlaneMaskHelper_H6(tdc, planeMask))
        return FALSE;

    return TRUE;
}

/*
 * Emit transparency state and color to the command buffer.
 */
static void
viaAccelTransparentHelper_H6 (ViaTwodContext * tdc, ViaCommandBuffer * cb,
                          CARD32 keyControl, CARD32 transColor,
                          Bool usePlaneMask)
{
    tdc->keyControl &= ((usePlaneMask) ? 0xF0000000 : 0x00000000);
    tdc->keyControl |= (keyControl & 0x0FFFFFFF);
    BEGIN_HEADER0_2D_H5(1);
    OUT_RING_QW(VIA_REG_KEYCONTROL_M1, tdc->keyControl);
    if (keyControl) {
	BEGIN_HEADER0_2D_H5(1);
        OUT_RING_QW(VIA_REG_SRCCOLORKEY_M1, transColor);
    }
}

void viaExaCopy_H6 (PixmapPtr pDstPixmap, int srcX, int srcY, int dstX, int dstY,
           int width, int height)
{
    ScrnInfoPtr pScrn = xf86Screens[pDstPixmap->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    ViaTwodContext *tdc = &pVia->td;
    CARD32 srcOffset = tdc->srcOffset;
    CARD32 dstOffset = exaGetPixmapOffset(pDstPixmap) + pScrn->fbOffset;

    RING_VARS;

    if (!width || !height)
        return;
    Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
    viaAccelTransparentHelper_H6(tdc, cb, 0x0, 0x0, TRUE);
    viaAccelCopyHelper_H6(cb, srcX, srcY, dstX, dstY, width, height,
                       srcOffset, dstOffset, tdc->mode, tdc->srcPitch,
                       exaGetPixmapPitch(pDstPixmap), tdc->cmd);
    Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
    ADVANCE_RING;
}



/*
 * Emit a copy blit operation to the command buffer.
 */
static void
viaAccelCopyHelper_H6 (ViaCommandBuffer * cb, int xs, int ys, int xd, int yd,
                   int w, int h, unsigned srcFbBase, unsigned dstFbBase,
                   CARD32 mode, unsigned srcPitch, unsigned dstPitch,
                   CARD32 cmd)
{
    if (cmd & VIA_GEC_DECY) {
        ys += h - 1;
        yd += h - 1;
    }

    if (cmd & VIA_GEC_DECX) {
        xs += w - 1;
        xd += w - 1;
    }

    BEGIN_HEADER0_2D_H5(8);
    OUT_RING_QW(VIA_REG_GEMODE_M1, mode);
    OUT_RING_QW(VIA_REG_SRCBASE_M1, srcFbBase >> 3);
    OUT_RING_QW(VIA_REG_DSTBASE_M1, dstFbBase >> 3);
    OUT_RING_QW(VIA_REG_PITCH_M1, ((dstPitch >> 3) << 16) | (srcPitch >> 3));
    OUT_RING_QW(VIA_REG_SRCPOS_M1, (ys << 16) | (xs & 0xFFFF));
    OUT_RING_QW(VIA_REG_DSTPOS_M1, (yd << 16) | (xd & 0xFFFF));
    OUT_RING_QW(VIA_REG_DIMENSION_M1, ((h - 1) << 16) | (w - 1));
    OUT_RING_QW(VIA_REG_GECMD_M1, cmd);
	
}

void viaExaDoneSolidCopy_H6(PixmapPtr pPixmap)
{
}


Bool viaExaPrepareSolid_H6(PixmapPtr pPixmap, int alu, Pixel planeMask, Pixel fg)
{
    ScrnInfoPtr pScrn = xf86Screens[pPixmap->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    ViaTwodContext *tdc = &pVia->td;

    RING_VARS;

    if (exaGetPixmapPitch(pPixmap) & 7)
        return FALSE;

    if (!viaAccelSetMode_H6(pPixmap->drawable.bitsPerPixel, tdc))
        return FALSE;

    if (!viaAccelPlaneMaskHelper_H6(tdc, planeMask))
        return FALSE;

    tdc->cmd = VIA_GEC_BLT | VIA_GEC_FIXCOLOR_PAT | VIAACCELPATTERNROP(alu);

    tdc->fgColor = fg;

    return TRUE;
}

void viaExaSolid_H6(PixmapPtr pPixmap, int x1, int y1, int x2, int y2)
{
    ScrnInfoPtr pScrn = xf86Screens[pPixmap->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    ViaTwodContext *tdc = &pVia->td;
    CARD32 dstPitch, dstOffset;

    RING_VARS;

    int w = x2 - x1, h = y2 - y1;

    dstPitch = exaGetPixmapPitch(pPixmap);
    dstOffset = exaGetPixmapOffset(pPixmap) + pScrn->fbOffset;
    Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
    viaAccelTransparentHelper_H6(tdc, cb, 0x0, 0x0, TRUE);
    viaAccelSolidHelper_H6(cb, x1, y1, w, h, dstOffset,
                        tdc->mode, dstPitch, tdc->fgColor, tdc->cmd);
    Command_CRSync_2D3D(&pVia->cb,(unsigned int)pVia->ChipId, FLAG_WAIT_2D_IDLE);
    ADVANCE_RING;
}

/*
 * Update our 2D state (TwoDContext) with a new mode.
 */
static Bool
viaAccelSetMode_H6(int bpp, ViaTwodContext * tdc)
{
    switch (bpp) {
        case 16:
            tdc->mode = VIA_GEM_16bpp;
            tdc->bytesPPShift = 1;
            return TRUE;
        case 32:
            tdc->mode = VIA_GEM_32bpp;
            tdc->bytesPPShift = 2;
            return TRUE;
        case 8:
            tdc->mode = VIA_GEM_8bpp;
            tdc->bytesPPShift = 0;
            return TRUE;
        default:
            tdc->bytesPPShift = 0;
            return FALSE;
    }
}

/*
 * Check if we can use a planeMask and update the 2D context accordingly.
 */
static Bool
viaAccelPlaneMaskHelper_H6(ViaTwodContext * tdc, CARD32 planeMask)
{
    CARD32 modeMask = (1 << ((1 << tdc->bytesPPShift) << 3)) - 1;
    CARD32 curMask = 0x00000000;
    CARD32 curByteMask;
    int i;

    if ((planeMask & modeMask) != modeMask) {

        /* Masking doesn't work in 8bpp. */
        if (modeMask == 0xFF) {
            tdc->keyControl &= 0x0FFFFFFF;
            return FALSE;
        }

        /* Translate the bit planemask to a byte planemask. */
        for (i = 0; i < (1 << tdc->bytesPPShift); ++i) {
            curByteMask = (0xFF << (i << 3));

            if ((planeMask & curByteMask) == 0) {
                curMask |= (1 << i);
            } else if ((planeMask & curByteMask) != curByteMask) {
                tdc->keyControl &= 0x0FFFFFFF;
                return FALSE;
            }
        }
        ErrorF("DEBUG: planeMask 0x%08x, curMask 0%02x\n",
               (unsigned)planeMask, (unsigned)curMask);

        tdc->keyControl = (tdc->keyControl & 0x0FFFFFFF) | (curMask << 28);
    }

    return TRUE;
}

Bool
viaExaTexUploadToScreen_H6(PixmapPtr pDst, int x, int y, int w, int h, char *src,
                        int src_pitch)
{
    ScrnInfoPtr pScrn = xf86Screens[pDst->drawable.pScreen->myNum];
    VIAPtr pVia = VIAPTR(pScrn);
    unsigned dstPitch = exaGetPixmapPitch(pDst);
    unsigned wBytes = (w * pDst->drawable.bitsPerPixel + 7) >> 3;
    CARD32 dstOffset;
    char *dst;
    return FALSE;
    #if 0
    dstOffset = x * pDst->drawable.bitsPerPixel;
    if ((dstOffset & 7) || (wBytes & 3))
        return FALSE;
    dst = (char *)pVia->FBBase + (exaGetPixmapOffset(pDst) + y * dstPitch + (dstOffset >> 3));
    while(h--) {
        via_ExaMoveDwords((CARD32*)dst, (CARD32*)src, wBytes/4);
        src += src_pitch;
        dst += dstPitch;
        }
    return TRUE;
    #endif
}


