/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "via_driver.h"
#include "via_gpioi2c.h"
#include "hw.h"

void HWGPIOI2C_SetSCL(VIABIOSInfoPtr pBIOSInfo, CARD8 flag);
void HWGPIOI2C_SetSDA(VIABIOSInfoPtr pBIOSInfo, CARD8 flag);
CARD8  HWGPIOI2C_GetSDA(VIABIOSInfoPtr pBIOSInfo);
void GPIOI2C_START(VIABIOSInfoPtr pBIOSInfo);
void GPIOI2C_STOP(VIABIOSInfoPtr pBIOSInfo);
Bool GPIOI2C_ACKNOWLEDGE(VIABIOSInfoPtr pBIOSInfo);
Bool GPIOI2C_SENDACKNOWLEDGE(VIABIOSInfoPtr pBIOSInfo);
Bool GPIOI2C_SENDNACKNOWLEDGE(VIABIOSInfoPtr pBIOSInfo);
Bool GPIOI2C_ReadBit(VIABIOSInfoPtr pBIOSInfo, int *psda, int timeout);
CARD8 GPIOI2C_ReadData(VIABIOSInfoPtr pBIOSInfo);
Bool GPIOI2C_WriteBit(VIABIOSInfoPtr pBIOSInfo, int sda, int timeout);
Bool GPIOI2C_WriteData(VIABIOSInfoPtr pBIOSInfo, CARD8 Data);
void I2C_RW_Control(VIABIOSInfoPtr pBIOSInfo, CARD8 Command, CARD8 Data);

int g_DeviceI2CPort;

/* I2C Functions */

Bool GPIOI2C_Initial(VIABIOSInfoPtr pBIOSInfo, CARD8 SlaveDevice)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "GPIOI2C_Initial\n"));

    if (!pBIOSInfo->GPIOI2CInfo.bGPIOPort)
    {       
        GPIOPORT = 0x2C;
    }
    else
    {
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "Use Option Setting Port x%x\n", pBIOSInfo->GPIOI2CInfo.bGPIOPort));
    }
    
    if (SlaveDevice == 0xA0 || SlaveDevice == 0xA2)
    {
        I2C_WAIT_TIME = 40;
        STARTTIMEOUT = 550;
        BYTETIMEOUT = 2200;
        HOLDTIME = 10;
        BITTIMEOUT = 10;
    }
    else
    {
        I2C_WAIT_TIME = 10;
        STARTTIMEOUT = 10;
        BYTETIMEOUT = 10;
        HOLDTIME = 10;
        BITTIMEOUT = 10;
    }

    SLAVEADDR = SlaveDevice;

    return TRUE;
}

/* I2C R/W Control */
void I2C_RW_Control(VIABIOSInfoPtr pBIOSInfo, CARD8 Command, CARD8 Data)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    CARD8   value;

    switch(Command) {
        case I2C_RELEASE:
            VGAOUT8(0x3c4, GPIOPORT);
            value = VGAIN8(0x3c5) & (CARD8)(~GPIOI2C_MASKD);
            VGAOUT8(0x3c5, value);
            break;
        case I2C_WRITE_SCL:
            VGAOUT8(0x3c4, GPIOPORT);
            value = VGAIN8(0x3c5) & (CARD8)(~GPIOI2C_SCL_MASK);
            value |= GPIOI2C_SCL_WRITE;
            if (Data) {
                VGAOUT8(0x3c5, (value | I2C_OUTPUT_CLOCK));
            }
            else {
                VGAOUT8(0x3c5, (value & (CARD8)(~I2C_OUTPUT_CLOCK)));
            }
            break;
        case I2C_READ_SCL:
            VGAOUT8(0x3c4, GPIOPORT);
            value = VGAIN8(0x3c5) & (CARD8)(~GPIOI2C_SCL_MASK);
            VGAOUT8(0x3c5, (value | GPIOI2C_SCL_READ));
            break;
        case I2C_WRITE_SDA:
            VGAOUT8(0x3c4, GPIOPORT);
            value = VGAIN8(0x3c5) & (CARD8)(~GPIOI2C_SDA_MASK);
            value |= GPIOI2C_SDA_WRITE;
            if (Data) {
                VGAOUT8(0x3c5, (value | I2C_OUTPUT_DATA));
            }
            else {
                VGAOUT8(0x3c5, (value & (CARD8)(~I2C_OUTPUT_DATA)));
            }
            break;
        case I2C_READ_SDA:
            VGAOUT8(0x3c4, GPIOPORT);
            value = VGAIN8(0x3c5) & (CARD8)(~GPIOI2C_SDA_MASK);
            VGAOUT8(0x3c5, (value | GPIOI2C_SCL_READ));
            break;
        default:
            break;
    }
}

/* Set SCL */
void HWGPIOI2C_SetSCL(VIABIOSInfoPtr pBIOSInfo, CARD8 flag)
{
    I2C_RW_Control(pBIOSInfo, I2C_WRITE_SCL, flag);
    if (flag) VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME);
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME);
}

/* Set SDA */
void HWGPIOI2C_SetSDA(VIABIOSInfoPtr pBIOSInfo, CARD8 flag)
{
    I2C_RW_Control(pBIOSInfo, I2C_WRITE_SDA, flag);
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME);
}

/* Get SDA */
CARD8  HWGPIOI2C_GetSDA(VIABIOSInfoPtr pBIOSInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    CARD8   value;

    VGAOUT8(0x3c4, GPIOPORT);
    value = VGAIN8(0x3c5);
    if (value & I2C_INPUT_DATA)
        return 1;
    else
        return 0;
}

/* START Condition */
void GPIOI2C_START(VIABIOSInfoPtr pBIOSInfo)
{
    HWGPIOI2C_SetSDA(pBIOSInfo, 1);
    HWGPIOI2C_SetSCL(pBIOSInfo, 1);
    VIADelayIn_usec(pBIOSInfo, STARTTIMEOUT);
    HWGPIOI2C_SetSDA(pBIOSInfo, 0);
    HWGPIOI2C_SetSCL(pBIOSInfo, 0);
}

/* STOP Condition */
void GPIOI2C_STOP(VIABIOSInfoPtr pBIOSInfo)
{
    HWGPIOI2C_SetSDA(pBIOSInfo, 0);
    HWGPIOI2C_SetSCL(pBIOSInfo, 1);
    HWGPIOI2C_SetSDA(pBIOSInfo, 1);
    I2C_RW_Control(pBIOSInfo, I2C_RELEASE, 0);
    /* to make the differentiation of next START condition */
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME);
}

/* Check ACK */
Bool GPIOI2C_ACKNOWLEDGE(VIABIOSInfoPtr pBIOSInfo)
{
    CARD8   Status;

    I2C_RW_Control(pBIOSInfo, I2C_READ_SDA, 0);
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME);
    HWGPIOI2C_SetSCL(pBIOSInfo, 1);
    Status = HWGPIOI2C_GetSDA(pBIOSInfo);
    I2C_RW_Control(pBIOSInfo, I2C_WRITE_SDA, Status);
    HWGPIOI2C_SetSCL(pBIOSInfo, 0);
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME);

    if(Status) return FALSE;
    else return TRUE;
}

/* Send ACK */
Bool GPIOI2C_SENDACKNOWLEDGE(VIABIOSInfoPtr pBIOSInfo)
{
    HWGPIOI2C_SetSDA(pBIOSInfo, 0);
    HWGPIOI2C_SetSCL(pBIOSInfo, 1);
    HWGPIOI2C_SetSCL(pBIOSInfo, 0);
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME);

    return TRUE;
}

/* Send NACK */
Bool GPIOI2C_SENDNACKNOWLEDGE(VIABIOSInfoPtr pBIOSInfo)
{
    HWGPIOI2C_SetSDA(pBIOSInfo, 1);
    HWGPIOI2C_SetSCL(pBIOSInfo, 1);
    HWGPIOI2C_SetSCL(pBIOSInfo, 0);
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME);

    return TRUE;
}

/* Write Data(Bit by Bit) to I2C */
Bool GPIOI2C_WriteData(VIABIOSInfoPtr pBIOSInfo, CARD8 Data)
{
    int     i;

    if (!GPIOI2C_WriteBit(pBIOSInfo, (Data >> 7) & 1, BYTETIMEOUT))
	    return FALSE;

    for (i = 6; i >= 0; i--)
    if (!GPIOI2C_WriteBit(pBIOSInfo, (Data >> i) & 1, BITTIMEOUT))
	    return FALSE;

    return GPIOI2C_ACKNOWLEDGE(pBIOSInfo);
}

Bool GPIOI2C_WriteBit(VIABIOSInfoPtr pBIOSInfo, int sda, int timeout)
{
    Bool ret = TRUE;

    HWGPIOI2C_SetSDA(pBIOSInfo, sda);
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME/5);
    HWGPIOI2C_SetSCL(pBIOSInfo, 1);
    VIADelayIn_usec(pBIOSInfo, HOLDTIME);
    VIADelayIn_usec(pBIOSInfo, timeout);
    HWGPIOI2C_SetSCL(pBIOSInfo, 0);
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME/5);

    return ret;
}

Bool GPIOI2C_ReadBit(VIABIOSInfoPtr pBIOSInfo, int *psda, int timeout)
{
    Bool ret = TRUE;

    I2C_RW_Control(pBIOSInfo, I2C_READ_SDA, 0);
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME/5);
    HWGPIOI2C_SetSCL(pBIOSInfo, 1);
    VIADelayIn_usec(pBIOSInfo, HOLDTIME);
    VIADelayIn_usec(pBIOSInfo, timeout);
    *psda = HWGPIOI2C_GetSDA(pBIOSInfo);
    HWGPIOI2C_SetSCL(pBIOSInfo, 0);
    VIADelayIn_usec(pBIOSInfo, I2C_WAIT_TIME/5);

    return ret;
}

/* Read Data(Bit by Bit) from I2C */
CARD8 GPIOI2C_ReadData(VIABIOSInfoPtr pBIOSInfo)
{
    int     i, sda;
    CARD8   data;

    if(!GPIOI2C_ReadBit(pBIOSInfo, &sda, BYTETIMEOUT))
	return 0;

    data = (sda > 0) << 7;
    for (i = 6; i >= 0; i--)
	if (!GPIOI2C_ReadBit(pBIOSInfo, &sda, BITTIMEOUT))
	    return 0;
	else
	    data |= (sda > 0) << i;

    return  data;
}

/* Write Data(one Byte) to Desired Device on I2C */
Bool GPIOI2C_Write(VIABIOSInfoPtr pBIOSInfo, int SubAddress, CARD8 Data)
{
    int     Retry;
    Bool    Done = FALSE;

    for(Retry = 1; Retry <= WRITE_MAX_RETRIES; Retry++)
    {
        GPIOI2C_START(pBIOSInfo);

        if(!GPIOI2C_WriteData(pBIOSInfo, SLAVEADDR)) {

            GPIOI2C_STOP(pBIOSInfo);
            continue;
        }

        if(SubAddress & 0xF00) {            /* write 12-bit sub address */
            if(!GPIOI2C_WriteData(pBIOSInfo, (CARD8)((SubAddress/0x100)&0x0F))) {

                GPIOI2C_STOP(pBIOSInfo);
                continue;
            }
            if(!GPIOI2C_WriteData(pBIOSInfo, (CARD8)(SubAddress%0x100))) {

                GPIOI2C_STOP(pBIOSInfo);
                continue;
            }
        } else {                            /* write 8-bit sub address */
            if(!GPIOI2C_WriteData(pBIOSInfo, (CARD8)(SubAddress))) {

                GPIOI2C_STOP(pBIOSInfo);
                continue;
            }
        }

        if(!GPIOI2C_WriteData(pBIOSInfo, Data)) {

            GPIOI2C_STOP(pBIOSInfo);
            continue;
        }
        Done = TRUE;
        break;
    }

    GPIOI2C_STOP(pBIOSInfo);

    return Done;
}

/* Read Data from Desired Device on I2C */
Bool GPIOI2C_Read(VIABIOSInfoPtr pBIOSInfo, int SubAddress, CARD8 *Buffer, int BufferLen)
{
    int     i, Retry;

    for(Retry = 1; Retry <= READ_MAX_RETRIES; Retry++)
    {
        GPIOI2C_START(pBIOSInfo);
        if(!GPIOI2C_WriteData(pBIOSInfo, SLAVEADDR&0xFE)) {

            GPIOI2C_STOP(pBIOSInfo);
            continue;
        }

        if(SubAddress & 0xF00) {        /* write 12-bit sub address */
            if(!GPIOI2C_WriteData(pBIOSInfo, (CARD8)((SubAddress/0x100)&0x0F))) {

                GPIOI2C_STOP(pBIOSInfo);
                continue;
            }
            if(!GPIOI2C_WriteData(pBIOSInfo, (CARD8)(SubAddress%0x100))) {

                GPIOI2C_STOP(pBIOSInfo);
                continue;
            }
        } else {                        /* write 8-bit sub address */
            if(!GPIOI2C_WriteData(pBIOSInfo, (CARD8)(SubAddress))) {

                GPIOI2C_STOP(pBIOSInfo);
                continue;
            }
        }

        break;
    }

    if (Retry > READ_MAX_RETRIES) return FALSE;

    for(Retry = 1; Retry <= READ_MAX_RETRIES; Retry++)
    {
        GPIOI2C_START(pBIOSInfo);
        if(!GPIOI2C_WriteData(pBIOSInfo, SLAVEADDR | 0x01)) {

            GPIOI2C_STOP(pBIOSInfo);
            continue;
        }
        for(i = 0; i < BufferLen; i++) {

            *Buffer = GPIOI2C_ReadData(pBIOSInfo);
            Buffer ++;
            if(BufferLen == 1)
                /*GPIOI2C_SENDACKNOWLEDGE(pBIOSInfo);*/ /* send ACK for normal operation */
                GPIOI2C_SENDNACKNOWLEDGE(pBIOSInfo);    /* send NACK for VT3191/VT3192 only */
            else if(i < BufferLen - 1)
                GPIOI2C_SENDACKNOWLEDGE(pBIOSInfo);     /* send ACK */
            else
                GPIOI2C_SENDNACKNOWLEDGE(pBIOSInfo);    /* send NACK */
        }
        GPIOI2C_STOP(pBIOSInfo);
        break;
    }

    if (Retry > READ_MAX_RETRIES)
        return FALSE;
    else
        return TRUE;
}

/* Read Data(one Byte) from Desired Device on I2C */
Bool GPIOI2C_ReadByte(VIABIOSInfoPtr pBIOSInfo, int SubAddress, CARD8 *Buffer)
{
    int     Retry;

    for(Retry = 1; Retry <= READ_MAX_RETRIES; Retry++)
    {
        GPIOI2C_START(pBIOSInfo);
        if(!GPIOI2C_WriteData(pBIOSInfo, SLAVEADDR & 0xFE)) {

            GPIOI2C_STOP(pBIOSInfo);
            continue;
        }

        if(SubAddress & 0xF00) {        /* write 12-bit sub address */
            if(!GPIOI2C_WriteData(pBIOSInfo, (CARD8)((SubAddress/0x100)&0x0F))) {

                GPIOI2C_STOP(pBIOSInfo);
                continue;
            }
            if(!GPIOI2C_WriteData(pBIOSInfo, (CARD8)(SubAddress%0x100))) {

                GPIOI2C_STOP(pBIOSInfo);
                continue;
            }
        } else {                        /* write 8-bit sub address */
            if(!GPIOI2C_WriteData(pBIOSInfo, (CARD8)(SubAddress))) {

                GPIOI2C_STOP(pBIOSInfo);
                continue;
            }
        }

        break;
    }

    if (Retry > READ_MAX_RETRIES) return FALSE;

    for(Retry = 1; Retry <= READ_MAX_RETRIES; Retry++) {

        GPIOI2C_START(pBIOSInfo);

        if(!GPIOI2C_WriteData(pBIOSInfo, SLAVEADDR | 0x01)) {

            GPIOI2C_STOP(pBIOSInfo);
            continue;
        }

        *Buffer = GPIOI2C_ReadData(pBIOSInfo);

        GPIOI2C_STOP(pBIOSInfo);
        break;
    }

    if (Retry > READ_MAX_RETRIES)
        return FALSE;
    else
        return TRUE;
}


/********************************************************************************/
/* Porting from FBDev driver.                             */
/********************************************************************************/

/* i2c delay for microsecond */
void delays(int count)
{
    volatile CARD8 data;
    
    while (count--)
    {
        /* delay 1 us */
        data = inb(DELAY_PORT);
        data = inb(DELAY_PORT);
        data = inb(DELAY_PORT);
        data = inb(DELAY_PORT);
        data = inb(DELAY_PORT);
    }
}

void WriteReg_I2C(VIABIOSInfoPtr pBIOSInfo, CARD8  data)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    
    VGAOUT8(0x3C4, g_DeviceI2CPort);
    VGAOUT8(0x3C5, data);
}

CARD8 ReadReg_I2C(VIABIOSInfoPtr pBIOSInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    CARD8  data;
    
    VGAOUT8(0x3C4, g_DeviceI2CPort);
    data = VGAIN8(0x3C5);

    return data;
}

void WriteReg_GPIO(VIABIOSInfoPtr pBIOSInfo, CARD8  data)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    
    VGAOUT8(0x3C4, g_DeviceI2CPort);
    VGAOUT8(0x3C5, data);
}

CARD8 ReadReg_GPIO(VIABIOSInfoPtr pBIOSInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    CARD8  data;
    
    VGAOUT8(0x3C4, g_DeviceI2CPort);
    data = VGAIN8(0x3C5);

    return data;
}

/*  Write I2C BUS SDA And SCL */
void i2cWriteSdaScl(VIABIOSInfoPtr pBIOSInfo, CARD8 sda, CARD8 scl)
{
    CARD8  data;

    switch (g_DeviceI2CPort)
    {
        case PORT_INDEX_I2C_26:
        case PORT_INDEX_I2C_31:
            data = ((scl << 1) | sda) << 4;
            data = data | BIT0;                         /* enable I2C port */

            /* Write Register Value */
            WriteReg_I2C(pBIOSInfo, data);
            break;

        case PORT_INDEX_GPIO_2C:                                         
        case PORT_INDEX_GPIO_25:
        case PORT_INDEX_GPIO_3D:
            data = ((scl << 1) | sda) << 4;
            data = data | (BIT6+BIT7);                  /* enable GPIO write port */

            /* BIT1:SW Control GPIO (From vt3353)*/
            if ((pBIOSInfo->Chipset >= VIA_VX800)
             && (g_DeviceI2CPort == PORT_INDEX_GPIO_2C))
            {
                data |= BIT1;
            }
            
            /* Write Register Value */
            WriteReg_GPIO(pBIOSInfo, data);
            break;
    }
}

void i2cWriteScl(VIABIOSInfoPtr pBIOSInfo, CARD8 scl)
{
    CARD8  data;

    switch (g_DeviceI2CPort)
    {
        case PORT_INDEX_I2C_26:    
        case PORT_INDEX_I2C_31:
            data = (scl << 1)<< 4;
            data = data | BIT0;                         /* enable I2C port */

            /* Write Register Value */
            WriteReg_I2C(pBIOSInfo, data);
            break;

        case PORT_INDEX_GPIO_2C:
        case PORT_INDEX_GPIO_25:
	    case PORT_INDEX_GPIO_3D:
            data = (scl << 1) << 4;
            data = data & 0xBF;                     /* enable GPIO write clock */

            /* Write Register Value */
            WriteReg_GPIO(pBIOSInfo, data);
            break;
    }
}

void i2cReadSdaScl(VIABIOSInfoPtr pBIOSInfo, CARD8 *pSda, CARD8 *pScl)
{
    CARD8  data;
    
    switch (g_DeviceI2CPort)
    {
        case PORT_INDEX_I2C_26:    
        case PORT_INDEX_I2C_31:
            data = ReadReg_I2C(pBIOSInfo);
            *pSda = (data >> 2) & BIT0;     /* get sda */
            *pScl = (data >> 3) & BIT0;     /* get scl */
            break;

        case PORT_INDEX_GPIO_2C:
        case PORT_INDEX_GPIO_25:
   	    case PORT_INDEX_GPIO_3D:
            data = ReadReg_GPIO(pBIOSInfo);
            *pSda = (data >> 2) & BIT0;  /* get sda */
            *pScl = (data >> 3) & BIT0;  /* get scl */
            break;
    }
}


void i2cWriteSdaSclDelay(VIABIOSInfoPtr pBIOSInfo, CARD8 sda, CARD8 scl)
{
    i2cWriteSdaScl(pBIOSInfo, sda, scl);
    delays(16);                 /* Wait 16 us */
}

void i2cStartSignal(VIABIOSInfoPtr pBIOSInfo)
{
    i2cWriteSdaSclDelay(pBIOSInfo, 1,1);
    i2cWriteSdaSclDelay(pBIOSInfo, 0,1);
    i2cWriteSdaSclDelay(pBIOSInfo, 0,0);
}

void i2cStopSignal(VIABIOSInfoPtr pBIOSInfo)
{
    CARD8  data;

    i2cWriteSdaSclDelay(pBIOSInfo, 0,0);
    i2cWriteSdaSclDelay(pBIOSInfo, 0,1);
    i2cWriteSdaSclDelay(pBIOSInfo, 1,1);

    if ((g_DeviceI2CPort == PORT_INDEX_GPIO_2C) ||
        (g_DeviceI2CPort == PORT_INDEX_GPIO_25)||
        (g_DeviceI2CPort == PORT_INDEX_GPIO_3D))
    {
        data = 0x3c;                                    /* disable GPIO write port */

        /* Write Register Value */
        WriteReg_GPIO(pBIOSInfo, data);
    }

    delays(2);
}

void disableSdaGPIO(VIABIOSInfoPtr pBIOSInfo)
{
    CARD8  data;

    if ((g_DeviceI2CPort == PORT_INDEX_GPIO_2C) ||
        (g_DeviceI2CPort == PORT_INDEX_GPIO_25)||
        (g_DeviceI2CPort == PORT_INDEX_GPIO_3D))
    {
        data = ReadReg_GPIO(pBIOSInfo);
        data = data & (~BIT6);                          /* disable GPIO write port */

        /* Write Register Value */
        WriteReg_GPIO(pBIOSInfo, data);
    }
}

void enableSdaGPIO(VIABIOSInfoPtr pBIOSInfo)
{
    CARD8  data;

    if ((g_DeviceI2CPort == PORT_INDEX_GPIO_2C) ||
        (g_DeviceI2CPort == PORT_INDEX_GPIO_25)||
        (g_DeviceI2CPort == PORT_INDEX_GPIO_3D))
    {
        data = ReadReg_GPIO(pBIOSInfo);
        data = data | (BIT6);                           /* disable GPIO write port */

        /* Write Register Value */
        WriteReg_GPIO(pBIOSInfo, data);
    }
}

void writeSclGPIO(VIABIOSInfoPtr pBIOSInfo, CARD8 scl)
{
    CARD8  data;

    if ((g_DeviceI2CPort == PORT_INDEX_GPIO_2C) ||
        (g_DeviceI2CPort == PORT_INDEX_GPIO_25)||
        (g_DeviceI2CPort == PORT_INDEX_GPIO_3D))
    {
        data = ReadReg_GPIO(pBIOSInfo);
        data = data & (~BIT5);
        data = (data | (scl<<5)) & (~BIT6);             /* write data to clock */

        /* Write Register Value */
        WriteReg_GPIO(pBIOSInfo, data);
    }
}

void writeSdaGPIO(VIABIOSInfoPtr pBIOSInfo, CARD8 sda)
{
    CARD8  data;

   if ((g_DeviceI2CPort == PORT_INDEX_GPIO_2C) ||
        (g_DeviceI2CPort == PORT_INDEX_GPIO_25)||
        (g_DeviceI2CPort == PORT_INDEX_GPIO_3D))
    {
        data = ReadReg_GPIO(pBIOSInfo);

        data = data & (~BIT4);
        data = (data | (sda<<4)) & (~BIT7);             /* write data to clock */

        /* Write Register Value */
        WriteReg_GPIO(pBIOSInfo, data);
    }
}

Bool i2CWaitForSlave(VIABIOSInfoPtr pBIOSInfo)
{
    int time_out = 20000;
    CARD8  sda, scl;

    while (time_out--)
    {
        i2cReadSdaScl(pBIOSInfo, &sda, &scl);
        
        if (scl)
        {
            return TRUE;        /* Successful stall */
        }
        
        delays(1);              /* wait 1 uS */
    }
    
    return FALSE;              /* Slave fail */
}

Bool i2cOutByte(VIABIOSInfoPtr pBIOSInfo, CARD8 data)
{
    CARD8  sda, scl;
    CARD8  out_byte;
    int bit_count = 8;
    Bool status;

    out_byte = data;
    
    while (bit_count--)
    {
        sda = (out_byte >> 7) & 1;      /* Load MSB */
        out_byte = out_byte << 1;       /* next bit. */
        i2cWriteSdaSclDelay(pBIOSInfo, sda, 0);
        i2cWriteSdaSclDelay(pBIOSInfo, sda, 1);

        status = i2CWaitForSlave(pBIOSInfo);
        
        if (status == FALSE)
        {
            return (status);
        }

        i2cWriteSdaSclDelay(pBIOSInfo, sda, 0);
    }

    switch (g_DeviceI2CPort)
    {
        case PORT_INDEX_I2C_26:    
        case PORT_INDEX_I2C_31:
            i2cWriteSdaSclDelay(pBIOSInfo, 1,0);
            i2cWriteSdaSclDelay(pBIOSInfo, 1,1);
            status = i2CWaitForSlave(pBIOSInfo);
            
            if (status == FALSE)
            {
                return (status);
            }

            i2cReadSdaScl(pBIOSInfo, &sda, &scl);
            
            if (sda == 0)
            {
                i2cWriteSdaSclDelay(pBIOSInfo, 1,0);
                status = TRUE;
            }
            else
            {
                i2cWriteSdaSclDelay(pBIOSInfo, 1,0);
                status = FALSE;
            }
            break;

        case PORT_INDEX_GPIO_2C:
        case PORT_INDEX_GPIO_25:
	    case PORT_INDEX_GPIO_3D:
            writeSclGPIO(pBIOSInfo, 0);
            disableSdaGPIO(pBIOSInfo);
            delays(2);
            writeSclGPIO(pBIOSInfo, 1);
            delays(2);
            i2cReadSdaScl(pBIOSInfo, &sda, &scl);
            writeSclGPIO(pBIOSInfo, 0);
            delays(2);
            
            if (sda == 0)
            {
                status = TRUE;
            }
            else
            {
                status = FALSE;
            }
            break;
    }

    return (status);
}

Bool i2cInputByte(VIABIOSInfoPtr pBIOSInfo, CARD8 *pInByte, int ack)
{
    int bit_count = 8;
    CARD8  sda, scl;
    CARD8  data = 0;
    Bool status;

    disableSdaGPIO(pBIOSInfo);

    while (bit_count--)
    {
        switch (g_DeviceI2CPort)
        {
            case PORT_INDEX_I2C_26:        
            case PORT_INDEX_I2C_31:
                i2cWriteSdaSclDelay(pBIOSInfo, 1,1);
                status = i2CWaitForSlave(pBIOSInfo);
                
                if (status == FALSE)
                {
                    return FALSE;
                }
                
                i2cReadSdaScl(pBIOSInfo, &sda, &scl);
                data = data << 1;
                data |= sda;
                i2cWriteSdaSclDelay(pBIOSInfo, 1, 0);
                break;
                
            case PORT_INDEX_GPIO_2C:
            case PORT_INDEX_GPIO_25:
	        case PORT_INDEX_GPIO_3D:
                writeSclGPIO(pBIOSInfo, 1);
                delays(2);
                status = i2CWaitForSlave(pBIOSInfo);
                
                if (status == FALSE)
                {
                    return FALSE;
                }
                
                i2cReadSdaScl(pBIOSInfo, &sda, &scl);
                data = data << 1;
                data |= sda;
                writeSclGPIO(pBIOSInfo, 0);
                delays(2);
                break;
        }        
    }
    
    *pInByte = data;

    if (ack)
    {
        i2cWriteSdaSclDelay(pBIOSInfo, 0, 0);
        i2cWriteSdaSclDelay(pBIOSInfo, 0, 1);
        status = i2CWaitForSlave(pBIOSInfo);

        if (status == FALSE)
        {
            return(status);
        }
        
        i2cWriteSdaSclDelay(pBIOSInfo, 0, 0);
    }
    else
    {
        i2cWriteSdaSclDelay(pBIOSInfo, 1, 0);
        i2cWriteSdaSclDelay(pBIOSInfo, 1, 1);
        status = i2CWaitForSlave(pBIOSInfo);
        
        if (status == FALSE)
        {
            return(status);
        }
    }
    
    i2cWriteSdaSclDelay(pBIOSInfo, 1, 0);

    return TRUE;
}

Bool i2cReadByte(VIABIOSInfoPtr pBIOSInfo, CARD8 slave_addr, CARD8 index, CARD8 *pData)
{
    Bool status;

    i2cStartSignal(pBIOSInfo);

    status = i2cOutByte(pBIOSInfo, slave_addr);
    
    if (status == FALSE)
    {
        i2cStopSignal(pBIOSInfo);
        return FALSE;
    }

    status = i2cOutByte(pBIOSInfo, index);

    if (status == FALSE)
    {
        i2cStopSignal(pBIOSInfo);
        return FALSE;
    }

    i2cStartSignal(pBIOSInfo);
    status = i2cOutByte(pBIOSInfo, slave_addr | BIT0);

    if (status == FALSE)
    {
        i2cStopSignal(pBIOSInfo);
        return FALSE;
    }

    status = i2cInputByte(pBIOSInfo, pData, 0);

    if (status == FALSE)
    {
        i2cStopSignal(pBIOSInfo);
        return FALSE;
    }

    i2cStopSignal(pBIOSInfo);
    return TRUE;
}

Bool i2cWriteByte(VIABIOSInfoPtr pBIOSInfo, CARD8 slave_addr, CARD8 index, CARD8 data)
{
    Bool status;

    i2cStartSignal(pBIOSInfo);
    status = i2cOutByte(pBIOSInfo, slave_addr);
    
    if (status == FALSE)
    {
        i2cStopSignal(pBIOSInfo);
        return FALSE;
    }
    
    status = i2cOutByte(pBIOSInfo, index);
    
    if (status == FALSE)
    {
        i2cStopSignal(pBIOSInfo);
        return FALSE;
    }
    
    status = i2cOutByte(pBIOSInfo, data);
    
    if (status == FALSE)
    {
        i2cStopSignal(pBIOSInfo);
        return FALSE;
    }
    
    i2cStopSignal(pBIOSInfo);
    return TRUE;
}


void Write_REG_LVDS(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo, CARD8 index, CARD8 data)
{
    g_DeviceI2CPort = pLVDSSettingInfo->I2CPort;
    i2cWriteByte(pBIOSInfo, pLVDSSettingInfo->I2CAddress, index, data);
}

Bool Read_REG_LVDS(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo, CARD8 index, CARD8 *pData)
{
    Bool  status;
    
    g_DeviceI2CPort = pLVDSSettingInfo->I2CPort;
    status = i2cReadByte(pBIOSInfo, pLVDSSettingInfo->I2CAddress, index, pData);

    return status;
}

void Write_Mask_REG_LVDS(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo, IODATA io_data)
{
    unsigned char index, data = 0;
    
    g_DeviceI2CPort = pLVDSSettingInfo->I2CPort;

    index = io_data.Index;
    Read_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, index, &data);
    data = (data & (~io_data.Mask)) | io_data.Data;

    i2cWriteByte(pBIOSInfo, pLVDSSettingInfo->I2CAddress, index, data);
}


void GPIOI2CWrite_TMDS(VIABIOSInfoPtr pBIOSInfo, TMDSSETTINGINFOPTR pTMDSSettingInfo, CARD8 index, CARD8 data)
{
    g_DeviceI2CPort = pTMDSSettingInfo->I2CPort;
    i2cWriteByte(pBIOSInfo, pTMDSSettingInfo->I2CAddress, index, data);
}

Bool GPIOI2CRead_TMDS(VIABIOSInfoPtr pBIOSInfo, TMDSSETTINGINFOPTR pTMDSSettingInfo, CARD8 index, CARD8 *pData)
{
    Bool  status;
    
    g_DeviceI2CPort = pTMDSSettingInfo->I2CPort;
    status = i2cReadByte(pBIOSInfo, pTMDSSettingInfo->I2CAddress, index, pData);

    return status;
}


