/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
#include "via_driver.h"
#include "via_lcd.h"
#include "hw.h"
#include "via_lcdtbl.h"
#include "via_vt1636.h"


void VIALCDSetMode(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    struct io_register      *reg = NULL;
    struct display_timing   mode_crt_reg, panel_crt_reg;
    int                     load_reg_num=0;
    int                     reg_value=0;
    int                     set_hres, set_vres, h_active, v_active, h_size, v_size;
    int                     panel_hres, panel_vres;
    struct crt_mode_table   *panel_crt_table;
    struct VideoModeTable   *PanelMode;
    struct crt_mode_table   *mode_crt_table;
    struct VideoModeTable   *VideoMode;  

    CARD32                  pll_D_N;
    int                     offset;
    int                     set_iga = pLVDSSettingInfo->IGAPath;
    BOOL                    lcd_use_modeline = FALSE;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIALCDSetMode!! ActiveDeviceID=0x%x, IGA=%d!!\n", pBIOSInfo->ActiveDevice, pLVDSSettingInfo->IGAPath));

    if (!(pBIOSInfo->Is3DScalingEnable == (DISP_3D_SCALING_ENABLE | VIA_DEVICE_LCD)) &&
		((pBIOSInfo->SAMM) ||
         (!pBIOSInfo->SAMM && (pBIOSInfo->EDIDType == VIA_DEVICE_LCD))) && 
         (pBIOSInfo->ModeLineStatus & (MODE_LINE_TYPE_USER | MODE_LINE_TYPE_EDID)))
    {
        lcd_use_modeline = TRUE;
    }

    /* In few case, we need to know the whole mode timing */
    if (pBIOSInfo->MergedFB || set_iga== IGA1_IGA2) {
        /* Get desired mode timing for centering used. */
        VideoMode = &VIARoutineModes[SearchModeSetting(pLVDSSettingInfo->ModeIndex)];           /* Get mode table */
        mode_crt_table = VideoMode->crtc;
        mode_crt_reg = mode_crt_table->crtc;
    /* In most case, we need to know the hor addr and ver addr of mode timing */
    } else {
        mode_crt_reg.hor_addr = pLVDSSettingInfo->HActive;
        mode_crt_reg.ver_addr = pLVDSSettingInfo->VActive;
    }
    
    /* Get LCD timing. */
    PanelMode = &LCDModes[SearchLCDModeIndex(VIAGetModeIndexByPanelSize(pLVDSSettingInfo->PanelSizeID))];    /* Get panel table */
    if (lcd_use_modeline && pBIOSInfo->isPanelModeLine) {
        panel_crt_table = &(pBIOSInfo->UserSpecifiedMode);
        panel_crt_reg = panel_crt_table->crtc;
    } else {
        panel_crt_table = PanelMode->crtc;
        panel_crt_reg = panel_crt_table->crtc;
    }

    /* The meaning of resolution to set here should not be view size but */
    /* real desktop size. */
    set_hres = pBIOSInfo->ActualDesktopSizeX;
    set_vres = pBIOSInfo->ActualDesktopSizeY;
    h_active = pLVDSSettingInfo->HActive;
    v_active = pLVDSSettingInfo->VActive;
    panel_hres = pLVDSSettingInfo->PanelSizeX;
    panel_vres = pLVDSSettingInfo->PanelSizeY;

    if(pLVDSSettingInfo->IGAPath == IGA1)
    {
        pLVDSSettingInfo->Center = TRUE;
    }

    if (pLVDSSettingInfo->ChipID == VIA_VT1636)
    {
        VIAInitLVDS_VT1636(pBIOSInfo, pLVDSSettingInfo);
    }
    
    /*add while LCD in Expansion,for mergedfb*/
    if(pBIOSInfo->MergedFB)
    {        
        fill_crtc_timing(pBIOSInfo,mode_crt_table, VideoMode->mode_array, (pBIOSInfo->bitsPerPixel >> 3), VIA_DEVICE_LCD, pLVDSSettingInfo->IGAPath, FALSE);
        pLVDSSettingInfo->VClk = mode_crt_table->clk;
    }
    else
    {
        fill_crtc_timing(pBIOSInfo,panel_crt_table, PanelMode->mode_array, (pBIOSInfo->bitsPerPixel >> 3), VIA_DEVICE_LCD, pLVDSSettingInfo->IGAPath, lcd_use_modeline);
        pLVDSSettingInfo->VClk = panel_crt_table->clk;
    }

    xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "-----LCD TIMING-----\n");

    if(set_iga == IGA1)
    { 
        unlock_crt();
        load_crtc_timing(pBIOSInfo, lcd_centering_timging(mode_crt_reg, panel_crt_reg), IGA1);
        write_reg_mask(CR79, VIACR, 0x00, BIT0+BIT1+BIT2);        /* LCD scaling disabled */
        pBIOSInfo->scaleY = FALSE;
        lock_crt();  
    }
    else
    {
        /* Expansion */
        if (pLVDSSettingInfo->Center == FALSE)
        {
            pLVDSSettingInfo->PanelSizeY = panel_crt_reg.ver_addr;
            pBIOSInfo->resY = mode_crt_reg.ver_addr;

            /* expansion timing IGA2 always loaded panel set timing */
            load_crtc_timing(pBIOSInfo, panel_crt_reg, IGA2);

            if(pBIOSInfo->Is3DScalingEnable & VIA_DEVICE_LCD)
            {
                write_reg_mask(CR79, VIACR, 0x00, BIT0+BIT1+BIT2);        /* LCD scaling disabled */
                pBIOSInfo->scaleY = FALSE;
            }
            else
            {
                if ((h_active < panel_hres ) || (v_active < panel_vres) ||
                    (set_hres < panel_hres ) || (set_vres < panel_vres))
                {
                    if (h_active < set_hres)
                    {
                        h_size = h_active;
                    }
                    else
                    {
                        h_size = set_hres;
                    }

                    if (v_active < set_vres)
                    {
                        v_size = v_active;
                    }
                    else
                    {
                        v_size = set_vres;
                    }
                
                    VIALoadLCDScaling(pBIOSInfo, h_size, v_size, pLVDSSettingInfo->PanelSizeX, pLVDSSettingInfo->PanelSizeY);
                    pBIOSInfo->scaleY = TRUE;
                }
                else
                {
                    /* LCD scaling disabled */
                    write_reg_mask(CR79, VIACR, 0x00, BIT0+BIT1+BIT2);        
                    pBIOSInfo->scaleY = FALSE;
                }
            }
        }
        else /* Centering */
        {
            /* centering timing IGA2 always loaded panel and mode releative timing */
            load_crtc_timing(pBIOSInfo, lcd_centering_timging(mode_crt_reg, panel_crt_reg), IGA2);
            write_reg_mask(CR79, VIACR, 0x00, BIT0+BIT1+BIT2);        /* LCD scaling disabled */
            pBIOSInfo->scaleY = FALSE;
        }
    }

    if (set_iga== IGA1_IGA2)
    {
        load_crtc_shadow_timing(pBIOSInfo, mode_crt_reg, panel_crt_reg);       /* Fill shadow registers */

        switch(pLVDSSettingInfo->PanelSizeID)
        {
            case VIA_640X480:
                offset = 80;
                break;
			/*because VIA_800X480_GEN's value is the same as VIA_800X480,so we just use one*/
			case VIA_800X480:
            case VIA_800X600:
                offset = 110;
                break;
            case VIA_1024X768:
                offset = 150;
                break;
            case VIA_1280X768:
            case VIA_1280X800:
            case VIA_1280X1024:
            case VIA_1400X1050:
                offset = 190;
                break;
            case VIA_1600X1200:
                offset = 250;
                break;
            case VIA_1366X768:
            case VIA_1360X768:
                offset = 212;                             
                break;
            case VIA_1440X900:
                offset = 224;
                break;
            default:
                offset = 140;
                break;
        }

        /* Offset for simultaneous */
        reg_value = offset;
        load_reg_num = offset_reg.iga2_offset_reg.reg_num;
        reg = offset_reg.iga2_offset_reg.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);

        load_fetch_count_reg(mode_crt_reg.hor_addr, 4, IGA2);           /* Fetch count for simultaneous*/
    }
    else  /* IGA2 for SAMM */
    {
        /*for mergedfb */ 
        if(pBIOSInfo->MergedFB)
        {
            if((pBIOSInfo->Scrn2Position == viaRightOf)||(pBIOSInfo->Scrn2Position == viaLeftOf))
            {
                load_offset_reg((mode_crt_reg.hor_addr*2),(pBIOSInfo->bitsPerPixel>>3),set_iga);
            }
            
            if((pBIOSInfo->Scrn2Position == viaBelow)||(pBIOSInfo->Scrn2Position == viaAbove))
            {
                load_offset_reg(mode_crt_reg.hor_addr,(pBIOSInfo->bitsPerPixel>>3),set_iga);
            }
        }
        else
        {
            load_offset_reg(mode_crt_reg.hor_addr, (pBIOSInfo->bitsPerPixel>>3), set_iga);
        }
        
        load_fetch_count_reg(mode_crt_reg.hor_addr, (pBIOSInfo->bitsPerPixel>>3),set_iga);    /* Fetch count for IGA2 only */
        load_FIFO_reg(pBIOSInfo, mode_crt_reg.hor_addr, mode_crt_reg.ver_addr, set_iga);
        set_color_depth(pBIOSInfo, pBIOSInfo->bitsPerPixel>>3, set_iga);
    }

    VIA_fill_lcd_format(pLVDSSettingInfo->IsDualEdge, pLVDSSettingInfo->IsDithering);

    if (lcd_use_modeline)
    {
        CalFromClock(pBIOSInfo, (double)(pBIOSInfo->UserSpecifiedMode.clk/1000000), &pll_D_N);    
        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO," PixelClock=%ld Hz\n", pBIOSInfo->UserSpecifiedMode.clk);                
        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO,"-----TIMING END-----\n");                        
    }
    else
    {
        pll_D_N = get_clk_value(pBIOSInfo, panel_crt_table[0].clk);
        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO," PixelClock=%ld Hz\n", panel_crt_table[0].clk);                
        xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO,"-----TIMING END-----\n");                        
    }
    DEBUG(ErrorF("PLL=0x%x\n", pll_D_N));
    SetVCLK(pBIOSInfo, pll_D_N, set_iga);

    VIASetOutputPath(pBIOSInfo, VIA_DEVICE_LCD, set_iga, pLVDSSettingInfo->DIPort);

    /* Adjust DPA Setting for clock skew problem: */
    VIALCDPatchSkew(pBIOSInfo, pLVDSSettingInfo);

    /* Set display channel. */
    if (pLVDSSettingInfo->ChipID == VIA_INTEGRATED_LVDS)
    {
        VIASetDisplayChannel(pBIOSInfo);
    }

    /* set CRT to IGA2 path */
    if ((set_iga == IGA1_IGA2)||(set_iga == IGA1))
    {
        write_reg_mask(SR16, VIASR, 0x40, BIT6);
    }

    /* If K8M800, enable LCD Prefetch Mode. */
    if (pBIOSInfo->Chipset == VIA_K8M890)
    {
        write_reg_mask(CR6A, VIACR, 0x01, BIT0);
    }

    VIALoadLCDPatchRegs(pBIOSInfo, set_iga); 
}

void VIALoadLCDScaling(VIABIOSInfoPtr pBIOSInfo, int set_hres, int set_vres, int panel_hres, int panel_vres)
{
    struct io_register  *reg = NULL;
    int                 reg_value;
    int                 load_reg_num;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, 
        "VIALoadLCDScaling: set_res=%dx%d, panel size=%dx%d\n", set_hres, set_vres, panel_hres, panel_vres));

    /* LCD Scaling Enable */
    write_reg_mask(CR79, VIACR, 0x07, BIT0+BIT1+BIT2);    

    /* P4M900_A0's scaling register has a little problem on reading it.
     * So we use another method to access it.
     */
    if (pBIOSInfo->Chipset == VIA_P4M900)
    {
        VIALoadScalingFactorForP4M900(pBIOSInfo, set_hres, set_vres, panel_hres, panel_vres);
        return;
    }

    /* Check if expansion for horizontal */
    if(set_hres < panel_hres)
    {
        /* Load Horizontal Scaling Factor */
        /* For VIA_K8M800 or later chipsets. */
        reg_value = K800_LCD_HOR_SCF_FORMULA(set_hres, panel_hres);
        write_reg_mask(CRA2, VIACR, 0xC0, BIT7+BIT6);       /* Horizontal scaling enabled */
        load_reg_num = lcd_scaling_factor.lcd_hor_scaling_factor.reg_num;
        reg = lcd_scaling_factor.lcd_hor_scaling_factor.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);

        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "Horizontal Scaling value = %d\n", reg_value));
    }
    else
    {
        write_reg_mask(CRA2, VIACR, 0x00, BIT7);      /* Horizontal scaling disabled */
    }

    
    /* Check if expansion for vertical */
    if(set_vres < panel_vres)
    {
        /* Load Vertical Scaling Factor */
        /* For VIA_K8M800 or later chipsets. */
        reg_value = K800_LCD_VER_SCF_FORMULA(set_vres, panel_vres);
        write_reg_mask(CRA2, VIACR, 0x08, BIT3);       /* Vertical scaling enabled */
        load_reg_num = lcd_scaling_factor.lcd_ver_scaling_factor.reg_num;
        reg = lcd_scaling_factor.lcd_ver_scaling_factor.reg;
        load_reg(reg_value, load_reg_num, reg, VIACR);                

        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "Vertical Scaling value = %d\n", reg_value));
    }
    else
    {
        write_reg_mask(CRA2, VIACR, 0x00, BIT3);      /* Vertical scaling disabled */
    }
}

void VIALoadLCDPatchRegs(VIABIOSInfoPtr pBIOSInfo, int set_iga)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIALoadLCDPatchRegs!!\n"));

    unlock_crt();

    /* Patch for simultaneous & Expansion */
    if((set_iga == IGA1_IGA2)&&(!pBIOSInfo->LVDSSettingInfo.Center))
    {
        VIA_load_lcd_p880_patch_tbl(pBIOSInfo);
    }
    lock_crt();
}


void VIA_load_lcd_p880_patch_tbl(VIABIOSInfoPtr pBIOSInfo)
{
    struct io_reg   *lcd_patch_reg = NULL;
    int             reg_num = 0;
    VIABIOSInfoPtr  pVia = pBIOSInfo;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIA_load_lcd_p880_patch_tbl!!\n"));

    switch(pBIOSInfo->LVDSSettingInfo.PanelSizeID)
    {
        case VIA_1400X1050:
            switch(pBIOSInfo->LVDSSettingInfo.ModeIndex)
            {
                case M640x480:
                    reg_num = NUM_TOTAL_P880_LCD_RES_6X4_14X10;
                    lcd_patch_reg = P880_LCD_RES_6X4_14X10;
                    break;
                case M800x600:
                    reg_num = NUM_TOTAL_P880_LCD_RES_8X6_14X10;
                    lcd_patch_reg = P880_LCD_RES_8X6_14X10;
                    break;
            }
            break;
        case VIA_1600X1200:
            switch(pBIOSInfo->LVDSSettingInfo.ModeIndex)
            {
                case M640x480:
                    reg_num = NUM_TOTAL_P880_LCD_RES_6X4_16X12;
                    lcd_patch_reg = P880_LCD_RES_6X4_16X12;
                    break;
                case M720x480:
                case M720x576:
                    reg_num = NUM_TOTAL_P880_LCD_RES_7X4_16X12;
                    lcd_patch_reg = P880_LCD_RES_7X4_16X12;
                    break;
                case M800x600:
                    reg_num = NUM_TOTAL_P880_LCD_RES_8X6_16X12;
                    lcd_patch_reg = P880_LCD_RES_8X6_16X12;
                    break;
                case M1024x768:
                    reg_num = NUM_TOTAL_P880_LCD_RES_10X7_16X12;
                    lcd_patch_reg = P880_LCD_RES_10X7_16X12;
                    break;
                case M1280x768:
                case M1280x960:
                case M1280x1024:
                    reg_num = NUM_TOTAL_P880_LCD_RES_12X10_16X12;
                    lcd_patch_reg = P880_LCD_RES_12X10_16X12;
                    break;
            }
            break;
    }

    if (reg_num != 0)
    {
        /* H.W. Reset : ON */
        write_reg_mask(CR17, VIACR, 0x00, BIT7);

        write_regx(lcd_patch_reg, reg_num);

        /* H.W. Reset : OFF */
        write_reg_mask(CR17, VIACR, 0x80, BIT7);

        /* Reset PLL */
        write_reg_mask(SR40, VIASR, 0x02, BIT1);
        write_reg_mask(SR40, VIASR, 0x00, BIT1);

        /* Fire! */
        VGAOUT8(VIAWMisc, VGAIN8(VIARMisc) | (BIT2+BIT3));
    }
}

void VIASetLCDOutputPath(VIABIOSInfoPtr pBIOSInfo, int set_iga, int output_interface)
{
    switch(set_iga)
    {
        case IGA1:
            write_reg_mask(CR6B, VIACR, 0x00, BIT3);
            /* Modify for sequence issue. */
            DisableSecondDisplayChannel();
            write_reg_mask(CR6A, VIACR, 0x08, BIT3);
            break;
        case IGA2:
            write_reg_mask(CR6B, VIACR, 0x00, BIT3);
            /*------IGA2 Enable sequence------*/
            /* Replaced by function call. */
            EnableSecondDisplayChannel();
            write_reg_mask(CR6A, VIACR, 0x08, BIT3);
            break;
        case IGA1_IGA2:
            write_reg_mask(CR6B, VIACR, 0x08, BIT3);
            /* Modify for sequence issue. */
            DisableSecondDisplayChannel();
            write_reg_mask(CR6A, VIACR, 0x08, BIT3);
            break;
    }

    switch(output_interface)
    {
        case VIA_DI_DVP0:
            if (set_iga == IGA1)
            {
                write_reg_mask(CR96, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg(CR91, VIACR, 0x00);
                write_reg_mask(CR96, VIACR, 0x10, BIT4);
            }

            break;
            
        case VIA_DI_DVP1:
            if (set_iga== IGA1)
            {
                write_reg_mask(CR9B, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg(CR91, VIACR, 0x00);
                write_reg_mask(CR9B, VIACR, 0x10, BIT4);
            }
            
            break;
            
        case VIA_DI_DFPHIGH:
            if (set_iga== IGA1)
            {
                write_reg_mask(CR97, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg(CR91, VIACR, 0x00);
                write_reg_mask(CR97, VIACR, 0x10, BIT4);
                write_reg_mask(CR96, VIACR, 0x10, BIT4);
            }
            break;
        case VIA_DI_DFPLOW:
            if (set_iga== IGA1)
            {
                write_reg_mask(CR99, VIACR, 0x00, BIT4);
                write_reg_mask(CR9B, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg(CR91, VIACR, 0x00);
                write_reg_mask(CR99, VIACR, 0x10, BIT4);
                write_reg_mask(CR9B, VIACR, 0x10, BIT4);
            }

            break;

        case VIA_DI_DFPHIGHLOW:
            if(( pBIOSInfo->Chipset==VIA_K8M890 ) ||( pBIOSInfo->Chipset==VIA_P4M890)) 
               write_reg_mask(CR97, VIACR, 0x84, BIT7+BIT2+BIT1+BIT0);
            if (set_iga== IGA1)
            {
                write_reg_mask(CR97, VIACR, 0x00, BIT4);
                write_reg_mask(CR99, VIACR, 0x00, BIT4);                
            }
            else
            {
                write_reg(CR91, VIACR, 0x00);
                write_reg_mask(CR97, VIACR, 0x10, BIT4);
                write_reg_mask(CR99, VIACR, 0x10, BIT4);
            }
            break;

        case VIA_DI_LVDS0:
	 	case VIA_DI_LVDS0LVDS1:
            if (set_iga == IGA1)
            {
                write_reg_mask(CR99, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg_mask(CR99, VIACR, 0x10, BIT4);
            }
            
            break;
            
        case VIA_DI_LVDS1:
            if (set_iga == IGA1)
            {
                write_reg_mask(CR97, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg_mask(CR97, VIACR, 0x10, BIT4);
            }
            
            break;
        case VIA_DI_LVDS0DVP1 | VIA_DI_TTL:
            if (set_iga == IGA1)
            {
                write_reg_mask(CR99, VIACR, 0x00, BIT4);
                write_reg_mask(CR9B, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg_mask(CR99, VIACR, 0x10, BIT4);
                write_reg_mask(CR9B, VIACR, 0x10, BIT4);
            }
            break;

        case VIA_DI_LVDS0LVDS1DVP1 | VIA_DI_TTL:
            if (set_iga == IGA1)
            {
                write_reg_mask(CR97, VIACR, 0x00, BIT4);
                write_reg_mask(CR9B, VIACR, 0x00, BIT4);
            }
            else
            {
                write_reg_mask(CR97, VIACR, 0x10, BIT4);
                write_reg_mask(CR9B, VIACR, 0x10, BIT4);
            }
            break;
            
    }
}

int VIAGetModeIndexByPanelSize(int PanelSizeID)
{
	if(PanelSizeID == VIA_800X480_GEN)
	{
		return M800x480_GEN;
	}
	
    switch(PanelSizeID)
    {        
        case VIA_640X480:
            return M640x480;
        case VIA_800X480:
            return M800x480;            
        case VIA_800X600:
            return M800x600;
        case VIA_1024X600:
            return M1024x600;
        case VIA_1024X768:
            return M1024x768;
        case VIA_1280X768:
            return M1280x768;
        case VIA_1280X800:
            return M1280x800;            
        case VIA_1280X1024:
            return M1280x1024;
        case VIA_1400X1050:
            return M1400x1050;
        case VIA_1600X1200:
            return M1600x1200;
        case VIA_1366X768:
            return M1366x768;
        case VIA_1360X768:
            return M1360x768;     
        case VIA_1440X900:
            return M1440x900;
        case VIA_480X640:
            return M480x640;
        default:
            return M1024x768;
    }
}

void VIA_fill_lcd_format(Bool bIsDualEdge, Bool bIsDithering)
{
    CARD8   bdithering = 0, bdual = 0;

    if(bIsDualEdge)
    {
       bdual = BIT4;
    }
    if(bIsDithering)
    {
       bdithering = BIT0;
    }

    write_reg_mask(CR88, VIACR, (bdithering | bdual), BIT4+BIT0);   /* Dual & Dithering */
}


Bool VIALVDSIdentify_VT1631(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo, CARD32 PortNum)
{
    CARD8 Buffer[2];

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIALVDSIdentify_VT1631: Try Port No. 0x%x\n", PortNum));

    pLVDSSettingInfo->I2CPort = PortNum;
    pLVDSSettingInfo->I2CAddress = LVDS_I2C_ADDR_VT1631;
    
    /* Check vendor ID first: */
    if (!Read_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, 0x00, &(Buffer[0])))
    {
        return FALSE;
    }
    
    if (!Read_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, 0x01, &(Buffer[1])))
    {
        return FALSE;
    }

    if (!((Buffer[0] == 0x06) && (Buffer[1] == 0x11)))
    {        
        return FALSE;        
    }

    /* Vendor ID is correct, so continue to check Chip ID: */
    if (!Read_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, 0x02, &(Buffer[0])))
    {
        return FALSE;
    }
    
    if (!Read_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, 0x03, &(Buffer[1])))
    {
        return FALSE;
    }

    if ((Buffer[0] == 0x91) && (Buffer[1] == 0x31))
    {
        if (pBIOSInfo->LVDSSettingInfo.IsDualEdge)
        {
            Write_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, 0x08, 0x0D);
        }
        else
        {
            Write_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, 0x08, 0x09);
        }

        pLVDSSettingInfo->ChipID = VIA_VT1631;
        pBIOSInfo->LVDSSettingInfo.DIPort = VIA_DI_DFPHIGHLOW;
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "Found VIA VT1631 LVDS on port %x!\n", PortNum));        

        return TRUE;
    }

    return FALSE;
}


/*  To determine LCD port.*/
void 
VIAInitLCDDIPort(VIABIOSInfoPtr pBIOSInfo, 
                    LVDSSETTINGINFOPTR pLVDSSettingInfo,
                    Bool lcd2_flag)
{
    if (pLVDSSettingInfo->DIPort != VIA_DI_NONE)        
        return; /* DIPort was assigned by option. */    
    
    if (pLVDSSettingInfo->ChipID == VIA_VT1636)
    {
        if(pBIOSInfo->Chipset == VIA_P4M800PRO) {
            if(lcd2_flag==TRUE)
                pLVDSSettingInfo->DIPort = VIA_DI_DFPHIGH;
            else
                pLVDSSettingInfo->DIPort = VIA_DI_DFPLOW;
        }else
            pLVDSSettingInfo->DIPort = VIA_DI_DVP1;
    }
    else if(pLVDSSettingInfo->ChipID == VIA_INTEGRATED_LVDS)
    {
        /* assign digital port for internal LCD */
        VIACheckDIPortOfIntegratedLVDS(pBIOSInfo, pLVDSSettingInfo,lcd2_flag);
    }
    else
    {        
        /* For the case VT1637 (PCIE LVDS Transmitter) on VT3336/VT3327, */
        /* we should use DFP Low to enable LCD, otherwise we use DFP High-Low as default */
        /* FixMe!! We should find a better method to do this. */
        if(((pBIOSInfo->Chipset == VIA_K8M890)||
            (pBIOSInfo->Chipset == VIA_P4M890)||
            (pBIOSInfo->Chipset == VIA_P4M900)) && 
           (pLVDSSettingInfo->BusWidth == VIA_DI_12BIT))
        {
            pLVDSSettingInfo->DIPort = VIA_DI_DFPLOW;
        }
        else
        {
            pLVDSSettingInfo->DIPort = VIA_DI_DFPHIGHLOW;
        }
           
        pLVDSSettingInfo->BusWidth = VIA_DI_24BIT;     /* Low-Swing LCD always use 24Bit */        
    }

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "LCD DIPort = %d\n", pLVDSSettingInfo->DIPort));
}


void VIALVDSIdentify_IntegratedLVDS(VIABIOSInfoPtr pBIOSInfo)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED,
          "VIALVDSIdentify_IntegratedLVDS\n"));

    if (pBIOSInfo->ActiveDevice == (VIA_DEVICE_LCD | VIA_DEVICE_LCD2)){
        
        if (pBIOSInfo->LVDSSettingInfo.ChipID){
            /* Two dual channel LCD (Internal LVDS + External LVDS): 
               If we have an external LVDS, such as VT1636, we should have
               its chip ID already. */            
            pBIOSInfo->LVDSSettingInfo2.ChipID = VIA_INTEGRATED_LVDS;
        }else{
            /* Two single channel LCD (Internal LVDS + Internal LVDS): */
            pBIOSInfo->LVDSSettingInfo.ChipID = VIA_INTEGRATED_LVDS;
            pBIOSInfo->LVDSSettingInfo2.ChipID = VIA_INTEGRATED_LVDS;
        }
            
    }else{
        /* Single LCD */    
        if (!pBIOSInfo->LVDSSettingInfo.ChipID)
            pBIOSInfo->LVDSSettingInfo.ChipID = VIA_INTEGRATED_LVDS;            
    }
}


/*
    To identify lvds.
*/
Bool VIALVDSIdentify(VIABIOSInfoPtr pBIOSInfo)
{
    /* Sense VT1636 */
    if (!VIALVDSIdentify_VT1636(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo), PORT_INDEX_I2C_31))
    {
        if(!VIALVDSIdentify_VT1636(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo), PORT_INDEX_GPIO_2C))
            VIALVDSIdentify_VT1636(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo), PORT_INDEX_GPIO_25);

        /* Randy two LCD for P4M800 */
        if((pBIOSInfo->TwoLCD)&&(pBIOSInfo->Chipset == VIA_P4M800PRO))
            VIALVDSIdentify_VT1636(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo2), PORT_INDEX_GPIO_3D);
    }

    /* Sense Integrated LVDS: */
    switch (pBIOSInfo->Chipset)
    {
        case VIA_CX700:
        case VIA_VX800:
        case VIA_VX855:
            VIALVDSIdentify_IntegratedLVDS(pBIOSInfo);
            break;
    }        

    if(pBIOSInfo->LVDSSettingInfo.ChipID)
        return TRUE;
    
    /* Sense VT1631 */
    if (VIALVDSIdentify_VT1631(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo), PORT_INDEX_I2C_31))
        return TRUE;    /* Found VT1631 on port 0x31! */

    if (VIALVDSIdentify_VT1631(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo), PORT_INDEX_GPIO_2C))        
        return TRUE;    /* Found VT1631 on port 0x2C! */

    if(!pBIOSInfo->LVDSSettingInfo.ChipID){
        /* Hardwire LVDS */
        pBIOSInfo->LVDSSettingInfo.ChipID = read_reg(VIACR, CR3E) >> 4;
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED,
            "Using Hardwire LVDS! LVDS id=0x%x\n", pBIOSInfo->LVDSSettingInfo.ChipID));
    }

    if(pBIOSInfo->LVDSSettingInfo.ChipID)
        return TRUE;
    else
        return FALSE;
}


void 
VIACheckDIPortOfIntegratedLVDS(VIABIOSInfoPtr pBIOSInfo,
                                       LVDSSETTINGINFOPTR pLVDSSettingInfo,
                                       Bool IsSecLCD)
{
    /* VX855 use LVDS0 as default setting */
    if(pBIOSInfo->Chipset == VIA_VX855) {
        pLVDSSettingInfo->DIPort = VIA_DI_LVDS0;
        return;
    }
        
    /* Dual Channel LCD must use two lvds. */
    if (pLVDSSettingInfo->IsDualEdge)
        pLVDSSettingInfo->DIPort = VIA_DI_LVDS0LVDS1;
    else{
        /* single lcd default use lvds1*/
        if (IsSecLCD)
            pLVDSSettingInfo->DIPort = VIA_DI_LVDS0;
        else
            pLVDSSettingInfo->DIPort = VIA_DI_LVDS1;
    }       
}


Bool VIADelay_Nmsec(VIABIOSInfoPtr pBIOSInfo, CARD32 dwCounter)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    CARD32 i = 0;
    
    if (pBIOSInfo->Chipset >= VIA_VX800 ||
        pBIOSInfo->Chipset == VIA_CX700)
    {        
        VGAOUT8(0x3C4, SR5D);
        VGAOUT8(0x3C5, 0x70);
        
        while (1)
        {
            VGAOUT8(0x3C4, SR5D);
            if(VGAIN8(0x3C5) & BIT7)
            {
                i++;
                VGAOUT8(0x3C4, SR5D);
                VGAOUT8(0x3C5, 0x70);
            }

            if (i==dwCounter)
            {
                break;
            }
        }
    }
    else
    {
        VIADelayIn_usec(pBIOSInfo, dwCounter*1000);
    }
    
    return TRUE;
}

/* TTL-LCD should use software power-sequence control. */
void TTLLCDPowerSequenceOn(VIABIOSInfoPtr pBIOSInfo)
{    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "TTLLCDPowerSequenceOn\n"));

    VIABIOSInfoPtr  pVia = pBIOSInfo;
    
    /* Software control power sequence ON*/    
    VGAOUT8(0x3D4, CR91);
    VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x01);

    /* VDD ON*/    
    VGAOUT8(0x3D4, CR91);
    VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x10);

    /* Delay 25 msec. (25msec is HW proposed) */
    VIADelay_Nmsec(pBIOSInfo, 25);    

    /* DATA ON (TTL-LCD's data is output from DVP1) */
    VGAOUT8(0x3C4, SR1E);
    VGAOUT8(0x3C5, VGAIN8(0x3C5) | 0x30);

    /* Delay 225 msec. (225msec is HW proposed) */
    VIADelay_Nmsec(pBIOSInfo, 225);

    /* Back-Light ON */
    VGAOUT8(0x3D4, CR91);
    VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x02);
}

 
/* TTL-LCD should use software power-sequence control. */
void TTLLCDPowerSequenceOff(VIABIOSInfoPtr pBIOSInfo)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "TTLLCDPowerSequenceOff\n"));

    VIABIOSInfoPtr  pVia = pBIOSInfo;
    
    /* Back-Light OFF */    
    VGAOUT8(0x3D4, CR91);
    VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xFD);

    /* Delay 225 msec. (225msec is HW proposed) */
    VIADelay_Nmsec(pBIOSInfo, 225);
    
    /* DATA OFF (TTL-LCD's data is output from DVP1) */    
    VGAOUT8(0x3C4, SR1E);
    VGAOUT8(0x3C5, VGAIN8(0x3C5) & 0xCF);

    /* Delay 25 msec. (25msec is HW proposed) */
    VIADelay_Nmsec(pBIOSInfo, 25);
    
    /* VDD OFF */
    VGAOUT8(0x3D4, CR91);
    VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xEF);
}


void VIAPowerOnLVDSChannel(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIAPowerOnLVDSChannel\n"));

    switch (pLVDSSettingInfo->DIPort)
    {
        case VIA_DI_LVDS0:
        {
            VGAOUT8(0x3d4, CRD2);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0x7F);
            break;
        }

        case VIA_DI_LVDS1:
        {
            VGAOUT8(0x3d4, CRD2);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0xBF);
            break;
        }

        case VIA_DI_LVDS0LVDS1:
        {
            VGAOUT8(0x3d4, CRD2);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0x3F);
            break;
        }

        case VIA_DI_LVDS0DVP1 | VIA_DI_TTL:
        {
            VGAOUT8(0x3d4, CRD2);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0x7F);
            break;
        }

        case VIA_DI_LVDS0LVDS1DVP1 | VIA_DI_TTL:
        {
            VGAOUT8(0x3d4, CRD2);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0x3F);
            break;
        }
        
    }
}

void VIAPowerOffLVDSChannel(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIAPowerOffLVDSChannel\n"));

    switch (pLVDSSettingInfo->DIPort)
    {
        case VIA_DI_LVDS0:
        {
            VGAOUT8(0x3d4, CRD2);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) | 0x80);
            break;
        }

        case VIA_DI_LVDS1:
        {
            VGAOUT8(0x3d4, CRD2);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) | 0x40);
            break;
        }

        case VIA_DI_LVDS0LVDS1:
        {
            VGAOUT8(0x3d4, CRD2);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) | 0xC0);
            break;
        }

        case VIA_DI_LVDS0DVP1 | VIA_DI_TTL:
        {
            VGAOUT8(0x3d4, CRD2);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) | 0x80);
            break;
        }

        case VIA_DI_LVDS0LVDS1DVP1 | VIA_DI_TTL:
        {
            VGAOUT8(0x3d4, CRD2);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) | 0xC0);
            break;
        }
        
    }
}

void EnableIntegratedLVDS(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    Bool bTurnOnFirstPowerSequence = FALSE;
    Bool bTurnOnSecondPowerSequence = FALSE;

     if(pLVDSSettingInfo->MsbEnable) {
        VGAOUT8(0x3d4, CRD2);
        VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0xFC);
    } else {
        VGAOUT8(0x3d4, CRD2);
        VGAOUT8(0x3d5, VGAIN8(0x3d5) | 0x03);
    }
    
    if (pLVDSSettingInfo->DIPort == VIA_DI_LVDS0LVDS1) {    
        bTurnOnFirstPowerSequence = TRUE;
    } else if (pLVDSSettingInfo->DIPort == VIA_DI_LVDS0) {
        bTurnOnFirstPowerSequence = TRUE;
    } else if (pLVDSSettingInfo->DIPort == VIA_DI_LVDS1) {
        /* For single channel LCD (use LVDS1) */
        bTurnOnSecondPowerSequence = TRUE;
    } else if (pLVDSSettingInfo->DIPort == VIA_DI_DVP0) {
        VGAOUT8(0x3C4, SR1E);
        VGAOUT8(0x3C5, VGAIN8(0x3C5) | 0xC0);
    } else if (pLVDSSettingInfo->DIPort == VIA_DI_DVP1) {
        VGAOUT8(0x3C4, SR1E);
        VGAOUT8(0x3C5, VGAIN8(0x3C5) | 0x30);
    } else if (pLVDSSettingInfo->DIPort == VIA_DI_DFPLOW) {
        VGAOUT8(0x3C4, SR2A);
        VGAOUT8(0x3C5, VGAIN8(0x3C5) | 0x03);
    } else if (pLVDSSettingInfo->DIPort == VIA_DI_DFPHIGH) {
        VGAOUT8(0x3C4, SR2A);
        VGAOUT8(0x3C5, VGAIN8(0x3C5) | 0x0C);
    } else if (pLVDSSettingInfo->DIPort == (VIA_DI_LVDS0DVP1|VIA_DI_TTL)) {
        /* Power on LVDS channel. */
        VIAPowerOnLVDSChannel(pBIOSInfo, pLVDSSettingInfo);
        
        /* For Qunta's TTL LCD (IL1) */
        /* Software control Power Sequence */        
        TTLLCDPowerSequenceOn(pBIOSInfo);      
         
        return;
    } else if (pLVDSSettingInfo->DIPort == (VIA_DI_LVDS0LVDS1DVP1|VIA_DI_TTL)) {
        /* for 409 18bit TTL panel */
        VGAOUT8(0x3D4, CRF3);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x80);
        /* Turn on DVP1 pad */
        VGAOUT8(0x3C4, SR1E);
        VGAOUT8(0x3C5, VGAIN8(0x3C5) | 0x30);
        bTurnOnFirstPowerSequence = TRUE;
    } else {
        /* For single channel LCD (use LVDS0) */
        bTurnOnFirstPowerSequence = TRUE;
    }
 
    if (bTurnOnFirstPowerSequence)
    {
        if (pBIOSInfo->Chipset == VIA_VX800 ||
            pBIOSInfo->Chipset == VIA_CX700)
        {
            VIASWPowerSequenceON(pBIOSInfo, pLVDSSettingInfo);
        }
        else
        {
            /* Use first power sequence control: */
            
            /* Use hardware control power sequence. */
            VGAOUT8(0x3d4, CR91);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0xFE);

            /* Turn on back light. */
            VGAOUT8(0x3d4, CR91);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0x3F);

            /* Turn on hardware power sequence. */
            VGAOUT8(0x3d4, CR6A);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) | 0x08);
        }
    }
        
    if (bTurnOnSecondPowerSequence)
    {
        if (pBIOSInfo->Chipset == VIA_VX800 ||
            pBIOSInfo->Chipset == VIA_CX700)
        {
            VIASWPowerSequenceON(pBIOSInfo, pLVDSSettingInfo);
        }
        else
        {
            /* Use second power sequence control: */
            
            /* Use hardware control power sequence. */
            VGAOUT8(0x3d4, CRD3);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0xFE);

            /* Turn on back light. */
            VGAOUT8(0x3d4, CRD3);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0x3F);

            /* Turn on hardware power sequence. */
            VGAOUT8(0x3d4, CRD4);
            VGAOUT8(0x3d5, VGAIN8(0x3d5) | 0x02);
        }
    }
    
    /* Turn DFP High/Low pad on. */
    VGAOUT8(0x3c4, SR2A);
    VGAOUT8(0x3c5, VGAIN8(0x3c5) | 0x0F);

    /* Power on LVDS channel. */
    VIAPowerOnLVDSChannel(pBIOSInfo, pLVDSSettingInfo);
}


void DisableIntegratedLVDS(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    Bool bTurnOffFirstPowerSequence = FALSE;
    Bool bTurnOffSecondPowerSequence = FALSE;

    if (pLVDSSettingInfo->DIPort == VIA_DI_LVDS0LVDS1)
    {
            bTurnOffFirstPowerSequence = TRUE;
    }
    else if (pLVDSSettingInfo->DIPort == VIA_DI_LVDS1)
    {
        /* For single channel LCD (use LVDS1) */
        bTurnOffSecondPowerSequence = TRUE;
    }
    else if (pLVDSSettingInfo->DIPort == (VIA_DI_LVDS0DVP1 | VIA_DI_TTL))
    {                
        TTLLCDPowerSequenceOff(pBIOSInfo);        
    }
    else if (pLVDSSettingInfo->DIPort == (VIA_DI_LVDS0LVDS1DVP1 | VIA_DI_TTL))
    {    
        bTurnOffFirstPowerSequence = TRUE;
        VGAOUT8(0x3C4, SR1E);
        VGAOUT8(0x3C5, VGAIN8(0x3C5) & 0xCF);        
    }    
    else
    {
        /* For single channel LCD (use LVDS0) */
        bTurnOffFirstPowerSequence = TRUE;
    }

    if (bTurnOffFirstPowerSequence)
    {
        if (pBIOSInfo->Chipset == VIA_VX800 ||
			pBIOSInfo->Chipset == VIA_CX700)
        {
            VIASWPowerSequenceOFF(pBIOSInfo, pLVDSSettingInfo);
        }
        else
        {
	        /* Use first power sequence control: */

	        /* Turn off power sequence. */
	        VGAOUT8(0x3d4, CR6A);
	        VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0xF7);

	        usleep(1);

	        /* Turn off back light. */
	        VGAOUT8(0x3d4, CR91);
	        VGAOUT8(0x3d5, 0xC0);
	    }
    }

    if (bTurnOffSecondPowerSequence)
    {
        if (pBIOSInfo->Chipset == VIA_VX800 ||
			pBIOSInfo->Chipset == VIA_CX700)
        {
            VIASWPowerSequenceOFF(pBIOSInfo, pLVDSSettingInfo);
        }
        else
	    {
	        /* Use second power sequence control: */

	        /* Turn off power sequence. */
	        VGAOUT8(0x3d4, CRD4);
	        VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0xFD);

	        usleep(1);

	        /* Turn off back light. */
	        VGAOUT8(0x3d4, CRD3);
	        VGAOUT8(0x3d5, 0xC0);
	    }
    }

    /* Turn DFP High/Low Pad off. */
    VGAOUT8(0x3c4, SR2A);
    VGAOUT8(0x3c5, VGAIN8(0x3c5) & 0xF0);

    /* Power off LVDS channel. */
    VIAPowerOffLVDSChannel(pBIOSInfo, pLVDSSettingInfo);
}


/*
    Set LCD power sequence.
*/
void VIASetPowerSequence(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIASetPowerSequence\n"));
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    CARD32 i;
    CARD32 TD0 = 0, TD1 = 0, TD2 = 0, TD3 = 0;

    /*
        1. Formula: 
            2^13 X 0.0698uSec [1/14.318MHz] = 8192 X 0.0698uSec =572.1uSec
            Timer = Counter x 572 uSec
        2. Note:
            0.0698 uSec is too small to compute for hardware. So we multipfy a 
            reference value(2^13) to make it big enough to compute for hardware.
        3. Note:
            The meaning of the TD0~TD3 are count of the clock. 
            TD(sec) = (sec)/(per clock) x (count of clocks)
    */
    
    TD0 = LCD_POWER_SEQUENCE_TD0;
    TD1 = LCD_POWER_SEQUENCE_TD1;
    TD2 = LCD_POWER_SEQUENCE_TD2;
    TD3 = LCD_POWER_SEQUENCE_TD3;
            
    /* Fill first power sequence timer */
    /* Set CRD4[0] to "0" to select 1st LCD power sequence. */ 
    VGAOUT8(0x3D4, CRD4);
    VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xFE);

    /* TD0 */
    VGAOUT8(0x3D4, CR8B);
    VGAOUT8(0x3D5, (TD0&0xFF));          
    VGAOUT8(0x3D4, CR8F);
    VGAOUT8(0x3D5, ((TD0>>8)& 0x0F) | (VGAIN8(0x3D5)&0xF0));     
    
    /* TD1 */   
    VGAOUT8(0x3D4, CR8C);
    VGAOUT8(0x3D5, (TD1&0xFF));      
    VGAOUT8(0x3D4, CR8F);
    VGAOUT8(0x3D5, ((TD1>>8)& 0xF0) | (VGAIN8(0x3D5)&0x0F));     

    /* TD2 */    
    VGAOUT8(0x3D4, CR8D);
    VGAOUT8(0x3D5, (TD2&0xFF));      
    VGAOUT8(0x3D4, CR90);
    VGAOUT8(0x3D5, ((TD2>>8)& 0x0F) | (VGAIN8(0x3D5)&0xF0));   

    /* TD3 */    
    VGAOUT8(0x3D4, CR8E);
    VGAOUT8(0x3D5, (TD3&0xFF));      
    VGAOUT8(0x3D4, CR90);
    VGAOUT8(0x3D5, ((TD3>>8)& 0xF0) | (VGAIN8(0x3D5)&0x0F));   

    /* Fill second power sequence timer */
    if((pBIOSInfo->Chipset == VIA_CX700) ||
       (pBIOSInfo->Chipset == VIA_VX800))
    {
        /* Set CRD4[0] to "1" to select 2nd LCD power sequence. */ 
        VGAOUT8(0x3D4, CRD4);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x01);

        /* TD0 */
 		VGAOUT8(0x3D4, CR8B);
 		VGAOUT8(0x3D5, (TD0&0xFF));          
   		VGAOUT8(0x3D4, CR8F);
		VGAOUT8(0x3D5, ((TD0>>8)& 0x0F) | (VGAIN8(0x3D5)&0xF0));     
 
		/* TD1 */   
		VGAOUT8(0x3D4, CR8C);
		VGAOUT8(0x3D5, (TD1&0xFF));      
		VGAOUT8(0x3D4, CR8F);
 		VGAOUT8(0x3D5, ((TD1>>8)& 0xF0) | (VGAIN8(0x3D5)&0x0F));     

		/* TD2 */    
		VGAOUT8(0x3D4, CR8D);
		VGAOUT8(0x3D5, (TD2&0xFF));      
		VGAOUT8(0x3D4, CR90);
		VGAOUT8(0x3D5, ((TD2>>8)& 0x0F) | (VGAIN8(0x3D5)&0xF0));   

        /* TD3 */    
		VGAOUT8(0x3D4, CR8E);
		VGAOUT8(0x3D5, (TD3&0xFF));      
		VGAOUT8(0x3D4, CR90);
		VGAOUT8(0x3D5, ((TD3>>8)& 0xF0) | (VGAIN8(0x3D5)&0x0F));   		
	}            
}



/*
    Set LCD software power sequence.
*/
void VIASWPowerSequenceON(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{   
    CARD32 i;
    CARD32 TD0 = 0, TD1 = 0, TD2 = 0, TD3 = 0;
    
    VIABIOSInfoPtr  pVia = pBIOSInfo;        
        
    TD0 = LCD_POWER_SEQUENCE_TD0;
    TD1 = LCD_POWER_SEQUENCE_TD1;
    TD2 = LCD_POWER_SEQUENCE_TD2;
    TD3 = LCD_POWER_SEQUENCE_TD3;  

    if ((pLVDSSettingInfo->DIPort == VIA_DI_LVDS0)||
        (pLVDSSettingInfo->DIPort == VIA_DI_LVDS0LVDS1))
    {
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "LVDS0 SW Power Sequence ON\n"));

        /* Software control power sequence ON*/    
        VGAOUT8(0x3D4, CR91);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0x7F);

        VGAOUT8(0x3D4, CR91);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x01);

        /* Delay TD0 msec.  */
        VIADelay_Nmsec(pBIOSInfo, TD0);

        /* VDD ON*/    
        VGAOUT8(0x3D4, CR91);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x10);

        /* Delay TD1 msec.*/
        VIADelay_Nmsec(pBIOSInfo, TD1);    

        /* DATA ON */    
        VGAOUT8(0x3D4, CR91);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x08);

        /* Delay TD2 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD2);

        /* VEE ON (unused on vt3353)*/
        VGAOUT8(0x3D4, CR91);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x04);        

        /* Delay TD3 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD3);

        /* Back-Light ON */    
        VGAOUT8(0x3D4, CR91);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x02);
    } 
   
    if (pLVDSSettingInfo->DIPort == VIA_DI_LVDS1)
    {
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "LVDS1 SW Power Sequence ON\n"));

        /* Secondary power hardware power sequence enable 0:off 1: on */
        VGAOUT8(0x3D4, CRD4);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xFD);
        
        /* Software control power sequence ON*/    
        VGAOUT8(0x3D4, CRD3);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x01);

        /* Delay TD0 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD0);

        /* VDD ON*/    
        VGAOUT8(0x3D4, CRD3);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x10);

        /* Delay TD1 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD1);

        /* DATA ON */    
        VGAOUT8(0x3D4, CRD3);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x08);

        /* Delay TD2 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD2);

        /* VEE ON (unused on vt3353)*/
        VGAOUT8(0x3D4, CRD3);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x04);        

        /* Delay TD3 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD3);

        /* Back-Light ON */    
        VGAOUT8(0x3D4, CRD3);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) | 0x02);
    }
}



/*
    Set LCD software power sequence.
*/
void VIASWPowerSequenceOFF(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{    
    CARD32 i;
    CARD32 TD0 = 0, TD1 = 0, TD2 = 0, TD3 = 0;
	
    VIABIOSInfoPtr  pVia = pBIOSInfo;        
        
    TD0 = LCD_POWER_SEQUENCE_TD0;
    TD1 = LCD_POWER_SEQUENCE_TD1;
    TD2 = LCD_POWER_SEQUENCE_TD2;
    TD3 = LCD_POWER_SEQUENCE_TD3;      

	if ((pLVDSSettingInfo->DIPort == VIA_DI_LVDS0)||(pLVDSSettingInfo->DIPort == VIA_DI_LVDS0LVDS1))
	{
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "LVDS0 SW Power Sequence OFF\n"));

        /* Back-Light OFF */    
        VGAOUT8(0x3D4, CR91);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xFD);

        /* Delay TD3 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD3);

        /* VEE OFF (unused on vt3353)*/
        VGAOUT8(0x3D4, CR91);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xFB);        

        /* Delay TD2 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD2);
        
        /* DATA OFF */    
        VGAOUT8(0x3D4, CR91);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xF7);

        /* Delay TD1 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD1);
        
        /* VDD OFF */    
        VGAOUT8(0x3D4, CR91);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xEF);
	} 
	
	if (pLVDSSettingInfo->DIPort == VIA_DI_LVDS1)
	{
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "LVDS1 SW Power Sequence OFF\n"));
        
        /* Back-Light OFF */    
        VGAOUT8(0x3D4, CRD3);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xFD);        

        /* Delay TD3 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD3);

        /* VEE OFF */
        VGAOUT8(0x3D4, CRD3);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xFB);

        /* Delay TD2 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD2);        
        
        /* DATA OFF */    
        VGAOUT8(0x3D4, CRD3);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xF7);

        /* Delay TD1 msec. */
        VIADelay_Nmsec(pBIOSInfo, TD1);
        
        /* VDD OFF */    
        VGAOUT8(0x3D4, CRD3);
        VGAOUT8(0x3D5, VGAIN8(0x3D5) & 0xEF);        
	}
}


void VIAEnableLCD(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIAEnableLCD\n"));

    /* VX800, CX700 have HW issue, so we'd better use SW power sequence */
    if (pBIOSInfo->Chipset != VIA_VX800 &&
        pBIOSInfo->Chipset != VIA_CX700)
    {
        VIASetPowerSequence(pBIOSInfo, pLVDSSettingInfo);
    }

    /* Because VT3324 has 2 power sequence controllers, */
    /* so it's better to divide the codes from other LVDS chips to be more clearly.*/
    if (pLVDSSettingInfo->ChipID == VIA_INTEGRATED_LVDS)
    {
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIAEnableLCD VIA_INTEGRATED_LVDS\n"));
        EnableIntegratedLVDS(pBIOSInfo, pLVDSSettingInfo);
        return;
    }

    if (pLVDSSettingInfo->ChipID == VIA_VT1636)
    {
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIAEnableLCD VIA_VT1636\n"));
        VIAEnableLVDS_VT1636(pBIOSInfo, pLVDSSettingInfo);
        return;
    }

    /* Hardware control power sequence */
    VGAOUT8(0x3d4, 0x91);
    VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0xFE);
    
    /* CLE266Cx: Enable LCD display path & back light */
    VGAOUT8(0x3d4, 0x91);
    VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0x3F);

    /* Enable LCD */
    VGAOUT8(0x3d4, 0x6A);
    VGAOUT8(0x3d5, VGAIN8(0x3d5) | 0x08);

    /* if active LCD, turn DFP High/Low pad on */
    switch(pLVDSSettingInfo->DIPort)
    {
    case VIA_DI_DFPHIGH:
        VGAOUT8(0x3c4, SR2A);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) | 0x0C);
        break;
    case VIA_DI_DFPLOW:
        VGAOUT8(0x3c4, SR2A);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) | 0x03);
        break;
    case VIA_DI_DFPHIGHLOW:
        VGAOUT8(0x3c4, SR2A);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) | 0x0f);
        break;
    }
    
}

void VIADisableLCD(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIADisableLCD\n"));

    /* Because VT3324 has 2 power sequence controllers, */
    /* so it's better to divide the codes from other LVDS chips to be more clearly.*/
    if (pLVDSSettingInfo->ChipID == VIA_INTEGRATED_LVDS)
    {
        DisableIntegratedLVDS(pBIOSInfo, pLVDSSettingInfo);
        return;
    }

    if (pLVDSSettingInfo->ChipID == VIA_VT1636)
    {
        VIADisableLVDS_VT1636(pBIOSInfo, pLVDSSettingInfo);
        return;
    }
    
    /* Disable LCD */
    VGAOUT8(0x3d4, 0x6A);
    VGAOUT8(0x3d5, VGAIN8(0x3d5) & 0xF7);
    usleep(1);

    /* if not active LCD, turn off VDD, turn off DFP High-Low */
    if (pBIOSInfo->IsBiosSupportLCD)
    {
        /* Software off Back Light */
        VGAOUT8(0x3d4, 0x91);
        VGAOUT8(0x3d5, 0xC0);
        /* DFP High/Low Pad off */
        VGAOUT8(0x3c4, 0x2A);
        VGAOUT8(0x3c5, VGAIN8(0x3c5) & 0xF0);
    }
}

Bool 
VIAIsPanelSizeValid(CARD32 width, CARD32 height)
{
	int i;
	
	for (i = 0; i < NUM_SUPPORT_PANEL; i++) {
		if ((ViaSupportPanel[i].Width == width)&&
			    (ViaSupportPanel[i].Height == height))
			return TRUE;
		else
			continue;
	}
	return FALSE;
}


/*
    Set LCD Panel Info according to Panel ID.
*/
Bool VIASetLCDPanelInfo(VIABIOSInfoPtr pBIOSInfo, int panel_id)
{
    switch (panel_id)
    {
        case 0x0:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_640X480;
            break;
        case 0x1:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_800X600;
            break;
        case 0x2:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1024X768;
            break;
        case 0x3:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1280X768;
            break;
        case 0x4:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1280X1024;
            break;
        case 0x5:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1400X1050;
            break;
        case 0x6:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1440X900;
            break;
        case 0x7:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1280X800;
            break;
        case 0x8:       /*For Quanta*/
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_800X480;
            break;
        case 0x9:  
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1024X600;
            break;  
        case 0xA:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1366X768;
            break;        
        case 0xB:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1600X1200;
            break;
        case 0xC: /* Need to be fixed to 1680x1050 */
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1280X768;
            break;
        case 0xD: /* Need to be fixed to 1920x1200 */
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1280X1024;
            break;
        case 0xE: /* Need to be fixed to 640x240 */
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1600X1200;
            break;
         case 0xF:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_480X640;
            break;              
        case 0x10:  /*For Panasonic 1280x768 18bit Dual-Channel Panel */
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1280X768;
            break;            
        case 0x11:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1360X768;
            break; 
        case 0x12:
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_1024X768;
            break;
        case 0x13:  /*General 8x4 panel use this setting*/
            pBIOSInfo->LVDSSettingInfo.PanelSizeID = VIA_800X480_GEN;
            break;
        default:
            return FALSE;
    }
    return TRUE;

}

void VIAPreInit_LVDS(VIABIOSInfoPtr pBIOSInfo)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIAPreInit_LVDS\n"));

    VIAGetPanelSizeByIndex(pBIOSInfo->LVDSSettingInfo.PanelSizeID,
                           &(pBIOSInfo->LVDSSettingInfo.PanelSizeX),
                           &(pBIOSInfo->LVDSSettingInfo.PanelSizeY));

    /* About LCD2, we just set the same panel info as LCD1 now: */
    pBIOSInfo->LVDSSettingInfo2.PanelSizeID = pBIOSInfo->LVDSSettingInfo.PanelSizeID;
    pBIOSInfo->LVDSSettingInfo2.IsDualEdge = pBIOSInfo->LVDSSettingInfo.IsDualEdge;
    pBIOSInfo->LVDSSettingInfo2.IsDithering = pBIOSInfo->LVDSSettingInfo.IsDithering;
    pBIOSInfo->LVDSSettingInfo2.PanelSizeX = pBIOSInfo->LVDSSettingInfo.PanelSizeX;
    pBIOSInfo->LVDSSettingInfo2.PanelSizeY = pBIOSInfo->LVDSSettingInfo.PanelSizeY;
    pBIOSInfo->LVDSSettingInfo2.ChipID = VIA_NONETRANSMITTER;
    

    VIALVDSIdentify(pBIOSInfo);
       
    /* If LVDS type is assigned, we have to determine LCD port. */
    if (pBIOSInfo->LVDSSettingInfo.ChipID)
        VIAInitLCDDIPort(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo),FALSE);
        
    if (pBIOSInfo->LVDSSettingInfo2.ChipID)
        VIAInitLCDDIPort(pBIOSInfo, &(pBIOSInfo->LVDSSettingInfo2),TRUE);
}

unsigned char VIAGetDeviceDetectLCD(VIABIOSInfoPtr pBIOSInfo)
{
    unsigned char   ret = 0x0;

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIAGetDeviceDetectLCD\n"));
    
    if (pBIOSInfo->LVDSSettingInfo.ChipID)        
    {
        ret = VIA_DEVICE_LCD;
    }

    if (pBIOSInfo->LVDSSettingInfo2.ChipID)
    {
        ret |= VIA_DEVICE_LCD2;
    }

    return ret;
}

struct display_timing lcd_centering_timging(struct display_timing mode_crt_reg, struct display_timing panel_crt_reg)
{
    struct display_timing crt_reg;

    crt_reg.hor_total = panel_crt_reg.hor_total;
    crt_reg.hor_addr = mode_crt_reg.hor_addr;
    crt_reg.hor_blank_start = (panel_crt_reg.hor_addr - mode_crt_reg.hor_addr)/2 + crt_reg.hor_addr;
    crt_reg.hor_blank_end = panel_crt_reg.hor_blank_end;
    crt_reg.hor_sync_start = (panel_crt_reg.hor_sync_start - panel_crt_reg.hor_blank_start) + crt_reg.hor_blank_start;
    crt_reg.hor_sync_end = panel_crt_reg.hor_sync_end;

    crt_reg.ver_total = panel_crt_reg.ver_total;
    crt_reg.ver_addr = mode_crt_reg.ver_addr;
    crt_reg.ver_blank_start = (panel_crt_reg.ver_addr - mode_crt_reg.ver_addr)/2 + crt_reg.ver_addr;
    crt_reg.ver_blank_end = panel_crt_reg.ver_blank_end;
    crt_reg.ver_sync_start = (panel_crt_reg.ver_sync_start - panel_crt_reg.ver_blank_start) + crt_reg.ver_blank_start;
    crt_reg.ver_sync_end = panel_crt_reg.ver_sync_end;

    return crt_reg;
}

void load_crtc_shadow_timing(VIABIOSInfoPtr pBIOSInfo, struct display_timing mode_timing, struct display_timing panel_timing)
{
    struct io_register  *reg = NULL;
    int                 i;
    int                 load_reg_Num = 0;
    int                 reg_value = 0;

    if(!(pBIOSInfo->LVDSSettingInfo.Center))    /* Expansion */
    {
       for(i=12;i<20;i++)
       {
          switch(i)
          {
           case H_TOTAL_SHADOW_INDEX:
                reg_value = IGA2_HOR_TOTAL_SHADOW_FORMULA(panel_timing.hor_total);
                load_reg_Num = iga2_shadow_crtc_reg.hor_total_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.hor_total_shadow.reg;
                break;
           case H_BLANK_END_SHADOW_INDEX:
                reg_value = IGA2_HOR_BLANK_END_SHADOW_FORMULA(panel_timing.hor_blank_start, panel_timing.hor_blank_end);
                load_reg_Num = iga2_shadow_crtc_reg.hor_blank_end_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.hor_blank_end_shadow.reg;
                break;
           case V_TOTAL_SHADOW_INDEX:
                reg_value = IGA2_VER_TOTAL_SHADOW_FORMULA(panel_timing.ver_total);
                load_reg_Num = iga2_shadow_crtc_reg.ver_total_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_total_shadow.reg;
                break;
           case V_ADDR_SHADOW_INDEX:
                reg_value = IGA2_VER_ADDR_SHADOW_FORMULA(panel_timing.ver_addr);
                load_reg_Num = iga2_shadow_crtc_reg.ver_addr_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_addr_shadow.reg;
                break;
           case V_BLANK_SATRT_SHADOW_INDEX:
                reg_value = IGA2_VER_BLANK_START_SHADOW_FORMULA(panel_timing.ver_blank_start);
                load_reg_Num = iga2_shadow_crtc_reg.ver_blank_start_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_blank_start_shadow.reg;
                break;
           case V_BLANK_END_SHADOW_INDEX:
                reg_value = IGA2_VER_BLANK_END_SHADOW_FORMULA(panel_timing.ver_blank_start, panel_timing.ver_blank_end);
                load_reg_Num = iga2_shadow_crtc_reg.ver_blank_end_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_blank_end_shadow.reg;
                break;
           case V_SYNC_SATRT_SHADOW_INDEX:
                reg_value = IGA2_VER_SYNC_START_SHADOW_FORMULA(panel_timing.ver_sync_start);
                load_reg_Num = iga2_shadow_crtc_reg.ver_sync_start_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_sync_start_shadow.reg;
                break;
           case V_SYNC_END_SHADOW_INDEX:
                reg_value = IGA2_VER_SYNC_END_SHADOW_FORMULA(panel_timing.ver_sync_start, panel_timing.ver_sync_end);
                load_reg_Num = iga2_shadow_crtc_reg.ver_sync_end_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_sync_end_shadow.reg;
                break;
          }
          load_reg(reg_value, load_reg_Num, reg, VIACR);
       }
    }
    else    /* Centering */
    {
       for(i=12;i<20;i++)
       {
          switch(i)
          {
           case H_TOTAL_SHADOW_INDEX:
                reg_value = IGA2_HOR_TOTAL_SHADOW_FORMULA(panel_timing.hor_total);
                load_reg_Num = iga2_shadow_crtc_reg.hor_total_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.hor_total_shadow.reg;
                break;
           case H_BLANK_END_SHADOW_INDEX:
                reg_value = IGA2_HOR_BLANK_END_SHADOW_FORMULA(panel_timing.hor_blank_start, panel_timing.hor_blank_end);
                load_reg_Num = iga2_shadow_crtc_reg.hor_blank_end_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.hor_blank_end_shadow.reg;
                break;
           case V_TOTAL_SHADOW_INDEX:
                reg_value = IGA2_VER_TOTAL_SHADOW_FORMULA(panel_timing.ver_total);
                load_reg_Num = iga2_shadow_crtc_reg.ver_total_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_total_shadow.reg;
                break;
           case V_ADDR_SHADOW_INDEX:
                reg_value = IGA2_VER_ADDR_SHADOW_FORMULA(mode_timing.ver_addr);
                load_reg_Num = iga2_shadow_crtc_reg.ver_addr_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_addr_shadow.reg;
                break;
           case V_BLANK_SATRT_SHADOW_INDEX:
                reg_value = IGA2_VER_BLANK_START_SHADOW_FORMULA(mode_timing.ver_blank_start);
                load_reg_Num = iga2_shadow_crtc_reg.ver_blank_start_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_blank_start_shadow.reg;
                break;
           case V_BLANK_END_SHADOW_INDEX:
                reg_value = IGA2_VER_BLANK_END_SHADOW_FORMULA(panel_timing.ver_blank_start, panel_timing.ver_blank_end);
                load_reg_Num = iga2_shadow_crtc_reg.ver_blank_end_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_blank_end_shadow.reg;
                break;
           case V_SYNC_SATRT_SHADOW_INDEX:
                reg_value = IGA2_VER_SYNC_START_SHADOW_FORMULA((panel_timing.ver_sync_start - panel_timing.ver_blank_start) + (panel_timing.ver_addr - mode_timing.ver_addr)/2 + mode_timing.ver_addr);
                load_reg_Num = iga2_shadow_crtc_reg.ver_sync_start_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_sync_start_shadow.reg;
                break;
           case V_SYNC_END_SHADOW_INDEX:
                reg_value = IGA2_VER_SYNC_END_SHADOW_FORMULA((panel_timing.ver_sync_start - panel_timing.ver_blank_start) + (panel_timing.ver_addr - mode_timing.ver_addr)/2 + mode_timing.ver_addr, panel_timing.ver_sync_end);
                load_reg_Num = iga2_shadow_crtc_reg.ver_sync_end_shadow.reg_num;
                reg = iga2_shadow_crtc_reg.ver_sync_end_shadow.reg;
                break;
          }
          load_reg(reg_value, load_reg_Num, reg, VIACR);
       }
    }
}



void VIALCDPatchSkew(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "VIALCDPatchSkew: DIPort=%d\n", pLVDSSettingInfo->DIPort));

    /* Load graphic chip and lvds's DPA setting. */     
    VIALoadDPASetting_Gfx(pBIOSInfo, pLVDSSettingInfo->ChipID,pLVDSSettingInfo->DIPort, VIAGetClkRangeIndex(pLVDSSettingInfo->VClk));    
    VIALoadDPASetting_LVDS(pBIOSInfo, pLVDSSettingInfo);

    /* Replace the default patch if get user DPA settings from xorg: */
    if (pBIOSInfo->UserDPASettingInfo.HasUserSetting_Gfx)
    {        
        VIASetDPA_Gfx(pLVDSSettingInfo->DIPort, &(pBIOSInfo->UserDPASettingInfo.GfxDPA));        
    }

    if ((pLVDSSettingInfo->ChipID == VIA_VT1636) && pBIOSInfo->UserDPASettingInfo.HasUserSetting_VT1636)
    {
        VIASetDPA_VT1636(pBIOSInfo, pLVDSSettingInfo, &(pBIOSInfo->UserDPASettingInfo.VT1636DPA));
    }
}



void VIALoadScalingFactorForP4M900(VIABIOSInfoPtr pBIOSInfo, int set_hres, int set_vres, int panel_hres, int panel_vres)
{    
    int     h_scaling_factor;
    int     v_scaling_factor;
    CARD8   cra2 = 0;
    CARD8   cr77 = 0;
    CARD8   cr78 = 0;
    CARD8   cr79 = 0;
    CARD8   cr9f = 0;
    
    /* Check if expansion for horizontal */
    if(set_hres < panel_hres)
    {
        /* Load Horizontal Scaling Factor */
    
        /* For VIA_K8M800 or later chipsets. */
        h_scaling_factor = K800_LCD_HOR_SCF_FORMULA(set_hres, panel_hres);
        cr9f = h_scaling_factor & 0x0003; /* HSCaleFactor[1:0] at CR9F[1:0] */        
        cr77 = (h_scaling_factor & 0x03FC)>>2; /* HSCaleFactor[9:2] at CR77[7:0] */        
        cr79 = (h_scaling_factor & 0x0C00)>>10; /* HSCaleFactor[11:10] at CR79[5:4] */
        cr79 <<= 4;
        
        /* Horizontal scaling enabled */
        cra2 = 0xC0;
        
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "Horizontal Scaling value = %d\n", h_scaling_factor));
    }
    else
    {
        /* Horizontal scaling disabled */
        cra2 = 0x00;        
    }

    
    /* Check if expansion for vertical */
    if(set_vres < panel_vres)
    {
        /* Load Vertical Scaling Factor */
        
        /* For VIA_K8M800 or later chipsets. */
        v_scaling_factor = K800_LCD_VER_SCF_FORMULA(set_vres, panel_vres);
        
        /* Vertical scaling enabled */
        cra2 |= 0x08;
        cr79 |= ((v_scaling_factor & 0x0001)<<3); /* VSCaleFactor[0] at CR79[3] */                
        cr78 |= (v_scaling_factor & 0x01FE) >> 1; /* VSCaleFactor[8:1] at CR78[7:0] */        
        cr79 |= ((v_scaling_factor & 0x0600) >> 9) << 6; /* VSCaleFactor[10:9] at CR79[7:6] */
        
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_INFO, "Vertical Scaling value = %d\n", v_scaling_factor));
    }
    else
    {
        /* Vertical scaling disabled */
        cra2 |= 0x00;      
    }

    write_reg_mask(CRA2, VIACR, cra2, BIT3+BIT6+BIT7);
    write_reg_mask(CR77, VIACR, cr77, 0xFF);
    write_reg_mask(CR78, VIACR, cr78, 0xFF);
    write_reg_mask(CR79, VIACR, cr79, 0xF8);
    write_reg_mask(CR9F, VIACR, cr9f, BIT0+BIT1);
    
}



