/*
 * Copyright 1998-2009 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2009 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */


#ifndef __VIA_MODE_H__
#define __VIA_MODE_H__

#include "via_common.h"

#define STD_SR_NUM   							0x04
#define STD_GR_NUM   							0x09
#define STD_AR_NUM   							0x14

/* Bits per pixel */
#define BPP8    									8
#define BPP16   									16
#define BPP32   									32

/* Definition CRTC Timing Index */
#define VIA_H_TOTAL_INDEX               				0
#define VIA_H_ADDR_INDEX                				1
#define VIA_H_BLANK_START_INDEX         			2
#define VIA_H_BLANK_END_INDEX           			3
#define VIA_H_SYNC_START_INDEX          			4
#define VIA_H_SYNC_END_INDEX           			5
#define VIA_V_TOTAL_INDEX               				6
#define VIA_V_ADDR_INDEX                				7
#define VIA_V_BLANK_START_INDEX         			8
#define VIA_V_BLANK_END_INDEX           			9
#define VIA_V_SYNC_START_INDEX          			10
#define VIA_V_SYNC_END_INDEX            			11
#define VIA_H_TOTAL_SHADOW_INDEX        		12
#define VIA_H_BLANK_END_SHADOW_INDEX    		13
#define VIA_V_TOTAL_SHADOW_INDEX        		14
#define VIA_V_ADDR_SHADOW_INDEX         			15
#define VIA_V_BLANK_SATRT_SHADOW_INDEX  		16
#define VIA_V_BLANK_END_SHADOW_INDEX    		17
#define VIA_V_SYNC_SATRT_SHADOW_INDEX   		18
#define VIA_V_SYNC_END_SHADOW_INDEX     		19

/* Define Register Number for IGA1 CRTC Timing */
#define VIA_IGA1_HOR_TOTAL_REG_NUM          		2           /* location: {CR00,0,7},{CR36,3,3} */
#define VIA_IGA1_HOR_ADDR_REG_NUM           		1           /* location: {CR01,0,7} */
#define VIA_IGA1_HOR_BLANK_START_REG_NUM    	1           /* location: {CR02,0,7} */
#define VIA_IGA1_HOR_BLANK_END_REG_NUM      	3           /* location: {CR03,0,4},{CR05,7,7},{CR33,5,5} */
#define VIA_IGA1_HOR_SYNC_START_REG_NUM     	2           /* location: {CR04,0,7},{CR33,4,4} */
#define VIA_IGA1_HOR_SYNC_END_REG_NUM       	1           /* location: {CR05,0,4} */
#define VIA_IGA1_VER_TOTAL_REG_NUM          		4           /* location: {CR06,0,7},{CR07,0,0},{CR07,5,5},{CR35,0,0} */
#define VIA_IGA1_VER_ADDR_REG_NUM           		4           /* location: {CR12,0,7},{CR07,1,1},{CR07,6,6},{CR35,2,2} */
#define VIA_IGA1_VER_BLANK_START_REG_NUM    	4           /* location: {CR15,0,7},{CR07,3,3},{CR09,5,5},{CR35,3,3} */
#define VIA_IGA1_VER_BLANK_END_REG_NUM      	1           /* location: {CR16,0,7} */
#define VIA_IGA1_VER_SYNC_START_REG_NUM     	4           /* location: {CR10,0,7},{CR07,2,2},{CR07,7,7},{CR35,1,1} */
#define VIA_IGA1_VER_SYNC_END_REG_NUM       	1           /* location: {CR11,0,3} */

/* Define Register Number for IGA2 CRTC Timing */
#define VIA_IGA2_HOR_TOTAL_REG_NUM          		2           /* location: {CR50,0,7},{CR55,0,3} */
#define VIA_IGA2_HOR_ADDR_REG_NUM           		2           /* location: {CR51,0,7},{CR55,4,6} */
#define VIA_IGA2_HOR_BLANK_START_REG_NUM    	2           /* location: {CR52,0,7},{CR54,0,2} */
#define VIA_IGA2_HOR_BLANK_END_REG_NUM      	3           /* location: CLE266: {CR53,0,7},{CR54,3,5} => CLE266's CR5D[6] is reserved, so it may have problem to set 1600x1200 on IGA2. */
                                                   						      /* Others: {CR53,0,7},{CR54,3,5},{CR5D,6,6} */
#define VIA_IGA2_HOR_SYNC_START_REG_NUM     	4           /* location: {CR56,0,7},{CR54,6,7},{CR5C,7,7} */
                                                   						      /* VT3314 and Later: {CR56,0,7},{CR54,6,7},{CR5C,7,7}, {CR5D,7,7} */
#define VIA_IGA2_HOR_SYNC_END_REG_NUM       	2           /* location: {CR57,0,7},{CR5C,6,6} */
#define VIA_IGA2_VER_TOTAL_REG_NUM          		2           /* location: {CR58,0,7},{CR5D,0,2} */
#define VIA_IGA2_VER_ADDR_REG_NUM           		2           /* location: {CR59,0,7},{CR5D,3,5} */
#define VIA_IGA2_VER_BLANK_START_REG_NUM    	2           /* location: {CR5A,0,7},{CR5C,0,2} */
#define VIA_IGA2_VER_BLANK_END_REG_NUM      	2           /* location: {CR5E,0,7},{CR5C,3,5} */
#define VIA_IGA2_VER_SYNC_START_REG_NUM     	2           /* location: {CR5E,0,7},{CR5F,5,7} */
#define VIA_IGA2_VER_SYNC_END_REG_NUM       	1           /* location: {CR5F,0,4} */

#define VIA_IGA1_FIFO_DEPTH_SELECT_REG_NUM          		1
#define VIA_IGA1_FIFO_THRESHOLD_REG_NUM             			2
#define VIA_IGA1_FIFO_HIGH_THRESHOLD_REG_NUM        		2
#define VIA_IGA1_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM   	1

#define VIA_IGA2_FIFO_DEPTH_SELECT_REG_NUM          		3
#define VIA_IGA2_FIFO_THRESHOLD_REG_NUM             			2
#define VIA_IGA2_FIFO_HIGH_THRESHOLD_REG_NUM        		2
#define VIA_IGA2_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM   	1


/***************************************************/
/* Definition IGA1 Design Method of CRTC Registers */
/***************************************************/
#define VIA_IGA1_HOR_TOTAL_FORMULA(x)           				((x)/8)-5
#define VIA_IGA1_HOR_ADDR_FORMULA(x)            				((x)/8)-1
#define VIA_IGA1_HOR_BLANK_START_FORMULA(x)     			((x)/8)-1
#define VIA_IGA1_HOR_BLANK_END_FORMULA(x)     				((x)/8)-1
#define VIA_IGA1_HOR_SYNC_START_FORMULA(x)      			((x)/8)-1
#define VIA_IGA1_HOR_SYNC_END_FORMULA(x)      				((x)/8)-1

#define VIA_IGA1_VER_TOTAL_FORMULA(x)           				(x)-2
#define VIA_IGA1_VER_ADDR_FORMULA(x)            				(x)-1
#define VIA_IGA1_VER_BLANK_START_FORMULA(x)     			(x)-1
#define VIA_IGA1_VER_BLANK_END_FORMULA(x)    				(x)-1
#define VIA_IGA1_VER_SYNC_START_FORMULA(x)      			(x)-1
#define VIA_IGA1_VER_SYNC_END_FORMULA(x)      				(x)-1

/***************************************************/
/* Definition IGA2 Design Method of CRTC Registers */
/***************************************************/
#define VIA_IGA2_HOR_TOTAL_FORMULA(x)           				(x)-1
#define VIA_IGA2_HOR_ADDR_FORMULA(x)            				(x)-1
#define VIA_IGA2_HOR_BLANK_START_FORMULA(x)     			(x)-1
#define VIA_IGA2_HOR_BLANK_END_FORMULA(x)     				(x)-1
#define VIA_IGA2_HOR_SYNC_START_FORMULA(x)      			(x)-1
#define VIA_IGA2_HOR_SYNC_END_FORMULA(x)      				(x)-1

#define VIA_IGA2_VER_TOTAL_FORMULA(x)           				(x)-1
#define VIA_IGA2_VER_ADDR_FORMULA(x)            				(x)-1
#define VIA_IGA2_VER_BLANK_START_FORMULA(x)     			(x)-1
#define VIA_IGA2_VER_BLANK_END_FORMULA(x)     				(x)-1
#define VIA_IGA2_VER_SYNC_START_FORMULA(x)      			(x)-1
#define VIA_IGA2_VER_SYNC_END_FORMULA(x)      				(x)-1

/*******************************************************/
/*Definition IGA1 Design Method of FIFO Registers                       */
/*******************************************************/
#define VIA_IGA1_FIFO_DEPTH_SELECT_FORMULA(x)           		(x/2)-1
#define VIA_IGA1_FIFO_THRESHOLD_FORMULA(x)              		(x/4)
#define VIA_IGA1_DISPLAY_QUEUE_EXPIRE_NUM_FORMULA(x)    	(x/4)
#define VIA_IGA1_FIFO_HIGH_THRESHOLD_FORMULA(x)         	(x/4)

/*******************************************************/
/*Definition IGA2 Design Method of FIFO Registers                       */
/*******************************************************/
#define VIA_IGA2_FIFO_DEPTH_SELECT_FORMULA(x)           		(x/2/4)-1
#define VIA_IGA2_FIFO_THRESHOLD_FORMULA(x)              		(x/4)
#define VIA_IGA2_DISPLAY_QUEUE_EXPIRE_NUM_FORMULA(x)    	(x/4)
#define VIA_IGA2_FIFO_HIGH_THRESHOLD_FORMULA(x)        	(x/4)

/*******************************************************/
/*Definition IGA2 Scaling Factor Registers                       */
/*******************************************************/
#define LCD_HOR_SCALE_FACTOR_FORMULA(x,y)       (((x-1)*4096)/(y-1))
#define LCD_VER_SCALE_FACTOR_FORMULA(x,y)       (((x-1)*2048)/(y-1))


/*******************************************************/
/* Cacalate PLL */
/*******************************************************/
#define VIA_SORT_BY_PHASE_MARGIN    1
#define VIA_SORT_BY_F_REF_BW        2
#define VIA_SORT_BY_PIXEL_CLOCK     3

#define VIA_FIN     14.31818
#define VIA_PI      3.141592654
#define VIA_TWO_PI  6.283185307

typedef struct _ClockSetting
{    
    double Clock;
    int M;
    int N;
    int R;
    double RefDivBW;
    double PhaseMargin;
}ClockSetting, *ClockSettingPtr;


/*******************************************************/
/*Display data structure*/
/*******************************************************/
typedef struct _DisplayTiming 
{
    unsigned short   HorTotal;
    unsigned short   HorAddr;
    unsigned short   HorBlankStart;
    unsigned short   HorBlankEnd;
    unsigned short   HorSyncStart;
    unsigned short   HorSyncEnd;

    unsigned short   VerTotal;
    unsigned short   VerAddr;
    unsigned short   VerBlankStart;
    unsigned short   VerBlankEnd;
    unsigned short   VerSyncStart;
    unsigned short   VerSyncEnd;
}DisplayTiming, *DisplayTimingPtr;

typedef struct _ViaTimingTable 
{
    unsigned long  ModeIndex;
    unsigned long   RefreshRate;
    unsigned long  Clk;
    long   HSyncPolarity;
    long   VSyncPolarity;
    DisplayTiming Timing;
}ViaTimingTable,*ViaTimingTablePtr;

/*Build-in timing table index*/
#define LCD_MODE_TABLE 		0x1

/****************************************************************************/
/*                              LCD Mode Table                              */
/* Note.1: On some LCD panel, we found that it can't receive the normal     */
/* timing. We must use a special timing to apply on it. But this maybe      */
/* not good for CRT. So we decide to make the LCD timing table independent. */
/*                                                                          */
/****************************************************************************/

static ViaTimingTable VIALCDModes[]= {
    /* Panel 480x640 (Special) */
    {VIA_480X640, REFRESH_60,  CLK_22_000M,        NEGATIVE,         POSITIVE,
        { 520,  480,    480,   40,  504,   8,  648,   640,  640,   8,   644,  2}},

    /* Panel 640x480 (DMT) */
    {VIA_640X480, REFRESH_60,  CLK_25_175M,        NEGATIVE,         NEGATIVE,
        { 800,  640,    640,  160,  656,  96,  525,   480,  480,  45,   490,   2}},  
    /* The LCD can't accept the VESA's 640x480 timing, because The VESA's 640x480 timing has border 
    and LCD doesn't have. Modify hbs and hbe to eliminate the border.*/


    /* Panel 800x480 (CVT) */
    {VIA_800X480, REFRESH_60,  CLK_29_581M,        NEGATIVE,         POSITIVE,
        { 992,   800,   800,  192,  824,  72,  500,   480,  480,  20,   483,   7}}, 

    /* Panel 800x600 (DMT)*/
    {VIA_800X600, REFRESH_60,  CLK_40_000M,        POSITIVE,         POSITIVE,
        {1056,   800,   800,  256,  840, 128,  628,   600,  600,  28,   601,   4}},           

    /* Panel 1024x600 */ 
    {VIA_1024X600, REFRESH_60,  CLK_48_875M,        NEGATIVE,         POSITIVE,
        {1312,  1024,  1024,  288, 1064, 104,  622,   600,  600,  22,   601,   3}},      

    /* Panel 1024x768 (CVT Reduce) */
    {VIA_1024X768, REFRESH_60,  CLK_56_250M,        POSITIVE,         NEGATIVE,
        {1184,  1024,  1024,  160, 1072,  32,  790,   768,  768,  22,   771,   4}},            

    /* Panel 1280x768 (GTF) */
    {VIA_1280X768, REFRESH_60,  CLK_80_136M,        NEGATIVE,         POSITIVE,
        {1680,  1280,  1280,  400, 1344, 136,  795,   768,  768,  27,   769,   3}},
        
    /* Panel 1280x800 (CVT Reduce) */
    {VIA_1280X800, REFRESH_60,  CLK_72_000M,        POSITIVE,         NEGATIVE,
        {1440,  1280,  1280,  160, 1328,  32,  823,   800,  800,  23,   803,   6}}, 

    /* Panel 1280x1024 (DMT) */
    {VIA_1280X1024, REFRESH_60, CLK_108_000M,        POSITIVE,         POSITIVE,
        {1688,  1280,  1280,  408, 1328, 112, 1066,  1024, 1024,  42,  1025,   3}},

    /* Panel 1366x768 (GTF) */ 
    {VIA_1366X768, REFRESH_60,  CLK_85_860M,        NEGATIVE,         POSITIVE,
        {1800,  1368,  1368,  432, 1440, 144,  795,   768,  768,  27,   769,   3}},

    /* Panel 1360x768 (CVT) */ 
    {VIA_1360X768, REFRESH_60,  CLK_84_750M,        POSITIVE,         POSITIVE,
        {1776,  1360,  1360,  416, 1432, 136,  798,   768,  768,  30,   771,   5}}, 

    /* Panel 1400x1050 (CVT Reduce) */
    {VIA_1400X1050, REFRESH_60, CLK_101_000M,        POSITIVE,         NEGATIVE,
        {1560,  1400,  1400,  160, 1448,  32, 1080,  1050, 1050,  30,  1053,   4}},

    /* Panel 1440x900*/
    {VIA_1440X900, REFRESH_60, CLK_88_750M,        POSITIVE,         NEGATIVE,
        {1760,  1440,  1440,  320, 1495,  105, 912,  900,  900,   12,  901,   2}},   
        
    /* Panel 16x12 (CVT Reduce)*/
    {VIA_1600X1200, REFRESH_60, CLK_130_250M,        POSITIVE,         NEGATIVE,
        {1760,  1600,  1600,  160, 1648,  32, 1235,  1200, 1200,  35,  1203,   4}}   
};
#define   NUM_VIALCD_TIMING_TBL   CMDISP_ARRAYSIZE(VIALCDModes)


/* IGA1 */
/* IGA1 Horizontal Total Register */
typedef struct  _Iga1HorTotalRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_HOR_TOTAL_REG_NUM];
}Iga1HorTotalRegs,*Iga1HorTotalRegsPtr;

/* VIA_IGA1 Horizontal Addressable Register */
typedef struct  _Iga1HorAddrRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_HOR_ADDR_REG_NUM];
}Iga1HorAddrRegs,*Iga1HorAddrRegsPtr;

/* IGA1 Horizontal Blank Start Register */
typedef struct  _Iga1HorBlankStartRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_HOR_BLANK_START_REG_NUM];
}Iga1HorBlankStartRegs,*Iga1HorBlankStartRegsPtr;

/* IGA1 Horizontal Blank End Register */
typedef struct _Iga1HorBlankEndRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_HOR_BLANK_END_REG_NUM];
}Iga1HorBlankEndRegs,*Iga1HorBlankEndRegsPtr;

/* IGA1 Horizontal Sync Start Register */
typedef struct _Iga1HorSyncStartRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_HOR_SYNC_START_REG_NUM];
}Iga1HorSyncStartRegs,*Iga1HorSyncStartRegsPtr;

/* IGA1 Horizontal Sync End Register */
typedef struct _Iga1HorSyncEndRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_HOR_SYNC_END_REG_NUM];
}Iga1HorSyncEndRegs,*Iga1HorSyncEndRegsPtr;

/* IGA1 Vertical Total Register */
typedef struct _Iga1VerTotalRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_VER_TOTAL_REG_NUM];
}Iga1VerTotalRegs,*Iga1VerTotalRegsPtr;

/* IGA1 Vertical Addressable Register */
typedef struct _Iga1VerAddrRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_VER_ADDR_REG_NUM];
}Iga1VerAddrRegs,*Iga1VerAddrRegsPtr;

/* IGA1 Vertical Blank Start Register */
typedef struct _Iga1VerBlankStartRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_VER_BLANK_START_REG_NUM];
}Iga1VerBlankStartRegs,*Iga1VerBlankStartRegsPtr;

/* IGA1 Vertical Blank End Register */
typedef struct _Iga1VerBlankEndRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_VER_BLANK_END_REG_NUM];
}Iga1VerBlankEndRegs,*Iga1VerBlankEndRegsPtr;

/* IGA1 Vertical Sync Start Register */
typedef struct _Iga1VerSyncStartRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_VER_SYNC_START_REG_NUM];
}Iga1VerSyncStartRegs,*Iga1VerSyncStartRegsPtr;

/* IGA1 Vertical Sync End Register */
typedef struct _Iga1VerSyncEndRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_VER_SYNC_END_REG_NUM];
}Iga1VerSyncEndRegs,*Iga1VerSyncEndRegsPtr;

/* IGA2 */
/* VIA_IGA2 Horizontal Total Register */
typedef struct _Iga2HorTotalRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_HOR_TOTAL_REG_NUM];
}Iga2HorTotalRegs,*Iga2HorTotalRegsPtr;

/* IGA2 Horizontal Addressable Register */
typedef struct _Iga2HorAddrRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_HOR_ADDR_REG_NUM];
}Iga2HorAddrRegs,*Iga2HorAddrRegsPtr;

/* IGA2 Horizontal Blank Start Register */
typedef struct _Iga2HorBlankStartRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_HOR_BLANK_START_REG_NUM];
}Iga2HorBlankStartRegs,*Iga2HorBlankStartRegsPtr;

/* IGA2 Horizontal Blank End Register */
typedef struct _Iga2HorBlankEndRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_HOR_BLANK_END_REG_NUM];
}Iga2HorBlankEndRegs,*Iga2HorBlankEndRegsPtr;

/* IGA2 Horizontal Sync Start Register */
typedef struct _Iga2HorSyncStartRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_HOR_SYNC_START_REG_NUM];
}Iga2HorSyncStartRegs,*Iga2HorSyncStartRegsPtr;

/* IGA2 Horizontal Sync End Register */
typedef struct _Iga2HorSyncEndRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_HOR_SYNC_END_REG_NUM];
}Iga2HorSyncEndRegs,*Iga2HorSyncEndRegsPtr;

/* IGA2 Vertical Total Register */
typedef struct _Iga2VerTotalRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_VER_TOTAL_REG_NUM];
}Iga2VerTotalRegs,*Iga2VerTotalRegsPtr;

/* IGA2 Vertical Addressable Register */
typedef struct _Iga2VerAddrRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_VER_ADDR_REG_NUM];
}Iga2VerAddrRegs,*Iga2VerAddrRegsPtr;

/* IGA2 Vertical Blank Start Register */
typedef struct _Iga2VerBlankStartRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_VER_BLANK_START_REG_NUM];
}Iga2VerBlankStartRegs,*Iga2VerBlankStartRegsPtr;

/* IGA2 Vertical Blank End Register */
typedef struct _Iga2VerBlankEndRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_VER_BLANK_END_REG_NUM];
}Iga2VerBlankEndRegs,*Iga2VerBlankEndRegsPtr;

/* IGA2 Vertical Sync Start Register */
typedef struct _Iga2VerSyncStartRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_VER_SYNC_START_REG_NUM];
}Iga2VerSyncStartRegs,*Iga2VerSyncStartRegsPtr;

/* IGA2 Vertical Sync End Register */
typedef struct _Iga2VerSyncEndRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_VER_SYNC_END_REG_NUM];
}Iga2VerSyncEndRegs,*Iga2VerSyncEndRegsPtr;

/* IGA1 CRTC Timing Register */
typedef struct _Iga1CrtcTimingRegs
{
	Iga1HorTotalRegs        HTotal;
	Iga1HorAddrRegs         HAddr;
	Iga1HorBlankStartRegs   HBlankSt;
	Iga1HorBlankEndRegs     HBlankEd;
	Iga1HorSyncStartRegs    HSyncSt;
	Iga1HorSyncEndRegs      HSyncEd;
	Iga1VerTotalRegs        VTotal;
	Iga1VerAddrRegs         VAddr;
	Iga1VerBlankStartRegs   VBlankSt;
	Iga1VerBlankEndRegs     VBlankEd;
	Iga1VerSyncStartRegs    VSyncSt;
	Iga1VerSyncEndRegs      VSyncEd;
}Iga1CrtcTimingRegs,*Iga1CrtcTimingRegsPtr;

/* IGA2 CRTC Timing Register */
typedef struct _Iga2CrtcTimingRegs
{
	Iga2HorTotalRegs        HTotal;
	Iga2HorAddrRegs         HAddr;
	Iga2HorBlankStartRegs   HBlankSt;
	Iga2HorBlankEndRegs     HBlankEd;
	Iga2HorSyncStartRegs    HSyncSt;
	Iga2HorSyncEndRegs      HSyncEd;
	Iga2VerTotalRegs        VTotal;
	Iga2VerAddrRegs         VAddr;
	Iga2VerBlankStartRegs   VBlankSt;
	Iga2VerBlankEndRegs     VBlankEd;
	Iga2VerSyncStartRegs    VSyncSt;
	Iga2VerSyncEndRegs      VSyncEd;
}Iga2CrtcTimingRegs,*Iga2CrtcTimingRegsPtr;

typedef struct _VpitRegTable 
{
    unsigned char  Misc;
    unsigned char  SR[STD_SR_NUM];
    unsigned char  GR[STD_GR_NUM];
    unsigned char  AR[STD_AR_NUM];
}VpitRegTable,*VpitRegTablePtr;

static VpitRegTable VpitRegs = 
{
    /* Msic*/
    0xC7,
    /*Sequencer*/
    {0x01,0x0F,0x00,0x0E },
    /*Graphic Controller*/
    {0x00,0x00,0x00,0x00,0x00,0x00,0x05,0x0F,0xFF},
    /*Attribute Controller*/
    {0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
     0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
     0x01,0x00,0x0F,0x00}
};

static Iga1CrtcTimingRegs Iga1CrtcRegs = 
{
    /* IGA1 Horizontal Total*/
    {VIA_IGA1_HOR_TOTAL_REG_NUM, {{REG_CR00,0,7}, {REG_CR36,3,3}}},
    /* IGA1 Horizontal Addressable Video*/
    {VIA_IGA1_HOR_ADDR_REG_NUM, {{REG_CR01,0,7}}},
    /*IGA1 Horizontal Blank Start*/
    {VIA_IGA1_HOR_BLANK_START_REG_NUM, {{REG_CR02,0,7}}},
    /*IGA1 Horizontal Blank End*/
    {VIA_IGA1_HOR_BLANK_END_REG_NUM, {{REG_CR03,0,4}, {REG_CR05,7,7}, {REG_CR33,5,5}}},
    /*IGA1 Horizontal Sync Start*/
    {VIA_IGA1_HOR_SYNC_START_REG_NUM, {{REG_CR04,0,7}, {REG_CR33,4,4}}},
    /*IGA1 Horizontal Sync End*/
    {VIA_IGA1_HOR_SYNC_END_REG_NUM, {{REG_CR05,0,4}}},
    /*IGA1 Vertical Total*/
    {VIA_IGA1_VER_TOTAL_REG_NUM, {{REG_CR06,0,7}, {REG_CR07,0,0}, {REG_CR07,5,5}, {REG_CR35,0,0}}},
    /*IGA1 Vertical Addressable Video*/
    {VIA_IGA1_VER_ADDR_REG_NUM, {{REG_CR12,0,7}, {REG_CR07,1,1}, {REG_CR07,6,6}, {REG_CR35,2,2}}},
    /*IGA1 Vertical Blank Start*/
    {VIA_IGA1_VER_BLANK_START_REG_NUM, {{REG_CR15,0,7}, {REG_CR07,3,3}, {REG_CR09,5,5}, {REG_CR35,3,3}}},
    /*IGA1 Vertical Blank End*/
    {VIA_IGA1_VER_BLANK_END_REG_NUM, {{REG_CR16,0,7}}},
    /*IGA1 Vertical Sync Start*/
    {VIA_IGA1_VER_SYNC_START_REG_NUM, {{REG_CR10,0,7}, {REG_CR07,2,2}, {REG_CR07,7,7}, {REG_CR35,1,1}}},
    /*IGA1 Vertical Sync End*/
    {VIA_IGA1_VER_SYNC_END_REG_NUM, {{REG_CR11,0,3}}}
};

static Iga2CrtcTimingRegs Iga2CrtcRegs = 
{
    /*IGA2 Horizontal Total*/
    {VIA_IGA2_HOR_TOTAL_REG_NUM, {{REG_CR50,0,7}, {REG_CR55,0,3}}},
    /*IGA2 Horizontal Addressable Video*/
    {VIA_IGA2_HOR_ADDR_REG_NUM, {{REG_CR51,0,7}, {REG_CR55,4,6}}},
    /*IGA2 Horizontal Blank Start*/
    {VIA_IGA2_HOR_BLANK_START_REG_NUM, {{REG_CR52,0,7}, {REG_CR54,0,2}}},
    /*IGA2 Horizontal Blank End*/
    {VIA_IGA2_HOR_BLANK_END_REG_NUM, {{REG_CR53,0,7}, {REG_CR54,3,5}, {REG_CR5D,6,6}}},
    /*IGA2 Horizontal Sync Start*/
    {VIA_IGA2_HOR_SYNC_START_REG_NUM, {{REG_CR56,0,7}, {REG_CR54,6,7}, {REG_CR5C,7,7}, {REG_CR5D,7,7}}},
    /*IGA2 Horizontal Sync End*/
    {VIA_IGA2_HOR_SYNC_END_REG_NUM, {{REG_CR57,0,7}, {REG_CR5C,6,6}}},
    /*IGA2 Vertical Total*/
    {VIA_IGA2_VER_TOTAL_REG_NUM, {{REG_CR58,0,7}, {REG_CR5D,0,2}}},
    /*IGA2 Vertical Addressable Video*/
    {VIA_IGA2_VER_ADDR_REG_NUM, {{REG_CR59,0,7}, {REG_CR5D,3,5}}},
    /*IGA2 Vertical Blank Start*/
    {VIA_IGA2_VER_BLANK_START_REG_NUM, {{REG_CR5A,0,7}, {REG_CR5C,0,2}}},
    /*IGA2 Vertical Blank End*/
    {VIA_IGA2_VER_BLANK_END_REG_NUM, {{REG_CR5B,0,7}, {REG_CR5C,3,5}}},
    /*IGA2 Vertical Sync Start*/
    {VIA_IGA2_VER_SYNC_START_REG_NUM, {{REG_CR5E,0,7}, {REG_CR5F,5,7}}},
    /*IGA2 Vertical Sync End*/
    {VIA_IGA2_VER_SYNC_END_REG_NUM, {{REG_CR5F,0,4}}}
};

#define VIA_IGA1_OFFSET_REG_NUM                     	2   /* location: {REG_CR13,0,7},{REG_CR35,5,7}*/
#define VIA_IGA2_OFFSET_REG_NUM                     	2   /*location: {REG_CR66,0,7},{REG_CR67,0,1}*/
#define VIA_IGA2_OFFSET_13BIT_REG_NUM          	3   /*location: {REG_CR66,0,7},{REG_CR67,0,1},{REG_CR71,7}*/

/* IGA Offset Register */
typedef struct _Iga1OffsetRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_OFFSET_REG_NUM];
}Iga1OffsetRegs,*Iga1OffsetRegsPtr;

/* IGA2 Offset Register */
typedef struct _Iga2OffsetRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_OFFSET_REG_NUM];
}Iga2OffsetRegs,*Iga2OffsetRegsPtr;

/* IGA2 13-bit Offset register */
typedef struct _Iga2Offset13BitRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_OFFSET_13BIT_REG_NUM];
}Iga2Offset13BitRegs,*Iga2Offset13BitRegsPtr;

/* IGA Offset Register */
typedef struct _OffsetRegs
{
    Iga1OffsetRegs   Iga1;
    Iga2OffsetRegs   Iga2;
}OffsetRegs,*OffsetRegsPtr;

static OffsetRegs ViaOffsetRegs = 
{
    /* IGA Offset Register*/
    {VIA_IGA1_OFFSET_REG_NUM, {{REG_CR13,0,7}, {REG_CR35,5,7}}},
    /*IGA2 Offset Register*/
    {VIA_IGA2_OFFSET_REG_NUM, {{REG_CR66,0,7}, {REG_CR67,0,1}}}
};

static Iga2Offset13BitRegs ViaIga2Offset13BitRegs =
{
    VIA_IGA2_OFFSET_13BIT_REG_NUM, {{REG_CR66,0,7}, {REG_CR67,0,1}, {REG_CR71,7,7}}
};

#define VIA_IGA1_FETCH_COUNT_REG_NUM                2   /*location: {REG_SR1C,0,7},{REG_SR1D,0,1}*/
#define VIA_IGA2_FETCH_COUNT_REG_NUM                2   /*location: {REG_CR65,0,7},{REG_CR67,2,3}*/

/* IGA1 Fetch Count Register */
typedef struct _Iga1FetchCountRegs
{
    unsigned char    RegNum;
    ViaIoBit Reg[VIA_IGA1_FETCH_COUNT_REG_NUM];
}Iga1FetchCountRegs,*Iga1FetchCountRegsPtr;

/* IGA2 Fetch Count Register */
typedef struct _Iga2FetchCountRegs
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_FETCH_COUNT_REG_NUM];
}Iga2FetchCountRegs,*Iga2FetchCountRegsPtr;

/* IGA Fetch Count Register */
typedef struct _IgaFetchCountRegs
{
      Iga1FetchCountRegs    Iga1;
      Iga2FetchCountRegs    Iga2;
}IgaFetchCountRegs,*IgaFetchCountRegsPtr;

static IgaFetchCountRegs ViaFetchCountRegs = 
{
    /*IGA Fetch Count Register*/
    {VIA_IGA1_FETCH_COUNT_REG_NUM, {{REG_SR1C,0,7}, {REG_SR1D,0,1}}},
    /*IGA2 Fetch Count Register*/
    {VIA_IGA2_FETCH_COUNT_REG_NUM, {{REG_CR65,0,7}, {REG_CR67,2,3}}}
};

typedef struct _FifoValue{
    unsigned long  MaxDepth;
    unsigned long  Threshold;
    unsigned long  HighThreshold;
    unsigned long  DispQueueExpiredNum;
}FifoValue,*FifoValuePtr;

typedef struct _DispFifo{
    unsigned long  ChipName;
    FifoValue Iga1Fifo;
    FifoValue Iga2Fifo;    
}DispFifo,*DispFifoPtr;

static DispFifo DispFifoTable[] = {
    /*  Chip               		IGA1 FIFO              	IGA2 FIFO */
    { VIA_P4M800PRO,      { 96,  80,  64,  0},   		{ 96,  80,  32, 128}},
    { VIA_CX700,      	  {192, 128, 128, 124},   	    { 96,  64,  32, 128}},
    { VIA_K8M890,     	  {360, 328, 296, 124},   	    {360, 328, 296, 124}},
    { VIA_P4M890,         { 96,  76,  64,  32},   		{ 96,  76,  64,  32}},
    { VIA_P4M900,         { 96,  76,  76,  32},   		{ 96,  76,  76,  32}},
    { VIA_VX800,          {192, 152, 152,  64},   	    { 96,  76,  76,  128}},
    { VIA_VX855,          {400, 320, 320, 160},           {200, 160, 160,  320}}    
};
#define NUM_TOTAL_DISP_FIFO_TABLE CMDISP_ARRAYSIZE(DispFifoTable)


/* Display FIFO Relation */
typedef struct _Iga1FifoDepthSelect
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_FIFO_DEPTH_SELECT_REG_NUM];
}Iga1FifoDepthSelect,*Iga1FifoDepthSelectPtr;

typedef struct _Iga1FifoThresholdSelect
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_FIFO_THRESHOLD_REG_NUM];
}Iga1FifoThresholdSelect,*Iga1FifoThresholdSelectPtr;

typedef struct _Iga1FifoHighThresholdSelect
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_FIFO_HIGH_THRESHOLD_REG_NUM];
}Iga1FifoHighThresholdSelect,*Iga1FifoHighThresholdSelectPtr;

typedef struct _Iga1DisplayQueueExpireNum
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA1_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM];
}Iga1DisplayQueueExpireNum,*Iga1DisplayQueueExpireNumPtr;

typedef struct _Iga2FifoDepthSelect
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_FIFO_DEPTH_SELECT_REG_NUM];
}Iga2FifoDepthSelect,*Iga2FifoDepthSelectPtr;

typedef struct _Iga2FifoThresholdSelect
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_FIFO_THRESHOLD_REG_NUM];
}Iga2FifoThresholdSelect,*Iga2FifoThresholdSelectPtr;

typedef struct _Iga2FifoHighThresholdSelect
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_FIFO_HIGH_THRESHOLD_REG_NUM];
}Iga2FifoHighThresholdSelect,*Iga2FifoHighThresholdSelectPtr;

typedef struct _Iga2DisplayQueueExpireNum
{
    unsigned char   RegNum;
    ViaIoBit Reg[VIA_IGA2_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM];
}Iga2DisplayQueueExpireNum,*Iga2DisplayQueueExpireNumPtr;

typedef struct _FifoDepthSelect
{
     Iga1FifoDepthSelect  Iga1FifoDepthSelectReg;
     Iga2FifoDepthSelect  Iga2FifoDepthSelectReg;
}FifoDepthSelect,*FifoDepthSelectPtr;

typedef struct _FifoThresholdSelect
{
     Iga1FifoThresholdSelect  Iga1FifoThresholdSelectReg;
     Iga2FifoThresholdSelect  Iga2FifoThresholdSelectReg;
}FifoThresholdSelect,*FifoThresholdSelectPtr;

typedef struct _FifoHighThresholdSelect
{
     Iga1FifoHighThresholdSelect  Iga1FifoHighThresoldSelectReg;
     Iga2FifoHighThresholdSelect  Iga2FifoHighThresoldSelectReg;
}FifoHighThresholdSelect,*FifoHighThresholdSelectPtr;

typedef struct _DisplayQueueExpireNum
{
     Iga1DisplayQueueExpireNum Iga1DisplayQueueExpireNumReg;
     Iga2DisplayQueueExpireNum Iga2DisplayQueueExpireNumReg;
}DisplayQueueExpireNum,*DisplayQueueExpireNumPtr;

static  FifoDepthSelect ViaDisplayFifoDepthReg= {
    /* IGA1 FIFO Depth_Select */
    {VIA_IGA1_FIFO_DEPTH_SELECT_REG_NUM, {{REG_SR17,0,7}}},
    /* IGA2 FIFO Depth_Select */
    {VIA_IGA2_FIFO_DEPTH_SELECT_REG_NUM, {{REG_CR68,4,7}, {REG_CR94,7,7}, {REG_CR95,7,7}}}
};

static  FifoThresholdSelect ViaFifoThresholdSelectReg= {
    /* IGA1 FIFO Threshold Select*/
    {VIA_IGA1_FIFO_THRESHOLD_REG_NUM, {{REG_SR16,0,5},{REG_SR16,7,7}}},
    /* IGA2 FIFO Threshold Select*/
    {VIA_IGA2_FIFO_THRESHOLD_REG_NUM, {{REG_CR68,0,3}, {REG_CR95,4,6}}}
};

static  FifoHighThresholdSelect ViaFifoHighThresholdSelectReg= {
    /* IGA1 GFX PREG Thresold Select*/
    {VIA_IGA1_FIFO_HIGH_THRESHOLD_REG_NUM, {{REG_SR18,0,5},{REG_SR18,7,7}}},
    /* IGA2 GFX PREG Thresold Select*/
    {VIA_IGA2_FIFO_HIGH_THRESHOLD_REG_NUM, {{REG_CR92,0,3}, {REG_CR95,0,2}}}
};

static  DisplayQueueExpireNum ViaDisplayQueueExpireNumReg= {
    /*IGA1 Display Queue Expire Num*/
    {VIA_IGA1_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM, {{REG_SR22,0,4}}},
    /*IGA2 Display Queue Expire Num*/
    {VIA_IGA2_DISPLAY_QUEUE_EXPIRE_NUM_REG_NUM, {{REG_CR94,0,6}}}
};
#endif /*__VIA_MODE_H__*/

