/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
 

#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <linux/version.h>
#include "via_driver.h"

#ifdef VIA_RANDR12_SUPPORT
#include "globals.h"
#include "debug.h"      /* for DBG_DD */
#include "xf86RAC.h"
#include "xf86Priv.h"
#include "xf86Crtc.h"
#include "xf86RandR12.h"
#include "via_bios.h"
#include "via_rotate.h"
#include "via_eng_regs.h"
#include "viamode.h"
#include "X11/Xatom.h"
#include "via_modedata.h"
#include "via_common.h"
#include "via_serial.h"
#include "via_regs.h"
#include "via_output.h"
#include "via_displcd.h"

/*External data structures and functions*/
extern Bool IsValidMode(CARD16 HDisplay, CARD16 VDisplay);


/* ================================================ */
/*                  Output property                              		   */
/* ================================================ */


void 
viaSetOutputPath (CARD32 diPort, int igaPath, int chipset)
{
    /*Set Iga source and turn on DI port clk*/
    switch (diPort) {
        case DISP_DI_DVP0:
            if (igaPath == IGA1)
                /* DVP0 Data Source Selection.*/
                viaWriteVgaIoBits(REG_CR96, 0x00, BIT4);
            else
                /* DVP0 Data Source Selection.*/
                viaWriteVgaIoBits(REG_CR96, 0x10, BIT4);
            /* enable dvp0 under CX700 */
            if (VIA_CX700 == chipset)
                viaWriteVgaIoBits(REG_CR91, 0x00, BIT5);
            
            /*Turn on DVP0 clk*/
            viaWriteVgaIoBits(REG_SR1E, 0xC0, BIT6+BIT7);
            break;
        case DISP_DI_DVP1:
            if (igaPath == IGA1)
                viaWriteVgaIoBits(REG_CR9B, 0x00, BIT4);
            else
                viaWriteVgaIoBits(REG_CR9B, 0x10, BIT4);
            
            /* enable dvp1 under this chipset */
            if ((VIA_CX700 == chipset) ||
                (VIA_VX800 == chipset) ||
                (VIA_VX855 == chipset))
                viaWriteVgaIoBits(REG_CRD3, 0x00, BIT5);
            /*Turn on DVP1 clk*/
            viaWriteVgaIoBits(REG_SR1E, 0x30, BIT4+BIT5);
            break;
            
        case DISP_DI_DFPH:
            if (igaPath == IGA1) {
                /*3324,3353 CR96[4] control DVP0*/
                if ((VIA_CX700 != chipset) &&
                    (VIA_VX800 != chipset) &&
                    (VIA_VX855 != chipset))
                    viaWriteVgaIoBits(REG_CR96, 0x00, BIT4);
                viaWriteVgaIoBits(REG_CR97, 0x00, BIT4);
            } else {
                /*3324,3353 CR96[4] control DVP0*/
                if ((VIA_CX700 != chipset) &&
                    (VIA_VX800 != chipset) &&
                    (VIA_VX855 != chipset))
                    viaWriteVgaIoBits(REG_CR96, 0x10, BIT4);
                viaWriteVgaIoBits(REG_CR97, 0x10, BIT4);
            }

            /*Turn on DFPH clk*/
            viaWriteVgaIoBits(REG_SR2A, 0x0C, BIT2+BIT3);
            break;
            
        case DISP_DI_DFPL:
            if (igaPath == IGA1) {
                /*3324,3353 CR9B[4] control DVP1*/
                if ((VIA_CX700 != chipset) &&
                    (VIA_VX800 != chipset) &&
                    (VIA_VX855 != chipset))
                    viaWriteVgaIoBits(REG_CR9B, 0x00, BIT4);
               viaWriteVgaIoBits(REG_CR99, 0x00, BIT4);
            } else {
                /*3324,3353 CR9B[4] control DVP1*/
                if ((VIA_CX700 != chipset) &&
                    (VIA_VX800 != chipset) &&
                    (VIA_VX855 != chipset))
                    viaWriteVgaIoBits(REG_CR9B, 0x10, BIT4);
                viaWriteVgaIoBits(REG_CR99, 0x10, BIT4);
            }

            /*Turn on DFPL clk*/
            viaWriteVgaIoBits(REG_SR2A, 0x03, BIT1+BIT0);
            break;
            
        case DISP_DI_DFP:
            if ((VIA_K8M890 == chipset) ||
                (VIA_P4M890 == chipset)) 
                viaWriteVgaIoBits(REG_CR97, 0x84, BIT7+BIT2+BIT1+BIT0);
            if (igaPath == IGA1) {
                viaWriteVgaIoBits(REG_CR97, 0x00, BIT4);
                viaWriteVgaIoBits(REG_CR99, 0x00, BIT4);
            } else {
                viaWriteVgaIoBits(REG_CR97, 0x10, BIT4);
                viaWriteVgaIoBits(REG_CR99, 0x10, BIT4);
            }

            /*Turn on DFP clk*/
            viaWriteVgaIoBits(REG_SR2A, 0x0F, BIT3+BIT2+BIT1+BIT0);
            break;

		/*For TTL Type LCD*/
    	case (DISP_DI_DFPL+DISP_DI_DVP1):
	    	if (igaPath == IGA1) {
				viaWriteVgaIoBits(REG_CR99, 0x00, BIT4);
				viaWriteVgaIoBits(REG_CR9B, 0x00, BIT4);
            } else {
                viaWriteVgaIoBits(REG_CR99, 0x10, BIT4);
                viaWriteVgaIoBits(REG_CR9B, 0x10, BIT4);
            }

			/*Turn on DFPL, DVP1 clk*/
			viaWriteVgaIoBits(REG_SR2A, 0x03, BIT1+BIT0);
			viaWriteVgaIoBits(REG_SR1E, 0x30, BIT4+BIT5);
            break;

        /*For 409 TTL Type LCD*/
        case (DISP_DI_DFPH + DISP_DI_DFPL + DISP_DI_DVP1):
            if (igaPath == IGA1) {
                viaWriteVgaIoBits(REG_CR97, 0x00, BIT4);                
                viaWriteVgaIoBits(REG_CR99, 0x00, BIT4);
                viaWriteVgaIoBits(REG_CR9B, 0x00, BIT4);
            } else {
                viaWriteVgaIoBits(REG_CR97, 0x10, BIT4);            
                viaWriteVgaIoBits(REG_CR99, 0x10, BIT4);
                viaWriteVgaIoBits(REG_CR9B, 0x10, BIT4);
            }

            /*Turn on DFPHL, DVP1 clk*/
            viaWriteVgaIoBits(REG_SR2A, 0x0F, BIT3+BIT2+BIT1+BIT0);
            viaWriteVgaIoBits(REG_SR1E, 0x30, BIT4+BIT5);
            break;

		default:
			xf86DrvMsg(0, X_ERROR, "viaSetOutputPath: Invalid DIPort.\n");
			break;

    }

    /*If Iga2 is used, enable Iga2 output for DI port. 
      In order to enable Iga2, 3324,3353,3293 use CR91 and CRD3 to control
      DVP0 and DVP1 seperately, but other chip only use CR91 to control all Di port*/
    if(IGA2 == igaPath && VIA_CX700 != chipset)
        viaWriteVgaIoBits(REG_CR91, 0x00, BIT5);
    
}

/* detect CRT VSYNC signal*/
xf86OutputStatus 
viaDetectCRTVsync  (VIAPtr  pVia)
{
    CARD8 regSR01, regCR36, regSR40;
    xf86OutputStatus    status;
    regSR01 = viaReadVgaIo(REG_SR01);
    regCR36 = viaReadVgaIo(REG_CR36);
    regSR40 = viaReadVgaIo(REG_SR40);
    /* Screen On */
    viaWriteVgaIoBits(REG_SR01, 0x0, BIT5);
    /* Power On DPMS */
    viaWriteVgaIoBits(REG_CR36, 0x0, BIT4+BIT5+BIT6+BIT7); 

    /*delay 16ms for Vertical Blank*/
    viaDelay_Nmsec(pVia, 16);
  
    /* Enable CRT Sense */
    viaWriteVgaIoBits(REG_SR40, BIT7, BIT7);                

    if ((VIA_CX700 == pVia->Chipset) || 
        (VIA_VX800 == pVia->Chipset) ||
        (VIA_VX855 == pVia->Chipset))
        viaWriteVgaIoBits(REG_SR40, 0, BIT7);
            
	/* 
	324,353: SR40[7]=1 --> SR40[7] = 0 --> check 3C2[4]
	other: SR40[7]=1 --> check 3C2[4] --> SR40[7]=0
	*/
    if (STANDVGA_R8(0x3C2) & BIT4) {
        status = XF86OutputStatusConnected;
        DEBUG(ErrorF("CRT Connected\n"));
    } else {
        status = XF86OutputStatusDisconnected; 
        DEBUG(ErrorF("CRT Dis-connected\n"));
    }

    if ((VIA_CX700 != pVia->Chipset) && 
        (VIA_VX800 != pVia->Chipset) &&
        (VIA_VX855 != pVia->Chipset))
        viaWriteVgaIoBits(REG_SR40, 0, BIT7);
            
     /* Restore */
     viaWriteVgaIo(REG_SR40, regSR40);
     viaWriteVgaIo(REG_CR36, regCR36);
     viaWriteVgaIo(REG_SR01, regSR01); 
     return status;
}

void 
viaDIPortPadOn (int DIPort)
{
    switch (DIPort) {
        case DISP_DI_DVP0:
            viaWriteVgaIoBits(REG_SR1E, BIT6+BIT7, BIT6+BIT7);
            break;

        case DISP_DI_DVP1:
            viaWriteVgaIoBits(REG_SR1E, BIT4+BIT5, BIT4+BIT5);
            break;

        case DISP_DI_DFPH:
            viaWriteVgaIoBits(REG_SR2A, BIT2+BIT3, BIT2+BIT3);
            break;

        case DISP_DI_DFPL:
            viaWriteVgaIoBits(REG_SR2A, BIT0+BIT1, BIT0+BIT1);
            break;  
			
 		case DISP_DI_DFP:
			viaWriteVgaIoBits(REG_SR2A, BIT2+BIT3+BIT1+BIT0, BIT2+BIT3+BIT1+BIT0);
 			break;

		/*TTL LCD,now used for Qunta case*/
		case DISP_DI_DFPL+DISP_DI_DVP1:
			viaWriteVgaIoBits(REG_SR1E, BIT4+BIT5, BIT4+BIT5);
			viaWriteVgaIoBits(REG_SR2A, BIT0+BIT1, BIT0+BIT1);
			break;

		case DISP_DI_DAC:
			break;
			
		default:
			xf86DrvMsg(0, X_ERROR, "viaDIPortPadOn: Invalid DIPort.\n");
			break;
    }
}

void
viaDIPortPadOff (int DIPort)
{
    switch (DIPort) {
        case DISP_DI_DVP0:
            viaWriteVgaIoBits(REG_SR1E, 0x00, BIT6+BIT7);
            break;

        case DISP_DI_DVP1:
            viaWriteVgaIoBits(REG_SR1E, 0x00, BIT4+BIT5);
            break;

        case DISP_DI_DFPH:
            viaWriteVgaIoBits(REG_SR2A, 0x00, BIT2+BIT3);
            break;

        case DISP_DI_DFPL:
            viaWriteVgaIoBits(REG_SR2A, 0x00, BIT0+BIT1);
            break;

 		case DISP_DI_DFP:
            viaWriteVgaIoBits(REG_SR2A, 0x00, BIT2+BIT3+BIT1+BIT0);
            break;

		/*TTL LCD,now used for Qunta case*/
		case DISP_DI_DFPL+DISP_DI_DVP1:
			viaWriteVgaIoBits(REG_SR1E, 0x00, BIT4+BIT5);
			viaWriteVgaIoBits(REG_SR2A, 0x00, BIT0+BIT1);
			break;

		case DISP_DI_DFPH+DISP_DI_DFPL+DISP_DI_DVP1:
			viaWriteVgaIoBits(REG_SR1E, 0x00, BIT4+BIT5);
			viaWriteVgaIoBits(REG_SR2A, 0x00, BIT3+BIT2+BIT0+BIT1);
			break;

		case DISP_DI_DAC:
			break;

		default:
			xf86DrvMsg(0, X_ERROR, "viaDIPortPadOff: Invalid DIPort.\n");
			break;
    }
}

static void
setDiportSyncPolarity (CARD32 port, DisplayModePtr mode)
{
    CARD8 syncreg = 0; 
    
    if (mode->Flags & XF86CONF_NVSYNC)
        syncreg |= NEGATIVE << 1;
    if (mode->Flags & XF86CONF_NHSYNC)
        syncreg |= NEGATIVE;
    
    switch (port) {
        case DISP_DI_DAC:
            viaWriteMiscIo((viaReadMiscIo() & (~(BIT6+BIT7))) | (syncreg<<6));
            break;
    	case VIA_DI_DVP0:
            viaWriteVgaIoBits(REG_CR96, syncreg<<5, BIT6 | BIT5);
			break;
		case VIA_DI_DVP1:
		    viaWriteVgaIoBits(REG_CR9B, syncreg<<5, BIT6 | BIT5);
			break;
		case VIA_DI_DFPHIGH:
		    viaWriteVgaIoBits(REG_CR97, syncreg<<5, BIT6 | BIT5);
			break;
		case VIA_DI_DFPLOW:
		    viaWriteVgaIoBits(REG_CR99, syncreg<<5, BIT6 | BIT5);
			break;
		default:
			xf86DrvMsg(0, X_ERROR, "setDiportSyncPolarity: Invalid DIPort.\n");
			break;    
    }
}

/*
*   Purpose: Set DPA value to register.
*/
static void 
setDPA (CARD32 port, GFX_DPA_VALUE_PTR pGfxDPASetting)
{
    DEBUG(ErrorF("VIASetDPA_Gfx, Use Port %x.\n", port));
    
    switch (port) {
        case DISP_DI_DVP0:
            /* DVP0 Clock Polarity and Adjust: */
			viaWriteVgaIoBits(REG_CR96, pGfxDPASetting->DVP0, 0x0F);

            /* DVP0 Clock and Data Pads Driving: */
            viaWriteVgaIoBits(REG_SR1E, pGfxDPASetting->DVP0ClockDri_S, BIT2);
            viaWriteVgaIoBits(REG_SR2A, pGfxDPASetting->DVP0ClockDri_S1, BIT4);
            viaWriteVgaIoBits(REG_SR1B, pGfxDPASetting->DVP0DataDri_S, BIT1);
            viaWriteVgaIoBits(REG_SR2A, pGfxDPASetting->DVP0DataDri_S1, BIT5);
            break;
        case DISP_DI_DVP1:
            /* DVP1 Clock Polarity and Adjust: */
            viaWriteVgaIoBits(REG_CR9B, pGfxDPASetting->DVP1, 0x0F);

            /* DVP1 Clock and Data Pads Driving: */
            viaWriteVgaIoBits(REG_SR65, pGfxDPASetting->DVP1Driving, 0x0F);
            break;
        case DISP_DI_DFPH:
            viaWriteVgaIoBits(REG_CR97, pGfxDPASetting->DFPHigh, 0x0F);
            break;
        case DISP_DI_DFPL:
            viaWriteVgaIoBits(REG_CR99, pGfxDPASetting->DFPLow, 0x0F);
            break;
        case DISP_DI_DFP:
            viaWriteVgaIoBits(REG_CR97, pGfxDPASetting->DFPHigh, 0x0F);
            viaWriteVgaIoBits(REG_CR99, pGfxDPASetting->DFPLow, 0x0F);
            break;
		default:
			xf86DrvMsg(0, X_ERROR, "setDPA: Invalid DIPort.\n");
			break;
	}
}

static void
LoadUserGfxDPASetting (CARD32 port, ViaGfxDPAPtr pVal)
{   
	switch (port) {
		case DISP_DI_DVP0:
			if (pVal->isClkPolarityUsed)
				viaWriteVgaIoBits(REG_CR96, pVal->clkPolarity<<3, BIT3);
			if (pVal->isClkAdjustUsed)
				viaWriteVgaIoBits(REG_CR96, pVal->clkAdjust, 0x07);
			if (pVal->isClkDrivingSelUsed) {
				/*pVal->clkDrivingSel[1:0],
				Bit0's value is S0, bit1's value is S1*/
				viaWriteVgaIoBits(REG_SR1E, pVal->clkDrivingSel<<2, BIT2);
				viaWriteVgaIoBits(REG_SR2A, pVal->clkDrivingSel<<4, BIT4);
			}
			if (pVal->isDataDrivingSelUsed) {
				/*pVal->clkDrivingSel[1:0],
				Bit0's value is S0, bit1's value is S1*/
				viaWriteVgaIoBits(REG_SR1B, pVal->dataDrivingSel<<1, BIT1);
				viaWriteVgaIoBits(REG_SR2A, pVal->dataDrivingSel<<5, BIT5);
			}
			break;
		case DISP_DI_DVP1:
			if (pVal->isClkPolarityUsed)
				viaWriteVgaIoBits(REG_CR9B, pVal->clkPolarity<<3, BIT3);
			if (pVal->isClkAdjustUsed)
				viaWriteVgaIoBits(REG_CR9B, pVal->clkAdjust, 0x07);
			if (pVal->isClkDrivingSelUsed)
				viaWriteVgaIoBits(REG_SR65, pVal->clkDrivingSel<<2, 0x0C);
			if (pVal->isDataDrivingSelUsed)
				viaWriteVgaIoBits(REG_SR65, pVal->dataDrivingSel, 0x03);
			break;
		case DISP_DI_DFPH:
			if (pVal->isClkPolarityUsed)
				viaWriteVgaIoBits(REG_CR97, pVal->clkPolarity<<3, BIT3);
			if (pVal->isClkAdjustUsed)
				viaWriteVgaIoBits(REG_CR97, pVal->clkAdjust, 0x07);
			break;
		case DISP_DI_DFPL:
			if (pVal->isClkPolarityUsed)
				viaWriteVgaIoBits(REG_CR99, pVal->clkPolarity<<3, BIT3);
			if (pVal->isClkAdjustUsed)
				viaWriteVgaIoBits(REG_CR99, pVal->clkAdjust, 0x07);
			break;
		case DISP_DI_DFP:
			if (pVal->isClkPolarityUsed) {
				viaWriteVgaIoBits(REG_CR97, pVal->clkPolarity<<3, BIT3);
				viaWriteVgaIoBits(REG_CR99, pVal->clkPolarity<<3, BIT3);
			}
			if (pVal->isClkAdjustUsed) {
				viaWriteVgaIoBits(REG_CR97, pVal->clkAdjust, 0x07);
				viaWriteVgaIoBits(REG_CR99, pVal->clkAdjust, 0x07);
			}
			break;
		default:
			xf86DrvMsg(0, X_ERROR, "LoadUserGfxDPASetting: Invalid DIPort.\n");
			break;
	}
}

/* Set lvds subchip DPA Setting*/
static void
LoadUserLvdsDPASetting (CARD32 subchipName, CARD32 serialPort,
                                ViaLvdsDPAPtr pVal)
{
	switch (subchipName) {
		case SUBCHIP_VT1636:
			if (pVal->isVt1636ClkSelST1Used)
				viaSerialWriteByteMask(serialPort, SUBCHIP_VT1636_SLAVE_ADDR,
				    0x09, pVal->Vt1636ClkSelST1, 0x1F);
			if (pVal->isVt1636ClkSelST2Used)
				viaSerialWriteByteMask(serialPort, SUBCHIP_VT1636_SLAVE_ADDR,
				    0x08, pVal->Vt1636ClkSelST2, 0x0F);
			break;
		default:
			break;
	}
}


/*For LCD DPA. Compute clock range*/
static int 
getClkRangeIndex (CARD32 Clk)
{
    if (Clk < DPA_CLK_30M) {
        return DPA_CLK_RANGE_30M;
    } else if ((DPA_CLK_30M < Clk) && (Clk < DPA_CLK_50M)) {
        return DPA_CLK_RANGE_30_50M;
    } else if ((DPA_CLK_50M < Clk) && (Clk < DPA_CLK_70M)) {
        return DPA_CLK_RANGE_50_70M;
    } else if ((DPA_CLK_70M < Clk) && (Clk < DPA_CLK_100M)) {
        return DPA_CLK_RANGE_70_100M;
    } else if ((DPA_CLK_100M < Clk) && (Clk < DPA_CLK_150M)) {
        return DPA_CLK_RANGE_100_150M;
    } else {
        return DPA_CLK_RANGE_150M;
    }
}

static void
loadLcdGfxDPA (CARD32 chipset, CARD32 subchipName, CARD32 port,
                    CARD32 clock)
{
	int i, rangeIdx;
	GFX_DPA_INFO_TABLE_PTR pGfxDPAInfoTbl = NULL;
	GFX_DPA_VALUE_PTR pDPASettingValue = NULL;
	
    /* To find the record set according to chipset. */
    for (i = 0; i < NUM_GFX_DPA_TABLE; i++) {        
        if (GFX_DPA_INDEX_TBL[i].ChipSet == chipset)
            break;
    }
	
	if (i == NUM_GFX_DPA_TABLE)
		return;
	rangeIdx = getClkRangeIndex(clock);
	
	switch (subchipName) {
		case SUBCHIP_VT1636:
			pGfxDPAInfoTbl = GFX_DPA_INDEX_TBL[i].pVT1636DPATbl;
			break;
		case SUBCHIP_HARDWIRED_LVDS:
		case SUBCHIP_TTL:
			pGfxDPAInfoTbl = GFX_DPA_INDEX_TBL[i].pHardwiredDPATbl;
			break;
		default:
			break;
	}

	if (pGfxDPAInfoTbl)
		pDPASettingValue = pGfxDPAInfoTbl[rangeIdx].pDPASettingValue;
	
	if (pDPASettingValue)
		setDPA(port, pDPASettingValue);
}

static void
loadLvdsTransmitterDPA(CARD32 chipset, CARD32 subchipName,
                                CARD32 serialPort, CARD32 panelIndex)
{
	int i;
    TRANSMITTER_DPA_INFO_PTR pGfxDPAInfoTbl = NULL; 
	CARD32 *pDPASetting = NULL;
	CARD8 index, data, mask;
	unsigned short slaveAddr = SUBCHIP_VT1636_SLAVE_ADDR;
	
    /* To find the record set according to chipset. */
    for (i = 0; i < NUM_TRANSMITTER_DPA_TABLE; i++) {        
        if (TRANSMITTER_DPA_INDEX_TBL[i].ChipSet == chipset)
            break;        
    }
	if (i == NUM_TRANSMITTER_DPA_TABLE)
		return;
        
    /* Find the desired DPA table according to the transmitter type. */
    switch (subchipName) {
        case SUBCHIP_VT1636:
            if (TRANSMITTER_DPA_INDEX_TBL[i].pVT1636DPATbl)
                pGfxDPAInfoTbl = TRANSMITTER_DPA_INDEX_TBL[i].pVT1636DPATbl;           
            break;
        default: /* Table havn't defined. */
            break;
    }
	if (!pGfxDPAInfoTbl)
		return;
            
    /* Find the DPA setting value table according to the panel size. */
    while (pGfxDPAInfoTbl->Index != VIA_INVALID) {
        if (pGfxDPAInfoTbl->Index == panelIndex) {
			pDPASetting = pGfxDPAInfoTbl->pTransmitterDPASetting;
			break;
        }
        pGfxDPAInfoTbl++;
    }
	
    if (!pDPASetting)
        return;

	while (*pDPASetting != 0xFFFFFF) {
		index = (*pDPASetting) & 0xFF;
		mask = (*pDPASetting >> 8) & 0xFF;
		data = (*pDPASetting >> 16) & 0xFF;
		viaSerialWriteByteMask(serialPort, slaveAddr, index, data, mask);
		pDPASetting++;
	}
}

static void
loadLcdDefaultDPASetting(xf86OutputPtr output, CARD32 clock)
{
	ScrnInfoPtr 			pScrn = output->scrn;
	VIAPtr 					pVia = VIAPTR(pScrn);  
	ViaLcdPrivateInfoPtr 	lcdInfo = (ViaLcdPrivateInfoPtr)(output->driver_private);
	CARD32					subchipName = lcdInfo->commonInfo.subChipName;
	CARD32					serialPort = lcdInfo->commonInfo.serialPort;
	CARD32					diPort = lcdInfo->commonInfo.diPort;
	CARD32 					panelWidth = lcdInfo->commonInfo.physicalWidth;
	CARD32 					panelHeigh = lcdInfo->commonInfo.physicalHeight;
	CARD32					panelIndex = VIA_MAKE_ID(panelWidth, panelHeigh);
	
	loadLcdGfxDPA(pVia->Chipset, subchipName, diPort, clock*1000);
	loadLvdsTransmitterDPA(pVia->Chipset, subchipName, serialPort, panelIndex);
}


static void
patchDviDvp0DPA(CARD32 chipset, int modeH, int modeV)
{
	switch (chipset) {
        case VIA_P4M890:
            if ((modeH == 1600) && (modeV == 1200))
                viaWriteVgaIoBits(REG_CR96, 0x03, BIT0+BIT1+BIT2);
            else
                viaWriteVgaIoBits(REG_CR96, 0x07, BIT0+BIT1+BIT2);
            break;
        case VIA_P4M900:
            /* Validate by CN896(VT5943A-2), VN896(VT6363A1-2)*/
            /* and P4M900(VT8450C/VT8450D). */
            viaWriteVgaIoBits(REG_CR96, 0x02, BIT0+BIT1+BIT2);
            break;
		default:
			break;
	}
}

static void
patchDviDfpLowDPA(CARD32 chipset)
{
    switch (chipset) {
        case VIA_K8M890:
            viaWriteVgaIoBits(REG_CR99, 0x03, BIT0+BIT1);
            break;
        case VIA_P4M900:        
            /* Validate by CN896(VT5943A-2), VN896(VT6363A1-2)*/
            /* and P4M900(VT8450C/VT8450D). */
            viaWriteVgaIoBits(REG_CR99, 0x09, BIT0+BIT1+BIT2+BIT3);
            break;
        case VIA_P4M890:        
            viaWriteVgaIoBits(REG_CR99, 0x0F, BIT0+BIT1+BIT2+BIT3);
            break;
        default:
           break;
    }
}

static void
loadDviDefaultDPASetting(CARD32 chipset, CARD32 port, DisplayModePtr mode)
{
	switch (port) {
		case DISP_DI_DVP0:
			patchDviDvp0DPA(chipset, mode->HDisplay, mode->VDisplay);
			break;
		case DISP_DI_DFPL:
			patchDviDfpLowDPA(chipset);
			break;
		default:
			break;
	}
}

static void
viaGetMonitorPhysicalSize(xf86MonPtr edidMon, CARD32* phyW, CARD32* phyH)
{
    CARD32 width=0, height=0;
    CARD32 i=0;

    /* established timing */
    if (edidMon->timings1.t1 & 0xD0) {
        width = 720;
        height = 400;
    }
    if (edidMon->timings1.t1 & 0x3D) {
        width = 640;
        height = 480;
    }
    if (edidMon->timings1.t1 & 0x03) {
        width = 800;
        height = 600;
    }
    if (edidMon->timings1.t2 & 0xD0) {
        width = 800;
        height = 600;
    }
    if (edidMon->timings1.t2 & 0x20) {
        width = 832;
        height = 624;
    }
    if (edidMon->timings1.t2 & 0x1E) {
        width = 1024;
        height = 768;
    }
    if (edidMon->timings1.t2 & 0x01) {
        width = 1280;
        height = 1024;
    }

    /* standard timing */    
    for (i=0; i<STD_TIMINGS; i++) {
        if (edidMon->timings2[i].hsize*edidMon->timings2[i].vsize 
            > width*height) {
            width = edidMon->timings2[i].hsize;
            height = edidMon->timings2[i].vsize;
        }                
    }      

    /* detail timing */
    for (i=0; i<DET_TIMINGS; i++) {
        if ((edidMon->det_mon[i].type == DT) && 
            (edidMon->det_mon[i].section.d_timings.h_active*
             edidMon->det_mon[i].section.d_timings.v_active 
             > width*height)) {
            width = edidMon->det_mon[i].section.d_timings.h_active;
            height = edidMon->det_mon[i].section.d_timings.v_active;
        }                
    }

    *phyW = width;
    *phyH = height;
}

/* ====================================================*/
/*                  Output Callback Functions                              */
/* ==================================================== */
/*****************************************************/
/*					CRT                              */
/*****************************************************/
/*
* Saves the crtc's state for restoration on VT switch.

static void 
via_crt_save (xf86OutputPtr output)
{
}
*/
/*
* Restore's the crtc's state at VT switch.

static void 
via_crt_restore (xf86OutputPtr output)
{
}
*/
static void 
via_crt_destroy (xf86OutputPtr output)
{
    if (output->driver_private)
        xfree (output->driver_private);
}

/*
* switch the level of DPMS. we should turn on/off the device in this function.
*/
static void
via_crt_dpms(xf86OutputPtr output, int mode)
{
    DEBUG(ErrorF("via_crt_dpms, mode = %d\n", mode));
    
    ScrnInfoPtr     pScrn = output->scrn;
    int             usedIga;
    ViaOutputInfo   commonInfo = ((ViaCrtPrivateInfoPtr)
                                    (output->driver_private))->commonInfo;
 
    /* Clear DPMS setting ??*/
    switch (mode) {
        case DPMSModeOn: 
            usedIga = ((VIACrtcPrivatePtr)(output->crtc->driver_private))->iga;
            if (NONE_SUBCHIP == commonInfo.subChipName) {
                switch(usedIga) {
                    case IGA1:
                        viaWriteVgaIoBits(REG_SR16, 0x00, BIT6);
                        break;
                    case IGA2:
                        viaWriteVgaIoBits(REG_SR16, 0x40, BIT6);
                        break;
                }
                viaWriteVgaIoBits(REG_CR36, 0x00, BIT4+BIT5);
            }                    
            break;
            
        case DPMSModeStandby:   /* 1 */
        case DPMSModeSuspend:   /* 2 */
        case DPMSModeOff:       /* 3 */       
            if (NONE_SUBCHIP == commonInfo.subChipName) {
                viaWriteVgaIoBits(REG_CR36, BIT5+BIT4, BIT5+BIT4);
                /*Reset CRT source to IGA1*/
                viaWriteVgaIoBits(REG_SR16, 0x00, BIT6);
            }   
             /*Turn off DI port clk*/
            viaDIPortPadOff(commonInfo.diPort);   
            break;
  
        default:
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Invalid DPMS mode %d\n", mode);
            break;
    }
}

static void 
via_crt_mode_set(xf86OutputPtr output,
               DisplayModePtr mode,
               DisplayModePtr adjusted_mode)
{   
    ViaCrtPrivateInfoPtr 	viaCrtInfo = (ViaCrtPrivateInfoPtr)
                                            (output->driver_private);
	CARD32   				diPort = viaCrtInfo->commonInfo.diPort;
	
    /* update sync polarity */
    setDiportSyncPolarity(diPort, adjusted_mode);	
}

/* To detect if the device is connect. */
static xf86OutputStatus 
via_crt_detect(xf86OutputPtr output)
{
    DEBUG(ErrorF("via_crt_detect\n"));
    
    ScrnInfoPtr         pScrn = output->scrn;
    VIAPtr              pVia = VIAPTR(pScrn);     
    xf86OutputStatus    status = XF86OutputStatusDisconnected;        
    xf86MonPtr          edid_mon; 
    Bool                result = FALSE;
    ViaCrtPrivateInfoPtr viaCrtInfo = output->driver_private;

    /* CRT detect first read EDID, if can't get EDID,detect vsync */     
    if (!viaCrtInfo->commonInfo.NoDDCValue) {   

        unsigned char       *RawEDID;
        RawEDID = xcalloc(1,sizeof(unsigned char) * (128));
        viaSerialReadBytes(viaCrtInfo->commonInfo.ddcPort, 
                       0xA0, 0, RawEDID, 128);
        edid_mon = xf86InterpretEDID(pScrn->scrnIndex,RawEDID);
        if (edid_mon) {
            /* Check input type of EDID,
               VGA: analog input 
               LCD/DVI: digital input */
            if (!DIGITAL(edid_mon->features.input_type)) {
                result = TRUE;
            }    
        }
    }

    if (result)
        status = XF86OutputStatusConnected;
    else 
        status = viaDetectCRTVsync(pVia); 

    return status;
}

static void 
via_crt_prepare (xf86OutputPtr output)
{
    output->funcs->dpms (output, DPMSModeOff);
}

static void 
via_crt_commit (xf86OutputPtr output)
{
    output->funcs->dpms (output, DPMSModeOn);
}

static DisplayModePtr 
via_crt_get_modes (xf86OutputPtr output)
{
    ScrnInfoPtr         pScrn = output->scrn;
    DisplayModePtr      modes = NULL;
    xf86MonPtr          edid_mon;  
    unsigned char       *RawEDID; 
    ViaCrtPrivateInfoPtr viaCrtInfo = output->driver_private;
    /* if "NoDDCValue", set physical size as 4096x2048,
       now, only support mode from modeline and xorg build-in */
    if ( viaCrtInfo->commonInfo.NoDDCValue ) {
        output->MonInfo = NULL;
        return NULL;
    }
    RawEDID = xcalloc(1,sizeof(unsigned char) * (128));

    /* Parse the EDID block */
    viaSerialReadBytes(viaCrtInfo->commonInfo.ddcPort, 
                       0xA0, 0, RawEDID, 128);
    
    edid_mon = xf86InterpretEDID(pScrn->scrnIndex,RawEDID);
    
    if (edid_mon) {
        xf86OutputSetEDID(output, edid_mon);  
        modes = xf86OutputGetEDIDModes(output);
    }                                
    else
        output->MonInfo = NULL;
    /* return a mode list, not a single mode */
    return modes;    
}


/*
 * To check whether the input mode is supported of not in our driver.
 */
static int 
via_crt_mode_valid (xf86OutputPtr output, DisplayModePtr mode)
{
    /* Check Clock Range */
    if(mode->Clock > 400000)
        return MODE_CLOCK_HIGH;        
    if(mode->Clock < 25000)
        return MODE_CLOCK_LOW;

    /* Check Resolution Range */
    if(IsValidMode(mode->HDisplay, mode->VDisplay) == 0)
         return MODE_BAD;
    return MODE_OK;
}

/* To fix up the input mode */
static Bool 
via_crt_mode_fixup (xf86OutputPtr output,
                 DisplayModePtr mode,
                 DisplayModePtr adjusted_mode)
{
    return TRUE;
}


static const xf86OutputFuncsRec via_crt_output_funcs = {
    .dpms = via_crt_dpms,
    .save = NULL,
    .restore = NULL,
    .mode_valid = via_crt_mode_valid,
    .mode_fixup = via_crt_mode_fixup,
    .prepare = via_crt_prepare,
    .mode_set = via_crt_mode_set,
    .commit = via_crt_commit,
    .detect = via_crt_detect,    
    .get_modes = via_crt_get_modes,
    .destroy = via_crt_destroy,        
};

/*****************************************************/
/*					LCD                              */
/*****************************************************/
/* Create a device dependent properties */
static void
via_lvds_create_resources (xf86OutputPtr output)
{
	ScrnInfoPtr pScrn = output->scrn;
	CARD32 centering = ((ViaLcdPrivateInfoPtr)(output->driver_private))->center;
	int	err,i;
	
	/*Now we only support one LCD dynamic change feature: Centering or not Centering*/

	/* Set up the Lcd_centering property, which takes effect on mode set
     * and accepts strings that match exactly
     */
    lcd_centering_atom = MakeAtom(LCD_CENTERING_NAME, 
                            sizeof(LCD_CENTERING_NAME) - 1, TRUE);

	for (i = 0; i < LCD_CENTERING_NUM; i++) {
		lcd_centering_name_atoms[i] = MakeAtom(lcd_centering_names[i],
		                                strlen(lcd_centering_names[i]),TRUE);
	}

	err = RRConfigureOutputProperty (output->randr_output, lcd_centering_atom,
                     TRUE, FALSE, FALSE, 
                     LCD_CENTERING_NUM, (INT32 *) lcd_centering_name_atoms);
	if (err != 0) {
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		   				"RRConfigureOutputProperty error, %d\n", err);
    }

	err = RRChangeOutputProperty(output->randr_output, lcd_centering_atom,
				 XA_ATOM, 32, PropModeReplace, 1,
				 &lcd_centering_name_atoms[centering],
				 FALSE, TRUE);
    if (err != 0) {
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		   				"failed to set panel fitting mode, %d\n", err);
    } 
}

static Bool
via_lvds_set_property(xf86OutputPtr output, Atom property,
                     RRPropertyValuePtr value)
{
	DEBUG(ErrorF("via_lvds_set_property\n"));
	ScrnInfoPtr pScrn = output->scrn;
	ViaLcdPrivateInfoPtr viaLcdInfo = output->driver_private;
	Atom atom;
	char *name;
	int center = -1;

	if (lcd_centering_atom == property) {
		if (value->type != XA_ATOM || value->format != 32 ||value->size != 1){
	    	return FALSE;
		}

		memcpy(&atom, value->data, 4);
		name = NameForAtom(atom);

		if (!strcmp(name, lcd_centering_names[0])) {
			DEBUG(ErrorF("Apply expand\n"));
			center = FALSE;
		}

		if (!strcmp(name, lcd_centering_names[1])) {
			DEBUG(ErrorF("Apply center\n"));
			center = TRUE;
		}

		if (center < 0) {
	    	return FALSE;
		}

		if (viaLcdInfo->center == center) {
			return TRUE;
		}

		viaLcdInfo->center = center;

        /*Note:Xorg call xf86CrtcSetMode when we set property if the property is Atom
        so we needn't do it here.But now I am not very sure of it, so I still do it here.
        Need to Fix*/
		if (output->crtc) {
	    	xf86CrtcPtr crtc = output->crtc;
	    	if (crtc->enabled) {
				if (!xf86CrtcSetMode(crtc, &crtc->desiredMode,
				     				crtc->desiredRotation,
				     				crtc->desiredX, crtc->desiredY)) {
		    			xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
			       		    "Failed to set mode after panel fitting change!\n");
		    			return FALSE;
				}
	    	}
		}
	}
	return TRUE;
}

static void 
via_lvds_dpms(xf86OutputPtr output, int mode)
{
	ScrnInfoPtr pScrn = output->scrn;
	VIAPtr pVia = VIAPTR(pScrn);
	int usedIga;
	ViaOutputInfo commonInfo = ((ViaLcdPrivateInfoPtr)(output->driver_private))
                                ->commonInfo;
	CARD32 panelSizeId = ((ViaLcdPrivateInfoPtr)(output->driver_private))
                                ->panelIndex;
	CARD32 serialPort = commonInfo.serialPort;                
	CARD32 subChipName = commonInfo.subChipName;
	CARD32 diPort = commonInfo.diPort;
	BOOL dualChannel = ((ViaLcdPrivateInfoPtr)(output->driver_private))
                        ->dualChannel;

	/*if the panel is swg panel, we need a panelSizeId to set power sequence*/
	if (!panelSizeId) {
		panelSizeId = VIA_MAKE_ID(commonInfo.physicalHeight,
                                    commonInfo.physicalWidth);  
	}
	
	switch (mode) {
	case DPMSModeOn:
		usedIga = ((VIACrtcPrivatePtr)(output->crtc->driver_private))->iga;  

        /*when using the EPIA-EX board, if we do not set this bit, light LCD will failed
          in nonRandR structure, when light LCD this bit is always setted, but I don't know
          why, here need to fix*/
        viaWriteVgaIoBits(REG_CR6A, 0x08, BIT3);
        
		/*1. Turn on DI port clock, set Iga source for DI port*/
		if (diPort) 
			viaSetOutputPath(diPort, usedIga, pVia->Chipset);
		
		/*2. Enable spread spectrum*/
		if (pVia->pChipInfo->biosIsSpreadSpectrum) {
			if ((pVia->Chipset == VIA_CX700) ||
				(pVia->Chipset == VIA_VX800) ||
				(pVia->Chipset == VIA_VX855)) {
				viaWriteVgaIoBits(REG_SR3D, 0x01, BIT0);
			} else {		
				viaWriteVgaIoBits(REG_SR2C, 0x01, BIT0);
			}
			viaWriteVgaIoBits(REG_SR1E, 0x08, BIT3);
		}

		/*3. Enable LVDS transmitter*/
		if (SUBCHIP_INTEGRATED_LVDS == subChipName) {
			enableInternalLvds(pVia, diPort, panelSizeId, dualChannel);
		} else if (SUBCHIP_VT1636 == subChipName) {
			enableLcdViaVt1636(serialPort);
        } else if (SUBCHIP_TTL == subChipName) {

            if (pVia->Chipset == VIA_VX855) {
                /* For 409 18bit TTL panel */
                viaWriteVgaIoBits(REG_CRF3, 0x80, BIT7);
                /* Power on LVDS channel. */
                viaWriteVgaIoBits(REG_CRD2, 0x00, BIT6+BIT7);
                /*Enable LCD*/
                viaWriteVgaIoBits(REG_CR6A, 0x08, BIT3);    
            } else {
                /* Power on LVDS channel. */
                viaWriteVgaIoBits(REG_CRD2, 0x00, BIT7);
            
                /* For Qunta's TTL LCD (IL1) */
                /* Software control Power Sequence */        
                ttlLcdPowerSequenceOn(pVia);
            }
        } else {   
		    /*Hw wired Lcd*/
			/*Turn on display period in the panel path(LCD only), turn on back light*/
			viaWriteVgaIoBits(REG_CR91, 0x00, BIT6+BIT7);
			/*Enable LCD*/
			viaWriteVgaIoBits(REG_CR6A, 0x08, BIT3);                 
		}
		break;
	case DPMSModeStandby:
	case DPMSModeSuspend:
	case DPMSModeOff:
		/*1.Disable spread spectrum, we put it to crtc DPMSModeOff*/ 

		/*2.Disable LVDS transmitter and turn off panel*/
		if (SUBCHIP_INTEGRATED_LVDS == subChipName) {
			disableInternalLvds(pVia, diPort, panelSizeId, dualChannel);
		} else if(SUBCHIP_VT1636 == subChipName) {
			disableLcdViaVt1636(serialPort);
		} else if(SUBCHIP_TTL == subChipName) {
		    if (pVia->Chipset == VIA_VX855) {
                /*Turn off display period in the panel path(LCD only), turn off back light*/
                viaWriteVgaIoBits(REG_CR91, 0xC0, BIT6+BIT7);
                /*Disable LCD*/
                viaWriteVgaIoBits(REG_CR6A, 0x00, BIT3);                 
            } else {
                ttlLCDPowerSequenceOff(pVia);
                /* Power off LVDS channel. */
                viaWriteVgaIoBits(REG_CRD2, 0x80, BIT7);
            }
		} else {    
		    /*Hw wired Lcd*/
			/*Turn off display period in the panel path(LCD only), turn off back light*/
			viaWriteVgaIoBits(REG_CR91, 0xC0, BIT6+BIT7);
			/*Disable LCD*/
			viaWriteVgaIoBits(REG_CR6A, 0x00, BIT3);                 
		}

		/* 3.Disable Iga2 LCD scaling, we put it to crtc DPMSModeOff*/

		/* 4. Turn off DI port clk*/
		viaDIPortPadOff(diPort);
		break;
	default:
		xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Invalid DPMS mode %d\n", mode);
		break;
	}
}

/*
* Saves the lvds's state for restoration on VT switch.

static void 
via_lvds_save (xf86OutputPtr output)
{

}
*/
/*
* Restore's the lvds's state at VT switch.

static void 
via_lvds_restore (xf86OutputPtr output)
{
	ScrnInfoPtr 			pScrn = output->scrn;
	VIAPtr 					pVia = VIAPTR(pScrn);

	if (pVia->pChipInfo->biosActiveDev & VIA_DEVICE_LCD) {
		via_lvds_dpms(output, DPMSModeOn);
	}
}
*/
/*
 * Callback for testing a video mode for a given output.
 *
 * This function should only check for cases where a mode can't be supported
 * on the pipe specifically, and not represent generic CRTC limitations.
 *
 * return MODE_OK if the mode is valid, or another MODE_* otherwise.
 */
static int 
via_lvds_mode_valid(xf86OutputPtr output, DisplayModePtr mode)
{    
    ViaOutputInfo commonInfo = ((ViaLcdPrivateInfoPtr)(output->driver_private))
                                ->commonInfo;
    if (!memcmp(mode->name, "PanelMode", 9)) {
        mode->type |= M_T_PREFERRED;
        return MODE_OK;
    }
    
    if (commonInfo.physicalWidth && commonInfo.physicalHeight) {
        /*Don't support mode larger than physical size*/
        if (mode->HDisplay > CMDISP_ALIGN_TO(commonInfo.physicalWidth,8))
            return MODE_PANEL;
        if (mode->VDisplay > CMDISP_ALIGN_TO(commonInfo.physicalHeight,8))
            return MODE_PANEL;
    }

    return MODE_OK;
}

static Bool 
via_lvds_mode_fixup(xf86OutputPtr output,
                                   DisplayModePtr mode,
                                   DisplayModePtr adjusted_mode)
{
    ScrnInfoPtr pScrn = output->scrn;
    VIAPtr pVia = VIAPTR(pScrn);  
    VIACrtcPrivatePtr crtcPriv = VIACrtcPrivate(output->crtc);
    CARD32 usedIga = crtcPriv->iga;
    ViaLcdPrivateInfoPtr lcdPriv = (ViaLcdPrivateInfoPtr)
                                    (output->driver_private);
    CARD32 isCenter = lcdPriv->center;
    CARD32 panelWidth = lcdPriv->commonInfo.physicalWidth;
    CARD32 panelHeight = lcdPriv->commonInfo.physicalHeight;

    /*If mode is equal to panel physical size, don't change it
        Maybe we should discuss it!*/
    if ((mode->HDisplay != panelWidth) ||
        (mode->VDisplay != panelHeight)) {

        if (lcdPriv->confMode) {
            memcpy(adjusted_mode, lcdPriv->confMode, sizeof(DisplayModeRec));
        } else {
            /*Get build-in timing*/
            ViaTimingTablePtr panelCrtcTable = NULL;
            panelCrtcTable = getDriverTiming(LCD_MODE_TABLE, lcdPriv->panelIndex);
            if (!panelCrtcTable)
                return FALSE;
            
            fillModeByDriverTiming(panelCrtcTable, adjusted_mode);
        }
        
        xf86SetModeCrtc(adjusted_mode, 0);
        /*Take care of 3353 downscaling*/
        if ((mode->HDisplay > panelWidth) ||
            (mode->VDisplay > panelHeight)) {
            if ((pVia->Chipset == VIA_VX800) ||
                (pVia->Chipset == VIA_VX855)) {
                /*Do downscaling*/
            }
        } else {
            if ((usedIga != IGA2) || isCenter) 
                /*Do centering according to mode and adjusted_mode*/
                centeringTiming(mode, adjusted_mode);
        }
    }

    return TRUE;
}

static void 
via_lvds_mode_set(xf86OutputPtr output,
                  		DisplayModePtr mode,
                  		DisplayModePtr adjusted_mode)
{
	VIACrtcPrivatePtr 		viaCrtc = VIACrtcPrivate(output->crtc);
	CARD32 					usedIga = viaCrtc->iga;
	ViaLcdPrivateInfoPtr	lcdPriv = (ViaLcdPrivateInfoPtr)
                                        (output->driver_private);
	CARD32 					isCenter = lcdPriv->center;
	CARD32					diPort = lcdPriv->commonInfo.diPort;
	CARD32					subchipName = lcdPriv->commonInfo.subChipName;
	CARD32					serialPort = lcdPriv->commonInfo.serialPort;
	CARD32 					panelWidth = lcdPriv->commonInfo.physicalWidth;
	CARD32 					panelHeigh = lcdPriv->commonInfo.physicalHeight;

    /*initial igastatus*/
    viaCrtc->igastatus = 0;
  
	if ((mode->HDisplay <= panelWidth) &&
		(mode->VDisplay <= panelHeigh)) {
		/* Only IGA2 can do expansion*/
		if ((usedIga == IGA2) && !isCenter) {
			if (mode->HDisplay < panelWidth)
				loadScaleFactorRegs(mode->HDisplay, panelWidth, TRUE);
			if (mode->VDisplay < panelHeigh) 
				loadScaleFactorRegs(mode->VDisplay, panelHeigh, FALSE);

            viaCrtc->igastatus = VIA_EXPAND;
		}
	} else {
		/*Take care of down scaling for 3353*/
	}

	/*Patch for clock skew*/
	loadLcdDefaultDPASetting(output, adjusted_mode->Clock);
	LoadUserGfxDPASetting(diPort, &lcdPriv->userGfxDPA);
	LoadUserLvdsDPASetting(subchipName, serialPort, &lcdPriv->userLvdsDPA);
}

static xf86OutputStatus 
via_lvds_detect(xf86OutputPtr output)
{
	ScrnInfoPtr pScrn = output->scrn;
	VIAPtr      pVia = VIAPTR(pScrn);
	ViaLcdPrivateInfoPtr viaLcdInfo = output->driver_private;
	CARD32 panelSizeId = ((ViaLcdPrivateInfoPtr)(output->driver_private))
                            ->panelIndex;
	xf86OutputStatus status = XF86OutputStatusDisconnected;
	xf86MonPtr edid_mon;
	
	/*if users set panelSizeId, we consider the lcd has no EDID
	  if users don't set panelSizeId, we consider the lcd is SPWG panel.*/
	if (panelSizeId) {
    	status = XF86OutputStatusConnected;
    /* if the DDC port is occupied by DVI, return disconnected status,
       because we can't make differece between LCD and DVI */	
	} else if (viaLcdInfo->commonInfo.ddcPort != pVia->ddcPortOfDvi) {
		unsigned char  *RawEDID;
		RawEDID = xcalloc(1,sizeof(unsigned char) * (128));
		viaSerialReadBytes(viaLcdInfo->commonInfo.ddcPort, 
						0xA0, 0, RawEDID, 128);
		edid_mon = xf86InterpretEDID(pScrn->scrnIndex, RawEDID);
        if (edid_mon) {
            /* Check input type of EDID,
               VGA: analog input 
               LCD/DVI: digital input */
            if (DIGITAL(edid_mon->features.input_type)) {
                status = XF86OutputStatusConnected;
            }    
        }
	}
	return status;
}

static void 
via_lvds_destroy (xf86OutputPtr output)
{
	if (output->driver_private) {
		xfree (output->driver_private);
	}
}

static void 
via_lvds_prepare (xf86OutputPtr output)
{
	via_lvds_dpms(output, DPMSModeOff);
}

static void
via_lvds_commit (xf86OutputPtr output)
{
	via_lvds_dpms(output, DPMSModeOn);
}

static DisplayModePtr 
via_lvds_get_modes (xf86OutputPtr output)
{
	ScrnInfoPtr pScrn = output->scrn;
	DisplayModePtr modes = NULL;
	xf86MonPtr edid_mon = NULL;
	unsigned char	*rawEDID = NULL; 
	ViaLcdPrivateInfoPtr viaLcdInfo = output->driver_private;

    if (viaLcdInfo->confMode){
        output->MonInfo = NULL;
        edid_mon = xcalloc (1, sizeof (xf86Monitor));
        if (edid_mon) {
            /* Set wide sync ranges so we get all modes
            * handed to valid_mode for checking
            */
            edid_mon->det_mon[0].type = DS_RANGES;
            edid_mon->det_mon[0].section.ranges.min_v = 0;
            edid_mon->det_mon[0].section.ranges.max_v = 200;
            edid_mon->det_mon[0].section.ranges.min_h = 0;
            edid_mon->det_mon[0].section.ranges.max_h = 200;

            output->MonInfo = edid_mon;
        }        
        return NULL;
    }
    
	/* if users set panelIndex, we add the related buildin mode timing to xorg*/
	if (viaLcdInfo->panelIndex) {
		output->MonInfo = NULL;
		modes = viaGetPanelModeRecord(pScrn, viaLcdInfo->panelIndex);
	} else {
		rawEDID = xcalloc(1,sizeof(unsigned char) * (128));

		/* Parse the EDID block */
		viaSerialReadBytes(viaLcdInfo->commonInfo.ddcPort, 0xA0, 0,
		                    rawEDID, 128);
   		edid_mon = xf86InterpretEDID(pScrn->scrnIndex,rawEDID);
		if (edid_mon) {
			/*how to get physicalWidth and physicalHeight of LCD*/
			viaGetMonitorPhysicalSize(edid_mon, 
			                            &viaLcdInfo->commonInfo.physicalWidth,
			                            &viaLcdInfo->commonInfo.physicalHeight);
			xf86OutputSetEDID(output, edid_mon);  
			modes = xf86OutputGetEDIDModes(output);
		} else 
			output->MonInfo = NULL;
	}

	if (!output->MonInfo) {
		edid_mon = xcalloc (1, sizeof (xf86Monitor));
		if (edid_mon) {
            /*defaultly display support continuous-freqencey
            if the parameter is not set, the xorg build in modes will not add,
            then lcd do not have edid, will have only one valid mode*/
	        edid_mon->features.msc |= 0x1;
			/* Set wide sync ranges so we get all modes
			* handed to valid_mode for checking
			*/
			edid_mon->det_mon[0].type = DS_RANGES;
			edid_mon->det_mon[0].section.ranges.min_v = 0;
			edid_mon->det_mon[0].section.ranges.max_v = 200;
			edid_mon->det_mon[0].section.ranges.min_h = 0;
			edid_mon->det_mon[0].section.ranges.max_h = 200;

			output->MonInfo = edid_mon;
		}
    }
	
    /* return a mode list, not a single mode */
    return modes;   	
}

static const xf86OutputFuncsRec via_lvds_output_funcs = {   
    .create_resources = via_lvds_create_resources,
    .set_property = via_lvds_set_property,   
    .dpms = via_lvds_dpms,
    .save = NULL,
    .restore = NULL,
    .mode_valid = via_lvds_mode_valid,
    .mode_fixup = via_lvds_mode_fixup,
    .prepare = via_lvds_prepare,
    .mode_set = via_lvds_mode_set,
    .commit = via_lvds_commit,
    .detect = via_lvds_detect,    
    .get_modes = via_lvds_get_modes,
    .destroy = via_lvds_destroy,        
};

/*****************************************************/
/*					DVI                              */
/*****************************************************/

static void 
via_tmds_dpms(xf86OutputPtr output, int mode)
{
     DEBUG(ErrorF("via_tmds_dpms, mode = %d\n", mode));
    
     ScrnInfoPtr     pScrn = output->scrn;
     VIAPtr          pVia = VIAPTR(pScrn);
     int             usedIga;
     ViaOutputInfo   commonInfo = ((ViaDviPrivateInfoPtr)
                                    (output->driver_private))->commonInfo;
     CARD32   serialPort = commonInfo.serialPort;                
     CARD32   subChipName = commonInfo.subChipName;
     CARD32   diPort = commonInfo.diPort;
     
     switch (mode) {
        case DPMSModeOn: 
            usedIga = ((VIACrtcPrivatePtr)(output->crtc->driver_private))->iga;               
            if (NONE_SUBCHIP == subChipName)
                return;
                
             /*1. Turn on DI port clock, set Iga source for DI port*/
            viaSetOutputPath(diPort, usedIga, pVia->Chipset);
             /*2. Enable TMDS transmitter*/
            if (SUBCHIP_VT1632 == subChipName) {
                /*Power on Vt1632*/
                viaSerialWriteByte(serialPort, SUBCHIP_VT1632_SLAVE_ADDR, 
                                    0x08, 0x3B);
            } else if (SUBCHIP_INTEGRATED_TMDS == subChipName) {
                /*Internal TMDS only use DFP_L*/
                /*Turn on DVI panel path(Only for internal), otherwise, 
                 the screen of DVI will be black.*/
                viaWriteVgaIoBits(REG_CR91, 0x00, BIT7);
                /*Power on TMDS*/
                viaWriteVgaIoBits(REG_CRD2, 0x00, BIT3);
            }
            break;

        case DPMSModeStandby:   /* 1 */
        case DPMSModeSuspend:   /* 2 */
        case DPMSModeOff:       /* 3 */                           
            if (NONE_SUBCHIP == subChipName)
                return;
                    
            if (SUBCHIP_VT1632 == subChipName) {
                /*Power on Vt1632*/
                viaSerialWriteByteMask(serialPort, 
                    SUBCHIP_VT1632_SLAVE_ADDR, 0x08, 0x00, BIT0);
            } else if (SUBCHIP_INTEGRATED_TMDS == subChipName) {
                viaWriteVgaIoBits(REG_CR91, 0x80, BIT7);
                /*Power off TMDS*/
                viaWriteVgaIoBits(REG_CRD2, 0x08, BIT3);
            }
            viaDIPortPadOff(diPort);
            break;
            
        default:
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Invalid DPMS mode %d\n",
                mode);
            break;
    }     
}

/*
* Saves the crtc's state for restoration on VT switch.

static void 
via_tmds_save (xf86OutputPtr output)
{
}
*/
/*
* Restore's the crtc's state at VT switch.

static void 
via_tmds_restore (xf86OutputPtr output)
{
	ScrnInfoPtr 			pScrn = output->scrn;
	VIAPtr 					pVia = VIAPTR(pScrn);

	if (pVia->pChipInfo->biosActiveDev & VIA_DEVICE_DFP) {
		via_tmds_dpms(output, DPMSModeOn);
	}
}
*/
/*
 * Callback for testing a video mode for a given output.
 *
 * This function should only check for cases where a mode can't be supported
 * on the pipe specifically, and not represent generic CRTC limitations.
 *
 * return MODE_OK if the mode is valid, or another MODE_* otherwise.
 */
static int 
via_tmds_mode_valid (xf86OutputPtr output, DisplayModePtr mode)
{
    ViaDviPrivateInfoPtr viaDviInfo = output->driver_private;

    /* Check Clock Range */
    if(mode->Clock > 400000)
        return MODE_CLOCK_HIGH;        
    if(mode->Clock < 25000)
        return MODE_CLOCK_LOW;

    /* Check monitor physical size */
    if (mode->HDisplay > viaDviInfo->commonInfo.physicalWidth)
        return MODE_BAD_HVALUE;
    if (mode->VDisplay > viaDviInfo->commonInfo.physicalHeight)
        return MODE_BAD_VVALUE;

    /* Check Resolution Range */
    if(IsValidMode(mode->HDisplay, mode->VDisplay) == 0)
         return MODE_BAD;
    
    return MODE_OK;
}

static Bool 
via_tmds_mode_fixup(xf86OutputPtr output,
                                   DisplayModePtr mode,
                                   DisplayModePtr adjusted_mode)
{
    return TRUE;
}

static void 
via_tmds_mode_set(xf86OutputPtr output,
                                 DisplayModePtr mode,
                                 DisplayModePtr adjusted_mode)
{
	ScrnInfoPtr         	pScrn = output->scrn;
	VIAPtr 					pVia = VIAPTR(pScrn);
    ViaDviPrivateInfoPtr 	viaDviInfo = output->driver_private;
	CARD32					diPort = viaDviInfo->commonInfo.diPort;

    /* update sync polarity */
    setDiportSyncPolarity(diPort, adjusted_mode);  

	/*Patch for clock skew*/
	loadDviDefaultDPASetting(pVia->Chipset, diPort, adjusted_mode);
	LoadUserGfxDPASetting(diPort, &viaDviInfo->userGfxDPA);
}

static xf86OutputStatus 
via_tmds_detect(xf86OutputPtr output)
{
    DEBUG(ErrorF("via_dvi_detect\n"));
    
    ScrnInfoPtr         pScrn = output->scrn;
    VIAPtr              pVia = VIAPTR(pScrn);
    xf86OutputStatus    status = XF86OutputStatusDisconnected;        
    xf86MonPtr          edid_mon;  
    ViaDviPrivateInfoPtr viaDviInfo = output->driver_private;
    
    /*if set NoDDCValue ,return XF86OutputStatusConnected .Else we read EDID*/
    if (viaDviInfo->commonInfo.NoDDCValue)
        status = XF86OutputStatusConnected;
    else {   
        unsigned char       *RawEDID;
        RawEDID = xcalloc(1,sizeof(unsigned char) * (128));
        viaSerialReadBytes(viaDviInfo->commonInfo.ddcPort, 
                       0xA0, 0, RawEDID, 128);
        edid_mon = xf86InterpretEDID(pScrn->scrnIndex,RawEDID);
        if (edid_mon) {
            /* Check input type of EDID,
               VGA: analog input 
               LCD/DVI: digital input */
            if (DIGITAL(edid_mon->features.input_type)) {
                pVia->ddcPortOfDvi = viaDviInfo->commonInfo.ddcPort;
                status = XF86OutputStatusConnected; 
            }    
        }
    }
    return status;
}

static void 
via_tmds_destroy (xf86OutputPtr output)
{
    if (output->driver_private)
        xfree (output->driver_private);         
}

static void 
via_tmds_prepare (xf86OutputPtr output)
{
    output->funcs->dpms (output, DPMSModeOff);           
}

static void 
via_tmds_commit (xf86OutputPtr output)
{
    output->funcs->dpms (output, DPMSModeOn);              
}

static DisplayModePtr 
via_tmds_get_modes (xf86OutputPtr output)
{
    ScrnInfoPtr         pScrn = output->scrn;
    DisplayModePtr      modes = NULL;
    xf86MonPtr          edid_mon;  
    unsigned char       *RawEDID; 
    ViaDviPrivateInfoPtr viaDviInfo = output->driver_private;
    /* if "NoDDCValue", set physical size as 4096x2048,
       now, only support mode from modeline and xorg build-in*/
    if ( viaDviInfo->commonInfo.NoDDCValue) {
        output->MonInfo = NULL;
        viaDviInfo->commonInfo.physicalWidth = 4096;
        viaDviInfo->commonInfo.physicalHeight = 2048;
        return NULL;
    }
    RawEDID = xcalloc(1,sizeof(unsigned char) * (128));

    /* Parse the EDID block */
    viaSerialReadBytes(viaDviInfo->commonInfo.ddcPort, 
                       0xA0, 0, RawEDID, 128);
    
    edid_mon = xf86InterpretEDID(pScrn->scrnIndex,RawEDID);
    
    if (edid_mon) {
        viaGetMonitorPhysicalSize(edid_mon, &viaDviInfo->commonInfo.physicalWidth,
                                  &viaDviInfo->commonInfo.physicalHeight);
        xf86OutputSetEDID(output, edid_mon);  
        modes = xf86OutputGetEDIDModes(output);
    } else {
        output->MonInfo = NULL;
    }
    /* return a mode list, not a single mode */
    return modes;
}

static const xf86OutputFuncsRec via_tmds_output_funcs = {
    .dpms = via_tmds_dpms,
    .save = NULL,
    .restore = NULL,
    .mode_valid = via_tmds_mode_valid,
    .mode_fixup = via_tmds_mode_fixup,
    .prepare = via_tmds_prepare,
    .mode_set = via_tmds_mode_set,
    .commit = via_tmds_commit,
    .detect = via_tmds_detect,    
    .get_modes = via_tmds_get_modes,
    .destroy = via_tmds_destroy,        
};


typedef enum 
{
    OPTION_CRT_TYPE,
    OPTION_CRT_DIPORT,
    OPTION_CRT_SERIALPORT,
    OPTION_CRT_DDCPORT,
    OPTION_CRT_NoDDCValue,
}ViaCrtOpts;

typedef enum 
{
    OPTION_DVI_TYPE,
    OPTION_DVI_DIPORT,
    OPTION_DVI_SERIALPORT,
    OPTION_DVI_DDCPORT,
    OPTION_DVI_NoDDCValue,
	OPTION_DVI_CLOCK_POLARITY,
	OPTION_DVI_CLOCK_ADJUST,
	OPTION_DVI_CLOCK_DRIVING_SELECTION,
	OPTION_DVI_DATA_DRIVING_SELECTION
}ViaDviOpts;


typedef enum 
{
    OPTION_LCD_TYPE,
    OPTION_LCD_DIPORT,
    OPTION_LCD_SERIALPORT,
    OPTION_LCD_DDCPORT,
    OPTION_LCD_PANEL_SIZE,
	OPTION_LCD_DUAL_CHANNEL,
	OPTION_LCD_MSB,
	OPTION_LCD_NO_DITHERING,
	OPTION_LCD_CENTER,
	OPTION_LCD_FIXONIGA1,
	OPTION_LCD_CLOCK_POLARITY,
	OPTION_LCD_CLOCK_ADJUST,
	OPTION_LCD_CLOCK_DRIVING_SELECTION,
	OPTION_LCD_DATA_DRIVING_SELECTION,
	OPTION_LCD_VT1636_CLOCK_SEL_ST1,
	OPTION_LCD_VT1636_CLOCK_SEL_ST2
}ViaLcdOpts;

static OptionInfoRec ViaCrtOptions[] =
{
	{OPTION_CRT_TYPE,       			"Type",         		OPTV_STRING,	{0}, FALSE},
	{OPTION_CRT_DIPORT,     			"DIPort",       		OPTV_STRING, 	{0}, FALSE},
	{OPTION_CRT_SERIALPORT, 			"SerialPort",   		OPTV_INTEGER,   {0}, FALSE},
	{OPTION_CRT_DDCPORT,    			"DDCPort",      		OPTV_INTEGER,   {0}, FALSE},
	{OPTION_CRT_NoDDCValue, 			"NoDDCValue",   		OPTV_BOOLEAN,   {0}, FALSE},
	{-1,                    			NULL,           		OPTV_NONE,      {0}, FALSE}	
};



static OptionInfoRec ViaDviOptions[] =
{
	{OPTION_DVI_TYPE,       			"Type",         		OPTV_STRING,    {0}, FALSE},
	{OPTION_DVI_DIPORT,     			"DIPort",       		OPTV_STRING,    {0}, FALSE},
	{OPTION_DVI_SERIALPORT, 			"SerialPort",   		OPTV_INTEGER,   {0}, FALSE},
	{OPTION_DVI_DDCPORT,    			"DDCPort",      		OPTV_INTEGER,   {0}, FALSE},
	{OPTION_DVI_NoDDCValue, 			"NoDDCValue",   		OPTV_BOOLEAN,   {0}, FALSE},
	{OPTION_DVI_CLOCK_POLARITY, 		"ClockPolarity",		OPTV_INTEGER,	{0}, FALSE},
	{OPTION_DVI_CLOCK_ADJUST,			"ClockAdjust",			OPTV_INTEGER,	{0}, FALSE},
	{OPTION_DVI_CLOCK_DRIVING_SELECTION,"ClockDrivingSelection",OPTV_INTEGER,	{0}, FALSE},
	{OPTION_DVI_DATA_DRIVING_SELECTION, "DataDrivingSelection", OPTV_INTEGER,	{0}, FALSE},
	{-1,                    			NULL,           		OPTV_NONE,      {0}, FALSE}	
};


static OptionInfoRec ViaLcdOptions[] =
{
	{OPTION_LCD_TYPE,       			"Type",         		OPTV_STRING,    {0}, FALSE},
	{OPTION_LCD_DIPORT,     			"DIPort",       		OPTV_STRING,    {0}, FALSE},
	{OPTION_LCD_SERIALPORT, 			"SerialPort",   		OPTV_INTEGER,   {0}, FALSE},
	{OPTION_LCD_DDCPORT,    			"DDCPort",      		OPTV_INTEGER,   {0}, FALSE},
	{OPTION_LCD_PANEL_SIZE,				"PanelSize",   			OPTV_STRING, 	{0}, FALSE},
	{OPTION_LCD_DUAL_CHANNEL,			"DualChannel",			OPTV_STRING,	{0}, FALSE},
	{OPTION_LCD_MSB,					"MSB",					OPTV_BOOLEAN,	{0}, FALSE},
	{OPTION_LCD_NO_DITHERING,			"NoDithering",			OPTV_BOOLEAN,	{0}, FALSE},
	{OPTION_LCD_CENTER,					"Center",				OPTV_BOOLEAN,	{0}, FALSE},
	{OPTION_LCD_FIXONIGA1,    			"FixOnIGA1",			OPTV_BOOLEAN,	{0}, FALSE},
	{OPTION_LCD_CLOCK_POLARITY,			"ClockPolarity",		OPTV_INTEGER,	{0}, FALSE},
	{OPTION_LCD_CLOCK_ADJUST,			"ClockAdjust",			OPTV_INTEGER,	{0}, FALSE},
	{OPTION_LCD_CLOCK_DRIVING_SELECTION,"ClockDrivingSelection",OPTV_INTEGER,	{0}, FALSE},
	{OPTION_LCD_DATA_DRIVING_SELECTION,	"DataDrivingSelection", OPTV_INTEGER,	{0}, FALSE},
	{OPTION_LCD_VT1636_CLOCK_SEL_ST1, 	"Vt1636ClockSelST1", 	OPTV_INTEGER,	{0}, FALSE},
	{OPTION_LCD_VT1636_CLOCK_SEL_ST2, 	"Vt1636ClockSelST2", 	OPTV_INTEGER,	{0}, FALSE},
	{-1,                    			NULL,           		OPTV_NONE,      {0}, FALSE}	
};

static CARD32
transformDiPort(char *pStr)
{
    if (!xf86NameCmp(pStr, "DFP_HIGH"))
        return DISP_DI_DFPH;
    
    else if (!xf86NameCmp(pStr, "DFP_LOW"))
        return DISP_DI_DFPL;
    
    else if (!xf86NameCmp(pStr, "DFP_HIGHLOW"))
        return DISP_DI_DFP;
    
    else if (!xf86NameCmp(pStr, "DVP0"))
        return DISP_DI_DVP0;
    
    else if (!xf86NameCmp(pStr, "DVP1"))
        return DISP_DI_DVP1;
    
    else if (!xf86NameCmp(pStr, "DFP_LOW,DVP1")) /*Quanta TTL*/
        return DISP_DI_DFPL+DISP_DI_DVP1;

    else if (!xf86NameCmp(pStr, "DFP_HIGHLOW,DVP1")) /*409 TTL*/
        return DISP_DI_DFPH+DISP_DI_DFPL+DISP_DI_DVP1;
    
    else
        return DISP_DI_NONE;
    
}

static CARD32
transformOutputType(char *pStr)
{
    if (!xf86NameCmp(pStr, "HardWired"))
        return DISP_TYPE_HARDWIRED;
    
    else if (!xf86NameCmp(pStr, "External"))
        return DISP_TYPE_EXTERNAL;
    
	else if (!xf86NameCmp(pStr, "TTL"))
		return DISP_TYPE_TTL;
	
    else
        return DISP_TYPE_INTERNAL;
    
}

static CARD32
transformPort(CARD32 port)
{
    switch (port) {
        case DISP_SERIALP_25:
        case DISP_SERIALP_26:
        case DISP_SERIALP_2C:
        case DISP_SERIALP_31:
        case DISP_SERIALP_3D:
            return port;
        default:
            return DISP_DEFAULT_SETTING;
    }
}


static void
viaSetDefaultLCDPanelInfo(ViaLcdPrivateInfoPtr viaLcdInfo)
{
    switch(viaLcdInfo->panelIndex){
    case VIA_1280X1024:
    case VIA_1400X1050:
    case VIA_1440X900:
    case VIA_1600X1200:
        viaLcdInfo->dualChannel = TRUE;
        break;
    default:
        viaLcdInfo->dualChannel = FALSE;
        break;
    }
}


static void
parseCrtOption(xf86OutputPtr output)
{
	char *s = NULL;
	int sPort = DISP_SERIALP_NONE;
	int dPort = DISP_SERIALP_NONE;
	ViaCrtPrivateInfoPtr viaCrtInfo = output->driver_private;
    OptionInfoPtr tempRecOptionsPtr;

    /* Default value */
    viaCrtInfo->commonInfo.diPort = DISP_DEFAULT_SETTING;
    viaCrtInfo->commonInfo.type = DISP_DEFAULT_SETTING;
    viaCrtInfo->commonInfo.serialPort = DISP_DEFAULT_SETTING;
    viaCrtInfo->commonInfo.ddcPort = DISP_DEFAULT_SETTING;
    viaCrtInfo->commonInfo.NoDDCValue = FALSE;

    if (!(tempRecOptionsPtr= (OptionInfoPtr)xalloc(sizeof(ViaCrtOptions)))) {
        ErrorF("%s:Allocate memory Failed\n",__FUNCTION__);
        return;
    }
    memcpy(tempRecOptionsPtr, ViaCrtOptions, sizeof(ViaCrtOptions));
    /* if no "CRT" monitor section, all use default settting */
    if (output->conf_monitor)
	    xf86ProcessOptions (output->scrn->scrnIndex,
			        output->conf_monitor->mon_option_lst,
			        tempRecOptionsPtr);
    
    /* parse option "DIPort" */
    if ((s = xf86GetOptValString(tempRecOptionsPtr, OPTION_CRT_DIPORT)))
        viaCrtInfo->commonInfo.diPort = transformDiPort(s);        
    
    /* parse option "Type" */
    if ((s = xf86GetOptValString(tempRecOptionsPtr, OPTION_CRT_TYPE)))
        viaCrtInfo->commonInfo.type = transformOutputType(s);        
    
    /* parse option "SerialPort" */
    if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_CRT_SERIALPORT, &sPort))
        viaCrtInfo->commonInfo.serialPort = transformPort(sPort);
    
    /* parse option "DDCPort" */
    if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_CRT_DDCPORT, &dPort))
        viaCrtInfo->commonInfo.ddcPort = transformPort(dPort);
    
    /* parse option "NoDDCValue" */    
    if (xf86ReturnOptValBool(tempRecOptionsPtr, OPTION_CRT_NoDDCValue, FALSE))
        viaCrtInfo->commonInfo.NoDDCValue = TRUE;   
    

    xfree(tempRecOptionsPtr);
}

static void
parseDviOption(xf86OutputPtr output)
{
	char *s = NULL;
	int value = 0;
	int sPort = DISP_SERIALP_NONE;
	int dPort = DISP_SERIALP_NONE;
    OptionInfoPtr tempRecOptionsPtr;
	ViaDviPrivateInfoPtr viaDviInfo = output->driver_private;
	viaDviInfo->userGfxDPA.isClkPolarityUsed = FALSE;
	viaDviInfo->userGfxDPA.isClkAdjustUsed = FALSE;
	viaDviInfo->userGfxDPA.isClkDrivingSelUsed = FALSE;
	viaDviInfo->userGfxDPA.isDataDrivingSelUsed = FALSE;

    /* Default value */
    viaDviInfo->commonInfo.diPort = DISP_DEFAULT_SETTING;
    viaDviInfo->commonInfo.type = DISP_DEFAULT_SETTING;
    viaDviInfo->commonInfo.serialPort = DISP_DEFAULT_SETTING;
    viaDviInfo->commonInfo.ddcPort = DISP_DEFAULT_SETTING;
    viaDviInfo->commonInfo.NoDDCValue = FALSE;

    if (!(tempRecOptionsPtr= (OptionInfoPtr)xalloc(sizeof(ViaDviOptions)))) {
        ErrorF("%s:Allocate memory Failed\n",__FUNCTION__);
        return;
    }
    memcpy(tempRecOptionsPtr, ViaDviOptions, sizeof(ViaDviOptions));
    /* if no "DVI" monitor section, all use default setting */ 
    if (output->conf_monitor)
	    xf86ProcessOptions (output->scrn->scrnIndex,
		    	    output->conf_monitor->mon_option_lst,
			        tempRecOptionsPtr);
    

    /* parse option "DIPort" */
    if ((s = xf86GetOptValString(tempRecOptionsPtr, OPTION_DVI_DIPORT)))
        viaDviInfo->commonInfo.diPort = transformDiPort(s);        
    
    /* parse option "Type" */
    if ((s = xf86GetOptValString(tempRecOptionsPtr, OPTION_DVI_TYPE)))
        viaDviInfo->commonInfo.type = transformOutputType(s);        
    
    /* parse option "SerialPort" */
    if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_DVI_SERIALPORT, &sPort))
        viaDviInfo->commonInfo.serialPort = transformPort(sPort);
    
    /* parse option "DDCPort" */
    if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_DVI_DDCPORT, &dPort))
        viaDviInfo->commonInfo.ddcPort = transformPort(dPort);
    
    /* parse option  "NoDDCValue" */
    if (xf86ReturnOptValBool(tempRecOptionsPtr, OPTION_DVI_NoDDCValue, FALSE))
        viaDviInfo->commonInfo.NoDDCValue = TRUE;
    

	/* parse user clock skew setting*/
	/* 1. parse option "ClockPolarity" */
	if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_DVI_CLOCK_POLARITY,
        &value)) {
		viaDviInfo->userGfxDPA.clkPolarity = value&BIT0;
		viaDviInfo->userGfxDPA.isClkPolarityUsed = TRUE;
	}

	/* 2. parse option "ClockAdjust" */
	if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_DVI_CLOCK_ADJUST,
        &value))
	{
		viaDviInfo->userGfxDPA.clkAdjust = value;
		viaDviInfo->userGfxDPA.isClkAdjustUsed = TRUE;
	}

	/* 3. parse option "ClockDrivingSelection" */
	if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_DVI_CLOCK_DRIVING_SELECTION, 
        &value)) {
		viaDviInfo->userGfxDPA.clkDrivingSel = value;
		viaDviInfo->userGfxDPA.isClkDrivingSelUsed = TRUE;
	}

	/* 4. parse option "DataDrivingSelection" */
	if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_DVI_DATA_DRIVING_SELECTION,
        &value)) {
		viaDviInfo->userGfxDPA.dataDrivingSel = value;
		viaDviInfo->userGfxDPA.isDataDrivingSelUsed = TRUE;
	}

    xfree(tempRecOptionsPtr);
}


static void
parseLcdOption(xf86OutputPtr output)
{
	char *s = NULL, *b = NULL;
	int value = 0;
	int sPort = DISP_SERIALP_NONE;
	int dPort = DISP_SERIALP_NONE;
	unsigned int panelWidth = 0, panelHeight = 0;
    OptionInfoPtr tempRecOptionsPtr;
	ViaLcdPrivateInfoPtr viaLcdInfo = output->driver_private;
	viaLcdInfo->userGfxDPA.isClkPolarityUsed = FALSE;
	viaLcdInfo->userGfxDPA.isClkAdjustUsed = FALSE;
	viaLcdInfo->userGfxDPA.isClkDrivingSelUsed = FALSE;
	viaLcdInfo->userGfxDPA.isDataDrivingSelUsed = FALSE;
	viaLcdInfo->userLvdsDPA.isVt1636ClkSelST1Used = FALSE;
	viaLcdInfo->userLvdsDPA.isVt1636ClkSelST2Used = FALSE;
	
    /* Default value */
    viaLcdInfo->commonInfo.diPort = DISP_DEFAULT_SETTING;
    viaLcdInfo->commonInfo.type = DISP_DEFAULT_SETTING;
    viaLcdInfo->commonInfo.serialPort = DISP_DEFAULT_SETTING;
    viaLcdInfo->commonInfo.ddcPort = DISP_DEFAULT_SETTING;
	viaLcdInfo->commonInfo.physicalWidth = 0;
	viaLcdInfo->commonInfo.physicalHeight = 0;
	viaLcdInfo->panelIndex = 0;
	viaLcdInfo->dualChannel = FALSE;
	viaLcdInfo->noDithering = FALSE;
	viaLcdInfo->msb = FALSE;
	viaLcdInfo->center = FALSE;
	viaLcdInfo->fixOnIGA1= FALSE;

    if (!(tempRecOptionsPtr= (OptionInfoPtr)xalloc(sizeof(ViaLcdOptions)))) {
        ErrorF("%s:Allocate memory Failed\n",__FUNCTION__);
        return;
    }
    memcpy(tempRecOptionsPtr, ViaLcdOptions, sizeof(ViaLcdOptions));
	/* if no LCD monitor section, all use default setting */
	if (output->conf_monitor) {
		xf86ProcessOptions (output->scrn->scrnIndex,
				    output->conf_monitor->mon_option_lst,
				    tempRecOptionsPtr);
	}
    /* parse option "DIPort" */
    if ((s = xf86GetOptValString(tempRecOptionsPtr, OPTION_LCD_DIPORT)))
        viaLcdInfo->commonInfo.diPort = transformDiPort(s);  

    /* parse option "Type" */
    if ((s = xf86GetOptValString(tempRecOptionsPtr, OPTION_LCD_TYPE)))
        viaLcdInfo->commonInfo.type = transformOutputType(s);   
   
    /* parse option "SerialPort" */   
    if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_LCD_SERIALPORT, &sPort))
        viaLcdInfo->commonInfo.serialPort = transformPort(sPort);
    
    /* parse option "DDCPort" */
    if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_LCD_DDCPORT, &dPort))
        viaLcdInfo->commonInfo.ddcPort = transformPort(dPort);
    
    /* parse option  "PanelSize" */
    if ((s = xf86GetOptValString(tempRecOptionsPtr, OPTION_LCD_PANEL_SIZE))) {
    	b = strdup(s);
		s = strtok(b, "x");
        panelWidth = (CARD32)atoi(s);
		s = strtok(NULL, "x");
		panelHeight = (CARD32)atoi(s);
		/*Check if the panel size is valid*/
		if (isPanelSizeValid(panelWidth, panelHeight)) {
			viaLcdInfo->commonInfo.physicalWidth = panelWidth;
			viaLcdInfo->commonInfo.physicalHeight = panelHeight;
			viaLcdInfo->panelIndex = VIA_MAKE_ID(panelWidth, panelHeight);

            viaSetDefaultLCDPanelInfo(viaLcdInfo);
		} else 
			/*Give a chance to get physical size from EDID*/
			ErrorF("parseLcdOption: Panel size %dx%d is not valid!\n", 
        		panelWidth, panelHeight);	
    } 

    /* get panel modeline from xorg.conf for special panel size */
    viaGetPanelModeLine(output);

	/* parse option "DualChannel"*/
    if ((s = xf86GetOptValString(tempRecOptionsPtr, OPTION_LCD_DUAL_CHANNEL))) {
        if ((*s == '\0') ||
            !xf86NameCmp(s, "TRUE") ||
            !xf86NameCmp(s, "True") ||
            !xf86NameCmp(s, "true"))
            viaLcdInfo->dualChannel = TRUE;
        else if (!xf86NameCmp(s, "FALSE") ||
                 !xf86NameCmp(s, "False") ||
                 !xf86NameCmp(s, "false"))
            viaLcdInfo->dualChannel = FALSE;
    } 
    
	/* parse option "Dithering"*/
    if (xf86ReturnOptValBool(tempRecOptionsPtr, OPTION_LCD_NO_DITHERING, FALSE)) 
        viaLcdInfo->noDithering = TRUE;
     
	/* parse option "MSB"*/
    if (xf86ReturnOptValBool(tempRecOptionsPtr, OPTION_LCD_MSB, FALSE))
        viaLcdInfo->msb = TRUE;
     
	/* parse option "Center"*/
	if (xf86ReturnOptValBool(tempRecOptionsPtr, OPTION_LCD_CENTER, FALSE)) 
        viaLcdInfo->center= TRUE;
      
	/* parse option "FixOnIGA1"*/
	if (xf86ReturnOptValBool(tempRecOptionsPtr, OPTION_LCD_FIXONIGA1, FALSE)) 
        viaLcdInfo->fixOnIGA1 = TRUE;
    
	
	/* parse user clock skew setting*/
	/* 1. parse option "ClockPolarity" */
	if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_LCD_CLOCK_POLARITY,
        &value)) {
		viaLcdInfo->userGfxDPA.clkPolarity = value&BIT0;
		viaLcdInfo->userGfxDPA.isClkPolarityUsed = TRUE;
	}

	/* 2. parse option "ClockAdjust" */
	if (xf86GetOptValInteger(tempRecOptionsPtr, OPTION_LCD_CLOCK_ADJUST,
        &value)) {
		viaLcdInfo->userGfxDPA.clkAdjust = value;
		viaLcdInfo->userGfxDPA.isClkAdjustUsed = TRUE;
	}

	/* 3. parse option "ClockDrivingSelection" */
	if (xf86GetOptValInteger(tempRecOptionsPtr,
        OPTION_LCD_CLOCK_DRIVING_SELECTION, &value)) {
		viaLcdInfo->userGfxDPA.clkDrivingSel = value;
		viaLcdInfo->userGfxDPA.isClkDrivingSelUsed = TRUE;
	}

	/* 4. parse option "DataDrivingSelection" */
	if (xf86GetOptValInteger(tempRecOptionsPtr,
        OPTION_LCD_DATA_DRIVING_SELECTION, &value)) {
		viaLcdInfo->userGfxDPA.dataDrivingSel = value;
		viaLcdInfo->userGfxDPA.isDataDrivingSelUsed = TRUE;
	}

	/* 5. parse option "Vt1636ClockSelST1" */
	if (xf86GetOptValInteger(tempRecOptionsPtr,
        OPTION_LCD_VT1636_CLOCK_SEL_ST1, &value)) {
		viaLcdInfo->userLvdsDPA.Vt1636ClkSelST1 = value;
		viaLcdInfo->userLvdsDPA.isVt1636ClkSelST1Used = TRUE;
	}

	/* 6. parse option "Vt1636ClockSelST2" */
	if (xf86GetOptValInteger(tempRecOptionsPtr,
        OPTION_LCD_VT1636_CLOCK_SEL_ST2, &value)) {
		viaLcdInfo->userLvdsDPA.Vt1636ClkSelST2 = value;
		viaLcdInfo->userLvdsDPA.isVt1636ClkSelST2Used = TRUE;
	}

    xfree(tempRecOptionsPtr);
}   

/*
Function Name:  checkDiPortUsage
Description:    Check whether the DI port is already occupied or not
*/
static Bool
checkDiPortUsage(VIAPtr pVia, CARD32 diPort)
{
    if (diPort != DISP_DI_NONE) {
        if (!(diPort & pVia->MbDiPortUsedInfo))
            return TRUE;
    }
    return FALSE;    
}


/*
Function Name:  senseSubChip
Description:    Check if sub chip exist or not
Note: if serial port is set to default value,
it will be modified if the sub chip is found
*/
CARD32 
senseSubChip(CARD32* serialPort, CARD32 subChipName)
{
    subChipVerInfoPtr chip = NULL;
    CARD32 retVal = NONE_SUBCHIP;
    CARD32 i=0, j=0;
    CARD8  retVersion[5];

    switch (subChipName) {
        case SUBCHIP_VT1632:
            chip = &vt1632Info;
            break;
        case SUBCHIP_VT1636:
            chip = &vt1636Info;
            break;
    }

    if (chip) {
        /* if user doesn't set port or set wrong port,
           sense chip: 0x2c->0x31->0x25->0x3D->0x26 */
        if (*serialPort == DISP_DEFAULT_SETTING) {
            for (i=0; i<NUM_SERIAL_PORT; i++) {
                for (j=0; j<chip->numOfRevReg; j++) {
                    if (viaSerialReadByte(serialPortTable[i], chip->address,
                        chip->revOffset[j], &retVersion[j])) {
                        if (retVersion[j] != chip->revId[j])
                            break;
                        if (j == chip->numOfRevReg-1) {
                            *serialPort = serialPortTable[i];
                            retVal = chip->subChipName;
                            return retVal;
                        }
                    }
                }
            }
        } else {
        /* if user set port, just use user setting */
            for (j=0; j<chip->numOfRevReg; j++) {
                if (viaSerialReadByte(*serialPort, chip->address,
                    chip->revOffset[j], &retVersion[j])) {
                    if (retVersion[j] != chip->revId[j])
                        break;
                    if (j == chip->numOfRevReg-1) {
                        retVal = chip->subChipName;
                        return retVal;
                    }
                }
            }
        }
    }
    return retVal;
}

/*
Function Name:  checkCrtSupport
Description:    Check if CRT supported by platform
*/
static Bool
checkCrtSupport(VIAPtr pVia, xf86OutputPtr output)
{
    Bool retVal = FALSE;
    ViaCrtPrivateInfoPtr viaCrtInfo = output->driver_private;

    /* default CRT type is internal CRT */
    if (viaCrtInfo->commonInfo.type == DISP_DEFAULT_SETTING)
        viaCrtInfo->commonInfo.type = DISP_TYPE_INTERNAL;

    /* deault CRT DIPort */
    if (viaCrtInfo->commonInfo.diPort == DISP_DEFAULT_SETTING) {
        if (viaCrtInfo->commonInfo.type == DISP_TYPE_INTERNAL)
            /* for internal CRT: DAC */
            viaCrtInfo->commonInfo.diPort = DISP_DI_DAC;
        else if (viaCrtInfo->commonInfo.type == DISP_TYPE_EXTERNAL) 
            /* for external CRT: DVP1 */
            viaCrtInfo->commonInfo.diPort = DISP_DI_DVP1;
        else 
            /* for hard-wired CRT, reserved */
            viaCrtInfo->commonInfo.diPort = DISP_DI_NONE;
    }

    /* check if CRT DI port is free */
    if (checkDiPortUsage(pVia, viaCrtInfo->commonInfo.diPort)) {
        /* for internal CRT: no serial port */
        if (viaCrtInfo->commonInfo.type == DISP_TYPE_INTERNAL) {
            viaCrtInfo->commonInfo.diPort = DISP_DI_DAC;
            viaCrtInfo->commonInfo.serialPort = DISP_SERIALP_NONE;

            /* if use DVI-VGA switch, ddc port may not be 0x26, 
               so user-define ddc port can override it */
            if (viaCrtInfo->commonInfo.ddcPort == DISP_DEFAULT_SETTING)
                viaCrtInfo->commonInfo.ddcPort = DISP_SERIALP_26;

            viaCrtInfo->commonInfo.subChipName = NONE_SUBCHIP;
            viaCrtInfo->commonInfo.slaveAddress = NONE_SUBCHIP_SLAVE_ADDR;
            pVia->MbDiPortUsedInfo |= viaCrtInfo->commonInfo.diPort;
            retVal = TRUE;
        } else { 
        }
        /* for hard-wired CRT, reserved */
    }
    
    return retVal;
}

/*
Function Name:  checkDviSupport
Description:    Check if DVI supported by platform
*/
static Bool
checkDviSupport(VIAPtr pVia, xf86OutputPtr output)
{
    Bool retVal = FALSE;
    ViaDviPrivateInfoPtr viaDviInfo = output->driver_private;

    /* default DVI type is internal */
    if (viaDviInfo->commonInfo.type == DISP_DEFAULT_SETTING) {
        if (pVia->GfxDispCaps & INTERNAL_TMDS)
            /* if the chip have the cap of integrated TMDS */
            viaDviInfo->commonInfo.type = DISP_TYPE_INTERNAL; 
        else 
            /* if the chip doesn't have, return FALSE directly */
            return retVal;
      
    }

    /* deault DVI DIPort */
    if (viaDviInfo->commonInfo.diPort == DISP_DEFAULT_SETTING) {
        if (viaDviInfo->commonInfo.type == DISP_TYPE_INTERNAL)
            /* for internal TMDS: DFP_LOW */
            viaDviInfo->commonInfo.diPort = DISP_DI_DFPL;
        else 
            /* for external and hard-wired TMDS: use DVP1 as default port */
            viaDviInfo->commonInfo.diPort = DISP_DI_DVP1;
        
    }

    /*if DVI DI port is free*/
    if (checkDiPortUsage(pVia, viaDviInfo->commonInfo.diPort)) {
        /* internal TMDS */
        if (viaDviInfo->commonInfo.type == DISP_TYPE_INTERNAL) {
            /* if the platform doesn't support internal TMDS, it fails  */
            if (INTERNAL_TMDS & pVia->GfxDispCaps) {
                viaDviInfo->commonInfo.subChipName = SUBCHIP_INTEGRATED_TMDS;
                viaDviInfo->commonInfo.slaveAddress = NONE_SUBCHIP_SLAVE_ADDR;
                /* default serial port (internal TMDS), NONE */
                if (viaDviInfo->commonInfo.serialPort == DISP_DEFAULT_SETTING)
                    viaDviInfo->commonInfo.serialPort = DISP_SERIALP_NONE;
                /* default DDC port (internal TMDS): 0x31 */
                if (viaDviInfo->commonInfo.ddcPort == DISP_DEFAULT_SETTING) {
                    if (pVia->Chipset == VIA_VX800) {
                        viaDviInfo->commonInfo.ddcPort = (VIAGetRevisionOfVX800() == REVISION_VX800_B2) ? DISP_SERIALP_31 : DISP_SERIALP_3D;
                    } else {
                        viaDviInfo->commonInfo.ddcPort = DISP_SERIALP_31;
                    }
                }
                /* DI port is occupied, and internal TMDS cap is lost */            
                pVia->MbDiPortUsedInfo |= viaDviInfo->commonInfo.diPort;
                pVia->GfxDispCaps &= ~INTERNAL_TMDS;
                retVal = TRUE;
            }
        /* external TMDS */    
        } else if (viaDviInfo->commonInfo.type == DISP_TYPE_EXTERNAL) {
            /*Case that graphic chip before 3324:
                    If user set DVI use DVP0, but found SR12[5]=1(jumper on MB set 
                    TV use DVP0), DVI support will fail*/
            if (DISP_DI_DVP0 == viaDviInfo->commonInfo.diPort &&
                !(DISP_DEV_DVI & pVia->MbDvp0Device)) {
                pVia->MbDiPortUsedInfo &= ~DISP_DI_DVP0;
            } else {
                /* Sense VT1632*/
                /* VT1632A needn't the clk on when sonsored through I2C,but VT1632 needs.
                           Turn on relevant clk according to diport used by vt1632. */
                viaDIPortPadOn(viaDviInfo->commonInfo.diPort);
                if (senseSubChip(&viaDviInfo->commonInfo.serialPort,
                    SUBCHIP_VT1632)) {
                    viaDviInfo->commonInfo.subChipName = SUBCHIP_VT1632;
                    viaDviInfo->commonInfo.slaveAddress =
                                SUBCHIP_VT1632_SLAVE_ADDR;
                    if (viaDviInfo->commonInfo.ddcPort ==
                                DISP_DEFAULT_SETTING)
                        viaDviInfo->commonInfo.ddcPort = 
                                viaDviInfo->commonInfo.serialPort;
                    pVia->MbDiPortUsedInfo |= viaDviInfo->commonInfo.diPort;
                    retVal = TRUE;
                }
                /*Turn off relevant di port clk set on previously to keep things consistent */
                viaDIPortPadOff(viaDviInfo->commonInfo.diPort);
            }

        } else {
            viaDviInfo->commonInfo.subChipName = SUBCHIP_HARDWIRED_TMDS;
            viaDviInfo->commonInfo.slaveAddress = NONE_SUBCHIP_SLAVE_ADDR;
            viaDviInfo->commonInfo.serialPort = DISP_SERIALP_NONE;
            viaDviInfo->commonInfo.ddcPort = DISP_SERIALP_NONE;
            pVia->MbDiPortUsedInfo |= viaDviInfo->commonInfo.diPort;
            retVal = TRUE;
        }
   }
   return retVal;
}



Bool
checkLcdSupport(VIAPtr pVia, xf86OutputPtr output)
{
    Bool retVal = FALSE;
    ViaLcdPrivateInfoPtr viaLcdInfo = output->driver_private;
	ViaOutputInfoPtr LcdCommonInfo = &viaLcdInfo->commonInfo;

	/*For default type*/
    if (LcdCommonInfo->type == DISP_DEFAULT_SETTING) {
    	if (pVia->GfxDispCaps & INTERNAL_LVDS)
    	    /* if Gfx have integrated LVDS capability, default as integrated LVDS */
			LcdCommonInfo->type = DISP_TYPE_INTERNAL;
    	else  
    		/* if Gfx have no integrated LVDS capability, return FALSE directly */
    		return retVal;
    	
    }
    /*For default DI port*/
    if(LcdCommonInfo->diPort == DISP_DEFAULT_SETTING) {
        if (LcdCommonInfo->type == DISP_TYPE_INTERNAL)
            /* Internal LVDS use DFP_HIGH as default DIPort */
            LcdCommonInfo->diPort = DISP_DI_DFPH;
        else if (LcdCommonInfo->type == DISP_TYPE_TTL) {
            if (pVia->Chipset == VIA_VX855) {
                LcdCommonInfo->diPort = DISP_DI_DFPH|DISP_DI_DFPL|DISP_DI_DVP1;                
            } else {
                LcdCommonInfo->diPort = DISP_DI_DFPL | DISP_DI_DVP1;
            }
        } else 
            /*External and Hardwired use DVP1 as default DI port,
            actually we recommend user set DI port on these cases!*/
            LcdCommonInfo->diPort = DISP_DI_DVP1;
        
    }
 
    /*If LCD DI port is free*/
    if (checkDiPortUsage(pVia, LcdCommonInfo->diPort)) {
        /*If user use internal LVDS, but found platform don't support 
             internal LVDS or DI port set wrong, it will fail*/
        if (LcdCommonInfo->type == DISP_TYPE_INTERNAL) {
            if((INTERNAL_LVDS & pVia->GfxDispCaps) &&
                ((DISP_DI_DFPL | DISP_DI_DFPH) & LcdCommonInfo->diPort)) {
            	LcdCommonInfo->subChipName = SUBCHIP_INTEGRATED_LVDS;
                /* default serial port (internal LVDS), NONE */
                if (LcdCommonInfo->serialPort == DISP_DEFAULT_SETTING) {
                    LcdCommonInfo->serialPort = DISP_SERIALP_NONE;
                }
                /* default DDC port (internal LVDS): 0x31.expect 409,409 DDC port is 2c */
                if (LcdCommonInfo->ddcPort == DISP_DEFAULT_SETTING) {
                    if ( pVia->Chipset == VIA_VX855 )
                        LcdCommonInfo->ddcPort = DISP_SERIALP_2C;
                    else
                        LcdCommonInfo->ddcPort = DISP_SERIALP_31;
                }
				LcdCommonInfo->slaveAddress = NONE_SUBCHIP_SLAVE_ADDR;
                /* DI port is occupied*/
                if (viaLcdInfo->dualChannel) 
                    pVia->MbDiPortUsedInfo |= (DISP_DI_DFPL | DISP_DI_DFPH);
                else
                    pVia->MbDiPortUsedInfo |= LcdCommonInfo->diPort;
                retVal = TRUE;
            }
        } else if (LcdCommonInfo->type == DISP_TYPE_EXTERNAL) {
            /*3324, 3353, 3293 don't need to care SR12(MB strapping) 
                    so MbDvp0Device will be 0, but other chip need.*/
            if ((DISP_DI_DVP0 == LcdCommonInfo->diPort) &&
                !(DISP_DEV_LCD&pVia->MbDvp0Device)) {
                pVia->MbDiPortUsedInfo &= ~DISP_DI_DVP0;
                retVal = FALSE;
            } else {
                /* Sense VT1636*/
				if (senseSubChip(&LcdCommonInfo->serialPort, SUBCHIP_VT1636)) {
					LcdCommonInfo->subChipName = SUBCHIP_VT1636;
					LcdCommonInfo->slaveAddress = SUBCHIP_VT1636_SLAVE_ADDR;
                    if (LcdCommonInfo->ddcPort == DISP_DEFAULT_SETTING) {   
                        LcdCommonInfo->ddcPort = LcdCommonInfo->serialPort;
                    }    
                    pVia->MbDiPortUsedInfo |= LcdCommonInfo->diPort;
                    retVal = TRUE;
				}
            }
		} else if (LcdCommonInfo->type == DISP_TYPE_TTL){
			LcdCommonInfo->subChipName = SUBCHIP_TTL;
			LcdCommonInfo->slaveAddress = NONE_SUBCHIP_SLAVE_ADDR;
			/* default serial port NONE */
			LcdCommonInfo->serialPort = DISP_SERIALP_NONE;
			/* default DDC port (internal LVDS): 0x31 */
			if (LcdCommonInfo->ddcPort == DISP_DEFAULT_SETTING)
				LcdCommonInfo->ddcPort = DISP_SERIALP_31;
			
			pVia->MbDiPortUsedInfo |= LcdCommonInfo->diPort;
			retVal = TRUE;
        } else { 
	        /*Hardwried*/
			LcdCommonInfo->subChipName = SUBCHIP_HARDWIRED_LVDS;
			LcdCommonInfo->slaveAddress = NONE_SUBCHIP_SLAVE_ADDR;
        	/* No need to program subchip*/
           	LcdCommonInfo->serialPort = DISP_SERIALP_NONE;
			/* Default DDC port is 0x31 */
			if (LcdCommonInfo->ddcPort == DISP_DEFAULT_SETTING)
				LcdCommonInfo->ddcPort = DISP_SERIALP_31;
			
			pVia->MbDiPortUsedInfo |= LcdCommonInfo->diPort;
			retVal = TRUE;
        }
    } else {
		/*if LCD DI port is occupied or unsupported*/
        retVal = FALSE;
    }
    return retVal;
}

void
via_crt_init(ScrnInfoPtr pScrn, const char* name)
{
    VIAPtr                  pVia = VIAPTR(pScrn);
    xf86OutputPtr           outputCrt;
    ViaCrtPrivateInfoPtr    viaCrtInfo;   

    outputCrt = xf86OutputCreate (pScrn, &via_crt_output_funcs, name);       
    if (!outputCrt) {
        DEBUG(ErrorF("xf86OutputCreate %s Fail.\n", name));
        return;        
    }
    viaCrtInfo = xnfcalloc (sizeof (ViaCrtPrivateInfo), 1);
    if (!viaCrtInfo) {
        xf86OutputDestroy(outputCrt);
        DEBUG(ErrorF("Allocate %s private info Fail.\n", name));
        return; 
    }
    outputCrt->driver_private = viaCrtInfo;   
    
    /* Read CRT options in CRT monitor section */
    parseCrtOption(outputCrt);

    if (checkCrtSupport(pVia, outputCrt)) {
        outputCrt->possible_crtcs = 0x1 | 0x2;
        outputCrt->possible_clones = 0;
        outputCrt->interlaceAllowed = TRUE;
        outputCrt->doubleScanAllowed = FALSE;

        if (!strcmp(name, OUTPUT_CRT_NAME))
            pVia->crt1Created = TRUE;         
    }
    else
        /* also free crt private date */
        xf86OutputDestroy(outputCrt);
       
}

void
via_dvi_init(ScrnInfoPtr pScrn, const char* name)
{
    VIAPtr                  pVia = VIAPTR(pScrn);
    xf86OutputPtr           outputDvi;
    ViaDviPrivateInfoPtr    viaDviInfo;   
    outputDvi = xf86OutputCreate (pScrn, &via_tmds_output_funcs, name);       
    if (!outputDvi) {
        DEBUG(ErrorF("xf86OutputCreate %s Fail.\n", name));
        return;        
    }
    viaDviInfo = xnfcalloc (sizeof (ViaDviPrivateInfo), 1);
    if (!viaDviInfo) {
        xf86OutputDestroy(outputDvi);
        DEBUG(ErrorF("Allocate %s private info Fail.\n", name));
        return; 
    }
    outputDvi->driver_private = viaDviInfo;   
    
    /* Read Dvi options in DVI monitor section */
    parseDviOption(outputDvi);

    if (checkDviSupport(pVia, outputDvi)) {
        if (viaDviInfo->commonInfo.subChipName == SUBCHIP_INTEGRATED_TMDS)
            pVia->numOfEmbDvi++;
    
        outputDvi->possible_crtcs = 0x1 | 0x2;
        outputDvi->possible_clones = 0;
        outputDvi->interlaceAllowed = TRUE;
        outputDvi->doubleScanAllowed = FALSE;

        if (!strcmp(name, OUTPUT_DVI_NAME))
            pVia->dvi1Created = TRUE;
    } else
        /* also free DVI private date */
        xf86OutputDestroy(outputDvi);
    
}


void
via_lcd_init(ScrnInfoPtr pScrn, const char* name)
{
    VIAPtr                  pVia = VIAPTR(pScrn);
    xf86OutputPtr           outputLcd;
    ViaLcdPrivateInfoPtr    viaLcdInfo;   
    outputLcd = xf86OutputCreate (pScrn, &via_lvds_output_funcs, name);       
    if (!outputLcd) {
        DEBUG(ErrorF("xf86OutputCreate %s Fail.\n", name));
        return;        
    }
    viaLcdInfo = xnfcalloc (sizeof (ViaLcdPrivateInfo), 1);
    if (!viaLcdInfo) {
        xf86OutputDestroy(outputLcd);
        DEBUG(ErrorF("Allocate %s private info Fail.\n", name));
        return; 
    }
    outputLcd->driver_private = viaLcdInfo;   
    
    /* Read Lcd options in LCD monitor section */
    parseLcdOption(outputLcd);

    if (checkLcdSupport(pVia, outputLcd)) {
		if (viaLcdInfo->commonInfo.subChipName == SUBCHIP_INTEGRATED_LVDS) {
			pVia->numOfEmbLcd++;
            if (viaLcdInfo->dualChannel)
			    pVia->dualChannelEmbLcdExist = TRUE;
		}	
    
		if (viaLcdInfo->fixOnIGA1)
			outputLcd->possible_crtcs = 0x01;
		else
			outputLcd->possible_crtcs = 0x02;
		
        outputLcd->possible_clones = 0;
        outputLcd->interlaceAllowed = TRUE;
        outputLcd->doubleScanAllowed = FALSE;

        if (!strcmp(name, OUTPUT_LCD_NAME))
            pVia->lcd1Created = TRUE;
    } else
       /* also free LCD private date */
        xf86OutputDestroy(outputLcd);
         
}

void 
via_output_init(ScrnInfoPtr pScrn)
{
    VIAPtr                  pVia = VIAPTR(pScrn);
    /* init value */
    pVia->numOfEmbLcd = 0;
    pVia->numOfEmbDvi = 0;
    pVia->dualChannelEmbLcdExist = FALSE;
    pVia->ddcPortOfDvi = 0;

    pVia->crt1Created = FALSE;
    pVia->lcd1Created = FALSE;
    pVia->dvi1Created = FALSE;

    /* CRT1 */
    via_crt_init(pScrn, OUTPUT_CRT_NAME);

    /* DVI1 */
    via_dvi_init(pScrn, OUTPUT_DVI_NAME);

    /* LCD1*/
    via_lcd_init(pScrn, OUTPUT_LCD_NAME);

    /* LCD2*/
    if (pVia->lcd1Created)
        via_lcd_init(pScrn, OUTPUT_LCD2_NAME);

    /* DVI2 */
    if (pVia->dvi1Created)
        via_dvi_init(pScrn, OUTPUT_DVI2_NAME);

}

#endif /*VIA_RANDR12_SUPPORT*/


