/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#include "via_driver.h"
#include "via_regs.h"
#include "via_rotate.h"

/*
*   Get reciprocal of source pitch. ex: 1/sp = 0.x, if sp=256d, x=004000h
*/ 
static unsigned int 
GetReciprocal(int dwSourcePitch)
{    
    if(dwSourcePitch != 0) {	    
    	return (1 << 22)/ dwSourcePitch;
    }
    else {
        return 0;	/*Error Flag,  FIXME*/
    }
}

/*
*   Enable HW CBU to rotate the screen.
*/
void 
EnableCBUrotate(ScrnInfoPtr pScrn)
{    
    VIAPtr  pVia = VIAPTR(pScrn);    
    int    dwRotDegree = pVia->RotateDegree;    
	/*Need to set the vaule before HW Rotate*/
    CARD32 dwWidth = pScrn->virtualX; 	 
    CARD32 dwHeight = pScrn->virtualY;
    CARD32 dwWidthAlign8 = 0;
    
    CARD32 dwRotCtl = 0;                     /* Degree, Bpp, Enable      */
    CARD32 dwStartAddr = pScrn->fbOffset;    /* start address            */
    CARD32 dwEndAddr = 0;                    /* end address              */
    CARD32 dwSrcPit = 0;                     /* source pitch             */
    CARD32 dwRotPit = 0;                     /* rotate pitch             */
    CARD32 dwRotWH = 0;                      /* W & H                    */    

    /* Rotate Window Register Interlace = 0x20 */
    int bankOffset = pScrn->scrnIndex * 0x20;     

    /* To do alignment, like 1366x768, 1400x1050... */
    dwWidthAlign8 = (dwWidth + 7) & (~7);     

    /*  Need to be fixed.
    *   when we rotate 1400x1050 8bpp mode into 90 or 270 degree. 
    *   The screen is fault because 1050 is not a correct value for rotate pitch.
    *   After my experiment, I found that 1052 is a suitable value.
    *   I can't explain it but I still apply this value to the rotate pitch.
    *   So this issue need to be solved in the future.
    */
    if ( (dwWidth == 1400) && (dwHeight == 1050) &&
        (pScrn->bitsPerPixel == 8) &&
        ((dwRotDegree == VIA_ROTATE_DEGREE_90) || 
		 (dwRotDegree == VIA_ROTATE_DEGREE_270)) ) {
        dwHeight += 2;
    }

    /* Set Degree */
    switch (dwRotDegree) {
    case VIA_ROTATE_DEGREE_90:
         dwRotCtl |= 0x8;
         break;
            
    case VIA_ROTATE_DEGREE_180:
         dwRotCtl |= 0x10;
         break;
                    
    case VIA_ROTATE_DEGREE_270:
         dwRotCtl |= 0x18;
         break;
    
    defalut:
         break;
    }

    /* Set Bpp */
    switch (pScrn->bitsPerPixel) {
    case 8:
         dwRotCtl |= 0x0;
         break;
            
    case 16:
         dwRotCtl |= 0x2;
         break;
                    
    case 32:
         dwRotCtl |= 0x4;
         break;

    default:
	 break;
    }

    /* Set Rotate End Address */
    dwEndAddr = dwStartAddr + (dwWidthAlign8 * dwHeight * (pScrn->bitsPerPixel/8));
            
    /* Rotate Source Pitch */
    /* Rotate Pitch */
    /* Rotate Height and Width */
    switch (dwRotDegree) {
    case VIA_ROTATE_DEGREE_90:
    case VIA_ROTATE_DEGREE_270:
         dwSrcPit = GetReciprocal(dwHeight); 
         dwRotPit = (dwWidthAlign8 << 16) | dwHeight; 
         dwRotWH = (dwWidthAlign8 << 16) | dwHeight;
         break;

    case VIA_ROTATE_DEGREE_0:
    case VIA_ROTATE_DEGREE_180:
         dwSrcPit = GetReciprocal(dwWidthAlign8);
         dwRotPit = (dwWidthAlign8 << 16) | dwWidthAlign8;
         dwRotWH = (dwHeight << 16) | dwWidthAlign8;
         break;
    
    default:
         break;
	}        
    
    /* Set Registers */       
    VIASETREG(0x1E04 + bankOffset, dwStartAddr);
    VIASETREG(0x1E08 + bankOffset, dwEndAddr);
    VIASETREG(0x1E0C + bankOffset, dwSrcPit);
    VIASETREG(0x1E10 + bankOffset, dwRotPit);
    VIASETREG(0x1E14 + bankOffset, dwRotWH);        
    VIASETREG(0x1E00 + bankOffset, dwRotCtl | 0x1); /* Fire */
}


void 
DisableCBUrotate(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn); 
	
    /* Rotate Window Register Interlace = 0x20 */
    int bankOffset = pScrn->scrnIndex * 0x20;        
    
    /* Clear */       
    VIASETREG(0x1E04 + bankOffset, 0);
    VIASETREG(0x1E08 + bankOffset, 0);
    VIASETREG(0x1E0C + bankOffset, 0);
    VIASETREG(0x1E10 + bankOffset, 0);
    VIASETREG(0x1E14 + bankOffset, 0);        
    VIASETREG(0x1E00 + bankOffset, 0);
}


/*
*   Enable 2D Engine rotate function to acceleration 2D function.  
*   Just for VX700 series, VX800 series don't use it
*/
void 
Enable2DEngineRotate(ScrnInfoPtr pScrn, CARD32* pGEMode, CARD32* pCommand)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    CARD32  dwWidth = 0;
    CARD32  dwHeight = 0;
    CARD32  RotateX = 0, RotateY = 0, RotateDegree = 0; /* For rotate used. */
    
    /* we need to get the actual screen size but not the mode we set */
    dwWidth = pScrn->virtualX;
    dwHeight = pScrn->virtualY;
    /*For XV set Clipping Windows, Rotate for 90/270 degree, virtualX/virtualY is swapped*/
    if ((pVia->RotateDegree == VIA_ROTATE_DEGREE_90) ||
        (pVia->RotateDegree == VIA_ROTATE_DEGREE_270)) {
        dwWidth = pScrn->virtualY;
        dwHeight = pScrn->virtualX;
    }
     
    dwWidth = (dwWidth + 7) & ~7;

    /*  Need to be fixed.
    *   when we rotate 1400x1050 8bpp mode into 90 or 270 degree. 
    *   The screen is fault because 1050 is not a correct value for rotate pitch.
    *   After my experiment, I found that 1052 is a suitable value.
    *   I can't explain it but I still apply this value to the rotate pitch.
    *   So this issue need to be solved in the future.
    */
    if ((dwWidth == 1400) && (dwHeight == 1050) &&
        (pScrn->bitsPerPixel == 8) &&
        ((pVia->RotateDegree == VIA_ROTATE_DEGREE_90) ||
         (pVia->RotateDegree == VIA_ROTATE_DEGREE_270))) {
        dwHeight += 2;
    }
    
    switch (pVia->RotateDegree) {
    case VIA_ROTATE_DEGREE_90:
         RotateX = dwHeight;
         RotateY = dwWidth;
         RotateDegree = VIA_GEM_ROTATE_90;            
         break;
            
    case VIA_ROTATE_DEGREE_180:
         RotateX = dwWidth;
         RotateY = dwHeight;
         RotateDegree = VIA_GEM_ROTATE_180;
         break;
            
    case VIA_ROTATE_DEGREE_270:
         RotateX = dwHeight;
         RotateY = dwWidth;
         RotateDegree = VIA_GEM_ROTATE_270;            
         break;

    default:
         break;
    }

    /* Modify GEMode and Command according to rotate degree. */
    *pGEMode |= RotateDegree;
    *pCommand |= VIA_GEC_ROT;

    /* Set resolution of rotation surface. */
    VIASETREG(VIA_REG_ROTSRC, (((RotateY-1)<<16))|(RotateX-1) );
    VIASETREG(VIA_REG_ROTDST, (((RotateY-1)<<16))|(RotateX-1) );        
}


/*
*   Only some chipsets have direct access window.
*/
Bool 
IsSupportDirectAccessWindow(int ChipID)
{
    int IsSupport = FALSE;
    
    switch (ChipID) {
    case VIA_CX700:
    case VIA_P4M890:
    case VIA_P4M900:
    case VIA_VX800:
    case VIA_VX855:
         IsSupport = TRUE;
         break;

    default:
         break;
    }

    return IsSupport;
}


/*
*   Only some chipsets have 2D rotation engine.
*/
Bool 
IsSupport2DRotateEngine(int ChipID)
{
    int IsSupport = FALSE;
    
    switch (ChipID) {
    case VIA_CX700:    
    case VIA_VX800:
    case VIA_VX855:
         IsSupport = TRUE;
        break;

    default:
        break;
    }

    return IsSupport;
}


/*
*   If the chipset doesn't have the capabilities to rotate, we have to modify the flag.
*   Note: This function must be called before 2D acceleration initialization.
*/
void 
ProbeRotateCaps(ScrnInfoPtr pScrn)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    VIABIOSInfoPtr  pBIOSInfo;
    pBIOSInfo = pVia->pBIOSInfo;    
    
    if (pVia->IsHWRotateEnabled) {
        /* Hardware rotation need direct access window, 
		 * otherwise we can't rotate the screen. */
        if (!IsSupportDirectAccessWindow(pBIOSInfo->Chipset)) {
            pVia->IsHWRotateEnabled = FALSE;
        }

        /* If the chipset doesn't support rotate engine, we 
		 * can't turn on 2D acceleration. */
        if (!IsSupport2DRotateEngine(pBIOSInfo->Chipset)) {
            pVia->NoAccel = TRUE;
        }
    }
}

/*
*   Dynamic rotate screen
*/
void 
EnableDynamicRotate(ScrnInfoPtr pScrn)
{    
    VIAPtr  pVia = VIAPTR(pScrn);    
    int    dwRotDegree = pVia->RotateDegree;    
    CARD32 dwWidth = pScrn->virtualX;    /* This value is before rotate */  
    CARD32 dwHeight = pScrn->virtualY;   /* This value is before rotate */
    /*For XV set Clipping Windows, Rotate for 90/270 degree, virtualX/virtualY is swapped*/
    if (( dwRotDegree == VIA_ROTATE_DEGREE_90) ||
        ( dwRotDegree == VIA_ROTATE_DEGREE_270)) {
        dwWidth = pScrn->virtualY;
        dwHeight = pScrn->virtualX;
    }
    CARD32 dwWidthAlign8 = 0;
    CARD32 dwWidthAlign32 = 0;
    
    CARD32 dwRotCtl = 0;                     /* Degree, Bpp, Enable      */
    CARD32 dwStartAddr = pScrn->fbOffset;    /* start address            */
    CARD32 dwEndAddr = 0;                    /* end address              */
    CARD32 dwSrcPit = 0;                     /* source pitch             */
    CARD32 dwRotPit = 0;                     /* rotate pitch             */
    CARD32 dwRotWH = 0;                      /* W & H                    */    

    /* Rotate Window Register Interlace: 0x20 */
    int bankOffset = pScrn->scrnIndex * 0x20;    

    if (dwRotDegree == VIA_ROTATE_DEGREE_0) {        
        DisableCBUrotate(pScrn);
        pVia->IsHWRotateEnabled = FALSE;
        return;
    }
    
    /* To do alignment, like 1366x768, 1400x1050... */
    dwWidthAlign32 = (dwWidth + 31) & (~31);        
    dwWidthAlign8 = (dwWidth + 7) & (~7);             

    /*  Need to be fixed.
    *   when we rotate 1400x1050 8bpp mode into 90 or 270 degree. 
    *   The screen is fault because 1050 is not a correct value for rotate pitch.
    *   After my experiment, I found that 1052 is a suitable value.
    *   I can't explain it but I still apply this value to the rotate pitch.
    *   So this issue need to be solved in the future.
    */
    if ((dwWidth == 1400) && (dwHeight == 1050) &&
        (pScrn->bitsPerPixel == 8) &&
        ((dwRotDegree == VIA_ROTATE_DEGREE_90) || 
		 (dwRotDegree == VIA_ROTATE_DEGREE_270))) {
        dwHeight += 2;
    }
    
    /* Set Degree */
    switch (dwRotDegree) {
    case VIA_ROTATE_DEGREE_90:
         dwRotCtl |= 0x8;
         break;
            
    case VIA_ROTATE_DEGREE_180:
         dwRotCtl |= 0x10;
         break;
                    
    case VIA_ROTATE_DEGREE_270:
         dwRotCtl |= 0x18;
         break;

    defalut:
         break;
    }

    /* Set Bpp */
    switch (pScrn->bitsPerPixel) {
    case 8:
         dwRotCtl |= 0x0;
         break;
         
    case 16:
         dwRotCtl |= 0x2;
         break;
                    
    case 32:
         dwRotCtl |= 0x4;
         break;

    default:
	    break;
    }    
                        
    switch (dwRotDegree) {
    case VIA_ROTATE_DEGREE_90:
         /* Set Rotate End Address */
         dwEndAddr = dwStartAddr + (dwWidthAlign32 * dwWidthAlign32 * (pScrn->bitsPerPixel/8));
         /* Rotate Source Pitch */
         dwSrcPit = GetReciprocal(dwWidth); 
         /* Rotate Pitch */
         dwRotPit = (dwWidthAlign32 << 16) | dwWidth; 
         /* Rotate Height and Width */
         dwRotWH = (dwWidthAlign8 << 16) | dwWidthAlign8;
         break;
            
    case VIA_ROTATE_DEGREE_270:            
         /* Set Rotate End Address */
         dwEndAddr = dwStartAddr + (dwWidthAlign32 * dwWidthAlign32 * (pScrn->bitsPerPixel/8));
         /* Rotate Source Pitch */
         dwSrcPit = GetReciprocal(dwWidthAlign32); 
         /* Rotate Pitch */
         dwRotPit = (dwWidthAlign32 << 16) | dwWidthAlign32; 
         /* Rotate Height and Width */
         dwRotWH = (dwWidthAlign8 << 16) | dwHeight;
         break;

    case VIA_ROTATE_DEGREE_0:
    case VIA_ROTATE_DEGREE_180:
         /* Set Rotate End Address */
         dwEndAddr = dwStartAddr + (dwWidthAlign32 * dwHeight * (pScrn->bitsPerPixel/8));
         dwSrcPit = GetReciprocal(dwWidthAlign32);
         dwRotPit = (dwWidthAlign32 << 16) | dwWidthAlign32;
         dwRotWH = (dwHeight << 16) | dwWidthAlign8;
         break;        

    default:
         break;
    }        
    
    /* Set Registers */       
    VIASETREG(0x1E04 + bankOffset, dwStartAddr);
    VIASETREG(0x1E08 + bankOffset, dwEndAddr);
    VIASETREG(0x1E0C + bankOffset, dwSrcPit);
    VIASETREG(0x1E10 + bankOffset, dwRotPit);
    VIASETREG(0x1E14 + bankOffset, dwRotWH);        
    VIASETREG(0x1E00 + bankOffset, dwRotCtl | 0x1); /* Fire */
}


/*
 *  Following Functions are for VX800/VX855 CBU/HW rotate usage, 
 *     which using M1 engine: 2D GE cmd/mode DON'T need 
 *   			set rotate degree and rotate cmd!!!
 * */

void 
DoConvertCoordinates(int *pX, int *pY, int maxX, int maxY, int rot_degree)
{
    int newX, newY;

    switch(rot_degree) {
    case VIA_ROTATE_DEGREE_90:
         newX = maxX - *pY;
         newY = *pX;
         break;

    case VIA_ROTATE_DEGREE_180:
         newX = maxX - *pX;
         newY = maxY - *pY;
         break;

    case VIA_ROTATE_DEGREE_270:
         newX = *pY;
         newY = maxY - *pX;
         break;

    default:			/* VIA_ROTATE_DEGREE_0 */
         newX = *pX;
         newY = *pY;
         break;
    }
	
    *pX = newX;
    *pY = newY;
}


void 
ConvertCoordinatesOne(ScrnInfoPtr pScrn, int* x1, int* y1, int w, int h)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    int dwWidth, dwHeight;
    
    dwWidth = pScrn->virtualX;
    dwHeight = pScrn->virtualY;
    /*For XV set Clipping Windows, Rotate for 90/270 degree, virtualX/virtualY is swapped*/
    if ((pVia->RotateDegree == VIA_ROTATE_DEGREE_90) ||
        (pVia->RotateDegree == VIA_ROTATE_DEGREE_270)) {
        dwWidth = pScrn->virtualY;
        dwHeight = pScrn->virtualX;
    }

    if( dwWidth == 1366) {
        dwWidth += 2;		/*Align*/
    }

    DoConvertCoordinates( x1, y1, dwWidth - w, dwHeight - h, pVia->RotateDegree);
}

void 
ConvertCoordinatesTwo(ScrnInfoPtr pScrn, int* x1, int* y1, 
					  int* x2, int* y2, int w , int h)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    int dwWidth, dwHeight;
    
    dwWidth = pScrn->virtualX;
    dwHeight = pScrn->virtualY;
    /*For XV set Clipping Windows, Rotate for 90/270 degree, virtualX/virtualY is swapped*/
    if ((pVia->RotateDegree == VIA_ROTATE_DEGREE_90) ||
        (pVia->RotateDegree == VIA_ROTATE_DEGREE_270)) {
        dwWidth = pScrn->virtualY;
        dwHeight = pScrn->virtualX;
    }
    
	if( dwWidth == 1366) {
        dwWidth += 2;		/*Align*/
    }

    DoConvertCoordinates( x1, y1, dwWidth - w, dwHeight - h, pVia->RotateDegree);
    DoConvertCoordinates( x2, y2, dwWidth - w, dwHeight - h, pVia->RotateDegree);
}


void 
ConvertDirection(ScrnInfoPtr pScrn, int xdir, int ydir, CARD32* cmd)
{
    VIAPtr  pVia = VIAPTR(pScrn);

    switch (pVia->RotateDegree) {                    
    case VIA_ROTATE_DEGREE_90:
         if (xdir < 0) {
             *cmd |= VIA_GEC_DECY;
         }
         if (ydir > 0) {
             *cmd |= VIA_GEC_DECX;
         }
         break;
            
    case VIA_ROTATE_DEGREE_180:
         if (xdir > 0) {
             *cmd |= VIA_GEC_DECX;
         }
         if (ydir > 0) {
             *cmd |= VIA_GEC_DECY;
         }
         break;
            
    case VIA_ROTATE_DEGREE_270:
         if (xdir > 0) {
             *cmd |= VIA_GEC_DECY;
         }
         if (ydir < 0) {
             *cmd |= VIA_GEC_DECX;
         }
         break;
         
    default:		 		/*VIA_ROTATE_DEGREE_0*/
          if (xdir < 0) {
              *cmd |= VIA_GEC_DECX;
          } 
          if (ydir < 0) {
              *cmd |= VIA_GEC_DECY;
          }
          break;
    }        
}

void 
ConvertSize(ScrnInfoPtr pScrn, int* width, int* height)
{    
    VIAPtr  pVia = VIAPTR(pScrn);
    int tmpSize;
    
    switch (pVia->RotateDegree) {                    
    case VIA_ROTATE_DEGREE_90:
    case VIA_ROTATE_DEGREE_270:
         tmpSize = *width;
         *width = *height;
         *height = tmpSize;
         break;

    default:
         break;
    }
}

/* ---------------------------------------------------------------------------*/
/* The following interface is for the uniform 3D texture bliting interface */
/* Using the standard pixmap-matrix transform operation to caculate the transform matrixes */

#define xFixedToFloat(val) \
	((float)xFixedToInt(val) + ((float)xFixedFrac(val) / 65536.0))
#define FloatToxFixed(val) \
	((pixman_fixed_t)(val * 65536.0))
	
#define F(x)	IntToxFixed(x)
#define toF(x)	((float) (x) / 65536.0f)

/* xorg version 1.60 */
#if XORG_VERSION_CURRENT  >= (((1) * 10000000) + ((6) * 100000) + ((0) * 1000) + 0)

static void
viaTransformRescale(struct pixman_f_transform *f_transform, double limit)
{
    double max = 0, v, scale;
    int i, j;

    for (j = 0; j < 3; j++)
	for (i = 0; i < 3; i++)
	    if ((v = abs (f_transform->m[j][i])) > max)
		max = v;
    scale = limit / max;
    for (j = 0; j < 3; j++)
	for (i = 0; i < 3; i++)
	    f_transform->m[j][i] *= scale;
}
/*
 * Compute the complete transformation matrix including
 * client-specified transform, rotation/reflection values and the crtc 
 * offset.
 *
 * Return TRUE if the resulting transform is not a simple translation.
 */
Bool
viaTransformCompute (int src_x, int src_y, int src_w, int src_h,
                                             int dst_x, int dst_y, int dst_w, int dst_h,
                                             unsigned int rotation, int width, int height, 
                                             PictTransformPtr	transform, 
                                             PictTransformPtr invTrans,
                                             PictTransformPtr rrTransform,
                                             PictTransformPtr rrInvTrans
                                             )
{
    struct pixman_f_transform tf_transform[4];
    struct pixman_f_transform *f_transform = &tf_transform[0];
    struct pixman_f_transform *f_invTrans = &tf_transform[1];
    struct pixman_f_transform *f_rr_transform = &tf_transform[2];
    struct pixman_f_transform *f_rr_invTrans = &tf_transform[3];

    Bool  overflow = FALSE;

    /* the transform matrix for the complete transform */
    pixman_transform_init_identity (transform);
    pixman_transform_init_identity (invTrans);
    pixman_f_transform_init_identity (f_transform);
    pixman_f_transform_init_identity (f_invTrans);

    /* the transform matrix for the final rotate and reflect transform only !*/
    pixman_transform_init_identity (rrTransform);
    pixman_transform_init_identity (rrInvTrans);
    pixman_f_transform_init_identity (f_rr_transform);
    pixman_f_transform_init_identity (f_rr_invTrans);

    /* step 1. scale and translate transform */
    if((src_x != 0) || (src_y != 0)) {
        pixman_transform_translate (transform, invTrans, F(-src_x), F(-src_y));
        pixman_f_transform_translate (f_transform, f_invTrans, -src_x, -src_y);
    }
    
    if((src_w != dst_w) || (src_h != dst_h)){
        double f_scale_x, f_scale_y;
        f_scale_x = (double)(dst_w)/(double)(src_w);
        f_scale_y = (double)(dst_h)/(double)(src_h);
        pixman_transform_scale (transform, invTrans, FloatToxFixed(f_scale_x), 
                                             FloatToxFixed(f_scale_y));
        pixman_f_transform_scale (f_transform, f_invTrans, f_scale_x, f_scale_y);
    }

    if((dst_x != 0) || (dst_y != 0)) {
        pixman_transform_translate (transform, invTrans, F(dst_x), F(dst_y));
        pixman_f_transform_translate (f_transform, f_invTrans, dst_x, dst_y);
    }

    /* step 2. rotate and reflect transform */
    if (rotation != RR_Rotate_0)
    {
        double	f_rot_cos, f_rot_sin, f_rot_dx, f_rot_dy;
        double	f_scale_x, f_scale_y, f_scale_dx, f_scale_dy;
        xFixed	rot_cos, rot_sin, rot_dx, rot_dy;
        xFixed	scale_x, scale_y, scale_dx, scale_dy;

        /* rotation */
        switch (rotation & 0xf) {
            default:
            case RR_Rotate_0:
                f_rot_cos = 1;	    f_rot_sin = 0;
                f_rot_dx  = 0;	    f_rot_dy  = 0;
                rot_cos = F ( 1);	    rot_sin = F ( 0);
                rot_dx  = F ( 0);	    rot_dy  = F ( 0);
                break;
            case RR_Rotate_90:
                f_rot_cos = 0;	    f_rot_sin = 1;
                f_rot_dx  = height-1;   f_rot_dy  = 0;
                rot_cos = F ( 0);	    rot_sin = F ( 1);
                rot_dx =  F (height-1); rot_dy  = F (0);
                break;
            case RR_Rotate_180:
                f_rot_cos = -1;	    f_rot_sin = 0;
                f_rot_dx  = width - 1;  f_rot_dy  = height - 1;
                rot_cos = F (-1);	    rot_sin = F ( 0);
                rot_dx  = F (width-1);  rot_dy  = F ( height-1);
                break;
            case RR_Rotate_270:
                f_rot_cos = 0;	    f_rot_sin = -1;
                f_rot_dx  = 0;	    f_rot_dy  = width-1;
                rot_cos = F ( 0);	    rot_sin = F (-1);
                rot_dx  = F ( 0);	    rot_dy  = F ( width-1);
                break;
        }
	
	pixman_transform_rotate (rrTransform, rrInvTrans, rot_cos, rot_sin);
	pixman_transform_translate (rrTransform, rrInvTrans, rot_dx, rot_dy);
	pixman_f_transform_rotate (f_rr_transform, f_rr_invTrans, f_rot_cos, f_rot_sin);
	pixman_f_transform_translate (f_rr_transform, f_rr_invTrans, f_rot_dx, f_rot_dy);

	/* reflection */
	f_scale_x = 1;
	f_scale_dx = 0;
	f_scale_y = 1;
	f_scale_dy = 0;
	scale_x = F (1);
	scale_dx = 0;
	scale_y = F (1);
	scale_dy = 0;
	if (rotation & RR_Reflect_X)
	{
	    f_scale_x = -1;
	    scale_x = F(-1);
	    if (rotation & (RR_Rotate_0|RR_Rotate_180)) {
		f_scale_dx = width-1;
		scale_dx = F(width-1);
	    } else {
		f_scale_dx = height-1;
		scale_dx = F(height-1);
	    }
	}
	if (rotation & RR_Reflect_Y)
	{
	    f_scale_y = -1;
	    scale_y = F(-1);
	    if (rotation & (RR_Rotate_0|RR_Rotate_180)) {
		f_scale_dy = height-1;
		scale_dy = F(height-1);
	    } else {
		f_scale_dy = width-1;
		scale_dy = F(width-1);
	    }
	}
	
	pixman_transform_scale (rrTransform, rrInvTrans, scale_x, scale_y);
	pixman_transform_translate (rrTransform, rrInvTrans, scale_dx, scale_dy);
	pixman_f_transform_scale (f_rr_transform, f_rr_invTrans, f_scale_x, f_scale_y);
	pixman_f_transform_translate (f_rr_transform, f_rr_invTrans, f_scale_dx, f_scale_dy);
    }
    
    /*
     * Compute the final resulting transform
     */

    pixman_f_transform_multiply (f_transform, f_rr_transform, f_transform);
    pixman_f_transform_multiply (f_invTrans, f_invTrans, f_rr_invTrans);

    overflow = pixman_transform_multiply (transform, rrTransform, transform);
    if (overflow)
    {
        struct pixman_f_transform f_scaled;
        f_scaled = *f_transform;
        viaTransformRescale(&f_scaled, 16384.0);
        pixman_transform_from_pixman_f_transform(transform, &f_scaled);
    }
    
    overflow= pixman_transform_multiply (invTrans, invTrans, rrInvTrans);
    if (overflow)
    {
        struct pixman_f_transform f_scaled;
        f_scaled = *f_transform;
        viaTransformRescale(&f_scaled, 16384.0);
        pixman_transform_from_pixman_f_transform(transform, &f_scaled);
    }
}

#else
static void
PictureTransformIdentity (PictTransformPtr matrix)
{
    int	i;
    memset (matrix, '\0', sizeof (PictTransform));
    for (i = 0; i < 3; i++)
	matrix->matrix[i][i] = F(1);
}

static Bool
PictureTransformMultiply (PictTransformPtr dst, PictTransformPtr l, PictTransformPtr r)
{
    PictTransform   d;
    int		    dx, dy;
    int		    o;

    for (dy = 0; dy < 3; dy++)
	for (dx = 0; dx < 3; dx++)
	{
	    xFixed_48_16    v;
	    xFixed_32_32    partial;
	    v = 0;
	    for (o = 0; o < 3; o++)
	    {
		partial = (xFixed_32_32) l->matrix[dy][o] * (xFixed_32_32) r->matrix[o][dx];
		v += partial >> 16;
	    }
	    if (v > MAX_FIXED_48_16 || v < MIN_FIXED_48_16)
		return FALSE;
	    d.matrix[dy][dx] = (xFixed) v;
	}
    *dst = d;
    return TRUE;
}

static void
PictureTransformInitScale (PictTransformPtr t, xFixed sx, xFixed sy)
{
    memset (t, '\0', sizeof (PictTransform));
    t->matrix[0][0] = sx;
    t->matrix[1][1] = sy;
    t->matrix[2][2] = F (1);
}

static xFixed
fixed_inverse (xFixed x)
{
    return (xFixed) ((((xFixed_48_16) F(1)) * F(1)) / x);
}

static Bool
PictureTransformScale (PictTransformPtr forward,
		       PictTransformPtr reverse,
		       xFixed sx, xFixed sy)
{
    PictTransform   t;
    
    PictureTransformInitScale (&t, sx, sy);
    if (!PictureTransformMultiply (forward, &t, forward))
	return FALSE;
    PictureTransformInitScale (&t, fixed_inverse (sx), fixed_inverse (sy));
    if (!PictureTransformMultiply (reverse, reverse, &t))
	return FALSE;
    return TRUE;
}

static void
PictureTransformInitRotate (PictTransformPtr t, xFixed c, xFixed s)
{
    memset (t, '\0', sizeof (PictTransform));
    t->matrix[0][0] = c;
    t->matrix[0][1] = -s;
    t->matrix[1][0] = s;
    t->matrix[1][1] = c;
    t->matrix[2][2] = F (1);
}

static Bool
PictureTransformRotate (PictTransformPtr forward,
			PictTransformPtr reverse,
			xFixed c, xFixed s)
{
    PictTransform   t;
    PictureTransformInitRotate (&t, c, s);
    if (!PictureTransformMultiply (forward, &t, forward))
	return FALSE;
    
    PictureTransformInitRotate (&t, c, -s);
    if (!PictureTransformMultiply (reverse, reverse, &t))
	return FALSE;
    return TRUE;
}

static void
PictureTransformInitTranslate (PictTransformPtr t, xFixed tx, xFixed ty)
{
    memset (t, '\0', sizeof (PictTransform));
    t->matrix[0][0] = F (1);
    t->matrix[0][2] = tx;
    t->matrix[1][1] = F (1);
    t->matrix[1][2] = ty;
    t->matrix[2][2] = F (1);
}

static Bool
PictureTransformTranslate (PictTransformPtr forward,
			   PictTransformPtr reverse,
			   xFixed tx, xFixed ty)
{
    PictTransform   t;
    PictureTransformInitTranslate (&t, tx, ty);
    if (!PictureTransformMultiply (forward, &t, forward))
	return FALSE;
    
    PictureTransformInitTranslate (&t, -tx, -ty);
    if (!PictureTransformMultiply (reverse, reverse, &t))
	return FALSE;
    return TRUE;
}

static Bool
PictureTransformIsIdentity(PictTransform *t)
{
    return ((t->matrix[0][0] == t->matrix[1][1]) &&
            (t->matrix[0][0] == t->matrix[2][2]) &&
            (t->matrix[0][0] != 0) &&
            (t->matrix[0][1] == 0) &&
            (t->matrix[0][2] == 0) &&
            (t->matrix[1][0] == 0) &&
            (t->matrix[1][2] == 0) &&
            (t->matrix[2][0] == 0) &&
            (t->matrix[2][1] == 0));
}

static void
PictureTransformErrorF (PictTransform *t)
{
    ErrorF ("{ { %f %f %f } { %f %f %f } { %f %f %f } }",
	    toF(t->matrix[0][0]), toF(t->matrix[0][1]), toF(t->matrix[0][2]), 
	    toF(t->matrix[1][0]), toF(t->matrix[1][1]), toF(t->matrix[1][2]), 
	    toF(t->matrix[2][0]), toF(t->matrix[2][1]), toF(t->matrix[2][2]));
}

static Bool
PictureTransformIsInverse (char *where, PictTransform *a, PictTransform *b)
{
    PictTransform   t;
/*
    PictureTransformMultiply (&t, a, b);
    if (!PictureTransformIsIdentity (&t))
    {
	ErrorF ("%s: ", where);
	PictureTransformErrorF (a);
	ErrorF (" * ");
	PictureTransformErrorF (b);
	ErrorF (" = ");
	PictureTransformErrorF (a);
	ErrorF ("\n");
	return FALSE;
    }
*/
    return TRUE;
}

void
PictureTransformBounds (BoxPtr b, PictTransformPtr matrix)
{
    PictVector	v[4];
    int		i;
    int		x1, y1, x2, y2;

    v[0].vector[0] = F (b->x1);    v[0].vector[1] = F (b->y1);	v[0].vector[2] = F(1);
    v[1].vector[0] = F (b->x2);    v[1].vector[1] = F (b->y1);	v[1].vector[2] = F(1);
    v[2].vector[0] = F (b->x2);    v[2].vector[1] = F (b->y2);	v[2].vector[2] = F(1);
    v[3].vector[0] = F (b->x1);    v[3].vector[1] = F (b->y2);	v[3].vector[2] = F(1);
    for (i = 0; i < 4; i++)
    {
	PictureTransformPoint (matrix, &v[i]);
	x1 = xFixedToInt (v[i].vector[0]);
	y1 = xFixedToInt (v[i].vector[1]);
	x2 = xFixedToInt (xFixedCeil (v[i].vector[0]));
	y2 = xFixedToInt (xFixedCeil (v[i].vector[1]));
	if (i == 0)
	{
	    b->x1 = x1; b->y1 = y1;
	    b->x2 = x2; b->y2 = y2;
	}
	else
	{
	    if (x1 < b->x1) b->x1 = x1;
	    if (y1 < b->y1) b->y1 = y1;
	    if (x2 > b->x2) b->x2 = x2;
	    if (y2 > b->y2) b->y2 = y2;
	}
    }
}

/*
 * Compute the complete transformation matrix including
 * client-specified transform, rotation/reflection values and the crtc 
 * offset.
 *
 * Return TRUE if the resulting transform is not a simple translation.
 */
Bool
viaTransformCompute (int src_x, int src_y, int src_w, int src_h,
                                             int dst_x, int dst_y, int dst_w, int dst_h,
                                             unsigned int rotation, int width, int height, 
                                             PictTransformPtr	transform, 
                                             PictTransformPtr invTrans,
                                             PictTransformPtr rrTransform,
                                             PictTransformPtr rrInvTrans
                                             )
{
    /* the transform matrix for the complete transform */
    PictureTransformIdentity (transform);
    PictureTransformIdentity (invTrans);

    /* the transform matrix for the final rotate and reflect transform only !*/
    PictureTransformIdentity (rrTransform);
    PictureTransformIdentity (rrInvTrans);

    /* step 1. scale and translate transform */
    if((src_x != 0) || (src_y != 0)) {
        PictureTransformTranslate (transform, invTrans, F(-src_x), F(-src_y));
        PictureTransformIsInverse ("translate before scale", transform, invTrans);
    }

    if((src_w != dst_w) ||(src_h != dst_h)){
        float f_scale_x, f_scale_y;
        f_scale_x = (float)(dst_w)/(float)(src_w);
        f_scale_y = (float)(dst_h)/(float)(src_h);
        PictureTransformScale(transform, invTrans, FloatToxFixed(f_scale_x), 
                                             FloatToxFixed(f_scale_y));
        PictureTransformIsInverse ("scale", transform, invTrans);
    }

    if((dst_x != 0) || (dst_y != 0)) {
        PictureTransformTranslate (transform, invTrans, F(dst_x), F(dst_y));
        PictureTransformIsInverse ("translate after scale", transform, invTrans);
    }

    /* step 2. rotate and reflect transform */
    if (rotation != RR_Rotate_0)
    {
        xFixed	rot_cos, rot_sin, rot_dx, rot_dy;
        xFixed	scale_x, scale_y, scale_dx, scale_dy;

        /* rotation */
        switch (rotation & 0xf) {
        default:
        case RR_Rotate_0:
            rot_cos = F ( 1);	    rot_sin = F ( 0);
            rot_dx  = F ( 0);	    rot_dy  = F ( 0);
            break;
        case RR_Rotate_90:
            rot_cos = F ( 0);	    rot_sin = F ( 1);
            rot_dx =  F (height-1); rot_dy  = F (0);
            break;
        case RR_Rotate_180:
            rot_cos = F (-1);	    rot_sin = F ( 0);
            rot_dx  = F (width-1);  rot_dy  = F ( height-1);
            break;
        case RR_Rotate_270:
            rot_cos = F ( 0);	    rot_sin = F (-1);
            rot_dx  = F ( 0);	    rot_dy  = F ( width-1);
            break;
        }
	
        PictureTransformRotate (rrTransform, rrInvTrans, rot_cos, rot_sin);
        PictureTransformIsInverse ("rotate", rrTransform, rrInvTrans);
        PictureTransformTranslate(rrTransform, rrInvTrans, rot_dx, rot_dy);
        PictureTransformIsInverse ("rotate translate", rrTransform, rrInvTrans);

        /* reflection */
        scale_x = F (1);
        scale_dx = 0;
        scale_y = F (1);
        scale_dy = 0;
        if (rotation & RR_Reflect_X)
        {
            scale_x = F(-1);
            if (rotation & (RR_Rotate_0|RR_Rotate_180)) {
                scale_dx = F(width-1);
            } else {
                scale_dx = F(height-1);
            }
        }
        if (rotation & RR_Reflect_Y)
        {
            scale_y = F(-1);
            if (rotation & (RR_Rotate_0|RR_Rotate_180)) {
                scale_dy = F(height-1);
            } else {
                scale_dy = F(width-1);
            }
        }

        PictureTransformScale(rrTransform, rrInvTrans, scale_x, scale_y);
        PictureTransformIsInverse ("reflect", rrTransform, rrInvTrans);
        PictureTransformTranslate (rrTransform, rrInvTrans, scale_dx, scale_dy);
        PictureTransformIsInverse ("reflect translate", rrTransform, rrInvTrans);
    }
    
    /*
     * Compute the final resulting transform
     */
    if(!PictureTransformMultiply(transform, rrTransform, transform))
        return FALSE;
    if(!PictureTransformMultiply (invTrans, invTrans, rrInvTrans))
        return FALSE;

    if(!PictureTransformIsInverse ("complete", transform, invTrans))
        return FALSE;

    return TRUE;
}
#endif
 
