/*
 * Copyright 1998-2009 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2009 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */


#include "via_serial.h"
#include "via_common.h"


static  SerialBusRec sNormalBusTime = {5, 5, 5, 5, 5, 2};

static  SerialBusRec sDDCBusTime =  {5, 40, 2200, 40, 550, 20};

static  SerialBusRecPtr spB = NULL;

static  signed int sSerialPort = 0;

static  unsigned short sSerialSlaveAddr = 0;

void
viaSerialUDelay ( signed int  usec )
{
    unsigned short port = 0x3c3;
    unsigned short data = 0;
    unsigned  int i = 0;
    while(usec--) {
        /*delay 1us*/
        for (i=0; i<3; i++) {
            __asm__ __volatile__
            (
                "mov %1,%%dx\n\t"
                "in     %%dx,%%ax\n\t"
                "mov  %%ax,%0\n\t"
                : "=r" (data) /* Output */
                : "m" (port) /* input */
                : "ax",  "dx" /* clobbered operands */ 
            );
        }    
    }
}

static void
viaSerialPutBits ( unsigned char scl,  unsigned char sda,  signed int ack )
{
    unsigned char  data;
    unsigned  int sRIndex = 0;

    data = ((scl << 1) | sda) << 4;
    switch (sSerialPort) {
        case PORT_INDEX_I2C_31:
        case PORT_INDEX_I2C_26:
            /* enable serial port */
            data = data | 0x01;
            break;
        case PORT_INDEX_GPIO_2C:
        case PORT_INDEX_GPIO_25:
	    case PORT_INDEX_GPIO_3D:
            if (!ack) {
                /* enable GPIO write port */
                data = data | 0xc0;                     
            } else {
                /* when master-receiver mode, just need to send clock */
                data =  (scl << 5) | 0x80;           
            }
            if (sSerialPort == PORT_INDEX_GPIO_2C)
                data |= BIT1;
            break;     
        default:
            return;
    }

    switch (sSerialPort) {
        case PORT_INDEX_I2C_31:
            sRIndex = REG_SR31;
            break;
        case PORT_INDEX_I2C_26:
            sRIndex = REG_SR26;
            break;
        case PORT_INDEX_GPIO_25:
            sRIndex = REG_SR25;
            break;
        case PORT_INDEX_GPIO_2C:
            sRIndex = REG_SR2C;
            break;
        case PORT_INDEX_GPIO_3D:
            sRIndex = REG_SR3D;
            break;
        default:
            return;
    }

    /* Write Register Value */
    viaWriteVgaIo(sRIndex, data);
    
}

static void 
viaSerialGetBits ( unsigned char * pScl,  unsigned char * pSda)
{
    unsigned char  data;
    unsigned  int  sRIndex = 0;

    switch (sSerialPort) {
        case PORT_INDEX_I2C_31:
            sRIndex = REG_SR31;
            break;
        case PORT_INDEX_I2C_26:
            sRIndex = REG_SR26;
            break;
        case PORT_INDEX_GPIO_25:
            sRIndex = REG_SR25;
            break;
        case PORT_INDEX_GPIO_2C:
            sRIndex = REG_SR2C;
            break;
        case PORT_INDEX_GPIO_3D:
            sRIndex = REG_SR3D;
            break;
        default:
            return;
    }
    
    data = viaReadVgaIo(sRIndex);
    /* get sda */
    *pSda = (data >> 2) & 0x01; 
    /* get scl */
    *pScl = (data >> 3) & 0x01;      
}

/* Some devices will hold SCL low to slow down the bus or until 
 * ready for transmission.
 *
 * This condition will be noticed when the master tries to raise
 * the SCL line. You can set the timeout to zero if the slave device
 * does not support this clock synchronization.
 */
static unsigned  int
viaSerialRaiseSCL ( unsigned char sda,  signed int timeout)
{
    signed int i;
    unsigned char scl;

    viaSerialPutBits(1, sda, 0);
    viaSerialUDelay(spB->RiseFallTime);

    for (i = timeout; i > 0; i -= spB->RiseFallTime) {
        viaSerialGetBits(&scl, &sda);
        if (scl) 
            break;
        viaSerialUDelay(spB->RiseFallTime);
    }
    
    if (i <= 0)
        return FALSE;
  
    return TRUE;
}

/* Send a start signal on the serial bus. The start signal notifies
 * devices that a new transaction is initiated by the bus master.
 *
 * The start signal is always followed by a slave address.
 * Slave addresses are 8+ bits. The first 7 bits identify the
 * device and the last bit signals if this is a read (1) or
 * write (0) operation.
 *
 * There may be more than one start signal on one transaction.
 * This happens for example on some devices that allow reading
 * of registers. First send a start bit followed by the device
 * address (with the last bit 0) and the register number. Then send
 * a new start bit with the device address (with the last bit 1)
 * and then read the value from the device.
 *
 * Note this is function does not implement a multiple master
 * arbitration procedure.
 */

static unsigned  int
viaSerialStart ( signed int timeout)
{
    if (!viaSerialRaiseSCL(1, timeout))
        return FALSE;
    
    viaSerialPutBits(1, 0, 0);
    viaSerialUDelay(spB->HoldTime);
    viaSerialPutBits(0, 0, 0);
    viaSerialUDelay(spB->HoldTime);

    return TRUE;
}

/* This is the default viaSerialStop function if not supplied by the driver.
 *
 * Signal devices on the serial bus that a transaction on the
 * bus has finished. There may be more than one start signal
 * on a transaction but only one stop signal.
 */

static void
viaSerialStop (void)
{
    viaSerialPutBits(0, 0, 0);
    viaSerialUDelay(spB->RiseFallTime);

    viaSerialPutBits(1, 0, 0);
    viaSerialUDelay(spB->HoldTime);
    viaSerialPutBits(1, 1,0);
    viaSerialUDelay(spB->HoldTime);

}

/* Write/Read a single bit to/from a device.
 * Return FALSE if a timeout occurs.
 */

static unsigned  int
viaSerialWriteBit ( unsigned char sda,  signed int timeout)
{
    unsigned  int r;

    viaSerialPutBits(0, sda, 0);
    viaSerialUDelay(spB->RiseFallTime);

    r = viaSerialRaiseSCL(sda, timeout);
    viaSerialUDelay(spB->HoldTime);

    viaSerialPutBits(0, sda, 0);
    viaSerialUDelay(spB->HoldTime);

    return r;
}

static unsigned  int
viaSerialReadBit ( unsigned char *pSda,  signed int timeout)
{
    unsigned  int r = TRUE;
    signed int i = 0;
    unsigned char scl, sda;

    viaSerialPutBits(1, 1, 1);
    viaSerialUDelay(spB->RiseFallTime);

    for (i = spB->HoldTime; i > 0; i -= spB->RiseFallTime) {
        viaSerialGetBits(&scl, &sda);
        if (scl)
            break;
        viaSerialUDelay(spB->RiseFallTime);
    }
    
    if(i <= 0)
        r = FALSE;
   
    viaSerialUDelay(spB->HoldTime);
    viaSerialGetBits(&scl, pSda);

    viaSerialPutBits(0, 1, 1);
    viaSerialUDelay(spB->HoldTime);

    return r;
}

static unsigned  int
viaSerialPutByte ( unsigned char data)
{
    unsigned  int r = TRUE;
    signed int i = 0, j = 0;
    unsigned char scl, sda;

    if (!viaSerialWriteBit((data >> 7) & 1, spB->ByteTimeout))
        return FALSE;

    for (i = 6; i >= 0; i--) {
        if (!viaSerialWriteBit((data >> i) & 1, spB->BitTimeout)) 
            return FALSE;
    }

    viaSerialPutBits(0, 1, 1);
    viaSerialUDelay(spB->RiseFallTime);

    viaSerialPutBits(1, 1, 1);
    viaSerialUDelay(spB->RiseFallTime);

    for (j = spB->HoldTime; j > 0; j -= spB->RiseFallTime) {
        viaSerialGetBits(&scl, &sda);
        if (scl)
            break;
        viaSerialUDelay(spB->RiseFallTime);
    }
    if (j > 0) {
        for (i = spB->AcknTimeout; i > 0; i -= spB->HoldTime) {
                viaSerialUDelay(spB->HoldTime);
                viaSerialGetBits(&scl, &sda);
                if (sda == 0)
                    break;
        }
        if (i <= 0)
            r = FALSE;
    } else {
        r = FALSE;
    }
    
    viaSerialPutBits(0, 1, 1);
    viaSerialUDelay(spB->HoldTime);

    return r;
}

static unsigned  int
viaSerialGetByte ( unsigned char * pData,  signed int last)
{
    signed int i;
    unsigned char sda;

    viaSerialPutBits(0, 1, 1);
    viaSerialUDelay(spB->RiseFallTime);

    if (!viaSerialReadBit(&sda, spB->ByteTimeout))
        return FALSE;
    
    *pData = (sda > 0) << 7;

    for (i = 6; i >= 0; i--) {
        if (!viaSerialReadBit(&sda, spB->BitTimeout))
            return FALSE;
        else
            *pData |= (sda > 0) << i;    
    }
    if (!viaSerialWriteBit(last ? 1 : 0, spB->BitTimeout))
        return FALSE;
    return TRUE;
}

static unsigned  int
viaSerialAddress ( unsigned short addr)
{
    if (viaSerialStart(spB->StartTimeout)) {
        if (viaSerialPutByte(addr & 0xFF)) {
            if ((addr & 0xF8) != 0xF0 &&(addr & 0xFE) != 0x00)
                return TRUE;
            if (viaSerialPutByte((addr >> 8) & 0xFF))
                return TRUE;
        }
	    viaSerialStop();
    }
    return FALSE;
}

static unsigned  int
viaSerialWriteRead ( unsigned char * pWriteBuffer,  signed int  nWrite,
unsigned char *pReadBuffer,  signed int  nRead) 
{
    unsigned  int r = TRUE;
    signed int s = 0;

    if (r && nWrite > 0) {
        r = viaSerialAddress(sSerialSlaveAddr& ~1);
        if (r) {
            for (; nWrite > 0; pWriteBuffer++, nWrite--) {
                if (!(r = viaSerialPutByte(*pWriteBuffer))) 
                    break;            
            }
            s++;
	    }
    }

    if (r && nRead > 0) {
        r = viaSerialAddress(sSerialSlaveAddr | 1);
        if (r) {
            for (; nRead > 0; pReadBuffer++, nRead--) {
                if (!(r = viaSerialGetByte(pReadBuffer, nRead == 1)))
                    break;
            }
            s++;
        }
    }
    if (s)
        viaSerialStop();
    return r;
}

static void
viaSerialInitInfo ( signed int port,  unsigned short slaveAddr)
{
    if(slaveAddr == 0xA0 || slaveAddr == 0xA2) {
        spB = &sDDCBusTime;
    } else {
        spB = &sNormalBusTime;
    }

    sSerialPort = port;
    sSerialSlaveAddr = slaveAddr;
}

/* Read a byte from one of the registers determined by its sub-address.
 */

unsigned  int
viaSerialReadByte ( signed int port,  unsigned short slaveAddr,
                        unsigned char index,  unsigned char *pByte)
{
    viaSerialInitInfo(port, slaveAddr);    
    return viaSerialWriteRead(&index, 1, pByte, 1);
}


/* Write a byte to one of the registers determined by its sub-address.
 */

unsigned  int
viaSerialWriteByte ( signed int port,  unsigned short slaveAddr,
                        unsigned char index,  unsigned char byte)
{
    unsigned char wb[2];

    wb[0] = index;
    wb[1] = byte;

    viaSerialInitInfo(port, slaveAddr);
    return viaSerialWriteRead(wb, 2, 0, 0);
}


/*
    Function Name:SerialWriteBytes
    Description:This function write a_nwrite bytes into the consecutive registers begin at a_index.
*/
unsigned  int
viaSerialWriteBytes ( signed int port,  unsigned short slaveAddr,
                            unsigned char index,  unsigned char *pByte, 
                            unsigned  int nWrite)
{
    /* since 20 bytes is enough for implementation now */
    unsigned char pWrite[20] = {0};     
    unsigned  int i = 0;
    unsigned  int ret = FALSE;
    
    pWrite[0] = index;
    for(i=1;i <= nWrite;i++)
        pWrite[i] = pByte[i-1];

    viaSerialInitInfo(port, slaveAddr);
    if(viaSerialWriteRead(pWrite, nWrite+1, 0, 0))
        ret = TRUE;
    else
        ret = FALSE;

    return ret ;
}

unsigned  int
viaSerialReadBytes ( signed int port,  unsigned short slaveAddr,
                            unsigned char index,  unsigned char *pByte, 
                            signed int nRead)
{
    viaSerialInitInfo(port, slaveAddr);    
    return viaSerialWriteRead(&index, 1, pByte, nRead);
}

unsigned  int
viaSerialWriteByteMask ( unsigned  int port,  unsigned short slaveAddr,
                                unsigned char index,  unsigned char byte,
                                unsigned char mask)
{
    unsigned char data = 0;

    if ( !viaSerialReadByte(port, slaveAddr, index, &data) )
        return FALSE;
    
    data = ( data & ~mask ) | (byte & mask);

    if ( !viaSerialWriteByte(port, slaveAddr, index, data) )
        return FALSE;

    return TRUE;
}


