/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 * INCLUDES 
 */
#include "via_driver.h"
#include "via_eng_regs.h"
#include "debug.h"              /* for DBG_DD */
#include "randr.h"
#include "via_eng_fops.h"
#include "via_eng_fops_cme.h"
#include "via_eng_fops_cle.h"
#include "video_rotate.h"
#include "via_rotate.h"
#include "via_eng_reg_fops.h"


/*
 * EXTERN FUNCTIONS
 */
extern unsigned long AddROTSurface(ScrnInfoPtr pScrn,viaPortPrivPtr pPriv);

/*
 * FUNCTION
 */
void inline construct_Engine_Matrix_Video(unsigned long dwDeviceID,VIAENGMATRIX *pViaEngineMatrix)
{
    pViaEngineMatrix[ENGINE_VIDEO].ulEngType = ENGINE_VIDEO;
    
    switch(dwDeviceID)
    {    
        case VIA_DEVICE_VT3314:
            pViaEngineMatrix[ENGINE_VIDEO].ulEngStatus =HAVE_VIDEO_3;
            pViaEngineMatrix[ENGINE_VIDEO].ulEngMask = HAVE_VIDEO_3;
            break;
        case VIA_DEVICE_VT3327:
        case VIA_DEVICE_VT3336:
        case VIA_DEVICE_VT3364:
            pViaEngineMatrix[ENGINE_VIDEO].ulEngStatus =HAVE_VIDEO_1;
            pViaEngineMatrix[ENGINE_VIDEO].ulEngMask = HAVE_VIDEO_1;
            break;
        case VIA_DEVICE_VT3324:
        case VIA_DEVICE_VT3353:
        case VIA_DEVICE_VT3409:    
            pViaEngineMatrix[ENGINE_VIDEO].ulEngStatus = HAVE_VIDEO_1|HAVE_VIDEO_3;
            pViaEngineMatrix[ENGINE_VIDEO].ulEngMask = HAVE_VIDEO_1|HAVE_VIDEO_3;
            break;
        default:
            DBG_DD(("unrecognized device ID\n"));
            break;
    } 
}


void inline construct_Engine_Matrix_HQV(unsigned long dwDeviceID,VIAENGMATRIX *pViaEngineMatrix)
{
    pViaEngineMatrix[ENGINE_HQV].ulEngType = ENGINE_HQV;

    switch(dwDeviceID)
    {    
        case VIA_DEVICE_VT3314:
        case VIA_DEVICE_VT3327:
        case VIA_DEVICE_VT3336:
        case VIA_DEVICE_VT3364:
            pViaEngineMatrix[ENGINE_HQV].ulEngStatus =HAVE_HQV_0;
            pViaEngineMatrix[ENGINE_HQV].ulEngMask = HAVE_HQV_0;
            break;
        case VIA_DEVICE_VT3324:
        case VIA_DEVICE_VT3353:
        case  VIA_DEVICE_VT3409:    
            pViaEngineMatrix[ENGINE_HQV].ulEngStatus =HAVE_HQV_0|HAVE_HQV_1;
            pViaEngineMatrix[ENGINE_HQV].ulEngMask = HAVE_HQV_0|HAVE_HQV_1;
            break;
        default:
            DBG_DD(("unrecognized device ID\n"));
            break;
    }
}
void construct_Engine_Matrix(unsigned long eng_type_index,unsigned long dwDeviceID,VIAENGMATRIX *pViaEngineMatrix)
{
    switch (eng_type_index)
    {    
        case ENGINE_VIDEO:
            construct_Engine_Matrix_Video(dwDeviceID,pViaEngineMatrix);
            break;
        case ENGINE_HQV:
            construct_Engine_Matrix_HQV(dwDeviceID,pViaEngineMatrix);
            break;
        default:
            break;
    } 
}

/* 
 * For every chip type,init engine matrix
 */
void initEngineMatrix(PVIDDATA pVidData)
{
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo; 
    unsigned long dwDeviceID = viaGfxInfo->chipId;
    int i = 0;

    DBG_DD(("init deviceID %lx engine matrix\n",dwDeviceID));
    
    memset(&pVidData->ViaEngManager,0x00,sizeof(VIAENGMGR));

    /*construct engine matrix*/
    for(;i<ENGINE_TYPE_NUM;i++){
        construct_Engine_Matrix(i,dwDeviceID,pVidData->ViaEngManager.ViaEngineMatrix);
    }
    
    pVidData->ViaEngManager.EngMatrixInit = TRUE;
}


void initHWDiff(PVIDDATA pVidData)
{
    LPVIDHWDIFFERENCE lpVideoHWDifference = &pVidData->VideoHWDifference;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo; 

    /* HW Difference Flag */
    switch( viaGfxInfo->chipId)
    {
        case VIA_DEVICE_VT3336:
            lpVideoHWDifference->dwNeedV1Prefetch = VID_HWDIFF_TRUE;
            lpVideoHWDifference->dwBypassHorExpandMagnify = VID_HWDIFF_FALSE;           
            lpVideoHWDifference->dwBypassVerExpandMagnify = VID_HWDIFF_TRUE;            
            lpVideoHWDifference->dwSupportVideoVQ = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwOnlyOneVideoOverlay = VID_HWDIFF_TRUE;
            lpVideoHWDifference->dwSupportYUVByPass = VID_HWDIFF_FALSE;
            break;
            
        case VIA_DEVICE_VT3327:		/*P4M890*/
        case VIA_DEVICE_VT3364:      /*P4M900*/    
            lpVideoHWDifference->dwNeedV1Prefetch = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwBypassHorExpandMagnify = VID_HWDIFF_FALSE;            
            lpVideoHWDifference->dwBypassVerExpandMagnify = VID_HWDIFF_TRUE;
            lpVideoHWDifference->dwSupportVideoVQ = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwOnlyOneVideoOverlay = VID_HWDIFF_TRUE;
            lpVideoHWDifference->dwSupportYUVByPass = VID_HWDIFF_FALSE;
            break;

        case VIA_DEVICE_VT3314:		/*CN700*/
            lpVideoHWDifference->dwNeedV1Prefetch = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwBypassHorExpandMagnify = VID_HWDIFF_FALSE;            
            lpVideoHWDifference->dwBypassVerExpandMagnify = VID_HWDIFF_TRUE;
            lpVideoHWDifference->dwSupportVideoVQ = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwOnlyOneVideoOverlay = VID_HWDIFF_TRUE;
            lpVideoHWDifference->dwSupportYUVByPass = VID_HWDIFF_FALSE;
            break;
        case VIA_DEVICE_VT3324:
            lpVideoHWDifference->dwNeedV1Prefetch = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwBypassHorExpandMagnify = VID_HWDIFF_TRUE;           
            lpVideoHWDifference->dwBypassVerExpandMagnify = VID_HWDIFF_TRUE;
            lpVideoHWDifference->dwSupportVideoVQ = VID_HWDIFF_TRUE;
            lpVideoHWDifference->dwOnlyOneVideoOverlay = VID_HWDIFF_FALSE;            
            lpVideoHWDifference->dwSupportYUVByPass = VID_HWDIFF_TRUE;
            break;
        case VIA_DEVICE_VT3353:
            lpVideoHWDifference->dwNeedV1Prefetch = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwBypassHorExpandMagnify = VID_HWDIFF_FALSE;           
            lpVideoHWDifference->dwBypassVerExpandMagnify = VID_HWDIFF_TRUE;
            lpVideoHWDifference->dwSupportVideoVQ = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwSupportYUVByPass = VID_HWDIFF_TRUE;
            lpVideoHWDifference->dwOnlyOneVideoOverlay = VID_HWDIFF_FALSE;            
            break;
        case VIA_DEVICE_VT3409:
            lpVideoHWDifference->dwNeedV1Prefetch = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwBypassHorExpandMagnify = VID_HWDIFF_FALSE;            
            lpVideoHWDifference->dwBypassVerExpandMagnify = VID_HWDIFF_TRUE;
            /*the spec unclearly mentioned CR.disable*/
            lpVideoHWDifference->dwSupportVideoVQ = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwOnlyOneVideoOverlay = VID_HWDIFF_FALSE;
            lpVideoHWDifference->dwSupportYUVByPass = VID_HWDIFF_TRUE;
            break;
        default:
            /*Unkown DeviceID*/
            DBG_DD((" Unknown dwDeviceID!!\n"));
            break;
    }/* End of dwProjectIDRevision */
}


VIAVidRegFuncRec VidRegFuncTable[2] = {
    {
        "CLE",
        vidSetVideoEngineCtrl,              /*setCtrl*/
        vidSetVideoEngineStartAddr,         /*setStartAddr*/
        vidSetVideoEngineFetch,             /*setFetch*/
        vidSetVideoEngineSrcPitch,          /*setSrcPitch*/
        vidSetVideoEngineDstStart,          /*setDstStart*/
        vidSetVideoEngineZoomFactor_CLE,    /*setZoomFactor*/
        vidSetVideoEngineDstWidthandHeight, /*setDstWidthandHeight*/
        vidSetVideoEngineFIFO_CLE,          /*setFIFO*/
        vidSetVideoEngineColorKey,          /*setColorKey*/
        vidSetVideoEngineComposeMode_CLE,   /*setComposeMode*/
        vidSetVideoEngineChromaKey,         /*setChromaKey*/
    },
    
    {
        "CME",
        vidSetVideoEngineCtrl,              /*setCtrl*/
        vidSetVideoEngineStartAddr,         /*setStartAddr*/
        vidSetVideoEngineFetch,             /*setFetch*/
        vidSetVideoEngineSrcPitch,          /*setSrcPitch*/
        vidSetVideoEngineDstStart,          /*setDstStart*/        
        vidSetVideoEngineZoomFactor_CME,    /*setZoomFactor*/
        vidSetVideoEngineDstWidthandHeight, /*setDstWidthandHeight*/
        vidSetVideoEngineFIFO_CME,          /*setFIFO*/         
        vidSetVideoEngineColorKey,          /*setColorKey*/        
        vidSetVideoEngineComposeMode_CME,   /*setComposeMode*/        
        vidSetVideoEngineChromaKey,         /*setChromaKey*/
       
    }
};

VIAHqvRegFuncRec HqvRegFuncTable[2] = {
    {
        "CLE",
        vidSetHQVEngineSrcAddr_CLE,         /*setSrcAddr*/
        vidSetHQVEngineDstAddr,             /*setDstAddr*/
        vidSetHQVAdvanceControl_CLE,        /*setAdvanceCtrl*/
        vidSetHQVEngineCtrl_CLE,            /*setCtrl*/
        vidSetHQVEngineOffset,              /*setOff*/  
        vidSetHQVEngineSrcDataOffset_CLE,   /*setSrcDataOff*/
        vidSetHQVEngineSrcFetch,            /*setFetch*/
        vidSetHQVEngineSrcPitch_CLE,        /*setSrcPitch*/
        vidSetHQVEngineDstPitch,            /*setDstPitch*/
        vidSetHQVEngineZoomFactor_CLE       /*setZoomFactor*/
    },
    
    {
        "CME",
        vidSetHQVEngineSrcAddr_CME,         /*setSrcAddr*/
        vidSetHQVEngineDstAddr,             /*setDstAddr*/
        vidSetHQVAdvanceControl_CME,        /*setAdvanceCtrl*/
        vidSetHQVEngineCtrl_CME,            /*setCtrl*/
        vidSetHQVEngineOffset,              /*setOff*/ 
        vidSetHQVEngineSrcDataOffset_CME,   /*setSrcDataOff*/        
        vidSetHQVEngineSrcFetch,            /*setFetch*/
        vidSetHQVEngineSrcPitch_CME,        /*setSrcPitch*/
        vidSetHQVEngineDstPitch,            /*setDstPitch*/
        vidSetHQVEngineZoomFactor_CME       /*setZoomFactor*/
    }
};

void initVidEngRegMgr(PVIDDATA pVidData)
{
    VIAVidRegFuncPtr vidRegFunc = &pVidData->vidRegFunc;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo; 
    if (viaGfxInfo->chipId == VIA_DEVICE_VT3314)
    {
        *vidRegFunc = VidRegFuncTable[0];
    }else{
        *vidRegFunc = VidRegFuncTable[1];
    }
    return;
}

void initHqvEngRegMgr(PVIDDATA pVidData)
{
    VIAHqvRegFuncPtr hqvRegFunc = &pVidData->hqvRegFunc;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo; 

    if (viaGfxInfo->chipId == VIA_DEVICE_VT3314)
    {
        *hqvRegFunc = HqvRegFuncTable[0];
    }else{
        *hqvRegFunc = HqvRegFuncTable[1];
    }
    return;
}

void vidInitVideoInfo(PVIDDATA pVidData)
{
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo; 

    memset(pVidData->Video_Hqv_Map, 0x00, sizeof(pVidData->Video_Hqv_Map));
    
    /* Init video engine and hqv engine map array */
    switch (viaGfxInfo->chipId){
        /*v3 exist*/
        case VIA_DEVICE_VT3314:  
            pVidData->Video_Hqv_Map[VIDEO_1] = HQV_NONE;
            pVidData->Video_Hqv_Map[VIDEO_3] = HQV_0;
            break;

        /* v1 exist*/ 
        case VIA_DEVICE_VT3336:
        case VIA_DEVICE_VT3327:
        case VIA_DEVICE_VT3364:    
            pVidData->Video_Hqv_Map[VIDEO_1] = HQV_0;
            pVidData->Video_Hqv_Map[VIDEO_3] = HQV_NONE;
            break;
                
        /*v1 v3 both exist*/
        case VIA_DEVICE_VT3324:
        case VIA_DEVICE_VT3353:  
        case VIA_DEVICE_VT3409:    
            pVidData->Video_Hqv_Map[VIDEO_1] = HQV_0;
            pVidData->Video_Hqv_Map[VIDEO_3] = HQV_1;
            break;
            
        default:
            break;    
    }  

    /* Init VidHWDifference structure for chip specific function. */
    initHWDiff(pVidData);
    
    if (pVidData->ViaEngManager.EngMatrixInit == FALSE){
        initEngineMatrix(pVidData);
    }

    initVidEngRegMgr(pVidData);
    initHqvEngRegMgr(pVidData);
}

void scaleOverlayDestWindow2( ViaXvRectPtr rSrc, ViaXvRectPtr rDest,
                            int OrigHActive, int OrigVActive,
                            int RealHActive, int RealVActive,
                            long XSIZEOffset, long YSIZEOffset,
                            long XPOSITIONOffset, long YPOSITIONOffset)
{
    int tmpLeft, tmpRight, tmpTop, tmpBottom;
    int screenX, screenY;

    DBG_DD(("scaleOverlayDestWindow: OriginalActiveSize=%dx%d \n", OrigHActive, OrigVActive));
    DBG_DD(("scaleOverlayDestWindow: RealActiveSize=%dx%d \n", RealHActive, RealVActive));
    DBG_DD(("scaleOverlayDestWindow: Offset XSize=%ld, YSize=%ld, XPosition=%ld, YPosition=%ld \n",
             XSIZEOffset, YSIZEOffset, XPOSITIONOffset, YPOSITIONOffset));
    DBG_DD(("scaleOverlayDestWindow: Old rDest >> left=%ld, right=%ld, top=%ld, bottom=%ld \n",
             rDest->left, rDest->right, rDest->top,
             rDest->bottom));

    tmpLeft = ((int) rDest->left * RealHActive) / OrigHActive;
    tmpRight = ((int) rDest->right * RealHActive) / OrigHActive;
    tmpTop = ((int) rDest->top * RealVActive) / OrigVActive;
    tmpBottom = ((int) rDest->bottom * RealVActive) / OrigVActive;

    screenX = RealHActive - XSIZEOffset * 2;
    screenY = RealVActive - YSIZEOffset * 2;

    tmpLeft = (screenX / 2) - (RealHActive / 2 - tmpLeft) * screenX / RealHActive;
    tmpRight = (screenX / 2) - (RealHActive / 2 - tmpRight) * screenX / RealHActive;
    tmpTop = (screenY / 2) - (RealVActive / 2 - tmpTop) * screenY / RealVActive;
    tmpBottom = (screenY / 2) - (RealVActive / 2 - tmpBottom) * screenY / RealVActive;

    /*
     * software shrink adjust windows size 
     */
    tmpLeft += XSIZEOffset + XPOSITIONOffset;
    tmpRight += XSIZEOffset + XPOSITIONOffset;
    tmpTop += YSIZEOffset - YPOSITIONOffset;
    tmpBottom += YSIZEOffset - YPOSITIONOffset;

    rDest->left = (unsigned long) tmpLeft;
    rDest->right = (unsigned long) tmpRight;
    rDest->top = (unsigned long) tmpTop;
    rDest->bottom = (unsigned long) tmpBottom;

    DBG_DD(("scaleOverlayDestWindow: new rDest >> left=%ld, right=%ld, top=%ld, bottom=%ld \n",
             rDest->left, rDest->right, rDest->top,
             rDest->bottom));
}

void overlay3DScalCalcuDetRect2(ViaXvRectPtr rDest, HW3D_Scaling_INFO * pgfx3DScal_info)
{
    int OrigHActive, OrigVActive, RealHActive, RealVActive;
    int tmpLeft, tmpRight, tmpTop, tmpBottom;
    int screenX, screenY;

    DBG_DD(("into overlay3DScalWindowAdjust \n"));
    DBG_DD(("before: lpUpdate.rDest  >> left=%ld, right=%ld, top = %ld, bottom=%ld \n",
             rDest->left, rDest->right, rDest->top,
             rDest->bottom));
    
    OrigHActive = pgfx3DScal_info->OrigHActive;
    OrigVActive = pgfx3DScal_info->OrigVActive;
    RealHActive = pgfx3DScal_info->RealHActive;
    RealVActive = pgfx3DScal_info->RealVActive;
    DBG_DD(("OrigHActive=%d, OrigVActive=%d, RealHActive = %d, RealVActive=%d \n", OrigHActive,
             OrigVActive, RealHActive, RealVActive));

    DBG_DD(("gfx3DScal_info.XSIZEOffset=%ld, gfx3DScal_info.YSIZEOffset=%ld \n",
             pgfx3DScal_info->XSIZEOffset, pgfx3DScal_info->YSIZEOffset));
    screenX = RealHActive - pgfx3DScal_info->XSIZEOffset * 2;
    screenY = RealVActive - pgfx3DScal_info->YSIZEOffset * 2;
    DBG_DD(("screenX=%d, screenY=%d \n", screenX, screenY));
    /*Calculate coordinates on the scaled screen*/
    tmpLeft = (int)(rDest->left) * screenX / OrigHActive;
    tmpRight = (int)(rDest->right) * screenX/ OrigHActive;
    tmpTop = (int)(rDest->top) * screenY / OrigVActive;
    tmpBottom = (int)(rDest->bottom) * screenY / OrigVActive;


    DBG_DD(("rDest >> left=%d, right=%d, top = %d, bottom=%d \n", 
                tmpLeft, tmpRight,tmpTop, tmpBottom));
             
    DBG_DD((": gfx3DScal_info.XPOSITIONOffset=%ld, gfx3DScal_info.YPOSITIONOffset=%ld \n",
             pgfx3DScal_info->XPOSITIONOffset, pgfx3DScal_info->YPOSITIONOffset));
    /*
     * software shrink adjust windows size 
     */
    tmpLeft += pgfx3DScal_info->XSIZEOffset + pgfx3DScal_info->XPOSITIONOffset;
    tmpRight += pgfx3DScal_info->XSIZEOffset + pgfx3DScal_info->XPOSITIONOffset;
    tmpTop += pgfx3DScal_info->YSIZEOffset - pgfx3DScal_info->YPOSITIONOffset;
    tmpBottom += pgfx3DScal_info->YSIZEOffset - pgfx3DScal_info->YPOSITIONOffset;

    if (tmpLeft < 0)
    {
        tmpLeft = 0;
    }
    if (tmpRight > RealHActive)
    {
        tmpRight = RealHActive;
    }
    if (tmpTop < 0)
    {
        tmpTop = 0;
    }
    if (tmpBottom > RealVActive)
    {
        tmpBottom = RealVActive;
    }
    
    rDest->top = tmpTop;
    rDest->bottom = tmpBottom;
    rDest->right = tmpRight;
    rDest->left = tmpLeft;
}

unsigned long determineVideoEngineandIGA(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    unsigned long ret = TRUE;
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;
    unsigned long *pVideoFlag = &pPriv->videoFlag;
      
    /* Which Video on which IGA, for example: 
     * { -1, VIDEO_3 , VIDEO_1 }: V3 ON IGA1(IGA1=1), V1 ON IGA2 */
    unsigned long *IGA_Video_Map = pVidData->IGA_Video_Map;

    memset(IGA_Video_Map, 0x00, sizeof(pVidData->IGA_Video_Map));

    if (viaGfxInfo->screenAttr.duoview == TRUE){
        /*a. The default situation*/
        switch (viaGfxInfo->chipId){
            /*v3 exist*/
            case VIA_DEVICE_VT3314:  
                IGA_Video_Map[pPriv->curIGA] = VIDEO_3;
                
                /* b. Special case, we can use option to set video on which screen.
                 *    For only one video engine V1 or V3 */
                if (viaGfxInfo->preferedVideoIga == IGA1){
                    IGA_Video_Map[IGA1] = VIDEO_3;                    
                    IGA_Video_Map[IGA2] = VIDEO_NONE;
                }else if (viaGfxInfo->preferedVideoIga == IGA2){
                    IGA_Video_Map[IGA2] = VIDEO_3;                    
                    IGA_Video_Map[IGA1] = VIDEO_NONE;
                }
                break;
                
            /* v1 exist*/ 
            case VIA_DEVICE_VT3336:
            case VIA_DEVICE_VT3327:
            case VIA_DEVICE_VT3364:    
                IGA_Video_Map[pPriv->curIGA] = VIDEO_1;  

                /* b. Special case, we can use option to set video on which screen.
                 *    For only one video engine V1 or V3 */
                if (viaGfxInfo->preferedVideoIga == IGA1){
                    IGA_Video_Map[IGA1] = VIDEO_1;                    
                    IGA_Video_Map[IGA2] = VIDEO_NONE;
                }else if (viaGfxInfo->preferedVideoIga == IGA2){
                    IGA_Video_Map[IGA2] = VIDEO_1;                    
                    IGA_Video_Map[IGA1] = VIDEO_NONE;
                }
                break;
                                    
            /*v1 v3 both exist*/
            case VIA_DEVICE_VT3324:
            case VIA_DEVICE_VT3353:  
            case VIA_DEVICE_VT3409:    
                IGA_Video_Map[IGA2] = VIDEO_1;
                IGA_Video_Map[IGA1] = VIDEO_3;
                break;
                                
            default:
                break;    
        }         
    }else{
        IGA_Video_Map[pPriv->curIGA] = (*pVideoFlag & VIDEO_1_INUSE)? \
            VIDEO_1 : VIDEO_3;
    }

    return ret;
}

unsigned long checkSrcandDstIfVideoOnSndIGA(PVIDDATA pVidData, ViaXvRectPtr rSrc, ViaXvRectPtr rDest)
{
    unsigned long dwRet = TRUE;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;

    if(viaGfxInfo->xrandrEnabled){
        if (viaGfxInfo->igaAttr.iga2_left){
            
            rDest->left = rDest->left - viaGfxInfo->igaInfo[0].start_x;
            rDest->right = rDest->right - viaGfxInfo->igaInfo[0].start_x;
            
        }else if (viaGfxInfo->igaAttr.iga2_above){
            
            rDest->top = rDest->top - viaGfxInfo->igaInfo[0].start_y;
            rDest->bottom = rDest->bottom - viaGfxInfo->igaInfo[0].start_y;
            
        }else if (viaGfxInfo->igaAttr.iga2_right){
            
            rDest->left = rDest->left - viaGfxInfo->igaInfo[1].start_x;
            rDest->right = rDest->right - viaGfxInfo->igaInfo[1].start_x;
            
        }else if (viaGfxInfo->igaAttr.iga2_below){
                
            rDest->top = rDest->top - viaGfxInfo->igaInfo[1].start_y;
            rDest->bottom = rDest->bottom - viaGfxInfo->igaInfo[1].start_y;
            
        }else{
            /* Nothing to do */
        }
    }

    return dwRet;
}

unsigned long checkSrcandDstIfPanning(PVIDDATA pVidData, ViaXvRectPtr rSrc, ViaXvRectPtr rDest, unsigned long curIGA)
{
    unsigned long dwRet = TRUE;

    DBG_DD(("Panning!!\n"));
    rDest->left = (int)((int)(rDest->left) - pVidData->panning_x[curIGA-1]); 
    rDest->top = (int)((int)(rDest->top) - pVidData->panning_y[curIGA-1]);
    rDest->right = (int)((int)(rDest->right) - pVidData->panning_x[curIGA-1]); 
    rDest->bottom =(int)((int)(rDest->bottom) -pVidData->panning_y[curIGA-1]); 
    DBG_DD(("Panning (%ld,%ld,%ld,%ld)\n",rDest->left,rDest->right,rDest->top,rDest->bottom));
    
    return dwRet;
}


unsigned long checkSrcandDstIfDownScaling(PVIDDATA pVidData, ViaXvRectPtr rSrc, ViaXvRectPtr rDest, unsigned long curIGA)
{
    unsigned long dwRet = TRUE;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    scaleOverlayDestWindow2(rSrc,rDest,viaGfxInfo->igaInfo[curIGA-1].desired_width,
                                        viaGfxInfo->igaInfo[curIGA-1].desired_height,
                                        viaGfxInfo->igaInfo[curIGA-1].crtc_width,
                                        viaGfxInfo->igaInfo[curIGA-1].crtc_height, 0, 0, 0, 0);

    return dwRet;
}

unsigned long checkSrcandDstIfExpand(PVIDDATA pVidData,  ViaXvRectPtr rSrc, ViaXvRectPtr rDest, unsigned long curIGA)
{
    unsigned long dwRet = TRUE;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    int top=rDest->top;
    int left=rDest->left;
    int right=rDest->right;
    int bottom=rDest->bottom;

    int crtcHeight = viaGfxInfo->igaInfo[curIGA-1].crtc_height;
    int crtcWidth  = viaGfxInfo->igaInfo[curIGA-1].crtc_width;
    int dwHeight = viaGfxInfo->igaInfo[curIGA-1].desired_height;
    int dwWidth = viaGfxInfo->igaInfo[curIGA-1].desired_width;

    if(viaGfxInfo->igaInfo[curIGA-1].igaStatus.expanded)
    {
        DBG_DD(("===dwExpand!! \n"));
        DBG_DD(("Original dest (%d,%d,%d,%d)\n",left,top,right,bottom));

        switch(viaGfxInfo->igaInfo[curIGA-1].igaRRStatus.unit & VIA_ROTATE_DEGREE_ALL){
            case VIA_ROTATE_DEGREE_90:
            case VIA_ROTATE_DEGREE_270:
                rDest->top = top*crtcWidth/dwWidth;
                rDest->bottom = bottom*crtcWidth/dwWidth;
                rDest->left = left*crtcHeight/dwHeight;
                rDest->right =right*crtcHeight/dwHeight;
                break;
            default:
                rDest->top  =top* crtcHeight/dwHeight;
                rDest->bottom=bottom * crtcHeight/ dwHeight;
                rDest->left = left *crtcWidth/dwWidth;
                rDest->right =right*crtcWidth/dwWidth;
                break;
         }   
        DBG_DD(("Expanded (%d,%d,%d,%d)\n",(int)rDest->left,(int)rDest->top,\
                    (int)rDest->right,(int)rDest->bottom));
    }
    return dwRet;
}

unsigned long checkSrcandDstIf3DScaling(PVIDDATA pVidData, ViaXvRectPtr rSrc, ViaXvRectPtr rDest, unsigned long curIGA)
{
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;   
    overlay3DScalCalcuDetRect2(rDest,&(viaGfxInfo->igaInfo[curIGA-1].igagfx3DScaling_info));

    DBG_DD(("After adjust: lpUpdate.rDest  >> left=%ld, right=%ld, top = %ld, bottom=%ld \n",
             rDest->left, rDest->right, rDest->top,rDest->bottom));

    return TRUE;
}


unsigned long checkSrcandDstIfMoveOut(PVIDDATA pVidData,  ViaXvRectPtr rSrc, ViaXvRectPtr rDest, 
    int dwWidth,int dwHeight, unsigned long curIGA)
{

    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    int nDstTop, nDstBottom, nDstLeft, nDstRight, nDstWidth, nDstHeight;
    
    int dwScnWidth=dwWidth;
    int dwScnHeight=dwHeight;
    int dwRawSrcWidth, dwRawSrcHeight;      
  
    nDstLeft = (int)(rDest->left);
    nDstTop = (int)(rDest->top);
    nDstRight = (int)(rDest->right);
    nDstBottom = (int)(rDest->bottom);
       
    /*
     * Screen width & height 
     */
    if (!(dwScnWidth)||!(dwScnHeight)){
        switch(viaGfxInfo->igaInfo[curIGA-1].igaRRStatus.unit & VIA_ROTATE_DEGREE_ALL){
            case VIA_ROTATE_DEGREE_90:
            case VIA_ROTATE_DEGREE_270:
                dwScnWidth = viaGfxInfo->igaInfo[curIGA-1].desired_height;
                dwScnHeight = viaGfxInfo->igaInfo[curIGA-1].desired_width;
                break;
            default:
                dwScnWidth = viaGfxInfo->igaInfo[curIGA-1].desired_width;
                dwScnHeight = viaGfxInfo->igaInfo[curIGA-1].desired_height;
                break;
         }   
    } 
    /*
     * the original width & height AP sent 
     */
    dwRawSrcWidth = rSrc->right - rSrc->left;
    dwRawSrcHeight = rSrc->bottom - rSrc->top;

    /*
     * Check if the Overlay Dest rectangle is moved out of current viewport totally ? if so hidden overlay.
     */
    if ((nDstBottom <= 0) ||              /* above of viewport */
        (nDstTop >= dwScnHeight) ||       /* below of viewport */
        (nDstRight <= 0 ) ||              /* left of viewport */
        (nDstLeft >= dwScnWidth))         /* right of viewport */
    {

        rDest->top     = 0x7F0;
        rDest->left    = 0x7F0;
        rDest->bottom  = 0x7FF;
        rDest->right   = 0x7FF;

     
    }
    else
    {
        /*
         * Sometimes AP will set wrong dest postion such as viaexp 
         * sometimes set dest right equal to dest left when panning 
         */
        nDstWidth = max(1, (nDstRight - nDstLeft));
        nDstHeight= max(1, (nDstBottom - nDstTop));
        
        if (nDstLeft < 0)
        {
             rSrc->left += (((-nDstLeft) * (dwRawSrcWidth)) +
                            ((nDstRight - nDstLeft) >> 1)) / nDstWidth;
             rDest->left=0;
        }
          
        if (nDstRight > dwScnWidth)
        {
            rSrc->right -= (((nDstRight - dwScnWidth) * (dwRawSrcWidth)) +
                            ((nDstRight - nDstLeft) >> 1)) / nDstWidth;
            rDest->right=dwScnWidth;
        }
          
        if (nDstTop < 0)
        {
            rSrc->top += (((-nDstTop) * (dwRawSrcHeight)) +
                          ((nDstBottom - nDstTop) >> 1)) / nDstHeight;
            rDest->top=0;
        }
          
        if (nDstBottom > dwScnHeight)
        {
            rSrc->bottom -= (((nDstBottom - dwScnHeight) * (dwRawSrcHeight)) +
                             ((nDstBottom - nDstTop) >> 1)) / nDstHeight;
            rDest->bottom=dwScnHeight;
        }
    }
      
    DBG_DD(("final Dest %ld,%ld,%ld,%ld\n",rDest->left,rDest->right,rDest->top,rDest->bottom)); 
    DBG_DD(("final Src %ld,%ld,%ld,%ld\n",rSrc->left,rSrc->right,rSrc->top,rSrc->bottom)); 
    return TRUE;
}


unsigned long calcSrcandDst(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    unsigned long ret = TRUE;

    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    unsigned long *pVideoFlag = &pPriv->videoFlag;
    LPOVERLAYRECORD pOverlayRecord = &pPriv->ovlInfo[pPriv->curIGA-1];
    ViaXvRectPtr rSrc = &pOverlayRecord->dwOverlaySrcRect;
    ViaXvRectPtr rDest = &pOverlayRecord->dwOverlayDestRect;

    HW3D_Scaling_INFO igagfx3DScal_info = viaGfxInfo->igaInfo[pPriv->curIGA-1].igagfx3DScaling_info;        

    DBG_DD((" current Iga index is  %d \n", pPriv->curIGA));    

#ifndef  VIA_DEFINE_OVERLAY_DISPLAY_PATH
    if(*pVideoFlag & VIDEO_ON_SND_IGA){
        checkSrcandDstIfVideoOnSndIGA(pVidData, rSrc, rDest);
    }

    /* Fixed for SAMM panning Mode */
    if(viaGfxInfo->igaInfo[pPriv->curIGA-1].igaStatus.panning)
    {
        checkSrcandDstIfPanning(pVidData, rSrc,  rDest, pPriv->curIGA);
    }

    /* rDest->left maybe negative if panning+fullscreen, so this function
       can not run before panning justify*/
    checkSrcandDstIfMoveOut(pVidData, rSrc, rDest, 0, 0, pPriv->curIGA);
    
    /*video image goes totally unseen*/
    if((*pVideoFlag & VIDEO_1_SHOW) || (*pVideoFlag & VIDEO_3_SHOW)){
        if (rDest->top==0x7F0 && rDest->bottom==0x7FF && rDest->left==0x7F0 && rDest->right ==0x7FF){
            if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_1){                
                *pVideoFlag &= ~VIDEO_1_SHOW;            
                *pVideoFlag |=  VIDEO_1_HIDE;
            }else if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_3){                
                *pVideoFlag &= ~VIDEO_3_SHOW;            
                *pVideoFlag |=  VIDEO_3_HIDE;
            }
        }else{
            if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_1){                
                *pVideoFlag |= VIDEO_1_SHOW;            
                *pVideoFlag &= ~VIDEO_1_HIDE;
            }else if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_3){                
                *pVideoFlag |= VIDEO_3_SHOW;            
                *pVideoFlag &= ~VIDEO_3_HIDE;
            }
        }
    }
#endif

    if(viaGfxInfo->igaInfo[pPriv->curIGA-1].igaStatus.scaling_hw)
    {
        checkSrcandDstIfDownScaling(pVidData, rSrc,  rDest,  pPriv->curIGA);
    }

    if(viaGfxInfo->igaInfo[pPriv->curIGA-1].igaStatus.expanded)
    {
        pOverlayRecord->dwExpandDestRect = *rDest; 
        checkSrcandDstIfExpand(pVidData, rSrc, &pOverlayRecord->dwExpandDestRect, pPriv->curIGA);
    }

    DBG_DD((" current IGA %lu, igaDISP3DScalIGAPath is  %d \n",
              pPriv->curIGA, igagfx3DScal_info.DISP3DScalIGAPath));

    if (igagfx3DScal_info.gfx3DScalingEnable)
    {   
        pOverlayRecord->dwScalingDestRect = *rDest;
        checkSrcandDstIf3DScaling(pVidData, rSrc,  &pOverlayRecord->dwScalingDestRect , pPriv->curIGA);
    }   
    return ret;
}


unsigned long setOverlayFlags(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv,LPDDUPDATEOVERLAY lpUpdate)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    LPOVERLAYRECORD pOverlayRecord = &pPriv->ovlInfo[pPriv->curIGA-1];
    unsigned long *pVideoFlag = &pPriv->videoFlag;    
    LPVIDHWDIFFERENCE lpVideoHWDifference = &pVidData->VideoHWDifference;
    int scrnOnIGA2 = -1;
    unsigned long dwFlags = lpUpdate->dwFlags;     
    
    /* Save unmodified overlay rSrc & rDest ,which are the values sent from 
     * AP,to OverlayRecord struct 
     */
    pOverlayRecord->dwOriSrcRect = lpUpdate->rSrc;
    pOverlayRecord->dwOriDestRect = lpUpdate->rDest;

    /*FIXME! not sure if pOverlayXXBackup is neede */
   
    /* Save values to  dwOverlaySrcRect,dwOverlayDestRect for later computation*/
    pOverlayRecord->dwOverlaySrcRect = lpUpdate->rSrc;
    pOverlayRecord->dwWidth  = (int)(lpUpdate->rSrc.right) - (int)(lpUpdate->rSrc.left);
    pOverlayRecord->dwHeight = (int)(lpUpdate->rSrc.bottom)- (int)(lpUpdate->rSrc.top);
    
    pOverlayRecord->dwOverlayDestRect = lpUpdate->rDest;
    pOverlayRecord->dwOverlayDestWidth = (int)(pOverlayRecord->dwOriDestRect.right)\
                                         -(int)(pOverlayRecord->dwOriDestRect.left);
    pOverlayRecord->dwOverlayDestHeight = (int)(pOverlayRecord->dwOriDestRect.bottom)\
                                          -(int)(pOverlayRecord->dwOriDestRect.top);

    if (dwFlags & DDOVER_KEYDEST){
        *pVideoFlag |= VIDEO_KEY_DEST;
        
        DBG_DD(("DDOVER_KEYDEST \n"));

        if(viaGfxInfo->screenInfo[pScrn->scrnIndex].bpp == 8){
            pOverlayRecord->dwKeyLow = pPriv->colorKey & 0xFF ;
        }else if(viaGfxInfo->screenInfo[pScrn->scrnIndex].bpp == 16){
            pOverlayRecord->dwKeyLow = pPriv->colorKey & 0xFFFF ;
        }else if(viaGfxInfo->screenInfo[pScrn->scrnIndex].bpp == 24){
            pOverlayRecord->dwKeyLow = pPriv->colorKey & 0x00FFFFFF;
        }else if(viaGfxInfo->screenInfo[pScrn->scrnIndex].bpp == 32){
            pOverlayRecord->dwKeyLow = pPriv->colorKey & 0x00FFFFFF;
        }   
        pOverlayRecord->dwColorKeyOn = 1;
    }else{
        pOverlayRecord->dwColorKeyOn = 0;
    }     

    /*set up deinterlaced mode select*/
    if (dwFlags & DDOVER_INTERLEAVED){
        pOverlayRecord->dwDeinterlaceMode |= DDOVER_INTERLEAVED;
    }

    if (dwFlags & DDOVER_BOB){
        pOverlayRecord->dwDeinterlaceMode |= DDOVER_BOB;
    }

    DBG_DD(("Dest Width,Dest Height:%ld,%ld\n",pOverlayRecord->dwOverlayDestWidth,pOverlayRecord->dwOverlayDestHeight));

    /*Check for IGA2 bypass horizontal/vertical expand function enable/disable*/
    if((lpVideoHWDifference->dwBypassHorExpandMagnify == VID_HWDIFF_TRUE)
        && ( viaGfxInfo->igaInfo[pPriv->curIGA-1].igaStatus.expanded)){
        *pVideoFlag |= VIDEO_BYPASS_HOR_EXPAND;
    }else{
        *pVideoFlag &= ~VIDEO_BYPASS_HOR_EXPAND;
    }

    if((lpVideoHWDifference->dwBypassVerExpandMagnify == VID_HWDIFF_TRUE)
        && ( viaGfxInfo->igaInfo[pPriv->curIGA-1].igaStatus.expanded)){
        *pVideoFlag |= VIDEO_BYPASS_VER_EXPAND;
    }else{
        *pVideoFlag &= ~VIDEO_BYPASS_VER_EXPAND;
    }    

    if(pPriv->curIGA == IGA2){
        *pVideoFlag |= VIDEO_ON_IGA2;
    }else{
        *pVideoFlag &= ~VIDEO_ON_IGA2;
    }

#ifndef  VIA_DEFINE_OVERLAY_DISPLAY_PATH
    if((viaGfxInfo->xrandrEnabled == TRUE) && (viaGfxInfo->screenAttr.extend == TRUE)){
        if ((viaGfxInfo->igaAttr.iga2_left) && 
            (lpUpdate->rDest.left >= (long)viaGfxInfo->igaInfo[1].visible_width)){
            *pVideoFlag |= VIDEO_ON_SND_IGA;        
        }else if ((viaGfxInfo->igaAttr.iga2_above) && 
            (lpUpdate->rDest.top >= (long)viaGfxInfo->igaInfo[1].visible_height)){
            *pVideoFlag |= VIDEO_ON_SND_IGA;        
        }else if ((viaGfxInfo->igaAttr.iga2_right) && 
            (lpUpdate->rDest.left >= (long)viaGfxInfo->igaInfo[0].visible_width)){
            *pVideoFlag |= VIDEO_ON_SND_IGA; 
        }else if ((viaGfxInfo->igaAttr.iga2_below) && 
                (lpUpdate->rDest.top >= (long)viaGfxInfo->igaInfo[0].visible_height)){
            *pVideoFlag |= VIDEO_ON_SND_IGA;        
        }else{        
            *pVideoFlag &= ~VIDEO_ON_SND_IGA;
        }    
    }else{
        *pVideoFlag &= ~VIDEO_ON_SND_IGA;
    }
#endif

    return TRUE;    
}

/* Desc:    Fullfil the structure pOverlayRecord
 *
 */
unsigned long setupUpdateOverlay(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv, LPDDUPDATEOVERLAY lpUpdate)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    unsigned long ret = TRUE;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    unsigned long dwFlags = lpUpdate->dwFlags;
    unsigned long *pVideoFlag = &pPriv->videoFlag;

    DBG_DD(("%s\n",__FUNCTION__));

    if(pVideoFlag == NULL){
        return TRUE;
    }

    if(dwFlags & DDOVER_HIDE){
        DBG_DD(("DDOVER_HIDE \n"));
#ifndef  VIA_DEFINE_OVERLAY_DISPLAY_PATH
        determineVideoEngineandIGA( pScrn,pPriv);          
#endif

        if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_1){
            *pVideoFlag |=  VIDEO_1_HIDE;
            *pVideoFlag &= ~VIDEO_1_SHOW;
        }else if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_3){
            *pVideoFlag |=  VIDEO_3_HIDE;
            *pVideoFlag &= ~VIDEO_3_SHOW;
        }

        return ret;
    }    

    if(dwFlags & DDOVER_SHOW){
        DBG_DD(("DDOVER_SHOW \n"));

#ifndef  VIA_DEFINE_OVERLAY_DISPLAY_PATH
        determineVideoEngineandIGA( pScrn,pPriv);          
#endif

        if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_1){
            *pVideoFlag &= ~VIDEO_1_HIDE;
            *pVideoFlag |=  VIDEO_1_SHOW;
        }else if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_3){
            *pVideoFlag &= ~VIDEO_3_HIDE;
            *pVideoFlag |=  VIDEO_3_SHOW;
        }
        
        if (dwFlags & DDOVER_ON_TOP){
            DBG_DD(("Raise to Top\n"));
            *pVideoFlag|= VIDEO_SHOW_TOP;
        }else {
            *pVideoFlag &=~VIDEO_SHOW_TOP;
        }

        DBG_DD(("duoview=%hx SAMM=%x ,*pVideoFlag=%lx \n",
                 viaGfxInfo->screenAttr.duoview, viaGfxInfo->screenAttr.samm,*pVideoFlag));        
        DBG_DD(("into viaUpdateOverlay : lpUpdate.rDest  >> left=%ld, right=%ld, top = %ld, bottom=%ld \n",
                 lpUpdate->rDest.left, lpUpdate->rDest.right, lpUpdate->rDest.top,lpUpdate->rDest.bottom));

            
        setOverlayFlags(pScrn,pPriv, lpUpdate);

        calcSrcandDst(pScrn,pPriv);
    }  
    return ret;
}

 
unsigned long checkAndSetRotateParam(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    unsigned long ret;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;   
    unsigned long curIGA = pPriv->curIGA - 1;
    unsigned long *pVideoFlag = &pPriv->videoFlag;
    int dst_width,dst_height;
    LPOVERLAYRECORD lpOverlayRecord = &pPriv->ovlInfo[pPriv->curIGA-1];
    ViaXvRectRec rSrc   =lpOverlayRecord->dwOriSrcRect;
    ViaXvRectRec rDest  = lpOverlayRecord->dwOriDestRect;;    
    unsigned long rotbufsize;
    DBG_DD(("%s\n",__FUNCTION__));  
    DBG_DD(("rSrc %ld,%ld,%ld,%ld,\n",rSrc.left,rSrc.right,rSrc.top,rSrc.bottom));
    DBG_DD(("rDest %ld,%ld,%ld,%ld,\n",rDest.left,rDest.right,rDest.top,rDest.bottom));    

#ifndef  VIA_DEFINE_OVERLAY_DISPLAY_PATH
    /* 
      *  Dynamically decide whether need to  create/destroy Rotate surface,
      *  According to the rotate degree.Clean it to rgb black when the rotation
      *  degree changes
      */
    if(viaGfxInfo->igaInfo[curIGA].igaRRStatus.unit){
        if ( !pPriv->pRotSurface[curIGA]){
            ret = AddROTSurface(pScrn,pPriv);
            if (ret == FALSE){
                DBG_DD(("  ROT Memory allocation failed\n"));
                return FALSE;
            }

            if (viaGfxInfo->chipId != VIA_DEVICE_VT3314){
                pPriv->pHqvSurface[curIGA]->format = RGB16;
                pPriv->pHqvSurface[curIGA]->formatType= DDPF_RGB;
            }
        }else if (pPriv->igaRRStatus[curIGA] != viaGfxInfo->igaInfo[curIGA].igaRRStatus.unit){                     
            pPriv->igaRRStatus[curIGA] = viaGfxInfo->igaInfo[curIGA].igaRRStatus.unit;
            if (pVia->IsPCI){
                pVia->myWaitIdle(pVia);
            }

            if(viaGfxInfo->igaInfo[pPriv->curIGA -1].igaRRStatus.rotate_90 ||
               viaGfxInfo->igaInfo[pPriv->curIGA -1].igaRRStatus.rotate_270) {
                pPriv->pRotSurface[curIGA]->pitch = ALIGN_TO(pPriv->pRotSurface[curIGA]->height*2 , 256);
                rotbufsize = pPriv->pRotSurface[curIGA]->pitch * pPriv->pRotSurface[curIGA]->width;
            } else {
                pPriv->pRotSurface[curIGA]->pitch = ALIGN_TO(pPriv->pRotSurface[curIGA]->width*2 , 256);
                rotbufsize = pPriv->pRotSurface[curIGA]->pitch * pPriv->pRotSurface[curIGA]->height;
            }
            rgbFillBlack(pVia, pPriv->rotMem[curIGA].base, rotbufsize*XV_ROTSURFACE_NUM);
        }
    } else if(!viaGfxInfo->igaInfo[curIGA].igaRRStatus.unit && pPriv->pRotSurface[curIGA]){
        if(pPriv->rotMem[curIGA].pool && pPriv->rotMem[curIGA].base){
            viaVideoMemFree(pScrn, &pPriv->rotMem[curIGA]);
        }
        free(pPriv->pRotSurface[curIGA]);
        pPriv->pRotSurface[curIGA] = NULL;
        if (viaGfxInfo->chipId != VIA_DEVICE_VT3314){
            pPriv->pHqvSurface[curIGA]->format = FOURCC_YUY2;
            pPriv->pHqvSurface[curIGA]->formatType= DDPF_FOURCC;
        }
    }
#endif

    if (viaGfxInfo->igaInfo[curIGA].igaRRStatus.unit) {
        
        dst_width = rDest.right-rDest.left;
        dst_height = rDest.bottom - rDest.top;
        
        switch(viaGfxInfo->igaInfo[curIGA].igaRRStatus.unit & VIA_ROTATE_DEGREE_ALL){   
            case VIA_ROTATE_DEGREE_90:
            case VIA_ROTATE_DEGREE_270:
                if (pPriv->hqvEngFlag & VAL_HQV_V_DOWN_SCALE_CME){
                    lpOverlayRecord->dwRotDestWidth= dst_height;
                }else {
                    lpOverlayRecord->dwRotDestWidth = lpOverlayRecord->dwOriSrcHeight;
                }

                if (pPriv->hqvEngFlag & VAL_HQV_H_DOWN_SCALE_CME){
                    lpOverlayRecord->dwRotDestHeight= dst_width;
                }else {
                    lpOverlayRecord->dwRotDestHeight = lpOverlayRecord->dwOriSrcWidth;
                }
               lpOverlayRecord->dwRotPitch = pPriv->pRotSurface[curIGA]->pitch;
                break;
                 
            case VIA_ROTATE_DEGREE_180:
            case VIA_ROTATE_DEGREE_0:
            default:
                if (pPriv->hqvEngFlag & VAL_HQV_V_DOWN_SCALE_CME){
                    lpOverlayRecord->dwRotDestHeight=dst_height;
                }else {
                    lpOverlayRecord->dwRotDestHeight=lpOverlayRecord->dwOriSrcHeight;
                }

                if (pPriv->hqvEngFlag & VAL_HQV_H_DOWN_SCALE_CME){
                    lpOverlayRecord->dwRotDestWidth = dst_width;
                }else {
                    lpOverlayRecord->dwRotDestWidth=lpOverlayRecord->dwOriSrcWidth;
                }
               lpOverlayRecord->dwRotPitch = pPriv->pRotSurface[curIGA]->pitch;
              
                break;
        }

        DBG_DD(("rot_dst_height:%ld,rot_dst_width:%ld\n",\
                     lpOverlayRecord->dwRotDestHeight,lpOverlayRecord->dwRotDestWidth));
        DBG_DD(("dstpitch= %ld\n",lpOverlayRecord->dwRotPitch));   
    }
    return TRUE;
}


unsigned long fireHQVandVideoEngine(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    unsigned long *pVideoFlag = &pPriv->videoFlag;
    LPOVERLAYRECORD pOverlayRecord = &pPriv->ovlInfo[pPriv->curIGA-1];
    LPVIDEOHWINFO lphwinfo = &pPriv->vidHWinfo;  
    unsigned long hqvIndex = 0;

    DBG_DD(("dwHQVCtl=%lx, dwVidCtl=%lx, dwCompose=%lx,pPriv->vidEngUpdate=%lx\n",
    	lphwinfo->dwHQVCtl[pPriv->curIGA-1],lphwinfo->dwVidCtl,lphwinfo->dwCompose,pPriv->vidEngUpdate));

    if(pPriv->hqvEngUpdate == HQV_1){
        hqvIndex = REG_HQV1_INDEX;
    }

    if(((pPriv->vidEngUpdate == VIDEO_1) && (*pVideoFlag & VIDEO_1_SHOW)) ||
        ((pPriv->vidEngUpdate == VIDEO_3) && (*pVideoFlag & VIDEO_3_SHOW))){     
        if (pOverlayRecord->dwHQVon != HQV_USE_ENABLE){       

            if(pPriv->vidEngUpdate == VIDEO_1){
                /*video initial sequence*/
                vfCM(V1_CONTROL,lphwinfo->dwVidCtl,pScrn,pPriv);                
                vfCM(V_COMPOSE_MODE, (lphwinfo->dwCompose|V1_COMMAND_FIRE ),pScrn,pPriv);
            }else if(pPriv->vidEngUpdate == VIDEO_3) {
                /*video initial sequence*/
                vfCM(V3_CONTROL, lphwinfo->dwVidCtl,pScrn,pPriv);
                vfCM(V_COMPOSE_MODE, (lphwinfo->dwCompose|V3_COMMAND_FIRE ),pScrn,pPriv);
            }

            /*HQV initial sequence.firset clear the frame staus and flip the current frame*/             
            vfCMHQVW(HQV_CONTROL|hqvIndex, lphwinfo->dwHQVCtl[pPriv->curIGA-1]|HQV_FLIP_STATUS,pScrn,pPriv);
            vfCMHQVW(HQV_CONTROL|hqvIndex, lphwinfo->dwHQVCtl[pPriv->curIGA-1]|HQV_SW_FLIP,pScrn,pPriv);            

            vfHM(pScrn,pPriv,HW_WAIT_HQV_FINISH);            
            vfCMHQVW(HQV_CONTROL|hqvIndex, lphwinfo->dwHQVCtl[pPriv->curIGA-1]|HQV_FLIP_STATUS,pScrn,pPriv);
            
            vfHM(pScrn,pPriv,HM_WAIT_FIRE);             
                                          
            pOverlayRecord->dwHQVon = HQV_USE_ENABLE;
        }else{   
            /**************************************************
             *  Step2 : Fire theVideo engine, include   
             *          lphwinfo->dwVidCtl lphwinfo->dwCompose,
             *          if using PCI method, just fire it. If 
             *          using AGP method, set command in the 
             *          AGP queue, wait for calling funciton 
             *          vfFireAGPcommand to fire
             **************************************************/
            if(pPriv->vidEngUpdate == VIDEO_1){                               
                vfCM(V1_CONTROL,lphwinfo->dwVidCtl,pScrn,pPriv);                   
                vfCM(V_COMPOSE_MODE, (lphwinfo->dwCompose|V1_COMMAND_FIRE ),pScrn,pPriv);
            }else if(pPriv->vidEngUpdate == VIDEO_3 ){
                vfCM(V3_CONTROL, lphwinfo->dwVidCtl,pScrn,pPriv);             
                vfCM(V_COMPOSE_MODE, (lphwinfo->dwCompose|V3_COMMAND_FIRE ),pScrn,pPriv);
            }
                    
            /**************************************************
             *  Step1 : Fire the HQV engine
             **************************************************/                     
            vfCMHQVW(HQV_CONTROL|hqvIndex, lphwinfo->dwHQVCtl[pPriv->curIGA-1]|HQV_FLIP_STATUS,pScrn,pPriv);
            vfCMHQVW(HQV_CONTROL|hqvIndex, lphwinfo->dwHQVCtl[pPriv->curIGA-1]|HQV_SW_FLIP,pScrn,pPriv);    
            vfHM(pScrn,pPriv,HW_WAIT_HQV_FINISH);            
            vfCMHQVW(HQV_CONTROL|hqvIndex, lphwinfo->dwHQVCtl[pPriv->curIGA-1]|HQV_FLIP_STATUS,pScrn,pPriv);
            
            vfHM(pScrn,pPriv,HM_WAIT_FIRE);             
        }
    }else if(((pPriv->vidEngUpdate == VIDEO_1) && (*pVideoFlag & VIDEO_1_HIDE)) ||
        ((pPriv->vidEngUpdate == VIDEO_3) && (*pVideoFlag & VIDEO_3_HIDE))){
        
        if(pOverlayRecord->dwHQVon == HQV_USE_DISABLE){            
            return TRUE;
        }

        /**************************************************
         *  Step2 : Fire theVideo engine, include   
         *          lphwinfo->dwVidCtl lphwinfo->dwCompose,
         *          if using PCI method, just fire it. If 
         *          using AGP method, set command in the 
         *          AGP queue, wait for calling funciton 
         *          vfFireAGPcommand to fire
         **************************************************/
        if(pPriv->vidEngUpdate == VIDEO_1){
            vfCM(V1_CONTROL,lphwinfo->dwVidCtl,pScrn,pPriv);                   
            vfCM(V_COMPOSE_MODE, (lphwinfo->dwCompose|V1_COMMAND_FIRE ),pScrn,pPriv);
        }else if(pPriv->vidEngUpdate == VIDEO_3){
            vfCM(V3_CONTROL, lphwinfo->dwVidCtl,pScrn,pPriv);             
            vfCM(V_COMPOSE_MODE, (lphwinfo->dwCompose|V3_COMMAND_FIRE ),pScrn,pPriv);
        }

        vfHM(pScrn,pPriv,HM_WAIT_FIRE);

        /**************************************************
         *  Step1 : Fire the HQV engine
         **************************************************/        
        if(pOverlayRecord->dwHQVon == HQV_USE_ENABLE){
            vfCMHQVW(HQV_CONTROL|hqvIndex, lphwinfo->dwHQVCtl[pPriv->curIGA-1],pScrn,pPriv);

            /* HQV engine should be enabled first before using it,
             * So set the dwHQVon flag to HQV_USE_DISABLE to make 
             * sure HQV will be enabled before using.*/
            pOverlayRecord->dwHQVon = HQV_USE_DISABLE;
        }
    }    
    return TRUE;
}



unsigned long fireHardwareIcon(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    
    unsigned long *pVideoFlag = &pPriv->videoFlag;

    LPOVERLAYRECORD pOverlayRecord = &pPriv->ovlInfo[pPriv->curIGA-1];

    DBG_DD(("%s\n",__FUNCTION__));

    if ((*pVideoFlag & VIDEO_1_SHOW) || (*pVideoFlag & VIDEO_3_SHOW)){  
        /*[ video on IGA2, so alpha/hi set to IGA2,too]
         *  V4 can work for Alpha Stream and Hardware Icon, and only one function 
         * will be enabled at a time. Also, we use the same register [Vid.260] to control the two functions, 
         * so we should make sure what the function it means when using this register. 
         * The following setting is for Alpha Stream, so we should not change the settings when Alpha Stream 
         * function is disabled (i.e., Hardware Icon is enabled) or it may affect the behavior of Hardware Icon. 
         */
        if (!viaGfxInfo->hwIconEnalbed)
         {
            if ( *pVideoFlag & VIDEO_ON_IGA2 )
             {
                 vfCM(HI_CONTROL, VIDInD(HI_CONTROL)|V4_DISPLAY_PATH,pScrn,pPriv);
             }else{
                 vfCM(HI_CONTROL, VIDInD(HI_CONTROL)&(~V4_DISPLAY_PATH),pScrn,pPriv);
             }
         }
    }
    return TRUE;
}


/* Desc:    Fullfil the structure lphwinfo
 *
 */
unsigned long subSequenceUpdateOverlay(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    unsigned long *pVideoFlag = &pPriv->videoFlag;
    LPOVERLAYRECORD pOverlayRecord = &pPriv->ovlInfo[pPriv->curIGA-1];
     unsigned long iga;
    unsigned long videng;    
    VIAVidRegFuncPtr vidEngfunc = &pVidData->vidRegFunc;    
    VIAHqvRegFuncPtr hqvEngfunc = &pVidData->hqvRegFunc;

    DBG_DD(("%s\n",__FUNCTION__));  

    pPriv->vidEngUpdate = pVidData->IGA_Video_Map[pPriv->curIGA];
    pPriv->hqvEngUpdate  = pVidData->Video_Hqv_Map[pPriv->vidEngUpdate];

    /* special case for duovie mode, only single video engine chips */
    if((pPriv->vidEngUpdate == VIDEO_NONE) || 
        (pPriv->hqvEngUpdate == HQV_NONE)) {
        return TRUE;
    }
    
#if VIA_APPLY_XVOVERLAY_AGP
    RING_VARS;
    if(viaGfxInfo->chipId != VIA_DEVICE_VT3314) {
        BEGIN_HEADER5_VIDEOAGP(256);
    }
#endif   

    if (((pPriv->vidEngUpdate == VIDEO_1) && (*pVideoFlag & VIDEO_1_SHOW)) ||
        ((pPriv->vidEngUpdate == VIDEO_3) && (*pVideoFlag & VIDEO_3_SHOW))){          /* show overlay path */
        
        if (*pVideoFlag & VIDEO_HQV_INUSE){  
            hqvEngfunc->setSrcAddr(pScrn, pPriv);
            hqvEngfunc->setDstAddr(pScrn, pPriv);
            hqvEngfunc->setAdvanceCtrl(pScrn, pPriv);
            hqvEngfunc->setCtrl(pScrn, pPriv);
            hqvEngfunc->setOff(pScrn, pPriv);
            hqvEngfunc->setSrcDataOff(pScrn, pPriv);
            hqvEngfunc->setFetch(pScrn, pPriv);
            hqvEngfunc->setSrcPitch(pScrn, pPriv);
            hqvEngfunc->setDstPitch(pScrn, pPriv);
            hqvEngfunc->setZoomFactor(pScrn, pPriv);
        } 
        
        /* Always check rotation surface, and set rotation parameter if needed! */
        checkAndSetRotateParam( pScrn, pPriv);

        /*Set the video engine control register*/        
        vidEngfunc->setCtrl(pScrn, pPriv);
        vidEngfunc->setStartAddr(pScrn, pPriv);
        vidEngfunc->setFetch(pScrn, pPriv);
        vidEngfunc->setSrcPitch(pScrn, pPriv);
        vidEngfunc->setDstStart(pScrn, pPriv);
        vidEngfunc->setZoomFactor(pScrn, pPriv);
        vidEngfunc->setDstWidthandHeight(pScrn, pPriv);
        vidEngfunc->setFIFO(pScrn, pPriv);
        vidEngfunc->setColorKey(pScrn, pPriv);
        vidEngfunc->setComposeMode(pScrn, pPriv);
        vidEngfunc->setChromaKey(pScrn, pPriv);
        
        vidWriteHQVandVideoCmd(pScrn, pPriv);
        fireHQVandVideoEngine( pScrn, pPriv);
        fireHardwareIcon( pScrn, pPriv); 
    }else if(((pPriv->vidEngUpdate == VIDEO_1) && (*pVideoFlag & VIDEO_1_HIDE)) ||
        ((pPriv->vidEngUpdate == VIDEO_3) && (*pVideoFlag & VIDEO_3_HIDE))){      
        /* hide overlay path */    
       
        hqvEngfunc->setCtrl(pScrn, pPriv);        
        /*Set the video engine control register*/        
        vidEngfunc->setCtrl(pScrn, pPriv);
        fireHQVandVideoEngine( pScrn, pPriv);
    } 

#if VIA_APPLY_XVOVERLAY_AGP
    if(viaGfxInfo->chipId != VIA_DEVICE_VT3314) {
/* For 324, if there is no command between BEGIN_HEADER5_VIDEOAGP and END_HEADER5_VIDEOAGP
 *   the system will hang. So add a patch for 324 to avoid this case.
 */
        if(viaGfxInfo->chipId == VIA_DEVICE_VT3324){
            vfCM(RegAddr_NULLCMD,0,pScrn,pPriv); 
        }
        END_HEADER5_VIDEOAGP;
        ADVANCE_RING;
    }
#endif
    return TRUE;
}


unsigned long UpdateOverlay(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv, LPDDUPDATEOVERLAY lpUpdate)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;  
    unsigned long ret = TRUE;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;    
    unsigned long igaIndex = 0;

    DBG_DD(("%s\n",__FUNCTION__));   

    DBG_DD(("current IGA:%ld\n",pPriv->curIGA));
    ret = setupUpdateOverlay(pScrn, pPriv, lpUpdate);
    ret = subSequenceUpdateOverlay(pScrn, pPriv);
    
    if(((viaGfxInfo->xrandrEnabled == FALSE) && (viaGfxInfo->screenAttr.duoview == TRUE)) ||
        ((viaGfxInfo->xrandrEnabled == TRUE) && (viaGfxInfo->screenAttr.clone == TRUE))){
        igaIndex = pPriv->curIGA;
        pPriv->curIGA = (igaIndex==IGA1)? IGA2: IGA1;

        DBG_DD(("current IGA:%ld\n",pPriv->curIGA));
        ret = setupUpdateOverlay(pScrn, pPriv, lpUpdate);
        ret = subSequenceUpdateOverlay(pScrn, pPriv);        
        pPriv->curIGA = igaIndex;
    }
    return ret;
}

unsigned long flipOut(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;         
    LPVIDEOHWINFO lphwinfo = &pPriv->vidHWinfo;  
    LPOVERLAYRECORD pOverlayRecord = &pPriv->ovlInfo[pPriv->curIGA-1];
    unsigned long dwVideoFlags = pPriv->videoFlag;
    unsigned long swCrPhysicalAddr = 0;
    unsigned long swCbPhysicalAddr = 0;
    unsigned long curIGAIndex = pPriv->curIGA - 1;
    unsigned long dispIndex = pPriv->pXvSurface->cur_idx;
    unsigned long rotIndex;
    unsigned long tmpHqvCtrl = 0;

    pPriv->vidEngUpdate = pVidData->IGA_Video_Map[pPriv->curIGA];
    pPriv->hqvEngUpdate  = pVidData->Video_Hqv_Map[pPriv->vidEngUpdate];

    /* special case for duovie mode, only single video engine chips */
    if((pPriv->vidEngUpdate == VIDEO_NONE) || 
        (pPriv->hqvEngUpdate == HQV_NONE)) {
        return TRUE;
    }

#if VIA_APPLY_XVOVERLAY_AGP
    RING_VARS;
    if(viaGfxInfo->chipId != VIA_DEVICE_VT3314) {
        BEGIN_HEADER5_VIDEOAGP(256);
    }
#endif  
    
    DBG_DD(("lpSWDevice->dwSWPhysicalAddr[0:1] location : %s\n",\
               (pPriv->xv_refine_using_pcie)? "SF: PCI-e memory":"SL: Video memory"));
    DBG_DD(("lpSWDevice->dwSWPhysicalAddr[0:1]=%lx %lx\n",\
               pPriv->pXvSurface->offset[0],pPriv->pXvSurface->offset[1]));
    DBG_DD(("lpSWDevice->dwHQVAddr[0:2]=%lx %lx %lx \n",\
               pPriv->pHqvSurface[curIGAIndex]->offset[0], \
               pPriv->pHqvSurface[curIGAIndex]->offset[1],\
               pPriv->pHqvSurface[curIGAIndex]->offset[2]));
    if (viaGfxInfo->igaInfo[pPriv->curIGA-1].igaRRStatus.unit){
        DBG_DD(("RotAddr[0:1]=%lx,%lx\n",\
                    pPriv->pRotSurface[curIGAIndex]->offset[0],\
                    pPriv->pRotSurface[curIGAIndex]->offset[1]));
    }  

    switch(pPriv->pXvSurface->format){
        case FOURCC_YUY2:
            DBG_DD((" Flip:lpDPFsrc->dwFourCC=%lx \n",pPriv->pXvSurface->format));
            
            if(pPriv->xv_refine_using_pcie) {
                vfCMHQVW(HQV_SRC_STARTADDR_Y, \
                    pPriv->pXvSurface->offset[dispIndex]|LOC_SF, pScrn,pPriv);
            } else {
                vfCMHQVW(HQV_SRC_STARTADDR_Y, \
                    pPriv->pXvSurface->offset[dispIndex], pScrn, pPriv);
            }

            tmpHqvCtrl = lphwinfo->dwHQVCtl[curIGAIndex];            
            tmpHqvCtrl &= ~(HQV_SW_FLIP|HQV_FLIP_STATUS);            
            vfCMHQVW(HQV_CONTROL,(( tmpHqvCtrl &~HQV_FLIP_ODD)|HQV_FLIP_STATUS),pScrn,pPriv);            
            vfCMHQVW(HQV_CONTROL,(( tmpHqvCtrl &~HQV_FLIP_ODD)|HQV_SW_FLIP),pScrn,pPriv);            
            vfHM(pScrn,pPriv,HW_WAIT_HQV_FINISH);
            vfCMHQVW(HQV_CONTROL,(( tmpHqvCtrl &~HQV_FLIP_ODD)|HQV_FLIP_STATUS),pScrn,pPriv);            

            /* In rotation case, we will update the frame buffer starting address
             * after the video is rotated. So there is no need to set these registers here in rotation case,
             * to do this can help to reduce garbage or flicker (rotation image overlaps unrotation image)
             */
            if(viaGfxInfo->igaInfo[curIGAIndex].igaRRStatus.unit){
#if VIA_APPLY_XVOVERLAY_AGP
                if(viaGfxInfo->chipId != VIA_DEVICE_VT3314) {
                    END_HEADER5_VIDEOAGP;
                }
#endif
                viaHWVideoRotation(pScrn, pPriv);
#if VIA_APPLY_XVOVERLAY_AGP
                if(viaGfxInfo->chipId != VIA_DEVICE_VT3314) {
                    BEGIN_HEADER5_VIDEOAGP(16);
                }
#endif    
                 rotIndex = pPriv->pRotSurface[curIGAIndex]->cur_idx;
                 DBG_DD(("Flip Rot Buffer %d\n", rotIndex));

                if (pPriv->vidEngUpdate == VIDEO_1){
                    vfCM(V1_STARTADDR0,pPriv->pRotSurface[curIGAIndex]->offset[rotIndex] + \
                        pOverlayRecord->dwOffset,pScrn,pPriv);
                    vfCM(V_COMPOSE_MODE, (VIDInD(V_COMPOSE_MODE) | V1_COMMAND_FIRE),pScrn,pPriv);     
                    
                }else if (pPriv->vidEngUpdate == VIDEO_3){
                    vfCM(V3_STARTADDR_0,pPriv->pRotSurface[curIGAIndex]->offset[rotIndex] +\
                        pOverlayRecord->dwOffset,pScrn,pPriv);
                    vfCM(V_COMPOSE_MODE, (VIDInD(V_COMPOSE_MODE) | V3_COMMAND_FIRE),pScrn,pPriv);                        
                }

                vfHM(pScrn,pPriv,HM_WAIT_FIRE);

                pPriv->pRotSurface[curIGAIndex]->cur_idx = !rotIndex;                
            }
            break;

        case FOURCC_NV12:
        case FOURCC_YV12:
        default:
            DBG_DD((" Flip:lpDPFsrc->dwFourCC=%lx \n",pPriv->pXvSurface->format));
            
            if(pPriv->xv_refine_using_pcie) {
                vfCMHQVW(HQV_SRC_STARTADDR_Y, pPriv->pXvSurface->offset[dispIndex]|LOC_SF, pScrn,pPriv);
            } else {
                vfCMHQVW(HQV_SRC_STARTADDR_Y, pPriv->pXvSurface->offset[dispIndex], pScrn,pPriv);
            }

            swCrPhysicalAddr = (pPriv->pXvSurface->offset[dispIndex]) + \
                (pPriv->pXvSurface->pitch)*(pPriv->pXvSurface->height);
            swCbPhysicalAddr = swCrPhysicalAddr + \
                (pPriv->pXvSurface->pitch >>1)*(pPriv->pXvSurface->height>>1);

            if(viaGfxInfo->chipId == VIA_DEVICE_VT3314){           
                 vfCMHQVW(HQV_SRC_STARTADDR_U, swCbPhysicalAddr, pScrn,pPriv);
                 vfCMHQVW(HQV_SRC_STARTADDR_V, swCrPhysicalAddr, pScrn,pPriv);
            } else {
                if(pPriv->xv_refine_using_pcie) {
                    vfCMHQVW(HQV_SRC_STARTADDR_U,swCrPhysicalAddr|LOC_SF, pScrn, pPriv);
                }else {
                    vfCMHQVW(HQV_SRC_STARTADDR_U,swCrPhysicalAddr, pScrn, pPriv);
                }
            }
            
            tmpHqvCtrl = lphwinfo->dwHQVCtl[curIGAIndex];            
            tmpHqvCtrl &= ~(HQV_SW_FLIP|HQV_FLIP_STATUS);
            vfCMHQVW(HQV_CONTROL,(( tmpHqvCtrl &~HQV_FLIP_ODD)|HQV_FLIP_STATUS),pScrn,pPriv);            
            vfCMHQVW(HQV_CONTROL,(( tmpHqvCtrl &~HQV_FLIP_ODD)|HQV_SW_FLIP),pScrn,pPriv);            
            vfHM(pScrn,pPriv,HW_WAIT_HQV_FINISH);
            vfCMHQVW(HQV_CONTROL,(( tmpHqvCtrl &~HQV_FLIP_ODD)|HQV_FLIP_STATUS),pScrn,pPriv);        

            if(viaGfxInfo->igaInfo[curIGAIndex].igaRRStatus.unit){              
#if VIA_APPLY_XVOVERLAY_AGP
                if(viaGfxInfo->chipId != VIA_DEVICE_VT3314) {
                    END_HEADER5_VIDEOAGP;
                 }
#endif
                viaHWVideoRotation(pScrn, pPriv);
#if VIA_APPLY_XVOVERLAY_AGP
                if(viaGfxInfo->chipId != VIA_DEVICE_VT3314) {
                    BEGIN_HEADER5_VIDEOAGP(16);
                }
#endif    
                rotIndex = pPriv->pRotSurface[curIGAIndex]->cur_idx;
                DBG_DD(("Flip Rot Buffer %d\n", rotIndex));
                
                if (pPriv->vidEngUpdate == VIDEO_1){
                    vfCM(V1_STARTADDR0,pPriv->pRotSurface[curIGAIndex]->offset[rotIndex] + \
                        pOverlayRecord->dwOffset,pScrn,pPriv);
                    vfCM(V_COMPOSE_MODE, (VIDInD(V_COMPOSE_MODE) | V1_COMMAND_FIRE),pScrn,pPriv);     
                    
                }else if (pPriv->vidEngUpdate == VIDEO_3){
                    vfCM(V3_STARTADDR_0,pPriv->pRotSurface[curIGAIndex]->offset[rotIndex]+ \
                        pOverlayRecord->dwOffset,pScrn,pPriv);
                    vfCM(V_COMPOSE_MODE, (VIDInD(V_COMPOSE_MODE) | V3_COMMAND_FIRE),pScrn,pPriv);                       
                }
                vfHM(pScrn,pPriv,HM_WAIT_FIRE);
                
                pPriv->pRotSurface[curIGAIndex]->cur_idx = !rotIndex;                
            }
            break;
    }

#if VIA_APPLY_XVOVERLAY_AGP
    if(viaGfxInfo->chipId != VIA_DEVICE_VT3314){
        END_HEADER5_VIDEOAGP;
        ADVANCE_RING;
    }
#endif 

    DBG_DD(("ovlCME.c: Flip_End \n\n"));
    return TRUE;
}



unsigned long Flip(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;
    unsigned long ret = TRUE;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;   
    unsigned long igaIndex = 0;

    DBG_DD(("current IGA:%ld\n",pPriv->curIGA));
    /* Take speical treatment for Xv Overlay 2 stream path support */
    determineVideoEngineandIGA( pScrn,pPriv);          

    if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_1){                
        if(pPriv->videoFlag & VIDEO_1_SHOW){
            ret = flipOut(pScrn, pPriv);
        }
    }else if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_3){         
        if(pPriv->videoFlag & VIDEO_3_SHOW){
           ret = flipOut(pScrn, pPriv);
        }
    }

    if(((viaGfxInfo->xrandrEnabled == FALSE) && (viaGfxInfo->screenAttr.duoview == TRUE)) ||
        ((viaGfxInfo->xrandrEnabled == TRUE) && (viaGfxInfo->screenAttr.clone == TRUE))){
        
        igaIndex = pPriv->curIGA;
        pPriv->curIGA = (igaIndex==IGA1)? IGA2: IGA1;         
        DBG_DD(("current IGA:%ld\n",pPriv->curIGA));
        
        if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_1){                
            if(pPriv->videoFlag & VIDEO_1_SHOW){
                ret = flipOut(pScrn, pPriv);
            }
        }else if(pVidData->IGA_Video_Map[pPriv->curIGA] == VIDEO_3){         
            if(pPriv->videoFlag & VIDEO_3_SHOW){
               ret = flipOut(pScrn, pPriv);
            }
        }

        pPriv->curIGA = igaIndex;
    }
    
    return ret;
}

#ifdef  VIA_DEFINE_OVERLAY_DISPLAY_PATH
unsigned long updateOvl(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv, viaOvlDisplayPathPtr ovlPath, 
    LPDDUPDATEOVERLAY lpUpdate)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;
    unsigned long ret = TRUE;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;   
    unsigned long igaIndex = 0;

    DBG_DD(("%s\n",__FUNCTION__));   

    DBG_DD(("current IGA:%ld\n", ovlPath->iga));

    pVidData->IGA_Video_Map[ovlPath->iga] = \
        (ovlPath->vidEng == HAVE_VIDEO_3)? VIDEO_3 : VIDEO_1;
    
    /* TMEP SOLUTION !*/
    pPriv->curIGA = ovlPath->iga;
    pPriv->ovlInfo[ovlPath->iga -1] = ovlPath->ovlInfo;
    pPriv->hqvMem[ovlPath->iga -1] = ovlPath->hqvMem;
    pPriv->pHqvSurface[ovlPath->iga -1] = ovlPath->hqvSurf;
    pPriv->rotMem[ovlPath->iga -1] = ovlPath->rotMem;
    pPriv->pRotSurface[ovlPath->iga -1] = ovlPath->rotSurf;
    pPriv->igaRRStatus[ovlPath->iga -1] = ovlPath->igaRRStatus;
    /* TMEP SOLUTION !*/ 

    ret = setupUpdateOverlay(pScrn, pPriv, lpUpdate);
    ret = subSequenceUpdateOverlay(pScrn, pPriv);

    /* TMEP SOLUTION !*/
    ovlPath->ovlInfo = pPriv->ovlInfo[ovlPath->iga -1];
    /* TMEP SOLUTION !*/

    return ret;    
}
unsigned long flipOvl(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv, viaOvlDisplayPathPtr ovlPath)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA pVidData = pVia->pVidData;
    unsigned long ret = TRUE;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;   
    unsigned long igaIndex = 0;

    DBG_DD(("current IGA:%ld\n", ovlPath->iga));

    /* TMEP SOLUTION !*/
    pVidData->IGA_Video_Map[ovlPath->iga] = \
        (ovlPath->vidEng == HAVE_VIDEO_3)? VIDEO_3 : VIDEO_1;
    pPriv->curIGA = ovlPath->iga;
    /* TMEP SOLUTION !*/ 

    ret = flipOut(pScrn, pPriv);

    return ret;
}
#endif

