/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

 
/*
 * I N C L U D E S
 */

/* seems can not get the MACRO of COMPOSITE from config.h, so we define it here by ourselves! */
/* Support Composite Extension */
#define COMPOSITE 1

#include "exa.h"
#include "dixstruct.h"
#include "xf86_OSproc.h"
#include "compiler.h"
#include "xf86PciInfo.h"
#include "via_driver.h"
#include "xf86Pci.h"
#include "xf86fbman.h"
#include "regionstr.h"
#include <time.h>
#include <sys/time.h>
#include <stdio.h>
#include "xf86.h"
#include "xf86xv.h"
#include "Xv.h"

#include "debug.h"      /* for DBG_DD */
#include "via_video.h"
#include "via_eng_regs.h"
#include "via_drm.h"
#include "via_texvideo.h"
#include "xaa.h"
#include "xaalocal.h"
#include "xf86Resources.h"
#include "via_3d.h"
#include "via_rotate.h" /* for rotate feature.*/

static unsigned
timeDiff(struct timeval *now, struct timeval *then)
{
    return (now->tv_usec >= then->tv_usec) ?
	now->tv_usec - then->tv_usec :
	1000000 - (then->tv_usec - now->tv_usec);
}

static void
syncHQV(VIAPtr pVia, unsigned long HQVIndex, unsigned int doSleep)
{
    unsigned char *mmioBase = pVia->MapBase;
    /*
     * Wait for HQV completion. Nothing strange here. We assume that the HQV
     * Handles syncing to the V1 / V3 engines by itself. It should be safe to
     * always wait for SUBPIC_FLIP completion although subpictures are not always
     * used. 
     */
    struct timeval now, then;
    struct timezone here;
    struct timespec sleep, rem;

    sleep.tv_nsec = 1;
    sleep.tv_sec = 0;
    here.tz_minuteswest = 0;
    here.tz_dsttime = 0;
    gettimeofday(&then, &here);
    
    while ((TexVInD(HQV_CONTROL | HQVIndex) & (HQV_SW_FLIP | HQV_SUB_FLIP)))
    {
        gettimeofday(&now, &here);
        if (timeDiff(&now, &then) > VIA_SYNCWAITTIMEOUT) {
	    if ((TexVInD(HQV_CONTROL | HQVIndex) & (HQV_SW_FLIP |HQV_SUB_FLIP))) {
		break;
	    }
	}
	if (doSleep)
	    nanosleep(&sleep, &rem);
    }
}

static void
sync3D(VIAPtr pVia, unsigned int doSleep)
{
    unsigned char *mmioBase = pVia->MapBase;
    struct timeval now, then;
    struct timezone here;
    struct timespec sleep, rem;
    unsigned long  waitflag;

    sleep.tv_nsec = 1;
    sleep.tv_sec = 0;
    here.tz_minuteswest = 0;
    here.tz_dsttime = 0;
    gettimeofday(&then, &here);

    switch(pVia->ChipId)
    {
        case VIA_DEVICE_VT3336:
        case VIA_DEVICE_VT3364:
            waitflag = VIA_3D_ENG_BUSY_H5|VIA_CMD_RGTR_BUSY_H5;
            break;
            
        case VIA_DEVICE_VT3353:
        case VIA_DEVICE_VT3409:
            waitflag = VIA_3D_ENG_BUSY_M1|VIA_CMD_RGTR_BUSY_M1;
            break;
            
        default:
            waitflag = VIA_3D_ENG_BUSY|VIA_CMD_RGTR_BUSY;
    }
    
    while ((TexVInD(VIA_REG_STATUS) & waitflag))
    {
        gettimeofday(&now, &here);
        if (timeDiff(&now, &then) > VIA_SYNCWAITTIMEOUT) {
	    if ((TexVInD(VIA_REG_STATUS) & waitflag)) {
		break;
	    }
	}
	if (doSleep)
	    nanosleep(&sleep, &rem);
    }
}

static int viaTexturedVideoOutSetup_3DPath(VIAPtr  pVia, viaPortPrivPtr pPriv,
    unsigned long src_offset[], unsigned long src_pitch, unsigned long src_format,
    unsigned long dst_offset, unsigned long dst_pitch, unsigned long dst_format,
    unsigned long src_width, unsigned long dst_height,
    ViaXvRectPtr src_rect, ViaXvRectPtr dst_rect,
    unsigned long src_filtermode, RegionPtr clip_region, int dstxoff, int dstyoff)
{
    ScrnInfoPtr pScrn = xf86Screens[(pVia->IsSecondary)? 1 : 0];
    viaTexture3DBltSrcRec texBltSrc;
    viaTexture3DBltDstRec texBltDst;
    viaTexture3DBltRotationRec texBltRotation;
    PVIDDATA  pVidData = pVia->pVidData;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;

    Bool doSleep;

    if(pVia->IsPCI)
    {
        sync3D(pVia, doSleep = 1);
    }

    if(!pPriv->compositeEnable && viaGfxInfo->xrandrEnabled) {
        if((dst_rect->right >= 2048) || (dst_rect->bottom >= 2048)) {
            switch(viaGfxInfo->chipId)
            {
                case VIA_DEVICE_VT3409:
                case VIA_DEVICE_VT3353:
                case VIA_DEVICE_VT3364:
                    /* It seems no need to patch on H5/H6 chips */
                    break;
                            
                case  VIA_DEVICE_VT3314:
                case  VIA_DEVICE_VT3324:
                    /* It should be patched on H2 chips */
                    dstxoff = min(dst_rect->left, 2048) & ~0xFF;
                    dstyoff = min(dst_rect->top, 2048);
                    dst_offset += dstyoff * dst_pitch + dstxoff * RGB_BytePerPixel(dst_format);
                    dstxoff = -dstxoff;
                    dstyoff = -dstyoff;
                    break;

                default:
                    break;
           }
        }
    }
    
    memset(&texBltSrc, 0x00, sizeof(texBltSrc));
    memset(&texBltDst, 0x00, sizeof(texBltDst));
    memset(&texBltRotation, 0x00, sizeof(texBltRotation));

    switch(src_format)
    {
        case FOURCC_YV12:
        case FOURCC_IYUV:
            texBltSrc.format = PIXMAN_yv12;
            break;
        case FOURCC_YUY2:
            texBltSrc.format = PIXMAN_yuy2;
            break;
        /* extend the source format for VDPAU use */
        case FOURCC_RGB555:
            texBltSrc.format = PICT_x1r5g5b5;
            break;
        case FOURCC_RGB565:
            texBltSrc.format = PICT_r5g6b5;
            break;
        case FOURCC_XRGB0888:
            texBltSrc.format = PICT_x8r8g8b8;
            break;
        case FOURCC_ARGB8888:
            texBltSrc.format = PICT_a8r8g8b8;
            break;            
        case FOURCC_VDPAU:     /* FIXME: this line should be removed later! */
            texBltSrc.format = PIXMAN_r8g8b8;
            break;             
        /* extend the source format for VDPAU use */
        default:
            texBltSrc.format = PICT_r5g6b5;
            break;
    }

    switch(dst_format)
    {
        case FOURCC_RGB555:
            texBltDst.format = PICT_x1r5g5b5;
            break;
        case FOURCC_RGB565:
            texBltDst.format = PICT_r5g6b5;
            break;
        case FOURCC_XRGB0888:
            texBltDst.format = PICT_x8r8g8b8;
            break;
        case FOURCC_ARGB8888:
            /* fall through */
        default:
            texBltDst.format = PICT_a8r8g8b8;
            break;
    }

    /* Adjust the Dest Rect and Clip Region for PixMap is not OnScreen case */
    if(dstxoff || dstyoff)
    {
        dst_rect->left += dstxoff;
        dst_rect->top += dstyoff;
        dst_rect->right += dstxoff;
        dst_rect->bottom += dstyoff;
        REGION_TRANSLATE(pScrn->pScreen, clip_region, dstxoff, dstyoff);
    }

    texBltDst.offset = dst_offset;
    texBltDst.pitch = dst_pitch;
    texBltDst.x = dst_rect->left;
    texBltDst.y = dst_rect->top;
    texBltDst.w = (dst_rect->right - dst_rect->left);
    texBltDst.h = (dst_rect->bottom - dst_rect->top);

    texBltSrc.offset = src_offset[0];
    texBltSrc.uoffset = src_offset[1];
    texBltSrc.voffset = src_offset[2];
    texBltSrc.pitch = src_pitch;
    texBltSrc.x = src_rect->left;
    texBltSrc.y = src_rect->top;
    texBltSrc.w = (src_rect->right - src_rect->left);
    texBltSrc.h = (src_rect->bottom - src_rect->top);
    texBltSrc.surfwidth = src_width;
    texBltSrc.surfheight = dst_height;
    texBltSrc.filter = ((texBltSrc.w == texBltDst.w) && (texBltSrc.h == texBltDst.h)) ?
                             via_FilterNearest: via_FilterBilinear;
    
    if(pPriv->xv_refine_using_pcie) {
        texBltSrc.memLoc = LOC_SF;
    } else {
        texBltSrc.memLoc = LOC_SL;
    }

    texBltRotation.rotate = VIA_TO_RR(0);
    texBltRotation.width = 0;
    texBltRotation.height = 0;
    
    if(!pPriv->compositeEnable && !viaGfxInfo->xrandrEnabled) {
        int IgaInuse; 

        if(viaGfxInfo->screenAttr.duoview) {
            IgaInuse = IGA1;
        } else {
            IgaInuse = viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse;
        }
        
        texBltRotation.rotate = VIA_TO_RR(viaGfxInfo->igaInfo[IgaInuse -1].igaRRStatus.unit);
        texBltRotation.width = viaGfxInfo->igaInfo[IgaInuse -1].visible_width;
        texBltRotation.height = viaGfxInfo->igaInfo[IgaInuse -1].visible_height;
    }

    viaAccelTexture3DBlt(pScrn, &texBltSrc, &texBltDst, &texBltRotation, clip_region);

    /* The Clip Region is used for calling DamageDamageRegion(), 
       So we must recover the clip region if it is changed before. 
     */
    /* Recover the Dest Rect and Clip Region for PixMap is not OnScreen case */
    if(dstxoff || dstyoff)
    {
        dst_rect->left -= dstxoff;
        dst_rect->top -= dstyoff;
        dst_rect->right -= dstxoff;
        dst_rect->bottom -= dstyoff;
        REGION_TRANSLATE(pScrn->pScreen, clip_region, -dstxoff, -dstyoff);
    }

    return Success;
}

static void  viaTexVHqvSetSrcFormat(ViaHqvRegPtr pHQVReg, 
    unsigned long src_format)
{
    pHQVReg->dwHQVCtl = HQV_SRC_SW|HQV_ENABLE|
                                   HQV_SW_FLIP|HQV_SINGLE_BUF;
    
    switch (src_format) 
    {
        case FOURCC_YV12 :
        case FOURCC_NV12:       
            pHQVReg->dwHQVCtl |=HQV_YUV420;
            break;
        case FOURCC_YUY2:
        default :
            pHQVReg->dwHQVCtl |= HQV_YUV422;
            break;
    }

    return;
}
static void  viaTexVHqvCalcSrcOffset(ViaHqvRegPtr pHQVReg, 
    ViaXvRectPtr src_rect, unsigned long src_format)
{
    if(src_rect->left || src_rect->top)
    {
        switch (src_format) 
        {
            case FOURCC_YV12 :
            case FOURCC_NV12:       
                pHQVReg->dwSrcOffset =((src_rect->left & ~0x00000001)<<16)|src_rect->top;
                break;
            case FOURCC_YUY2:
            default :
                pHQVReg->dwSrcOffset =(((src_rect->left & ~0x00000001)<<1)<<16)|src_rect->top;
                break;
        }
    }
    else
    {
        pHQVReg->dwSrcOffset = 0;
    }
    
    return;
}
static void  viaTexVHqvCalcSrcDstDataOffset(ViaHqvRegPtr pHQVReg, 
    ViaXvRectPtr src_rect)
{
    pHQVReg->dwSrcDataOffsetStart = 0;
    pHQVReg->dwSrcDataOffsetEnd = (max((src_rect->right -1), 0) << 16) | max((src_rect->bottom - 1), 0);

    /* It is used for VT3409 new HQV interface */    
    pHQVReg->dwDstDataOffsetStart = pHQVReg->dwSrcDataOffsetStart;
    pHQVReg->dwDstDataOffsetEnd = pHQVReg->dwSrcDataOffsetEnd;

    return;
}
static void  viaTexVHqvCalcSrcDstDataOffset_V2(ViaHqvRegPtr pHQVReg, 
    ViaXvRectPtr dst_rect)
{
    pHQVReg->dwSrcDataOffsetStart = (max((dst_rect->left - 1), 0) << 16) | max((dst_rect->top - 1), 0);;
    pHQVReg->dwSrcDataOffsetEnd = (max((dst_rect->right - 1), 0) << 16) | max((dst_rect->bottom - 1), 0);

    /* It is used for VT3409 new HQV interface */    
    pHQVReg->dwDstDataOffsetStart = pHQVReg->dwSrcDataOffsetStart;
    pHQVReg->dwDstDataOffsetEnd = pHQVReg->dwSrcDataOffsetEnd;

    return;
}
static void  viaTexVHqvCalcSrcCountWH(ViaHqvRegPtr pHQVReg, 
    ViaXvRectPtr src_rect, unsigned long src_format)
{
    unsigned long dwHqvFetchPerLine = src_rect->right - src_rect->left;
    unsigned long dwLineCount = src_rect->bottom - src_rect->top;

    if(pHQVReg->dwSrcOffset)
    {
        if(src_rect->left)
            dwHqvFetchPerLine += src_rect->left;

        if(src_rect->top)
            dwLineCount += src_rect->top;
    }
    
    switch(src_format)
    {
        case FOURCC_YUY2:
            dwHqvFetchPerLine <<= 1;
            pHQVReg->dwSrcCountWH = (dwHqvFetchPerLine-1)<<16 | (dwLineCount-1);
            break;
        case FOURCC_YV12:
        case FOURCC_NV12:
        default:
            pHQVReg->dwSrcCountWH = (dwHqvFetchPerLine-1)<<16 | (dwLineCount-1);
            break;
    }

    return;
}
static void  viaTexVHqvCalcHScaleControl(ViaHqvRegPtr pHQVReg,
    ViaXvRectPtr src_rect, ViaXvRectPtr dst_rect)
{
    unsigned long src_width = src_rect->right - src_rect->left;
    unsigned long dst_width = dst_rect->right - dst_rect->left;
    unsigned long src_width1, tmpval;

    if((src_width == 0) ||(dst_width == 0) ||(src_width == dst_width))
    {
        pHQVReg->dwMinizoomCtl &= ~HQV_H_SCALE_ENABLE;
        pHQVReg->dwHScaleCtl &= ~HQV_H_NEWSCALE_ENABLE;
        return;
    }

    if(src_width < dst_width) /* zoom in */
    {
        tmpval = (src_width - 1)*0x0800*0x400 / max((dst_width - 1), 1);
        tmpval = tmpval / 0x400 + ((tmpval&0x3ff)?1:0);
        pHQVReg->dwMinizoomCtl |=(tmpval & 0x7FF)|HQV_H_SCALE_ENABLE|HQV_H_UP_SCALE;

        pHQVReg->dwHScaleCtl = (src_width * 0x1000) / dst_width;
        pHQVReg->dwHScaleCtl = ((pHQVReg->dwHScaleCtl)&0x7FFF)|
                                        HQV_H_NEWSCALE_ENABLE|HQV_H_NEWSCALE_UP;
    }
    else
    {
        tmpval = dst_width*0x0800*0x400 /src_width;
        tmpval = tmpval / 0x400 + ((tmpval&0x3ff)?1:0);
        pHQVReg->dwMinizoomCtl |=(tmpval & 0x7FF)|HQV_H_SCALE_ENABLE|HQV_H_DOWN_SCALE;
        
        pHQVReg->dwExtendCtlColorFormat |= HQV_COLOR_FORMAT_YUV444;

        src_width1 = src_width>>2;

        if( src_width1 >= dst_width)
        {
            pHQVReg->dwHScaleCtl = (src_width * 0x2000) / ((dst_width<<1)+1);
            pHQVReg->dwHScaleCtl = ((pHQVReg->dwHScaleCtl)&0x7FFF)|
                                        HQV_H_NEWSCALE_ENABLE|HQV_H_NEWSCALE_DOWN_DIV4_TO_1;
        }
        else
        {
            src_width1 = src_width << 1;

            if( src_width1 >= dst_width )
            {
                pHQVReg->dwHScaleCtl = (src_width * 0x1000) / dst_width;
                pHQVReg->dwHScaleCtl = ((pHQVReg->dwHScaleCtl)&0x7FFF)|
                                            HQV_H_NEWSCALE_ENABLE|HQV_H_NEWSCALE_DOWN_DIV8_TO_DIV4;
            }
            else
            {
                pHQVReg->dwHScaleCtl = (dst_width * 0x2000) /src_width;
                pHQVReg->dwHScaleCtl = ((pHQVReg->dwHScaleCtl)&0x7FFF)|
                                            HQV_H_NEWSCALE_ENABLE|HQV_H_NEWSCALE_DOWN_UNDER_DIV8;
            }
        }
    }

    return;
}
static void  viaTexVHqvCalcVScaleControl(ViaHqvRegPtr pHQVReg, 
    ViaXvRectPtr src_rect, ViaXvRectPtr dst_rect)
{
    unsigned long src_height = src_rect->bottom - src_rect->top;
    unsigned long dst_height = dst_rect->bottom - dst_rect->top;
    unsigned long tmpval;
    
    if((src_height == 0)||(dst_height == 0)||(src_height == dst_height))
    {
        pHQVReg->dwMinizoomCtl &= ~HQV_V_SCALE_ENABLE;
        pHQVReg->dwVScaleCtl &= ~HQV_V_NEWSCALE_ENABLE;
        return;
    }

    if(src_height < dst_height)
    {
        tmpval = (src_height -1) * 0x0800*0x400 / max((dst_height - 1),1);
        tmpval = tmpval / 0x400 + ((tmpval&0x3ff)?1:0);
        pHQVReg->dwMinizoomCtl |= ((tmpval & 0x7FF)<<16) | 
                            HQV_V_SCALE_ENABLE|HQV_V_UP_SCALE;
    
        pHQVReg->dwVScaleCtl = (src_height * 0x1000) / dst_height;
        pHQVReg->dwVScaleCtl = (pHQVReg->dwVScaleCtl & 0x1FFFF) |
                                          HQV_V_NEWSCALE_ENABLE|HQV_V_NEWSCALE_UP;
    }
    else
    {

        tmpval = dst_height *0x0800*0x400 / src_height;
        tmpval = tmpval / 0x400 + ((tmpval & 0x3ff)?1:0);
        pHQVReg->dwMinizoomCtl |= ((tmpval& 0x7FF)<<16)|
                     HQV_V_MINIFY_ENABLE |HQV_V_DOWN_SCALE;

        pHQVReg->dwHScaleCtl = (src_height * 0x2000) / ((dst_height<<1)+1);
        pHQVReg->dwHScaleCtl = ((pHQVReg->dwHScaleCtl)&0x1FFFF)|
                                    HQV_V_NEWSCALE_ENABLE|HQV_V_NEWSCALE_DOWN;
    }

    return;
}
static void  viaTexVHqvSetCSCControl(ViaHqvRegPtr pHQVReg, 
    unsigned long src_format, unsigned long dst_format)
{
    switch(src_format)
    {
        case FOURCC_YV12:
        case FOURCC_NV12:
        case FOURCC_YUY2:
            
            switch(dst_format)
            {
                case FOURCC_ARGB8888:
                case FOURCC_XRGB0888:
                    pHQVReg->dwCSCTileDeblockingCtl |= HQV_CSC_RGB32|HQV_CSC_ENABLE;
                break;
                
                case FOURCC_RGB555:
                case FOURCC_RGB565:
                    pHQVReg->dwCSCTileDeblockingCtl |= HQV_CSC_RGB16|HQV_CSC_ENABLE;
                break;
            }
            break;

        default:
            break;
    }
    pHQVReg->dwExtendCtlColorFormat = HQV_COLOR_FORMAT_DEFAULT;

    return;
}
static void  viaTexVHqvSetDeinterlace(ViaHqvRegPtr pHQVReg, 
    ViaXvRectPtr src_rect, unsigned long src_filtermode)
{
    unsigned long src_width = src_rect->right - src_rect->left;
    unsigned long src_height = src_rect->bottom - src_rect->top;
    unsigned long imageSize;
    
    if(src_filtermode != TEXVIDEO_SRC_FRAME)
    {
        pHQVReg->dwHQVCtl |= HQV_FIELD_UV | HQV_FIELD_2_FRAME |
                           HQV_FRAME_2_FIELD;
        pHQVReg->dwHQVCtl |= HQV_MOTION_ADAPTIVE_ENABLE|
                           HQV_V_484_LOWPASS_FILTER;

        pHQVReg->dwHQVCtl |= (src_filtermode == TEXVIDEO_SRC_EVENFILED) ? 
                                               HQV_FLIP_EVEN: HQV_FLIP_ODD;

        imageSize = ((src_width * src_height)>>10)& HQV_IMAGE_SIZE_MASK;
        pHQVReg->dwCSCTileDeblockingCtl |= imageSize;
        
        pHQVReg->dwDeinterlaceCtl |=
                                      PULL_DOWN_INTER_FIELD_THRESHOLD_16|                                  
                                      PULL_DOWN_INTRA_FIELD_THRESHOLD_FACTOR_2|
                                      PULL_DOWN_FACTOR2_4_8|
                                      PULL_DOWN_FACTOR1_2_8|
                                      PULL_DOWN_MIN_THRESHOLD_2|
                                      HQV_EDGE_DETECTION_THRESHOLD_16|
                                      HQV_EDGE_DETECTION_ENABLE;    
        pHQVReg->dwDeinterlaceCtl |=
                                      PULL_DOWN_2_2_ENABLE |
                                      PULL_DOWN_3_2_ENABLE;
        pHQVReg->dwDeinterlaceCtl |= PULL_DOWN_23_32_ENABLE;
        pHQVReg->dwPullDownParameter |=
                                    PULL_DOWN_DETECTION_INTER_FIELD_THRESHOLD|
                                    INTER_FIELD_THRESHOLD_CALCULATING_FACTOR|
                                    STATIC_JUDGMENT_NUMBER|
                                    INCREMENT_STATIC_RECORD_FIELD_NUMBER|
                                    PULL_DOWN_FACTOR_2|
                                    PULL_DOWN_FACTOR_1|
                                    PULL_DOWN_DETECTION_THRESHOLD|
                                    MOTION_DETECTION_THRESHOLD |
                                    EDGE_DETECTION_THRESHOLD|
                                    BLOCK_BOUNDARY_MAX_DIFF;
    }

    return;
}

static int viaTexturedVideoOutSetup_HQVPath(VIAPtr  pVia, viaPortPrivPtr pPriv,
    unsigned long src_offset[], unsigned long src_pitch, unsigned long src_format,
    unsigned long dst_offset, unsigned long dst_pitch, unsigned long dst_format,
    unsigned long src_width, unsigned long dst_height,
    ViaXvRectPtr src_rect, ViaXvRectPtr dst_rect,
    unsigned long src_filtermode, RegionPtr clip_region, int dstxoff, int dstyoff)
{
    unsigned char *mmioBase = pVia->MapBase;
    unsigned long HQVIndex;
    ViaHqvRegRec HQVReg, *pHQVReg = &HQVReg;
    Bool newScaleCtl, doSleep;
    
/* Because VT3314 's HQV can not do Scaling up, so Texture video HQV path dose not handle VT3314 case */
    switch(pVia->ChipId)
    {
        case VIA_DEVICE_VT3353:
        case VIA_DEVICE_VT3409:
            newScaleCtl = TRUE;
            HQVIndex = VIA_HQV1_INDEX;
            break;

        case VIA_DEVICE_VT3324:
            newScaleCtl = FALSE;
            HQVIndex = VIA_HQV1_INDEX;
            break;

        default:
            newScaleCtl = FALSE;
            HQVIndex = VIA_HQV0_INDEX;
            break;               
    }

    memset(pHQVReg, 0 , sizeof(ViaHqvRegRec));

    /* Adjust Dest Clip Rect for PixMap is not OnScreen case */
    if(dstxoff || dstyoff)
    {
        dst_rect->left += dstxoff;
        dst_rect->top += dstyoff;
        dst_rect->right += dstxoff;
        dst_rect->bottom += dstyoff;
    }
    
    pHQVReg->dwSrcAddr = src_offset[0];
    pHQVReg->dwSrcUVAddr = src_offset[1];
    pHQVReg->dwSrcPitch = src_pitch;
    
    pHQVReg->dwDstAddr = dst_offset;
    pHQVReg->dwDstPitch = dst_pitch;

    viaTexVHqvSetSrcFormat(pHQVReg, src_format);
    viaTexVHqvCalcSrcOffset(pHQVReg, src_rect, src_format);

#if !(VIA_HQV_OUT_CLIP)
/* HQV dose not handle Dst buffer output image Clip */
    if(dst_rect->top || dst_rect->left)
    {
        pHQVReg->dwDstAddr += dst_pitch * dst_rect->top + 
                 dst_rect->left * RGB_BytePerPixel(dst_format);
    }
    viaTexVHqvCalcSrcDstDataOffset(pHQVReg, src_rect);
#else
    viaTexVHqvCalcSrcDstDataOffset_V2(pHQVReg, dst_rect);
#endif

    viaTexVHqvCalcSrcCountWH(pHQVReg, src_rect, src_format);
    viaTexVHqvCalcHScaleControl(pHQVReg, src_rect, dst_rect);
    viaTexVHqvCalcVScaleControl(pHQVReg, src_rect, dst_rect);
    viaTexVHqvSetCSCControl(pHQVReg, src_format, dst_format);
    viaTexVHqvSetDeinterlace(pHQVReg, src_rect, src_filtermode);

    /* Adjust Dest Clip Rect for PixMap is not OnScreen case */
    if(dstxoff || dstyoff)
    {
        dst_rect->left -= dstxoff;
        dst_rect->top -= dstyoff;
        dst_rect->right -= dstxoff;
        dst_rect->bottom -= dstyoff;
    }
/*---------------------------------------------------------*/
/* Sync HQV engine Idle */
   syncHQV(pVia, HQVIndex, doSleep=1);
/*--------------------------------------------------------*/    
/* Start program the following HQV's registers for Textured Video Out */

    if(HQVIndex)
    {
        TexVOutD(V3_ColorSpaceReg_1, Default_ColorSpaceValue1);
        TexVOutD(V3_ColorSpaceReg_2, Default_ColorSpaceValue2);
    }
    else
    {
        TexVOutD(ColorSpaceReg_1, Default_ColorSpaceValue1);
        TexVOutD(ColorSpaceReg_2, Default_ColorSpaceValue2);
    }

    TexVOutD(HQVIndex|HQV_SRC_OFFSET, pHQVReg->dwSrcOffset);
    TexVOutD(HQVIndex|HQV_SRC_DATA_OFFSET_CONTROL1, pHQVReg->dwSrcDataOffsetStart);
    TexVOutD(HQVIndex|HQV_SRC_DATA_OFFSET_CONTROL3, pHQVReg->dwSrcDataOffsetEnd);
    TexVOutD(HQVIndex|HQV_SRC_FETCH_LINE, pHQVReg->dwSrcCountWH); 

    if(pPriv->xv_refine_using_pcie) {
        TexVOutD(HQVIndex|HQV_SRC_STARTADDR_Y, pHQVReg->dwSrcAddr|LOC_SF);                
    } else {
        TexVOutD(HQVIndex|HQV_SRC_STARTADDR_Y, pHQVReg->dwSrcAddr);                
    }
    
    if(pPriv->xv_refine_using_pcie) {
        TexVOutD(HQVIndex|HQV_SRC_STARTADDR_U, pHQVReg->dwSrcUVAddr|LOC_SF);   
    } else {
        TexVOutD(HQVIndex|HQV_SRC_STARTADDR_U, pHQVReg->dwSrcUVAddr);   
    }
    
    TexVOutD(HQVIndex|HQV_HW_TUNING_PERFORMANCE, pHQVReg->dwPullDownParameter);
    TexVOutD(HQVIndex|HQV_EXTENDED_CONTROL, pHQVReg->dwExtendCtlColorFormat);

    TexVOutD(HQVIndex|HQV_TILE_CSC_CONTROL, pHQVReg->dwCSCTileDeblockingCtl);       
    TexVOutD(HQVIndex|HQV_DEINTERLACE_CONTROL, pHQVReg->dwDeinterlaceCtl);
    TexVOutD(HQVIndex|HQV_DST_STRIDE, pHQVReg->dwDstPitch);                
    TexVOutD(HQVIndex|HQV_SRC_STRIDE, pHQVReg->dwSrcPitch);

    if (newScaleCtl)
    {
        TexVOutD(HQVIndex|HQV_H_SCALE_CONTROL, pHQVReg->dwHScaleCtl);
        TexVOutD(HQVIndex|HQV_V_SCALE_CONTROL, pHQVReg->dwVScaleCtl);
    }
    else
    {
        TexVOutD(HQVIndex|HQV_MINI_CONTROL, pHQVReg->dwMinizoomCtl);                
    }

    TexVOutD(HQVIndex|HQV_DST_STARTADDR0, pHQVReg->dwDstAddr);
    TexVOutD(HQVIndex|HQV_CONTROL, pHQVReg->dwHQVCtl|HQV_END_OF_FRAME);     
    TexVOutD(HQVIndex|HQV_CONTROL, pHQVReg->dwHQVCtl|HQV_SW_FLIP);  

/* Finish program the following HQV's registers for Textured Video Out */    
/*---------------------------------------------------------*/

    return Success;
}

static Bool viaCheckSWRotateCondition(ScrnInfoPtr pScrn, VIAPtr pVia)
{
    PVIDDATA  pVidData = pVia->pVidData;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;

    if(!viaGfxInfo->xrandrEnabled) {
        int IgaInuse; 

        if(viaGfxInfo->screenAttr.duoview) {
            IgaInuse = IGA1;
        }else {
            IgaInuse = viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse;
        }

        /* The sepeical treatment for Non-XRandR SW Rotate path */
        if((viaGfxInfo->screenInfo[pScrn->scrnIndex].rotatetype == VIA_ROTATE_TYPE_SW) &&
            (viaGfxInfo->igaInfo[IgaInuse -1].igaRRStatus.unit != VIA_ROTATE_DEGREE_0)) {
            return TRUE;
        }
    }
    return FALSE;
}

static void viaPatchForNonRandRRotate(ScrnInfoPtr pScrn, VIAPtr pVia, 
    unsigned long * dst_offset, unsigned long * dst_pitch)
{
    PVIDDATA  pVidData = pVia->pVidData;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;
    int Bpp, realWidth;
    int IgaInuse; 

    if(viaGfxInfo->screenAttr.duoview) {
        IgaInuse = IGA1;
    }else {
        IgaInuse = viaGfxInfo->screenInfo[pScrn->scrnIndex].igaInuse;
    }

    /* The sepeical treatment for Non-XRandR SW Rotate path */
    if((viaGfxInfo->screenInfo[pScrn->scrnIndex].rotatetype == VIA_ROTATE_TYPE_SW) &&
        (viaGfxInfo->igaInfo[IgaInuse -1].igaRRStatus.unit != VIA_ROTATE_DEGREE_0)) {
        *dst_offset = viaGfxInfo->screenInfo[pScrn->scrnIndex].fboffset_rot;
    }
    
    /* correct the OnScreen Pixmap pitch value in rotate 90/270 */
    if(viaGfxInfo->igaInfo[IgaInuse -1].igaRRStatus.rotate_90 ||
        viaGfxInfo->igaInfo[IgaInuse -1].igaRRStatus.rotate_270 ) {
        Bpp = viaGfxInfo->screenInfo[pScrn->scrnIndex].bpp;
        realWidth = viaGfxInfo->igaInfo[IgaInuse -1].visible_height;
        /* the pitch aligement is 32 bytes for OnScreen (IGA buffer) */
        *dst_pitch = ALIGN_TO((realWidth * Bpp/8), 32);
    }
    
}

int viaClipVideoToViewPort(ScrnInfoPtr pScrn, short *src_x, short *src_y, 
    short *src_w, short *src_h, short *drw_x, short *drw_y, short *drw_w, short *drw_h, 
    short width, short height, RegionPtr clipBoxes)
{
        BoxRec dstBox;
        INT32 x1, x2, y1, y2;


        /* Clip */
        x1 = *src_x;
        x2 = *src_x + *src_w;
        y1 = *src_y;
        y2 = *src_y + *src_h;

        dstBox.x1 = *drw_x;
        dstBox.x2 = *drw_x + *drw_w;
        dstBox.y1 = *drw_y;
        dstBox.y2 = *drw_y + *drw_h;

        if (!xf86XVClipVideoHelper(&dstBox, (INT32 *)&x1, (INT32 *)&x2, (INT32 *)&y1, (INT32 *)&y2, \
            clipBoxes, width, height))
    	return FALSE;

        if ((x1 >= x2) || (y1 >= y2))
    	return FALSE;

        *src_x = x1 >> 16;
        *src_y = y1 >> 16;
        *src_w = (x2 - x1) >> 16;
        *src_h = (y2 - y1) >> 16;
        *drw_x = dstBox.x1;
        *drw_y = dstBox.y1;
        *drw_w = dstBox.x2 - dstBox.x1;
        *drw_h = dstBox.y2 - dstBox.y1;
        /* Clip */

        return TRUE;
}

Bool viaCheckVideoTargetPixmap(ScrnInfoPtr pScrn, 
    viaPortPrivPtr pPriv, PixmapPtr pPixmap)
{
    ScreenPtr pScreen = pScrn->pScreen;
    VIAPtr  pVia = VIAPTR(pScrn);
    unsigned long onScreenSize;
    unsigned long pPixmap_offset;
    unsigned long pPixmap_pitch;
    Bool nonRandRSwRotate;

    nonRandRSwRotate = viaCheckSWRotateCondition(pScrn, pVia);

#ifdef  VIA_HAVE_EXA
    if (pVia->useEXA && !exaPixmapIsOffscreen(pPixmap)) {
        /* Force the pixmap into framebuffer so we can draw to it. */
        exaMoveInPixmap(pPixmap);
    }
#endif
    /* The sepeical treatment for Non-XRandR SW Rotate path */
    if (!pVia->useEXA && !nonRandRSwRotate &&
      (((char *)pPixmap->devPrivate.ptr < (char *)pVia->FBBase) ||
      ((char *)pPixmap->devPrivate.ptr >= (char *)pVia->FBBase + pVia->videoRambytes))) 
    {
        /* If the pixmap wasn't in framebuffer, then we have no way in XAA to
        * force it there. So, we simply refuse to draw and fail.
        */
        return FALSE;
    }

    onScreenSize =  (pScrn->virtualX * pScrn->virtualY * (pScrn->bitsPerPixel/8));

#ifdef VIA_HAVE_EXA
    if (pVia->useEXA) 
    {
        pPixmap_offset = exaGetPixmapOffset(pPixmap) + pScrn->fbOffset;
        pPixmap_pitch = exaGetPixmapPitch(pPixmap);
    } 
    else
#endif
    {
        pPixmap_offset = (unsigned long)pPixmap->devPrivate.ptr 
                         - (unsigned long)pVia->FBBase + pScrn->fbOffset;
        pPixmap_pitch = (unsigned long)pPixmap->devKind;
    }

    /* The sepeical treatment for Non-XRandR SW Rotate path */
    if(nonRandRSwRotate) {
        pPixmap_offset = pScrn->fbOffset;
    }

   /* whether EXA Composite or 3D Desktop effect enabled ? */
    if((pPixmap_offset == pScrn->fbOffset) &&
#if 1  /* more exactly condition check */
        (pPixmap->drawable.width == pScrn->virtualX) &&
        (pPixmap->drawable.height == pScrn->virtualY) &&
        (pPixmap->drawable.depth == pScrn->depth) &&
        (pPixmap->drawable.bitsPerPixel == pScrn->bitsPerPixel) &&
#endif
        (pPixmap_pitch == (pScrn->displayWidth * pScrn->bitsPerPixel/8))) {
        
        /* 1. exactly in the OnScreen area */
        pPriv->compositeEnable = FALSE;
        
    } else  if(pPixmap_offset < (pScrn->fbOffset + onScreenSize)) {
    
        /* 2. overlap with the OnScreen area */
        pPriv->compositeEnable = FALSE;
        
    } else {
    
        /* 3. out of the OnScreen area */
        pPriv->compositeEnable = TRUE;
        
    }
        
    return TRUE;
}

static int viaAdjustClipRegion(VIAPtr pVia, viaPortPrivPtr pPriv)
{
    ViaXvRectPtr src_rect = &pPriv->src_rect;
    ViaXvRectPtr dst_rect = &pPriv->drw_rect;
    ViaXvSurfacePtr pXvSurface = pPriv->pXvSurface;
    RegionPtr clip_region =&pPriv->clip;
    BoxPtr pbox;
    float xratio, yratio;
    unsigned long width, height;

    if(pPriv->vdp_image_data) /* vdpau path */
    {
        width = pPriv->vdp_image_data->w;
        height = pPriv->vdp_image_data->h;
    } else {
        width = pXvSurface->width;
        height = pXvSurface->height;
    }
    
    pbox = &clip_region->extents;

    xratio = (dst_rect->right - dst_rect->left + 0.1)/(src_rect->right - src_rect->left + 0.1);
    yratio = (dst_rect->bottom - dst_rect->top + 0.1)/(src_rect->bottom - src_rect->top + 0.1);

    dst_rect->left -= (long)(src_rect->left*xratio);
    dst_rect->top -= (long)((src_rect->top)*yratio);
    dst_rect->right += (long)((width - src_rect->right)*xratio);
    dst_rect->bottom += (long)((height - src_rect->bottom)*yratio);

    src_rect->left = 0;
    src_rect->top = 0;
    src_rect->right = width;
    src_rect->bottom = height;

    pbox->x1 = dst_rect->left;
    pbox->y1 = dst_rect->top;
    pbox->x2 = dst_rect->right;
    pbox->y2 = dst_rect->bottom;

    return Success;
}

int viaDisplayTexturedVideo(ScrnInfoPtr pScrn, viaPortPrivPtr pPriv)
{
    VIAPtr  pVia = VIAPTR(pScrn);
    PVIDDATA  pVidData = pVia->pVidData;
    viaGfxInfoPtr viaGfxInfo = pVidData->viaGfxInfo;

    PixmapPtr pPixmap = pPriv->pPixmap;
    ViaXvSurfacePtr pXvSurface = pPriv->pXvSurface;
    ViaXvRectPtr src_rect = &pPriv->src_rect;
    ViaXvRectPtr dst_rect = &pPriv->drw_rect;

    unsigned long src_filtermode, src_format, src_pitch;
    unsigned long src_offset[3]; /* y u v */
    unsigned long dst_format, dst_offset, dst_pitch;
    unsigned long width, height;

    BoxPtr pBox = REGION_RECTS(&pPriv->clip);
    int nBox = REGION_NUM_RECTS(&pPriv->clip);
    int dstxoff, dstyoff, pixel_shift;

#ifdef VIA_HAVE_EXA
    if (pVia->useEXA) 
    {
        dst_offset = exaGetPixmapOffset(pPixmap) + pScrn->fbOffset;
        dst_pitch = exaGetPixmapPitch(pPixmap);
    } 
    else
#endif
    {
        dst_offset = (unsigned long)pPixmap->devPrivate.ptr - (unsigned long)pVia->FBBase 
                        + pScrn->fbOffset;
        dst_pitch = (unsigned long)pPixmap->devKind;
    }

    if(pPriv->compositeEnable) {
        viaAdjustClipRegion(pVia, pPriv);
    } else {
        /* Non-RandR path without Desktop Effect, Dst Pixmap is just the OnScreen surface. */
        if(!viaGfxInfo->xrandrEnabled) {
            viaPatchForNonRandRRotate(pScrn, pVia, &dst_offset, &dst_pitch);
        }
    }
    
    switch (pPixmap->drawable.bitsPerPixel) 
    {
        case 16:
            dst_format = (pPixmap->drawable.depth == 15)?
                FOURCC_RGB555 : FOURCC_RGB565;
            break;

        case 24:
            dst_format = FOURCC_XRGB0888;
            break;
        case 32:
            dst_format = FOURCC_ARGB8888;
            break;

        default:
            return;
    }

    /* it should be related to XV's Deinterlace mode setting */
    src_filtermode = TEXVIDEO_SRC_FRAME;

    if(pPriv->vdp_image_data)
    {
        src_offset[0] = pPriv->vdp_image_data->offset;
        src_pitch = pPriv->vdp_image_data->pitch;
        src_format = pPriv->vdp_image_data->fourcc;
        width = pPriv->vdp_image_data->w;
        height = pPriv->vdp_image_data->h;
        src_offset[1] = 0;
        src_offset[2] = 0;        
    } else {
        src_offset[0] = pXvSurface->offset[pXvSurface->cur_idx];
        src_pitch = pXvSurface->pitch;
        src_format = pXvSurface->format;
        width = pXvSurface->width;
        height = pXvSurface->height;

        switch(pXvSurface->format)
        {
            case FOURCC_YV12:
            case FOURCC_IYUV:
                src_offset[1] = src_offset[0] + pXvSurface->pitch * pXvSurface->height;
                src_offset[2] = src_offset[1] + ((pXvSurface->pitch * pXvSurface->height)>>2);
                break;

            case FOURCC_NV12:
                src_offset[1] = src_offset[0] + pXvSurface->pitch * pXvSurface->height;
                src_offset[2] = 0;
                break;

            case FOURCC_YUY2:
            default:
                src_offset[1] = 0;
                src_offset[2] = 0;
        }
    }

#ifdef COMPOSITE        
    dstxoff = -pPixmap->screen_x + pPixmap->drawable.x;
    dstyoff = -pPixmap->screen_y + pPixmap->drawable.y;
#else
    dstxoff = 0;
    dstyoff = 0;
#endif

    pixel_shift = pPixmap->drawable.bitsPerPixel >> 4;

    if(pPriv->texV3DPath)
    {
        /* texture video go 3D path */
        viaTexturedVideoOutSetup_3DPath(pVia, pPriv, src_offset, src_pitch, src_format,
        dst_offset, dst_pitch, dst_format, width, height, src_rect, dst_rect, src_filtermode, 
        &pPriv->clip, dstxoff, dstyoff);    
    }
    else
    {
        /* texture video go HQV path */
        viaTexturedVideoOutSetup_HQVPath(pVia, pPriv, src_offset, src_pitch, src_format,
        dst_offset, dst_pitch, dst_format, width, height, src_rect, dst_rect, src_filtermode, 
        &pPriv->clip, dstxoff, dstyoff);    
    }

   /*
   * Report that as damaged so it will be redrawn
   */
    if ((pPriv->textured || pPriv->redirected))
    {
        DamageDamageRegion(pPriv->pDraw, &pPriv->clip);
    }

    return Success;
}

/*===========================================================*/
int viaPutImageTexturedG(
    ScrnInfoPtr pScrn,
    short src_x, short src_y,
    short drw_x, short drw_y,
    short src_w, short src_h,
    short drw_w, short drw_h,
    int id, unsigned char* buf,
    short width, short height,
    Bool sync,
    RegionPtr clipBoxes, pointer data, DrawablePtr pDraw)
{
    ScreenPtr pScreen = pScrn->pScreen;
    VIAPtr  pVia = VIAPTR(pScrn);
    viaPortPrivPtr pPriv = (viaPortPrivPtr)data;
    PVIDDATA pVidData = pVia->pVidData;    
    PixmapPtr pPixmap;
    BoxRec dstBox;
    INT32 x1, x2, y1, y2;
    int ret = Success;

    DBG_DD((" via_video.c : viaPutImageTexturedG : called\n"));
    DBG_DD((" via_video.c : Screen[%d]\n", pScrn->scrnIndex));
    DBG_DD((" via_video.c : FourCC=0x%x width=%d height=%d sync=%d\n",id,width,height,sync));
    DBG_DD((" via_video.c : src_x=%d src_y=%d src_w=%d src_h=%d\n",src_x, src_y, src_w, src_h));
    DBG_DD((" via_video.c : drw_x=%d drw_y=%d drw_w=%d drw_h=%d\n",drw_x,drw_y,drw_w,drw_h));


    if((!pPriv->textured && !pPriv->redirected) || 
      (pPriv->xv_portnum >= XV_PORT_NUM_TEXTURE))
    {
        DBG_DD((" via_video.c : XV Port not supported\n"));
        return BadAlloc;
    }

    if (pDraw->type == DRAWABLE_WINDOW) {
        pPixmap = (*pScreen->GetWindowPixmap)((WindowPtr)pDraw);
    } else{
        pPixmap = (PixmapPtr)pDraw;
    }

    /* Patch for the XV client Drawable info DO NOT SYNC with XCompmgr side:
    * Description:
    *  when frequently press F to switch between window mode and Fullscreen mode, in some 
    *  extreme sitution it will cause garbage and screen crashed. 
    */
    drw_x = pDraw->x;
    drw_y = pDraw->y;
    drw_w = pDraw->width;
    drw_h = pDraw->height;
    drw_x = max(drw_x, pPixmap->screen_x);
    drw_y = max(drw_y, pPixmap->screen_y);
    drw_w = min(drw_w, pPixmap->drawable.width);
    drw_h = min(drw_h, pPixmap->drawable.height);
    
    if(!viaClipVideoToViewPort(pScrn, &src_x, &src_y, &src_w, &src_h, \
        &drw_x, &drw_y, &drw_w, &drw_h, width, height, clipBoxes)) {
        return Success;
    }

    if(!viaCheckVideoTargetPixmap(pScrn, pPriv, pPixmap)) {
        return BadAlloc;
    }

    if(id == FOURCC_VDPAU) /* For vdpau path, do not need to create surface */
    {
        pPriv->vdp_image_data =  (viaVdpImageDataPtr)buf;
        pPriv->pXvSurface = (ViaXvSurfacePtr)NULL;
    }
    else
    { 
        pPriv->vdp_image_data =  (viaVdpImageDataPtr)NULL;
        
        if ((pPriv->width != width) || (pPriv->height != height)){
            viaDestroyXvSurface(pScrn, pPriv);
        }

        if(!pPriv->pXvSurface){
            if(viaCreateXvSurface( pScrn, pPriv,id, width, height) != TRUE){
                viaDestroyXvSurface(pScrn, pPriv);
                DBG_DD((" Xv surface memory allocation failure!\n"));
                return BadAlloc;
            }
#ifndef  VIA_DEFINE_OVERLAY_DISPLAY_PATH
            pPriv->skipDataCopy = FALSE;
#endif
        }

        pPriv->pXvSurface->cur_idx = pPriv->framenum%XV_SURFACE_NUM;
        pPriv->framenum++;

#ifdef  VIA_DEFINE_OVERLAY_DISPLAY_PATH
        viaUploadToXVSurface(pScrn, pPriv, id, buf, pPriv->pXvSurface, width, height);
#else
        if(pPriv->skipDataCopy) {
            pPriv->skipDataCopy = FALSE;
        } else {
            viaUploadToXVSurface(pScrn, pPriv, id, buf, pPriv->pXvSurface, width, height);
        }
#endif
    }

     /* update cliplist */
    if (!REGION_EQUAL(pScrn->pScreen, &pPriv->clip, clipBoxes)) {
	REGION_COPY(pScrn->pScreen, &pPriv->clip, clipBoxes);
    }

    if(pPriv->textured || pPriv->redirected)
    {
        pPriv->src_rect.left = src_x;
        pPriv->src_rect.top = src_y;
        pPriv->src_rect.right = src_x + src_w;
        pPriv->src_rect.bottom = src_y + src_h;
        
        pPriv->drw_rect.left  = drw_x;
        pPriv->drw_rect.top = drw_y;
        pPriv->drw_rect.right = drw_x + drw_w;
        pPriv->drw_rect.bottom = drw_y + drw_h;

        pPriv->pDraw = pDraw;
        pPriv->pPixmap = pPixmap;
        ret = viaDisplayTexturedVideo(pScrn, pPriv);
    }
    
    BACKUP_PARAMETERS(pPriv, id, src_x, src_y, drw_x, drw_y, \
        src_w, src_h, drw_w, drw_h, width, height, buf, sync, pDraw);
    
    return ret;
}

