/*
 * Copyright 1998-2008 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2008 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) OR COPYRIGHT HOLDER(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */
#include "via_driver.h"
#include "hw.h"
#include "via_tbl1636.h"
#include "via_vt1636.h"
#include "via_tblDPASetting.h"


void VIAInitLVDS_VT1636(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    int reg_num, i;

    /* Common settings: */
    reg_num = ARRAY_SIZE(COMMON_INIT_TBL_VT1636);

    for (i = 0; i < reg_num; i++)
    {
        Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, COMMON_INIT_TBL_VT1636[i]);
    }

    /* Input Data Mode Select */
    if (pLVDSSettingInfo->IsDualEdge)
    {
        Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, DUAL_CHANNEL_ENABLE_TBL_VT1636[0]);
    }
    else
    {
        Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, SINGLE_CHANNEL_ENABLE_TBL_VT1636[0]);
    }

    if(pLVDSSettingInfo->MsbEnable)
    {
        Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, MSB_DATAFORMAT_OUTPUT[0]);
    }
    else
    {
        Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, LSB_DATAFORMAT_OUTPUT[0]);
    }
    
    if (pLVDSSettingInfo->IsDithering)
    {
        Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, DITHERING_ENABLE_TBL_VT1636[0]);
    }
    else
    {
        Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, DITHERING_DISABLE_TBL_VT1636[0]);
    }
}


void VIAEnableLVDS_VT1636(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    
    Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, VDD_ON_TBL_VT1636[0]);

    /* Pad on: */
    switch (pBIOSInfo->LVDSSettingInfo.DIPort)
    {
        case VIA_DI_DVP0:
        {
            VGAOUT8(0x3C4, SR1E);
            VGAOUT8(0x3C5, VGAIN8(0x3C5) | 0xC0);
            break;
        }
        
        case VIA_DI_DVP1:
        {
            VGAOUT8(0x3C4, SR1E);
            VGAOUT8(0x3C5, VGAIN8(0x3C5) | 0x30);
            break;
        }

        case VIA_DI_DFPLOW:
        {
            VGAOUT8(0x3C4, SR2A);
            VGAOUT8(0x3C5, VGAIN8(0x3C5) | 0x03);
            break;
        }

        case VIA_DI_DFPHIGH:
        {
            VGAOUT8(0x3C4, SR2A);
            VGAOUT8(0x3C5, VGAIN8(0x3C5) | 0x0C);
            break;
        }
       
        
    }
}


void VIADisableLVDS_VT1636(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{
    VIABIOSInfoPtr  pVia = pBIOSInfo;
    
    Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, VDD_OFF_TBL_VT1636[0]);

    /* Pad off: */
    switch (pBIOSInfo->LVDSSettingInfo.DIPort)
    {
        case VIA_DI_DVP0:
        {
            VGAOUT8(0x3C4, SR1E);
            VGAOUT8(0x3C5, VGAIN8(0x3C5) & 0x3F);
            break;
        }

        case VIA_DI_DVP1:
        {
            VGAOUT8(0x3C4, SR1E);
            VGAOUT8(0x3C5, VGAIN8(0x3C5) & 0xCF);
            break;
        }

        case VIA_DI_DFPLOW:
        {
            VGAOUT8(0x3C4, SR2A);
            VGAOUT8(0x3C5, VGAIN8(0x3C5) & 0xFC);
            break;
        }

        case VIA_DI_DFPHIGH:
        {
            VGAOUT8(0x3C4, SR2A);
            VGAOUT8(0x3C5, VGAIN8(0x3C5) & 0xF3);
            break;
        }
        
    }
}


Bool VIALVDSIdentify_VT1636(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo, CARD32 PortNum)
{
    CARD8 Buffer[2];

    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "VIALVDSIdentify_VT1636: Try Port No. 0x%x\n", PortNum));
    
    /* Sense VT1636 LVDS Transmiter */
    pLVDSSettingInfo->I2CPort = PortNum;
    pLVDSSettingInfo->I2CAddress = LVDS_I2C_ADDR_VT1636;

    /* Check vendor ID first: */
    if (!Read_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, 0x00, &(Buffer[0])))
    {
        return FALSE;
    }
    
    if (!Read_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, 0x01, &(Buffer[1])))
    {
        return FALSE;
    }

    if (!((Buffer[0] == 0x06) && (Buffer[1] == 0x11)))
    {        
        return FALSE;        
    }

    /* Vendor ID is correct, so continue to check Chip ID: */
    if (!Read_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, 0x02, &(Buffer[0])))
    {
        return FALSE;
    }
    
    if (!Read_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, 0x03, &(Buffer[1])))
    {
        return FALSE;
    }

    if ((Buffer[0] == 0x45) && (Buffer[1] == 0x33))
    {
        DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED, "Found VIA VT1636 LVDS on port %x!\n", PortNum));
        pLVDSSettingInfo->ChipID = VIA_VT1636;
        return TRUE;
    }

    return FALSE;
}

void VIASaveLVDS_VT1636(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo, VIARegPtr save)
{
    int     i, reg_num;
    CARD8   index;

    /* Save from Tx08 to Tx10: */
    reg_num = ARRAY_SIZE(save->VT1636Regs);
    index = 0x08;
    
    for (i = 0; i < reg_num; i++ )
    {
        Read_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, index, &(save->VT1636Regs[i]));
        index++;
    }
}


void VIARestoreLVDS_VT1636(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo, VIARegPtr restore)
{
    int     i, reg_num;
    CARD8   index;

    /* Restore from Tx08 to Tx10: */
    reg_num = ARRAY_SIZE(restore->VT1636Regs);
    index = 0x08;
    
    for (i = 0; i < reg_num; i++ )
    {
        Write_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, index, restore->VT1636Regs[i]);        
        index++;
    }
}

int VIAGetClkRangeIndex(CARD32 Clk)
{
    if (Clk < DPA_CLK_30M)
    {
        return DPA_CLK_RANGE_30M;
    }
    else if ((DPA_CLK_30M < Clk) && (Clk < DPA_CLK_50M))
    {
        return DPA_CLK_RANGE_30_50M;
    }
    else if ((DPA_CLK_50M < Clk) && (Clk < DPA_CLK_70M))
    {
        return DPA_CLK_RANGE_50_70M;
    }
    else if ((DPA_CLK_70M < Clk) && (Clk < DPA_CLK_100M))
    {
        return DPA_CLK_RANGE_70_100M;
    }
    else if ((DPA_CLK_100M < Clk) && (Clk < DPA_CLK_150M))
    {
        return DPA_CLK_RANGE_100_150M;
    }
    else
    {
        return DPA_CLK_RANGE_150M;
    }
}


void VIASetDPA_VT1636(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo, VT1636DPASETTINGPTR pVT1636DPASetting)
{
    IODATA io_data;

    io_data.Index = 0x09;
    io_data.Mask = 0x1F;
    io_data.Data = pVT1636DPASetting->CLK_SEL_ST1;
    Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, io_data);

    io_data.Index = 0x08;
    io_data.Mask = 0x0F;
    io_data.Data = pVT1636DPASetting->CLK_SEL_ST2;
    Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, io_data);
}


/* Set DPA value to lvds's register. */
void VIASetDPA_LVDS(VIABIOSInfoPtr pBIOSInfo,
                          LVDSSETTINGINFOPTR pLVDSSettingInfo,
                          CARD32* pDPASettingValue)
{
    DEBUG(ErrorF("VIASetDPA_LVDS.\n"));
    IODATA io_data;
        
	while (*pDPASettingValue != 0xFFFFFF)
	{
		io_data.Index = (*pDPASettingValue) & 0xFF;
		io_data.Mask = (*pDPASettingValue >> 8) & 0xFF;
		io_data.Data = (*pDPASettingValue >> 16) & 0xFF;
		Write_Mask_REG_LVDS(pBIOSInfo, pLVDSSettingInfo, io_data);
		DEBUG(ErrorF("io_data.Index = %x.\n", io_data.Index));
		DEBUG(ErrorF("io_data.Mask = %x.\n", io_data.Mask));
		DEBUG(ErrorF("io_data.Data = %x.\n", io_data.Data));
		pDPASettingValue++;
	}
}



/* Load DPA table and fill in the graphic chip's registers.*/
void VIALoadDPASetting_Gfx(VIABIOSInfoPtr pBIOSInfo,
                                  CARD32 TransmitterID,
                                  CARD32 DIPort,
                                  CARD32 CLKRangeIndex)
{    
    DEBUG(xf86DrvMsg(pBIOSInfo->scrnIndex, X_PROBED,
        "VIALoadDPASetting_Gfx, TransmitterID = %d, CLKRangeIndex = %d, DIPort = %d.\n", TransmitterID, CLKRangeIndex, DIPort));    
    
    int i = 0;        
    Bool IsDPATblExist = FALSE;
    GFX_DPA_INFO_TABLE_PTR pGfxDPAInfoTbl;
    GFX_DPA_VALUE_PTR pDPASettingValue;    

    /* To find the record set according to chipset. */
    for (i=0; i<NUM_GFX_DPA_TABLE; i++)
    {        
        if (GFX_DPA_INDEX_TBL[i].ChipSet == pBIOSInfo->Chipset)
        {
            IsDPATblExist = TRUE;
            break;
        }
    }        
    
    /* If the DPA table of the specified chipset havn't build, doing nothing. */
    if (IsDPATblExist)
    {
        /* Find the desired DPA table according to the transmitter type. */
        switch (TransmitterID)
        {
            case VIA_VT1636:
                if (GFX_DPA_INDEX_TBL[i].pVT1636DPATbl)
                {
                    pGfxDPAInfoTbl = GFX_DPA_INDEX_TBL[i].pVT1636DPATbl;
                }
                else
                {
                    return; /* Table doesn't exist. */
                }
                break;
                
            case VIA_Hardwired: /* For Hardwired LCD */
            case VIA_TTLTYPE:   /* For VT1637 LCD    */
                if (GFX_DPA_INDEX_TBL[i].pHardwiredDPATbl)
                {
                    pGfxDPAInfoTbl = GFX_DPA_INDEX_TBL[i].pHardwiredDPATbl;
                }
                else
                {
                    return; /* Table doesn't exist. */
                }
                break; 
                
            default:    /* Table havn't defined. */
                return;
                break;
        }
        
        /* Write the value to the register. */       
        pDPASettingValue = pGfxDPAInfoTbl[CLKRangeIndex].pDPASettingValue;        
        VIASetDPA_Gfx(DIPort, pDPASettingValue);
    }
}



/* Load DPA table and fill in the lvds chip's registers.*/
void VIALoadDPASetting_LVDS(VIABIOSInfoPtr pBIOSInfo, LVDSSETTINGINFOPTR pLVDSSettingInfo)
{ 
    DEBUG(ErrorF("VIALoadDPASetting_LVDS.\n"));
    int i=0;    
    TRANSMITTER_DPA_INFO_PTR pLVDSDPASetting = NULL; 
	CARD32 *pDPASetting = NULL;
    
    /* To find the record set according to chipset. */
    for (i=0; i<NUM_TRANSMITTER_DPA_TABLE; i++)
    {        
        if (TRANSMITTER_DPA_INDEX_TBL[i].ChipSet== pBIOSInfo->Chipset)
            break;        
    }
	if (i == NUM_TRANSMITTER_DPA_TABLE)
		return;
        
    /* Find the desired DPA table according to the transmitter type. */
    switch (pLVDSSettingInfo->ChipID)
    {
        case VIA_VT1636:
            if (TRANSMITTER_DPA_INDEX_TBL[i].pVT1636DPATbl)
                pLVDSDPASetting = TRANSMITTER_DPA_INDEX_TBL[i].pVT1636DPATbl;           
            break;
        default: /* Table havn't defined. */
            break;
    }
	if (!pLVDSDPASetting)
		return;
            
    /* Find the DPA setting value table according to the panel size. */
    while(pLVDSDPASetting->Index != VIA_INVALID)
    {
        if (pLVDSDPASetting->Index == pLVDSSettingInfo->PanelSizeID){
			pDPASetting = pLVDSDPASetting->pTransmitterDPASetting;
			break;
        }
        pLVDSDPASetting++;
    }
	
    if (!pDPASetting)
        return;

    /* Write the value to the register. */        
    VIASetDPA_LVDS(pBIOSInfo, pLVDSSettingInfo, pDPASetting);                        
}


