require "yast"

require "installation/auto_client"
require "bootloader/bootloader_factory"
require "bootloader/autoyast_converter"
require "bootloader/exceptions"
require "bootloader/main_dialog"

Yast.import "AutoInstall"
Yast.import "Bootloader"
Yast.import "BootStorage"
Yast.import "Initrd"
Yast.import "Progress"
Yast.import "PackagesProposal"

module Bootloader
  # Autoyast client for bootloader
  class AutoClient < ::Installation::AutoClient
    include Yast::I18n

    class << self
      attr_accessor :changed
    end

    def run
      progress_orig = Yast::Progress.set(false)
      ret = super
      Yast::Progress.set(progress_orig)

      ret
    end

    def import(data)
      begin
        Yast::Bootloader.Import(data)
      rescue ::Bootloader::UnsupportedBootloader => e
        textdomain "bootloader"
        possible_values = BootloaderFactory.supported_names + [BootloaderFactory::DEFAULT_KEYWORD]
        Yast::AutoInstall.issues_list.add(:invalid_value, "bootloader", "loader_type",
          e.bootloader_name,
          _("The selected bootloader is not supported on this architecture. Possible values: ") +
            possible_values.join(", "),
          :fatal)
        # AutoInstall issues itself will abort import, so do not stop here prematurely.
        return true
      end

      Yast::PackagesProposal.AddResolvables("yast2-bootloader",
        :package, BootloaderFactory.current.packages)

      true
    end

    def summary
      formatted_summary = Yast::Bootloader.Summary.map { |l| "<LI>#{l}</LI>" }

      "<UL>" + formatted_summary.join("\n") + "</UL>"
    end

    def modified?
      self.class.changed
    end

    def modified
      self.class.changed = true
    end

    def reset
      Yast::Bootloader.Reset
    end

    def change
      ::Bootloader::MainDialog.new.run_auto
    end

    # Return configuration data
    #
    # Some of the sections are useless as they're ignored during import.
    # (for example, entries are generated by Grub2 itself).
    #
    # More details can be found in the original pull request at
    # https://github.com/yast/yast-bootloader/pull/272
    #
    # return map or list
    def export
      Yast::Bootloader.Export
    end

    def write
      Yast::Bootloader.Write
    end

    def read
      Yast::Initrd.Read
      Yast::Bootloader.Read
    end
  end
end
