#!/bin/bash

# Copyright (c) 2005 Gentoo Foundation.
# $Id$
# This file is part of the 'eselect' tools framework.
#
# eselect is free software; you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# eselect is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# eselect; if not, write to the Free Software Foundation, Inc., 59 Temple
# Place, Suite 330, Boston, MA  02111-1307  USA

# Colours
COLOUR_LIST_HEADER="\033[1;32m"
COLOUR_LIST_LEFT="\033[0;1m"
COLOUR_LIST_RIGHT="\033[00m"
COLOUR_ERROR="\033[1;31m"
COLOUR_NORMAL="\033[00m"
COLOUR_HI="\033[1;34m"
COLOUR_WARN="\033[1;31m"

# set all colours to COLOURS_NORMAL
nocolours() {
    COLOUR_LIST_HEADER=""
    COLOUR_LIST_LEFT=""
    COLOUR_LIST_RIGHT=""
    COLOUR_ERROR=""
    COLOUR_NORMAL=""
    COLOUR_HI=""
    COLOUR_WARN=""
}

# write_error_msg PUBLIC
# write an error
write_error_msg() {
    echo -e "${COLOUR_ERROR}!!! Error: ${COLOUR_NORMAL}${*}" 1>&2
}

# write_warning_msg PUBLIC
# write a warning
write_warning_msg() {
    echo -e "${COLOUR_WARN}!!! Warning: ${COLOUR_NORMAL}${*}" 1>&2
}

# write_list_start PUBLIC
# Write a list heading. Args may include text highlighting. If -p is passed,
# use 'plain' highlighting.
write_list_start() {
    local colour="${COLOUR_LIST_HEADER}"
    if [[ ${1} == "-p" ]] ; then
        colour=
        shift
    fi
    echo -n -e "${colour}"
    echo -n -e $(apply_text_highlights "${colour}" "$*")
    echo -n -e "${COLOUR_NORMAL}"
    echo
}

# write_kv_list_entry PUBLIC
# Write a key/value list entry with $1 on the left and $2 on the right.
# Args may include text highlighting. If -p is passed, use 'plain'
# highlighting rather than bold.
write_kv_list_entry() {
    local n text left="${COLOUR_LIST_LEFT}" right="${COLOUR_LIST_RIGHT}"
    local key val lindent rindent ifs_save="${IFS}"

    IFS=$' \t\n'

    if [[ ${1} == "-p" ]] ; then
        left=
        right=
        shift
    fi

    lindent=${1%%[^[:space:]]*}
    rindent=${2%%[^[:space:]]*}
    key=${1##*([[:space:]])}
    val=${2##*([[:space:]])}

    echo -n -e "  ${lindent}${left}"
    echo -n -e "$(apply_text_highlights "${left}" "${key}")"
    echo -n -e "${COLOUR_NORMAL} "

    text=${key//\%%%??%%%/}
    n=$(( 25 - ${#text} ))

    # if ${n} is less than or equal to zero then we have a long ${key}
    # that will mess up the formatting of ${val}, so end the line, indent
    # and let ${val} go on the next line.
    if [[ ${n} -le 0 ]] ; then
        n=$(( ${#rindent} + 28 ))
        echo -n -e "\n$(space ${n})${right}"
    else
        space ${n}
        n=$(( ${n} + 2 + ${#text} ))
        # ${val} will already be indented by ${lindent} so only use the
        # difference of ${rindent} and ${lindent} as the right indent
        if [[ ${#rindent} -eq ${#lindent} ]] ; then
            n=$(( ${n} + ${#lindent} ))
        else
            space $(( ${#rindent} - ${#lindent} ))
            n=$(( ${n} + ${#rindent} - ${#lindent} ))
        fi
        echo -n -e "${right}"
    fi

    local cols=$(get_column_width) \
        cwords="$(apply_text_highlights "${right}" "${val}")"

    text=${val//\%%%??%%%/}
    # only loop if it doesn't fit on the same line
    if [[ $(( ${n} + ${#text} )) -ge ${cols} ]] ; then
        local i=0
        rindent="${rindent}$(space 28)"
        cwords=( ${cwords} )
        for text in ${val} ; do
            text=${text//\%%%??%%%/}
            # put the word on the same line if it fits
            if [[ $(( ${n} + ${#text} + 1 )) -lt ${cols} ]] ; then
                echo -n -e "${cwords[i]}"
                n=$(( ${n} + ${#text} ))
                # only append a space if we're not on the last word
                if [[ ${i} -ne ${#cwords} ]] ; then
                    echo -n ' '
                    n=$(( ${n} + 1 ))
                fi
            # otherwise, start a new line and indent
            else
                echo -n -e "\n${rindent}${cwords[i]}"
                n=$(( ${#rindent} + ${#text} ))
                # only append a space if we're not on the last word
                if [[ ${i} -ne ${#cwords} ]] ; then
                    echo -n ' '
                    n=$(( ${n} + 1 ))
                fi
            fi
            i=$(( ${i} + 1 ))
        done
    else
        echo -n -e "${cwords}"
    fi
    echo -e "${COLOUR_NORMAL}"
    IFS="${ifs_save}"
}

# write_numbered_list_entry PUBLIC
# Write out a numbered list entry with index $1 and text $2. Args may
# include text highlighting. If -p is passed, use 'plain' highlighting.
write_numbered_list_entry() {
    local n left="${COLOUR_LIST_LEFT}" right="${COLOUR_LIST_RIGHT}"
    if [[ ${1} == "-p" ]] ; then
        left=""
        right=""
        shift
    fi
    echo -n -e "  ${left}"
    echo -n -e "[$(apply_text_highlights "${left}" "$1")]"
    echo -n -e "${COLOUR_NORMAL}"
    space $(( 4 - ${#1} ))
    echo -n -e "${right}"
    echo -n -e "$(apply_text_highlights "${right}" "$2")"
    echo -n -e "${COLOUR_NORMAL}"
    echo
}

# write_numbered_list PUBLIC
# Write out a numbered list. Args may include text highlighting.
write_numbered_list() {
    local n=1 p=
    if [[ ${1} == "-p" ]] ; then
        p="-p"
        shift
    fi

    while [[ ${#@} -gt 0 ]] ; do
        item=${1}
        shift
        if [[ ${item##*\\} == "" ]] ; then
            item="${item%\\} ${1}"
            shift
        fi
        write_numbered_list_entry ${p} "${n}" "${item}"
        n=$(( ${n} + 1 ))
    done
}

# get_column_width INTERNAL
# Get current column width
get_column_width() {
    if [[ -z "${COLS}" ]] ; then
        COLS=( $(stty size 2>/dev/null) )
        is_number "${COLS[1]}" && COLS=${COLS[1]} || COLS=79
    fi

    echo -n ${COLS}
}

# apply_text_highlights INTERNAL
# Apply text highlights. First arg is the 'restore' colour, second arg
# is the text.
apply_text_highlights() {
    local restore=${1} text=${2}
    text="${text//?%%HI%%%/${COLOUR_HI}}"
    text="${text//?%%WA%%%/${COLOUR_WARN}}"
    text="${text//?%%RE%%%/${restore}}"
    echo -n "${text}"
}

# highlight PUBLIC
# Highlight all arguments. Text highlighting function.
highlight() {
    echo -n -e "%%%HI%%%${*}%%%RE%%%"
}

# highlight_warning PUBLIC
# Highlight all arguments as a warning (red). Text highlighting function.
highlight_warning() {
    echo -n -e "%%%WA%%%${*}%%%RE%%%"
}

# space PUBLIC
# Write $1 numbers of spaces
space() {
    local ret=""
    for (( n = 1 ; n <= ${1} ; ++n )) ; do
        ret="${ret} "
    done
    echo -n "${ret}"
}

# vim: set sw=4 et sts=4 tw=80 :
