#!/bin/sh
##############################################################################
# mkdvd
#
# Take Red Hat Linux/Fedora Core Linux CD images and build a DVD image
#
# Copyright (c) 2004
#   Chris Adams <cmadams@iruntheinter.net>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 2 as published by
# the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#
# History:
# 1.0 - 2004-03-25
#	initial version
#

usage()
{
	cmd=`basename "$0"`
	if [ "$1" != "" ]; then
		echo "$cmd: $1" 1>&2
	fi
	echo "Usage: $cmd <ISO directory> <DVD image>" 1>&2
	echo "    ISO directory: directory holding CD images" 1>&2
	echo "    DVD image: name of DVD image output file" 1>&2
	echo "" 1>&2
	echo "The following RPMs must be installed:" 1>&2
	echo "    $PKGREQ" 1>&2
	exit 1
}

PKGREQ="anaconda-runtime mkisofs"

PATH=/usr/local/bin:/usr/bin:/bin
PATH=$PATH:/usr/local/sbin:/usr/sbin:/sbin

[ "`id -u`" != 0 ] && usage "This must be run as root"

# Check the input directory
start=`/bin/pwd`
base=$1
shift
if [ "$base" = "" -o ! -d "$base" ]; then
	usage "Invalid ISO directory"
fi
if [ ${base#/} = $base ]; then
	base=$start/$base
fi

# Find the output file name
dvd=$1
if [ "$dvd" = "" ]; then
	usage "Invalid DVD image name"
fi
if [ -e $dvd ]; then
	usage "DVD image \"$dvd\" already exists"
fi
if [ ${dvd#/} = $dvd ]; then
	dvd=$base/$dvd
fi

# Make sure necessary tools exist
for p in $PKGREQ; do
	rpm -q $p 2> /dev/null | grep -q $p && continue
	usage "$p not installed"
done

# If the loopback module isn't loaded, load it (with a higher max_loop)
modprobe loop max_loop=16 > /dev/null 2>&1

# Trap errors for cleanup
mnted=""
tree=$base/.tree
trap '
	rm -rf $tree $dvd
	wait
	cd $start
	if [ "$mnted" != "" ]; then
		for d in $mnted; do
			grep -q " $d iso9660" /proc/mounts || continue
			umount $d
			rmdir $d
		done
	fi
    ' ERR 0 2 3 15
set -e

# Mount all the CD images
mnt=0
volid=""
discs=""
for f in $base/*.iso; do
	if [ "$volid" = "" ]; then
		volid=`isoinfo -d -i $f | grep "^Volume id:" | cut -d' ' -f3-`
	fi
	mnt=$(($mnt + 1))
	idir=$base/.disc$mnt
	mnted="$mnted $idir"
	mkdir $idir 
	echo "Mounting `basename $f`"
	mount -o ro,loop -t iso9660 $f $idir
	disc=`tail +4 $idir/.discinfo | head -1`
	discs="$discs $disc"
done
discs=`echo $discs | tr ' ' '\n' | sort -un | tr '\n' , | sed 's/,$//'`

# Make a symlink tree to the CD images
mkdir $tree
for mnt in $mnted; do
	echo "Symlinking to `basename $mnt`"
	cd $mnt
	find . -depth -type d -print | while read d; do
		[ ! -d $tree/$d ] && mkdir -p $tree/$d
		find $d -maxdepth 1 -not -type d -print | while read f; do
			[ -e $tree/$f ] && continue
			ln -s $mnt/$f $tree/$f
		done
	done
done
find $tree -name TRANS.TBL -print | xargs rm -f

# Set up the boot image
cd $tree/isolinux
cp isolinux.bin isolinux.bin.new
rm -f isolinux.bin
mv isolinux.bin.new isolinux.bin
rm -f boot.cat

# Make the full discinfo file
cd $tree
mv .discinfo .discinfo.orig
head -3 .discinfo.orig > .discinfo
echo $discs >> .discinfo
tail +5 .discinfo.orig >> .discinfo
rm .discinfo.orig

# Make the ISO
echo "Building the DVD ISO image \"$dvd\" for \"$volid\""
mkisofs \
    -f \
    -A "$volid" \
    -V "$volid" \
    -J \
    -R \
    -q \
    -T \
    -o $dvd \
    -b isolinux/isolinux.bin \
    -c isolinux/boot.cat \
    -no-emul-boot \
    -boot-load-size 4 \
    -boot-info-table \
    $tree

# Set the MD5SUM in the ISO
echo "Setting DVD MD5 sum"
/usr/lib/anaconda-runtime/implantisomd5 --force $dvd

# Clean up
echo "Cleaning up"
umount $mnted
rm -rf $mnted $tree
trap - ERR 0 2 3 15

