#include <gio/gio.h>
#include <libdbusmenu-glib/server.h>

#include "unity-webapps-action-manager.h"
#include "unity-webapps-interest-manager.h"
#include "unity-webapps-interest-tracker.h"
#include "unity-webapps-window-tracker.h"
#include "unity-webapps-window-tracker-mock.h"
#include "unity-webapps-debug.h"

typedef struct _ActionTestFixture {
  UnityWebappsActionManager *action_manager;
  UnityWebappsInterestManager *interest_manager;
  UnityWebappsInterestTracker *interest_tracker;
  UnityWebappsWindowTrackerMock *window_tracker;
  
  gint interest_one;
  gint interest_two;
  gint interest_three;
} ActionTestFixture;


static void
setup_fixture_no_server_flat_single_interest (ActionTestFixture *fixture, gconstpointer user_data)
{
  UnityWebappsInterestTracker *tracker;
  gchar *menu_path;
  gboolean use_hierarchy, track_activity;
  
  fixture->interest_manager = unity_webapps_interest_manager_new ();
  fixture->interest_tracker = unity_webapps_interest_tracker_new (unity_webapps_window_tracker_get_default (), fixture->interest_manager);
  fixture->action_manager = unity_webapps_action_manager_new_flat (fixture->interest_tracker, NULL);
  fixture->window_tracker = UNITY_WEBAPPS_WINDOW_TRACKER_MOCK (unity_webapps_window_tracker_get_default ());
  unity_webapps_window_tracker_mock_set_active_window (fixture->window_tracker, 1);

  
  fixture->interest_one = unity_webapps_interest_manager_add_interest (fixture->interest_manager,
								       ":1");
  unity_webapps_interest_manager_set_interest_window (fixture->interest_manager, 
						      fixture->interest_one, 1);
  unity_webapps_interest_manager_set_interest_is_active (fixture->interest_manager, 
						     fixture->interest_one, TRUE);
  
  g_object_get (fixture->action_manager, "menu-path", &menu_path,
		"use-hierarchy", &use_hierarchy,
		"track-activity", &track_activity,
		"interest-tracker", &tracker, NULL);
  
  g_assert (menu_path == NULL);
  g_assert (use_hierarchy == FALSE);
  g_assert (track_activity == TRUE);
  g_assert (tracker == fixture->interest_tracker);
}

static void
setup_fixture_no_server_flat_multi_interest (ActionTestFixture *fixture, gconstpointer user_data)
{
  UnityWebappsInterestTracker *tracker;
  gchar *menu_path;
  gboolean use_hierarchy, track_activity;

  fixture->interest_manager = unity_webapps_interest_manager_new ();
  fixture->interest_tracker = unity_webapps_interest_tracker_new (unity_webapps_window_tracker_get_default (), fixture->interest_manager);
  fixture->action_manager = unity_webapps_action_manager_new_flat (fixture->interest_tracker, NULL);
  fixture->window_tracker = UNITY_WEBAPPS_WINDOW_TRACKER_MOCK (unity_webapps_window_tracker_get_default ());
  unity_webapps_window_tracker_mock_set_active_window (fixture->window_tracker, 1);
  
  fixture->interest_one = unity_webapps_interest_manager_add_interest (fixture->interest_manager,
								       ":1");
  unity_webapps_interest_manager_set_interest_window (fixture->interest_manager, 
						      fixture->interest_one, 1);
  unity_webapps_interest_manager_set_interest_is_active (fixture->interest_manager, 
						     fixture->interest_one, TRUE);
  fixture->interest_two = unity_webapps_interest_manager_add_interest (fixture->interest_manager,
								       ":2");
  unity_webapps_interest_manager_set_interest_window (fixture->interest_manager, 
						      fixture->interest_two, 2);
  unity_webapps_interest_manager_set_interest_is_active (fixture->interest_manager, 
						     fixture->interest_two, TRUE);
  fixture->interest_three = unity_webapps_interest_manager_add_interest (fixture->interest_manager,
								       ":3");
  unity_webapps_interest_manager_set_interest_window (fixture->interest_manager, 
						      fixture->interest_three, 3);
  unity_webapps_interest_manager_set_interest_is_active (fixture->interest_manager, 
						     fixture->interest_three, TRUE);

  g_object_get (fixture->action_manager, "menu-path", &menu_path,
		"use-hierarchy", &use_hierarchy,
		"track-activity", &track_activity,
		"interest-tracker", &tracker, NULL);
  
  g_assert (menu_path == NULL);
  g_assert (use_hierarchy == FALSE);
  g_assert (track_activity == TRUE);
  g_assert (tracker == fixture->interest_tracker);
}

static void
setup_fixture_no_server_hierarchal_single_interest (ActionTestFixture *fixture, gconstpointer user_data)
{
  UnityWebappsInterestTracker *tracker;
  gchar *menu_path;
  gboolean use_hierarchy, track_activity;
  
  fixture->interest_manager = unity_webapps_interest_manager_new ();
  fixture->interest_tracker = unity_webapps_interest_tracker_new (unity_webapps_window_tracker_get_default (), fixture->interest_manager);
  fixture->action_manager = unity_webapps_action_manager_new (fixture->interest_tracker, NULL);
  fixture->window_tracker = UNITY_WEBAPPS_WINDOW_TRACKER_MOCK (unity_webapps_window_tracker_get_default ());
  unity_webapps_window_tracker_mock_set_active_window (fixture->window_tracker, 1);

  
  fixture->interest_one = unity_webapps_interest_manager_add_interest (fixture->interest_manager,
								       ":1");
  unity_webapps_interest_manager_set_interest_window (fixture->interest_manager, 
						      fixture->interest_one, 1);
  unity_webapps_interest_manager_set_interest_is_active (fixture->interest_manager, 
						     fixture->interest_one, TRUE);
  
  g_object_get (fixture->action_manager, "menu-path", &menu_path,
		"use-hierarchy", &use_hierarchy,
		"track-activity", &track_activity,
		"interest-tracker", &tracker, NULL);
  
  g_assert (menu_path == NULL);
  g_assert (use_hierarchy == TRUE);
  g_assert (track_activity == TRUE);
  g_assert (tracker == fixture->interest_tracker);
}

static gchar *
print_action_manager (UnityWebappsActionManager *manager)
{
  GVariant *serialized_variant;
  gchar *ret;

  serialized_variant = unity_webapps_action_manager_serialize (manager);
  ret = g_variant_print (serialized_variant, FALSE);
  g_variant_unref (serialized_variant);
  return ret;
}

static void
assert_action_manager_equal (ActionTestFixture *fixture, const gchar *compare_to)
{
  gchar *serialized;

  serialized = print_action_manager (fixture->action_manager);
  g_assert_cmpstr(serialized, ==, compare_to);
  g_free (serialized);
}

#define SINGLE_INTEREST_FLAT_SERIALIZED "([('Foo', 1, true)],)"

static void
test_single_interest_flat (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  
  assert_action_manager_equal (fixture, SINGLE_INTEREST_FLAT_SERIALIZED);
}

#define SINGLE_INTEREST_FLAT_WHITESPACE_SERIALIZED "([('Foo', 1, true)],)"

static void
test_single_interest_flat_whitespace (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "\n   \n Foo\n", fixture->interest_one);
  
  assert_action_manager_equal (fixture, SINGLE_INTEREST_FLAT_SERIALIZED);
}

#define SINGLE_INTEREST_FLAT1_SERIALIZED "([('Bar', 1, true), ('Foo', 1, true)],)"

static void
test_single_interest_flat1 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Bar", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Bar", fixture->interest_one);
  
  assert_action_manager_equal (fixture, SINGLE_INTEREST_FLAT1_SERIALIZED);
}

#define SINGLE_INTEREST_FLAT2_SERIALIZED "([('Bar', 1, true)],)"

static void
test_single_interest_flat2 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Bar", fixture->interest_one);
  unity_webapps_action_manager_remove_action (fixture->action_manager, "Foo", fixture->interest_one);
  
  assert_action_manager_equal (fixture, SINGLE_INTEREST_FLAT2_SERIALIZED);
}

#define SINGLE_INTEREST_FLAT3_SERIALIZED "([],)"

static void
test_single_interest_flat3 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Bar", fixture->interest_one);
  unity_webapps_interest_manager_remove_interest (fixture->interest_manager, fixture->interest_one, FALSE);
  
  assert_action_manager_equal (fixture, SINGLE_INTEREST_FLAT3_SERIALIZED);
}


#define SINGLE_INTEREST_FLAT4_SERIALIZED "([],)"

static void
test_single_interest_flat4 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Bar", fixture->interest_one);
  unity_webapps_action_manager_remove_all_actions (fixture->action_manager, fixture->interest_one);
  
  assert_action_manager_equal (fixture, SINGLE_INTEREST_FLAT3_SERIALIZED);
}

#define MULTI_INTEREST_FLAT0_SERIALIZED "([('Foo', 2, true)],)"

static void
test_multi_interest_flat0 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_two);
  
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT0_SERIALIZED);
}

#define MULTI_INTEREST_FLAT1_SERIALIZED "([('Foo', 1, true)],)"

static void
test_multi_interest_flat1 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_two);
  unity_webapps_action_manager_remove_action (fixture->action_manager, "Foo", fixture->interest_two);
  
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT1_SERIALIZED);
}

#define MULTI_INTEREST_FLAT2_SERIALIZED "([],)"

static void
test_multi_interest_flat2 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_two);
  unity_webapps_action_manager_remove_action (fixture->action_manager, "Foo", fixture->interest_two);
  unity_webapps_action_manager_remove_action (fixture->action_manager, "Foo", fixture->interest_one);
  
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT2_SERIALIZED);
}

#define MULTI_INTEREST_FLAT3_SERIALIZED "([('Foo', 1, true)],)"

static void
test_multi_interest_flat3 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_two);
  unity_webapps_action_manager_remove_action (fixture->action_manager, "Foo", fixture->interest_two);
  unity_webapps_action_manager_remove_action (fixture->action_manager, "Foo", fixture->interest_two);
  
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT3_SERIALIZED);
}

#define MULTI_INTEREST_FLAT4_SERIALIZED_0 "([('Foo', 1, false)],)"
#define MULTI_INTEREST_FLAT4_SERIALIZED_1 "([('Foo', 1, true)],)"

static void
test_multi_interest_flat4 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Bar", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_two);
  unity_webapps_interest_manager_remove_interest (fixture->interest_manager, fixture->interest_one, FALSE);
  
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT4_SERIALIZED_0);
  unity_webapps_window_tracker_mock_set_active_window (fixture->window_tracker, 2);
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT4_SERIALIZED_1);

}

#define MULTI_INTEREST_FLAT5_SERIALIZED "([],)"

static void
test_multi_interest_flat5 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Bar", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_two);
  unity_webapps_interest_manager_remove_interest (fixture->interest_manager, fixture->interest_one, FALSE);
  unity_webapps_interest_manager_remove_interest (fixture->interest_manager, fixture->interest_two, FALSE);
  
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT5_SERIALIZED);
}

#define MULTI_INTEREST_FLAT6_SERIALIZED "([('Bar', 1, true), ('Foo', 1, true)],)"

static void
test_multi_interest_flat6 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Bar", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_two);
  unity_webapps_action_manager_remove_all_actions (fixture->action_manager, fixture->interest_two);
  
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT6_SERIALIZED);
}

#define MULTI_INTEREST_FLAT7_SERIALIZED "([('Bar', 1, false), ('Foo', 1, true)],)"

static void
test_multi_interest_flat7 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Bar", fixture->interest_two);
  
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT7_SERIALIZED);
}

#define MULTI_INTEREST_FLAT8_SERIALIZED "([('Foo', 2, true)],)"

static void
test_multi_interest_flat8 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "\nFoo", fixture->interest_two);
  
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT8_SERIALIZED);
}

#define MULTI_INTEREST_FLAT9_SERIALIZED "([('Foo', 2, false)],)"

static void
test_multi_interest_flat9 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_two);
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_three);
  
  assert_action_manager_equal (fixture, MULTI_INTEREST_FLAT9_SERIALIZED);
}

#define SINGLE_INTEREST_HIERARCHAL_SERIALIZED "([('/Foo', 1, true), ('/Foo/Bar', 1, true)],)"

static void
test_single_interest_hierarchal (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "/Foo/Bar", fixture->interest_one);
  
  assert_action_manager_equal (fixture, SINGLE_INTEREST_HIERARCHAL_SERIALIZED);
}

#define SINGLE_INTEREST_HIERARCHAL1_SERIALIZED "([('/Foo', 2, true), ('/Foo/Bar', 1, true)],)"

static void
test_single_interest_hierarchal1 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "/Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "/Foo/Bar", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "/Foo/Bar", fixture->interest_one);
  
  assert_action_manager_equal (fixture, SINGLE_INTEREST_HIERARCHAL1_SERIALIZED);
}

#define SINGLE_INTEREST_HIERARCHAL2_SERIALIZED "([],)"

static void
test_single_interest_hierarchal2 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "%Foo/Bar", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "/Foo/Bar/Baz/This/Is/So/Long", fixture->interest_one);
  
  assert_action_manager_equal (fixture, SINGLE_INTEREST_HIERARCHAL2_SERIALIZED);
}

#define SINGLE_INTEREST_HIERARCHAL3_SERIALIZED "([('/Foo', 3, true), ('/Foo/Bar', 2, true), ('/Foo/Bar/Baz', 1, true), ('/Foo/Bar/Boy', 1, true)],)"

static void
test_single_interest_hierarchal3 (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_action_manager_add_action (fixture->action_manager, "/Foo", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "/Foo/Bar/Baz", fixture->interest_one);
  unity_webapps_action_manager_add_action (fixture->action_manager, "/Foo/Bar/Boy", fixture->interest_one);
  
  assert_action_manager_equal (fixture, SINGLE_INTEREST_HIERARCHAL3_SERIALIZED);
}

static void
teardown_fixture_single_interest (ActionTestFixture *fixture, gconstpointer user_data)
{
  unity_webapps_interest_manager_remove_interest (fixture->interest_manager,
						  fixture->interest_one, FALSE);
  assert_action_manager_equal (fixture, "([],)");
  g_object_unref (G_OBJECT (fixture->action_manager));
  g_object_unref (G_OBJECT (fixture->interest_manager));
  g_object_unref (G_OBJECT (fixture->interest_tracker));
}

static void
teardown_fixture (ActionTestFixture *fixture, gconstpointer user_data)
{
  g_object_unref (G_OBJECT (fixture->action_manager));
  g_object_unref (G_OBJECT (fixture->interest_manager));
  g_object_unref (G_OBJECT (fixture->interest_tracker));
}

int 
main (int argc, char **argv)
{
  g_type_init ();
  g_test_init (&argc, &argv, NULL);
  
  unity_webapps_debug_initialize_flags ();
  
  g_test_add("/Actions/Flat/SingleInterestAddOne", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_single_interest, test_single_interest_flat, teardown_fixture_single_interest);
  g_test_add("/Actions/Flat/SingleInterestWhitespace", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_single_interest, test_single_interest_flat_whitespace, teardown_fixture_single_interest);
  g_test_add("/Actions/Flat/SingleInterestAddTwo", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_single_interest, test_single_interest_flat1, teardown_fixture_single_interest);
  g_test_add("/Actions/Flat/SingleInterestAddTwoRemoveOne", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_single_interest, test_single_interest_flat2, teardown_fixture_single_interest);
  g_test_add("/Actions/Flat/SingleInterestAddTwoRemoveInterest", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_single_interest, test_single_interest_flat3, teardown_fixture_single_interest);
  g_test_add("/Actions/Flat/SingleInterestAddTwoRemoveAll", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_single_interest, test_single_interest_flat4, teardown_fixture_single_interest);
  g_test_add("/Actions/Flat/MultiInterest/AddOneTwice", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_multi_interest, test_multi_interest_flat0, teardown_fixture);
  g_test_add("/Actions/Flat/MultiInterest/AddOneTwiceRemoveOnce", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_multi_interest, test_multi_interest_flat1, teardown_fixture);
  g_test_add("/Actions/Flat/MultiInterest/AddOneTwiceRemoveTwice", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_multi_interest, test_multi_interest_flat2, teardown_fixture);
  g_test_add("/Actions/Flat/MultiInterest/AddOneTwiceRemoveTwiceSameOwner", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_multi_interest, test_multi_interest_flat3, teardown_fixture);
  g_test_add("/Actions/Flat/MultiInterest/AddTwoTwiceRemoveOneInterest", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_multi_interest, test_multi_interest_flat4, teardown_fixture);
  g_test_add("/Actions/Flat/MultiInterest/AddTwoTwiceRemoveBothInterest", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_multi_interest, test_multi_interest_flat5, teardown_fixture);
  g_test_add("/Actions/Flat/MultiInterest/AddManyTwiceRemoveAllOnce", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_multi_interest, test_multi_interest_flat6, teardown_fixture);
  g_test_add("/Actions/Flat/MultiInterest/AddTwoOneInvisible", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_multi_interest, test_multi_interest_flat7, teardown_fixture);
  
  g_test_add("/Actions/Flat/MultiInterest/PathNormalization", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_multi_interest, test_multi_interest_flat8, teardown_fixture);
  g_test_add("/Actions/Flat/MultiInterest/AddHidden", ActionTestFixture, NULL,
	     setup_fixture_no_server_flat_multi_interest, test_multi_interest_flat9, teardown_fixture);
	     

  g_test_add("/Actions/Hierarchal/SingleInterestAddOne", ActionTestFixture, NULL,
	     setup_fixture_no_server_hierarchal_single_interest, test_single_interest_hierarchal, teardown_fixture_single_interest);
  g_test_add("/Actions/Hierarchal/SingleInterestAddTwoManyTimes", ActionTestFixture, NULL,
	     setup_fixture_no_server_hierarchal_single_interest, test_single_interest_hierarchal1, teardown_fixture_single_interest);
  g_test_add("/Actions/Hierarchal/SingleInterestInvalidActions", ActionTestFixture, NULL,
	     setup_fixture_no_server_hierarchal_single_interest, test_single_interest_hierarchal2, teardown_fixture_single_interest);
  g_test_add("/Actions/Hierarchal/SingleInterestDeepActions", ActionTestFixture, NULL,
	     setup_fixture_no_server_hierarchal_single_interest, test_single_interest_hierarchal3, teardown_fixture_single_interest);

  
  return g_test_run ();
}
