#!/usr/bin/env python

CopyRight = '''
/**************************************************************************
 *
 * Copyright 2010 VMware, Inc.
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL VMWARE AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 **************************************************************************/

/**
 * @file
 * SRGB translation.
 *
 * @author Brian Paul <brianp@vmware.com>
 * @author Michal Krol <michal@vmware.com>
 * @author Jose Fonseca <jfonseca@vmware.com>
 */
'''


import math
import struct


def srgb_to_linear(x):
    if x <= 0.04045:
        return x / 12.92
    else:
        return math.pow((x + 0.055) / 1.055, 2.4)


def linear_to_srgb(x):
    if x >= 0.0031308:
        return 1.055 * math.pow(x, 0.41666666) - 0.055
    else:
        return 12.92 * x


def generate_srgb_tables():
    print 'const float'
    print 'util_format_srgb_8unorm_to_linear_float_table[256] = {'
    for j in range(0, 256, 4):
        print '   ',
        for i in range(j, j + 4):
            print '%.7e,' % (srgb_to_linear(i / 255.0),),
        print
    print '};'
    print
    print 'const uint8_t'
    print 'util_format_srgb_to_linear_8unorm_table[256] = {'
    for j in range(0, 256, 16):
        print '   ',
        for i in range(j, j + 16):
            print '%3u,' % (int(srgb_to_linear(i / 255.0) * 255.0 + 0.5),),
        print
    print '};'
    print
    print 'const uint8_t'
    print 'util_format_linear_to_srgb_8unorm_table[256] = {'
    for j in range(0, 256, 16):
        print '   ',
        for i in range(j, j + 16):
            print '%3u,' % (int(linear_to_srgb(i / 255.0) * 255.0 + 0.5),),
        print
    print '};'
    print

# calculate the table interpolation values used in float linear to unorm8 srgb
    numexp = 13
    mantissa_msb = 3
# stepshift is just used to only use every x-th float to make things faster,
# 5 is largest value which still gives exact same table as 0
    stepshift = 5
    nbuckets = numexp << mantissa_msb
    bucketsize = (1 << (23 - mantissa_msb)) >> stepshift
    mantshift = 12
    valtable = []
    sum_aa = float(bucketsize)
    sum_ab = 0.0
    sum_bb = 0.0
    for i in range(0, bucketsize):
        j = (i << stepshift) >> mantshift
        sum_ab += j
        sum_bb += j*j
    inv_det = 1.0 / (sum_aa * sum_bb - sum_ab * sum_ab)

    for bucket in range(0, nbuckets):
        start = ((127 - numexp) << 23) + bucket*(bucketsize << stepshift)
        sum_a = 0.0
        sum_b = 0.0
 
        for i in range(0, bucketsize):
            j = (i << stepshift) >> mantshift
            fint = start + (i << stepshift)
            ffloat = struct.unpack('f', struct.pack('I', fint))[0]
            val = linear_to_srgb(ffloat) * 255.0 + 0.5
            sum_a += val
            sum_b += j*val

        solved_a = inv_det * (sum_bb*sum_a - sum_ab*sum_b)
        solved_b = inv_det * (sum_aa*sum_b - sum_ab*sum_a)

        scaled_a = solved_a * 65536.0 / 512.0
        scaled_b = solved_b * 65536.0
 
        int_a = int(scaled_a + 0.5)
        int_b = int(scaled_b + 0.5)

        valtable.append((int_a << 16) + int_b)

    print 'const unsigned'
    print 'util_format_linear_to_srgb_helper_table[104] = {'

    for j in range(0, nbuckets, 4):
        print '   ',
        for i in range(j, j + 4):
            print '0x%08x,' % (valtable[i],),
        print
    print '};'
    print

def main():
    print '/* This file is autogenerated by u_format_srgb.py. Do not edit directly. */'
    print
    # This will print the copyright message on the top of this file
    print CopyRight.strip()
    print
    print '#include "u_format_srgb.h"'
    print
    generate_srgb_tables()    


if __name__ == '__main__':
    main()
