package test.dr.inference.trace;

import dr.evolution.alignment.SimpleAlignment;
import dr.evolution.coalescent.CoalescentSimulator;
import dr.evolution.coalescent.ConstantPopulation;
import dr.evolution.datatype.DataType;
import dr.evolution.io.NewickImporter;
import dr.evolution.sequence.Sequence;
import dr.evolution.tree.Tree;
import dr.evolution.tree.SimpleNode;
import dr.evolution.tree.SimpleTree;
import dr.evolution.util.Date;
import dr.evolution.util.Taxon;
import dr.evolution.util.Units;
import dr.evomodel.coalescent.ConstantPopulationModel;
import dr.evomodel.tree.TreeModel;
import dr.evomodelxml.tree.TreeModelParser;
import dr.inference.model.Parameter;
import dr.inference.trace.TraceCorrelation;
import test.dr.math.MathTestCase;

/**
 * @author Alexei Drummond
 */
public class TraceCorrelationAssert extends MathTestCase {

    protected static final String TREE_HEIGHT = TreeModel.TREE_MODEL + "." + TreeModelParser.ROOT_HEIGHT;

    protected TreeModel treeModel;
    protected SimpleAlignment alignment;
    protected Taxon[] taxa;


    public TraceCorrelationAssert(String name) {
        super(name);
    }
       

    protected void createAlignment(Object[][] taxa_sequence, DataType dataType) {

        alignment = new SimpleAlignment();
        alignment.setDataType(dataType);
//        alignment.setDataType(Nucleotides.INSTANCE);

        taxa = new Taxon[taxa_sequence[0].length]; // 6, 17
        System.out.println("Taxon len = " + taxa_sequence[0].length);
        System.out.println("Alignment len = " + taxa_sequence[1].length);
        if (taxa_sequence.length > 2) System.out.println("Date len = " + taxa_sequence[2].length);                          

        for (int i=0; i < taxa_sequence[0].length; i++) {
            taxa[i] = new Taxon(taxa_sequence[0][i].toString());

            if (taxa_sequence.length > 2) {
                Date date = new Date((Double) taxa_sequence[2][i], Units.Type.YEARS, (Boolean) taxa_sequence[3][0]);
                taxa[i].setDate(date);
            }

            //taxonList.addTaxon(taxon);
            Sequence sequence = new Sequence(taxa_sequence[1][i].toString());
            sequence.setTaxon(taxa[i]);
            sequence.setDataType(dataType);

            alignment.addSequence(sequence);
        }
    }

    protected void createRandomInitialTree(double popSize) throws Exception {
        ConstantPopulation constant = new ConstantPopulation(Units.Type.YEARS);
        constant.setN0(popSize); // popSize

        createTreeModel(constant);
    }

    protected ConstantPopulationModel createRandomInitialTree(Parameter popSize) {        
        ConstantPopulationModel startingTree = new ConstantPopulationModel(popSize, Units.Type.YEARS);
        ConstantPopulation constant = (ConstantPopulation) startingTree.getDemographicFunction();

        createTreeModel(constant);

        return startingTree;
    }

    private void createTreeModel (ConstantPopulation constant) {
        CoalescentSimulator simulator = new CoalescentSimulator();
        Tree tree = simulator.simulateTree(alignment, constant);
        treeModel = new TreeModel(tree);//treeModel
    }

    protected void createSpecifiedTree(String t) throws Exception {
//        Tree.Utils.newick(tree)
        //create tree
        NewickImporter importer = new NewickImporter(t);
        Tree tree = importer.importTree(null);          

        treeModel = new TreeModel(tree);//treeModel
    }

    /**
     * Asserts that the given trace correlation stats are not significantly different to
     * provided expectation value
     *
     * @param name  the name of the statistic
     * @param stats the summary of the MCMC trace of the statistic
     * @param v     the expected value for the statistic
     */
    protected void assertExpectation(String name, TraceCorrelation stats, double v) {
        double mean = stats.getMean();
        double stderr = stats.getStdErrorOfMean();
        double upper = mean + 2 * stderr;
        double lower = mean - 2 * stderr;

        assertTrue("Expected " + name + " is " + v + " but got " + mean + " +/- " + stderr,
                upper > v && lower < v);
    }

    //************************** data ****************************
    protected TreeModel createPrimateTreeModel () {

        SimpleNode[] nodes = new SimpleNode[10];
        for (int n=0; n < 10; n++) {
            nodes[n] = new SimpleNode();
        }

//        nodes[0].setHeight(0);
        nodes[0].setTaxon(taxa[0]); // human

        nodes[1].setTaxon(taxa[1]); // chimp

        nodes[2].setTaxon(taxa[2]); // bonobo

        nodes[3].setHeight(0.010772);
        nodes[3].addChild(nodes[1]);
        nodes[3].addChild(nodes[2]);

        nodes[4].setHeight(0.024003);
        nodes[4].addChild(nodes[0]);
        nodes[4].addChild(nodes[3]);

        nodes[5].setTaxon(taxa[3]); // gorilla

        nodes[6].setHeight(0.036038);
        nodes[6].addChild(nodes[4]);
        nodes[6].addChild(nodes[5]);

        nodes[7].setTaxon(taxa[4]); // orangutan

        nodes[8].setHeight(0.069125);
        nodes[8].addChild(nodes[6]);
        nodes[8].addChild(nodes[7]);

        nodes[9].setTaxon(taxa[5]); // siamang

        SimpleNode root = new SimpleNode();
        root.setHeight(0.099582);
        root.addChild(nodes[8]);
        root.addChild(nodes[9]);

        Tree tree = new SimpleTree(root);
        tree.setUnits(Units.Type.YEARS);

        return new TreeModel(tree); //treeModel
    }

    protected static final String[][] PRIMATES_TAXON_SEQUENCE = {{"human", "chimp", "bonobo", "gorilla", "orangutan", "siamang"},
          {"AGAAATATGTCTGATAAAAGAGTTACTTTGATAGAGTAAATAATAGGAGCTTAAACCCCCTTATTTCTACTAGGACTATGAGAATCGAACCCATCCCTGAGAATCCAAAATTCTCCGTGCCACCTATCACACCCCATCCTAAGTAAGGTCAGCTAAATAAGCTATCGGGCCCATACCCCGAAAATGTTGGTTATACCCTTCCCGTACTAAGAAATTTAGGTTAAATACAGACCAAGAGCCTTCAAAGCCCTCAGTAAGTTG-CAATACTTAATTTCTGTAAGGACTGCAAAACCCCACTCTGCATCAACTGAACGCAAATCAGCCACTTTAATTAAGCTAAGCCCTTCTAGACCAATGGGACTTAAACCCACAAACACTTAGTTAACAGCTAAGCACCCTAATCAAC-TGGCTTCAATCTAAAGCCCCGGCAGG-TTTGAAGCTGCTTCTTCGAATTTGCAATTCAATATGAAAA-TCACCTCGGAGCTTGGTAAAAAGAGGCCTAACCCCTGTCTTTAGATTTACAGTCCAATGCTTCA-CTCAGCCATTTTACCACAAAAAAGGAAGGAATCGAACCCCCCAAAGCTGGTTTCAAGCCAACCCCATGGCCTCCATGACTTTTTCAAAAGGTATTAGAAAAACCATTTCATAACTTTGTCAAAGTTAAATTATAGGCT-AAATCCTATATATCTTA-CACTGTAAAGCTAACTTAGCATTAACCTTTTAAGTTAAAGATTAAGAGAACCAACACCTCTTTACAGTGA",
           "AGAAATATGTCTGATAAAAGAATTACTTTGATAGAGTAAATAATAGGAGTTCAAATCCCCTTATTTCTACTAGGACTATAAGAATCGAACTCATCCCTGAGAATCCAAAATTCTCCGTGCCACCTATCACACCCCATCCTAAGTAAGGTCAGCTAAATAAGCTATCGGGCCCATACCCCGAAAATGTTGGTTACACCCTTCCCGTACTAAGAAATTTAGGTTAAGCACAGACCAAGAGCCTTCAAAGCCCTCAGCAAGTTA-CAATACTTAATTTCTGTAAGGACTGCAAAACCCCACTCTGCATCAACTGAACGCAAATCAGCCACTTTAATTAAGCTAAGCCCTTCTAGATTAATGGGACTTAAACCCACAAACATTTAGTTAACAGCTAAACACCCTAATCAAC-TGGCTTCAATCTAAAGCCCCGGCAGG-TTTGAAGCTGCTTCTTCGAATTTGCAATTCAATATGAAAA-TCACCTCAGAGCTTGGTAAAAAGAGGCTTAACCCCTGTCTTTAGATTTACAGTCCAATGCTTCA-CTCAGCCATTTTACCACAAAAAAGGAAGGAATCGAACCCCCTAAAGCTGGTTTCAAGCCAACCCCATGACCTCCATGACTTTTTCAAAAGATATTAGAAAAACTATTTCATAACTTTGTCAAAGTTAAATTACAGGTT-AACCCCCGTATATCTTA-CACTGTAAAGCTAACCTAGCATTAACCTTTTAAGTTAAAGATTAAGAGGACCGACACCTCTTTACAGTGA",
           "AGAAATATGTCTGATAAAAGAATTACTTTGATAGAGTAAATAATAGGAGTTTAAATCCCCTTATTTCTACTAGGACTATGAGAGTCGAACCCATCCCTGAGAATCCAAAATTCTCCGTGCCACCTATCACACCCCATCCTAAGTAAGGTCAGCTAAATAAGCTATCGGGCCCATACCCCGAAAATGTTGGTTATACCCTTCCCGTACTAAGAAATTTAGGTTAAACACAGACCAAGAGCCTTCAAAGCTCTCAGTAAGTTA-CAATACTTAATTTCTGTAAGGACTGCAAAACCCCACTCTGCATCAACTGAACGCAAATCAGCCACTTTAATTAAGCTAAGCCCTTCTAGATTAATGGGACTTAAACCCACAAACATTTAGTTAACAGCTAAACACCCTAATCAGC-TGGCTTCAATCTAAAGCCCCGGCAGG-TTTGAAGCTGCTTCTTTGAATTTGCAATTCAATATGAAAA-TCACCTCAGAGCTTGGTAAAAAGAGGCTTAACCCCTGTCTTTAGATTTACAGTCCAATGCTTCA-CTCAGCCATTTTACCACAAAAAAGGAAGGAATCGAACCCCCTAAAGCTGGTTTCAAGCCAACCCCATGACCCCCATGACTTTTTCAAAAGATATTAGAAAAACTATTTCATAACTTTGTCAAAGTTAAATTACAGGTT-AAACCCCGTATATCTTA-CACTGTAAAGCTAACCTAGCATTAACCTTTTAAGTTAAAGATTAAGAGGACCAACACCTCTTTACAGTGA",
           "AGAAATATGTCTGATAAAAGAGTTACTTTGATAGAGTAAATAATAGAGGTTTAAACCCCCTTATTTCTACTAGGACTATGAGAATTGAACCCATCCCTGAGAATCCAAAATTCTCCGTGCCACCTGTCACACCCCATCCTAAGTAAGGTCAGCTAAATAAGCTATCGGGCCCATACCCCGAAAATGTTGGTCACATCCTTCCCGTACTAAGAAATTTAGGTTAAACATAGACCAAGAGCCTTCAAAGCCCTTAGTAAGTTA-CAACACTTAATTTCTGTAAGGACTGCAAAACCCTACTCTGCATCAACTGAACGCAAATCAGCCACTTTAATTAAGCTAAGCCCTTCTAGATCAATGGGACTCAAACCCACAAACATTTAGTTAACAGCTAAACACCCTAGTCAAC-TGGCTTCAATCTAAAGCCCCGGCAGG-TTTGAAGCTGCTTCTTCGAATTTGCAATTCAATATGAAAT-TCACCTCGGAGCTTGGTAAAAAGAGGCCCAGCCTCTGTCTTTAGATTTACAGTCCAATGCCTTA-CTCAGCCATTTTACCACAAAAAAGGAAGGAATCGAACCCCCCAAAGCTGGTTTCAAGCCAACCCCATGACCTTCATGACTTTTTCAAAAGATATTAGAAAAACTATTTCATAACTTTGTCAAGGTTAAATTACGGGTT-AAACCCCGTATATCTTA-CACTGTAAAGCTAACCTAGCGTTAACCTTTTAAGTTAAAGATTAAGAGTATCGGCACCTCTTTGCAGTGA",
           "AGAAATATGTCTGACAAAAGAGTTACTTTGATAGAGTAAAAAATAGAGGTCTAAATCCCCTTATTTCTACTAGGACTATGGGAATTGAACCCACCCCTGAGAATCCAAAATTCTCCGTGCCACCCATCACACCCCATCCTAAGTAAGGTCAGCTAAATAAGCTATCGGGCCCATACCCCGAAAATGTTGGTTACACCCTTCCCGTACTAAGAAATTTAGGTTA--CACAGACCAAGAGCCTTCAAAGCCCTCAGCAAGTCA-CAGCACTTAATTTCTGTAAGGACTGCAAAACCCCACTTTGCATCAACTGAGCGCAAATCAGCCACTTTAATTAAGCTAAGCCCTCCTAGACCGATGGGACTTAAACCCACAAACATTTAGTTAACAGCTAAACACCCTAGTCAAT-TGGCTTCAGTCCAAAGCCCCGGCAGGCCTTAAAGCTGCTCCTTCGAATTTGCAATTCAACATGACAA-TCACCTCAGGGCTTGGTAAAAAGAGGTCTGACCCCTGTTCTTAGATTTACAGCCTAATGCCTTAACTCGGCCATTTTACCGCAAAAAAGGAAGGAATCGAACCTCCTAAAGCTGGTTTCAAGCCAACCCCATAACCCCCATGACTTTTTCAAAAGGTACTAGAAAAACCATTTCGTAACTTTGTCAAAGTTAAATTACAGGTC-AGACCCTGTGTATCTTA-CATTGCAAAGCTAACCTAGCATTAACCTTTTAAGTTAAAGACTAAGAGAACCAGCCTCTCTTTGCAATGA",
           "AGAAATACGTCTGACGAAAGAGTTACTTTGATAGAGTAAATAACAGGGGTTTAAATCCCCTTATTTCTACTAGAACCATAGGAGTCGAACCCATCCTTGAGAATCCAAAACTCTCCGTGCCACCCGTCGCACCCTGTTCTAAGTAAGGTCAGCTAAATAAGCTATCGGGCCCATACCCCGAAAATGTTGGTTATACCCTTCCCATACTAAGAAATTTAGGTTAAACACAGACCAAGAGCCTTCAAAGCCCTCAGTAAGTTAACAAAACTTAATTTCTGCAAGGGCTGCAAAACCCTACTTTGCATCAACCGAACGCAAATCAGCCACTTTAATTAAGCTAAGCCCTTCTAGATCGATGGGACTTAAACCCATAAAAATTTAGTTAACAGCTAAACACCCTAAACAACCTGGCTTCAATCTAAAGCCCCGGCAGA-GTTGAAGCTGCTTCTTTGAACTTGCAATTCAACGTGAAAAATCACTTCGGAGCTTGGCAAAAAGAGGTTTCACCTCTGTCCTTAGATTTACAGTCTAATGCTTTA-CTCAGCCACTTTACCACAAAAAAGGAAGGAATCGAACCCTCTAAAACCGGTTTCAAGCCAGCCCCATAACCTTTATGACTTTTTCAAAAGATATTAGAAAAACTATTTCATAACTTTGTCAAAGTTAAATCACAGGTCCAAACCCCGTATATCTTATCACTGTAGAGCTAGACCAGCATTAACCTTTTAAGTTAAAGACTAAGAGAACTACCGCCTCTTTACAGTGA"}};

    protected static final Object[][] DENGUE4_TAXON_SEQUENCE = {{"D4Brazi82", "D4ElSal83", "D4ElSal94", "D4Indon76",
            "D4Indon77", "D4Mexico84", "D4NewCal81", "D4Philip64", "D4Philip56", "D4Philip84", "D4PRico86", "D4SLanka78",
            "D4Tahiti79", "D4Tahiti85", "D4Thai63", "D4Thai78", "D4Thai84"}, // 17
          {"ATGCGATGCGTAGGAGTAGGAAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGACCTGGTGCTAGAACATGGAGGATGCGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGACCAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCAATATCAAACATAACTACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTATCTGAAAGAGGAACAGGACCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGTTCGGGGAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAATCATGGAGTTACAGCCATGATAACTCCCAGGTCACCATCGGTGGAAGTCAAATTGCCGGACTATGGAGAACTAACACTCGATTGTGAACCCAGGTCTGGAATTGACTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACATGGCTCGTGCATAAGCAATGGTTTTTGGATCTGCCTCTTCCATGGACAGCAGGAGCAGACACATCAGAGGTTCACTGGAATTACAAAGAGAGAATGGTGACATTTAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTGGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCGCTGGAGCCACAGAAGTGGACTCCGGTGATGGAAATCACATGTTTGCAGGACATCTCAAGTGCAAAGTCCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGTTCAGGAAAGTTTTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTGAAAGTCAAGTATGAAGGTGCTGGAGCTCCGTGTAAAGTCCCCATAGAGATAAGAGATGTAAACAAGGAAAAAGTGGTTGGGCGTATCATCTCATCCACCCCTTTGGCTGAGAATACCAACAGTGTAACCAACATAGAATTAGAACCCCCCTTTGGGGACAGCTACATAGTGATAGGTGTTGGAAACAGCGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAACGAATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCCGTTGGTGGATTGTTCACATCATTGGGAAAGGCTGTGCACCAGGTTTTTGGAAGTGTGTATACAACCATGTTTGGAGGAGTCTCATGGATGATTAGAATCCTAATTGGGTTCTTAGTGTTGTGGATTGGCACGAACTCAAGGAACACTTCAATGGCTATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGCTTCACAGTTCAAGCA",
           "ATGCGATGCGTAGGAGTAGGAAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGACCTGGTGCTAGAACATGGAGGATGCGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGACTAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCAATATCAAACATAACTACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTATCTGAAAGAGGAACAGGACCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGTTCGGGGAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAATCATGGAGTTACAGCCATGATAACTCCCAGGTCACCATCGGTGGAAGTCAAATTGCCGGACTATGGAGAACTAACACTCGATTGTGAACCCAGGTCTGGAATTGACTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACATGGCTCGTGCATAAGCAATGGTTTTTGGATCTGCCTCTTCCATGGACAGCAGGAGCAGACACATCAGAGGTTCACTGGAATTACAAAGAGAGAATGGTGACATTTAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTGGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCGCTGGAGCCACAGAAGTGGACTCCGGTGATGGAAATCATATGTTTGCAGGACATCTCAAGTGCAAAGTCCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGTTCAGGAAAGTTTTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTGAAAGTCAAGTATGAAGGTGCTGGAGCTCCGTGTAAAGTCCCCATAGAGATAAGAGATGTAAACAAGGAAAAAGTGGTTGGGCGTATCATCTCATCCACCCCTTTGGCTGAGAATACCAACAGTGTAACCAACATAGAATTAGAACCCCCCTTTGGGGACAGCTACATAGTGATAGGTGTTGGAAACAGCGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAACGAATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCCGTTGGTGGACTGTTCACATCATTGGGAAAGGCTGTGCACCAGGTTTTTGGAAGTGTGTATACAACCATGTTTGGAGGAGTCTCATGGATGATTAGAATCCTAATTGGGTTCTTAGTGTTGTGGATTGGCACGAACTCAAGGAACACTTCAATGGCTATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGCTTCACAGTTCAAGCA",
           "ATGCGATGCGTAGGAGTAGGAAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGACCTGGTGCTAGAACATGGAGGATGCGTCACAACCATAGCCCAGGGAAAACCAACCTTGGATTTTGAATTGACTAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCAATATCAAACATAACTACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTATCTGAAAGAGGAACAGGACCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGGAATGGCTGTGGCTTGCTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGTTCGGGGAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGATACCCATGCAGTAGGAAATGACACATCCAATCATGGAGTTACAGCCACGATAACTCCCAGGTCACCATCGGTGGAAGTCAAATTGCCGGACTATGGAGAACTAACACTCGATTGTGAACCCAGATCTGGAATTGATTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACATGGCTCGTGCATAAGCAATGGTTTTTGGATCTGCCTCTTCCATGGACAGCAGGAGCAGACACATCAGAGGTTCACTGGAATTACAAAGAGAGAATGGTGACATTCAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTGGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCGCTGGAGCCACAGAAGTGGACTCCGGTGATGGAAATCACATGTTTGCAGGACATCTCAAGTGCAAAGTCCGCATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGTTCAGGAAAGTTTTCAATTGATAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTGAAAGTCAAGTATGAAGGTGCTGGAGCTCCGTGTAAAGTCCCCATAGAGATAAGAGATGTAAACAAGGAAAAAGTGGTTGGGCGTATCATCTCATCCACCCCTTTGGCTGAGAATACCAACAGTGTAACCAACATAGAATTAGAACCCCCCTTTGGGGACAGCTACATAGTGATAGGTGTCGGAAACAGCGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAACGAATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCCGTTGGTGGACTGTTCACATCATTGGGAAAGGCTGTGCACCAGGTTTTTGGAAGTGTGTACACAACCATGTTTGGAGGAGTCTCATGGATGATTAGAATCCTAATTGGGTTCTTAGTGTTATGGATTGGCACGAACTCAAGGAACACTTCAATGGCTATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGCTTCACAGCTCAAGCA",
           "ATGCGATGCGTAGGAGTAGGAAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGATCTGGTGCTAGAACATGGAGGATGCGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGACTAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCAATATCAAACATAACCACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTATCTAAAAGAGGAACAAGACCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGTTCGGGGAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAATCATGGAGTTACAGCCACGATAACTCCCAGGTCACCATCGGTGGAAGTCAAATTGCCGGACTATGGAGAACTAACACTCGATTGTGAACCCAGGTCTGGAATTGACTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACATGGCTTGTGCATAAGCAATGGTTTTTGGATCTACCTCTACCATGGACAGCAGGAGCAGACACATCAGAGGTTCACTGGAATTACAAAGAGAGAATGGTGACATTTAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTGGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCGCTGGAGCCACAGAAGTGGACTCCGGTGATGGAAATCACATGTTTGCAGGACATCTCAAGTGCAAAGTCCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGTCCAGGAAAGTTCTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTGAAAGTCAAGTATGAAGGTGCTGGAGCTCCGTGTAAAGTCCCCATAGAGATAAGAGATGTGAACAAGGAAAAAGTGGTTGGGCGTATCATCTCATCCACCCCTTTGGCTGAGAATACCAACAGTGCAACCAACATAGAGTTAGAACCCCCCTTTGGGGACAGCTACATAGTGATAGGTGTTGGAAACAGTGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAACGAATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCCGTTGGTGGACTGCTCACATCATTGGGAAAGGCTGTGCACCAGGTTTTTGGAAGTGTGTATACAACCATGTTTGGAGGAGTCTCATGGATGATTAGAATCCTAATTGGGTTCCTAGTGTTGTGGATTGGCACGAACTCAAGGAACACTTCAATGGCTATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGCTTCACAGTTCAAGCA",
           "ATGCGATGCGTAGGAGTAGGAAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGATCTGGTGCTAGAACATGGAGGATGCGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGACTAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCAATATCAAACATAACCACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTATCTAAAAGAGGAACAAGACCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGTTCGGGGAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATACACAGTAGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAACCATGGAGTTACAGCCACGATAACTCCCAGGTCACCATCGGTGGAAGTCAAATTGCCGGACTATGGAGAACTAACACTCGATTGTGAACCCAGGTCTGGAATTGACTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACATGGCTTGTGCATAAGCAATGGTTTTTGGATCTACCTCTACCATGGACAGCAGGAGCAGACACATCAGAGGTTCACTGGAATTACAAAGAGAGAATGGTGACATTTAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTGGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCGCTGGAGCCACAGAAGTGGACTCCGGTGATGGAAATCACATGTTTGCAGGACATCTCAAGTGCAAAGTCCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGTTCAGGAAAGTTCTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTGAAAGTCAAGTATGAAGGTGCTGGAGCTCCGTGCAAAGTCCCCATAGAGATAAGAGATGTAAACAAGGAAAAAGTGGTTGGGCGTATCATCTCATCCACCCCTTTGGCTGAGAATACCAACAGTGTAACCAACATAGAATTAGAACCCCCCTTTGGGGACAGCTACATAGTGATAGGTGTTGGAAACAGTGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAACGAATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCCGTTGGTGGACTGTTCACATCATTGGGAAAGGCTGTGCACCAGGTTTTTGGAAGTGTGTATACAACCATGTTTGGAGGAGTCTCATGGATGATTAGAATCCTAATTGGCTTCTTAGTGTTGTGGATTGGCACGAACTCAAGGAACACTTCAATGGCTATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGCTTCACAGTTCAAGCA",
           "ATGCGATGCGTAGGAGTAGGAAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGACCTAGTGCTAGAACATGGAGGATGCGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGACTAAGACAACAGCCAAGGAAGTGGCTCTGCTAAGAACCTATTGCATTGAAGCCTCAATATCAAACATAACTACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTATCTGAAAGAGGAACAGGACCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGTTCGGGGAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAATCATGGAGTTACAGCCATGATAACTCCCAGGTCACCATCGGTGGAAGTCAAATTGCCGGACTATGGAGAACTAACACTCGATTGTGAACCCAGGTCTGGAATTGACTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACATGGCTCGTGCATAAGCAATGGTTTTTGGATCTGCCTCTTCCATGGACAGCAGGAGCAGACACATCAGAGGTTCACTGGAATTACAAAGAGAGAATGGTGACATTTAAGGTTCCCCATGCCAAGAGACAGGATGTGACAGTGCTGGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCGCTGGAGCCACAGAAGTGGACTCCGGTGATGGAAATCACATGTTTGCAGGACATCTCAAGTGCAAAGTCCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGTTCAGGAAAGTTTTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTGAAAGTCAAGTGTGAAGGTGCTGGAGCTCCCTGTAAAGTCCCCATAGAGATAAGAGATGTAAACAAGGAAAAAGTGGTTGGGCGTATCATCTCATCCACCCCTTTGGCTGAGAATACCAACAGTGTAACCAACATAGAATTAGAACCTCCCTTTGGGGACAGCTACATAGTGATAGGTGTTGGAAACAGCGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAACGAATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCCGTTGGTGGACTGTTCACATCATTGGGAAAGGCTGTGCACCAGGTTTTTGGAAGTGTGTATACAACCATGTTTGGAGGAGTCTCATGGATGATTAGAATCCTAATTGGGTTCTTAGTGTTGTGGATTGGCACGAACTCAAGGAACACTTCAATGGCTATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGCTTCACAGTTCAAGCA",
           "ATGCGATGCGTAGGAGTAGGAAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGACCTGGTGCTAGAACATGGAGGATGCGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGACTAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCAATATCAAACATAACTACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTATCTGAAAGAGGAACAGGACCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGTTCGGGGAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAATCATGGAGTTACAGCCATGATAACTCCCAGGTCACCATCGGTGGAAGTCAAATTGCCGGACTATGGAGAACTAACACTCGATTGTGAACCCAGGTCTGGAATTGACTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACATGGCTCGTGCATAAGCAATGGTTTTTGGATCTGCCTCTTCCATGGACAGCAGGAGCAGACACATCAGAGGTTCACTGGAATTACAAAGAGAGAATGGTGACATTTAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTGGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCGCTGGAGCCACAGAAGTGGACTCCGGTGATGGAAATCACATGTTTGCAGGACATCTCAAGTGCAAAGTCCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGTTCAGGAAAGTTTTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTGAAAGTCAAGTATGAAGGTGCTGGAGCTCCGTGTAAAGTCCCCATAGAGATAAGAGATGTAAACAAGGAAAAAGTGGTTGGGCGTATCATCTCATCCACCCCTTTGGCTGAGAATACCAACAGTGTAACCAACATAGAATTAGAACCCCCCTTTGGGGACAGCTACATAGTGATAGGTGTTGGAAACAGCGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAACGAATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCCGTTGGTGGACTGTTCACATCATTGGGAAAGGCTGTGCACCAGGTTTTTGGAAGTGTGTATACAACCATGTTTGGAGGAGTCTCATGGATGATTAGAATCCTAATTGGGTTCTTAGTGTTGTGGATTGGCACGAACTCAAGGAACACTTCAATGGCTATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGCTTCACAGTTCAAGCA",
           "ATGCGATGCGTGGGAGTGGGGAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGATTTGGTGCTAGAACATGGAGGATGTGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGATCAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCGATATCAAACATAACCACGGCAACAAGATGTCCAACGCAAGGAGAACCTTATCTCAAAGAGGAACAAGATCAACAGTACATCTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGGAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGCTCGGGGAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGATACATCCAACCATGGAGTGACAGCCACGATAACCCCCAGGTCACCATCGGTAGAAGTTAAATTACCGGATTATGGAGAATTAACACTCGATTGTGAACCCAGGTCCGGAATTGATTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACGTGGCTTGTGCACAAGCAATGGTTTTTGGATCTACCTCTACCATGGGCAGCAGGAGCAGATACATCAGAAGTTCATTGGAATTACAAAGAGAGAATGGTGACATTCAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTAGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCACCGGAGCTACAGAAGTGGATTCCGGTGATGGAAACCACATGTTTGCAGGACATCTGAAATGCAAAGTTCGCATGGAGAAATTGAGAATTAAGGGAATGTCATACACGATGTGCTCAGGAAAGTTCTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTAAAAGTCAAATATGAGGGTGCTGGAGCTCCATGTAAAGTTCCCATAGAGATAAGAGATGTGAACAAGGAAAAAGTGGTAGGGCGTATCATCTCATCTACCCCTTTGGCTGAGAACACCAACAGTGTAACCAACATAGAATTAGAACCCCCTTTTGGGGACAGCTACATAGTAATAGGTGTTGGAGACAGTGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAGCGAATGGCCATTCTAGGTGAAACAGCCTGGGATTTTGGTTCGGTTGGTGGACTGCTCACATCATTGGGAAAGGCTGTACACCAGGTTTTTGGTAGTGTGTATACAACTATGTTTGGAGGAGTCTCATGGATGGTTAGAATCCTAATTGGGTTCTTAGTGTTGTGGATTGGCACGAATTCGAGAAACACCTCAATGGCAATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGTTTCACAGTTCACGCA",
           "ATGCGATGCGTGGGAGTGGGGAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGATTTGGTGCTAGAACATGGAGGATGTGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGATCAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCGATATCAAACATAACCACGGCAACAAGATGTCCAACGCAAGGAGAACCTTATCTCAAAGAGGAACAAGATCAACAGTACATCTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGCTTGGGAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGCTCGGGGAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATACACAGTAGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACATATCCAACCATGGAGTGACAGCCACGATAACCCCCAGGTCACCATCGGTAGAAGTTAAATTACCGGATTATGGAGAATTAACACTCGATTGTGAACCCAGGTCCGGAATTGATTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACGTGGCTTGTGCACAAGCAATGGTTTTTGGATCTACCTCTACCATGGGCAGCAGGAGCAGACACATCAGAAGTTCATTGGAATTACAAAGAGAGAATGGTGACATTCAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTAGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCACCGGAGCTACAGAAGTGGATTCCGGTGATGGAAACCACATGTATGCAGGACATCTGAAATGCAAAGTTCGCATGGAGAAATTGAGAATTAAGGGAATGTCATACACGATGTGCTCAGGAAAGTTCTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTAAAAGTCAAGTATGAGGGTGCTGGAGCTCCATGTAAAGTTCCCATAGAGATAAGAGATGTGAACAAGGAAAAAGTGGTAGGGCGCATCATCTCATCTACCCCTTTGGCTGAGTATACCAACAGTGTAACCAACATAGAATTAGAACCCCCCTTTGGGGACAGCTACATAGTAATAGGTGTTGGAGACAGTGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGCGCAAAGCGAATGGCCATTCTAGGTGAAACAGCCTGGGATTTTGGTTCTGTTGGTGGACTGCTCACATCATTGGGAAAGGCTGTACACCAGGTTTTTGGTAGTGTGTATACAACTATGTTTGGAGGAGTCTCATGGATGGTTAGAATCCTAATTGGGTTCTTAGTGTTGTGGATTGGCACGAATTCGAGAAACACCTCAATGGCAATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGTTTCACAGTTCACGCA",
           "ATGCGATGCGTAGGAGTGGGGAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGACTTAGTGCTAGAACATGGAGGATGTGTCACAACCATGGCCCAAGGAAAACCAACCTTGGATTTTGAACTGATCAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCGATATCAAACATAACCACGGCAACAAGATGCCCAACGCAAGGAGAACCTTATCTCAAAGAGGAACAAGATCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGGAAAGGAGGAGTTGTGACATGTGCGAAGTTCTCATGCTCGGGAAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATATACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAACCATGGAGTGACAGCCACGATAACCCCTAGGTCACCATCGGTAGAAGTTAAATTACCGGATTATGGAGAATTAACACTTGATTGTGAACCCAGGTCCGGAATTGACTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACGTGGCTTGTGCACAAGCAATGGTTTCTGGATCTGCCTCTACCATGGGCAGCAGGAGCAGATACATCAGAAGTTCATTGGAATTACAAAGAGAGAATGGTGACATTCAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTAGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCACCGGAGCTACAGAAGTGGATTCCGGTGATGGAAACCACATGTTTGCAGGACATCTGAAATGCAAAGTTCGCATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGCTCAGGGAAGTTCTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTTAAAGTCAAATATGAAGGTGCTGGAGCTCCGTGCAAAGTTCCCATAGAGATAAGAGATGTGAACAGGGAAAAAGTGGTAGGGCGTGTCATCTCATCTACCCCTTTGGCCGAGAATACCAACAGTGTAACCAACATAGAATTAGAACCCCCTTTTGGGGACAGCTACATAGTAATAGGTGTTGGAGACAGTGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAGCGAATGGCCATTCTAGGTGAAACAGCCTGGGATTTTGGTTCTGTTGGTGGACTGCTCACATCATTGGGAAAGGCTGTACACCAGGTTTTTGGTAGTGTGTATACAACTATGTTTGGAGGAGTCTCATGGATGGTTAGAATCCTAATTGGGTTCCTAGTGTTGTGGATTGGCACGAATTCGAGAAACACCTCAATGGCAATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTCATGGGTTTCACAGTTCACGCA",
           "ATGCGATGCGTAGGAGTAGGAAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGACCTGGTGCTAGAACATGGAGGATGCGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGACTAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCAATATCAAACATAACTACGGCAACAAGATGTCCAACGCAAGGGGAGCCCTATCTGAAAGAGGAACAGGACCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGTTCGGGGAAGATAACAGGCAATCTGGTCCAAATTGAGAACCTTGAGTACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAATCATGGAGTTACAGCCACGATAACTCCCAGGTCACCATCGGTGGAAGTCAAATTGCCGGACTATGGAGAACTAACACTCGATTGTGAACCCAGGTCTGGAATTGACTTTAATGAGATGATTCTAATGAAAATGAAAAAGAAAACATGGCTCGTGCATAAGCAATGGTTTTTGGATCTGCCTCTTCCATGGACAGCAGGAGCAGACACATCAGAGGTTCACTGGAATTACAAAGAGAGAATGGTGACATTTAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTGGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCGCTGGAGCCACAGAAGTGGACTCCGGTGATGGAAATCACATGTTTGCAGGACATCTCAAGTGCAAAGTCCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGTTCAGGAAAGTTTTCAATCGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTGAAAGTCAAGTATGAAGGTGCTGGAGCTCCGTGTAAAGTCCCCATAGAGATAAGAGATGTAAACAAGGAAAAAGTGGTTGGGCGCGTCATCTCATCCACCCCTTTGGCTGAGAATACCAACAGTGTAACCAACATAGAATTAGAACCCCCCTTTGGGGACAGCTACATAGTGATAGGTGTTGGAAACAGCGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAACGAATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCCGTTGGTGGACTGTTCACATCATTGGGAAAGGCTGTGCACCAGGTTTTTGGAAGCGTGTATACAACCATGTTTGGAGGAGTCTCATGGATGATTAGAATCCTAATTGGGTTCTTAGTGTTGTGGATTGGCACGAACTCAAGGAACACTTCAATGGCTATGACGTGCATAGCTGTTGGAGGTATCACTCTGTTTCTGGGCTTCACAGTTCAAGCG",
           "ATGCGATGCGTGGGAGTGGGGAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGATCTGGTGCTAGAACATGGAGGATGTGTCACAACCATGGCCCAAGGAAAACCAACCTTGGATTTTGAACTGATCAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCCATATCAAACATAACCACGGCAACAAGATGTCCAACGCAAGGAGAGCCCTGTCTCAAAGAGGAACAGGATCAACAGTACATCTGCCGGAGAGACGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCCTGCTTGGGAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGCTCGGGGAAGATAACAGGCAACTTAGTCCGAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAACCACGGAGTGACAGCCACGATAACCCCCAGGTCACCATCGGTAGAAGTTAAATTACCGGACTATGGAGAATTGACACTCGATTGTGAACCCAGGTCCGGAATTGATTTTAATGAGATGATTCTGATGGAAATGAGAAAGAAAACGTGGCTTGTGCACAAGCAATGGTTTTTGGATCTACCTCTACCATGGACAGCAGGAGCAGACACGTCAGAAGTTCATTGGAATCACAAAGAGAGAATGGTGACGTTCAAGGTCCCTCATGCCAAGAGACAGGATGTGACAGTGCTAGGATCTCAGGAAGGAGCCATGCATTCAGCCCTCACCGGAGCCACAGAAGTGGATTCCGGTGATGGAAACCACATGTTTGCAGGACATTTGAAGTGCAAAGTTCGCATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGCTCAGGAAAGTTCTCAATTGATAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTAAAAGTCAAGTATGAGGGTGCCGGAGCTCCATGTAAAGTTCCCATAGAGATAAGAGATGTGAACAAGGAAAAAGTGGTTGGGCGCATCATCTCATCTACCCCTTTTGCTGAGAATACCAACAGTGTGACCAACATAGAATTGGAACCCCCCTTTGGGGATAGCTACATAGTAATAGGTGTAGGAAACAGTGCATTAACACTCCATTGGTTTAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGCGCAAAGCGCATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCTGTTGGTGGACTGCTCACATCATTGGGAAAGGCTGTACACCAGGTTTTTGGTAGTGTGTATACAACTATGTTTGGAGGAGTCTCATGGATGGTTAGAATCCTAATCGGGTTCTTAGTATTGTGGATTGGCACGAATTCAAGAAACACTTCAATGGCAATGTCGTGCATAGCTGTTGGAGGAATTACTTTGTTTCTGGGTTTCACAGTTCATGCA",
           "ATGCGATGCGTAGGAGTAGGAAACAGAGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGATCTGGTGCTAGAACATGGAGGATGCGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGACTAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCAATATCAAACATAACTACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTATCTGAAAGAGGAACAGGACCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGTTCGGGGAAGATAACAGGCAATTTGGTCCAAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAATCATGGAGTTACAGCCACGATAACTCCCAGGTCACCATCGGTGGAAGTCAAATTGCCGGACTATGGAGAACTAACACTCGATTGTGAACCCAGGTCTGGAATTGACTTTAATGAGATGATCCTGATGAAAATGAGAAAGAAGACATGGCTCGTGCATAAGCAATGGTTTTTGGATCTGCCTCTTCCATGGACAGCAGGAGCAGACACATCAGAGGTTCACTGGAATTACAAAGAGAGAATGGTGACATTTAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTGGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCGCTGGAGCCACAGAAGTGGACTCCGGTGATGGAAATCACATGTTTGCAGGACATCTCAAGTGCAAAGTCCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGTTCAGGAAAGTTTTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTGAAAGTCAAGTATGAAGGTGCTGGAGCTCCGTGTAAAGTCCCCATAGAGATAAGAGATGTAAACAAGGAAAAAGTGGTTGGGCGTATCATCTCATCCACCCCTTTGGCTGAGAATACCAACAGTGTAACCAACATAGAATTAGAACCCCCCTTTGGGGACAGCTACATAGTGATAGGTGTTGGAAACAGCGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAACGAATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCCGTTGGTGGACTGTTCACATCATTGGGAAAGGCTGTGCACCAGGTTTTTGGAAGTGTGTATACAACCATGTTTGGAGGAGTCTCATGGATGATTAGAATCCTAATTGGGTTCTTAGTGTTGTGGATTGGCACGAACTCAAGGAACACTTCAATGGCTATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGCTTCACAGTTCAAGCA",
           "ATGCGATGCGTAGGAGTAGGAAACAGAGACTTTGTGGAAGGAGTTTCAGGTGGAGCATGGGTCGATTTGGTGCTAGAACATGGAGGATGCGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGACTAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCAATATCAAACATAACTACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTATCTGAAAGAGGAACAGGACCAACAGTACATTTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTGTTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGTTCGGGGAAGATAACAGGCAATCTGGTCCAAATTGAGAACCTTGAATACACAGTGGTCATAACAGTCCACAATGGAGACACCCATGCAGTAGGAAATGACACATCCAATCATGGAGTTACAGCCACGATAACTCCCAGGTCACCATCGGTGGAAGTCAAATTGCCGGACTATGGAGAACTAACACTCGATTGTGAACCCAGGTCTGGAATTGACTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACATGGCTCGTGCATAAGCAATGGTTTTTGGATCTGCCTCTTCCATGGACAGCAGGAGCAGACACAACAGAGGTTCACTGGAATTACAAAGAGAGAATGGTGACATTTAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTACTGGGATCTCAGGAAGGAGCCATGCATTCTGCCCTAGCTGGAGCTACAGAAGTGGACTCCGGTGATGGGAATCACATGTTTGCAGGACATCTCAAGTGCAAAATCCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGTTCAGGAAAGTTTTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTGAAAGTCAAGTATGAAGGTGCTGGAGCTCCGTGTAAAGTCCCCATAGAGATAAGAGATGTAAACAAGGAAAAAGTGGTTGGGCGTGTCATCTCATCCACCCCTTTGGCTGAGAATACCAACAGTGTAACCAACATAGAATTAGAACCCCCCTTTGGGGACAGCTACATAGTGATAGGTGTTGGAAACAGCGCATTAACACTCCATTGGTTCAGGAAAGGGAGTTCCATCGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAACGAATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCCGTTGGTGGACTGTTCACATCATTGGGAAAGGCTGTGCACCAGGTTTTTGGAAGTGTGTATACAACCATGTTTGGAGGAGTCTCATGGATGATTAGAATCCTAATTGGGTTCTTAGTGTTGTGGATTGGCACGAACTCAAGGAACACTTCAATGGCTATGACGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGCTTCACAGTTCAAGCA",
           "ATGCGATGCGTAGGAGTGGGGAACAGGGACTTTGTGGAAGGAGTCTCAGGTGGAGCATGGGTCGATCTGGTGCTAGAACATGGAGGATGTGTCACAACCATGGCTCAAGGAAAACCAACCTTGGATTTTGAACTGATCAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCGATATCAAACATAACCACGGCGACAAGATGTCCAACGCAAGGAGAGCCTTATCTCAAAGAGGAACAAGATCAACAGTACATCTGCCGGAGAGATGTGGTAGACAGAGGGTGGGGCAATGGCTGTGGCTTACTTGGAAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGCTCGGGGAAGATAACAGGCAACTTGGTTCGAATTGAGAACCTTGAATACACAGTGGTTGTGACAGTCCACAACGGAGACACCCATGCAGTAGGAAATGACATATCCAACCATGGAGTGACAGCCACGATAACTCCCAGGTCACCATCGGTAGAAGTCAAATTACCGGATTATGGAGAATTAACGCTCGATTGTGAACCCAGGTCCGGAATTGATTTTAATGAGATGATTCTGATGGAAATGAGAAAGAAGACGTGGCTTGTGCACAAGCAATGGTTTTTGGATCTACCTCTACCATGGACAGCAGGAGCAGACACAGCAGAAGTTCATTGGAATTACAAAGAGAGAATGGTGACATTCAAGGTCCCTCATGCTAAGAGACAAGATGTGACAGTGCTAGGATCTCAGGAAGGAGCCATGCATTCTGCCCTCACCGGAGCTACAGAAGTGGATTCCGGTGATGGAAACCACATGTTTGCAGGACATCTTAAGTGCAAGGTTCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGCTCAGGAAAGTTCTCAATTGACAAAGAGATGGCAGAAACACAGCATGGGACAACAGTAGTGAAAGTTAAGTATGAAGGCGCTGAAGCTCCATGTAAAATCCCCATAGAGATAAGAGATGTGAACAAGGAAAAAGTTGTTGGGCGCATCATCTCATCCACTCCTTTGGCTGAAAACACCAACAGCGTGACCAATATAGAATTAGAACCTCCCTTTGGGGACAGCTACATAGTAATAGGTGTTGGAGACAGTGCATTAACACTCCATTGGTTCAGGAAGGGGAGTTCCATTGGCAAGATGTTTGAGTCTACATATAGAGGAGCAAAGCGAATGGCCATTCTAGGTGAAACAGCTTGGGACTTTGGCTCTGTTGGCGGACTGTCTACATCATTGGGAAAGGCTGTACACCAGGTTTTTGGTAGTGTGTACACAACCATGTTTGGAGGAGTCTCATGGATGGTTAGAATCCTAATCGGGCTCTTGGTGTTGTGGATTGGCACAAATTCAAGAAACACCTCAATGGCAATGACGTGCATAGCTGTTGGAGGAATCACTCTATTTCTGGGTTTCACAGCTCACGCA",
           "ATGCGATGCGTAGGAGTGGGGAACAGAGACTTTGTAGAAGGAGTCTCAGGTGGAGCATGGGTCGATCTGGTGCTAGAACATGGAGGATGTGTCACAACCATGGCCCAGGGAAAACCAACCTTGGATTTTGAACTGATCAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCCATATCAAACATAACCACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTACCTCAAAGAGGAACAAGATCAACAGTACATCTGCCGGAGAGACGTGGTAGATAGAGGGTGGGGCAACGGCTGTGGCTTGCTTGGGAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGCTCGGGGAAGATAACAGGCAACTTAGTCCAAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCTGTAGGAAATGATACATCCAACCACGGAGTGACAGCCACGATAACCCCCAGGTCACCATCGGTAGAAGTTAAATTACCGGACTATGGAGAATTAACACTTGATTGTGAACCTAGGTCCGGAATTGACTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACGTGGCTCGTGCACAAGCAATGGTTTTTGGATCTACCTCTACCATGGACAGCAGGAGCAGACACGTCAGAAGTTCACTGGAATCACAAAGAGAGAATGGTGACATTCAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTAGGATCTCAGGAAGGAGCTATGCATTCAGCCCTCACCGGAGCCACAGAAGTGGATTCCGGTGATGGAAACCATATGTTTGCAGGACATCTTAAGTGTAAAGTTCGTATGGAGAAATTGAGGATCAAGGGAATGTCATACACGATGTGCTCAGGAAAGTTCTCAATTGATAAAGAGATGGCAGAAACACAGCATGGGACAACAGTGGTAAAAGTCAAGTATGAAGGTGCTGGAGCTCCATGTAAAGTCCCCATAGAGATAAGAGATGTGAACAAGGAAAAAGTGGTTGGGCGTATCATCTCATCTACCCCTTTTGCTGAGAATACCAACAGTGTGACCAATATAGAATTGGAACCCCCTTTTGGGGATAGCTACATAGTAATAGGTGTAGGAGACAGTGCATTAACACTCCATTGGTTCAGGAAAGGGAGCTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGCGCAAAGCGCATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCTGTCGGTGGACTGCTCACATCATTGGGAAAGGCTGTACACCAGGTTTTTGGCAGTGTGTATACAACTATGTTTGGAGGAGTCTCATGGATGGTTAGAATCCTAATCGGGTTCTTAGTGTTGTGGATTGGCACGAATTCAAGAAACACTTCAATGGCAATGTCGTGCATAGCTGTTGGAGGAATCACTCTGTTTCTGGGCTTCACAGTTCATGCA",
           "ATGCGATGCGTAGGAGTAGGGAACAGAGACTTTGTAGAAGGAGTCTCAGGTGGAGCATGGGTCGATCTGGTGCTAGAACATGGAGGATGTGTCACAACCATGGCCCAGGGAAAACCAACCCTGGATTTTGAACTGATCAAGACAACAGCCAAGGAAGTGGCTCTGTTAAGAACCTATTGCATTGAAGCCTCCATATCAAACATAACCACGGCAACAAGATGTCCAACGCAAGGAGAGCCTTACCTCAAAGAGGAACAAGATCAACAGTACATCTGCCGGAGAGACGTGGTAGACAGAGGGTGGGGCAACGGCTGTGGCTTGTTTGGGAAAGGAGGAGTTGTGACATGTGCGAAGTTTTCATGCTCGGGGAAGATAACGGGCAACTTAGTCCAAATTGAGAACCTTGAATACACAGTGGTTGTAACAGTCCACAATGGAGACACCCATGCTGTAGGAAATGATACATCCAACCACGGAGTGACAGCCACGATAACCCCCAGGTCACCATCGGTAGAAGTTAAATTACCGGACTATGGAGAATTAACACTTGATTGTGAACCTAGGTCCGGAATTGATTTTAATGAGATGATTCTGATGAAAATGAAAAAGAAAACGTGGCTCGTGCACAAGCAATGGTTTTTGGATCTACCTCTACCATGGACAGCAGGAGCAGACACGTCAGAAGTTCACTGGAATCACAAAGAGAGAATGGTGACATTCAAGGTTCCTCATGCCAAGAGACAGGATGTGACAGTGCTAGGATCTCAGGAAGGAGCCATGCATTCAGCCCTCGCCGGAGCCACAGAAGTGGATTCCGGTGATGGAAACCATATGTTTGCAGGACACTTAAAGTGTAAAGTTCGTATGGAGAAATTGAGAATCAAGGGAATGTCATACACGATGTGCTCAGGAAAGTTCTCAATTGATAAAGAGATGGCAGAAACACAGCATGGAACAACAGTGGTAAAGGTCAAGTATGAAGGCACTGGAGCTCCATGTAAAGTCCCCATAGAGATAAGAGATGTGAACAAGGAAAAAGTGGTTGGGCGTATCATCTCATCTACCCCTTTGGCTGAGAATACCAACAGTGTGACCAATATAGAATTGGAACCTCCTTTTGGGGATAGCTACATAGTGATAGGTGTGGGAGACAGTGCATTAACACTCCATTGGTTCAGGAAAGGGAGCTCCATTGGCAAGATGTTTGAGTCCACATACAGAGGTGCAAAGCGCATGGCCATTCTAGGTGAAACAGCTTGGGATTTTGGTTCTGTTGGCGGACTGCTCACATCATTGGGAAAGGCTGTACACCAGGTTTTTGGCAGTGTGTATACAACTATGTTTGGAGGAGTCTCATGGATGGTTAGAATCCTAATCGGGTTCTTAGTGTTGTGGATTGGCACGAATTCAAGAAACACTTCAATGGCTATGTCGTGCATAGCTGTTGGAGGAATTACTCTGTTTCTGGGCTTCACAGTTCATGCA"},
          {1982.0, 1983.0, 1994.0, 1976.0, 1977.0, 1984.0, 1981.0, 1964.0, 1956.0, 1984.0, 1986.0, 1978.0, 1979.0, 1985.0, 1963.0, 1978.0, 1984.0},
          {false}//forwards
          };

    protected static final Object[][] NUMBER_TAXON_SEQUENCE = {{"119_27629.7737681096", "103_23193.258377165", "125_29554.4074069898",
            "174_44884.4881635122", "49_10373.6477823059", "127_30942.0757935152", "28_3361.21216092344", "74_15748.6180439834",
            "177_45915.951076986", "191_49643.2699171139", "178_46029.1511374065", "56_11331.1641925041", "168_43292.0579663998",
            "114_26229.5956907316", "15_0", "141_33993.5931642486", "122_28640.244120699", "82_18050.6404047613", "170_43771.7660792247",
            "96_20040.6057328143", "77_16282.1519049503", "110_25036.1893581697", "140_33828.3027974152", "156_40108.1945225169",
            "33_4384.54318311674", "27_3286.56306065565", "85_18255.9046701693", "4_0", "153_38146.7620575103", "184_47341.425959174",
            "51_10775.2913004955", "123_28640.5036683123", "92_19534.5571294814", "120_28246.6434921313", "3_0", "100_21990.610543414",
            "104_23534.6629469335", "55_11253.6535216988", "39_6682.88783307691", "44_8408.1014145367", "41_7404.85178303718",
            "95_19679.5532566769", "78_16589.2643746657", "161_41593.5335733483", "137_32557.4078204266", "36_5552.138463554",
            "132_31834.5211268845", "151_37913.6480934186", "35_4938.64887919884", "84_18247.8773121982", "34_4574.41581769302",
            "14_0", "89_19180.5623573959", "23_636.189290657008", "43_8099.14005152468", "146_35466.3683277678", "139_33127.9489146048",
            "145_35381.7303597347", "7_0", "52_10779.7772429738", "179_46372.9943446877", "159_41304.7019296968", "54_10907.3255329076",
            "171_43910.8604796992", "47_9521.18728201725", "172_44214.8006250223", "30_3705.03706028433", "130_31434.5369495946", "19_0",
            "13_0", "143_34603.626076205", "18_0", "116_26793.8784675557", "181_46978.5776842404", "88_18806.7808982742", "97_20490.9510163654",
            "166_42394.1578476862", "67_14513.2626559194", "155_40041.3554929369", "101_22371.4779273036", "149_36695.4502876162",
            "66_14067.7896152956", "58_12005.4521099569", "176_45607.6057972176", "25_2371.2206425099", "63_13292.2356334715", "80_16894.0572542597",
            "129_31020.2746327466", "90_19225.652102134", "11_0", "163_41880.1956106261", "147_35724.0245734204", "187_48416.9563909283",
            "154_39389.3148541711", "46_9414.51106651936", "157_41121.5359539506", "175_45320.4281001263", "112_25908.6648955616", "38_6555.85933111986",
            "169_43657.7713233115", "2_0", "8_0", "60_12604.1293602869", "12_0", "68_14842.8769023166", "158_41192.2105510858", "57_11401.6223352573",
            "131_31740.8103272138", "29_3701.61977516492", "128_30979.5732193265", "87_18798.4147409621", "26_3037.56781984674", "115_26532.8669292666",
            "83_18230.4237031799", "126_30117.3723750415", "64_13874.2451117345", "186_48325.4503775733", "65_14004.0635236353", "9_0",
            "165_41965.0795901386", "75_15764.982406929", "102_22381.2122539226", "61_12849.1264583715", "98_20887.3773053226", "160_41392.4669452232",
            "37_6283.17530013991", "94_19608.8809590778", "167_43275.1927866184", "42_7946.90339174959", "31_4233.30469585643", "21_0",
            "45_9337.29802780956", "93_19598.0175163873", "6_0", "190_49122.6573595682", "1_0", "91_19248.8185402411", "76_16261.5202457145",
            "109_24783.6533440257", "133_31843.1314081288", "111_25766.4342624988", "72_15441.1310395924", "81_17818.8547932802", "118_27555.2102761873",
            "69_14985.058461061", "48_10049.7838572", "144_35274.9409822078", "5_0", "22_0", "105_23554.5335852946", "40_6978.04339209664",
            "71_15220.6096886138", "79_16618.9893260873", "117_27072.7976656318", "152_38055.7486513712", "50_10647.4101614314",
            "24_837.033020313349", "164_41934.5519352841", "121_28350.0980527709", "180_46592.9320853448", "86_18499.5499625087",
            "59_12397.5172641178", "183_47102.3332595296", "32_4342.24017973754", "16_0", "188_48880.2396711122", "135_32225.9431260072",
            "20_0", "99_21539.2831007524", "142_34456.2398246651", "189_49045.5635156947", "70_15050.1970003035", "106_24077.8343464443",
            "134_31843.2696677631", "113_26017.8648699345", "150_36940.8346778002", "148_36279.7812045972", "185_47657.4290437036",
            "138_32706.8658376431", "53_10843.0641530676", "17_0", "124_28908.6423833304", "10_0", "73_15611.6403423159", "173_44573.2566334234",
            "62_12951.778531092", "108_24460.8209168726", "136_32336.5710606241", "182_47026.7000545894", "107_24250.081674673", "162_41827.381497564"},
            // 191
          {"CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGAGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTGTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGGAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACACAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGATCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATGCTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGGCTTGCTTTGTTCTCGCGCTTGGACACCAGGGTTTTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCTTAACCGTATCGGCTTGACACTTAGGTTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCTCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCTATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTACATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAATTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAATCACGGTTGGTCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACTCACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACTGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTATTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAACTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTACTCAACGAGTAGTATCACTTCCCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGGCACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTTCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCAAATCTACTTAATGGCGTTTTATCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAGGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCTTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGCACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGCCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACATGCTTTGTTTTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCAACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCAAATCTACTTAATGGCGTTTTATCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAGGTAAGTTAAGACGTACTTGGGACTACGCAACGCATTCCGCCCGCATGAAGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGTTTTTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACTACGGTTGATCTTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGACCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGCCTAGGGGGTTCAGCAAGCTCCGCC",
           "TATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGTGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTGTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACCAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTAGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGATGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATACATTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACACGCTTGGGACTACGCAACGCGTTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACTAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACATAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTGGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTGCCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTGCTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGGCTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTACTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTCAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTCGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAATGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCGACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGTCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCGCTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTATCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACACAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGGTCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAACATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCACTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTGCTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGATCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATGCTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTCGTTCTCGTGCTTGGACACCAGGGTTTTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCTTAACCGTACCGGCTTGACACTTAGGTTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCTCGCACTCGGCATGCGTAGCCATCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTGTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACTAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAACCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCCCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCCACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCACATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTACTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTCAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATGTAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCGACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC", 
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATTACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCGGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCGATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTAGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATGTAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGATATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGTTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTCTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTGCTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAGACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTGCTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCACCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGATTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTGCTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACATCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACTGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTATTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAACTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGCAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGGCAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGGGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTCTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTGCTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGATTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACTGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGACATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCAAATCTACTTAATGGCGTTTTATCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACGGGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACCTGCTTTGTTCTCGCGCTTGGACACCAGGTTTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACTACGGTTGATCTTAACCGTACCGGCTTGACACTTAGATTAGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGCCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTACCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCACTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCGATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCTCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAATCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAACCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCTGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTGTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGATTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAAGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGCTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTTACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTTACTGTTACCTACCAGTGTACCAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTCGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGGGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATGATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGAGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGGGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCAAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTCTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCACGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGCTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACCTGCTTTGTTCTCGCGCTTGGACACCACGATTGTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCACTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCGGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTGTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTGTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTCGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGTAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGAGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATGATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCTGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGCCCGGTCACTGTTACCCACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCGTTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCGACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCGGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTAATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGATCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATGCTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGGTTTTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCTTAACCGTACCGGCTTGACACTTAGGTTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCTCGCACTCGGCATGCGTAGCCATCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCTCGAATCCACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGGTCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGATCACTGTTACCCACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGCTAAGACATACTTGGGACTACGCAACGCATTCCGCTCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCTGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGTTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCTCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAACCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCACCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACATTTAGATTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTGAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCTCTAGGTTTTGGTCCGGTCACTGTTACCTACCAATGTACTAATCGGTCAGTGGACGGAGTGGGCCTATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCAACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATAGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTGCTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAGACCGTCCTTTCAGTTCGCTGACATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGATCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATGCTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGGTTTTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCTTAACCGTACCGGCTTGACACTTAGGTTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCTCGCACTCGGCATGCGTAGCCATCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACGAATTTCGACCAGTACTCCTAGGTTTTGGTCCGGTCACCGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTTCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGTCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAATCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACTAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCTCCGAATCTACTTAATGGCGTTTTACCGTTTGTCCTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCACTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGATTAAATCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTATGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCAGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCGTTCCGCCCGCATGAGGATGCAGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCAAATCTACTTAATGGCGTTTTATCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCACGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCTTAACCGTACCGGCTTGACACTTAGATTGAGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGCCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCCAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCTCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAACCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCTGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTGTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATGATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGTTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCAAATCTACTTAATGGCGTTTTATCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAGGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGTGCTTGGACACCAGGTTTTTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACTACGGTTGATCTTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGCCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCCTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTACGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATTGGCTGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCCAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTTAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACTATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGACCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATACATTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCGTTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCTCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAACCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCTGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACACAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGTTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTAACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGTTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACATGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCCAACCACGGTTGATCCTAACCGTACCGGCTTGACACCTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTGTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATGCTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCACC",
           "CATGACACCTAACCTCAATAACATTCCCCGAACCTACTTAATGGCGTTTTACCGTTTGTCTTGAGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACACAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGTTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGTCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCTCGAATCCACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGGTCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCAGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCCACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATGTTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGCCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACCAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCGTTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCGATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTCCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGACCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGATTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACACAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTGTGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACATTTAGATTGGGTACCCCTAGAGCCTCTGATTGGATCTCAATTTCTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAGACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAATTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGTTATATAGTTCCATGTATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCATACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTGTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGAAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCGGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTGCCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCTATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGTATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGGGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTGCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTCTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGCTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTGCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGTGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGACATGCGTAGCCATCAACAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGCGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCTCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAACCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCTGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTGCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGTATTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTACTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTCAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAATCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGCTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTCTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCGACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGTTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATGTTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACGCTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATCAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACTATCACAAGTAAGTCAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCCACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CACGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCGGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAGCCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATGACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACTTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTCTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGAGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTGTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTAATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCATTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGTCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAAACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCTGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACACAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTGTGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACATTTAGATTGGGTACCCCTAGAGCCTCTGATTGGATCTCAATTTCTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAACCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAGCCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTCTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGAATTTTCTGGTAAATAATTGTATATTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGAAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGCTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCCGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGACCGGAACTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCCTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGTTGTCGGAGGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACATGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCACATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCCAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGATTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAATTCGCTGATATTAAGCTAGCCATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTCTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGATGCCCGCCCAGATAGCCATAAACTGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAGCCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTACGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCGTTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGCTGATCCTAACCGCACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTAGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATGATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACACAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTAGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCGACGCATTCCACCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGTTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACATGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGTTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGAGACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACACAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACTAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATGCTTGAGACTACACAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTATCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGGGTAGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAACAACATTCCCCGAATCTACTTAATGGCGTTTTACCGCTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGACCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTTCGCC",
           "CATTACACCTAACCTCAATAACATTCCCCAAATCTACTTAATGGCGTTTTATCGTTTGTCTTGCGGTTATGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATGTATAAGGGACTAAACCGAACCATCACAGGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGTTTTTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACTACGGTTGATCTTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTGGCCATCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTCGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGCCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCACGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCGCTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGCTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGACCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGTTTGGGACTACGCAACGCGTTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTGTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCACTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCGGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTGCTTAACGGCGTTTTACCGGTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAGACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATATTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTGCCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTGAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAAGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGTGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAATCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCACGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCGTTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTAAATAATTGTATGTTGTGGCTGCGGCCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATTGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAACGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCAGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTTTTGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCGACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGGTTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTTGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGCCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAACAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCAGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGTTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTACTCAACGAGTAGTATCAGTTCTCGGTTTCGCGGGGGTGCCCGCCCAGATAGCCATAAACCGTCCTTTCAGTTCGCTGATATTAAGCTAGCTATATAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTCAAGACATACTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTACGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTTTCTGGTAAATAATTGTATGTTGTGGCCGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTACCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCATCAGCAGACAAATCTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGGACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC",
           "CATGACACCTAACCTCAATAACATTCCCCGAATCTACTTAATGGCGTTTTACCGCTTGTCTTGCGGTTGTGAGGCTGTCGGAAGTTGCTCAACGAGTAGTATCAGTTCTCGGTCTCGCGGGGGTGCCCGCCCAAATAGCCATAAACCGTCCTTTCGGTTCGCTGATATTAAGCTAGCTATACAGTTCCATATATAAGGGACTAAACCGAACCATCACAAGTAAGTTGAGACATGCTTGGGACTACGCAACGCATTCCGCCCGCATGAGGATGCGGTATGGAGACTTGCTTTGTTCTCGCGCTTGGACACCAGGATTCTCTGGTGAATAATTGTATGTTGTGGCTGCGACCTCGAACCACGGTTGATCCTAACCGTACCGGCTTGACACTTAGATTGGGTATCCCTAGAGCCTCTGATCGGATCTCAATTTGTGCTGGAATATATATTCGTGAGTCTCCCGCACTCGGCATGCGTAGCCGTCAGCAGACAAATTTCGACCAGTACCCCTAGGTTTTGGTCCGGTCACTGTTACCTACCAGTGTACTAATCGGCCGGTGAACGGAGTGGGCCCATATACTACGTCTAGGGGGTTCAGCAAGCTCCGCC"},
          {27629.7737681096, 23193.258377165, 29554.4074069898, 44884.4881635122, 10373.6477823059, 30942.0757935152, 3361.21216092344,
                  15748.6180439834, 45915.951076986, 49643.2699171139, 46029.1511374065, 11331.1641925041, 43292.0579663998,
                  26229.5956907316, 0.0, 33993.5931642486, 28640.244120699, 18050.6404047613, 43771.7660792247, 20040.6057328143,
                  16282.1519049503, 25036.1893581697, 33828.3027974152, 40108.1945225169, 4384.54318311674, 3286.56306065565,
                  18255.9046701693, 0.0, 38146.7620575103, 47341.425959174, 10775.2913004955, 28640.5036683123, 19534.5571294814,
                  28246.6434921313, 0.0, 21990.610543414, 23534.6629469335, 11253.6535216988, 6682.88783307691, 8408.1014145367,
                  7404.85178303718, 19679.5532566769, 16589.2643746657, 41593.5335733483, 32557.4078204266, 5552.138463554,
                  31834.5211268845, 37913.6480934186, 4938.64887919884, 18247.8773121982, 4574.41581769302, 0.0, 19180.5623573959,
                  636.189290657008, 8099.14005152468, 35466.3683277678, 33127.9489146048, 35381.7303597347, 0.0, 10779.7772429738,
                  46372.9943446877, 41304.7019296968, 10907.3255329076, 43910.8604796992, 9521.18728201725, 44214.8006250223,
                  3705.03706028433, 31434.5369495946, 0.0, 0.0, 34603.626076205, 0.0, 26793.8784675557, 46978.5776842404,
                  18806.7808982742, 20490.9510163654, 42394.1578476862, 14513.2626559194, 40041.3554929369, 22371.4779273036,
                  36695.4502876162, 14067.7896152956, 12005.4521099569, 45607.6057972176, 2371.2206425099, 13292.2356334715,
                  16894.0572542597, 31020.2746327466, 19225.652102134, 0.0, 41880.1956106261, 35724.0245734204, 48416.9563909283,
                  39389.3148541711, 9414.51106651936, 41121.5359539506, 45320.4281001263, 25908.6648955616, 6555.85933111986,
                  43657.7713233115, 0.0, 0.0, 12604.1293602869, 0.0, 14842.8769023166, 41192.2105510858, 11401.6223352573,
                  31740.8103272138, 3701.61977516492, 30979.5732193265, 18798.4147409621, 3037.56781984674, 26532.8669292666,
                  18230.4237031799, 30117.3723750415, 13874.2451117345, 48325.4503775733, 14004.0635236353, 0.0, 41965.0795901386,
                  15764.982406929, 22381.2122539226, 12849.1264583715, 20887.3773053226, 41392.4669452232, 6283.17530013991,
                  19608.8809590778, 43275.1927866184, 7946.90339174959, 4233.30469585643, 0.0, 9337.29802780956, 19598.0175163873,
                  0.0, 49122.6573595682, 0.0, 19248.8185402411, 16261.5202457145, 24783.6533440257, 31843.1314081288, 25766.4342624988,
                  15441.1310395924, 17818.8547932802, 27555.2102761873, 14985.058461061, 10049.7838572, 35274.9409822078, 0.0, 0.0,
                  23554.5335852946, 6978.04339209664, 15220.6096886138, 16618.9893260873, 27072.7976656318, 38055.7486513712,
                  10647.4101614314, 837.033020313349, 41934.5519352841, 28350.0980527709, 46592.9320853448, 18499.5499625087,
                  12397.5172641178, 47102.3332595296, 4342.24017973754, 0.0, 48880.2396711122, 32225.9431260072, 0.0, 21539.2831007524,
                  34456.2398246651, 49045.5635156947, 15050.1970003035, 24077.8343464443, 31843.2696677631, 26017.8648699345,
                  36940.8346778002, 36279.7812045972, 47657.4290437036, 32706.8658376431, 10843.0641530676, 0.0, 28908.6423833304,
                  0.0, 15611.6403423159, 44573.2566334234, 12951.778531092, 24460.8209168726, 32336.5710606241, 47026.7000545894,
                  24250.081674673, 41827.381497564},
          {true}//backwords
          };

}
