/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * rapolicy - check argus records against cisco access control
 *            configuration.
 *
 * written by Carter Bullard, Olaf Gellert
 *
 */


#include <argus_client.h>
#include <rapolicy.h>
#include <signal.h>
#include <ctype.h>

int RaInitialized = 0;

int RaReadPolicy (struct RaPolicyPolicyStruct **, char *);
int RaParsePolicy (struct RaPolicyPolicyStruct *, char *);
int RaCheckPolicy (struct ArgusRecord *, struct RaPolicyPolicyStruct *);
int RaMeetsPolicyCriteria (struct ArgusRecord *, struct RaPolicyPolicyStruct *);
int RaDoNotification (struct ArgusRecord *, struct RaPolicyPolicyStruct *);

struct RaPolicyPolicyStruct *policy = NULL;

void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      RaInitialized++;
      RaWriteOut = 0;

      if (ArgusFlowModelFile != NULL) {
         RaReadPolicy (&policy, ArgusFlowModelFile);
      }
   }
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if (!RaParseCompleting) {
      RaParseCompleting++;
   }
}

void
ArgusClientTimeout ()
{
}

void
parse_arg (int argc, char**argv)
{ 
}


void
usage ()
{
   extern char version[];
   fprintf (stderr, "Rapolicy Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [-f policyfile] [ra-options] [- filter-expression]\n", ArgusProgramName);

   fprintf (stderr, "options: -f <policyfile>    read Cisco access control policy file.\n");
   fprintf (stderr, "         -a                 print record summaries on termination.\n");
   fprintf (stderr, "         -A                 print application bytes.\n");
   fprintf (stderr, "         -b                 dump packet-matching code.\n");
   fprintf (stderr, "         -C                 treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "         -d <bytes>         print number of <bytes> from user data capture buffer.\n");
   fprintf (stderr, "                   format:  num | s<num> | d<num> | s<num>:d<num>\n");
#if defined (ARGUSDEBUG)
   fprintf (stderr, "         -D <level>         specify debug level\n");
#endif
   fprintf (stderr, "         -e <encode>        convert user data using <encode> method.\n");
   fprintf (stderr, "                            Supported types are <Ascii> and <Encode64>.\n");
   fprintf (stderr, "         -E <file>          write records that are rejected by the filter into <file>\n");
   fprintf (stderr, "         -F <conffile>      read configuration from <conffile>.\n");
   fprintf (stderr, "         -h                 print help.\n");
   fprintf (stderr, "         -n                 don't convert numbers to names.\n");
   fprintf (stderr, "         -p <digits>        print fractional time with <digits> precision.\n");
   fprintf (stderr, "         -q                 quiet mode. don't print record outputs.\n");
   fprintf (stderr, "         -r <file>          read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -R                 print out response data when availabile.\n");
   fprintf (stderr, "         -s [-][+[#]]field  specify fields to print.\n");
   fprintf (stderr, "         -S <host[:port]>   specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "         -t <timerange>     specify <timerange> for reading records.\n");
   fprintf (stderr, "                   format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                            timeSpecification: [[[yyyy/]mm/]dd.]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                                 [yyyy/]mm/dd\n");
   fprintf (stderr, "                                                 -%%d{yMdhms}\n");
   fprintf (stderr, "         -T <secs>          attach to remote server for T seconds.\n");
   fprintf (stderr, "         -u                 print time in Unix time format.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth>     specify <user/auth> authentication information.\n");
#endif
   fprintf (stderr, "         -w <file>          write output to <file>. '-' denotes stdout.\n");
   fprintf (stderr, "         -z                 print Argus TCP state changes.\n");
   fprintf (stderr, "         -Z <s|d|b>         print actual TCP flag values.<'s'rc | 'd'st | 'b'oth>\n");
   exit(1);
}

int RaLabelCounter = 0;
void RaProcessThisRecord (struct ArgusRecord *argus);

void
RaProcessRecord (struct ArgusRecord *argus)
{
   struct ArgusRecordStore *store = NULL;
   int process = 0;

   if (argus->ahdr.type & ARGUS_MAR)
      process++;

   else {
      switch (argus->ahdr.status & 0xFFFF) {
         case ETHERTYPE_IP:
            switch (argus->argus_far.flow.ip_flow.ip_p) {
               case IPPROTO_TCP:
               case IPPROTO_UDP:
               case IPPROTO_ICMP:
                  process++;

               default:
                  break;
            }
            break;

         case ETHERTYPE_ARP:
         case ETHERTYPE_REVARP:
         default:
            break;
      }
   }


   if (process) {
      if (RaCheckPolicy (argus, policy)) {
         if ((store = RaNewArgusStore(argus)) != NULL) {
            if ((store->data[0] = RaNewArgusData(argus)) != NULL) {
               store->data[0]->status |= RA_MODIFIED;
               RaSendArgusRecord (store);
               RaDeleteArgusStore (store);

            } else
               ArgusLog (LOG_ERR, "RaProcessThisRecord: RaNewArgusData(0x%x) %s\n",
                                  argus, strerror(errno));
         } else
            ArgusLog (LOG_ERR, "RaProcessThisRecord: RaNewArgusStore(0x%x) %s\n",
                                  argus, strerror(errno));
      }
   }

   if (Nflag > 0) {
      if (--Nflag == 0) {
         RaParseComplete(SIGINT);
      }
   }
}


int
RaSendArgusRecord(struct ArgusRecordStore *store)
{
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   int retn = 0, i;

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if ((data = store->data[i]) != NULL) {
         argus = data->argus;

         if (!aflag) {
#ifdef _LITTLE_ENDIAN
            ArgusHtoN(argus);
#endif
            ArgusGenerateCanonicalRecord (argus, RaThisCanon);
 
#ifdef _LITTLE_ENDIAN
            ArgusNtoH(argus);
#endif
            if ((retn = argus_filter (ArgusFilterCode.bf_insns, RaThisCanon)) == 0)
               return (retn);
         }

         if ((ArgusWfileList != NULL) && (!(ArgusListEmpty(ArgusWfileList)))) {
            struct ArgusWfileStruct *wfile = NULL, *start = NULL;
 
            if ((wfile = ArgusFrontList(ArgusWfileList)) != NULL) { 
               start = wfile; 
               do { 
                  if ((exceptfile == NULL) || strcmp(wfile->filename, exceptfile)) { 
#ifdef _LITTLE_ENDIAN
                     ArgusHtoN(argus); 
#endif
                     ArgusWriteNewLogfile (wfile, argus); 
 
#ifdef _LITTLE_ENDIAN
                     ArgusNtoH(argus); 
#endif
                  }   
                  ArgusPopFrontList(ArgusWfileList); 
                  ArgusPushBackList(ArgusWfileList, wfile); 
                  wfile = ArgusFrontList(ArgusWfileList); 
 
               } while (wfile != start); 
            }   

         } else {
            if (argus->ahdr.type & ARGUS_MAR)
               printf ("%s\n", get_man_string (argus));
                  
            else {

               if (Lflag && !(qflag)) {
                  if (RaLabel == NULL)
                     RaLabel = RaGenerateLabel(argus);

                  if (!(RaLabelCounter++ % Lflag)) {
                     printf ("%s\n", RaLabel);
                     fflush (stdout);
                  }

                  if (Lflag < 0)
                     Lflag = 0;
               }

               ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);

               switch (argus->ahdr.status & 0xFFFF) {
                  case ETHERTYPE_IP:
                     switch (argus->argus_far.flow.ip_flow.ip_p) {
                        case IPPROTO_TCP: { 
                           int srcportind = 0;
                           if (argus->argus_far.flow.ip_flow.sport == 20) {
                              if (RaPrintAlgorithms[3] == ArgusPrintDstPort) {
                                 RaPrintAlgorithms[3] = ArgusPrintSrcPort;
                                 srcportind++;
                              }
                           }
                           
                           printf ("%s", get_tcp_string (argus));

                           if (srcportind)
                              RaPrintAlgorithms[3] = ArgusPrintDstPort;
                           break;
                        }
         
                        case IPPROTO_ICMP:              
                           printf ("%s", get_icmp_string (argus));
                           break;
      
                        default:
                           printf ("%s", get_ip_string (argus));
                           break;
                     }
                     break;
         
                  case ETHERTYPE_ARP:
                  case ETHERTYPE_REVARP:
                     printf ("%s", get_arp_string (argus));
                     break;
         
                  default:
                     printf ("%s", get_nonip_string (argus));
                     break;
               }

               printf ("\n");
            }

            fflush (stdout);
         }
      }
   }

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif

   return (retn);
}

#include <errno.h>


char *RaParseErrorStr [POLICYERRORNUM] = {
   "access-list identifier not found",
   "policy id number not found",
   "permit/deny indication not found",
   "protocol indentifier not found",
   "no source address defined",
   "no source address mask defined",
   "wrong source port operator",
   "wrong source port specification"
   "no destination address defined",
   "no destination address mask defined",
   "wrong destination port operator",
   "wrong destination port specification",
   "access violation notification not found",
};

int
RaReadPolicy (struct RaPolicyPolicyStruct **policy, char *file)
{
   int retn = 1, linenum = 0;
   struct RaPolicyPolicyStruct *pol, *policyLast = NULL;
   FILE *fd;
   char buffer [1024], *str;

   if (file) {
      if ((fd = fopen (file, "r")) != NULL) {
         while (fgets (buffer, 1024, fd)) {
            linenum++;
            if ((*buffer != '#') && (*buffer != '\n') && (*buffer != '!')) {
               if ((pol = (struct RaPolicyPolicyStruct *) calloc (1, sizeof (struct RaPolicyPolicyStruct))) != NULL) {
                  pol->str = strdup (buffer);

                  if (!(RaParsePolicy (pol, buffer))) {
                     str = RaParseErrorStr[RaParseError];
                     ArgusLog (LOG_ERR, "RaReadPolicy %s parse error line %d: %s\n", file, linenum, str);
                     retn = 0;
                  }
                  if (policyLast)  {
                     policyLast->nxt = pol;
                     pol->prv = policyLast;
                     policyLast = pol;
                  } else
                     *policy = policyLast = pol;
               }
            }
         }
         fclose (fd);
      } else {
         retn = 0;
         ArgusLog (LOG_ERR, "RaReadPolicy: fopen %s\n", strerror(errno));
      }
   }

#if defined(ARGUSDEBUG)
   ArgusDebug (2, "RaReadPolicy (0x%x, %s) returning %d\n", policy, file, retn);
#endif

   return (retn);
}

#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <ctype.h>

int
RaParsePolicy (struct RaPolicyPolicyStruct *policy, char *buf)
{
   int retn = 1, error = -1, state = 0;
   char *act_word, *nxt_word;

   if (strncmp (buf, "access", 6)) {
      if (strstr (buf, "ip"))
         if (strstr (buf, "source-route"))
            policy->type = RA_SRCROUTED;

   } else {
      policy->type = RA_ACCESSLIST;

      /* set act_word to the word to be processed now, nxt_word to the
         following word */
      act_word=strtok(buf, " ");
      if (act_word==NULL) {
        error++;
      } else {
        nxt_word=strtok(NULL, " ");
        if (nxt_word==NULL) {
          error++;
        } else {
          while ((nxt_word != NULL) && (*nxt_word != '\n') && (error==-1)) {
            switch (state) {
               case POLICYSTRING: 
                  if ((strcmp (act_word, POLICY_STRING))) {
                    error=POLICYERR_NOACL;
                    }
                  break;
   
               case POLICYID: 
                  if (!(policy->policyID = atoi (act_word))) {
                    error=POLICYERR_NOID;
                    }
                  break;
   
               case POLICYACTION: 
                  if (!(strcmp (act_word, "permit")))
                     policy->flags |= RA_PERMIT;
                  else if (!(strcmp (act_word, "deny")))
                     policy->flags |= RA_DENY;
                  else error=POLICYERR_NOACTION;
                  break;
   
               case POLICYPROTO: 
                     if (isdigit((int)*act_word)) {
                        policy->proto = atoi(act_word);
                        policy->flags |= RA_PROTO_SET;
                     } else {
                        struct protoent *proto;
                        if ((proto = getprotobyname(act_word)) != NULL) {
                           policy->proto = proto->p_proto;
                           policy->flags |= RA_PROTO_SET;
                        } else {
                           error=POLICYERR_NOPROTO;
                        }
                     }

                     break;

               case POLICYSRC:
                    if (!(strcmp(act_word, "any"))) {
                      policy->src.addr = ntohl(inet_addr("0.0.0.0"));
                      policy->src.mask = ntohl(inet_addr("255.255.255.255"));
                    } else if (!(strcmp(act_word, "host"))) {
                      policy->src.addr = ntohl(inet_addr(nxt_word));
                      policy->src.mask = ntohl(inet_addr("0.0.0.0"));
                      act_word=nxt_word;
                      nxt_word=strtok(NULL, " ");
		      if (nxt_word == NULL) break;
		    } else {
		      policy->src.addr = ntohl(inet_addr(act_word));
		      policy->src.mask = ntohl(inet_addr(nxt_word));
                      act_word=nxt_word;
                      nxt_word=strtok(NULL, " ");
		      if (nxt_word == NULL) break;
		    }

		    /* now get next words, check for port number */
		    if (!(strncmp(nxt_word, "eq", 2))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->src_action = RA_EQ;
		       if (!(policy->src_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_SPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                    else if (!(strncmp (nxt_word, "lt", 2))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->src_action = RA_LT;
		       if (!(policy->src_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_SPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                    else if (!(strncmp (nxt_word, "gt", 2))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->src_action = RA_GT;
		       if (!(policy->src_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_SPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                    else if (!(strncmp (nxt_word, "neq", 3))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->src_action = RA_NEQ;
		       if (!(policy->src_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_SPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                    else if (!(strncmp (nxt_word, "established",11))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->src_action = RA_EST;
		       if (!(policy->src_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_SPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                    else if (!(strncmp (nxt_word, "range", 5))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->src_action = RA_RANGE;
		       if (!(policy->src_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_SPORT;
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) {
		         error=POLICYERR_NODSTADR;
		         break;
                         }
		       if (!(policy->src_port_hi = (arg_uint16) atoi (nxt_word))) error=POLICYERR_SPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                  break;

               case POLICYDST:
                    if (!(strcmp(act_word, "any"))) {
                      policy->dst.addr = ntohl(inet_addr("0.0.0.0"));
                      policy->dst.mask = ntohl(inet_addr("255.255.255.255"));
                    } else if (!(strcmp(act_word, "host"))) {
                      policy->dst.addr = ntohl(inet_addr(nxt_word));
                      policy->dst.mask = ntohl(inet_addr("0.0.0.0"));
                      act_word=nxt_word;
                      nxt_word=strtok(NULL, " ");
		      if (nxt_word == NULL) {
		        break;
		        }
		    } else {
		      policy->dst.addr = ntohl(inet_addr(act_word));
		      policy->dst.mask = ntohl(inet_addr(nxt_word));
                      act_word=nxt_word;
                      nxt_word=strtok(NULL, " ");
		      if (nxt_word == NULL) {
		        break;
		        }
		    }

		    /* now get next words, check for port number */
		    if (!(strncmp(nxt_word, "eq", 2))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->dst_action = RA_EQ;
		       /* printf("xxx dst policy RA_EQ\n"); */
		       if (!(policy->dst_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_DPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                    else if (!(strncmp (nxt_word, "lt", 2))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->dst_action = RA_LT;
		       if (!(policy->dst_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_DPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                    else if (!(strncmp (nxt_word, "gt", 2))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->dst_action = RA_GT;
		       if (!(policy->dst_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_DPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                    else if (!(strncmp (nxt_word, "neq", 3))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->dst_action = RA_NEQ;
		       if (!(policy->dst_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_DPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                    else if (!(strncmp (nxt_word, "established",11))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->dst_action = RA_EST;
		       if ((nxt_word != NULL) && !(policy->dst_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_DPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                    else if (!(strncmp (nxt_word, "range", 5))) {
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       policy->dst_action = RA_RANGE;
		       if (!(policy->dst_port_low = (arg_uint16) atoi (nxt_word))) error=POLICYERR_DPORT;
		       act_word=nxt_word; nxt_word=strtok(NULL, " ");
		       if (!(policy->dst_port_hi = (arg_uint16) atoi (nxt_word))) error=POLICYERR_DPORT;
                       act_word=nxt_word;
                       nxt_word=strtok(NULL, " ");
		       if (nxt_word == NULL) break;
		       }
                  break;
   
               case POLICYNOTIFICATION: 
                  break;
            } /* end of switch */
   
          buf = NULL;
          state++;
          act_word=nxt_word; nxt_word=strtok(NULL, " ");
          } /* end of while */
        }
     }
   }

   if (error != -1) {
      RaParseError = error; retn = 0;
   }

   return (retn);
}


int
RaCheckPolicy (struct ArgusRecord *argus, struct RaPolicyPolicyStruct *policy)
{
   int retn = 0, policymatch = 0;
   /* xxx weg damit */

   /* printf("\nxxx entry: src %u sp %u  -> dst %u dp %u\n",
     flow->ip_flow.ip_src, flow->ip_flow.sport,
     flow->ip_flow.ip_dst, flow->ip_flow.dport); */

   if (!((argus->ahdr.status) & ARGUS_MAR)) {
      if (policy) {
         while (policy) {
            if ((retn = RaMeetsPolicyCriteria (argus, policy))) {
               retn = RaDoNotification (argus, policy);
               policymatch = 1;
               break;
            }
            policy = policy->nxt;
         }
      }
   }

   return (retn);
}


int
RaMeetsPolicyCriteria (struct ArgusRecord *argus, struct RaPolicyPolicyStruct *policy)
{
   int retn = 0, i = 0;
   struct ArgusFlow *flow = &argus->argus_far.flow;
   u_char proto = flow->ip_flow.ip_p;

   arg_uint32 saddr = 0, daddr = 0;
   arg_uint16 sport = 0, dport = 0;
   
   switch (policy->type) {
   case RA_SRCROUTED:
      if ((argus->argus_far.attr_ip.soptions & (ARGUS_SSRCROUTE | ARGUS_LSRCROUTE)) || 
          (argus->argus_far.attr_ip.doptions & (ARGUS_SSRCROUTE | ARGUS_LSRCROUTE)))
         retn++;
      break;

   case RA_ACCESSLIST:
      saddr = flow->ip_flow.ip_src;
      daddr = flow->ip_flow.ip_dst;
      sport = flow->ip_flow.sport;
      dport = flow->ip_flow.dport;
   
      for (i = 0, retn = 1; ((i < POLICYTESTCRITERIA) && retn); i++) {
         retn = 0;
         switch (i) {
            case POLICYTESTPROTO:
               if (policy->flags & (RA_PROTO_SET)) {
                  if (policy->proto) {
                     if (policy->proto == proto)
                        retn++;
                  } else
                     retn++;
               }
               break;
   
            case POLICYTESTSRC:
               if ((saddr & ~policy->src.mask) == policy->src.addr) {
                  retn++;
                  }
               break;
            case POLICYTESTDST:
               if ((daddr & ~policy->dst.mask) == policy->dst.addr) {
                  retn++;
                  }
               break;
            case POLICYTESTSRCPORT:
               switch (policy->src_action) {
                  case  RA_EQ:
                     if (sport == policy->src_port_low) {
                        retn++;
                     } else {
                     }
                     continue;
                  case  RA_LT:
                     if (sport < policy->src_port_low) {
                        retn++;
                        }
                        continue;
                  case  RA_GT:
                     if (sport > policy->src_port_low) {
                        retn++;
                        }
                        continue;
                  case RA_NEQ:
                     if (sport != policy->src_port_low) {
                        retn++;
                        }
                        continue;
                  case RA_RANGE:
                     if ((sport < policy->src_port_low) ||
                         (sport > policy->src_port_hi)) {
                        retn++;
                        }
                        continue;
                  case  RA_EST: {
                     if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
                        struct ArgusTCPObject *tcp = NULL;

                        tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

                        if (!(tcp->status & ARGUS_SAW_SYN)) {
                           retn++;
                           }
                     }
                     continue;
                  }
               }

            case POLICYTESTDSTPORT:
               switch (policy->dst_action) {
                  case  RA_EQ:
                     if (dport == policy->dst_port_low) {
                        retn++;
                     } else {
                     }
                     continue;
                  case  RA_LT:
                     if (dport < policy->dst_port_low) {
                        retn++;
                        }
                        continue;
                  case  RA_GT:
                     if (dport > policy->dst_port_low) {
                        retn++;
                        }
                        continue;
                  case RA_NEQ:
                     if (dport != policy->dst_port_low) {
                        retn++;
                        }
                        continue;
                  case RA_RANGE:
                     if ((dport >= policy->dst_port_low) ||
                         (dport <= policy->dst_port_hi)) {
                        retn++;
		     }
                     continue;
                  case  RA_EST: {
                     if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
                        struct ArgusTCPObject *tcp = NULL;

                        tcp = (struct ArgusTCPObject *)ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX];

                        if (!(tcp->status & ARGUS_SAW_SYN))
                           retn++;
                     }
                     continue;
                  }
                  default:
                     retn++; break;
           } /* end of inner switch */
         } /* end of outer switch */
      } /* end of for */
   }

   return (retn);
}

int
RaDoNotification (struct ArgusRecord *argus, struct RaPolicyPolicyStruct *policy)
{
   int retn = 1;

   if (policy) {
      if (policy->flags & RA_PERMIT) {
         if (Argusdflag > 1) {
            printf ("%s %s ", "policy: permitted", policy->str);
         } else
            retn = 0;

      } else {
         if (Argusdflag)
            printf ("%s %s ", "policy: denyed", policy->str);
      }
   }

   return (retn);
}
