/*
------------------------------------------------------------------------------
denef - Decode NEF image files
Copyright (C) 2000 Daniel Stephens (daniel@cheeseplant.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
------------------------------------------------------------------------------
*/

#include <iostream.h>
#include <math.h>
#include <stdlib.h>

#include "interp.h"

static const char *rcsid="$Id: interp_gradlumin.cc,v 1.4 2000/09/18 07:51:43 daniel Exp $";

// Luminance based interpolator, which uses all colors to interpolate
// luminance channel and then interpolates colors from that.
// Algorithm has some 'gradient based' image quality enhancements.

// Define the region we're willing to interpolate in a single pass (used to
// determine how large of a 'workspace' area to allocate.

static const int zoneWidth  = 256;
static const int zoneHeight = 256;
static const int zoneSize = zoneWidth * zoneHeight;

// Define coeficients for each color when determining luminance.
// These may be a little off but I dont know the ACTUAL colors of the filters
// over the CCD on a D1. This works on the basic assumption that the D1's
// CCD returns a linear strength for each channel.

// From C. Poynton's FAQ on Gamma (http://www.inforamp.net/~poynton/)
// CIE Rec 709
static const double luminCoefRed   = 0.2126;
static const double luminCoefGreen = 0.7152;
static const double luminCoefBlue  = 0.0722;

gradientLuminanceInterpolator::gradientLuminanceInterpolator()
{
    tmpY  = new double[zoneSize];
    tmpCr = new double[zoneSize];
    tmpCb = new double[zoneSize];
}

gradientLuminanceInterpolator::~gradientLuminanceInterpolator()
{
    delete [] tmpY;
    delete [] tmpCr;
    delete [] tmpCb;
}

int
gradientLuminanceInterpolator::Fringe() const
{
    return 2;
}

int
gradientLuminanceInterpolator::Max_Width() const
{
    return zoneWidth - 4;
}

int
gradientLuminanceInterpolator::Max_Height() const
{
    return zoneHeight - 4;
}

void
gradientLuminanceInterpolator::Interpolate(const nefImageData &nef,
					   int xofs, int yofs,
					   int width, int height,
					   fullImageData &result,
					   int rxofs, int ryofs)
{
    /* This function takes a chunk of the NEF data and creates the decoded
       image from it.

       Little bit more complex luminance based algorithm, attempts to be
       intelligent about 'edges' too.
    */

    // Interpolate 'intermediate' red and blue values, normalize them all
    // to 0.0 - 1.0
    
    for (int y=-2; y<(height+2); ++y) {
	const short *pn  = nef.Data(xofs, yofs + y - 1);
	const short *p   = nef.Data(xofs, yofs + y    );
	const short *ps  = nef.Data(xofs, yofs + y + 1);

	int ofs = 0;
	if (y & 1) {ofs=1; ++pn; ++p; ++ps;}

	for (int x=ofs-2; x<(width+2); x+=2, pn+=2, p+=2, ps+=2) {
	    double X;
	    double gh = fabs(pn[-1]-ps[ 1]);
	    double gv = fabs(pn[ 1]-ps[-1]);

	    if (gh > 2.0 * gv) {
		X=(double)(pn[ 1]+ps[-1]) * 0.5;
	    } else if (gv > 2.0 * gh) {
		X=(double)(pn[-1]+ps[ 1]) * 0.5;
	    } else {
		X=(double)(pn[-1]+pn[1]+ps[-1]+ps[1]) * 0.25;
	    }

	    X /=  (double)nef.Max_Value();

	    if (y & 1) {
		tmpCr[(y+2)*zoneWidth + (x+2)] = ((double)*p /
						  (double)nef.Max_Value());
		tmpCb[(y+2)*zoneWidth + (x+2)] = X;
	    } else {
		tmpCr[(y+2)*zoneWidth + (x+2)] = X;
		tmpCb[(y+2)*zoneWidth + (x+2)] = ((double)*p  /
						  (double)nef.Max_Value());
	    }
	}
    }

    // Calculate Luminance for each pixel, normalized to 0.0 - 1.0

    for (int y=-2; y<(height+2); ++y) {
	const short *pn  = nef.Data(xofs, yofs + y - 1);
	const short *p   = nef.Data(xofs, yofs + y    );
	const short *ps  = nef.Data(xofs, yofs + y + 1);

	int ofs = (y+2)*zoneWidth;

	bool green_pixel = (y & 1);

	for (int x=-2; x<(width+2);
	     ++x, ++pn, ++p, ++ps, ++ofs, green_pixel = !green_pixel) {
	    double R,B,G;
	    
	    if (green_pixel) {
		G=(double)*p;

		double gh = fabs(tmpCr[ofs-1]-tmpCr[ofs+1]);
		double gv = fabs(tmpCr[ofs-zoneWidth]-tmpCr[ofs+zoneWidth]);
//		gh=gv;

		if (gh > 2.0 * gv) {
		    R=(tmpCr[ofs-zoneWidth]+tmpCr[ofs+zoneWidth]) * 0.5;
		} else if (gv > 2.0 * gh) {
		    R=(tmpCr[ofs-1]+tmpCr[ofs+1]) * 0.5;
		} else {
		    R=(tmpCr[ofs-1] + tmpCr[ofs+1] +
		       tmpCr[ofs-zoneWidth] + tmpCr[ofs+zoneWidth]) * 0.25;
		}

		gh = fabs(tmpCb[ofs-1]-tmpCb[ofs+1]);
		gv = fabs(tmpCb[ofs-zoneWidth]-tmpCb[ofs+zoneWidth]);
//		gh=gv;

		if (gh > 2.0 * gv) {
		    B=(tmpCb[ofs-zoneWidth]+tmpCb[ofs+zoneWidth]) * 0.5;
		} else if (gv > 2.0 * gh) {
		    B=(tmpCb[ofs-1]+tmpCb[ofs+1]) * 0.5;
		} else {
		    B=(tmpCb[ofs-1] + tmpCb[ofs+1] +
		       tmpCb[ofs-zoneWidth] + tmpCb[ofs+zoneWidth]) * 0.25;
		}
	    } else {
		double gh = fabs(p[-1]-p[1]);
		double gv = fabs(pn[0]-ps[0]);

		if (gh > 2.0 * gv) {
		    G=(double)(pn[0]+ps[0]) * 0.5;
		} else if (gv > 2.0 * gh) {
		    G=(double)(p[1]+p[-1]) * 0.5;
		} else {
		    G=(double)(pn[0]+p[1]+p[-1]+ps[0]) * 0.25;
		}

		R = tmpCr[ofs];
		B = tmpCb[ofs];
	    }

	    G /= (double)nef.Max_Value();

	    double Y = (luminCoefRed   * R + 
			luminCoefGreen * G +
			luminCoefBlue  * B);
	    
	    tmpY[(y+2)*zoneWidth + (x+2)] = Y;
	}
    }

    // Now calculate chrominances.. Start by transferring known values

    for (int y=-2; y<(height+2); ++y) {
	const short *p   = nef.Data(xofs, yofs + y    );

	int x = -2;
	int ofs = (y+2) * zoneWidth;
	if (y & 1) {++x; ++ofs; ++p;}

	for (; x<(width+2); x+=2, ofs+=2, p+=2) {
	    double Y = tmpY[ofs];
	    if (y & 1) {
		tmpCr[ofs] = (double)*p / (double)nef.Max_Value() - Y;
	    } else {
		tmpCb[ofs] = (double)*p / (double)nef.Max_Value() - Y;
	    }
	}
    }

    // Next interpolate 'diagonals' again

    for (int y=-2; y<(height+2); ++y) {
	int x = -2;
	int ofs = (y+2) * zoneWidth;
	if (y & 1) {++x; ++ofs;}

	for (; x<(width+2); x+=2, ofs+=2) {
	    double *Cx;
	    if (y & 1) {
		Cx = tmpCb;
	    } else {
		Cx = tmpCr;
	    }
	    double X;
	    double gh = fabs(Cx[ofs - zoneWidth - 1]-Cx[ofs + zoneWidth + 1]);
	    double gv = fabs(Cx[ofs - zoneWidth + 1]-Cx[ofs + zoneWidth - 1]);

	    if (gh > 2.0 * gv) {
		X=(Cx[ofs - zoneWidth + 1] + Cx[ofs + zoneWidth - 1]) * 0.5;
	    } else if (gv > 2.0 * gh) {
		X=(Cx[ofs - zoneWidth - 1] + Cx[ofs + zoneWidth + 1]) * 0.5;
	    } else {
		X=(Cx[ofs - zoneWidth - 1] + Cx[ofs + zoneWidth + 1] +
		   Cx[ofs - zoneWidth + 1] + Cx[ofs + zoneWidth - 1]) * 0.25;
	    }

	    Cx[ofs] = X;
	}
    }

    // Finally calculate color values for each pixel

    for (int y=0; y<height; ++y) {
	int ofs = (y+2)*zoneWidth + 2;

	bool green_pixel = (y & 1);

	for (int x=0; x<width; ++x, ++ofs, green_pixel = !green_pixel) {
	    double Y,Cr,Cb;
	    
	    Y=tmpY[ofs];

	    if (green_pixel) {
		double gh = fabs(tmpCr[ofs-1]-tmpCr[ofs+1]);
		double gv = fabs(tmpCr[ofs-zoneWidth]-tmpCr[ofs+zoneWidth]);
//		gh=gv;

		if (gh > 2.0 * gv) {
		    Cr=(tmpCr[ofs-zoneWidth]+tmpCr[ofs+zoneWidth]) * 0.5;
		} else if (gv > 2.0 * gh) {
		    Cr=(tmpCr[ofs-1]+tmpCr[ofs+1]) * 0.5;
		} else {
		    Cr=(tmpCr[ofs-1] + tmpCr[ofs+1] +
		       tmpCr[ofs-zoneWidth] + tmpCr[ofs+zoneWidth]) * 0.25;
		}

		gh = fabs(tmpCb[ofs-1]-tmpCb[ofs+1]);
		gv = fabs(tmpCb[ofs-zoneWidth]-tmpCb[ofs+zoneWidth]);
//		gh=gv;

		if (gh > 2.0 * gv) {
		    Cb=(tmpCb[ofs-zoneWidth]+tmpCb[ofs+zoneWidth]) * 0.5;
		} else if (gv > 2.0 * gh) {
		    Cb=(tmpCb[ofs-1]+tmpCb[ofs+1]) * 0.5;
		} else {
		    Cb=(tmpCb[ofs-1] + tmpCb[ofs+1] +
			tmpCb[ofs-zoneWidth] + tmpCb[ofs+zoneWidth]) * 0.25;
		}
	    } else {
		Cr = tmpCr[ofs];
		Cb = tmpCb[ofs];
	    }

	    rgbTriple &res = result(rxofs+x, ryofs+y);

	    res.r = Cr + Y;
	    res.b = Cb + Y;
	    
	    res.g = (Y - luminCoefRed * (Cr + Y) - luminCoefBlue * (Cb + Y)) 
		/ luminCoefGreen;
	}
    }
}

