/**
 * Copyright (C) 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "pngoutput.h"
#include "exrinput.h"

static double intensity_to_voltage_srgb( double val )
{
    /* Handle invalid values before doing a gamma transform. */
    if( val < 0.0 ) return 0.0;
    if( val > 1.0 ) return 1.0;

    /* Note: sRGB uses an exponent of 2.4 but it still advertises
     * a display gamma of 2.2.
     */
    if( val <= 0.0031308 ) {
        return 12.92 * val;
    }
    return ( ( 1.055 * pow( val, 1.0 / 2.4 ) ) - 0.055 );
}

int main( int argc, char **argv )
{
    exrinput_t *exrin;
    pngoutput_t *pngout;
    int width, height;
    uint8_t *scanline;
    int y;

    if( argc < 3 ) {
        fprintf( stderr, "Usage: exrtopng input.exr output.png\n" );
        return 1;
    }

    exrin = exrinput_new( argv[ 1 ] );
    if( !exrin ) {
        fprintf( stderr, "exrtopng: Cannot open %s\n", argv[ 1 ] );
        return 1;
    }
    width = exrinput_get_width( exrin );
    height = exrinput_get_height( exrin );
    fprintf( stdout, "exrtopng: %s [%dx%d]\n", argv[ 1 ], width, height );

    scanline = malloc( width * 4 );
    if( !scanline ) {
        fprintf( stderr, "exrtopng: Cannot allocate memory.\n" );
        exrinput_delete( exrin );
        return 1;
    }

    pngout = pngoutput_new( argv[ 2 ], width, height, 1, 1 );
    if( !pngout ) {
        fprintf( stderr, "exrtopng: Cannot open %s.\n", argv[ 2 ] );
        exrinput_delete( exrin );
        free( scanline );
        return 1;
    }

    for( y = 0; y < height; y++ ) {
        double *inscanline = exrinput_get_scanline( exrin, y );
        int x;

        fprintf( stderr, "exrtopng: Writing scanline %d\r", y );

        for( x = 0; x < width; x++ ) {
            double r = inscanline[ (x * 4) + 0 ];
            double g = inscanline[ (x * 4) + 1 ];
            double b = inscanline[ (x * 4) + 2 ];
            double a = inscanline[ (x * 4) + 3 ];
            double rp, gp, bp;

            rp = intensity_to_voltage_srgb( r );
            gp = intensity_to_voltage_srgb( g );
            bp = intensity_to_voltage_srgb( b );

            scanline[ (x * 4) + 0 ] = (int) ( ( rp * 255.0 ) + 0.5 );
            scanline[ (x * 4) + 1 ] = (int) ( ( gp * 255.0 ) + 0.5 );
            scanline[ (x * 4) + 2 ] = (int) ( ( bp * 255.0 ) + 0.5 );
            scanline[ (x * 4) + 3 ] = (int) ( (  a * 255.0 ) + 0.5 );
        }
        pngoutput_scanline( pngout, scanline );
    }
    fprintf( stderr, "\nexrtopng: Done.\n" );

    pngoutput_delete( pngout );
    exrinput_delete( exrin );
    free( scanline );
    return 0;
}

