/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_cred_data.c
    \brief  Routines to handle lcmaps credential data
    \author Oscar Koeroo and Martijn Steenbakkers for the EU DataGrid.
*/


/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
/* #include <malloc.h> */
#include <string.h>

#include "_lcmaps_cred_data.h"
#include "lcmaps_log.h"
#include "lcmaps_vo_data.h"


#include "_lcmaps_credential.h"
#include "_lcmaps_runvars.h"

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int lcmaps_compare_gids(const void *, const void *);


/******************************************************************************
                       Define module specific variables
******************************************************************************/
static cred_data_t credData =
{
    (char *) NULL,                /*!< user globus DN */
    (uid_t *) NULL,               /*!< list of userIDs */
    (gid_t *) NULL,               /*!< list of primary groupIDs */
    (gid_t *) NULL,               /*!< list of secondary groupIDs */
    (lcmaps_vo_data_t *) NULL,    /*!< list of VO data structures */
    (char **) NULL,               /*!< list of VO data strings */
    (lcmaps_vo_mapping_t *) NULL, /*!< list of VO mapping structures */
    0,                            /*!< number of userIDs */
    0,                            /*!< number of primary groupIDs (in principle only one) */
    0,                            /*!< number of secondary groupIDs (could be any number) */
    0,                            /*!< number of VO data structures */
    0,                            /*!< number of VO data strings */
    0,                            /*!< number of VO mapping structures */
    (char *) NULL,                /*!< pool_index is the concat of a DN+GIDs for the DAS */
};

/******************************************************************************
Function:   addCredentialData
Description:
    Add a credential to the list of found credentials (uids, gids etc)

Parameters:
    datatype: type of credential
    data:     pointer to credential
Returns:
    0:  success
    -1: failure (unknown data type, realloc error)
******************************************************************************/
/*!
    \fn addCredentialData(
        int datatype,
        void *data
        )
    \brief Add a credential to the list of found credentials (uids, gids etc)

    The credential value is copied into list (memory is allocated for this)

    \param datatype type of credential
    \param data     pointer to credential

    \return 0 in case of succes
*/
int addCredentialData(
        int datatype,
        void *data
)
{
    switch(datatype) 
    {
        case DN                    :
                          if (data)
                          {
                              if ( (credData.dn = strdup(*(char **)data))==NULL)
				  return -1;
                          }
                          break;

        case UID                   :
                          if (data)
                          {
			      if ( (credData.uid = (uid_t *) realloc(credData.uid, ((credData.cntUid+1) * sizeof(uid_t)))) == NULL)
				  return -1;
                              credData.uid[credData.cntUid] = *((uid_t *)data);
                              credData.cntUid++;
                          }
                          break;
 
        case PRI_GID               :
                          if (data)
                          {
                              credData.cntPriGid++;
                              if ( (credData.priGid = (gid_t *) realloc(credData.priGid, (credData.cntPriGid * sizeof(gid_t)))) == NULL)
				  return -1;
                              credData.priGid[credData.cntPriGid - 1] = *((gid_t *)data);
                          }
                          break;

        case SEC_GID               :
                          if (data)
                          {
                              int foundgid;
                              int igid = 0;
                              gid_t newgid;

                              newgid = *((gid_t *)data);

                              /* Check if gid is already stored in the list */
                              foundgid = 0;
                              for (igid = 0; igid < credData.cntSecGid; igid++)
                              {
                                  if (newgid == credData.secGid[igid])
                                  {
                                      foundgid = 1;
                                      break;
                                  }
                              }
                              if (foundgid) break;

                              /* New gid so increase cntSecGid by 1, reallocate the array and sort it */
                              credData.cntSecGid++;
			      if ( (credData.secGid = (gid_t *) realloc(credData.secGid, (credData.cntSecGid * sizeof(gid_t)))) == NULL)
				  return -1;
                              credData.secGid[credData.cntSecGid - 1] = newgid;

                              /* sort the secondaries with qsort */
                              if (credData.cntSecGid > 1)
                              {
                                  qsort(
                                      credData.secGid,
                                      credData.cntSecGid,
                                      sizeof(gid_t),
                                      (int (*)(const void *,const void *))lcmaps_compare_gids
                                  );
                              }
                          }
                          break;

        case LCMAPS_VO_CRED        :
                          if (data)
                          {
			      if ( (credData.VoCred = (lcmaps_vo_data_t *) realloc(credData.VoCred, ((credData.cntVoCred+1) * sizeof(lcmaps_vo_data_t)))) == NULL)
				  return -1;
                              lcmaps_copyVoData(&(credData.VoCred[credData.cntVoCred]), (lcmaps_vo_data_t *) data);
                              credData.cntVoCred++;
                          }
                          break;

        case LCMAPS_VO_CRED_STRING :
                          if (data)
                          {
			      if ( (credData.VoCredString = (char **) realloc(credData.VoCredString, ((credData.cntVoCredString+1) * sizeof(char *)))) == NULL)
				  return -1;
			      if ( (credData.VoCredString[credData.cntVoCredString] = strdup(*((char **)data))) == NULL)
				  return -1;
                              credData.cntVoCredString++;
                          }
                          break;

        case LCMAPS_VO_CRED_MAPPING:
                          if (data)
                          {
			      if ( (credData.VoCredMapping = (lcmaps_vo_mapping_t *) realloc(credData.VoCredMapping, ((credData.cntVoCredMapping+1) * sizeof(lcmaps_vo_mapping_t)))) == NULL )
				  return -1;
                              lcmaps_copyVoMapping(&(credData.VoCredMapping[credData.cntVoCredMapping]), (lcmaps_vo_mapping_t *) data);
                              credData.cntVoCredMapping++;
                          }
                          break;

        case POOL_INDEX            :
                          if (data)
                          {
			      if ( (credData.pool_index = strdup(*(char **)data)) == NULL)
				  return -1;
                          }
                          break;



        default         :
                          return -1;
    }
    return 0;
}

/******************************************************************************
Function:   getCredentialData
Description:
    Get pointer to a list of credential data of a certain type

Parameters:
    datatype: type of credential
    count:    number of credentials found in list of datatype
Returns:
    pointer to list of credential data or NULL in case of failure
******************************************************************************/
/*!
    \fn getCredentialData(
        int datatype,
        int *count
        )
    \brief Get pointer to a list of credential data of a certain type

    \param datatype type of credential
    \param count    number of credentials found in list of datatype (filled by routine)

    \return pointer to list of credential data or NULL in case of failure
*/
void *getCredentialData(
        int datatype,
        int *count
)
{
    switch(datatype)
    {
        case DN:
            if (credData.dn != NULL) *count = 1;
            else *count = 0;
            return &(credData.dn);

        case UID: 
            *count = credData.cntUid;
            return (credData.uid);

        case PRI_GID: 
            *count = credData.cntPriGid;
            return (credData.priGid);

        case SEC_GID:
            *count = credData.cntSecGid;
            return (credData.secGid);

        case LCMAPS_VO_CRED: 
            *count = credData.cntVoCred;
            return (credData.VoCred);

        case LCMAPS_VO_CRED_STRING: 
            *count = credData.cntVoCredString;
            return (credData.VoCredString);

        case LCMAPS_VO_CRED_MAPPING : 
            *count = credData.cntVoCredMapping;
            return (credData.VoCredMapping);

        case POOL_INDEX:
            if (credData.pool_index != NULL) *count = 1;
            else *count = 0;
            return &(credData.pool_index);


        default         : return NULL;
    }
}

/******************************************************************************
Function:   lcmaps_cleanCredentialData
Description:
    Clean the credData structure

Parameters: none
Returns: 0
******************************************************************************/
/*!
    \fn lcmaps_cleanCredentialData()
    \brief Clean the credData structure

    \return 0
    \internal
*/
int lcmaps_cleanCredentialData(void)
{
    int i = 0;

    for (i = 0; i < credData.cntVoCred; i++)
        lcmaps_cleanVoData(&(credData.VoCred[i]));
    for (i = 0; i < credData.cntVoCredString; i++)
        if (credData.VoCredString[i]) free(credData.VoCredString[i]);
    for (i = 0; i < credData.cntVoCredMapping; i++)
        lcmaps_cleanVoMapping(&(credData.VoCredMapping[i]));

    if (credData.dn)            free(credData.dn);
    if (credData.uid)           free(credData.uid);
    if (credData.priGid)        free(credData.priGid);
    if (credData.secGid)        free(credData.secGid);
    if (credData.VoCred)        free(credData.VoCred);
    if (credData.VoCredString)  free(credData.VoCredString);
    if (credData.VoCredMapping) free(credData.VoCredMapping);
    if (credData.pool_index)    free(credData.pool_index);


    credData.dn               = NULL;
    credData.uid              = NULL;
    credData.priGid           = NULL;
    credData.secGid           = NULL;
    credData.VoCred           = NULL;
    credData.VoCredString     = NULL;
    credData.VoCredMapping    = NULL;
    credData.pool_index       = NULL;

    credData.cntUid           = 0;
    credData.cntPriGid        = 0;
    credData.cntSecGid        = 0;
    credData.cntVoCred        = 0;
    credData.cntVoCredString  = 0;
    credData.cntVoCredMapping = 0;

    return 0;
}

/******************************************************************************
Function:   lcmaps_printCredData
Description:
    print out the credData structure

Parameters:
    debug_level: the debug level
Returns:
    nothing
******************************************************************************/
/*!
    \fn lcmaps_printCredData(int debug_level)
    \brief Get pointer to a list of credential data of a certain type

    \param debug_level  the debug level

    \return nothing
*/
void lcmaps_printCredData(int debug_level)
{
    int i = 0;
    int bufsize = 1500;
    char * buffer = NULL;

    /* 2048 is the max log line for sys log, 1500 is very safe */
    if ( (buffer = calloc (sizeof(char), (bufsize + 1))) == NULL)   {
	lcmaps_log(LOG_ERR, "%s: Out of memory\n", __func__);
	return;
    }

    lcmaps_log_debug(5, "Credential Print:\n");

    if (credData.dn != NULL) 
    {
        if ((bufsize - strlen(buffer)) <= snprintf (&buffer[strlen(buffer)], bufsize - strlen(buffer), "DN:\"%s\"%s", 
                                                                                                       credData.dn,
                                                                                                       (credData.cntUid > 0) || (credData.cntPriGid > 0) || (credData.cntSecGid > 0) ? "->" : "")) 
            lcmaps_log(LOG_INFO, "LCMAPS: Warning: output truncated for DN.\n");
        /* lcmaps_log(debug_level, "dn: %s\n", credData.dn); */
    }
    for (i = 0; i < credData.cntUid; i++)
    {
        if ((bufsize - strlen(buffer)) <= snprintf (&buffer[strlen(buffer)], bufsize - strlen(buffer), "mapped uid:\'%d\'", 
                                                                                                       credData.uid[i]))
            lcmaps_log(LOG_INFO, "LCMAPS: Warning: output truncated for uid. %d\n", bufsize);
        /* lcmaps_log(debug_level, "uid                   : %d  [%d/%d]\n", credData.uid[i], i+1, credData.cntUid); */
    }
    for (i = 0; i < credData.cntPriGid; i++)
    {
        if ((bufsize - strlen(buffer)) <= snprintf (&buffer[strlen(buffer)], bufsize - strlen(buffer), ",pgid:\'%d\'", 
                                                                                                       credData.priGid[i]))
            lcmaps_log(LOG_INFO, "LCMAPS: Warning: output truncated for pgid.\n");
        /* lcmaps_log(debug_level, "pgid                  : %d  [%d/%d]\n", credData.priGid[i], i+1, credData.cntPriGid); */
    }
    for (i = 0; i < credData.cntSecGid; i++)
    {
        if ((bufsize - strlen(buffer)) <= snprintf (&buffer[strlen(buffer)], bufsize - strlen(buffer), ",sgid:\'%d\'", 
                                                                                                       credData.secGid[i]))
            lcmaps_log(LOG_INFO, "LCMAPS: Warning: output truncated for sgid.\n");
        /* lcmaps_log(debug_level, "sgid                  : %d  [%d/%d]\n", credData.secGid[i], i+1, credData.cntSecGid); */
    }

    /* Write generic DN to UID, GID and Secondary GID mappings */
    if (strlen(buffer) > 0)
        lcmaps_log (LOG_NOTICE, "LCMAPS CRED FINAL: %s\n", buffer);

    /* Memory liberation */
    free(buffer);
    buffer = NULL;

    /* Trying to one line VOMS info */
    for (i = 0; i < credData.cntVoCred; i++)
    {
        lcmaps_log_debug(debug_level, "LCMAPS CRED FINAL: VO credential         :     [%d/%d]\n", i+1, credData.cntVoCred);
        lcmaps_printVoData(debug_level, &(credData.VoCred[i]));
    }
    for (i = 0; i < credData.cntVoCredString; i++)
        lcmaps_log(LOG_INFO, "LCMAPS CRED FINAL: VO credential string  : %s  [%d/%d]\n", credData.VoCredString[i], i+1, credData.cntVoCredString);

    /* Typically used */
    for (i = 0; i < credData.cntVoCredMapping; i++)
    {
        lcmaps_log_debug(debug_level, "LCMAPS CRED FINAL: VO credential mapping : [%d/%d]\n", i+1, credData.cntVoCredMapping);
        /* lcmaps_printVoMapping(debug_level, &(credData.VoCredMapping[i])); */

        if (credData.VoCredMapping[i].groupname)
            lcmaps_log(LOG_NOTICE,"LCMAPS CRED FINAL: FQAN:\"%s\"->mapped group:%d(%s)\n", credData.VoCredMapping[i].vostring, (int)credData.VoCredMapping[i].gid, credData.VoCredMapping[i].groupname);
        else
            lcmaps_log(LOG_NOTICE,"LCMAPS CRED FINAL: FQAN:\"%s\"->mapped group:%d\n", credData.VoCredMapping[i].vostring, (int)credData.VoCredMapping[i].gid);
    }

    /* Bonus */
    if (credData.pool_index != NULL)
    {
        lcmaps_log(LOG_DEBUG,"LCMAPS CRED FINAL: POOL_INDEX:\"%s\"\n", credData.pool_index);
    }
}



static int lcmaps_compare_gids(const void * pgid1, const void * pgid2)
{
    gid_t gid1, gid2;

    if (pgid1 == NULL) return 0;
    if (pgid2 == NULL) return 0;

    gid1 = *(const gid_t *) pgid1;
    gid2 = *(const gid_t *) pgid2;
    if (gid1 < gid2)
        return -1;
    else if (gid1 > gid2)
        return 1;
    return 0;
}


/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/pluginmanager/lcmaps_cred_data.c,v $
    $Date: 2012-03-16 16:16:23 +0100 (Fri, 16 Mar 2012) $
    $Revision: 16177 $
    $Author: msalle $
******************************************************************************/
