/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     error.c                                                        */
/*                                                                          */
/* description:  routines for computing different types of errors           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#include "alberta.h"

static const QUAD_FAST  *quad_fast = nil;
static int              relative;

/*--------------------------------------------------------------------------*/
/*  discrete solution                                                       */
/*--------------------------------------------------------------------------*/
static const DOF_REAL_VEC    *err_uh;
static const DOF_REAL_D_VEC  *err_uh_d;
static const BAS_FCTS  *bas_fcts;

/*--------------------------------------------------------------------------*/
/*  data on a single element                                                */
/*--------------------------------------------------------------------------*/
static const EL_INFO *elinfo;
static const REAL    *(*get_real_vec_el)(const EL *, const DOF_REAL_VEC *,
					   REAL *);
static const REAL_D  *(*get_real_d_vec_el)(const EL *, const DOF_REAL_D_VEC *,
					   REAL_D *);
static REAL          *(*rw_error)(EL *el);

static PARAMETRIC    *el_parametric = nil;

/*--------------------------------------------------------------------------*/
/*  functions handed over to the error routines evaluated in local          */
/*  coordinates                                                             */
/*--------------------------------------------------------------------------*/

static REAL   (*p_u)(const REAL_D);

static REAL err_u(const REAL lambda[N_LAMBDA])
{
  REAL_D  x[1];

  if (el_parametric) {
    el_parametric->coord_to_world(elinfo, nil, 1,
				  (const REAL (*)[N_LAMBDA])lambda, x);
  }
  else {
    coord_to_world(elinfo, lambda, *x);
  }
  return((*p_u)(x[0]));
}

static const REAL   *(*p_u_d)(const REAL_D, REAL_D);

static const REAL *err_u_d(const REAL lambda[N_LAMBDA])
{
  REAL_D  x[1];

  if (el_parametric) {
    el_parametric->coord_to_world(elinfo, nil, 1,
				  (const REAL (*)[N_LAMBDA])lambda, x);
  }
  else {
    coord_to_world(elinfo, lambda, x[0]);
  }
  return((*p_u_d)(x[0], nil));
}

static const REAL  *(*p_grd_u)(const REAL_D, REAL_D);

static const REAL *err_grd_u(const REAL lambda[N_LAMBDA])
{
  REAL_D  x[1];

  if (el_parametric) {
    el_parametric->coord_to_world(elinfo, nil, 1,
				  (const REAL (*)[N_LAMBDA])lambda, x);
  }
  else {
    coord_to_world(elinfo, lambda, *x);
  }
  
  return((*p_grd_u)(x[0], nil));
}

static const REAL_D  *(*p_grd_u_d)(const REAL_D, REAL_DD);

static const REAL_D *err_grd_u_d(const REAL lambda[N_LAMBDA])
{
  REAL_D  x[1];

  if (el_parametric) {
    el_parametric->coord_to_world(elinfo, nil, 1,
				  (const REAL (*)[N_LAMBDA])lambda, x);
  }
  else {
    coord_to_world(elinfo, lambda, *x);
  }
  
  return((*p_grd_u_d)(x[0], nil));
}

/*--------------------------------------------------------------------------*/
/*  and now, the error functions :-)))                                      */
/*--------------------------------------------------------------------------*/

typedef struct err_traverse_data {

  REAL max_err;

  REAL  rel_norm2;

  REAL l2_err_2, l2_norm2;

  REAL h1_err_2, h1_norm2;

} ERR_TRAVERSE_DATA;

/*--------------------------------------------------------------------------*/
/*  max error at the quadrature points :-)))                                */
/*--------------------------------------------------------------------------*/

static void max_err_at_qp_fct(const EL_INFO *el_info, void *data)
{
  int         i;
  REAL        err;
  const REAL  *u_vec, *uh_vec, *uh_el;
  PARAMETRIC  *parametric = el_info->mesh->parametric;

  elinfo = el_info;
  if (parametric) {
    parametric->init_element(el_info, parametric);
    el_parametric = parametric;
  }
  else {
    el_parametric = nil;
  }

  u_vec = f_at_qp(quad_fast->quad, err_u, nil);

  uh_el = (*get_real_vec_el)(el_info->el, err_uh, nil);
  uh_vec = uh_at_qp(quad_fast, uh_el, nil);

  for (i = 0; i < quad_fast->n_points; i++)
  {
    err = u_vec[i] > uh_vec[i] ? u_vec[i] - uh_vec[i] : uh_vec[i] - u_vec[i];
    ((ERR_TRAVERSE_DATA *)data)->max_err = 
      MAX(((ERR_TRAVERSE_DATA *)data)->max_err, err);
  }

  return;
}

/*--------------------------------------------------------------------------*/
/*  max error at the quadrutare points of quadrature formula of degree      */
/*  of the continous function u and discrete function uh                    */
/*--------------------------------------------------------------------------*/

REAL max_err_at_qp(REAL (*u)(const REAL_D), const DOF_REAL_VEC *uh,
		   const QUAD *quad)
{
  FUNCNAME("max_err_at_qp");
  ERR_TRAVERSE_DATA td[1] = {{0}};
  const FE_SPACE *fe_space;

  if (!(p_u = u))
  {
    ERROR("no function u specified; doing nothing\n");
    return(-1.0);
  }
  if (!(err_uh = uh)  ||  !(fe_space = uh->fe_space))
  {
    ERROR("no discrete function or no fe_space for it; doing nothing\n");
    return(-1.0);
  }
  if (!uh->vec)
  {
    ERROR("no coefficient vector at discrete solution ; doing nothing\n");
    return(-1.0);
  }
  if (!(bas_fcts = fe_space->bas_fcts))
  {
    ERROR("no basis functions at discrete solution ; doing nothing\n");
    return(-1.0);
  }
  if (!quad)
    quad = get_quadrature(bas_fcts->dim,
			  2*bas_fcts->degree-2);

  quad_fast = get_quad_fast(bas_fcts, quad, INIT_PHI);

  get_real_vec_el = fe_space->bas_fcts->get_real_vec;

  td->max_err = 0.0;
  mesh_traverse(fe_space->mesh,-1,FILL_COORDS|CALL_LEAF_EL, 
		max_err_at_qp_fct, td);

  return(td->max_err);
}

/*--------------------------------------------------------------------------*/
/*  max error at the vertices of the grid :-)))                             */
/*--------------------------------------------------------------------------*/

static REAL vl[N_LAMBDA][N_LAMBDA] = {{1,0,0,0},{0,1,0,0},{0,0,1,0},{0,0,0,1}};

static void max_err_at_v_fct(const EL_INFO *el_info, void *data)
{
  int         i;
  int         dim = el_info->mesh->dim;
  REAL        err;
  const REAL  *uh_el;
  PARAMETRIC  *parametric = el_info->mesh->parametric;

  elinfo = el_info;
  if (parametric) {
    parametric->init_element(el_info, parametric);
    el_parametric = parametric;
  }
  else {
    el_parametric = nil;
  }

  uh_el = get_real_vec_el(el_info->el, err_uh, nil);
  for (i = 0; i < N_VERTICES(dim); i++)
  {
    err = err_u(vl[i]) - eval_uh(vl[i], uh_el, bas_fcts);
    err = ABS(err);
    ((ERR_TRAVERSE_DATA *)data)->max_err = 
      MAX(((ERR_TRAVERSE_DATA *)data)->max_err, err);
  }

  return;
}

/*--------------------------------------------------------------------------*/
/*  max error at the quadrutare points of quadrature formula of degree      */
/*  of the continous function u and discrete function uh                    */
/*--------------------------------------------------------------------------*/

REAL max_err_at_vert(REAL (*u)(const REAL_D), const DOF_REAL_VEC *uh)
{
  FUNCNAME("max_err_at_vert");
  ERR_TRAVERSE_DATA td[1] = {{0}};
  const FE_SPACE *fe_space;

  if (!(p_u = u))
  {
    ERROR("no function u specified; doing nothing\n");
    return(-1.0);
  }
  if (!(err_uh = uh)  ||  !(fe_space = uh->fe_space))
  {
    ERROR("no discrete function or no fe_space for it; doing nothing\n");
    return(-1.0);
  }
  if (!uh->vec)
  {
    ERROR("no coefficient vector at discrete solution ; doing nothing\n");
    return(-1.0);
  }
  if (!(bas_fcts = fe_space->bas_fcts))
  {
    ERROR("no basis functions at discrete solution ; doing nothing\n");
    return(-1.0);
  }

  get_real_vec_el = fe_space->bas_fcts->get_real_vec;

  td->max_err = 0.0;
  mesh_traverse(fe_space->mesh,-1,FILL_COORDS|CALL_LEAF_EL, 
		max_err_at_v_fct, td);

  return(td->max_err);
}

/*--------------------------------------------------------------------------*/
/*  function for setting relative error on the elements: used by both       */
/*  h1_err and l2_err                                                       */
/*--------------------------------------------------------------------------*/
static void rel_fct(const EL_INFO *el_info, void *data)
{
  REAL *exact = rw_error(el_info->el);
  *exact /= ((ERR_TRAVERSE_DATA *)data)->rel_norm2;
  return;
}

/*--------------------------------------------------------------------------*/
/*  L2 error on the mesh                                                    */
/*--------------------------------------------------------------------------*/
static void l2_err_fct(const EL_INFO *el_info, void *data)
{
  int          i, dim = el_info->mesh->dim;
  REAL         err, l2_err_el, norm_el, exact;
  REAL         det[MAX_N_QUAD_POINTS];
  const REAL   *uh_el, *u_vec, *uh_vec;
  PARAMETRIC   *parametric = el_info->mesh->parametric;

  elinfo = el_info;
  if (parametric) {
    parametric->init_element(el_info, parametric);
    el_parametric = parametric;
  }
  else {
    el_parametric = nil;
  }

  u_vec = f_at_qp(quad_fast->quad, err_u, nil);

  uh_el = (*get_real_vec_el)(el_info->el, err_uh, nil);
  uh_vec = uh_at_qp(quad_fast, uh_el, nil);

  if (el_parametric) {
    el_parametric->det(el_info, quad_fast->quad, 0, nil, det);

    for (l2_err_el = i = 0; i < quad_fast->n_points; i++) {
      err = u_vec[i] - uh_vec[i];
      l2_err_el += det[i] * quad_fast->w[i] * SQR(err);
    }

    exact = l2_err_el;

    if (relative)
      for (i = 0; i < quad_fast->n_points; i++)
	((ERR_TRAVERSE_DATA *)data)->l2_norm2 += 
	  det[i] * quad_fast->w[i] * SQR(u_vec[i]);
  }
  else {
    switch(dim) {
    case 1:
      det[0] = el_det_1d(el_info);
      break;
#if DIM_OF_WORLD > 1
    case 2:
      det[0] = el_det_2d(el_info);
      break;
#if DIM_OF_WORLD > 2
    case 3:
      det[0] = el_det_3d(el_info);
#endif
#endif
    }

    for (l2_err_el = i = 0; i < quad_fast->n_points; i++) {
      err = u_vec[i] - uh_vec[i];
      l2_err_el += quad_fast->w[i]*SQR(err);
    }

    exact = det[0]*l2_err_el;

    if (relative) {
      for (norm_el = i = 0; i < quad_fast->n_points; i++)
	norm_el += quad_fast->w[i]*SQR(u_vec[i]);
      ((ERR_TRAVERSE_DATA *)data)->l2_norm2 += det[0]*norm_el;
    }
  }

  ((ERR_TRAVERSE_DATA *)data)->l2_err_2 += exact;
  ((ERR_TRAVERSE_DATA *)data)->max_err = 
    MAX(((ERR_TRAVERSE_DATA *)data)->max_err, exact);

  if (rw_error) *(*rw_error)(el_info->el) = exact;

  return;
}

REAL L2_err(REAL (*u)(const REAL_D), const DOF_REAL_VEC *uh, const QUAD *quad,
	    int rel_err, REAL *(*rw_err_el)(EL *), REAL *max_l2_err2)
{
  FUNCNAME("L2_err");
  ERR_TRAVERSE_DATA td[1] ={{0}};
  const FE_SPACE *fe_space;
  
  if (!(p_u = u))
  {
    ERROR("no function u specified; doing nothing\n");
    return(0.0);
  }
  if (!(err_uh = uh)  ||  !(fe_space = uh->fe_space))
  {
    ERROR("no discrete function or no fe_space for it; doing nothing\n");
    return(0.0);
  }
  if (!uh->vec)
  {
    ERROR("no coefficient vector at discrete solution ; doing nothing\n");
    return(0.0);
  }
  if (!(bas_fcts = fe_space->bas_fcts))
  {
    ERROR("no basis functions at discrete solution ; doing nothing\n");
    return(0.0);
  }
  if (!quad)
    quad = get_quadrature(fe_space->mesh->dim,
			  2*fe_space->bas_fcts->degree -2);
  quad_fast = get_quad_fast(bas_fcts, quad, INIT_PHI);

  get_real_vec_el = fe_space->bas_fcts->get_real_vec;

  relative = rel_err;

  rw_error = rw_err_el;

  td->max_err = td->l2_err_2 = td->l2_norm2 = 0.0;
  mesh_traverse(fe_space->mesh, -1, FILL_COORDS|CALL_LEAF_EL, l2_err_fct, td);

  if (relative)
  {
    td->rel_norm2 = td->l2_norm2+1.e-15;
    if (rw_error)  mesh_traverse(fe_space->mesh, -1, CALL_LEAF_EL, rel_fct, td);
    td->l2_err_2 /= td->rel_norm2;
  }

  if (max_l2_err2)  *max_l2_err2 = td->max_err;

  return(sqrt(td->l2_err_2));
}

/*--------------------------------------------------------------------------*/
/*  H1 error on the mesh                                                    */
/*--------------------------------------------------------------------------*/

static void h1_err_fct(const EL_INFO *el_info, void *data)
{
  int           i, j;
  int           dim = el_info->mesh->dim;
  REAL          err, err_2, h1_err_el, norm_el, norm2, exact;
  REAL          det[MAX_N_QUAD_POINTS];
  REAL_D        Lambda[MAX_N_QUAD_POINTS][N_LAMBDA];
  const REAL   *uh_el;
  const REAL_D *grdu_vec, *grduh_vec;
  PARAMETRIC   *parametric = el_info->mesh->parametric;

  elinfo = el_info;
  if (parametric) {
    parametric->init_element(el_info, parametric);
    el_parametric = parametric;
  }
  else {
    el_parametric = nil;
  }

  grdu_vec = grd_f_at_qp(quad_fast->quad, err_grd_u, nil);

  uh_el = (*get_real_vec_el)(el_info->el, err_uh, nil);

  if (el_parametric) {
    el_parametric->grd_lambda(el_info, quad_fast->quad, 0, nil, Lambda, det);

    grduh_vec = param_grd_uh_at_qp(quad_fast, Lambda, uh_el, nil);

    for (h1_err_el = i = 0; i < quad_fast->n_points; i++) {
      for (err_2 = j = 0; j < DIM_OF_WORLD; j++) {
	err = grdu_vec[i][j] - grduh_vec[i][j];
	err_2 += SQR(err);
      }
      h1_err_el += det[i] * quad_fast->w[i]*err_2;
    }

    exact = h1_err_el;

    if (relative) {
      for (i = 0; i < quad_fast->n_points; i++) {
	for (norm2 = j = 0; j < DIM_OF_WORLD; j++) 
	  norm2 += SQR(grdu_vec[i][j]);
	((ERR_TRAVERSE_DATA *)data)->h1_norm2 += det[i] * quad_fast->w[i]*norm2;
      }
    }
  }
  else {
    switch(dim) {
    case 1:
      det[0] = el_grd_lambda_1d(el_info, Lambda[0]);
      break;
#if DIM_OF_WORLD > 1
    case 2:
      det[0] = el_grd_lambda_2d(el_info, Lambda[0]);
      break;
#if DIM_OF_WORLD > 2
    case 3:
      det[0] = el_grd_lambda_3d(el_info, Lambda[0]);
#endif
#endif
    }

    grduh_vec = grd_uh_at_qp(quad_fast, (const REAL_D *)Lambda[0], uh_el, nil);

    for (h1_err_el = i = 0; i < quad_fast->n_points; i++) {
      for (err_2 = j = 0; j < DIM_OF_WORLD; j++) {
	err = grdu_vec[i][j] - grduh_vec[i][j];
	err_2 += SQR(err);
      }
      h1_err_el += quad_fast->w[i]*err_2;
    }

    exact = det[0]*h1_err_el;

    if (relative) {
      for (norm_el = i = 0; i < quad_fast->n_points; i++) {
	for (norm2 = j = 0; j < DIM_OF_WORLD; j++) 
	  norm2 += SQR(grdu_vec[i][j]);
	norm_el += quad_fast->w[i]*norm2;
      }
      ((ERR_TRAVERSE_DATA *)data)->h1_norm2 += det[0]*norm_el;
    }
  }

  ((ERR_TRAVERSE_DATA *)data)->h1_err_2 += exact;
  ((ERR_TRAVERSE_DATA *)data)->max_err = 
    MAX(((ERR_TRAVERSE_DATA *)data)->max_err, exact);

  if (rw_error) *(*rw_error)(el_info->el) = exact;

  return;
}

REAL H1_err(const REAL *(*grd_u)(const REAL_D, REAL_D),
	    const DOF_REAL_VEC *uh, 
	    const QUAD *quad, int rel_err, REAL *(*rw_err_el)(EL *),
	    REAL *max_h1_err2)
{
  FUNCNAME("H1_err");
  ERR_TRAVERSE_DATA td[1] = {{0}};
  const FE_SPACE *fe_space;
  
  if (!(p_grd_u = grd_u))
  {
    ERROR("no gradient function grd_u specified; doing nothing\n");
    return(0.0);
  }
  if (!(err_uh = uh)  ||  !(fe_space = uh->fe_space))
  {
    ERROR("no discrete function or no fe_space for it; doing nothing\n");
    return(0.0);
  }
  if (!uh->vec)
  {
    ERROR("no coefficient vector at discrete solution ; doing nothing\n");
    return(0.0);
  }
  if (!(bas_fcts = fe_space->bas_fcts))
  {
    ERROR("no basis functions at discrete solution ; doing nothing\n");
    return(0.0);
  }
  if (!quad)
    quad = get_quadrature(fe_space->mesh->dim, 2*fe_space->bas_fcts->degree-2);
  quad_fast = get_quad_fast(bas_fcts, quad, INIT_GRD_PHI);

  get_real_vec_el = fe_space->bas_fcts->get_real_vec;

  relative = rel_err;

  rw_error = rw_err_el;

/*
  if (!rw_error) 
    MSG("rw_err_el nil pointer; can not write errors to elements\n");
*/

  td->max_err = td->h1_err_2 = td->h1_norm2 = 0.0;
  mesh_traverse(fe_space->mesh, -1, FILL_COORDS|CALL_LEAF_EL, h1_err_fct, td);

  if (relative)
  {
    td->rel_norm2 = td->h1_norm2+1.e-15;
    if (rw_error)  mesh_traverse(fe_space->mesh, -1, CALL_LEAF_EL, rel_fct, td);
    td->h1_err_2 /= td->rel_norm2;
    td->max_err /= td->rel_norm2;
  }

  if (max_h1_err2)  *max_h1_err2 = td->max_err;

  return(sqrt(td->h1_err_2));
}

/*--------------------------------------------------------------------------*/
/*  and now, the error functions for _d :-)))                               */
/*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*/
/*  max error at the quadrature points :-)))                                */
/*--------------------------------------------------------------------------*/

static void max_err_d_at_qp_fct(const EL_INFO *el_info, void *data)
{
  int           i, k;
  REAL          err;
  const REAL_D  *u_vec, *uh_vec, *uh_el;
  PARAMETRIC  *parametric = el_info->mesh->parametric;

  elinfo = el_info;
  if (parametric) {
    parametric->init_element(el_info, parametric);
    el_parametric = parametric;
  }
  else {
    el_parametric = nil;
  }

  u_vec = f_d_at_qp(quad_fast->quad, err_u_d, nil);

  uh_el = (*get_real_d_vec_el)(el_info->el, err_uh_d, nil);
  uh_vec = uh_d_at_qp(quad_fast, uh_el, nil);

  for (i = 0; i < quad_fast->n_points; i++)
  {
    for (err = k = 0; k < DIM_OF_WORLD; k++)
      err += SQR(u_vec[i][k] - uh_vec[i][k]);

    ((ERR_TRAVERSE_DATA *)data)->max_err = 
      MAX(((ERR_TRAVERSE_DATA *)data)->max_err, err);
  }

  return;
}

/*--------------------------------------------------------------------------*/
/*  max error at the quadrature points of quadrature formula of degree      */
/*  of the continous function u and discrete function uh                    */
/*--------------------------------------------------------------------------*/

REAL max_err_d_at_qp(const REAL *(*u)(const REAL_D, REAL_D),
		     const DOF_REAL_D_VEC *uh, const QUAD *quad)
{
  FUNCNAME("max_err_d_at_qp");
  ERR_TRAVERSE_DATA td[1] = {{0}};
  const FE_SPACE *fe;
  
  if (!(p_u_d = u))
  {
    ERROR("no function u specified; doing nothing\n");
    return(0.0);
  }
  if (!(err_uh_d = uh)  ||  !(fe = uh->fe_space))
  {
    ERROR("no discrete function or no fe_space for it; doing nothing\n");
    return(0.0);
  }
  if (!uh->vec)
  {
    ERROR("no coefficient vector at discrete solution ; doing nothing\n");
    return(0.0);
  }
  if (!(bas_fcts = fe->bas_fcts))
  {
    ERROR("no basis functions at discrete solution ; doing nothing\n");
    return(0.0);
  }
  if (!quad)
    quad = get_quadrature(fe->mesh->dim, 2*bas_fcts->degree-2);
  quad_fast = get_quad_fast(bas_fcts, quad, INIT_PHI);

  get_real_d_vec_el = bas_fcts->get_real_d_vec;

  td->max_err = 0.0;
  mesh_traverse(fe->mesh, -1, FILL_COORDS|CALL_LEAF_EL,
		max_err_d_at_qp_fct, td);

  return sqrt(td->max_err);
}

/*--------------------------------------------------------------------------*/
/*  L2_d error on the mesh                                                  */
/*--------------------------------------------------------------------------*/

static void l2_err_fct_d(const EL_INFO *el_info, void *data)
{
  int            i, k, dim = el_info->mesh->dim;
  REAL           err = 0, l2_err_el, norm_el, exact;
  REAL           det[MAX_N_QUAD_POINTS];
  const REAL_D   *uh_el, *u_vec, *uh_vec;
  PARAMETRIC     *parametric = el_info->mesh->parametric;

  elinfo = el_info;
  if (parametric) {
    parametric->init_element(el_info, parametric);
    el_parametric = parametric;
  }
  else {
    el_parametric = nil;
  }

  u_vec = f_d_at_qp(quad_fast->quad, err_u_d, nil);

  uh_el = (*get_real_d_vec_el)(el_info->el, err_uh_d, nil);
  uh_vec = uh_d_at_qp(quad_fast, uh_el, nil);

  if (el_parametric) {
    el_parametric->det(el_info, quad_fast->quad, 0, nil, det);

    for (l2_err_el = i = 0; i < quad_fast->n_points; i++) {
      for (err = k = 0; k < DIM_OF_WORLD; k++)
	err += SQR(u_vec[i][k] - uh_vec[i][k]);
      
      l2_err_el += det[i] * quad_fast->w[i]*err;
    }

    exact = l2_err_el;

    if (relative) {
      for (i = 0; i < quad_fast->n_points; i++) {
	for (err = k = 0; k < DIM_OF_WORLD; k++)
	  err += SQR(u_vec[i][k]);
	((ERR_TRAVERSE_DATA *)data)->l2_norm2 += det[i] * quad_fast->w[i] * err;
      }
    }
  }
  else {
    switch(dim) {
    case 1:
      det[0] = el_det_1d(el_info);
      break;
#if DIM_OF_WORLD > 1
    case 2:
      det[0] = el_det_2d(el_info);
      break;
#if DIM_OF_WORLD > 2
    case 3:
      det[0] = el_det_3d(el_info);
#endif
#endif
    }

    for (l2_err_el = i = 0; i < quad_fast->n_points; i++) {
      for (err = k = 0; k < DIM_OF_WORLD; k++)
	err += SQR(u_vec[i][k] - uh_vec[i][k]);
      
      l2_err_el += quad_fast->w[i]*err;
    }

    exact = det[0]*l2_err_el;

    if (relative) {
      for (norm_el = i = 0; i < quad_fast->n_points; i++)
	for (err = k = 0; k < DIM_OF_WORLD; k++)
	  err += SQR(u_vec[i][k]);
      norm_el += quad_fast->w[i]*err;
      ((ERR_TRAVERSE_DATA *)data)->l2_norm2 += det[0]*norm_el;
    }
  }

  ((ERR_TRAVERSE_DATA *)data)->l2_err_2 += exact;
  ((ERR_TRAVERSE_DATA *)data)->max_err = 
    MAX(((ERR_TRAVERSE_DATA *)data)->max_err, exact);
  
  if (rw_error) *(*rw_error)(el_info->el) = exact;

  return;
}

REAL L2_err_d(const REAL *(*u)(const REAL_D, REAL_D), 
	       const DOF_REAL_D_VEC *uh,
	       const QUAD *quad, int rel_err, REAL *(*rw_err_el)(EL *),
	       REAL *max_l2_err2)
{
  FUNCNAME("L2_err_d");
  ERR_TRAVERSE_DATA td[1] = {{0}};
  const FE_SPACE *fe;

  if (!(p_u_d = u))
  {
    ERROR("no function u specified; doing nothing\n");
    return(0.0);
  }
  if (!(err_uh_d = uh)  ||  !(fe = uh->fe_space))
  {
    ERROR("no discrete function or no fe_space for it; doing nothing\n");
    return(0.0);
  }
  if (!uh->vec)
  {
    ERROR("no coefficient vector at discrete solution ; doing nothing\n");
    return(0.0);
  }
  if (!(bas_fcts = fe->bas_fcts))
  {
    ERROR("no basis functions at discrete solution ; doing nothing\n");
    return(0.0);
  }
  if (!quad)
    quad = get_quadrature(fe->mesh->dim, 2*bas_fcts->degree -2);
  quad_fast = get_quad_fast(bas_fcts, quad, INIT_PHI);

  get_real_d_vec_el = bas_fcts->get_real_d_vec;

  relative = rel_err;

  rw_error = rw_err_el;

  td->max_err = td->l2_err_2 = td->l2_norm2 = 0.0;
  mesh_traverse(fe->mesh, -1, FILL_COORDS|CALL_LEAF_EL, l2_err_fct_d, td);

  if (relative)
  {
    td->rel_norm2 = td->l2_norm2+1.e-15;
    if (rw_error)  mesh_traverse(fe->mesh, -1, CALL_LEAF_EL, rel_fct, td);
    td->l2_err_2 /= td->rel_norm2;
  }

  if (max_l2_err2) *max_l2_err2 = td->max_err;

  return(sqrt(td->l2_err_2));
}

/*--------------------------------------------------------------------------*/
/*  H1_d error on the mesh                                                  */
/*--------------------------------------------------------------------------*/

static void h1_err_fct_d(const EL_INFO *el_info, void *data)
{
  int            i, j, k, dim = el_info->mesh->dim;
  REAL           err, err_2, h1_err_el, norm_el, norm2, exact;
  REAL           det[MAX_N_QUAD_POINTS];
  REAL_D         Lambda[MAX_N_QUAD_POINTS][N_LAMBDA];
  const REAL_D   *uh_el;
  const REAL_DD  *grdu_vec, *grduh_vec;
  PARAMETRIC     *parametric = el_info->mesh->parametric;

  elinfo = el_info;
  if (parametric) {
    parametric->init_element(el_info, parametric);
    el_parametric = parametric;
  }
  else {
    el_parametric = nil;
  }

  grdu_vec = grd_f_d_at_qp(quad_fast->quad, err_grd_u_d, nil);

  uh_el = (*get_real_d_vec_el)(el_info->el, err_uh_d, nil);

  if (el_parametric) {
    el_parametric->grd_lambda(el_info, quad_fast->quad, 0, nil, Lambda, det);

    grduh_vec = param_grd_uh_d_at_qp(quad_fast, Lambda, uh_el, nil);
    
    for (h1_err_el = i = 0; i < quad_fast->n_points; i++) {
      for (err_2 = k = 0; k < DIM_OF_WORLD; k++)
	for (j = 0; j < DIM_OF_WORLD; j++)  {
	  err = grdu_vec[i][k][j] - grduh_vec[i][k][j];
	  err_2 += SQR(err);
	}
      
      h1_err_el += det[i] * quad_fast->w[i]*err_2;
    }

    exact = h1_err_el;

    if (relative) {
      for (i = 0; i < quad_fast->n_points; i++) {
	for (norm2 = j = 0; j < DIM_OF_WORLD; j++) 
	  for (k = 0; k < DIM_OF_WORLD; k++)
	    norm2 += SQR(grdu_vec[i][k][j]);

	((ERR_TRAVERSE_DATA *)data)->h1_norm2 += det[i] * quad_fast->w[i]*norm2;
      }
    }
  }
  else {
    switch(dim) {
    case 1:
      det[0] = el_grd_lambda_1d(el_info, Lambda[0]);
      break;
#if DIM_OF_WORLD > 1
    case 2:
      det[0] = el_grd_lambda_2d(el_info, Lambda[0]);
      break;
#if DIM_OF_WORLD > 2
    case 3:
      det[0] = el_grd_lambda_3d(el_info, Lambda[0]);
#endif
#endif
    }

    grduh_vec = grd_uh_d_at_qp(quad_fast, (const REAL_D *)Lambda[0],
			       uh_el, nil);
    
    for (h1_err_el = i = 0; i < quad_fast->n_points; i++) {
      for (err_2 = k = 0; k < DIM_OF_WORLD; k++)
	for (j = 0; j < DIM_OF_WORLD; j++)  {
	  err = grdu_vec[i][k][j] - grduh_vec[i][k][j];
	  err_2 += SQR(err);
	}
      
      h1_err_el += quad_fast->w[i]*err_2;
    }

    exact = det[0]*h1_err_el;

    if (relative) {
      for (norm_el = i = 0; i < quad_fast->n_points; i++) {
	for (norm2 = j = 0; j < DIM_OF_WORLD; j++) 
	  for (k = 0; k < DIM_OF_WORLD; k++)
	    norm2 += SQR(grdu_vec[i][k][j]);

	norm_el += quad_fast->w[i]*norm2;
      }
      ((ERR_TRAVERSE_DATA *)data)->h1_norm2 += det[0] * norm_el;
    }
  }

  ((ERR_TRAVERSE_DATA *)data)->h1_err_2 += exact;
  ((ERR_TRAVERSE_DATA *)data)->max_err = 
    MAX(((ERR_TRAVERSE_DATA *)data)->max_err, exact);

  if (rw_error) *(*rw_error)(el_info->el) = exact;

  return;
}

REAL H1_err_d(const REAL_D *(*grd_u)(const REAL_D, REAL_DD),
	       const DOF_REAL_D_VEC *uh, const QUAD *quad, int rel_err, 
	       REAL *(*rw_err_el)(EL *), REAL *max_h1_err2)
{
  FUNCNAME("H1_err_d");
  ERR_TRAVERSE_DATA td[1] = {{0}};
  const FE_SPACE *fe;
  
  if (!(p_grd_u_d = grd_u))
  {
    ERROR("no gradient function grd_u specified; doing nothing\n");
    return(0.0);
  }
  if (!(err_uh_d = uh)  ||  !(fe = uh->fe_space))
  {
    ERROR("no discrete function or no admin for it; doing nothing\n");
    return(0.0);
  }
  if (!uh->vec)
  {
    ERROR("no coefficient vector at discrete solution ; doing nothing\n");
    return(0.0);
  }
  if (!(bas_fcts = fe->bas_fcts))
  {
    ERROR("no basis functions at discrete solution ; doing nothing\n");
    return(0.0);
  }
  if (!quad)
    quad = get_quadrature(fe->mesh->dim, 2*bas_fcts->degree-2);
  quad_fast = get_quad_fast(bas_fcts, quad, INIT_GRD_PHI);

  get_real_d_vec_el = bas_fcts->get_real_d_vec;

  relative = rel_err;
  rw_error = rw_err_el;
  
  td->max_err = td->h1_err_2 = td->h1_norm2 = 0.0;
  mesh_traverse(fe->mesh, -1, FILL_COORDS|CALL_LEAF_EL, h1_err_fct_d, td);

  if (relative)
  {
    td->rel_norm2 = td->h1_norm2+1.e-15;
    if (rw_error)  mesh_traverse(fe->mesh, -1, CALL_LEAF_EL, rel_fct, td);
    td->h1_err_2 /= td->rel_norm2;
    td->max_err /= td->rel_norm2;
  }

  if (max_h1_err2) *max_h1_err2 = td->max_err;

  return(sqrt(td->h1_err_2));
}
