/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "Font.h"

#include <epoxy/gl.h>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>

#include <cstdio>

using namespace std;

namespace libgltf
{

Font::Font()
    : mLoadedPixelSize(0)
    , mNewLine(0)
    , bLoaded(false)
    , mVAO(0)
    , mDataBuffer()
    , mBuffer(0)
    , mShaderId(0)
{
    for (int i = 0; i < 256; i++)
    {
        mAdvX[i] = 0;
        mAdvY[i] = 0;
        mBearingX[i] = 0;
    }
}

inline int next_p2(int n)
{
    int res = 1;
    while (res < n)
    {
        res <<= 1;
    } return res;
}

void Font::createChar(int iIndex, unsigned char* pBitmap, int iW, int iH)
{
    int iTW = next_p2(iW), iTH = next_p2(iH);

    GLubyte* bData = new GLubyte[(unsigned int)(iTW * iTH)];
    for (int ch = 0; ch < iTH; ch++)
    {
        for (int cw = 0; cw < iTW; cw++)
        {
            bData[ch * iTW + cw] = (ch >= iH || cw >= iW) ?
                                   (GLubyte)0 :
                                   pBitmap[(iH - ch - 1) * iW + cw];
        }
    }

    mCharTextures[iIndex].createFromData(bData, iTW, iTH, GL_DEPTH_COMPONENT);
    mCharTextures[iIndex].setFiltering(TEXTURE_FILTER_MAG_BILINEAR,
                                       TEXTURE_FILTER_MIN_BILINEAR);
    mCharTextures[iIndex].setTextureParameter(GL_TEXTURE_WRAP_S,
                                              GL_CLAMP_TO_EDGE);
    mCharTextures[iIndex].setTextureParameter(GL_TEXTURE_WRAP_T,
                                              GL_CLAMP_TO_EDGE);

    glm::vec2 vQuad[] =
    {
        glm::vec2(0.0f, float(-mAdvY[iIndex] + iTH)),
        glm::vec2(0.0f, float(-mAdvY[iIndex])),
        glm::vec2(float(iTW), float(-mAdvY[iIndex] + iTH)),
        glm::vec2(float(iTW), float(-mAdvY[iIndex]))
    };

    glm::vec2 vTexQuad[] = {glm::vec2(0.0f, 1.0f), glm::vec2(0.0f, 0.0f),
                            glm::vec2(1.0f, 1.0f), glm::vec2(1.0f, 0.0f)
                           };

    for (int i = 0; i < 4; i++)
    {
        mDataBuffer.insert(mDataBuffer.end(), reinterpret_cast<unsigned char*>(&vQuad[i]),
                           reinterpret_cast<unsigned char*>(&vQuad[i]) + sizeof(glm::vec2));
        mDataBuffer.insert(mDataBuffer.end(), reinterpret_cast<unsigned char*>(&vTexQuad[i]),
                           reinterpret_cast<unsigned char*>(&vTexQuad[i]) + sizeof(glm::vec2));
    }
    delete[] bData;
}
bool Font:: loadFont()
{
    loadTextureFromBitmap();
    mLoadedPixelSize = 32;
    setGLBufferData();
    return true;
}

void Font::setGLBufferData()
{
    glGenVertexArrays(1, &mVAO);
    glBindVertexArray(mVAO);
    glGenBuffers(1, &mBuffer);
    glBindBuffer(GL_ARRAY_BUFFER, mBuffer);
    glBufferData(GL_ARRAY_BUFFER, (GLsizeiptr)mDataBuffer.size(),
                 &mDataBuffer[0], GL_STATIC_DRAW);
    mDataBuffer.clear();
}

void Font::loadTextureFromBitmap()
{
    mDataBuffer.insert(mDataBuffer.end(), 48 * sizeof(glm::vec2) * 2 * 4, '1');
    int tmpAdvX[10] = {20, 20, 20, 21, 20, 20, 20, 21, 20, 20};
    int tmpAdvY[10] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    int tmpBearingX[10] = {2, 3, 2, 2, 1, 2, 2, 3, 2, 2};
    for (int i = 0; i < 10; i++)
    {
        mAdvX[48 + i] = tmpAdvX[i];
        mAdvY[48 + i] = tmpAdvY[i];
        mBearingX[48 + i] = tmpBearingX[i];
    }
    mNewLine = 29;
    createChar(0 + 48, zeroBitmap, bitmapWidths[0], bitmapHeight);
    createChar(1 + 48, oneBitmap, bitmapWidths[1], bitmapHeight);
    createChar(2 + 48, twoBitmap, bitmapWidths[2], bitmapHeight);
    createChar(3 + 48, threeBitmap, bitmapWidths[3], bitmapHeight);
    createChar(4 + 48, fourBitmap, bitmapWidths[4], bitmapHeight);
    createChar(5 + 48, fiveBitmap, bitmapWidths[5], bitmapHeight);
    createChar(6 + 48, sixBitmap, bitmapWidths[6], bitmapHeight);
    createChar(7 + 48, sevenBitmap, bitmapWidths[7], bitmapHeight);
    createChar(8 + 48, eightBitmap, bitmapWidths[8], bitmapHeight);
    createChar(9 + 48, nineBitmap, bitmapWidths[9], bitmapHeight);
    bLoaded = true;
}

void Font::printString(string sText, int x, int y, int iPXSize)
{
    if (!bLoaded)
    {
        return;
    }
    glBindBuffer(GL_ARRAY_BUFFER, mBuffer);

    GLuint positionId;
    GLuint coordId;
    {
        GLint iTmp = glGetAttribLocation(mShaderId, "inPosition");
        if( iTmp == -1 )
            return;
        positionId = static_cast<GLuint>(iTmp);

        iTmp = glGetAttribLocation(mShaderId, "inCoord");
        if( iTmp == -1 )
            return;
        coordId = static_cast<GLuint>(iTmp);
    }

    glEnableVertexAttribArray(positionId);
    glVertexAttribPointer(positionId, 2, GL_FLOAT, GL_FALSE, sizeof(glm::vec2) * 2, 0);

    glEnableVertexAttribArray(coordId);
    glVertexAttribPointer(coordId, 2, GL_FLOAT, GL_FALSE, sizeof(glm::vec2) * 2, reinterpret_cast<void*>(sizeof(glm::vec2)));
    GLint iLoc = glGetUniformLocation(mShaderId, "gSampler");

    glUniform1i(iLoc, 0);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    int iCurX = x, iCurY = y;
    if (iPXSize == -1)
    {
        iPXSize = mLoadedPixelSize;
    }
    float fScale = float(iPXSize) / float(mLoadedPixelSize);
    for (unsigned int i = 0; i < sText.size(); i++)
    {
        if (sText[i] == '\n')
        {
            iCurX = x;
            iCurY -= mNewLine * iPXSize / mLoadedPixelSize;
            continue;
        }
        int iIndex = int(sText[i]);
        iCurX += mBearingX[iIndex] * iPXSize / mLoadedPixelSize;
        if (sText[i] != ' ')
        {
            mCharTextures[iIndex].bindTexture(0);
            glm::mat4 mModelView = glm::translate(glm::mat4(1.0f),
                                                  glm::vec3(float(iCurX),
                                                            float(iCurY),
                                                            0.0f));
            mModelView = glm::scale(mModelView, glm::vec3(fScale));
            GLint iLoc2 = glGetUniformLocation(mShaderId, "modelViewMatrix");

            glUniformMatrix4fv(iLoc2, 1, false, glm::value_ptr(mModelView));
            glDrawArrays(GL_TRIANGLE_STRIP, iIndex * 4, 4);
        }

        iCurX += (mAdvX[iIndex] - mBearingX[iIndex]) *
                 iPXSize / mLoadedPixelSize;
    }
    glDisable(GL_BLEND);
}

void Font::printDecimalInt(int nOutput, int x, int y, int iPXSize)
{
    static char pBuffer[32] = {0};

    int i = 30;

    for(; nOutput && i > 0 ; --i, nOutput /= 10)
    {
        pBuffer[i] = "0123456789"[nOutput % 10];
    }

    printString(&pBuffer[i+1], x, y, iPXSize);
}

void Font::deleteFont()
{
    for (int i = 0; i < 128; i++)
    {
        mCharTextures[i].deleteTexture();
    }
    glDeleteBuffers(1, &mBuffer);
    mDataBuffer.clear();
    glDeleteVertexArrays(1, &mVAO);
}

void Font::setShaderProgram(unsigned int shaderId)
{
    mShaderId = shaderId;
}

} // namespace libgltf

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
