// Author(s): Bas Ploeger, Carst Tankink, Ruud Koolen
// Copyright: see the accompanying file COPYING or copy at
// https://svn.win.tue.nl/trac/MCRL2/browser/trunk/COPYING
//
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

#include "mcrl2/trace/trace.h"
#include "mcrl2/utilities/logger.h"
#include "lts.h"
#include "simulation.h"
#include "transition.h"
#include "state.h"
#include <algorithm>

using namespace std;

Simulation::Simulation(QObject *parent, LTS *lts):
  QObject(parent),
  m_lts(lts),
  m_initialState(0),
  m_currentState(0),
  m_currentTransition(0)
{
}

Simulation::~Simulation()
{
  stop();
}

void Simulation::operator=(const Simulation &other)
{
  if (&other == this)
  {
    return;
  }

  stop();
  m_lts = other.m_lts;
  m_initialState = other.m_initialState;
  m_currentState = other.m_currentState;
  m_currentTransition = other.m_currentTransition;
  m_history = other.m_history;

  if (m_currentState)
  {
    m_currentState->increaseSimulation();
    for (int i = 0; i < m_history.size(); i++)
    {
      m_history[i]->getBeginState()->increaseSimulation();
    }
  }

  emit changed();
}

QList<Transition *> Simulation::availableTransitions() const
{
  if (!m_currentState)
  {
    return QList<Transition *>();
  }

  QList<Transition *> output;
  for (int i = 0; i < m_currentState->getNumOutTransitions(); i++)
  {
    output += m_currentState->getOutTransition(i);
  }
  for (int i = 0; i < m_currentState->getNumLoops(); i++)
  {
    output += m_currentState->getLoop(i);
  }
  return output;
}

void Simulation::start()
{
  if (!m_currentState)
  {
    m_currentState = (m_initialState ? m_initialState : m_lts->getInitialState());
    m_currentState->increaseSimulation();
    m_currentTransition = 0;
    emit started();
    emit changed();
  }
}

void Simulation::stop()
{
  if (m_currentState)
  {
    m_history.clear();
    m_currentTransition = 0;
    m_currentState->decreaseSimulation();
    m_currentState = 0;
    emit stopped();
    emit changed();
  }
}

void Simulation::selectTransition(Transition *transition)
{
  if (m_currentTransition != transition)
  {
    m_currentTransition = transition;
    emit selectionChanged();
  }
}

void Simulation::followTransition(Transition *transition)
{
  m_history += transition;
  m_currentTransition = 0;
  m_currentState = transition->getEndState();
  m_currentState->increaseSimulation();
  emit changed();
}

void Simulation::undo()
{
  if (canUndo())
  {
    m_history.last()->getEndState()->decreaseSimulation();
    m_currentState = m_history.last()->getBeginState();
    m_currentTransition = m_history.takeLast();
    emit changed();
  }
}

void Simulation::traceback()
{
  // TODO: this algorithm only works for iterative state ranking
  if (m_currentState)
  {
    State *initialState = m_lts->getInitialState();
    State *currentState = m_history.isEmpty() ? m_currentState : m_history.first()->getBeginState();

    while (currentState != initialState)
    {
      // Loop through the incoming transitions of currentState.
      // Because of the way the LTS is structured, we know that there is at least
      // one incoming transition that is not a selfloop and not a backpointer.
      // We take the first such transition, because that guarantees that the
      // source state of that transition is in the previous level.
      // Just taking the first incoming transition may lead to infinite loops.
      for (int i = 0; i < currentState->getNumInTransitions(); i++)
      {
        Transition *transition = currentState->getInTransition(i);
        if (!transition->isBackpointer() && !transition->isSelfLoop())
        {
          m_history.prepend(transition);
          currentState = transition->getBeginState();
          currentState->increaseSimulation();
          break;
        }
      }
    }

    emit changed();
  }
}

bool Simulation::loadTrace(QString filename)
{
  mcrl2::trace::Trace trace;
  try
  {
    trace.load(filename.toStdString());
  }
  catch (...)
  {
    mCRL2log(mcrl2::log::error) << "Error loading file: " << filename.toStdString();
    return false;
  }

  if (trace.number_of_states() == 0)
  {
    mCRL2log(mcrl2::log::error) << "The trace in " << filename.toStdString() << " has no state information, which is currently not allowed by ltsview.\nThis typically occurs with traces generated by ltscompare, which can currently not be used in ltsview.";
    return false;
  }

  if (!m_lts->hasStateInfo())
  {
     mCRL2log(mcrl2::log::error) << "The trace in " << filename.toStdString() << " has state information but the labelled transition system does not have it, which is currently not allowed by ltsview.";
     return false;
  }

  Simulation simulation(0, m_lts);
  State* initialState = m_lts->getInitialState();
  if (trace.currentState().size() != m_lts->getNumParameters())
  {
    mCRL2log(mcrl2::log::error) << "The trace in " << filename.toStdString() << " and the labelled transition system have state information of unequal lengths.";
    return false;
  }
  simulation.setInitialState(initialState);
  simulation.start();

  // Get the first state of the trace 
  mcrl2::lps::state currentState = trace.currentState();

  // Load the rest of the trace.
  while (trace.getPosition() != trace.number_of_actions())
  {
    std::string action = pp(trace.currentAction());
    trace.increasePosition();

    QList<Transition *> transitions = simulation.availableTransitions();
    int possibilities = 0;
    Transition *transition = 0;

    for (int i = 0; i < transitions.size(); i++)
    {
      if (action == m_lts->getLabel(transitions[i]->getLabel()))
      {
        possibilities++;
        transition = transitions[i];
      }
    }

    if (possibilities > 1)
    {
      // More than one possibility, meaning that choosing on action name is
      // ambiguous. Solve disambiguation by looking at states

      currentState = trace.currentState();

      // Match is the score keeping track of how well a state matches an LPS
      // state. The (unique) state with the maximum match will be chosen.
      // The value of this match should be the number of variables which have
      // the same value as in the LPS, minus the number of free variables (
      // which are undetectable).
      int maxmatch = -1;

      for (int i = 0; i < transitions.size(); i++)
      {
        State *state = transitions[i]->getEndState();
        int match = 0;

        for (size_t j = 0; j < currentState.size(); j++)
        {
          if (mcrl2::data::pp(currentState[j]) == m_lts->getStateParameterValueStr(state, j))
          {
            match++;
          }
        }

        if (match > maxmatch)
        {
          maxmatch = match;
          transition = transitions[i];
        }
      }
    }
    else if (possibilities == 1)
    {
      // Exactly one possibility, so skip
    }
    else
    {
      // This cannot occur, unless there was some mismatch between lps and lts
      mCRL2log(mcrl2::log::error) << "Could not regenerate trace, does it belong to the loaded LTS?";
      return false;
    }

    simulation.followTransition(transition);
  }

  // Set simulation to the LTS
  *this = simulation;
  return true;
}
