/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

//
// .NAME:
//  Plot Mod
//
// .AUTHOR:
//  Gilberto Camara, Baudoin Raoult and Fernando Ii
//
// .SUMMARY:
//  Defines an concrete class for higher level of control of Plot Mod.
//
//
// .CLIENTS:
//   This class is used the "main" module and by any METVIEW
//   object which needs to know, whether METVIEW is in interative
//   or in batch mode
//
// .RESPONSIBILITIES:
//   - Inform the other classes about the operation of METVIEW
//     (interactive or bacth mode)
//
//
// .COLLABORATORS:
//   PlotModApp ( which will be either PlotModXApp or PlotModBatchApp).
//
// .DESCENDENT:
//   (none)
//
// .RELATED:
//   (none)
//
// .ASCENDENT:
//   (none)
//
#ifndef PlotMod_H
#define PlotMod_H

#include "MvRequest.h"
#include "MagicsObserver.h"

class MvStopWatch;

class PlotMod : protected magics::MagicsObserver
{
public:
    // Access to singleton object
    static PlotMod& Instance();
    static void Instance(PlotMod*);

    // Default Constructor/Destructor
    PlotMod();
    ~PlotMod() {}

    // Returns running mode: true: interactive, false: batch
    bool IsInteractive() { return isInteractive_; }

    // Window mode functions
    //	bool CheckSetWindowOutputFormat( MvRequest& );	//Check and set
    bool IsWindow(const char*);            //Check a given driver
    bool IsWindow() const { return isWindow_; }  //Inquire current status

    // Check/initialize request came from a Macro
    void CalledFromMacro(bool flag) { calledFromMacro_ = flag; }
    bool CalledFromMacro() { return calledFromMacro_; }
    bool CalledFirstFromMacro() { return calledFirstFromMacro_; }
    bool CalledFromMacro(MvRequest&);

    // Output Format methods
    MvRequest OutputFormatInfo();         // Get output info
    MvRequest OutputFormat();             // Get request
    MvRequest MakeDefaultOutputFormat();  // Get default
    void OutputFormat(MvRequest&);        // Initialize
    const char* Destination()             // "SCREEN","PRINTER","PREVIEW"
    {
        return plotDestination_.c_str();
    }
    const char* PrinterName()  // name of the printer
    {
        return printerName_.c_str();
    }

    // Internal Error should not happen in user version
    //	static void InternalError ( const string & message, const string & title = string() );

    // System Error is used when probems are in one of the modules that PlotMod uses
    // including OpenGL
    virtual void SystemError(const string& message, const string& title = string()) = 0;

    // Metview Error is due to user defining wrong parameters
    void MetviewError(const string& message, const string& errorlevel = string());

    // public interface to message logging and popup
    void UserInfoMessage(const string&, bool priority=true);
    void UserWarningMessage(const string&, bool priority=true);
    void UserErrorMessage(const string&, bool priority=true);

    // the module will exit with an error code
    void exitWithError();

    // change/query the error message behaviour
    void setExitOnErrorMessage(bool toggle) { exitOnErrorMessage_ = toggle; }
    bool exitOnErrorMessage() const { return exitOnErrorMessage_; }
    void setPopupErrorMessage(bool toggle) { popupErrorMessage_ = toggle; }
    bool popupErrorMessage() const { return popupErrorMessage_; }

    // Timing methods
    void watchStart(const char*);    // Starts and names the session
    void watchLaptime(const char*);  // Prints the laptime since the previous call to
                                     // watchLaptime() or from watchStart()
    void watchStop();                // Stops and prints a report

protected:
    // from MagicsObserver
    void warningMessage(const string&) override;
    void errorMessage(const string&) override;
    void infoMessage(const string&) override;
    void progressMessage(const string&) override {}

    // log handling
    enum MessageLevel {InfoMessageLevel, WarningMessageLevel, ErrorMessageLevel};
    virtual void showInfoMessagePopup(const string&) {}
    virtual void showWarningMessagePopup(const string&) {}
    virtual void showErrorMessagePopup(const string&) {}

    // print a message to the log window or the output so it will be visible without '-slog'
    virtual void printMessage(const string&, MessageLevel, bool) = 0;
    void sendMessageToService(const string&);
    void writeToLog(const string&, MessageLevel);

    bool isInteractive_{false};
    bool isWindow_{false};            // output on screen?
    bool exitOnErrorMessage_{true};  // do we exit when we get an error message?
    bool popupErrorMessage_{true};   // pop-up error messages?
    MvRequest outFormat_;      // output format info
    string plotDestination_;   // "SCREEN","PRINTER","PREVIEW"."FILE"
    string printerName_;       // name of the printer
    MvStopWatch* watch_{nullptr};       // timing debug

    // If request came from a Macro then both variables are initialised to
    // true. In the interactive mode, after calling the ploting routine for
    // the first time, variable calledFromMacro_ is updated to false.
    // This is needed because uPlot should treat all the icons dropped
    // afterwards as interactive mode, instead of batch mode.
    bool calledFromMacro_{false};     // request came from Macro?
    bool calledFirstFromMacro_{false};  // request came from Macro?

    static PlotMod* plotModInstance_;
};
#endif
