/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var resolve = require( 'path' ).resolve;
var tape = require( 'tape' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var EPS = require( '@stdlib/constants/float64/eps' );
var abs = require( './../../../../base/special/abs' );
var sqrt = require( './../../../../base/special/sqrt' );
var pow = require( './../../../../base/special/pow' );
var isnan = require( './../../../../base/assert/is-nan' );
var isPositiveZero = require( './../../../../base/assert/is-positive-zero' );
var tryRequire = require( '@stdlib/utils/try-require' );


// VARIABLES //

var hypot = tryRequire( resolve( __dirname, './../lib/native.js' ) );
var opts = {
	'skip': ( hypot instanceof Error )
};


// FIXTURES //

var data = require( './fixtures/julia/data.json' );


// TESTS //

tape( 'main export is a function', opts, function test( t ) {
	t.ok( true, __filename );
	t.strictEqual( typeof hypot, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns `+infinity` if either argument is `+-infinity`', opts, function test( t ) {
	var h;

	h = hypot( PINF, 3.14 );
	t.strictEqual( h, PINF, 'returns +infinity' );

	h = hypot( 3.14, PINF );
	t.strictEqual( h, PINF, 'returns +infinity' );

	h = hypot( NINF, 3.14 );
	t.strictEqual( h, PINF, 'returns +infinity' );

	h = hypot( 3.14, NINF );
	t.strictEqual( h, PINF, 'returns +infinity' );

	h = hypot( PINF, PINF );
	t.strictEqual( h, PINF, 'returns +infinity' );

	h = hypot( NINF, PINF );
	t.strictEqual( h, PINF, 'returns +infinity' );

	h = hypot( PINF, NINF );
	t.strictEqual( h, PINF, 'returns +infinity' );

	h = hypot( NINF, NINF );
	t.strictEqual( h, PINF, 'returns +infinity' );

	t.end();
});

tape( 'the function returns `NaN` if either argument is `NaN`', opts, function test( t ) {
	var h;

	h = hypot( NaN, 3.14 );
	t.strictEqual( isnan( h ), true, 'returns NaN' );

	h = hypot( 3.14, NaN );
	t.strictEqual( isnan( h ), true, 'returns NaN' );

	h = hypot( NaN, NaN );
	t.strictEqual( isnan( h ), true, 'returns NaN' );

	t.end();
});

tape( 'the function returns `+0` if both arguments are `+-0`', opts, function test( t ) {
	var h;

	h = hypot( +0.0, +0.0 );
	t.strictEqual( isPositiveZero( h ), true, 'returns +0' );

	h = hypot( -0.0, +0.0 );
	t.strictEqual( isPositiveZero( h ), true, 'returns +0' );

	h = hypot( +0.0, -0.0 );
	t.strictEqual( isPositiveZero( h ), true, 'returns +0' );

	h = hypot( -0.0, -0.0 );
	t.strictEqual( isPositiveZero( h ), true, 'returns +0' );

	t.end();
});

tape( 'the function computes the hypotenuse', opts, function test( t ) {
	var expected;
	var delta;
	var tol;
	var h;
	var x;
	var y;
	var i;

	x = data.x;
	y = data.y;
	expected = data.expected;

	for ( i = 0; i < x.length; i++ ) {
		h = hypot( x[ i ], y[ i ] );
		if ( h === expected[ i ] ) {
			t.ok( true, 'x: '+x[i]+'. y: '+y[i]+'. h: '+h+'. Expected: '+expected[i]+'.' );
		} else {
			delta = abs( h - expected[ i ] );
			tol = 1.4 * EPS * abs( expected[ i ] );
			t.strictEqual( delta <= tol, true, 'within tolerance. x: '+x[i]+'. y: '+y[i]+'. h: '+h+'. Expected: '+expected[i]+'. Delta: '+delta+'. Tol: '+tol+'.' );
		}
	}
	t.end();
});

tape( 'the function computes the hypotenuse (canonical inputs)', opts, function test( t ) {
	var h;

	h = hypot( 3.0, 4.0 );
	t.strictEqual( h, 5.0, 'returns 5.0' );

	h = hypot( 6.0, 8.0 );
	t.strictEqual( h, 10.0, 'returns 10.0' );

	h = hypot( 5.0, 12.0 );
	t.strictEqual( h, 13.0, 'returns 13.0' );

	t.end();
});

tape( 'the function avoids overflow', opts, function test( t ) {
	var h;

	h = sqrt( pow( 1.0e308, 2 ) + pow( 1.0e308, 2 ) );
	t.strictEqual( h, PINF, 'returns +infinity' );

	h = hypot( 1.0e308, 1.0e308 );
	t.strictEqual( h, 1.4142135623730951e308, 'avoids overflow' );

	t.end();
});

tape( 'the function avoids underflow', opts, function test( t ) {
	var h;

	h = sqrt( pow( 1.0e-200, 2 ) + pow( 1e-200, 2 ) );
	t.strictEqual( h, 0.0, 'returns 0' );

	h = hypot( 1.0e-200, 1.0e-200 );
	t.strictEqual( h, 1.414213562373095e-200, 'avoids underflow' );

	t.end();
});
