/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var EPS = require( '@stdlib/constants/float64/eps' );
var MAX_FLOAT64 = require( '@stdlib/constants/float64/max' );
var isnan = require( './../../../../base/assert/is-nan' );
var isPositiveZero = require( './../../../../base/assert/is-positive-zero' );
var diff = require( './../lib' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof diff, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function throws an error if provided an unrecognized/unsupported `scale` function name', function test( t ) {
	t.throws( foo, Error, 'throws an error' );
	t.end();

	function foo() {
		diff( -2, 5, 'abcdefg' );
	}
});

tape( 'if both `x` and `y` equal `+infinity`, the function returns `NaN`', function test( t ) {
	var d = diff( PINF, PINF );
	t.equal( isnan( d ), true, 'returns NaN' );
	t.end();
});

tape( 'if both `x` and `y` equal `-infinity`, the function returns `NaN`', function test( t ) {
	var d = diff( NINF, NINF );
	t.equal( isnan( d ), true, 'returns NaN' );
	t.end();
});

tape( 'if `x` or `y` equals `+infinity` and the other value equals any other number (beside +infinity), the function returns `+infinity`', function test( t ) {
	var values;
	var d;
	var i;

	values = [
		NINF,
		5.0,
		10.0,
		3.14,
		-1.0e308
	];

	for ( i = 0; i < values.length; i++ ) {
		d = diff( PINF, values[i] );
		t.equal( d, PINF, 'returns +infinity' );

		d = diff( values[i], PINF );
		t.equal( d, PINF, 'returns +infinity' );
	}
	t.end();
});

tape( 'if `x` or `y` equals `-infinity` and the other value equals any other number (beside -infinity), the function returns `+infinity`', function test( t ) {
	var values;
	var d;
	var i;

	values = [
		PINF,
		5.0,
		10.0,
		3.14,
		-1.0e308
	];

	for ( i = 0; i < values.length; i++ ) {
		d = diff( NINF, values[i] );
		t.equal( d, PINF, 'returns +infinity' );

		d = diff( values[i], NINF );
		t.equal( d, PINF, 'returns +infinity' );
	}
	t.end();
});

tape( 'if `x` and/or `y` is `NaN`, the function returns `NaN`', function test( t ) {
	var d;

	d = diff( NaN, 5.0 );
	t.equal( isnan( d ), true, 'returns NaN' );

	d = diff( 3.0, NaN );
	t.equal( isnan( d ), true, 'returns NaN' );

	t.end();
});

tape( 'if `x = y` and `x` and `y` are both finite, the function always returns `0`', function test( t ) {
	var values;
	var v;
	var d;
	var i;

	values = [
		5.0,
		-1.0,
		3.14,
		-3.14,
		0.0,
		-0.0,
		1.0e308,
		-1.0e-324
	];

	for ( i = 0; i < values.length; i++ ) {
		v = values[ i ];
		d = diff( v, v );
		t.equal( d, 0.0, 'returns 0' );
	}
	t.end();
});

tape( 'if `x` and `y` both equal `+-0`, the function returns `0`', function test( t ) {
	var d;

	d = diff( -0.0, 0.0 );
	t.equal( isPositiveZero( d ), true, 'returns +0' );

	d = diff( -0.0, -0.0 );
	t.equal( isPositiveZero( d ), true, 'returns +0' );

	d = diff( 0.0, 0.0 );
	t.equal( isPositiveZero( d ), true, 'returns +0' );

	d = diff( 0.0, -0.0 );
	t.equal( isPositiveZero( d ), true, 'returns +0' );

	t.end();
});

tape( 'if a scale function returns `0`, the function returns `NaN`', function test( t ) {
	var d;

	d = diff( -1.0, 1.0, 'mean' );
	t.equal( isnan( d ), true, 'returns NaN' );

	d = diff( -1.0, 0.0, 'max' );
	t.equal( isnan( d ), true, 'returns NaN' );

	d = diff( 0.0, 1.0, 'min' );
	t.equal( isnan( d ), true, 'returns NaN' );

	d = diff( 0.0, -1.0, 'min-abs' );
	t.equal( isnan( d ), true, 'returns NaN' );

	d = diff( 0.0, 5.0, 'x' );
	t.equal( isnan( d ), true, 'returns NaN' );

	d = diff( 5.0, 0.0, 'y' );
	t.equal( isnan( d ), true, 'returns NaN' );

	d = diff( 2.0, 5.0, scale );
	t.equal( isnan( d ), true, 'returns NaN' );

	t.end();

	function scale() {
		return 0.0;
	}
});

tape( 'the function computes the relative difference (default)', function test( t ) {
	var expected;
	var d;

	d = diff( -2.0, 5.0 );
	expected = (7.0/5.0) / EPS;

	t.equal( d, expected, 'returns '+expected );
	t.end();
});

tape( 'the function computes the relative difference (max-abs)', function test( t ) {
	var expected;
	var d;

	d = diff( 5.0, -2.0, 'max-abs' );
	expected = ((5.0+2.0)/5.0) / EPS;

	t.equal( d, expected, 'returns '+expected );
	t.end();
});

tape( 'the function computes the relative difference (max)', function test( t ) {
	var expected;
	var d;

	d = diff( 1.0, 10.0, 'max' );
	expected = (9.0/10.0) / EPS;

	t.equal( d, expected, 'returns '+expected );
	t.end();
});

tape( 'the function computes the relative difference (min-abs)', function test( t ) {
	var expected;
	var d;

	d = diff( -1.0, 10.0, 'min-abs' );
	expected = 11.0 / EPS;

	t.equal( d, expected, 'returns '+expected );
	t.end();
});

tape( 'the function computes the relative difference (min)', function test( t ) {
	var expected;
	var d;

	d = diff( 1.0, 10.0, 'min' );
	expected = 9.0 / EPS;

	t.equal( d, expected, 'returns '+expected );
	t.end();
});

tape( 'the function computes the relative difference (x)', function test( t ) {
	var expected;
	var d;

	d = diff( 10.0, -1.0, 'x' );
	expected = (11.0/10.0) / EPS;

	t.equal( d, expected, 'returns '+expected );
	t.end();
});

tape( 'the function computes the relative difference (y)', function test( t ) {
	var expected;
	var d;

	d = diff( 10.0, -1.0, 'y' );
	expected = 11.0 / EPS;

	t.equal( d, expected, 'returns '+expected );
	t.end();
});

tape( 'the function computes the relative difference (mean)', function test( t ) {
	var expected;
	var d;

	d = diff( -1.0, 10.0, 'mean' );
	expected = (11.0/4.5) / EPS;

	t.equal( d, expected, 'returns '+expected );
	t.end();
});

tape( 'the function computes the relative difference (mean-abs)', function test( t ) {
	var expected;
	var d;

	d = diff( 10.0, -1.0, 'mean-abs' );
	expected = (11.0/5.5) / EPS;

	t.equal( d, expected, 'returns '+expected );
	t.end();
});

tape( 'the function supports custom scale functions', function test( t ) {
	var expected;
	var d;

	d = diff( 12.15, 12.149999999999999, scale );
	expected = ((12.15-12.149999999999999)/12.15) / EPS;

	t.equal( d, expected, 'returns '+expected );
	t.end();

	function scale( x, y ) {
		return ( x < y ) ? y : x;
	}
});

tape( 'if computing the relative difference will result in overflow, the function returns the maximum double-precision floating-point number', function test( t ) {
	var d = diff( 1.0e304, 1.0, 'min' );
	t.equal( 1.0e304/EPS, PINF, 'conversion will result in overflow' );
	t.equal( d, MAX_FLOAT64, 'returns '+MAX_FLOAT64 );
	t.end();
});
