/*	Multibuf_ImageInputStream

PIRL CVS ID: Multibuf_ImageInputStream.java,v 1.8 2012/04/16 06:10:20 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Image_Tools;

import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.ImageInputStreamImpl;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.List;
import java.util.Vector;
import java.lang.IllegalArgumentException;

/**	An <i>Multibuf_ImageInputStream</i> provides an <i>ImageInputStream</i>
	interface to a List of ByteBuffers.
<p>
	The buffer list is thread-safe.
<p>
	@author		Bradford Castalia UA/PIRL
	@version	1.8
	@see		javax.imageio.stream.ImageInputStream
*/
public class Multibuf_ImageInputStream
	extends ImageInputStreamImpl
{
public static final String
	ID = "PIRL.Image_Tools.Multibuf_ImageInputStream (1.8 2012/04/16 06:10:20)";

/*------------------------------------------------------------------------------
	Buffer Management
*/
/**	The list of ByteBuffers that provide the data source.
*/
private List
	Data_Buffers		= Collections.synchronizedList (new Vector ());

/**	Current image data buffer.
<p>
	The Data_Buffer contains the current data accessed by the
	ImageInputStream interface. It's position is where the next
	ImageInputStream read byte will be obtained; this is equivalent to
	the virtual stream location. It's limit is the end of valid source
	data in the buffer (the buffer may not be full to capacity).
*/
protected ByteBuffer
	Data_Buffer			= null;

/**	The index of the Data_Buffer in the Data_Buffers.
*/
private int
	Buffer_Index		= 0;

/*	!!! Workaround !!!

	The reset mehod of ImageInputStreamImpl throws an exception if the
	position popped from the markByteStack is located before the
	flushedPos (beginning of the buffer). This implementation allows
	seeking to a location before the beginning of the buffer. The
	workaround is to leave the flushedPos at zero and use an alternate
	variable for the buffer stream location.
*/
/**	Virtual stream location of the Data_Buffer
<p>
	The first byte of the buffer is located at Buffer_Location offset in
	the stream.
*/
protected long
	Buffer_Location		= 0;

/**	Total size (bytes) of all buffers in the list that constitute the
	source data stream.
*/
private long
	Stream_Size			= 0;	// Sum of all buffer content amounts.

//------------------------------------------------------------------------------
//  DEBUG control
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTORS	= 1 << 0,
	DEBUG_ACCESSORS		= 1 << 1,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Multibuf_ImageInputStream on a List of ByteBuffers.
<p>
	@param	buffers	The List of ByteBuffers to use to use as the data source
		in the order they occur in the List.
	@throws	IllegalArgumentException If the List contains a non-null
		Object that is not a.ByteBuffer instance.
	@see	#Add(ByteBuffer)
*/
public Multibuf_ImageInputStream
	(
	List		buffers
	)
	throws IllegalArgumentException
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> Multibuf_ImageInputStream: " + buffers);
if (buffers != null)
	{
	for (int
			index = 0;
			index < buffers.size ();
			index++)
		{
		Object
			buffer = buffers.get (index);
		if (buffer != null &&
			! (buffer instanceof ByteBuffer))
			throw new IllegalArgumentException (ID + '\n'
				+ "Can't construct a Multibuf_ImageInputStream from a List\n"
				+ "where entry " + index + '/' + buffers.size ()
					+ " is not a ByteBuffer.");
		Add ((ByteBuffer)buffer);
		}
	}
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println ("<<< Multibuf_ImageInputStream");
}

/**	Construct a Multibuf_ImageInputStream that initial contains a single
	ByteBuffer.
<p>
	@param	buffer	The ByteBuffer to provide stream data.
	@see	#Add(ByteBuffer)
*/
public Multibuf_ImageInputStream
	(
	ByteBuffer		buffer
	)
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> Multibuf_ImageInputStream: " + buffer);
Add (buffer);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println ("<<< Multibuf_ImageInputStream");
}

/**	Construct an empty Multibuf_ImageInputStream.
*/
public Multibuf_ImageInputStream ()
{}

/*==============================================================================
	Accessors
*/
/**	Add a ByteBuffer to the list of buffers providing stream data.
<p>
	If the buffer is null or empty (zero limit) it is ignored.
<p>
	A duplicate of the buffer (pointers are copied, data is not) is
	added to the end of the buffers list and rewound (position set
	to zero). The {@link #length() length} of the stream is incremented
	by the amount of data (limit) in the buffer. If a current buffer
	has not be set the new buffer is set as the current buffer.
<p>
	@param	buffer	The ByteBuffer to be added to the buffers list.
	@return	This Multibuf_ImageInputStream.
*/
synchronized public Multibuf_ImageInputStream Add
	(
	ByteBuffer	buffer
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> Multibuf_ImageInputStream.Add: " + buffer);
if (buffer != null &&
	buffer.limit () != 0)
	{
	Data_Buffers.add (buffer = buffer.duplicate ());
	Stream_Size += buffer.rewind ().limit ();
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("    Buffer " + (Data_Buffers.size () - 1)
				+ ": adds " + buffer.limit ()
				+ " of " + Stream_Size + " bytes.");
	if (Data_Buffer == null)
		Data_Buffer = buffer;
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< Multibuf_ImageInputStream.Add");
return this;
}

/**	Add the contents of another Multibuf_ImageInputStream to this
	Multibuf_ImageInputStream.
<p>
	Each data buffer in the source stream is {@link #Add(ByteBuffer)
	Add}ed to the end of the list of data buffers in this stream in the
	order in which they are listed.
<p>
	@return	This Multibuf_ImageInputStream.
*/
synchronized public Multibuf_ImageInputStream Add
	(
	Multibuf_ImageInputStream	stream
	)
{
for (int
		index = 0;
		index < stream.Data_Buffers.size ();
		index++)
	Add ((ByteBuffer)stream.Data_Buffers.get (index));
return this;
}

/**	Test if the specified buffer is contained in the buffers list.
<p>
	<b>Warning</b>: The {@link ByteBuffer#equals(Object)} method is used
	to compare the specified buffer against the contents of the buffers
	list. This compares the structure and contents of the buffers, not
	their object reference values. This can be expensive for large and/or
	many buffers. It will also test true if a buffer in the list happens
	to have the same content as the specified buffer even if the buffer
	objects are different.
<p>
	@param	buffer	The ByteBuffer to test for.
	@return	true if the specified buffer equals some other 
*/
public boolean Contains
	(
	ByteBuffer	buffer
	)
{return Data_Buffers.contains (buffer);}

/**	Remove a buffer from the buffers list.
<p>
	The indexed buffer is removed from the buffers list and the {@link
	#length() length} of the stream decremented by the amount of data
	(limit) in the buffer.
<p>
	If the index of the removed buffer is less than the index of the
	current buffer the current {@link
	ImageInputStreamImpl#getStreamPosition() stream position} and current
	buffer location are decremented by the amount of data in the removed
	buffer.
<p>
	If the index of the removed buffer is equal to the index of the
	current buffer the current {@link
	ImageInputStreamImpl#getStreamPosition() stream position} is set to
	the current buffer location. If there is another buffer in the
	buffers list following the removed buffer, the next buffer is set as
	the current buffer; thus the fist byte of the next available buffer
	will be the next byte read (unless the stream position is {@link
	#seek(long) moved}). However, if there is no next buffer - the
	removed buffer is at the end of the buffers list - but there is a
	buffer in the buffers list immediately preceeding the removed buffer,
	then the previous buffer becomes the current buffer and its position
	is set to its end (limit); thus the stream position is now st the end
	of the stream. Finally, if the removed buffer is the last one in the
	buffers list the current buffer is set to null and all pointers are
	at zero which is the end of the (now empty) stream.
<p>
	@param	index	The index of the buffer to be removed.
	@return	This Multibuf_ImageInputStream.
	@throws	ArrayIndexOutOfBoundsException	If the specified index is
		less than zero or greater than or equal to the size of the
		buffers list.
*/
synchronized public Multibuf_ImageInputStream Remove
	(
	int		index
	)
	throws ArrayIndexOutOfBoundsException
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> Multibuf_ImageInputStream.Remove");
if (index >= Data_Buffers.size () ||
	index < 0)
	throw new ArrayIndexOutOfBoundsException (ID + '\n'
		+ "Can't remove buffer at index "
			+ index + '/' + Data_Buffers.size () + '.');
ByteBuffer
	buffer = (ByteBuffer)Data_Buffers.remove (index);
int
	size = buffer.limit ();
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("    Buffer " + index
			+ ": removes " + size
			+ " of " + Stream_Size + " bytes.\n"
		+"    Buffer_Index = " + Buffer_Index + '/' + Data_Buffers.size () + '\n'
		+"    Buffer_Location = " + Buffer_Location + '\n'
		+"          streamPos = " + streamPos);
Stream_Size -= size;
if (index < Buffer_Index)
	{
	/*	All locations above the removed buffer
		shift down by the amount of removed data.
		Locations remain attached to the same data.
	*/
	streamPos -= size;
	Buffer_Location -= size;
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("    Buffer_Location = " + Buffer_Location + '\n'
			+"          streamPos = " + streamPos);
	}
else
if (index == Buffer_Index)
	{
	/*	The rug's been removed!

		Locations within the range of the removed buffer
		are all clamped to the location of the next datum
		at the beginning of the next buffer
		which has been moved down to the Buffer_Location.
	*/
	streamPos = Buffer_Location;
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("          streamPos = " + streamPos);

	if (Buffer_Index <= Data_Buffers.size ())
		{
		//	Use the next buffer, at the current Buffer_Index.
		Data_Buffer = (ByteBuffer)Data_Buffers.get (Buffer_Index);
		if ((DEBUG & DEBUG_ACCESSORS) != 0)
			System.out.println
				("    Using next buffer: " + Data_Buffer);
		}
	else
		{
		//	There is no next buffer.
		if (Previous_Buffer ())
			{
			//	Position at EOF which is Stream_Size.
			Data_Buffer.position (Data_Buffer.limit ());
			if ((DEBUG & DEBUG_ACCESSORS) != 0)
				System.out.println
					("    Buffer_Location = " + Buffer_Location + '\n'
					+"    Using previous buffer: " + Data_Buffer);
			}
		else
			{
			/*	There is no previous buffer; no buffers remain.

				At this point the stream is empty
				and all locations are at EOF which is zero.
			*/
			Data_Buffer = null;
			if ((DEBUG & DEBUG_ACCESSORS) != 0)
				System.out.println
					("    No buffers reamin.");
			}
		}
	}
/*
	All locations below the removed buffer
	remain unaffected.
*/
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< Multibuf_ImageInputStream.Remove");
return this;
}

/**	Remove the specified buffer from the buffers list.
<p>
	<b>Warning</b>: The specified buffer is sought using the {@link
	List#indexOf(Object)} method on the list of data buffers which
	employs the {@link ByteBuffer#equals(Object)} method. See the {@link
	#Contains(ByteBuffer) Contains} method for details.
<p>
	@param	buffer	The ByteBuffer to be removed.
	@return	This Multibuf_ImageInputStream.
	@see	#Remove(int)
*/
public Multibuf_ImageInputStream Remove
	(
	ByteBuffer	buffer
	)
{
int
	index = Data_Buffers.indexOf (buffer);
if (index >= 0)
	Remove (index);
return this;
}

/**	Get the List of data buffers backing this object.
<p>
	<b>Caution</b>: Modifying the contents of the list by reordering or
	adding to the List, or changing the amount of valid data in a buffer
	(its {@link ByteBuffer#limit(int) limit}) could invalidate the
	current {@link ImageInputStream#getStreamPosition() stream position}
	and/or {@link #length() stream length} information with indeterminate
	(i.e. bad) consequences. Use the {@link #Add(ByteBuffer) Add} and
	{@link #Remove(int) Remove} methods to safely manipulate the data
	buffer List.
<p>
	@return	The List of ByteBuffer objects backing this
		Multibuf_ImageInputStream.
*/
public List Data_Buffers ()
{return Data_Buffers;}

/*==============================================================================
	Manipulators
*/
/**	Set the next buffer as the current buffer.
<p>
	A next buffer is present if the current buffer index plus one is less
	than the size of the buffers list. In this case the Buffer_Location
	is incremented by the amount of data (limit) in the current buffer,
	the buffer index is incremented and the correspoding buffer from the
	buffer list is set as the current Data_Buffer. It's position is
	rewound to zero.
<p>
	@return	true if a next buffer has been set; false if there is no next
		buffer.
*/
protected boolean Next_Buffer ()
{
if ((Buffer_Index + 1) >= Data_Buffers.size ())
	//	Thre is no next buffer.
	return false;

//	Move the Buffer_Location to the beginning of the next buffer.
Buffer_Location += Data_Buffer.limit ();

//	Get the next buffer.
Data_Buffer = (ByteBuffer)Data_Buffers.get (++Buffer_Index);
//	Expect sequential reads; seeks will repostion.
Data_Buffer.rewind ();
return true;
}

/**	Set the previous data buffer as the current buffer.
<p>
	If the current buffer index in the buffers list is not zero, the
	buffer index is decremented and the corresponding buffer from the
	buffer list is set as the current Data_Buffer. It's position
	remains unchanged. The Buffer_Location is decremented by the amount
	of data (limit) in the new buffer.
<p>
	@return	true if a previous buffer has been set; false if there is no
		previous buffer.
*/
protected boolean Previous_Buffer ()
{
if (Buffer_Index == 0)
	//	There is no previous buffer.
	return false;

//	Get the previous buffer.
Data_Buffer = (ByteBuffer)Data_Buffers.get (--Buffer_Index);
Buffer_Location -= Data_Buffer.limit ();
return true;
}

/*==============================================================================
	ImageInputStream
*/
//	Required method.
/**	Read a source byte.
<p>
	The {@link #getBitOffset() bit offset} is set to zero.
<p>
	@return	The next byte value from the source. This will be -1
		if the current source location is the end of the stream.
*/
public int read ()
{
int
	datum = -1;
if (streamPos < Stream_Size)
	{
	if (Data_Buffer.remaining () > 0 ||
		Next_Buffer ())
		{
		//	Required ImageInputStream functionality.
		bitOffset = 0;
		++streamPos;
		datum = (int)Data_Buffer.get () & 0xFF;
		}
	}
return datum;
}

//	Required method.
/**	Read a sequence of source bytes and store them in an array.
<p>
	The number of bytes actually read may be less than the specified
	length. The length will be reduced to the smaller of the amount
	of space in the data array following the offset or the amount of
	buffered data remaining after an attempt to provide length bytes.
<p>
	The {@link #getBitOffset() bit offset} is set to zero.
<p>
	The current source input location is advanced by the amount of
	data read.
<p>
	@param	data	The byte array into which the data is to be stored.
	@param	offset	The offset of the array where the first byte read is
		to be stored.
	@param	length	The number of bytes to read.
	@return	The number of bytes read and stored. This will be -1 if the
		current source location is the end of the stream.
	@throws	NullPointerException If the data array is null.
	@throws IndexOutOfBoundsException If the offset or length are
		negative, or the offset is greater than the length of the data
		array.
*/
public int read
	(
	byte[]	data,
	int		offset,
	int		length
	)
{
if (streamPos >= Stream_Size)
	return -1;

if (data == null)
	throw new NullPointerException (ID + '\n'
		+ "Attempt to read into a null data array.");
if (length < 0 ||
	offset < 0 ||
	offset >= data.length)
	throw new IndexOutOfBoundsException (ID + '\n'
		+ "Attempt to read into array of length " + data.length
			+ " at offset " + offset
			+ " for amount " + length + '.');
if ((offset + length) > data.length)
	length = data.length - offset;

int
	amount = 0;
while (length > 0)
	{
	int
		count = Math.min (length, Data_Buffer.remaining ());
	Data_Buffer.get (data, offset, count);
	amount += count;
	if ((length -= count) == 0 ||
		! Next_Buffer ())
		break;
	offset += count;
	}
//	Required ImageInputStream functionality.
bitOffset = 0;
streamPos += amount;
return amount;
}

/**	Set the current source stream location where the next byte will be read.
<p>
	The {@link #getBitOffset() bit offset} is set to zero.
<p>
	@param	location	The source stream offset for the next read
		position. A location less than zero will be set to zero; a
		location greater than the source stream size will be set to the
		end of the source stream.
*/
public void seek
	(
	long	location
	)
{
if (location < Stream_Size)
	{
	if (location < 0)
		location = 0;

	//	Inter-buffer positioning.
	if (location > (Buffer_Location + Data_Buffer.limit ()))
		while (Next_Buffer () &&
		location > (Buffer_Location + Data_Buffer.limit ()));
	else
	if (location < Buffer_Location)
		while (Previous_Buffer () &&
		location < Buffer_Location);

	//	Intra-buffer positioning.
	Data_Buffer.position ((int)(location - Buffer_Location));
	streamPos = location;
	}
else
	streamPos = Stream_Size;
bitOffset = 0;
}

/**	Determine if the source data is cached.
<p>
	@return	true
*/
public boolean isCached ()
{return true;}

/**	Determine if the source data is cached in memory.
<p>
	@return	true
*/
public boolean isCachedMemory ()
{return true;}

/**	Get the total size of all the buffer contents.
<p>
	@return	The size of the source stream.
*/
public long length ()
{return Stream_Size;}

/**	Suggests that data before a source location may be discared.
<p>
	<b>N.B.</b>: No buffered data is ever discared; the suggestion is
	ignored.
<p>
	@param	location	The source stream offset before which buffered
		data may be discarded.
*/
public void flushBefore
	(
	long	location
	)
{}

/**	Pretends to close the stream.
<p>
	Nothing is done since the source data is externally owned.
*/
public void close ()
{}

}
