/*	Memory_History

PIRL CVS ID: Memory_History.java,v 1.5 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Viewers;

import	javax.swing.Timer;
import	javax.swing.event.ChangeEvent;
import	javax.swing.event.ChangeListener;
import	java.awt.event.ActionEvent;
import	java.awt.event.ActionListener;
import	java.util.Vector;
import	java.util.Iterator;


/**	A <i>Memory_History</i> provides a history of Java Runtime Environment
	(JRE) memory use.
<p>
	The JRE reports three categories of memory: the total amount of
	{@link #Available_Memory() available memory} in the JRE heap space,
	the amount of {@link #Allocated_Memory allocated memory} used by
	objects, and the amount of {@link #Free_Memory() free memory} that is
	no longer in use by objects which can be garbage collected.

*/
public class Memory_History
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Viewers.Memory_Chart (1.5 2012/04/16 06:22:59)";

/**	The 
*/
public static final int
	DEFAULT_MAX_SAMPLES		= 32;

//!	Memory amount samples.
protected int
	Samples					= 0;

/**	The {@link #Free_Memory() free memory} sampling history.
*/
protected long[]
	Free;

/**	The {@link #Allocated_Memory() allocated memory} sampling history.
*/
protected long[]
	Allocated;

//!	Polling manager.
private Timer
	Poll_Timer				= null;

//!	The previous non-zero sampling rate.
private int
	Previous_Rate			= 0;

//!	ChangeListener objects that have been registered.
private Vector
	Change_Listeners		= new Vector ();

//!	The Java virtual machine (JVM) runtime environment (JRE) object.
private static final Runtime
	JRE						= Runtime.getRuntime ();

/*==============================================================================
	Constructors
*/
/**	Constructs at Memory_History with specified max samples and
	sampling rate.
<p>
	@param	max_samples	The maximum number of JRE memory use samples
		to acquire.
	@param	rate	The rate, in seconds, between memory samples.
*/
public Memory_History
	(
	int		max_samples,
	int		rate
	)
{
if (max_samples <= 0)
	max_samples = DEFAULT_MAX_SAMPLES;
Free      = new long[max_samples];
Allocated = new long[max_samples];
Rate (rate);
}

/**	Constructs an idle Memory_History with {@link #DEFAULT_MAX_SAMPLES}
	sample capacity.
<p>
	The Memory_History will have no history capacity and a memory
	sampling will be disabled.
*/
public Memory_History ()
{this (0, 0);}

/*==============================================================================
	Convenience functions
*/
/**	Get the amount of memory allocated by the Java runtime environment
	but not currently in use.
<p>
	Free memory is part of the {@link #Allocated_Memory() allocated} memory
	pool but is not currently in use. Garbage collection frees memory.
<p>
	@return	The amount, in bytes, of allocated but free memory.
	@see	Runtime#freeMemory()
*/
public static long Free_Memory ()
{return JRE.freeMemory ();}

/**	Get the amount of memory currently in use by the Java runtime
	environment.
<p>
	Allocated memory is taken from the total amount of memory made
	available to the Java runtime environment when the Java virtual
	machine is started. Memory that is allocated, but not {@link
	#Free_Memory() free} is currently in use by the virtual machine,
	including any user objects it is managing.
<p>
	@return	The amount, in bytes, of allocated memory.
	@see	#Available_Memory()
	@see	Runtime#totalMemory()
*/
public static long Allocated_Memory ()
{return JRE.totalMemory ();}

/**	Get the total amount of memory available to the Java runtime
	environment.
<p>
	When the Java virtual machine is started some amount of the host
	system memory is reserved for use by the Java runtime environment.
	{@link #Allocated_Memory() allocated} memory is taken from available
	memory when needed. The default amount of available memory can be
	changed using the java -Xmx command line option.
<p>
	@return	The total amount of memory, in bytes, available to the Java
		runtime environment.
	@see	Runtime#maxMemory
*/
public static long Available_Memory ()
{return JRE.maxMemory ();}

/*==============================================================================
	Sampling
*/
/**	Get the rate at which memory usage is being sampled.
<p>
	@return	The sampling rate in seconds. This will be zero if sampling
		is not active.
	@see	#Rate(int)
	@see	#Previous_Rate()
*/
public int Rate ()
{
if (Poll_Timer == null ||
	! Poll_Timer.isRunning ())
	return 0;
return Poll_Timer.getDelay () / 1000;
}
	
/**	Set the rate at which to sample memory usage.
<p>
	If the current sampling rate is the same as the new sampling
	rate, nothing is done.
<p>
	If the new sampling rate is less than or equal to zero the
	sampling timer is stopped.
<p>
	If the new sampling rate is positive a samping timer will be
	constructed if one has not already been provided; otherwise the
	current sampling history will be {@link #Clear() cleared}. If the
	sampling timer is not already running it will be started. The
	sampling timer's action event will {@link #Update() updated} the
	memory history at the specified rate, and will continue to be updated
	at the same rate until sampling is discontinued by setting the rate
	at or below zero. The sampling rate may be changed while sampling is
	active.
<p>
	The intention is to have a sample at each rate point in time.
	However, the timer can not guarantee this: Other activity on the
	system may prevent the timer from generating an event at the
	requested time. Nevertheless, by preventing events from being
	coalesced there will still be the correct number of events generated
	for the overall amount of elapsed time even though the burst of
	catch-up events will result in memory values not being sampled at the
	expected time; no sampling events will be dropped.
<p>
	After the sampling rate has been changed a ChangeEvent is {@link
	#fireChange() fired}.
<p>
	@param	rate	The rate, in seconds, between each memory usage sample.
		If less than or equal to zero sampling will be discontinued.
	@return	The previous sampling rate. If this value is not zero the
		{@link #Previous_Rate() previous rate} will be recorded.
*/
public synchronized int Rate
	(
	int		rate
	)
{
int
	previous_rate = Rate ();
if (previous_rate == rate)
	return previous_rate;
if (rate <= 0)
	{
	if (Poll_Timer != null)
		Poll_Timer.stop ();
	}
else
	{
	rate *= 1000; 
	if (Poll_Timer == null)
		{
		//	First time setup.
		Poll_Timer = new Timer (rate, new ActionListener ()
			{public void actionPerformed (ActionEvent event)
			{Update ();}});
		Poll_Timer.setRepeats (true);
		Poll_Timer.setCoalesce (false);
		}
	else
		Clear ();
	if (Poll_Timer.getDelay () * 1000 != rate)
		Poll_Timer.setDelay (rate);
	if (! Poll_Timer.isRunning ())
		Poll_Timer.start ();
	}
if (previous_rate != 0)
	Previous_Rate = previous_rate;
fireChange ();
return previous_rate;
}

/**	Get the previous sampling rate.
<p>
	The value returned is the previous non-zero sampling rate that was
	used. However, if sampling was never enabled then this will be zero.
<p>
	@return	The previous non-zero sampling rate, or zero if sampling was
		never enabled.
	@see	#Rate()
*/
public int Previous_Rate ()
{return Previous_Rate;}

/**	Update the memory history with new samples.
<p>
	After the memory use samples have been recorded a ChangeEvent is
	{@link #fireChange() fired}.
<p>
	@see	#Historical(long, long)
*/
private void Update ()
{
Historical (Free_Memory (), Allocated_Memory ());
fireChange ();
}

/**	Add values to the memory history.
<p>
	If the history {@link #Max_Samples() capacity} has not been reached
	the valid {@link #Samples() samples count} is incremented. All the
	current allocated and free memory sample values are shifted down in
	their arrays and then the new values are set as the current (index
	zero) entries.
<p>
	It might be argued that new sample values should be added to the end
	of the history arrays rather than at the beginning. However, once the
	history capacity has been reached the array contents would need to be
	shifted anyway. It is conceptually and programatically easier to work
	with the history arrays if the most recent sample is first.
<p>
	@param	free		The new free memory sample value.
	@param	allocated	The new allocated memory sample value.
	@see	#Allocated_History()
	@see	#Free_History()
*/
protected synchronized void Historical
	(
	long	free,
	long	allocated
	)
{
int
	there = Samples;
if (Samples == Allocated.length)
	--there;
else
	++Samples;
for (int
		here = there - 1;
	 	there != 0;
	  --here,
	  --there)
	{
	Free[there]      = Free[here];
	Allocated[there] = Allocated[here];
	}
Free[0]      = free;
Allocated[0] = allocated;
}

/**	Get the count of valid memory history samples.
<p>
	@return	The number of valid memory history samples that are available.
	@see	#Allocated_History()
	@see	#Free_History()
*/
public int Samples ()
{return Samples;}

/**	Get the maximum number of history samples that can be recorded.
<p>
	@return	The maximum number of history samples that will be recorded.
	@see	#Allocated_History()
	@see	#Free_History()
*/
public int Max_Samples ()
{return Allocated.length;}

/**	Set the maximum number of history samples that can be recorded.
<p>
	If the new sample capacity is different from the current sample
	capacity new sample history arrays are allocated and the previous
	valid sample values, up to the current {@link #Samples() sample count}
	or the new capacity (whichever is smaller), are copied into the
	new history arrays. Then the new history arrays are set as the
	current history arrays.
<p>
	After the history arrays have been changed a ChangeEvent is {@link
	#fireChange() fired}.
<p>
	@param	max_samples	The memory history sample capacity. If the value
		is less than or equal to zero the {@link #DEFAULT_MAX_SAMPLES}
		will be used.
*/
public synchronized void Max_Samples
	(
	int		max_samples
	)
{
if (max_samples <= 0)
	max_samples = DEFAULT_MAX_SAMPLES;
if (max_samples == Allocated.length)
	//	No change.
	return;

//	Allocate the new history buffers.
long[]
	free_history      = new long[max_samples],
	allocated_history = new long[max_samples];

if (Samples > max_samples)
	//	Drop the excess samples.
	Samples = max_samples;

//	Copy the samples from the old history to the new history.
for (int
		index = 0;
		index < Samples;
	  ++index)
	{
	free_history[index]      = Free[index];
	allocated_history[index] = Allocated[index];
	}

//	Use the new histroy buffers.
Free      = free_history;
Allocated = allocated_history;

//	Let everyone know that the history buffers have changed.
fireChange ();
}

/**	The memory history is emptied.
<p>
	After the memory history has been reset a ChangeEvent is {@link
	#fireChange() fired}.
<p>
	@see	#Clear()
*/
public void Reset ()
{
Clear ();
fireChange ();
}

/**	Clear the memory history.
<p>
	The memory sample history array values and the count of valid
	samples are set to zero.
<p>
	@see	#Samples()
	@see	#Allocated_History()
	@see	#Free_History()
*/
protected synchronized void Clear ()
{
while (Samples > 0)
	{
	Free[--Samples]    = 0;
	Allocated[Samples] = 0;
	}
}

/**	Get free memory samples history.
<p>
	@return	An array of long values that contain the {@link
		#Historical(long, long) history} of {@link #Free_Memory()
		free memory} samples. This array will have as many entries as
		the {@link #Samples() samples count}.
*/
public synchronized long[] Free_History ()
{
long[]
	history = new long[Samples];
for (int
		index = 0;
		index < Samples;
	  ++index)
	history[index] = Free[index];
return history;
}

/**	Get allocated memory samples history.
<p>
	@return	An array of long values that contain the {@link
		#Historical(long, long) history} of {@link #Allocated_Memory()
		allocated memory} samples. This array will have as many entries as
		the {@link #Samples() samples count}.
*/
public synchronized long[] Allocated_History ()
{
long[]
	history = new long[Samples];
for (int
		index = 0;
		index < Samples;
	  ++index)
	history[index] = Allocated[index];
return history;
}

/**	Get the memory samples history.
<p>
	@return An array of long value pairs that contain the {@link
		#Historical(long, long) history} of {@link #Allocated_Memory()
		allocated memory} samples in the first value of the pair
		(array[i][0] and {@link #Free_Memory() free memory} samples in
		the second value of the pair (array[i][1]). This array will have
		as many sample value pair entries as the {@link #Samples()
		samples count}.
*/
public synchronized long[][] History ()
{
long[][]
	history = new long[Samples][2];
for (int
		index = 0;
		index < Samples;
	  ++index)
	{
	history[index][0] = Allocated[index];
	history[index][1] = Free[index];
	}
return history;
}

/**	Register a ChangeListener.
<p>
	@param	listener	The ChangeListener to be added to the list
		registered for this Memory_History. The listener is not
		added if it is already registered..
*/
public void addChangeListener
	(
	ChangeListener	listener
	)
{
if (listener != null &&
	! Change_Listeners.contains (listener))
	Change_Listeners.add (listener);
}

/**	Remove a registered ChangeListener.
<p>
	@param	listener	The ChangeListener to be removed.
*/
public boolean removeChangeListener
	(
	ChangeListener	listener
	)
{return Change_Listeners.remove (listener);}

/**	Send a ChangeEvent referencing this Memory_History to each
	registered ChangeListener.
<p>
	@see	#addChangeListener(ChangeListener)
*/
protected void fireChange ()
{
Iterator
	listeners = Change_Listeners.iterator ();
while (listeners.hasNext ())
	((ChangeListener)listeners.next ()).stateChanged (new ChangeEvent (this));
}

}	//	End of Memory_History class.

