"""
Pydap proxy.

Works as a "frontend" to another DAP server. Normal responses are
simply proxied, but additional ones (like JSON, KML, etc.) are
generated by connecting to the server as a DAP client.

Sample usage. Create a file called ``server.ini``::

    [server:main]
    use = egg:Paste#http
    # Change to 0.0.0.0 to make public
    host = 127.0.0.1
    port = 8080

    [app:main]
    use = egg:dap#proxy
    url = http://nomad2.ncep.noaa.gov:9090/dods/sst
    responses = json kml wms
    verbose = 1
    debug = 1
    cache = .pydap-cache

And run ``paster serve server.ini``. This will forward a request
originally to::

    http://localhost:8080/oiv2.dds

To the URL::

    http://nomad2.ncep.noaa.gov:9090/dods/sst/oiv2.dds

But a request to::

    http://localhost:8080/oiv2.json
    
Will be processed by pydap by connecting to the remote dataset as
a client, and generating the JSON response on the fly.
"""

import urlparse

from paste.proxy import TransparentProxy
from paste.request import construct_url
from wsgifilter import Filter
from wsgigilter.fixuplinks import fixup_text_links

from dap.client import open
from dap.wsgi.application import SimpleApplication


def make_proxy(global_conf, url, responses, verbose=False, debug=False, cache=None, **kwargs):
    from paste.deploy.converters import aslist, asbool
    responses = aslist(responses)
    verbose = asbool(verbose)
    debug = asbool(debug)
    return DapDispatcher(url, responses, verbose, debug, cache, **kwargs)


class URLFilter(Filter):
    def filter(self, environ, headers, data):
        repl = environ.get('repl')
        # TODO: check relocatereponse etc.
        func = lambda s: s.replace(*repl)
        return fixup_text_links(data, func)


class DapDispatcher(object):
    def __init__(self, url, responses, verbose=False, debug=False, cache=None):
        self.url = url
        self.responses = responses
        self.verbose = verbose
        self.debug = debug
        self.cache = cache
    
    def __call__(self, environ, start_response):
        environ['x-wsgiorg.throw_errors'] = self.debug
        base = construct_url(environ, with_query_string=False, with_path_info=False) 

        # Extract response.
        request = environ.get('PATH_INFO', '').lstrip('/')
        response = request.split('.')[-1]

        if response not in self.responses:
            proxy = TransparentProxy()
            scheme, netloc, path, queries, fragment = urlparse.urlsplit(self.url)
            environ['wsgi.url_scheme'] = scheme
            environ['HTTP_HOST'] = netloc
            environ['SCRIPT_NAME'] = path.rstrip('/')
            environ['repl'] = (self.url, base)
            app = URLFilter(proxy)
        else:
            # Connect to server.
            request = request[:-len(response)-1]
            url = urlparse.urljoin(self.url + '/', request)
            dataset = open(url, verbose=self.verbose, cache=self.cache)
            app = SimpleApplication(dataset)

        return app(environ, start_response)
