/**************************************************************************
*   Copyright (C) 2000-2012 by Johan Maes                                 *
*   on4qz@telenet.be                                                      *
*   http://users.telenet.be/on4qz                                         *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include "soundio.h"
#include "qsstvglobal.h"
#include"soundcontrol.h"
#include <QDebug>
#include <QApplication>
#include <stdio.h>
#include <unistd.h>
#include "utils/supportfunctions.h"
//#include "scope/scopeview.h"
#include <sys/time.h>
#include <sys/timex.h>


soundIO *soundIOPtr;


static const unsigned int rates[] =
{
  5512,
  8000,
  11025,
  16000,
  22050,
  32000,
  44100,
  48000,
  64000,
  88200,
  96000,
  176400,
  192000,
};


soundIO::soundIO(int s)
{
  setSamplerate(s);
  soundOK=false;

}


void soundIO::setSamplerate(int s)
{
  samplerate=s;
}

void  soundIO::run()
{
  unsigned int delay;
  delay=((playbackBufferSize*1000)/samplerate);
  abort=false;
  playbackState=PBINIT;
  captureState=CPINIT;
  if(!soundOK) return;
  while(!abort)
    {
      switch(playbackState)
        {
        case PBINIT:
          txBuffer.reset();
          detectedPlaybackState= PBINIT;
        break;
        case PBSTARTING:
          if (play()==0) msleep(10);
          else
            {
              playbackState=PBRUNNING;
              addToLog("playback started",LOGPERFORM);
            }
          detectedPlaybackState= PBSTARTING;
        break;
        case PBRUNNING:
          if (play()==0)
            {
               addToLog(QString("playback stopped: delay=%1").arg(delay),LOGPERFORM);
               msleep(delay);
               waveOut.close();
               addToLog("playback stopped",LOGPERFORM);
               emit playbackStopped();
               playbackState=PBINIT;
            }
          detectedPlaybackState= PBRUNNING;
        break;
        case PBCALIBRATE1:
          {
            detectedPlaybackState= PBCALIBRATE1;
            if (playbackCalibration()!=0)
              {
                playbackState=PBCALIBRATE2;
              }
          }
        break;
        case PBCALIBRATE2:
          {
            if (playbackCalibration()==0) break;
            detectedPlaybackState= PBCALIBRATE2;
          }
          break;
       }
      switch(captureState)
        {
        case CPINIT:
//          rxBuffer.reset();
          detectedCaptureState=CPINIT;
        break;
        case CPSTARTING:
          rxBuffer.reset(); //clear the rxBuffer
          snd_pcm_prepare (captureHandle );
          snd_pcm_start (captureHandle);
          detectedCaptureState=CPSTARTING;
          captureState=CPRUNNING;
        break;
        case CPRUNNING:
          if (capture()==0) msleep(60);
          detectedCaptureState=CPRUNNING;
        break;
        case CPCALIBRATE:
          {
            if (captureCalibration()==0) msleep(0);
            detectedCaptureState= CPCALIBRATE;
          }
          break;
        case CPEND:
          msleep(1000);
          captureState=CPINIT;
         break;
       }
      if((captureState==CPINIT) &&(playbackState==PBINIT))   msleep(200);
    }
  abort=false;
  playbackState=PBINIT;
  captureState=CPINIT;
}


double soundIO::getPlaybackStartupTime()
{
  return (double)playbackBufferSize/BASESAMPLERATE;
}



/**
 * @brief play sound from txBuffer
 *
 * The sound is controlled by soundRoutingOutput (via configuration).
 * - SNDOUTCARD  sound to card only
 * - SNDOUTTOFILE sound to card and saved in wav file
 * - SNDOUTFILETOCARD sound from wav file to card
 *
 * @return number of frames sent.
 *
 * The readIndex must always be a multiple of 2*PERIODSIZE because we don't check the txBuffer for wrap-around.
 *
 */

int soundIO::play()
{
 unsigned int numFrames,restFrames;
 int framesWritten,error;
  addToLog(QString("readIndex: %1,writeIndex: %2 count: %3").arg(txBuffer.getReadIndex()).arg(txBuffer.getWriteIndex()).arg(txBuffer.count()),LOGSOUND);
  if(playbackState==PBSTARTING)
    {
      // prefill buffers to the max
      if((numFrames=txBuffer.count())<playbackBufferSize) return 0;
      addToLog(QString("start numFrame: %1").arg(numFrames),LOGSOUND);
      numFrames=playbackBufferSize;
    }
  else
    {
      if((numFrames=txBuffer.count())>=(2*PERIODSIZE))
        {
          numFrames=2*PERIODSIZE;
        }
    }

  if(soundRoutingOutput==SNDOUTTOFILE)
    {
      if(storedFrames<=(ulong)recordingSize*1048576L)
        {
          waveOut.write((quint16*)txBuffer.readPointer(),numFrames,isStereo);
          storedFrames+=numFrames;
//          msleep(10); // give some processingtime
//          return numFrames;
        }
    }
  restFrames=numFrames;
  while(restFrames)
    {
      framesWritten =  snd_pcm_writei ( playbackHandle, txBuffer.readPointer(), restFrames);
//      arrayDump(QString("sio %1").arg(frameCounter++),txBuffer.readPointer(),32,true);
      if(framesWritten<0)
        {
          if ( framesWritten ==  -EAGAIN )
            {
              return -1;
            }
          else if ( framesWritten == -EPIPE )
            {
              /* underrun */
              errorHandler(framesWritten,QString("Underrun recovery for %1").arg(numFrames));
              error = snd_pcm_prepare (playbackHandle);
              if ( error < 0 )
                {
                  errorHandler(framesWritten,"Can't recover from underrun, prepare failed");
                  snd_pcm_drop (playbackHandle);
                }
            }
          else
            {
              errorHandler(framesWritten,"Unhandled error in playback");
              snd_pcm_drop (playbackHandle );
            }
        }
      else
        {
          addToLog(QString("framesWritten:%1").arg(framesWritten),LOGSOUND);
          txBuffer.skip(framesWritten);
          if(restFrames!=numFrames)
            {
              addToLog(QString("restFrames: %1; framesWritten %2").arg(restFrames).arg(framesWritten),LOGSOUND);
            }
          msleep(10);
          restFrames-=framesWritten;
        }
    }
  return numFrames;
}

int soundIO::playbackCalibration()
{
  int numFrames,framesWritten;
  if(playbackState==PBCALIBRATE1)
    {
      numFrames=playbackBufferSize;
      framesWritten =  snd_pcm_writei ( playbackHandle, txBuffer.readPointer(), numFrames);
      calibrationFrames=0;
      return numFrames;
    }
  else
    {
      numFrames=1*PERIODSIZE;
    }
  framesWritten =  snd_pcm_writei ( playbackHandle, txBuffer.readPointer(), numFrames);
  if(numFrames!=framesWritten)
    {
       framesWritten =  snd_pcm_writei ( playbackHandle, txBuffer.readPointer(), numFrames-framesWritten);
    }
  else
    {

    }
  mutex.lock();
  calibrationFrames+=1;
  calibrationTime=stopwatch.elapsed();
  if(calibrationFrames==CALIBRATIONLEADIN)
    {
      stopwatch.start();
      addToLog("start calrx",LOGCAM);
    }
  mutex.unlock();
  return numFrames;
}

/**
 * @brief capture sound to rxBuffer
 *
 * The sound recording is controlled by soundRoutingInput (via configuration).
 *
 * - SNDINCARD sound from card
 * - SNDINFILE sound from a wav file
 * - SNDINCARDTOFILE sound from card and recording to wav file
 *
 * @return number of frames recorded.
 */

int soundIO::capture()
{
  int count;
  QString debugStr;
  if(rxBuffer.spaceLeft()<PERIODSIZE) return 0;
  if(soundRoutingInput==SNDINFILE)
    {
      count=waveIn.read((quint32*)tempRXBuffer,PERIODSIZE);
     //delay to give realtime feeling
      msleep((100*count)/samplerate);
      if(count<=0)
        {
          waveIn.close();
          captureState=CPEND;
          return 0;
        }
    }
  else
    {
      count=snd_pcm_avail(captureHandle); // check for available frames
//      addToLog(QString("countcheck %1 ").arg(count),LOGSOUND);

      if(count>=PERIODSIZE)
        {
          count = snd_pcm_readi(captureHandle, tempRXBuffer,PERIODSIZE);
        }
      if ( count < 0 )
        {
          if ( count != -EAGAIN )
            {
              if ( count == -EPIPE )
                {
                  // Overrun
                  snd_pcm_prepare (captureHandle );
                  snd_pcm_start (captureHandle);
                  qDebug()<< "Overrun";
                }
              else
                {
                  snd_pcm_drop (captureHandle );
                  qDebug()<<"Overrun , reason: "<< count << "Stopping device";
                }
            }
          addToLog("soundIO: sound eagain",LOGSOUND);
          return 0;
        }
      if(count!=PERIODSIZE)
        {
          return 0;
        }
      else
        {
          if(soundRoutingInput==SNDINCARDTOFILE)
            {
              if(storedFrames<=(ulong)recordingSize*1048576L)
                {
                  addToLog(QString("writen %1 tofile").arg(count),LOGSOUND);
                  waveOut.write((quint16*)tempRXBuffer,count,isStereo);
                  storedFrames+=count;
                }
              else
                {
                  captureState=CPINIT;
                }
            }
        }
    }
//  addToLog(QString("rxBuffercount: %1").arg(count),LOGSOUND);

  if((isStereo) || (soundRoutingInput==SNDINFILE))
    {
      for(int i=0;i<count;i++)
        {
          tempRXBuffer[i]=tempRXBuffer[2*i];  // setup as mono channel
        }
     }
  rxBuffer.putNoCheck(tempRXBuffer,count);
//  addToLog(QString("in buffer count: %1").arg(rxBuffer.count()),LOGSOUND);
  return count;
}

int soundIO::captureCalibration()
{
  int count;
  addToLog("calibration",LOGSOUND);
  count = snd_pcm_readi( captureHandle, tempRXBuffer,PERIODSIZE);
//  addToLog(QString("calibration count: %1").arg(count),LOGSOUND);
  if (count==PERIODSIZE)
    {
//      addToLog("calibration 2",LOGSOUND);
      mutex.lock();
      calibrationFrames++;
      calibrationTime=stopwatch.elapsed();
      if(calibrationFrames==CALIBRATIONLEADIN)
        {
          stopwatch.start();
        }
      mutex.unlock();
    }
  else if(count>=0)
    {

      count = snd_pcm_readi( captureHandle, tempRXBuffer,PERIODSIZE-count);
      mutex.lock();
      calibrationFrames++;
      calibrationTime=stopwatch.elapsed();
      mutex.unlock();

    }
  else
    {
      calibrationFrames=0; // restart calibration
      qDebug() << "restarting calibration";
      if ( count != -EAGAIN )
        {
          if ( count == -EPIPE )
            {
              // Overrun
              snd_pcm_prepare (captureHandle );
              snd_pcm_start (captureHandle);
              qDebug() << "Overrun";
            }
          else
            {
              snd_pcm_drop (captureHandle );
              qDebug()<<"Overrun , reason: "<< count << "Stopping device";
            }
        }
      addToLog("soundIO: sound eagain",LOGSOUND);
      return 0;
    }

  return count;
}

/**
 * @brief Initialisation of the soundcard hardware in full-duplex
 *
 * This function must be called before using the soundcard and after the configuration has been loaded
 *@return bool true if call was succesfull
 */

bool soundIO::init()
{
  int err;
  QString tempDevice;
  lastError.clear();
  if(soundOK)
    {
     // we have to stop thread first
      stopAndWait();
      snd_pcm_close(playbackHandle);
      snd_pcm_close(captureHandle);
    }
  soundOK=false;
  tempDevice=outputAudioDevice.left(outputAudioDevice.indexOf(" "));
  err = snd_pcm_open(&playbackHandle,tempDevice.toLatin1().data(), SND_PCM_STREAM_PLAYBACK,0); //open in blocking mode
  if(!errorHandler(err,"Unable to open "+outputAudioDevice)) return false;
  tempDevice=inputAudioDevice.left(inputAudioDevice.indexOf(" "));
  err = snd_pcm_open(&captureHandle,tempDevice.toLatin1().data(), SND_PCM_STREAM_CAPTURE, 0);
  if(!errorHandler(err,"Unable to open "+inputAudioDevice)) return false;
  snd_pcm_hw_params_malloc ( &hwparams );
  snd_pcm_sw_params_malloc ( &swparams );
  if(setupSoundParams(true))
    {
      if(setupSoundParams(false)) soundOK=true;
    }

  snd_pcm_hw_params_free ( hwparams );
  snd_pcm_sw_params_free ( swparams );
   return soundOK;
}

bool soundIO::stoppedPlaying()
{
  if(playbackState==PBINIT)
    {
      return true;
    }
  else
    {
      return false;
    }
}

bool soundIO::setupSoundParams(bool isCapture)
{
  int err;
  int dir=0;
  snd_pcm_t *handle;

  playbackPeriodSize=PERIODSIZE;
  playbackBufferSize=BUFFERSIZE;
  capturePeriodSize=PERIODSIZE;
  captureBufferSize=BUFFERSIZE;

  if(isCapture) handle=captureHandle;
  else handle=playbackHandle;

  /* Choose all parameters */
  err = snd_pcm_hw_params_any ( handle, hwparams );
  if(!errorHandler(err,"Broken configuration : no configurations available")) return false;

  /* Set the interleaved read/write format */
  err = snd_pcm_hw_params_set_access ( handle, hwparams, SND_PCM_ACCESS_RW_INTERLEAVED );
  if(!errorHandler(err,"Access type not available : ")) return false;

  /* Set the sample format */
  err = snd_pcm_hw_params_set_format ( handle, hwparams, SND_PCM_FORMAT_S16_LE );
  if(!errorHandler(err,"Sample format Float not available : ")) return false;
  /* Set the count of channels */
  if(isCapture)
    {
      err=snd_pcm_hw_params_get_channels_min(hwparams,&minChannelsCapture);
      err=snd_pcm_hw_params_get_channels_max(hwparams,&maxChannelsCapture);
      err = snd_pcm_hw_params_set_channels ( handle, hwparams, minChannelsCapture);
      if(!errorHandler(err,"Channels count not correct; " )) return false;
     }
  else
    {

      err=snd_pcm_hw_params_get_channels_min(hwparams,&minChannelsPlayback);
      err=snd_pcm_hw_params_get_channels_max(hwparams,&maxChannelsPlayback);
      err = snd_pcm_hw_params_set_channels ( handle, hwparams, 2); //allways stereo output
      if(!errorHandler(err,"Channels count not correct; " ))
        {
          return false;
        }
    }

  err = snd_pcm_hw_params_set_rate ( handle, hwparams, samplerate, 0 );
  if(!errorHandler(err,QString("Samplerate %1 not available").arg(samplerate))) return false;

  if(isCapture)
    {
      err = snd_pcm_hw_params_set_period_size_near ( handle, hwparams, &capturePeriodSize, &dir );
      if(!errorHandler(err,QString("Unable to set period size %1 for capture").arg(capturePeriodSize))) return false;
      err = snd_pcm_hw_params_set_buffer_size_near ( handle, hwparams, &captureBufferSize );
      if(!errorHandler(err,QString("Unable to set buffersize %1 for capture").arg(captureBufferSize))) return false;
    }
  else
    {
      err = snd_pcm_hw_params_set_period_size_near ( handle, hwparams, &playbackPeriodSize, &dir );
      if(!errorHandler(err,QString("Unable to set period size %1 for playback").arg(playbackPeriodSize))) return false;
      err = snd_pcm_hw_params_set_buffer_size_near ( handle, hwparams, &playbackBufferSize );
      if(!errorHandler(err,QString("Unable to set buffersize %1 for playback").arg(playbackBufferSize))) return false;
    }
  err = snd_pcm_hw_params ( handle, hwparams );
  if(isCapture)
    {
      if(!errorHandler(err,QString("Unable to set hw params for capture:"))) return false;
    }
  else
    {
      if(!errorHandler(err,QString("Unable to set hw params for playback:"))) return false;
    }

  /* Get the current swparams */
  err = snd_pcm_sw_params_current ( handle, swparams );
  if(!errorHandler(err,"Unable to determine current swparams")) return false;
  err = snd_pcm_sw_params_set_start_threshold ( handle, swparams, 2048 );
  if(!errorHandler(err,"Unable to set start threshold mode")) return false;
   /* Write the parameters to the record/playback device */
  err = snd_pcm_sw_params ( handle, swparams );
  if(!errorHandler(err,"Unable to set sw params for output")) return false;
  return true;
 }

bool soundIO::listCards()
{
  int  err;
  int  cardNum, totalCards;
  char   str[64];
  snd_ctl_card_info_t *cardInfo;
  int      devNum;
//  int totalDevices;
  snd_pcm_info_t *pcmInfo;
  snd_ctl_t *cardHandle;
  inputPCMNameList << "default";
  outputPCMNameList << "default";

  // No cards found yet
  totalCards = 0;

  // Start with first card
  cardNum = -1;

  for (;;)
    {

      // Get next sound card's card number. When "cardNum" == -1, then ALSA fetches the first card
      if ((err = snd_card_next(&cardNum)) < 0)
        {
          qDebug() << "Can't get the next card number:" << snd_strerror(err);
          return false;
        }

      if (cardNum < 0) break; // No more cards? ALSA sets "cardNum" to -1 if so
      sprintf(str, "hw:%i", cardNum);
      if ((err = snd_ctl_open(&cardHandle, str, 0)) < 0)
        {
          qDebug() << "Can't open card "<< cardNum << snd_strerror(err);
          continue;
        }
      // We need to get a snd_ctl_card_info_t. Just alloc it on the stack
      snd_ctl_card_info_alloca(&cardInfo);

      // Tell ALSA to fill in our snd_ctl_card_info_t with info about this card
      if ((err = snd_ctl_card_info(cardHandle, cardInfo)) >= 0)
        qDebug() << "Card " << cardNum << "name: " << snd_ctl_card_info_get_name(cardInfo);
//      totalDevices = 0;
      devNum = -1; // Start with the first wave device on this card
      for (;;)
        {
          if ((err = snd_ctl_pcm_next_device(cardHandle, &devNum)) < 0)  // Get the number of the next wave device on this card
            {
              qDebug()<< "Can't get next wave device number: " << snd_strerror(err);
              break;
            }

          // No more wave devices on this card? ALSA sets "devNum" to -1 if so.
          // NOTE: It's possible that this sound card may have no wave devices on it
          // at all, for example if it's only a MIDI card
          if (devNum < 0) break;
          // To get some info about the subdevices of this wave device (on the card), we need a
          // snd_pcm_info_t, so let's allocate one on the stack
          snd_pcm_info_alloca(&pcmInfo);
          memset(pcmInfo, 0, snd_pcm_info_sizeof());

          // Tell ALSA which device (number) we want info about
          snd_pcm_info_set_device(pcmInfo, devNum);
          // Get info on the wave outs of this device
          getDevices(cardHandle,cardNum,devNum,pcmInfo, SND_PCM_STREAM_PLAYBACK);
          getDevices(cardHandle,cardNum,devNum,pcmInfo, SND_PCM_STREAM_CAPTURE);
        }
      ++totalCards;
      // Close the card's control interface after we're done with it
      snd_ctl_close(cardHandle);
//      qDebug() << "Found:" << totalDevices << " digital audio devices on card " <<  cardNum;
    }

//  qDebug() << "ALSA found cards" << totalCards;

  // ALSA allocates some mem to load its config file when we call
  // snd_card_next. Now that we're done getting the info, let's tell ALSA
  // to unload the info and free up that mem
  snd_config_update_free_global();
  return true;
}


void soundIO::getDevices(snd_ctl_t *cardHandle,int cardNum,int devNum,snd_pcm_info_t *pcmInfo, snd_pcm_stream_t direction)
{
  int  err;
  int i= -1;
  int subDevCount = 1;
  snd_pcm_info_set_stream(pcmInfo, direction);


  // More subdevices?
  while (++i < subDevCount)
    {
      // Tell ALSA to fill in our snd_pcm_info_t with info on this subdevice
      snd_pcm_info_set_subdevice(pcmInfo, i);
      if ((err = snd_ctl_pcm_info(cardHandle, pcmInfo)) < 0)
        {
//          qDebug() << QString("Can't get info for wave output subdevice hw:%1,%2,%3: %4").arg(cardNum).arg(devNum).arg(i).arg(snd_strerror(err));
          continue;
        }
//      qDebug() << "PCM name" << snd_pcm_info_get_name(pcmInfo);
      // Print out how many subdevices (once only)
      if (!i)
        {
          subDevCount = snd_pcm_info_get_subdevices_count(pcmInfo);
//          qDebug() << QString("Found %1 wave output subdevices on card %2").arg(subDevCount).arg(cardNum);
        }

      // NOTE: If there's only one subdevice, then the subdevice number is immaterial,
      // and can be omitted when you specify the hardware name
      if(subDevCount>1)
        {
//          qDebug()<< QString("hw:%1,%2,%3").arg(cardNum).arg(devNum).arg(i);
          if(direction==SND_PCM_STREAM_CAPTURE)
            {
              inputPCMNameList << QString("hw:%1,%2,%3  %4").arg(cardNum).arg(devNum).arg(i).arg(snd_pcm_info_get_name(pcmInfo));
            }

          else
            {
              outputPCMNameList << QString("hw:%1,%2,%3  %4").arg(cardNum).arg(devNum).arg(i).arg(snd_pcm_info_get_name(pcmInfo));
            }

         }
      else
        {
//          qDebug()<< QString("hw:%1,%2").arg(cardNum).arg(devNum);
          if(direction==SND_PCM_STREAM_CAPTURE)
            {
              inputPCMNameList << QString("hw:%1,%2  %3").arg(cardNum).arg(devNum).arg(snd_pcm_info_get_name(pcmInfo));
            }

          else
            {
              outputPCMNameList << QString("hw:%1,%2  %3").arg(cardNum).arg(devNum).arg(snd_pcm_info_get_name(pcmInfo));
           }
        }

    }

}

bool soundIO::errorHandler(int err,QString info)
{
  if(err<0)
    {
      lastError=info+ ": "+ QString ( snd_strerror ( err ) );
      qDebug() << lastError;
      addToLog(lastError,LOGALL);
      return false;
    }
  return true;
}

void soundIO::errorHandler(QString info)
{
  lastError=info;
  qDebug() << lastError;
  addToLog(lastError,LOGALL);
}



bool soundIO::startCapture()
{
  playbackState=PBINIT;
  if(!soundOK) return false;
  switch(soundRoutingInput)
    {
      case SNDINCARD:
      if(minChannelsCapture==1) isStereo=false; else isStereo=true;
      break;
      case SNDINFILE:
      if(!waveIn.openFileForRead("",true))
        {
          errorHandler("File not opened");
          return false;
        }
      if(waveIn.getNumberOfChannels()==1) isStereo=false; else isStereo=true;
      break;
      case SNDINCARDTOFILE:
        {
        if(minChannelsCapture==1) isStereo=false; else isStereo=true;
          if(!waveOut.openFileForWrite("",true,isStereo))
            {
              errorHandler("File not opened");
              return false;
            }
        }
      break;
    }
  rxBuffer.reset();
  captureState=CPSTARTING;
  addToLog("start capturing",LOGSOUND);
  return true;
}

bool soundIO::startPlayback()
{
  frameCounter=0;
  captureState=CPINIT;
  if(!soundOK) return false;
  soundIOPtr->txBuffer.reset();

  if(soundRoutingOutput==SNDOUTTOFILE)
    {
      if(!waveOut.openFileForWrite("",true,true)) // indicate stereo
        {
          errorHandler("File not opened");
          return false;
        }
    }
  playbackState=PBSTARTING;
  snd_pcm_prepare (playbackHandle);
  addToLog(QString("start playback, txbuffercount: %1").arg(txBuffer.count()),LOGSOUND);

  return true;
}

bool soundIO::startCalibration(bool capture)
{
  captureState=CPINIT;
  playbackState=PBINIT;
  if(!soundOK) return false;
  while(detectedCaptureState!=CPINIT)
    {
      qApp->processEvents();
    };
  while(detectedPlaybackState!=PBINIT)
    {
      qApp->processEvents();
    };
  calibrationFrames=0;
  if(capture)
    {
     snd_pcm_prepare (captureHandle );
     snd_pcm_start (captureHandle);
     captureState=CPCALIBRATE;
    }
  else
    {
      txBuffer.fill(0);
     //
      snd_pcm_start (playbackHandle);
      //snd_pcm_prepare (playbackHandle );
      playbackState=PBCALIBRATE1;
    }
  return true;
}

void soundIO::idle()
{
  waveOut.close();
  waveIn.closeFile();
  captureState=CPINIT;
  playbackState=PBINIT;
}

/**
 * @brief stopping soundio thread
 */

void soundIO::stopAndWait()
{
  if(!soundOK) return;
  waveOut.write(NULL,0,false);
  abort=true;
  while(abort)
    {
      qApp->processEvents();
    }
}


int soundIO::calibrationCount(int &frames)
{
  int tempT,tempC;
  mutex.lock();
  tempC=calibrationFrames;
  tempT=calibrationTime;
  mutex.unlock();
  frames=tempC-CALIBRATIONLEADIN;
  return tempT;
}





